/*
 *  linux/fs/hpfs/alloc.c
 *
 *  Mikulas Patocka (mikulas@artax.karlin.mff.cuni.cz), 1998-1999
 *
 *  HPFS bitmap operations
 */

#include "hpfs_fn.h"

/*
 * Check if a sector is allocated in bitmap
 * This is really slow. Turned on only if chk==2
 */

static int chk_if_allocated(struct super_block *s, secno sec, char *msg)
{
	struct quad_buffer_head qbh;
	unsigned *bmp;
	if (!(bmp = map_bitmap(s, sec >> 14, &qbh))) goto fail;
	if ((bmp[(sec & 0x3fff) >> 5] >> (sec & 0x1f)) & 1) {
		hpfs_error(s, "sector '%s' - %08x not allocated in bitmap", msg, sec);
		goto fail1;
	}
	brelse4(&qbh);
	if (sec >= s->s_hpfs_dirband_start && sec < s->s_hpfs_dirband_start + s->s_hpfs_dirband_size) {
		unsigned ssec = (sec - s->s_hpfs_dirband_start) / 4;
		bmp = map_dnode_bitmap(s, &qbh);
		if ((bmp[ssec >> 5] >> (ssec & 0x1f)) & 1) {
			hpfs_error(s, "sector '%s' - %08x not allocated in directory bitmap", msg, sec);
			goto fail1;
		}
		brelse4(&qbh);
	}
	return 0;
	fail1:
	brelse4(&qbh);
	fail:
	return 1;
}

/*
 * Check if sector(s) have proper number and additionally check if they're
 * allocated in bitmap.
 */
	
int chk_sectors(struct super_block *s, secno start, int len, char *msg)
{
	if (start + len < start || start < 0x12 ||
	    start + len > s->s_hpfs_fs_size) {
	    	hpfs_error(s, "sector(s) '%s' badly placed at %08x", msg, start);
		return 1;
	}
	if (s->s_hpfs_chk>=2) {
		int i;
		for (i = 0; i < len; i++)
			if (chk_if_allocated(s, start + i, msg)) return 1;
	}
	return 0;
}

static secno tst_alloc(unsigned *bmp, unsigned base, unsigned n,
		       unsigned forward)
{
	unsigned i;
	base &= 0x3fff;
	for (i = base; i < base + ALLOC_GR && i < 0x4000; i += n) {
		unsigned t;
		if (!(i&0x1f)) if (!bmp[i >> 5]) {
			i += 32 - n;
			continue;
		}
		if (i + n + forward > 0x4000) break;
		if (!(t = tstbits(bmp, i, n + forward))) {
			return i;
		}	
		i += t - 1;
		i &= ~(n - 1);
	}
	return -1;
}

static secno alloc_in_bmp(struct super_block *s, secno near, unsigned n,
			  unsigned forward)
{
	struct quad_buffer_head qbh;
	unsigned *bmp;
	unsigned bs = near & ~0x3fff;
	unsigned nr = (near & 0x3fff) & ~(n - 1);
	unsigned mnr;
	unsigned i;
	secno ret = 0;
	lock_super(s);
	if (bs != ~0x3fff) {
		if (!(bmp = map_bitmap(s, near >> 14, &qbh))) goto uls;
	} else {
		if (!(bmp = map_dnode_bitmap(s, &qbh))) goto uls;
	}
	if (!tstbits(bmp, nr, n+forward)) {
		ret = bs + nr;
		goto rt;
	}
	if (!tstbits(bmp, nr + n, n+forward)) {
		ret = bs + nr + n;
		goto rt;
	}
	nr &= ~0x1f;
	mnr = nr > 0x1fff ? nr : 0x4000 - nr;
	for (i = 0; i <= mnr; i += ALLOC_GR ) {
		secno r;
		if (nr + i < 0x4000) if ((r = tst_alloc(bmp, nr + i, n, forward)) != -1) {
			ret = bs + r;
			goto rt;
		}
		if ((unsigned)(nr - i - ALLOC_GR) < 0x4000)
			if ((r = tst_alloc(bmp, nr - i - ALLOC_GR, n, forward)) != -1) {
				ret = bs + r;
				goto rt;
			}
	}
	rt:
	if (ret) {
		bmp[(ret & 0x3fff) >> 5] &= ~(((1 << n) - 1) << (ret & 0x1f));
		mark_4buffers_dirty(&qbh);
	}
	brelse4(&qbh);
	uls:
	unlock_super(s);
	return ret;
}

/*
 * Allocation strategy:	1) search place near the sector specified
 *			2) search bitmap where free sectors last found
 *			3) search all bitmaps
 *			4) search all bitmaps ignoring number of pre-allocated
 *				sectors
 */

secno alloc_sector(struct super_block *s, secno near, unsigned n,
		   unsigned forward, int lock)
{
	secno sec;
	unsigned i;
	unsigned n_bmps;
	int b = s->s_hpfs_c_bitmap;
	if (lock) lock_creation(s);
	if (near && near < s->s_hpfs_fs_size)
		if ((sec = alloc_in_bmp(s, near, n, forward/4))) goto ret;
	if (b != -1) {
		if (b < 0x10000000) if ((sec = alloc_in_bmp(s, b<<14, n, forward/2))) goto ret;
		else if ((sec = alloc_in_bmp(s, (b&0xfffffff)<<14, n, 0))) goto ret;
	}	
	n_bmps = (s->s_hpfs_fs_size + 0x4000 - 1) >> 14;
	for (i = 0; i < n_bmps / 2; i++) {
		if ((sec = alloc_in_bmp(s, (n_bmps/2+i) << 14, n, forward))) {
			s->s_hpfs_c_bitmap = n_bmps/2+i;
			goto ret;
		}	
		if ((sec = alloc_in_bmp(s, (n_bmps/2-i-1) << 14, n, forward))) {
			s->s_hpfs_c_bitmap = n_bmps/2-i-1;
			goto ret;
		}
	}
	if ((sec = alloc_in_bmp(s, (n_bmps-1) << 14, n, forward))) {
		s->s_hpfs_c_bitmap = n_bmps-1;
		goto ret;
	}	
	for (i = 0; i < n_bmps; i++)
		if ((sec = alloc_in_bmp(s, i << 14, n, 0))) {
			s->s_hpfs_c_bitmap = 0x10000000 + i;
			goto ret;
		}
	sec = 0;
	ret:
	if (lock) unlock_creation(s);
	return sec;
}

static secno alloc_in_dirband(struct super_block *s, secno near, int lock)
{
	unsigned nr = near;
	secno sec;
	if (nr < s->s_hpfs_dirband_start)
		nr = s->s_hpfs_dirband_start;
	if (nr >= s->s_hpfs_dirband_start + s->s_hpfs_dirband_size)
		nr = s->s_hpfs_dirband_start + s->s_hpfs_dirband_size - 4;
	nr -= s->s_hpfs_dirband_start;
	nr >>= 2;
	if (lock) lock_creation(s);
	sec = alloc_in_bmp(s, (~0x3fff) | nr, 1, 0);
	if (lock) unlock_creation(s);
	if (!sec) return 0;
	return ((sec & 0x3fff) << 2) + s->s_hpfs_dirband_start;
}

/* Alloc sector if it's free */

int alloc_if_possible(struct super_block *s, secno sec)
{
	struct quad_buffer_head qbh;
	unsigned *bmp;
	lock_creation(s);
	/*printk("3 - ");*/
	lock_super(s);
	if (!(bmp = map_bitmap(s, sec >> 14, &qbh))) goto end;
	if (bmp[(sec & 0x3fff) >> 5] & (1 << (sec & 0x1f))) {
		bmp[(sec & 0x3fff) >> 5] &= ~(1 << (sec & 0x1f));
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
		unlock_super(s);
		unlock_creation(s);
		return 1;
	}
	brelse4(&qbh);
	end:
	unlock_super(s);
	unlock_creation(s);
	return 0;
}

/* Free sectors in bitmaps */

void free_sectors(struct super_block *s, secno sec, unsigned n)
{
	struct quad_buffer_head qbh;
	unsigned *bmp;
	/*printk("2 - ");*/
	lock_super(s);
	new_map:
	if (!(bmp = map_bitmap(s, sec >> 14, &qbh))) {
		unlock_super(s);
		return;
	}	
	new_tst:
	if ((bmp[(sec & 0x3fff) >> 5] >> (sec & 0x1f) & 1)) {
		hpfs_error(s, "sector %08x not allocated", sec);
		brelse4(&qbh);
		unlock_super(s);
		return;
	}
	bmp[(sec & 0x3fff) >> 5] |= 1 << (sec & 0x1f);
	if (!--n) {
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
		unlock_super(s);
		return;
	}	
	if (!(++sec & 0x3fff)) {
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
		goto new_map;
	}
	goto new_tst;
}

/*
 * Check if there are at least n free dnodes on the filesystem.
 * Called before adding to dnode. If we run out of space while
 * splitting dnodes, it would corrupt dnode tree.
 */

int check_free_dnodes(struct super_block *s, int n)
{
	int n_bmps = (s->s_hpfs_fs_size + 0x4000 - 1) >> 14;
	int b = s->s_hpfs_c_bitmap;
	int i, j;
	unsigned *bmp;
	struct quad_buffer_head qbh;
	bmp = map_dnode_bitmap(s, &qbh);
	for (j = 0; j < 512; j++) {
		unsigned k;
		for (k = bmp[j]; k; k >>= 1) if (k & 1) if (!--n) {
			brelse4(&qbh);
			return 0;
		}
	}
	brelse4(&qbh);
	i = 0;
	if (b != -1) {
		bmp = map_bitmap(s, b, &qbh);
		goto chk_bmp;
	}
	chk_next:
	if (i == b) i++;
	if (i >= n_bmps) return 1;
	bmp = map_bitmap(s, i, &qbh);
	chk_bmp:
	for (j = 0; j < 512; j++) {
		unsigned k;
		for (k = 0xf; k; k <<= 4)
			if ((bmp[j] & k) == k)
				if (!--n) {
					brelse4(&qbh);
					return 0;
				}
	}
	brelse4(&qbh);
	i++;
	goto chk_next;
}

void free_dnode(struct super_block *s, dnode_secno dno)
{
	if (s->s_hpfs_chk) if (dno & 3) {
		hpfs_error(s, "free_dnode: dnode %08x not aligned", dno);
		return;
	}
	if (dno < s->s_hpfs_dirband_start ||
	    dno >= s->s_hpfs_dirband_start + s->s_hpfs_dirband_size) {
		free_sectors(s, dno, 4);
	} else {
		struct quad_buffer_head qbh;
		unsigned *bmp;
		unsigned ssec = (dno - s->s_hpfs_dirband_start) / 4;
		if (!(bmp = map_dnode_bitmap(s, &qbh))) return;
		bmp[ssec >> 5] |= 1 << (ssec & 0x1f);
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
	}
}

struct dnode *alloc_dnode(struct super_block *s, secno near,
			 dnode_secno *dno, struct quad_buffer_head *qbh,
			 int lock)
{
	struct dnode *d;
	if (!(*dno = alloc_in_dirband(s, near, lock)))
		if (!(*dno = alloc_sector(s, near, 4, 0, lock))) return NULL;
	if (!(d = get_4sectors(s, *dno, qbh))) {
		free_dnode(s, *dno);
		return NULL;
	}
	memset(d, 0, 2048);
	d->magic = DNODE_MAGIC;
	d->first_free = 52;
	d->dirent[0] = 32;
	d->dirent[2] = 8;
	d->dirent[30] = 1;
	d->dirent[31] = 255;
	d->self = *dno;
	return d;
}

struct fnode *alloc_fnode(struct super_block *s, secno near, fnode_secno *fno,
			  struct buffer_head **bh)
{
	struct fnode *f;
	if (!(*fno = alloc_sector(s, near, 1, FNODE_ALLOC_FWD, 1))) return NULL;
	if (!(f = get_sector(s, *fno, bh))) {
		free_sectors(s, *fno, 1);
		return NULL;
	}	
	memset(f, 0, 512);
	f->magic = FNODE_MAGIC;
	f->ea_offs = 0xc4;
	f->btree.n_free_nodes = 8;
	f->btree.first_free = 8;
	return f;
}

struct anode *alloc_anode(struct super_block *s, secno near, anode_secno *ano,
			  struct buffer_head **bh)
{
	struct anode *a;
	if (!(*ano = alloc_sector(s, near, 1, ALLOC_FWD, 1))) return NULL;
	if (!(a = get_sector(s, *ano, bh))) {
		free_sectors(s, *ano, 1);
		return NULL;
	}
	memset(a, 0, 512);
	a->magic = ANODE_MAGIC;
	a->self = *ano;
	a->btree.n_free_nodes = 40;
	a->btree.n_used_nodes = 0;
	a->btree.first_free = 8;
	return a;
}

