/*   symtab.cc : May 5, 1994   */

/* Copyright (C) 1994-1999  Sekhar Bala,
 *                          Ramchander Balasubramanian, and
 *                          Alphax Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "symtab.h"

/*------------------------------------------------------------------------*/

GLOBAL SYMTAB_TYP  symtab;
       CHAR       *SYMBOL_TYP::typenames[ST_TYPE_MAX] = { NULL, };

/*------------------------------------------------------------------------*/


/*+
 *      NAME : zero
 *
 *  FUNCTION : clear the symbol object
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
VOID SYMBOL_TYP::zero( VOID )
{

   symbol     = NULL;
   type       = ST_TYPE_NONE;
   value_free = TRUE;
   value_len  = 0;
   value      = NULL;
   args       = 0;
   return;

} /* zero */


/*+
 *      NAME : make
 *
 *  FUNCTION : creates a symbol object and clear all data values
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMBOL_TYP::make( VOID )
{

   zero();
   return( TRUE );

} /* make */


/*+
 *      NAME : make
 *
 *  FUNCTION : creates a symbol object by coping the raw data passed
 *
 *     ENTRY : etc.   all the args are values of the symbol class components
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMBOL_TYP::make(
   CHAR    *usymbol,
   INT_2    utype,
   BOOLEAN  uval_free,
   INT_2    uval_len,
   VOID    *uval,
   INT_2    uargs       )
{

   zero();
   if ( (usymbol != NULL) && (WC::strxfer(symbol, usymbol) != 0) )
      return( FALSE );
   type       = utype;
   value_free = uval_free;
   value_len  = uval_len;
   value      = uval;
   args       = uargs;
   return( TRUE );

} /* make */


/*+
 *      NAME : make
 *
 *  FUNCTION : creates a symbol object by coping the data contained an
 *             existing symbol object
 *
 *     ENTRY : sym -> existing symbol with data appropriately evaluated
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMBOL_TYP::make( SYMBOL_TYP *sym )
{
 BOOLEAN   result;

   result = make( sym->symbol,
                  sym->type,
                  sym->value_free,
                  sym->value_len,
                  sym->value,
                  sym->args );
   sym->value_free = FALSE;
   return( result );

} /* make */


/*+
 *      NAME : unmake
 *
 *  FUNCTION : destroy a symbol object
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
VOID SYMBOL_TYP::unmake( VOID )
{

   if ( symbol != NULL )
      ::free( symbol );
   symbol = NULL;
   if ( value_free ) {
       if ( value != NULL )
          ::free( value );
       value = NULL;
   }
   zero();
   return;

} /* unmake */


/*+
 *      NAME : allocate
 *
 *  FUNCTION : grow symbol array by 1 element
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::allocate( VOID )
{
 INT_2   size;

   // allocate next cell
   size = sizeof(FILE *) * (table_len + 1);
   if ( table_len == 0 )
      table = (SYMBOL_TYP *(*)[]) ::malloc( size );
   else
      table = (SYMBOL_TYP *(*)[]) ::realloc( table, size );

   // test for error
   if ( table == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      return( FALSE );
   }
   memset( &(*table)[table_len], 0, sizeof(FILE *) );

   // bump counter
   table_len++;

   // done
   return( TRUE );

} /* allocate */


/*+
 *      NAME : deallocate
 *
 *  FUNCTION : shrink symbol array by 1 element
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::deallocate( VOID )
{
 INT_2      size;

   // test for validity
   if ( table_len <= 0 ) {
      set_errhdl( ERR_ST_UNDERFLOW );
      return( FALSE );
   }

    // allocate next cell
   size  = sizeof(FILE *) * (table_len - 1);
   if ( table_len == 1 ) {
      if ( table != NULL )
         ::free( table );
      table = NULL;
   } else {
      table = (SYMBOL_TYP *(*)[]) ::realloc( table, size );
      if ( table == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         table_len--;
         return( FALSE );
      }
   }
   table_len--;

   // done
   return( TRUE );

} /* deallocate */


/*+
 *      NAME : sort_idx
 *
 *  FUNCTION : sort the dynamic array for bsearch routines
 *             is a bsort.  is a slow way, as each insertion
 *             resorting the entire dynamic array. Fix later
 *             if necessary.
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::sort_idx( VOID )
{
 INT_2         i;
 INT_2         j;
 INT_2         cmp;
 SYMBOL_TYP   *tsym;

   // test for validity
   if ( table == NULL )
      return( FALSE );

   // a slow sorting routine (should use qsort)
   for( i=0; (i < (table_len-1)); i++ ) {
      for( j=(i+1); (j < table_len); j++ ) {
         cmp = strcmpi( (*table)[i]->symbol, (*table)[j]->symbol );
         if ( cmp > 0 ) {
            tsym        = (*table)[i];
            (*table)[i] = (*table)[j];
            (*table)[j] = tsym;
         }
      }
   }

   // done ok
   return( TRUE );

} /* sort_idx */


/*+
 *      NAME : find_idx
 *
 *  FUNCTION : find the index of the symbol in the dynamic array
 *             uses the basic binary search algorithm
 *
 *     ENTRY : symbol -> name of symbol to find
 *             ibeg   -> beginning element of array to start searching
 *             iend   -> ending element of array to end seaching
 *
 *   RETURNS : index into dynamic array of symbols
 *
-*/
INT_2 SYMTAB_TYP::find_idx( CHAR *symbol, INT_2 ibeg, INT_2 iend )
{
 INT_2    idx;
 INT_2    cmp;

   if ( iend == -1 )
      iend = table_len;
   if ( (table == NULL) || (ibeg >= iend) )
      return( -1 );

   idx = ibeg + ((iend - ibeg) / 2);
   cmp = strcmpi( symbol, (*table)[idx]->symbol );
   if ( cmp == 0 )
       return( idx );
   if ( cmp < 0 ) {
      if ( idx == iend )
         return( -1 );
      iend = idx;
   } else { // if ( cmp > 0 )
      if ( idx == ibeg )
         return( -1 );
      ibeg = idx;
   }
   return( find_idx(symbol, ibeg, iend) );

} /* find_idx */


/*+
 *      NAME : find
 *
 *  FUNCTION : find a symbol in the table object
 *
 *     ENTRY : usymbol -> name of symbol to find
 *
 *   RETURNS : ptr to actual symbol in the table object
 *
-*/
SYMBOL_TYP *SYMTAB_TYP::find( CHAR *usymbol )
{
 INT_2    idx;


   idx = find_idx( usymbol );
   if ( idx == -1 )
       return( NULL );
   return( (*table)[idx] );

} /* find */


/*+
 *      NAME : insert
 *
 *  FUNCTION : insert a new symbol into the table object
 *             as a copy of the supplied symbol object
 *
 *     ENTRY : usym -> symbol object to copy into the table
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::insert( SYMBOL_TYP *usym )
{
 SYMBOL_TYP   *sym;

   //
   // THIS insertion algorithm is very slow
   // 1- insert at end
   // 2- then sort it
   //

   // create new symbol
   sym = NEW SYMBOL_TYP( usym );
   if ( sym == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      return( FALSE );
   }

   // alocate entry & insert at end
   if ( !allocate() )
      return( FALSE );
   (*table)[table_len-1] = sym;

   // sort & return
   if ( !sort_idx() )
      return( FALSE );
   return( TRUE );

} /* insert */


/*+
 *      NAME : remove
 *
 *  FUNCTION : remove a symbol from the table object
 *
 *     ENTRY : usymbol -> name of symbol to find and remove
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::remove( CHAR *usymbol )
{
 INT_2      idx;

   // find the entry
   idx = find_idx( usymbol );
   if ( idx == -1 )
      return( FALSE );

   // delete the object
   DELETE( (*table)[idx] );
   (*table)[idx] = NULL;

   // restore table usage
   for( ; (idx < (table_len-1)); idx++ ) {
      (*table)[idx] = (*table)[idx+1];
   }
   (*table)[table_len-1] = NULL;

   // remove the entry
   if ( !deallocate() )
      return( FALSE );

   // done ok
   return( TRUE );

} /* remove */


/*+
 *      NAME : zero
 *
 *  FUNCTION : clears data of symbol table object
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
VOID SYMTAB_TYP::zero( VOID )
{

   table_len = 0;
   table     = NULL;
   return;

} /* zero */


/*+
 *      NAME : make
 *
 *  FUNCTION : constructor-like initialization of the symbol table object
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
BOOLEAN SYMTAB_TYP::make( VOID )
{

   if ( SYMBOL_TYP::typenames[0] == NULL ) {
      SYMBOL_TYP::typenames[ST_TYPE_NONE]  = "NONE"     ;
      SYMBOL_TYP::typenames[ST_TYPE_INT32] = "INT32"    ;
      SYMBOL_TYP::typenames[ST_TYPE_INT]   = "INT"      ;
      SYMBOL_TYP::typenames[ST_TYPE_CHAR]  = "CHAR"     ;
      SYMBOL_TYP::typenames[ST_TYPE_FUNCT] = "FUNCTION" ;
   }
   zero();
   return( TRUE );

} /* make */


/*+
 *      NAME : make
 *
 *  FUNCTION : destructor-like initialization of the symbol table object
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
VOID SYMTAB_TYP::unmake( VOID )
{
 INT_2   i;

   for( i=0; ((table != NULL) && (i < table_len)); i++ ) {
      DELETE( (*table)[i] );
      (*table)[i] = NULL;
   }
   if ( table != NULL )
      ::free( table );
   table = NULL;
   zero();
   return;

} /* unmake */


/*------------------------------------------------------------------------*/


