/*   input1.cc : May 5, 1994   */

/* Copyright (C) 1994-1999  Sekhar Bala,
 *                          Ramchander Balasubramanian, and
 *                          Alphax Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "std.h"
#include "wildcard.h"
#include "parse.h"
#include "symtab.h"
#include "input.h"
#include "cgets.h"

/*-------------------------------------------------------------------------*/

STATIC CHAR       *Cstring  = "btrnf";
STATIC CHAR        Cvalue[] = { '\b', '\t', '\r', '\n', '\f', 0 };

/*-------------------------------------------------------------------------*/


/*+
 *      NAME : toINT
 *
 *  FUNCTION : parses "text" and determined the valid INT value; the "seps"
 *             are used as terminating characters in the parse
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE on success; FALSE on failure
 *
-*/
GLOBAL BOOLEAN toINT( INT_4 *result, CHAR *text, CHAR *seps )
{
 INT_4    value;
 INT_2    base;
 CHAR    *endp;
 CHAR    *pbuf;

   // default settings
   base = 0;
   endp = 
   pbuf = text;

   // parse string to add binary 0bxxxx
   while ( *endp == ' ' )
      endp++;
   if ( (endp[0] == '0') && ((endp[1] == 'b') || (endp[1] == 'B')) ) {
      base = 2;
      pbuf = &endp[2];
   }

   // test for 1-byte char string value
   if ( endp[0] == '\'' ) {
      endp++;
      if ( endp[0] == '\\' ) {
         endp++;
         pbuf = strchr( Cstring, *endp );
         if ( pbuf != NULL )
            endp = &Cvalue[(pbuf - Cstring)];
      }
      value = ( ((INT_4) *endp) & 0x00FF );
      endp++;
      if ( endp[0] != '\'' ) {
         set_errhdl( ERR_SEM_BADNUM1 );
         *result= 0;
         return( FALSE );
      }
   } else {
      value = (INT_4) strtol( pbuf, &endp, base );
      if ( *endp != 0 ) {
         if ( (seps == NULL) || (strchr(seps, *endp) == NULL) ) {
            set_errhdl( ERR_SEM_BADNUM2 );
            *result= 0;
            return( FALSE );
         }
      }
    }
    *result = value;
    return( TRUE );

} /* toINT */


/*+
 *      NAME : exec_debug
 *
 *  FUNCTION : main level semantics for the DEBUG directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a DEBUG directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_debug( VOID )
{

   // debug
   if ( strcmpi(P.word, "DEBUG") != 0 )
      return( FALSE );

   // test to display current state
   if ( P.get() ) {
      printf( "Debug state is %s\n", ((udebug) ? "ON" : "OFF") );
      return( TRUE );
   }

   // test to turn flag on
   if ( WC::streq(P.word, "ON")  ) {
      udebug = TRUE;
      return( TRUE );
   }

   // test to turn flag off
   if ( WC::streq(P.word, "OFF") ) {
      udebug = FALSE;
      return( TRUE );
   }

   // return was processed
   set_errhdl( ERR_CMD_UNKNOWN );
   return( TRUE );
    
} /* exec_debug */


/*+
 *      NAME : exec_quote
 *
 *  FUNCTION : main level semantics for the QUOTE directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a QUOTE directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_quote( VOID )
{

   // quote
   if ( strcmpi(P.word, "QUOTE") != 0 )
      return( FALSE );

   // test to display current state
   if ( P.get() ) {
      printf( "Quote state is %s\n", ((uquote) ? "ON" : "OFF") );
      return( TRUE );
   }

   // test to turn flag on
   if ( WC::streq(P.word, "ON")  ) {
      uquote = TRUE;
      return( TRUE );
   }

   // test to turn flag off
   if ( WC::streq(P.word, "OFF") ) {
      uquote = FALSE;
      return( TRUE );
   }

   // return was processed
   set_errhdl( ERR_CMD_UNKNOWN );
   return( TRUE );
    
} /* exec_echo */


/*+
 *      NAME : exec_echo
 *
 *  FUNCTION : main level semantics for the ECHO directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a ECHO directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_echo( BOOLEAN *echo_flag )
{

   // echo
   if ( strcmpi(P.word, "ECHO") != 0 )
      return( FALSE );

   // test to display current state
   if ( P.get() ) {
      printf( "Echo state is %s\n", ((*echo_flag) ? "ON" : "OFF") );
      return( TRUE );
   }

   // test to turn flag on
   if ( WC::streq(P.word, "ON")  ) {
      *echo_flag = TRUE;
      return( TRUE );
   }

   // test to turn flag off
   if ( WC::streq(P.word, "OFF") ) {
      *echo_flag = FALSE;
      return( TRUE );
   }

   // return was processed
   set_errhdl( ERR_CMD_UNKNOWN );
   return( TRUE );
    
} /* exec_echo */


/*+
 *      NAME : exec_file
 *
 *  FUNCTION : main level semantics for the FILE directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a FILE directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_file( VOID )
{
 STATIC CHAR         filename[64] = { 0, };
        CHAR        *mode;
        BOOLEAN      append;
        CHAR        *ptr;
        CHAR        *eptr;

   // print
   if ( strcmpi(P.word, "FILE") != 0 )
      return( FALSE );

   // get arg
   ptr = P.cmd_ptr();
   if ( P.get() ) {
      printf( "Output File: " );
      if ( filename[0] == 0 )
         printf( "screen" );
      else
         printf( "%s", filename );
      printf( "\n" );
      return( TRUE );
   }

   // test for close
   if ( strcmpi(P.word, "CLOSE") == 0 ) {
      if ( outfp != stdout )
         fclose( outfp );
      outfp       = stdout;
      filename[0] = 0;
      return( TRUE );
   }

   // check for opt
   append = FALSE;
   if ( P.word[0] == '-' ) {
      if ( P.get() )
         return( TRUE );
      if ( (P.word[0] == 'a') || (P.word[0] == 'A') )
         append = TRUE;
      ptr = P.cmd_ptr();
      if ( P.get() )
         return( TRUE );
   }

   // else open fname
   if ( (P.word[0] == '\'') || (P.word[0] == '\"') ) {
      if ( P.delimited() )
         return( TRUE );
      ptr = P.word;
   } else {
      while ( *ptr == ' ' )
         ptr++;
      eptr = ptr;
      while ( TRUE ) {
         if ( *eptr == 0 )
            break;
         if ( *eptr == ' ' ) {
            *eptr = 0;
            break;
         }
         eptr++;
      }
   }
   strncpy( filename, ptr, sizeof(filename) );

   // open it
   mode  = ( (append) ? "a" : "w" );
   outfp = fopen( filename, mode );
   if ( outfp == NULL ) {
      set_errhdl( ERR_INP_FNAME );
      outfp       = stdout;
      filename[0] = 0;
   }

   // done
   return( TRUE );

} /* exec_file */


/*+
 *      NAME : exec_print
 *
 *  FUNCTION : main level semantics for the PRINT directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a PRINT directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_print( VOID )
{
 #define pbuf  smallbuf
 INT_2         pbuf_len;
 INT_2         last_semi;
 FILE         *ofp;
 CHAR         *ptr;
 BOOLEAN       xlate;
 BOOLEAN       nl;

   // print
   if ( strcmpi(P.word, "PRINT") != 0 )
      return( FALSE );

   // setup
   ofp       = stdout;
   nl        = TRUE;
   xlate     = FALSE;
   *pbuf     = 0;
   pbuf_len  = 0;
   last_semi = -1;
   ptr       = P.cmd_ptr();

   // handle options
   while (TRUE) {
      if ( *ptr == ' ' ) {
         ptr++;
         continue;
      }
      if ( *ptr == '#' )
         goto x_done;
      if ( *ptr != '-' )
         break;
      while (TRUE) {
         ptr++;
         if        ( (*ptr == 'n') || (*ptr == 'N') ) {
            nl = FALSE;
         } else if ( (*ptr == 'e') || (*ptr == 'E') ) {
            xlate = TRUE;
         } else if ( (*ptr == 'o') || (*ptr == 'O') ) {
            ofp = outfp;
         } else {
            break;
         }
      }
   }

   // transfer string
   while (TRUE) {
      pbuf[pbuf_len] = *ptr;
      if ( (*ptr == 0) || (*ptr == '#') )
         break;
      if ( *ptr == ';' )
         last_semi = pbuf_len;
      if ( !xlate ) {
         ptr++;
         pbuf_len++;
         continue;
      }
      if ( *ptr == '\\' ) {
         ptr++;
         if        ( (*ptr == 'e') || (*ptr == 'E') ) {
            pbuf[pbuf_len] = 27;
         } else if ( (*ptr == 'n') || (*ptr == 'N') ) {
            pbuf[pbuf_len] = '\n';
         } else if ( (*ptr == 'r') || (*ptr == 'R') ) {
            pbuf[pbuf_len] = '\r';
         } else if ( (*ptr == 'f') || (*ptr == 'F') ) {
            pbuf[pbuf_len] = '\f';
         } else if ( (*ptr == 't') || (*ptr == 't') ) {
            pbuf[pbuf_len] = '\t';
         } else {
            pbuf[pbuf_len] = *ptr;
         }
      }
      ptr++;
      pbuf_len++;
   }
   pbuf[pbuf_len] = 0;

x_done:

   // clear last ';'
   if ( last_semi != -1 )
      pbuf[last_semi] = 0;

   fprintf( ofp, "%s", pbuf );
   if ( nl )
      fprintf( ofp, "\n" );

   // done
   return( TRUE );

 #undef pbuf
} /* exec_print */


/*+
 *      NAME : dump_sym
 *
 *  FUNCTION : lookup symtab and dumps information about symbol
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
STATIC VOID dump_sym( SYMBOL_TYP *usym, BOOLEAN hex, BOOLEAN local=FALSE )
{
 CHAR    *ptr;
 CHAR    *fmt;

   printf( "%-32.32s  ", usym->symbol );
   printf( "%-5.5s  "  , SYMBOL_TYP::typenames[usym->type] );
   printf( "%6d  "     , usym->value_len );
   switch( usym->type ) {
      case ST_TYPE_INT:
         fmt = (hex) ? "%x" : "%d";
         printf( fmt, (*(INT_2 *) usym->value) );
         break;
      case ST_TYPE_INT32:
         fmt = (hex) ? "%lx" : "%lx";
         printf( fmt, (*(INT_4 *) usym->value) );
         break;
      case ST_TYPE_CHAR:
         printf( "'%s'", (CHAR *) usym->value );
         break;
      case ST_TYPE_FUNCT:
         printf( "%d-Args", usym->args );
         if ( !local )
            break;
         ptr = (CHAR *) usym->value;
         printf( "\n     {\n" );
         while ( *ptr != 0 ) {
            printf( "     " );
            while ( *ptr != 0 ) {
               if ( (*ptr > 0) && (*ptr < ' ') )
                  printf( "$%d", ((INT_2) *ptr & 0x00FF) );
               else
                  printf( "%c", *ptr );
               ptr++;
            }
            printf( "\n" );
            ptr++;
         }
         printf( "     }" );
         return;
   }
   printf( "\n" );
   return;

} /* dump_sym */


/*+
 *      NAME : exec_show
 *
 *  FUNCTION : main level semantics for the SHOW directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a SHOW directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_show( VOID )
{
 INT_2         idx;
 SYMBOL_TYP   *sym;
 BOOLEAN       skip;
 BOOLEAN       hex;

   // echo
   if ( strcmpi(P.word, "SHOW") != 0 )
      return( FALSE );

   // test to see if any exists
   if ( symtab.table_len == 0 ) {
      printf( "\n" );
      printf( "No variables have been declared!\n" );
      printf( "\n" );
      return( TRUE );
   }

   // dump header
   printf( "\n" );
   printf( "%-32.32s  %-5.5s  %-6.6s  %s\n", "NAME", "TYPE", "LENGTH", "VALUE" );
   printf( "%-32.32s  %-5.5s  %-6.6s  %s\n", "----", "----", "------", "-----" );
   printf( "\n" );

   // if arg supplied, tell about it only
   hex  = FALSE;
   skip = FALSE;
   while ( !P.get() && (P.word[0] != '*') ) {
      if ( strcmpi("HEX", P.word) == 0 ) {
         hex = TRUE;
         continue;
      }
      if ( (P.word[0] == ';') || (P.word[0] == '#') )
         break;
      if ( (skip) && (P.word[0] == ',') )
         continue;
      skip = TRUE;
      sym  = symtab.find( P.word );
      if ( sym == NULL ) {
         printf( "%-32.32s  %-5.5s  %s\n", P.word, "NONE", "NOT DEFINED" );
         continue;
      }
      dump_sym( sym, TRUE );
      printf( "\n" );
   }
   if ( skip ) {
      printf( "\n" );
      return( TRUE );
   }

   // otherwise list all
   for( idx=0; (idx < symtab.table_len ); idx++ ) {
      sym = (*symtab.table)[idx];
      dump_sym( sym, hex );
   }
   printf( "\n" );

   // return was processed
   return( TRUE );

} /* exec_show */


/*+
 *      NAME : exec_defvar
 *
 *  FUNCTION : main level semantics for the DEFINE VARIABLE directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a DEFINE VARIABLE directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_defvar( VOID )
{
 INT_4          value;
 SYMBOL_TYP     sym;
 INT_2          type;
 BOOLEAN        found;
 BOOLEAN        braced;

   // determine type declaration
   found = FALSE;
   type  = ST_TYPE_NONE;
   if ( strcmpi(P.word, SYMBOL_TYP::typenames[ST_TYPE_INT32]) == 0 )
      type = ST_TYPE_INT32;
   if ( strcmpi(P.word, SYMBOL_TYP::typenames[ST_TYPE_INT]) == 0 )
      type = ST_TYPE_INT;
   if ( strcmpi(P.word, SYMBOL_TYP::typenames[ST_TYPE_CHAR]) == 0 )
      type = ST_TYPE_CHAR;
   // are we not defining a valid variable type
   if ( type == ST_TYPE_NONE )
      return( FALSE );

x_next:

   // get variable name from user
   if ( (P.get()) || (P.word[0] == ';') || (P.word[0] == '#') ) {
      if ( !found ) {
         set_errhdl( ERR_SEM_TYPE );
      }
      return( TRUE );
   }
   if ( (found) && (P.word[0] == ',') )
      goto x_next;
   found = TRUE;

   // if variable already exists, then a error
   if ( symtab.find(P.word) != NULL ) {
      set_errhdl( ERR_SEM_VARDEFINED );
      goto x_next;
   }

   // transfer data
   sym.type = type;
   if ( WC::strxfer(sym.symbol, P.word) != 0 )
      return( TRUE );
   if ( type == ST_TYPE_INT ) {
      sym.value_len  = sizeof(INT_2);
      sym.value_free = TRUE;
   }
   if ( type == ST_TYPE_INT32 ) {
      sym.value_len  = sizeof(INT_4);
      sym.value_free = TRUE;
   }
   if ( type == ST_TYPE_CHAR ) {
      sym.value_free = TRUE;
      sym.value_len  = DEFAULT_STRINGLEN;
      if ( !P.get() ) {
         braced = FALSE;
         if ( P.word[0] == '[' ) {
            if ( !eval_expr( &value, "]") ) {
               set_errhdl( ERR_SEM_CHAR );
               goto x_next;
            }
         } else {
            if ( !toINT(&value, P.word) ) {
               set_errhdl( ERR_SEM_CHAR );
               goto x_next;
            }
         }
         sym.value_len = (INT_2) value;
      }
   }

   // insert it into our table
   if ( sym.value_len != 0 ) {
      sym.value = ::malloc( sym.value_len );
      if ( sym.value == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         return( FALSE );
      }
      memset( sym.value, 0, sym.value_len );
      symtab.insert(&sym);
   }
   goto x_next;

} /* exec_defvar */


/*+
 *      NAME : get
 *
 *  FUNCTION : get an input from user and store in predefined var
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
GLOBAL BOOLEAN exec_get( VOID )
{
 INT_4          value;
 CHAR          *prompt;
 SYMBOL_TYP    *psym;

   // get
   if ( strcmpi(P.word, "GET") != 0 )
      return( FALSE );

   // obtain name of var to stuff with user value
   if ( P.get() ) {
      set_errhdl( ERR_P_NOTOKENS );
      return( TRUE );
   }
   psym = symtab.find(P.word);
   if ( (psym == NULL) || (psym->type == ST_TYPE_FUNCT) ) {
      set_errhdl( ERR_SEM_REF );
      return( TRUE );
   }

   // prompt
   prompt = NULL;
   if ( (!P.get()) && ((P.word[0] == '\'') || (P.word[0] == '\"')) ) {
      if ( P.delimited() ) {
         set_errhdl( ERR_P_NOTOKENS );
         return( TRUE );
      }
      prompt = P.word;
   }
   cgets( smallbuf, prompt );

   // stuff value
   switch( psym->type ) {
      case ST_TYPE_INT:
         if ( !toINT(&value, smallbuf) ) {
            set_errhdl( ERR_SEM_BADNUM1 );
            return( TRUE );
         }
         (*(INT_2 *) psym->value) = (INT_2) value;
         break;
      case ST_TYPE_INT32:
         if ( !toINT(&value, smallbuf) ) {
            set_errhdl( ERR_SEM_BADNUM1 );
            return( TRUE );
         }
         (*(INT_4 *) psym->value) = (INT_4) value;
         break;
      case ST_TYPE_CHAR:
         memset( psym->value, 0, psym->value_len );
         strncpy( (CHAR *) psym->value, smallbuf, psym->value_len-1 );
         break;
   }

   // done
   return( TRUE );

} /* exec_get */


/*+
 *      NAME : getline
 *
 *  FUNCTION : get an input line and prevent any interpretations
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
STATIC VOID getline( CHAR *prompt, BOOLEAN iseval=FALSE )
{

   while ( TRUE ) {
      if ( input.setup(prompt, iseval) )
         break;
   }
   return;

} /* getline */


/*+
 *      NAME : getword
 *
 *  FUNCTION : (scanner) gets a word continually across multiple lines
 *
 *     ENTRY : none
 *
 *   RETURNS : none
 *
-*/
STATIC VOID getword( CHAR *prompt )
{

   while ( (P.get()) || (P.word[0] == '#') ) {
      getline(prompt);
   }
   return;

} /* getword */


/*+
 *      NAME : exec_deffunct
 *
 *  FUNCTION : main level semantics for defining FUNCTIONS directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a define FUNCTION directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_deffunct( VOID )
{
 // output
 #define        obuf    hugebuf
 #define        tbuf    smallbuf
 INT_2          obuf_len;
 INT_2          tbuf_len;
 CHAR          *ptr;
 INT_2          widx;
 BOOLEAN        brace;
 INT_4          value;
 SYMBOL_TYP     sym;

   // test for keyword designator
   if ( strcmpi(P.word, SYMBOL_TYP::typenames[ST_TYPE_FUNCT]) != 0 )
      return( FALSE );

   // get funct name
   getword("FUNCTION> ");
   if ( symtab.find(P.word) != NULL ) {
      set_errhdl( ERR_SEM_VARDEFINED );
      goto x_done;
   }

   // setup
   sym.type = ST_TYPE_FUNCT;
   sym.args = 0;
   if ( WC::strxfer(sym.symbol, P.word) != 0 )
      goto x_done;

   // arglist
   getword("FUNCTION> ");
   if ( P.word[0] != '(' )
      goto x_err;
   getword("FUNCTION> ");
   if ( P.word[0] != ')' ) {
      if ( !toINT(&value, P.word) )
         goto x_err;
      sym.args = (INT_2) value;
      getword("FUNCTION> ");
      if ( P.word[0] != ')' )
         goto x_err;
   }

   // setup
   ptr  = P.cmd_ptr();
   widx = P.word_idx;

   // skip until opening brace
   while ( TRUE ) {
      if ( *ptr == '{' )
         break;
      if ( (*ptr == 0) || (*ptr == '#') || (*ptr == ';') ) {
         getline("FUNCTION> ");
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }

   // setup to get function code
   ptr++;
   widx++;
   brace    = 0;
   obuf_len = 0;

   // get function code
   while (TRUE) {
      obuf[obuf_len] = *ptr;
      if ( *ptr == '$' ) {
         if ( (ptr[1] >= '0') && (ptr[1] <= '9') ) {
            ptr++;
            widx++;
            tbuf_len = 0;
            while ((*ptr >= '0') && (*ptr <= '9')) {
               tbuf[tbuf_len++] = *ptr++;
               widx++;
            }
            tbuf[tbuf_len] = 0;
            if ( !toINT(&value, tbuf) ) {
               set_errhdl( ERR_SEM_REF );
               continue;
            }
            obuf[obuf_len] = (CHAR) value;
            obuf_len++;
            continue;
         }
      } else if ( *ptr == '{' ) {
         brace++;
      } else if ( *ptr == '}' ) {
         if ( brace == 0 )
            break;
         brace--;
      } else if ( *ptr == 0 ) {
         getline("FUNCTION> ");
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         if ( (obuf_len != 0) && (obuf[obuf_len-1] != 0) )
            obuf_len++;
         continue;
      }
      ptr++;
      widx++;
      obuf_len++;
   }
   obuf[obuf_len++] = 0;
   obuf[obuf_len++] = 0;

   // save function in symtab
   sym.value_free = TRUE;
   sym.value_len  = obuf_len;
   sym.value      = ::malloc( obuf_len );
   if ( sym.value == NULL ) {
      set_errhdl( ERR_SYS_NOMEM );
      goto x_done;
   }
   memcpy( (CHAR *) sym.value, obuf, obuf_len );
   symtab.insert( &sym );

   // done
   ptr = &P.cmd[widx+1];
   while (*ptr == ' ')
      ptr++;
   if ( (*ptr != 0) && (*ptr != ';') && (*ptr != '#') ) {
      P.word_idx = widx+1;
      input.bypass = TRUE;
   }
   goto x_done;

x_err:

   set_errhdl( ERR_SEM_BADFDECL );

x_done:

   // return was processed
   return( TRUE );

 #undef tbuf
 #undef obuf
} /* exec_deffunct */


/*+
 *      NAME : exec_call
 *
 *  FUNCTION : main level semantics for the CALL directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a CALL directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_call( VOID )
{
 #define tmpbuf  smallbuf
 #define obuf    hugebuf
 INT_2           tmpbuf_len;
 INT_2           obuf_idx;
 CHAR           *ibuf;
 INT_2           ibuf_idx;
 INT_2           oparen;
 INT_2           widx;
 CHAR           *ptr;
 CHAR           *token;
 SYMBOL_TYP     *sym        = NULL;
 INT_2           args_idx   = 0;
 INT_2           args_len   = 0;
 CHAR         *(*args)[]    = NULL;
 CHAR            uch        = 0;

   // call stmt
   if ( strcmpi(P.word, "CALL") != 0 )
      return( FALSE );

   // setup
   if ( P.get() ) {
      set_errhdl( ERR_SEM_BADFUNC );
      return( TRUE );
   }

   // find function name to invoke
   sym = symtab.find( P.word );
   if ( sym == NULL ) {
      set_errhdl( ERR_SEM_NOFUNC );
      return( TRUE );
   }
   args_len = sym->args;

   // setup arg holder
   if ( args_len != 0 ) {
      args = (CHAR *(*)[]) ::malloc( sizeof(CHAR *)*sym->args );
      if ( args == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         goto x_done;
      }
      memset( args, 0, sizeof(CHAR *)*sym->args );
   }

   // get opening to argument value list
   ptr   = P.cmd_ptr();
   widx  = P.word_idx;
   while (TRUE) {
      if ( *ptr == '(' )
         break;
      if ( (*ptr == 0) || (*ptr == ';') || (*ptr == '#') ) {
         getline("CALL> ", TRUE);
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }
   ptr++;
   token = ptr;
   if ( *token == 0 )
      token = NULL;

   // for each argument in list not as 0-args
   oparen = 0;
   while ( args != NULL ) {
      uch = *ptr;
      if ( uch == '(' )
         oparen++;
      if (
            (uch == ',') ||
            (uch == ')') ||
            (uch == 0  ) ||
            (uch == ';') ||
            (uch == '#') 
         ) {
         if ( args_idx >= args_len )
            goto x_badarg;
         if ( token != NULL ) {
            *ptr = 0;
            if ( WC::strxfer((*args)[args_idx], token) != 0 )
               return( TRUE );
            args_idx++;
            token = &ptr[1];
         }
         if ( uch == ')' ) {
            if ( oparen == 0 ) {
               *ptr = uch;
               break;
            }
            oparen--;
         }
         if ( (uch == 0) || (uch == ';') || (uch == '#') ) {
            getline( "CALL> ", TRUE );
            ptr   = 
            token = P.cmd_ptr();
            widx  = P.word_idx;
            continue;
         }
      }
      ptr++;
      widx++;
   }

   // get closing to argument value list for 0-args
   oparen = 0;
   while ( args == NULL ) {
      if ( *ptr == '(' )
         oparen++;
      if ( *ptr == ')' ) {
         if ( oparen == 0 ) {
            while (*token == ' ')
               token++;
            *ptr = *token;
            break;
         }
         oparen--;
      }
      if ( (*ptr == 0) || (*ptr == ';') || (*ptr == '#') ) {
         getline( "CALL> ", TRUE );
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }
   if ( (*ptr != ')') || (args_idx != args_len) )
      goto x_badarg;
   widx++;

   // get funct-code and perform arg replacements
   ibuf     = (CHAR *) sym->value;
   ibuf_idx = 0;
   obuf[0]  = 0;
   obuf_idx = 0;
   while (TRUE) {
      if ( ibuf_idx >= sym->value_len )
         break;
      if ( (*ibuf >= 1) && (*ibuf <= args_len) ) {
         ptr = (*args)[ (*ibuf-1) ];
         while (*ptr != 0 )
            obuf[obuf_idx++] = *ptr++;
         ibuf++;
      } else
         obuf[obuf_idx++] = *ibuf++;
      ibuf_idx++;
   }

   // finally push input stack & clean-up
   ptr = &P.cmd[widx+1];
   while ( (*ptr == ' ') || (*ptr == ';') )
      ptr++;
   if ( *ptr != 0 ) {
      strcpy( tmpbuf, ptr );
      tmpbuf_len           = strlen(ptr);
      tmpbuf[tmpbuf_len++] = 0;
      tmpbuf[tmpbuf_len++] = 0;
      input.push( tmpbuf, tmpbuf_len );
   }
   input.push( obuf, obuf_idx );
   goto x_done;

x_badarg:

    set_errhdl( ERR_SEM_BADARGS );

x_done:

    // free the args holder
    if ( args != NULL ) {
       for( args_idx=0; (args_idx < args_len); args_idx++ ) {
          if ( (*args)[args_idx] != NULL )
             ::free( (*args)[args_idx] );
       }
       ::free( args );
    }
    args = NULL;

    // return was processed
    return( TRUE );

 #undef obuf
 #undef tmpbuf
} /* exec_call */


/*+
 *      NAME : exec_while
 *
 *  FUNCTION : main level semantics for the WHILE construct
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a WHILE construct, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_while( VOID )
{
 #define     tmpbuf     smallbuf
 #define     condbuf    bigbuf
 #define     databuf    hugebuf
 INT_2       tmpbuf_len;
 INT_2       condbuf_len;
 INT_2       databuf_len;
 INT_2       widx;
 INT_2       brace;
 CHAR       *ptr;

   // while
   if ( strcmpi(P.word, "WHILE") != 0 )
      return( FALSE );

   // get the condition
   condbuf_len = 0;
   ptr         = P.cmd_ptr();
   widx        = P.word_idx;
   while (TRUE ) {
      condbuf[condbuf_len] = *ptr;
      if ( *ptr == '{' )
         break;
      if ( (*ptr == 0) || (*ptr == ';') || (*ptr == '#') ) {
         getline("WHILE> ");
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         condbuf[condbuf_len++] = ' ';
         continue;
      }
      condbuf_len++;
      ptr++;
   }
   condbuf[condbuf_len] = 0;

   // get datacode
   ptr++;
   widx++;
   while ( *ptr == ' ' ) {
      widx++;
      ptr++;
   }
   databuf_len = 0;
   brace      = 0;
   while (TRUE) {
      databuf[databuf_len] = *ptr;
      if        ( *ptr == '{' ) {
         brace++;
      } else if ( *ptr == '}' ) {
         if ( brace == 0 )
            break;
         brace--;
      } else if ( *ptr == 0 ) {
         getline("WHILE> ");
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         while ( *ptr == ' ' ) {
            ptr++;
            widx++;
         }
         if ( (databuf_len != 0) && (databuf[databuf_len-1] != 0) )
            databuf_len++;
         continue;
      }
      ptr++;
      widx++;
      databuf_len++;
   }
   databuf[databuf_len++] = 0;
   databuf[databuf_len++] = 0;

   // finally push input stack and clean-up
   ptr = &P.cmd[widx+1];
   while ( *ptr == ' ' )
      ptr++;
   if ( *ptr != 0 ) {
      strcpy( tmpbuf, ptr );
      tmpbuf_len           = strlen(ptr);
      tmpbuf[tmpbuf_len++] = 0;
      tmpbuf[tmpbuf_len++] = 0;
      input.push( tmpbuf, tmpbuf_len );
   }
   input.push( databuf, databuf_len, condbuf, condbuf_len );
   return( TRUE );

 #undef databuf
 #undef condbuf
 #undef tmpbuf
} /* exec_while */


/*+
 *      NAME : exec_for
 *
 *  FUNCTION : main level semantics for the FOR construct
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a FOR construct, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_for( VOID )
{
 #define    pbuf     hugebuf
 INT_2      pbuf_len;
 INT_2      widx;
 INT_2      brace;
 CHAR      *ptr;
 CHAR      *pdata       = NULL;
 CHAR      *pcond       = NULL;
 CHAR      *pinc        = NULL;
 CHAR      *pinit       = NULL;
 INT_2      pinit_len   = 0;
 INT_2      pdata_len   = 0;
 INT_2      pcond_len   = 0;
 INT_2      pinc_len    = 0;

   // for
   if ( strcmpi(P.word, "FOR") != 0 )
      return( FALSE );

   // get to the init-stmts  --past '('--
   ptr  = P.cmd_ptr();
   widx = P.word_idx;
   while (TRUE) {
      if ( *ptr == '(' )
         break;
      if ( (*ptr == 0) || (*ptr == '#') ) {
         getline( "FOR> " );
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }
   ptr++;
   widx++;

   // get init-stmts
   pbuf_len = 0;
   while (TRUE) {
      pbuf[pbuf_len] = *ptr;
      if ( *ptr == ';' )
         break;
      if ( *ptr == ',' )
         pbuf[pbuf_len] = 0;
      if ( (*ptr == 0) || (*ptr == '#') ) {
         getline("FOR> ");
         ptr            = P.cmd_ptr();
         widx           = P.word_idx;
         pbuf[pbuf_len] = ' ';
         pbuf_len++;
         continue;
      }
      ptr++;
      widx++;
      pbuf_len++;
   }
   pbuf[pbuf_len++] = 0;
   pbuf[pbuf_len++] = 0;

   // save
   pinit_len = pbuf_len;
   pinit     = (CHAR *) ::malloc( pinit_len );
   if ( pinit == NULL )
      goto x_errmem;
   memcpy( pinit, pbuf, pbuf_len );

   // get condition
   ptr++;
   widx++;
   pbuf_len = 0;
   while (TRUE) {
      pbuf[pbuf_len] = *ptr;
      if ( *ptr == ';' )
         break;
      if ( (*ptr == 0) || (*ptr == '#') ) {
         getline("FOR> ");
         ptr            = P.cmd_ptr();
         widx           = P.word_idx;
         pbuf[pbuf_len] = ' ';
         pbuf_len++;
         continue;
      }
      ptr++;
      widx++;
      pbuf_len++;
   }
   pbuf[pbuf_len] = 0;

   // save
   pcond_len = pbuf_len;
   pcond     = (CHAR *) ::malloc( pcond_len );
   if ( pcond == NULL )
      goto x_errmem;
   memcpy( pcond, pbuf, pbuf_len );

   // get inc-stmts
   brace = 0;
   ptr++;
   widx++;
   pbuf_len = 0;
   while (TRUE) {
      pbuf[pbuf_len] = *ptr;
      if ( *ptr == '(' )
         brace++;
      if ( *ptr == ')' ) {
         if ( brace == 0 )
            break;
         brace--;
      }
      if ( *ptr == ',' )
         pbuf[pbuf_len] = 0;
      if ( (*ptr == 0) || (*ptr == '#') ) {
         getline("FOR> ");
         ptr            = P.cmd_ptr();
         widx           = P.word_idx;
         pbuf[pbuf_len] = ' ';
         pbuf_len++;
         continue;
      }
      ptr++;
      widx++;
      pbuf_len++;
   }
   pbuf[pbuf_len++] = 0;
   pbuf[pbuf_len++] = 0;

   // save
   pinc_len = pbuf_len;
   pinc     = (CHAR *) ::malloc( pinc_len );
   if ( pinc == NULL )
      goto x_errmem;
   memcpy( pinc, pbuf, pbuf_len );

   // skip until brace
   while (TRUE) {
      if ( *ptr == '{' )
         break;
      if ( (*ptr == 0) || (*ptr == ';') || (*ptr == '#') ) {
         getline("FOR> ");
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }

   // get data/code-stmts
   ptr++;
   widx++;
   brace    = 0;
   pbuf_len = 0;
   while (TRUE) {
      pbuf[pbuf_len] = *ptr;
      if ( *ptr == '{' )
         brace++;
      if ( *ptr == '}' ) {
         if ( brace == 0 )
            break;
         brace--;
      }
      if ( *ptr == 0 ) {
         getline("FOR> ");
         ptr            = P.cmd_ptr();
         widx           = P.word_idx;
         if ( (pbuf_len != 0) && (pbuf[pbuf_len-1] != 0) )
            pbuf_len++;
         continue;
      }
      ptr++;
      widx++;
      pbuf_len++;
   }
   pbuf[pbuf_len++] = 0;
   pbuf[pbuf_len++] = 0;

   // save
   pdata_len = pbuf_len;
   pdata     = (CHAR *) ::malloc( pdata_len );
   if ( pdata == NULL )
      goto x_errmem;
   memcpy( pdata, pbuf, pbuf_len );

   // push inputs
   ptr = &P.cmd[widx+1];
   while (*ptr == ' ')
      ptr++;
   if ( *ptr != 0 ) {
      strcpy( smallbuf, ptr );
      widx             = strlen(ptr);
      smallbuf[widx++] = 0;
      smallbuf[widx++] = 0;
      input.push( smallbuf, widx );
   }
   input.push( pdata, pdata_len, pcond, pcond_len, pinc, pinc_len );
   input.push( pinit, pinit_len );

   // done ok
   goto x_done;

x_errmem:

   set_errhdl( ERR_SYS_NOMEM );

x_done:

   if ( pdata != NULL )
      ::free( pdata );
   if ( pcond != NULL )
      ::free( pcond );
   if ( pinc != NULL )
      ::free( pinc );
   if ( pinit != NULL )
      ::free( pinit );
   return( TRUE );

 #undef pbuf
} /* exec_for */


/*+
 *      NAME : exec_loopctl
 *
 *  FUNCTION : control level for loop constructs
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a LOOPCTL-related construct, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_loopctl( VOID )
{
 STREAM_TYP  *stream;
 INT_2        which;

   // find loop control keyword
   which = 0;
   if ( strcmpi(P.word, "BREAK") == 0 )
      which = 1;
   if ( (strcmpi(P.word, "CONTINUE") == 0) || (strcmpi(P.word, "CONT") == 0) )
      which = 2;
   if ( which == 0 )
      return( FALSE );

   // setup
   stream = input.stream_ptr();
   if ( stream == NULL )
      return( TRUE );

   // test for loop active
   if ( stream->cond == NULL ) {
      set_errhdl( ERR_P_CTL_LOOP );
      return( TRUE );
   }

   // break should kill entire loop control
   if ( which == 1 ) {
      input.pop();
      stream->elseskip = FALSE;
      return( TRUE );
   }

   // continue execution to next iteration round
   if ( which == 2 ) {
      which            = strlen(stream->data);
      stream->cptr     = &stream->data[which];
      stream->elseskip = FALSE;
   }

   // done
   return( TRUE );

} /* exec_loopctl */


/*+
 *      NAME : exec_if
 *
 *  FUNCTION : main level semantics for the IF construct
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a IF construct, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_if( VOID )
{
 #define condbuf    smallbuf
 INT_2              condbuf_len;
 INT_2              widx;
 INT_2              brace;
 CHAR              *ptr;
 INT_4              value;
 STREAM_TYP        *stream;

   // if stmt test
   if ( strcmpi(P.word, "IF") != 0 )
      return( FALSE );

   // get the condition
   condbuf_len = 0;
   widx        = P.word_idx;
   ptr         = P.cmd_ptr();
   while (TRUE) {
      condbuf[condbuf_len] = *ptr;
      if ( *ptr == '{' )
         break;
      if ( (*ptr == 0) || (*ptr == ';') || (*ptr == '#') ) {
         getline( "IF> ", TRUE );
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         condbuf[condbuf_len++] = ' ';
         continue;
      }
      condbuf_len++;
      ptr++;
      widx++;
   }
   condbuf[condbuf_len] = 0;

   if ( !eval_expr(&value,";{") )
      return( TRUE );

   // obtain current input stream info
   stream = input.stream_ptr();
   if ( stream == NULL )
      return( TRUE );

   // if if-cond is satisfied, then execute code
   if ( value != 0 )
      goto x_done;

   // otherwise if-cond is not satisfied, so skip code
   brace = 0;
   while ( TRUE ) {
      if        ( *ptr == '{' ) {
         brace++;
      } else if ( *ptr == '}' ) {
         brace--;
         if ( brace == 0 )
            break;
      } else if ( *ptr == 0 ) {
         getline("IF> ", TRUE);
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }

x_done:

   ptr = &P.cmd[widx+1];
   while (*ptr == ' ' )
      ptr++;
   if ( (*ptr != 0) && (*ptr != ';') && (*ptr != '#') ) {
      input.bypass = TRUE;
      P.word_idx   = widx+1;
   }
   return( TRUE );

 #undef condbuf
} /* exec_if */


/*+
 *      NAME : exec_ifctl
 *
 *  FUNCTION : control level processing for ifctl
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a IFCTL-related construct, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_ifctl( VOID )
{
 STREAM_TYP  *stream;
 BOOLEAN      elseskip;
 INT_2        widx;
 INT_2        brace;
 CHAR        *ptr;

   // stream should exist
   stream = input.stream_ptr(FALSE);
   if ( stream == NULL )
      return( FALSE );

   // reset else-skipper flag
   elseskip         = stream->elseskip;
   stream->elseskip = FALSE;

   // test to set skipper
   if ( P.word[0] == '}' ) {
      stream->elseskip = TRUE;
      input.bypass = TRUE;
      return( TRUE );
   }

   // test for ELSE or '} ELSE'
   if ( strcmpi(P.word, "ELSE") != 0 )
      return( FALSE );

   // test to execute else-code
   if ( !elseskip ) {
      while (TRUE) {
         if ( P.get() ){
            getline("ELSE> ", TRUE);
            continue;
         }
         if ( strcmpi(P.word, "IF") == 0 )
            return( exec_if() );
         if ( P.word[0] == '{' )
            break;
      }
      input.bypass = TRUE;
      return( TRUE );
   }

   // otherwise skip the else-code
   widx  = P.word_idx;
   brace = 0;
   ptr   = P.cmd_ptr();
   while (TRUE) {
      if        ( *ptr == '{' ) {
         brace++;
      } else if ( *ptr == '}' ) {
         brace--;
         if ( brace == 0 )
            break;
      } else if ( *ptr == 0 ) {
         getline("ELSE> ", TRUE);
         ptr  = P.cmd_ptr();
         widx = P.word_idx;
         continue;
      }
      ptr++;
      widx++;
   }
   ptr = &P.cmd[widx+1];
   while (*ptr == ' ' )
      ptr++;
   if ( (*ptr != 0) && (*ptr != ';') && (*ptr != '#') ) {
      input.bypass = TRUE;
      P.word_idx   = widx+1;
   }
   return( TRUE );

} /* exec_ifctl */


/*+
 *      NAME : exec_delete
 *
 *  FUNCTION : main level semantics for the DELETE directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a DELETE directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_delete( VOID )
{
 BOOLEAN     found;

   // echo
   if ( (strcmpi(P.word, "DELETE") != 0) && (strcmpi(P.word, "DEL") != 0) )
      return( FALSE );

   // loop thru all specified
   found = FALSE;
   while ( !P.get() ) {
      if ( (P.word[0] == ';') || (P.word[0] == '#') )
         break;
      if ( (found) && (P.word[0] == ',') )
         continue;
      found = TRUE;
      if ( !symtab.remove(P.word) ) {
         set_errhdl( ERR_SEM_DEL );
      }
   }

   // blank deletion line
   if ( !found ) {
      set_errhdl( ERR_SEM_DEL );
      return( TRUE );
   }

   // return was processed
   return( TRUE );

} /* exec_delete */


/*+
 *      NAME : exec_expr
 *
 *  FUNCTION : evaluates an expression
 *             NOTE: this is a bogus expression analyizer.  No precedence
 *             rules implemented.  The user must use parenthesis when
 *             precendences in an expression are required.
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was an EXPRESSION was determined, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_expr( VOID )
{
 CHAR          *valptr;
 INT_4          value;
 SYMBOL_TYP    *psym      = NULL;
 BOOLEAN        array_ref = FALSE;
 
   // determine if expr is requested; parse 'var ='
   psym = symtab.find( P.word );
   if ( psym == NULL )
      return( FALSE );
   valptr = (CHAR *) psym->value;

   // make sure token has valid type
   switch( psym->type ) {
      case ST_TYPE_INT32:
      case ST_TYPE_INT:
         if ( P.get() )
            goto x_err;
         break;
      case ST_TYPE_CHAR:
         if ( P.get() )
            goto x_err;
         if ( P.word[0] != '[' )
            break;
         if ( P.get() )
            goto x_err;
         if ( !toINT(&value, P.word) ) {
            set_errhdl( ERR_SEM_BADNUM3 );
            return( TRUE );
         }
         if ( P.get() )
            goto x_err;
         if (
              (P.word[0] != ']')          ||
              (value < 0)                 ||
              (value >= psym->value_len)
            ) {
            set_errhdl( ERR_SEM_ARRIDX );
            return( TRUE );
         }
         valptr   += value;
         array_ref = TRUE;
         if ( P.get() )
            goto x_err;
         break;
      default:
         set_errhdl( ERR_SEM_BADTOKEN );
         return( TRUE );
   }
   if ( P.word[0] != '=' )
      goto x_err;

   if ( (psym->type == ST_TYPE_CHAR) && (!array_ref) ) {
      if ( !eval_strexpr(valptr, psym->value_len) )
         goto x_done;
      goto x_done;
   }

   // evaluate the integer or 1-byte char expression
   if ( !eval_expr(&value) )
      goto x_done;
   // set the variable with expression results
   switch( psym->type ) {
      case ST_TYPE_INT:
         (*(INT_2 *) valptr) = (INT_2) value;
         break;
      case ST_TYPE_INT32:
         (*(INT_4 *) valptr) = (INT_4) value;
         break;
      case ST_TYPE_CHAR:
         *valptr = (CHAR) value;
         break;
   }
   goto x_done;

x_err:

   set_errhdl( ERR_SEM_BADLINE );

x_done:

   // return was processed
   return( TRUE );

} /* exec_expr */


/*+
 *      NAME : exec_shell
 *
 *  FUNCTION : execute a single system (shell command) directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a SHELL directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_shell( VOID )
{
 INT_2   l_ret;

   // shell
   if (
        (strcmpi(P.word, "SHELL") != 0) &&
        (strcmpi(P.word, "SH") != 0)    &&
        (P.word[0] != '!')
      )
      return( FALSE );

   // execute
   P.get( (PARSE::_ASIS|PARSE::_COUNT), 99 );
   if ( (l_ret=system(P.word)) != 0 )
      printf( "Error: %d\n", l_ret );

   // return was processed
   return( TRUE );

} /* exec_shell */


/*-------------------------------------------------------------------------*/
#if 0


/*+
 *      NAME : exec_deffunct
 *
 *  FUNCTION : main level semantics for defining FUNCTIONS directive
 *
 *     ENTRY : none
 *
 *   RETURNS : TRUE if it was a define FUNCTION directive, else FALSE
 *
-*/
GLOBAL BOOLEAN exec_deffunct( VOID )
{
 // output
 #define        obuf    hugebuf
 INT_2          oidx;
 // input
 CHAR          *ibuf;
 INT_2          iidx;
 INT_2          ilen;
 INT_2          itmp;
 // internal
 #define        tbuf    smallbuf
 INT_2          tbuf_len;
 INT_2          ibrace;
 INT_4          value;
 SYMBOL_TYP     sym;
 BOOLEAN        found;

   // test for keyword designator
   if ( strcmpi(P.word, SYMBOL_TYP::typenames[ST_TYPE_FUNCT]) != 0 )
      return( FALSE );

   // get funct name
   getword("FUNCTION> ");
   if ( symtab.find(P.word) != NULL ) {
      set_errhdl( ERR_SEM_VARDEFINED );
      goto x_done;
   }

   // setup
   sym.type = ST_TYPE_FUNCT;
   sym.args = 0;
   if ( WC::strxfer(sym.symbol, P.word) != 0 )
      goto x_done;

   // arglist
   getword("FUNCTION> ");
   if ( P.word[0] != '(' )
      goto x_err;
   getword("FUNCTION> ");
   if ( P.word[0] != ')' ) {
      if ( !toINT(&value, P.word) )
         goto x_err;
      sym.args = (INT_2) value;
      getword("FUNCTION> ");
      if ( P.word[0] != ')' )
         goto x_err;
   }
     
   // fdecl opening
   getword("FUNCTION> ");
   if ( P.word[0] != '{' )
      goto x_err;

   // get line
   ibuf   = &P.cmd[P.word_idx];
   ilen   = strlen( ibuf );
   iidx   = 0;

   // setup for loop
   obuf[0] = 0;
   oidx    = 0;
   ibrace  = 0;

   // fdecl body
   while ( TRUE ) {

      // skip starting blanks
      while ( ibuf[iidx] == ' ' )
         iidx++;

      // test for completion
      if ( (ibrace <= 0) && (ibuf[iidx] == '}') )
         break;

      // for every char in ibuf
      while (TRUE) {
         if ( (iidx >= ilen) || (ibuf[iidx] == 0) )
            break;
         if ( ibuf[iidx] == '$' ) {
            itmp  = iidx + 1;
            found = (BOOLEAN) ((ibuf[itmp] >= '0') && (ibuf[itmp] <= '9'));
            if ( found )  { // argument reference found
               itmp     = iidx++;
               tbuf[0]  = 0;
               tbuf_len = 0;
               while ( (ibuf[iidx] >= '0') && (ibuf[iidx] <= '9') ) {
                  tbuf[tbuf_len++] = ibuf[iidx++];
               }
               tbuf[tbuf_len] = 0;
               if ( toINT(&value, tbuf) ) {
                  if ( (value >= 1) && (value <= sym.args) ) {
                     obuf[oidx++] = (CHAR) value;
                     continue;
                  }
                  set_errhdl( ERR_SEM_REF );
               }
               iidx = itmp;
            }
         } else if ( ibuf[iidx] == '{' )
            ibrace++;
         else if ( ibuf[iidx] == '}' )
            ibrace--;
         obuf[oidx++] = ibuf[iidx++];
      }
      if ( iidx != 0 )
         obuf[oidx++] = 0;

      // get the next line
      getline("FUNCTION> ");
      ibuf = P.cmd;
      ilen = strlen( ibuf );
      iidx = 0;

   } /* end while */

   // obuf filled
   if ( oidx != 0 ) {
      obuf[oidx++]   = 0;
      sym.value_free = TRUE;
      sym.value_len  = oidx;
      sym.value      = ::malloc( oidx );
      if ( sym.value == NULL ) {
         set_errhdl( ERR_SYS_NOMEM );
         goto x_done;
      }
      memset( sym.value, 0, oidx );
      WC::memcpy( (CHAR *) sym.value, obuf, oidx );
      symtab.insert( &sym );
   }

   // clean up
   goto x_done;

x_err:

   set_errhdl( ERR_SEM_BADFDECL );

x_done:

   // return was processed
   return( TRUE );

 #undef tbuf
 #undef obuf
} /* exec_deffunct */


#endif
/*-------------------------------------------------------------------------*/



