/*   ext2fs.cc : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <time.h>

#include "std.h"
#include "diskio.h"
#include "ext2fs.h"

/*------------------------------------------------------------------------*/

GLOBAL BOOLEAN             g_verbose_ext2fs           = FALSE;

GLOBAL SUPERBLOCK_TYP     *sb                         = NULL;

GLOBAL INT_4               groups_len                 = 0;
GLOBAL GROUPDESC_TYP      *groups                     = NULL;

GLOBAL INT_4               inode_no                   = -1;
GLOBAL INODE_TYP           inode                      = { 0, };

GLOBAL DIR_TYP            *dir                        = NULL;

GLOBAL INT_2               badinode_len               = 0;
GLOBAL BADINODE_TYP        badinode[MAX_BADINODES]    = { { 0, },  };

STATIC INT_4               blocksize                  = 0;
STATIC INT_4               fragsize                   = 0;

STATIC INT_2   allocdir    ( DIR_TYP **udir );
STATIC VOID    freedir     ( DIR_TYP **udir );
STATIC VOID    removedir   ( DIR_TYP **udir );

/*------------------------------------------------------------------------*/


GLOBAL INT_2 init_FS( VOID )
{

   clean_FS();
   return( 0 );

} /* init_FS */


GLOBAL VOID clean_FS( VOID )
{

   if ( sb != NULL )
      ::free( sb );
   if ( groups != NULL )
      ::free( groups );
   sb         = NULL;
   groups     = NULL;
   groups_len = 0;
   return;

} /* clean_FS */


GLOBAL INT_2 loadsb_FS( INT_4 loc )
{
 INT_2   l_ret;

   if ( (g_disk == 0xFFFF) || (g_p_idx == -1) )
      return( set_errhdl(ERR_SYS_NOINIT) );
   sb = (SUPERBLOCK_TYP *) ::malloc( sizeof(SUPERBLOCK_TYP) );
   if ( sb == NULL )
      return( set_errhdl(ERR_SYS_NOMEM) );
   memset( sb, 0, sizeof(SUPERBLOCK_TYP) );
   l_ret = read_LIBIO(sb, sizeof(SUPERBLOCK_TYP), loc);
   if ( l_ret == 0 ) {
      blocksize = ( EXT2_BASE_BLOCK     << sb->s_log_block_size );
      fragsize  = ( EXT2_BASE_FRAGMENT  << sb->s_log_frag_size  );
      if ( sb->s_magic != EXT2_SUPER_MAGIC ) {
         blocksize = EXT2_BASE_BLOCK;
         fragsize  = EXT2_BASE_FRAGMENT;
         l_ret     = set_errhdl( ERR_IO_FS );
      }
   } else {
      free( sb );
      sb = NULL;
   }
   return( l_ret );

} /* loadsb_FS */


GLOBAL INT_2 loadgroups_FS( INT_4 len_fake, INT_4 loc )
{
 INT_4     size;
// INT_2     l_ret           = 0;

   /* validity check */
   if ( sb == NULL )
      return( set_errhdl(ERR_SYS_NOINIT) );
   if ( groups != NULL ) {
      ::free( groups );
      groups     = NULL;
      groups_len = 0;
   }

   /* calculate # of groups */
   if ( len_fake == -1 ) {
      groups_len = (sb->s_blocks_count - sb->s_first_data_block +
                  sb->s_blocks_per_group -1 ) / sb->s_blocks_per_group;
   } else {
      groups_len = len_fake;
   }
   size  = sizeof(GROUPDESC_TYP) * groups_len;

   /* allocate storage for groups */
   groups = (GROUPDESC_TYP *) ::malloc( (size_t) size );
   if ( groups == NULL )
      return( set_errhdl(ERR_SYS_NOMEM) );

   /* hard read */
   return( read_LIBIO(groups, size, loc) );

} /* loadgroups_FS */


GLOBAL INT_2 readinode_FS( INT_4 ino, INODE_TYP *i )
{
 INT_4      gidx;
 INT_4      gidx_rem;
 INT_4      loc;
 INT_2      j;
 INT_2      l_ret       = 0;

   // validity tests
   if ( (sb == NULL) || (groups == NULL) ) {
      l_ret = set_errhdl( ERR_SYS_NOINIT );
      goto x_done;
   }
   if ( (ino < 1) || (ino > (INT_4) sb->s_inodes_count) ) {
      l_ret = set_errhdl( ERR_SYS_BADINO );
      goto x_done;
   }

   // test for inode on badinode list
   for( j=0; (j < badinode_len); j++ ) {
      if ( badinode[j].inode_num == ino ) {
         memcpy( i, &badinode[j].inode, sizeof(INODE_TYP) );
         goto x_done;
      }
   }

   // determine index and validate
   ino--;
   gidx = (ino / sb->s_inodes_per_group);
   if ( (gidx < 0) || (gidx > groups_len) ) {
      l_ret = set_errhdl( ERR_SYS_INOLEN );
      goto x_done;
   }
   gidx_rem = (ino % sb->s_inodes_per_group);

   // get actual location
   loc   = groups[gidx].bg_inode_table * blocksize + (gidx_rem * 128);
   l_ret = read_LIBIO( i, sizeof(INODE_TYP), loc );

x_done:

   return( l_ret );

} /* readinode_FS */


GLOBAL INT_2 loadinode_FS( INT_4 ino )
{
 INT_2      l_ret       = 0;

   l_ret = readinode_FS( ino, &inode );
   if ( l_ret == 0 )
      inode_no = ino;
   return( l_ret );

} /* loadinode_FS */


STATIC INT_4     llength    = 0;
STATIC INT_4     totals     = 0;
STATIC INT_2     lidx       = 0;
STATIC INT_2     lbufidx[3] = { 0, 0, 0 };
STATIC CHAR     *lbuf[3]    = { NULL, NULL, NULL };


STATIC INT_2 nextblock( INT_4 *block, INT_2 idx )
{
 INT_4   *piblock    = NULL;
 INT_2    l_ret      = 0;

   // test for current data table completion
   if ( lbufidx[idx] >= (INT_2) (blocksize/sizeof(INT_4)) ) {
      if ( idx <= _IBLOCK(_SINGLE) )
         return( -1 );
      l_ret = nextblock( block, idx-1 );
      if ( (l_ret != 0) || (*block == 0) )
         return( l_ret );
      l_ret = read_LIBIO( lbuf[idx], blocksize, *block*blocksize );
      if ( l_ret != 0 )
         return( l_ret );
      lbufidx[idx] = 0;
   }

   // transfer value
   piblock = (INT_4 *) lbuf[idx];
   *block  = piblock[ lbufidx[idx]++ ];
   if ( *block == 0 ) {
      l_ret        = 0;
      lbufidx[idx] = -1;
   }

   // done
   return( l_ret );

} /* nextblock */



/*--------------------------------------------------*
 |                                                  |
 |  readblock_FS - called with NULL first to setup  |
 |  returns -1 when done                            |
 |                                                  |
 *--------------------------------------------------*/
GLOBAL INT_2 readblock_FS( INODE_TYP *i, INT_4 *block, INT_4 *length )
{
 INT_2     l_ret  = 0;
 INT_2     l_tmp;
 INT_4     l_blk;

   // validity & initialization tests
   if ( block == NULL ) {
      lidx = 0;
      for( l_tmp=0; (l_tmp < 3); l_tmp++ ) {
         lbufidx[l_tmp] = 0;
         if ( lbuf[l_tmp] != NULL )
            free( lbuf[l_tmp] );
         lbuf[l_tmp] = NULL;
      }
      totals = 0;
      if ( i != NULL )
         totals = i->i_size;
      return( l_ret );
   }
   if ( lidx == -1 ) {
      l_ret = set_errhdl( ERR_IO_BLKINIT );
      return( l_ret );
   }
   if ( length == NULL )
      length = &llength;

   // test for normal data return
   if ( lidx < EXT2_IBLOCK_NR ) {
      *block = i->i_block[lidx++];
      if ( *block == 0 ) {
         *length = 0;
         l_ret   =
         lidx    = -1;
      } else {
         *length = blocksize;
         if ( (totals - blocksize) <= 0 )
            *length = totals;
         totals -= *length;
      }
      return( l_ret );
   }

   // test for single index data return
   if ( lidx == EXT2_IBLOCK_SINGLE ) {
      l_tmp = _IBLOCK(_SINGLE);
      if ( lbuf[l_tmp] == NULL ) {
         lbuf[l_tmp] = (CHAR *) ::malloc( blocksize );
         if ( lbuf[l_tmp] == NULL )
            return( set_errhdl(ERR_SYS_NOMEM) );
         lbufidx[l_tmp] = 0;
         l_ret          = read_LIBIO( lbuf[l_tmp],
                                      blocksize,
                                      i->i_block[lidx] * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
      }
      l_ret = nextblock( block, l_tmp );
      if ( l_ret == 0 ) {
         if ( *block == 0 ) {
            *length = 0;
            l_ret   =
            lidx    = -1;
         } else {
            *length = blocksize;
            if ( (totals - blocksize) <= 0 )
               *length = totals;
            totals -= *length;
         }
         return( l_ret );
      } else {
         if ( l_ret != -1 )
            return( l_ret );
         lidx++;
      }
   }

   // test for double index data return
   if ( lidx == EXT2_IBLOCK_DOUBLE ) {
      l_tmp = _IBLOCK(_DOUBLE);
      if ( lbuf[l_tmp] == NULL ) {
         lbuf[l_tmp] = (CHAR *) ::malloc( blocksize );
         if ( lbuf[l_tmp] == NULL )
            return( set_errhdl(ERR_SYS_NOMEM) );
         lbufidx[l_tmp-1] = 1;
         l_ret            = read_LIBIO( lbuf[l_tmp-1],
                                        blocksize,
                                        i->i_block[lidx] * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
         l_blk          = *((INT_4 *) lbuf[l_tmp-1]);
         lbufidx[l_tmp] = 0;
         l_ret          = read_LIBIO( lbuf[l_tmp],
                                      blocksize,
                                      l_blk * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
      }
      l_ret = nextblock( block, l_tmp );
      if ( l_ret == 0 ) {
         if ( *block == 0 ) {
            *length = 0;
            l_ret   =
            lidx    = -1;
         } else {
            *length = blocksize;
            if ( (totals - blocksize) <= 0 )
               *length = totals;
            totals -= *length;
         }
         return( l_ret );
      } else {
         if ( l_ret != -1 )
            return( l_ret );
         lidx++;
      }
   }

   // test for triple index data return
   if ( lidx == EXT2_IBLOCK_TRIPLE ) {
      l_tmp = _IBLOCK(_TRIPLE);
      if ( lbuf[l_tmp] == NULL ) {
         lbuf[l_tmp] = (CHAR *) ::malloc( blocksize );
         if ( lbuf[l_tmp] == NULL )
            return( set_errhdl(ERR_SYS_NOMEM) );
         lbufidx[l_tmp-2] = 1;
         l_ret            = read_LIBIO( lbuf[l_tmp-2],
                                        blocksize,
                                        i->i_block[lidx] * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
         l_blk            = *((INT_4 *) lbuf[l_tmp-2]);
         lbufidx[l_tmp-1] = 1;
         l_ret            = read_LIBIO( lbuf[l_tmp-1],
                                        blocksize,
                                        l_blk * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
         l_blk          = *((INT_4 *) lbuf[l_tmp-1]);
         lbufidx[l_tmp] = 0;
         l_ret            = read_LIBIO( lbuf[l_tmp],
                                        blocksize,
                                        l_blk * blocksize );
         if ( l_ret != 0 )
            return( l_ret );
      }
      l_ret = nextblock( block, l_tmp );
      if ( l_ret == 0 ) {
         if ( *block == 0 ) {
            *length = 0;
            l_ret   =
            lidx    = -1;
         } else {
            *length = blocksize;
            if ( (totals - blocksize) <= 0 )
               *length = totals;
            totals -= *length;
         }
         return( l_ret );
      } else {
         if ( l_ret != -1 )
            return( l_ret );
         lidx++;
      }
   }

   *block  = 0;
   l_ret   = 
   lidx    = -1;
   *length = 0;
   return( l_ret );

} /* readblock_FS */


GLOBAL INT_2 getblock_FS( INT_4 *block, INT_4 *length )
{

   return( readblock_FS(&inode,block,length) );

} /* getblock_FS */


STATIC INT_2 allocdir( DIR_TYP **udir )
{

   (*udir) = (DIR_TYP *) ::malloc( sizeof(DIR_TYP) );
   if ( (*udir) == NULL )
      return( set_errhdl(ERR_SYS_NOMEM) );
   memset( (*udir), 0, sizeof(DIR_TYP) );
   return( 0 );

} /* allocdir */


STATIC VOID freedir( DIR_TYP **udir )
{

   if ( (*udir) != NULL ) {
      if ( (*udir)->name != NULL )
         free( (*udir)->name );
      free( (*udir) );
   }
   (*udir) = NULL;
   return;

} /* freedir */


STATIC VOID removedir( DIR_TYP **udir )
{
 DIR_TYP  *pdir;
 DIR_TYP  *ddir;

   pdir = (*udir);
   while ( pdir != NULL ) {
       ddir = pdir;
       pdir = pdir->next;
       freedir( &ddir );
   }
   (*udir) = NULL;
   return;

} /* removedir */


GLOBAL INT_2 readdir_FS( INODE_TYP *i, DIR_TYP **pdir )
{
 INT_2           l_ret        = 0;
 INT_4           block        = 0;
 INT_4           total        = 0;
 ENTRY_TYP      *pentry       = NULL;
 DIR_TYP        *ptdir        = NULL;
 CHAR           *lptr         = NULL;
 STATIC CHAR     lbuf[1024];           // no, should be variable

   if ( i == NULL )
      return( set_errhdl(ERR_SYS_BADINO) );
   if ( !_ISDIR(i->i_mode) )
      return( set_errhdl(ERR_IO_NOTDIR) );
   removedir( pdir );
   readblock_FS( NULL, NULL, NULL );
   while ( TRUE ) {

      l_ret = readblock_FS(i, &block, NULL );
      if ( l_ret != 0 ) {
         if (l_ret == -1)
            l_ret = 0;
         goto x_done;
      }

      l_ret = read_LIBIO( lbuf, blocksize, block*blocksize );
      if ( l_ret != 0 )
         goto x_done;

      total = 0;
      lptr  = lbuf;

      while ( total != blocksize ) {
         pentry = (ENTRY_TYP *) lptr;
         if (
             (pentry->name_len == 0)  ||
             (pentry->name[0] == 0)   ||
             (pentry->inode_num == 0)
            ) {
            total = blocksize;
            // should be able to short-circuit; but linux ? who knows
            continue;
         }
         if ( ptdir == NULL ) {
            l_ret = allocdir( &ptdir );
            if ( l_ret != 0 )
               goto x_done;
            (*pdir) = ptdir;
         } else {
            l_ret = allocdir( &ptdir->next );
            if ( l_ret != 0 )
               break;
            ptdir = ptdir->next;
         }
         ptdir->inode_num = pentry->inode_num;
         ptdir->name      = (CHAR *) ::malloc( pentry->name_len+1 );
         if ( ptdir->name == NULL ) {
            l_ret = set_errhdl( ERR_SYS_NOMEM );
            goto x_done;
         }
         strncpy( ptdir->name, pentry->name, pentry->name_len );
         ptdir->name[pentry->name_len] = 0;
         l_ret = readinode_FS( ptdir->inode_num, &ptdir->inode );
         if ( l_ret != 0 ) {
            // print error & continue reading
            printf( "Err-Read: inode %ld for entry %s\n",
                    ptdir->inode_num, ptdir->name );
         }
         total += pentry->rec_len;
         lptr   = &lptr[pentry->rec_len];

      } /* endwhile */
      
   } /* endwhile */

x_done:

   return( l_ret );

} /* readdir_FS */


GLOBAL INT_2 loaddir_FS( VOID )
{

   return( readdir_FS(&inode, &dir) );

} /* loaddir_FS */


/*------------------------------------------------------------------------*/


