/*   diskio.cc : February 1, 1994    */

/* Copyright (C) 1994-1999  Sekhar Bala, Rama Bala, and
 *                          Alphax System, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>

#ifdef GO32
   #include <dos.h>
#else
   #include <unistd.h>
   #include <stdlib.h>
   #include <fcntl.h>
#endif

#include "std.h"
#include "diskio.h"

/*------------------------------------------------------------------------*/

GLOBAL TAGDESC_TYP      ext2errs[]           = {
  { 0x0001,             "BIOS - invalid command"                     },
  { 0x0002,             "BIOS - address mark not found"              },
  { 0x0003,             "BIOS - disk write protected"                },
  { 0x0004,             "BIOS - sector not found"                    },
  { 0x0005,             "BIOS - reset failed"                        },
  { 0x0006,             "BIOS - floppy disk removed"                 },
  { 0x0007,             "BIOS - bad parameter table"                 },
  { 0x0008,             "BIOS - DMA overrun"                         },
  { 0x0009,             "BIOS - DMS DMS crossed 64K boundary"        },
  { 0x000A,             "BIOS - bad sector flag"                     },
  { 0x000B,             "BIOS - bad track flag"                      },
  { 0x000C,             "BIOS - media type not found"                },
  { 0x000D,             "BIOS - invalid # of sectors on format"      },
  { 0x000E,             "BIOS - control data address mark detected"  },
  { 0x000F,             "BIOS - DMA arbitration level out of range"  },
  { 0x0010,             "BIOS - uncorrectable CRC or ECC data error" },
  { 0x0011,             "BIOS - ECC corrected data error"            },
  { 0x0020,             "BIOS - controller failure"                  },
  { 0x0040,             "BIOS - seek failed"                         },
  { 0x0080,             "BIOS - disk timeout"                        },
  { 0x00AA,             "BIOS - drive not ready"                     },
  { 0x00BB,             "BIOS - undefined error"                     },
  { 0x00CC,             "BIOS - write fault"                         },
  { 0x00E0,             "BIOS - status register error"               },
  { 0x00FF,             "BIOS - status operation failed"             },
  { ERR_SYS_NOINIT,     "System not init'ed (use set disk,part)"     },
  { ERR_SYS_BADINO,     "Bad inode number encountered"               },
  { ERR_SYS_INOLEN,     "Inode number exceeds system specs"          },
  { ERR_SYS_FILE,       "File System Error"                          },
  { ERR_IO_FS,          "Disk does not contain an ext2 file system"  },
  { ERR_IO_PART,        "Disk partitioning is corrupted"             },
  { ERR_IO_EXTPART,     "Extended disk partitioning is corrupted"    },
  { ERR_IO_NOPART,      "Disk partition not selected"                },
  { ERR_IO_MAXPART,     "Disk contains too many extended partitions" },
  { ERR_IO_BLKINIT,     "Initialization required for getblock"       },
  { ERR_IO_NOTDIR,      "Inode not directory"                        },
  { ERR_IO_SLINK,       "Inode is a symbolic link"                   },
  { ERR_IO_BADWRITE,    "Invalid Write-Data Request"                 },
  { ERR_IO_DATA,        "Read/Write Data Request Failed"             },
  { ERR_IO_SEEK,        "Sector Seek Request Failed"                 },
  { ERR_IO_RDONLY,      "device opened in readonly mode"             },
  { 0,                  NULL                                         }
};

GLOBAL BOOLEAN          g_verbose_libio      = FALSE;
GLOBAL INT_2            g_retry              = MAX_RETRIES;

#if !defined(GO32)
GLOBAL BOOLEAN          g_readonly           = TRUE;
GLOBAL CHAR             g_devname[20]        = { "/dev/hda" };
GLOBAL int              g_dev                = -1;
#endif

GLOBAL INT_U2           g_disk               = 0xFFFF;
GLOBAL INT_U2           g_heads              = 0;
GLOBAL INT_U2           g_sectors            = 0;
GLOBAL INT_U2           g_cylinders          = 0;

GLOBAL INT_2            g_p_idx              = -1;
GLOBAL INT_2            g_p_len              = -1;
GLOBAL PARTITION_TYP    g_p[MAX_PARTITIONS]  = { { 0, }, };

#define PART_ACTIVE     0x80
#define PART_MAGIC      0xAA55

typedef struct partblock_typ {
   UCHAR          boot_ind;         /* 0x80 - active                         */
   UCHAR          head;             /* starting head                         */
   UCHAR          sector;           /* starting sector                       */
   UCHAR          cyl;              /* starting cylinder                     */
   UCHAR          sys_ind;          /* What partition type                   */
   UCHAR          end_head;         /* end head                              */
   UCHAR          end_sector;       /* end sector                            */
   UCHAR          end_cyl;          /* end cylinder                          */
   INT_U2         start_sectlo;     /* starting sector counting from 0       */
   INT_U2         start_secthi;     /* starting sector counting from 0       */
   INT_U2         nr_sectslo;       /* nr of sectors in partition            */
   INT_U2         nr_sectshi;       /* nr of sectors in partition            */
} PARTBLOCK_TYP;

typedef struct {
   UCHAR          fill[0x1BE];      /* reserverd bytes                       */
   PARTBLOCK_TYP  part[4];          /* 4 phycial partitions                  */
   INT_U2         magic;            /* magic part designation field          */
} BOOTBLOCK_TYP;

#define BIOS_READ             0x02
#define BIOS_WRITE            0x03
#define BIOS_PARMS            0x08

#define MAX_L_BUF_LEN         (4096*2)
#define MAX_L_BUF_SECTS       (MAX_L_BUF_LEN / 512)

STATIC UCHAR                  l_buf[MAX_L_BUF_LEN] = { 0, };

STATIC INT_2   bios_LIBIO      ( INT_2    cmd,
                                 INT_U4   sector,
                                 INT_U2   nsectors,
                                 VOID    *pbuf      );
STATIC INT_2   xdiskparms      ( VOID );
STATIC INT_2   xget_partition  ( VOID );

#ifdef GO32

STATIC INT_2   xbios           ( INT_U2      cmd,
                                 INT_U2      head,
                                 INT_U2      sector,
                                 INT_U2      cylinder,
                                 INT_U2      nsectors,
                                 VOID       *buf );

#else

#if 0
typedef struct {                    /* Boot Block Definition                 */
   UCHAR          jump[3];          /* 3 byte near jump to boot code         */
   CHAR           oem_name[8];      /* name of vendor & version              */
   INT_U2         sector_size;      /* size of sector in bytes               */
   UCHAR          cluster_size;     /* size of cluster in sectors (per/alu)  */
   INT_U2         starting_fat;     /* FAT start sector (#-rsved sectors)    */
   UCHAR          fat_cnt;          /* number of FAT's                       */
   INT_U2         root_entries;     /* number of root directory entries      */
   INT_U2         sector_cnt;       /* total sectors on volume               */
   UCHAR          media;            /* media descriptor byte                 */
   INT_U2         fat_size;         /* size of FAT in sectors (sects/fat)    */
   INT_U2         track_size;       /* track size in sectors (sects/track)   */
   INT_U2         heads;            /* number of heads                       */
   INT_U2         hidden;           /* number of hidden sectors              */
} RAWBOOTBLOCK_TYP;

STATIC RAWBOOTBLOCK_TYP           pboot = { { 0, }, };
#endif

#endif

/*------------------------------------------------------------------------*/


GLOBAL INT_2 init_LIBIO( INT_U2 disk )
{
 STATIC BOOLEAN      loaded = FALSE;
        INT_2        i;
        INT_2        l_ret;

   if ( g_disk != 0xFFFF )
      clean_LIBIO();
   if ( !loaded ) {
      for( i=0; (TRUE); i++ ) {
         if ( ext2errs[i].value == 0 )
            break;
         setusr_err( ext2errs[i].value, ext2errs[i].desc );
      }
      loaded = TRUE;
   }
   if ( disk == 0xFFFF )
      return( 0 );
   g_disk = disk;
   if ( (l_ret = xdiskparms()) == 0 )
      l_ret = xget_partition();
   return( l_ret );

} /* init_LIBIO */


GLOBAL VOID clean_LIBIO( VOID )
{
 INT_2   i;

   g_disk      = 0xFFFF;
#if !defined(GO32)
   if ( g_dev != -1 )
      close( g_dev );
   g_dev       = -1;
#endif
   g_heads     = 0;
   g_sectors   = 0;
   g_cylinders = 0;
   g_p_len     = -1;
   for( i=0; (i < MAX_PARTITIONS); i++ )
      memset( &g_p[i], 0, sizeof(PARTITION_TYP) );
   return;

} /* clean_LIBIO */


GLOBAL INT_2 readbios_LIBIO( INT_U4 sector, INT_U2 nsectors, VOID *pbuf )
{

   return( bios_LIBIO(BIOS_READ,sector,nsectors,pbuf) );

} /* readbios_LIBIO */


GLOBAL INT_2 writebios_LIBIO( INT_U4 sector, INT_U2 nsectors, VOID *pbuf )
{

#if !defined(GO32)
   if ( g_readonly )
      return(  set_errhdl(ERR_IO_RDONLY)  );
#endif
   return( bios_LIBIO(BIOS_WRITE,sector,nsectors,pbuf) );

} /* writebios_LIBIO */


GLOBAL INT_2 read_LIBIO( VOID *upbuf, INT_4 upbuf_len, INT_U4 loc )
{
 INT_U4     l_s;                 /* starting sector #                   */
 INT_U2     l_s_rem;             /* overflow bytes in starting sector # */
 INT_U2     l_n;                 /*                                     */
 INT_U2     l_n_rem;             /*                                     */
 INT_U2     l_n_xtra;            /*                                     */
 INT_U2     l_cnt;               /*                                     */
 INT_4      l_len;               /*                                     */
 INT_4      pbuf_len    = upbuf_len;
 CHAR      *pbuf        = (CHAR *) upbuf;
 INT_2      l_ret       = 0;

   /* validity test */
   if ( pbuf_len <= 0 )
      goto x_done;
   if ( (g_p_idx < 0) || (g_p[g_p_idx].type != PART_EXT2) ) {
      l_ret = set_errhdl( ERR_SYS_NOINIT );
      goto x_done;
   }

   /* calculate starting position */
   l_s     = ((INT_U4) ( loc / 512L )) + g_p[g_p_idx].start ;
   l_s_rem = (INT_U2) ( loc % 512L );

x_again:

   /* calculate # of sectors to transfer */
   l_n      = (INT_U2) ( pbuf_len / 512L );
   l_n_rem  = (INT_U2) ( pbuf_len % 512L );
   l_n_xtra = ( l_n_rem == 0 ) ? 0 : 1 ;

   l_cnt = l_n + l_n_xtra;
   l_len = pbuf_len;
   if ( l_cnt > MAX_L_BUF_SECTS ) {
      l_cnt = MAX_L_BUF_SECTS;
      l_len = MAX_L_BUF_LEN - l_s_rem;
   }

   /* test to output translation */
   if ( g_verbose_libio ) {
      printf( "loc=%lu --> (l_s, l_cnt, ...)=(%lu,%u)\n", loc, l_s, l_cnt );
   }

   /* read data from bios */
   l_ret = readbios_LIBIO( l_s, l_cnt, l_buf );
   if ( l_ret != 0 )
      goto x_done;

   memcpy( pbuf, &l_buf[l_s_rem], (size_t) l_len );
   pbuf      = &pbuf[l_len];
   pbuf_len -= l_len;
   l_s       += l_cnt;
   l_s_rem    = 0;
   if ( pbuf_len > 0 )
      goto x_again;

x_done:

   return( l_ret );

} /* read_LIBIO */


//
// Write can only be called with totally even boundaries
//
GLOBAL INT_2 write_LIBIO( VOID *upbuf, INT_4 upbuf_len, INT_U4 loc )
{
 INT_U4     l_s;                 /* starting sector #                   */
 INT_U2     l_s_rem;             /* overflow bytes in starting sector # */
 INT_U2     l_n;                 /*                                     */
 INT_U2     l_n_rem;             /*                                     */
 INT_4      pbuf_len    = upbuf_len;
 CHAR      *pbuf        = (CHAR *) upbuf;
 INT_2      l_ret       = 0;

#if !defined(GO32)
   if ( g_readonly )
      return(  set_errhdl(ERR_IO_RDONLY)  );
#endif

   /* validity test */
   if ( pbuf_len <= 0 )
      goto x_done;
   if ( (g_p_idx < 0) || (g_p[g_p_idx].type != PART_EXT2) ) {
      l_ret = set_errhdl( ERR_SYS_NOINIT );
      goto x_done;
   }

   /* calculate starting position */
   l_s     = ((INT_U4) ( loc / 512L )) + g_p[g_p_idx].start ;
   l_s_rem = (INT_U2) ( loc % 512L );
   if ( l_s_rem != 0 ) {
      l_ret = set_errhdl( ERR_IO_BADWRITE );
      goto x_done;
   }

   /* calculate # of sectors to transfer */
   l_n     = ((INT_U4) pbuf_len / 512L );
   l_n_rem = (INT_U2) ( pbuf_len % 512L );
   if ( (l_n_rem != 0) || (l_n > MAX_L_BUF_SECTS) ) {
      l_ret = set_errhdl( ERR_IO_BADWRITE );
      goto x_done;
   }

   /* test to output translation */
   if ( g_verbose_libio ) {
      printf( "loc=%lu --> (l_s, l_n, ...)=(%lu,%u)\n", loc, l_s, l_n );
   }

   /* write data using bios */
   l_ret = writebios_LIBIO( l_s, l_n, pbuf );

x_done:

   return( l_ret );

} /* write_LIBIO */


#ifdef GO32
STATIC INT_2 bios_LIBIO(
   INT_2    cmd,
   INT_U4   sector,
   INT_U2   nsectors,
   VOID    *pbuf        )
{
 INT_2      l_ret;
 INT_2      l_retry;
 INT_U2     l_head;
 INT_U2     l_sector;
 INT_U2     l_cylinder;

   /* translate logical sector-# to physical head/sector/cylinder */
   l_sector   = ( sector % (INT_U4) g_sectors ) + 1 ;
   l_cylinder = sector / ((INT_U4) g_sectors * (INT_U4) g_heads) ;
   l_head     = ( sector / (INT_U4) g_sectors ) % (INT_U4) g_heads ;

   /* show translations */
   if ( g_verbose_libio ) {
      printf( "cmd=%s head=%6u %04X sector=%6u"
              "%04x cylinder=%6u %04X count=%6u\n",
              ( (cmd == BIOS_READ) ? "READ" : "WRITE" ),
              l_head, l_head,
              l_sector, l_sector,
              l_cylinder, l_cylinder,
              nsectors );
      printf("\n");
   }

   /* retry with MAX_RETRIES */
   l_retry = 0;
   while (TRUE) {
      l_ret = xbios(cmd, l_head, l_sector, l_cylinder, nsectors, pbuf);
      if ( (l_ret == 0) || (l_ret != 6) )
         break;
      if ( ++l_retry >= g_retry )
         break;
   }

   /* done */
   if ( l_ret != 0 )
      set_errhdl( l_ret );
   return( l_ret );

} /* bios_LIBIO */
#else
STATIC INT_2 bios_LIBIO(
   INT_2    cmd,
   INT_U4   sector,
   INT_U2   nsectors,
   VOID    *pbuf        )
{
 INT_2       l_ret;
 INT_2       l_retry;
 long long   a_position;
 long long   l_position;
 INT_U4      a_length;
 INT_U4      l_length;

   /* setup */
   l_position = (long long) ( sector * 512 );
   l_length   = nsectors * 512;

   /* show translations */
   if ( g_verbose_libio ) {
      printf( "cmd=%s sector=%6lu %04lx count=%6u\n",
              ( (cmd == BIOS_READ) ? "READ" : "WRITE" ),
              sector, sector,
              nsectors );
      printf("\n");
   }

   /* retry with MAX_RETRIES */
   l_ret   = 0;
   l_retry = 0;
   while (TRUE) {
      a_position = llseek( g_dev, l_position, SEEK_SET );
      if ( a_position != l_position ) {
         l_ret = ERR_IO_SEEK;
      } else {
         if ( cmd == BIOS_READ )
            a_length = (INT_U4) read( g_dev, pbuf, l_length );
         else
            a_length = (INT_U4) write( g_dev, pbuf, l_length );
         if ( a_length != l_length ) {
            l_ret = ERR_IO_DATA;
         }
      }
      if ( ++l_retry >= g_retry )
         break;
   }

   /* done */
   if ( l_ret != 0 )
      set_errhdl( l_ret );
   return( l_ret );

} /* bios_LIBIO */
#endif


/*------------------------------------------------------------------------*/
#ifdef GO32


STATIC INT_2 xdiskparms( VOID )
{
 union REGS     iregs;
 union REGS     oregs;
// struct SREGS   sregs;

   /*-------------------------------------------------------------------*
    |  Modern machine bios's return incorrect disk parameters to allow  |
    |  access to larger hard disks (logical to physical translations).  |
    *-------------------------------------------------------------------*/

   /* get sector data using bios */
//   segread( &sregs );
   iregs.h.ah = (UCHAR) BIOS_PARMS;
   iregs.h.dl = (UCHAR) g_disk;
//   INT86X( 0x13, &iregs, &oregs, &sregs );
   int86( 0x13, &iregs, &oregs );
   if ( oregs.x.cflag != 0 )
      return( set_errhdl(__INT_2(oregs.h.ah)) );

   /* store values for future calculations */
   g_heads      = __INT_U2(oregs.h.dh) + 1;
   g_sectors    = __INT_U2(oregs.h.cl) & 0x003F;
   g_cylinders  = ((__INT_U2(oregs.h.cl) & 0x00C0) << 2);
   g_cylinders += (__INT_U2(oregs.h.ch));

   /* done */
   return( 0 );

} /* xdiskparms */


STATIC INT_2 xbios(
   INT_U2      cmd,
   INT_U2      head,
   INT_U2      sector,
   INT_U2      cylinder,
   INT_U2      nsectors,
   VOID       *pbuf )
{
 union REGS     iregs;
 union REGS     oregs;
 struct SREGS   sregs;
 UCHAR          upper;

//   segread( &sregs );
   iregs.h.ah = (UCHAR) cmd;
   iregs.h.al = (UCHAR) nsectors;
   iregs.h.ch = (UCHAR) cylinder & 0xFF;
   upper      = ((UCHAR) (cylinder >> 2)) & 0xC0;
   iregs.h.cl = (UCHAR) sector | upper;
   iregs.h.dh = (UCHAR) head;
   iregs.h.dl = (UCHAR) g_disk;
   iregs.x.bx = (INT_U2) pbuf;
   sregs.es   = (INT_U2) ((INT_U4) pbuf >> 16);
   INT86X( 0x13, &iregs, &oregs, &sregs );
   if ( oregs.x.cflag != 0 )
      return( __INT_2(oregs.h.ah) );
   return( 0 );

} /* xbios */


#else


STATIC INT_2 xdiskparms( VOID )
{
 INT_2             disk;
 INT_2             dev_len;
 CHAR              device[20];

   // setup the device name
   disk = g_disk;
   strcpy( device, "/dev/fd0" );
   if ( disk & 0x80 ) {
      disk &= 0x07;
      strcpy( device, g_devname );
   }
   dev_len            = strlen( device );
   device[dev_len-1] += (char) disk;

   // open the device
   if ( g_readonly ) 
      g_dev = open( device, O_RDONLY );
   else
      g_dev = open( device, O_RDWR );
   if ( g_dev < 0 ) {
      set_errhdl( ERR_SYS_PERM );
      return( ERR_SYS_PERM );
   }

#if 0
   /* read boot sector */
   disk = readbios_LIBIO( 0, 1, &l_buf );
   if ( disk != 0 )
      return( disk );

   // special copy
   memcpy( &pboot.jump,         &l_buf[0],   3 );     /* UCHAR [3] */
   memcpy( &pboot.oem_name,     &l_buf[3],   8 );     /* CHAR  [8] */
   memcpy( &pboot.sector_size,  &l_buf[11],  2 );     /* INT_U2    */
   memcpy( &pboot.cluster_size, &l_buf[13],  1 );     /* UCHAR     */
   memcpy( &pboot.starting_fat, &l_buf[14],  2 );     /* INT_U2    */
   memcpy( &pboot.fat_cnt,      &l_buf[16],  1 );     /* UCHAR     */
   memcpy( &pboot.root_entries, &l_buf[17],  2 );     /* INT_U2    */
   memcpy( &pboot.sector_cnt,   &l_buf[19],  2 );     /* INT_U2    */
   memcpy( &pboot.media,        &l_buf[21],  1 );     /* UCHAR     */
   memcpy( &pboot.fat_size,     &l_buf[22],  2 );     /* INT_U2    */
   memcpy( &pboot.track_size,   &l_buf[24],  2 );     /* INT_U2    */
   memcpy( &pboot.heads,        &l_buf[26],  2 );     /* INT_U2    */
   memcpy( &pboot.hidden,       &l_buf[28],  2 );     /* INT_U2    */

   if ( g_verbose_libio ) {
      printf( "        jump = x%02x x%02x x%02x\n",
              pboot.jump[0], pboot.jump[1], pboot.jump[2] );
      printf( "    oem_name = -%8.8s\n",  pboot.oem_name );
      printf( " sector_size = %u\n",      pboot.sector_size );
      printf( "clustor_size = %d\n",      (INT_2) pboot.cluster_size&0x00FF );
      printf( "starting_fat = %d\n",      pboot.starting_fat );
      printf( "     fat_cnt = %d\n",      (INT_2) pboot.fat_cnt&0x00FF );
      printf( "root_entries = %d\n",      pboot.root_entries );
      printf( "  sector_cnt = %d\n",      pboot.sector_cnt );
      printf( "       media = x%02X\n",   (INT_2) pboot.media&0x00FF );
      printf( "    fat_size = %d\n",      pboot.fat_size );
      printf( "  track_size = %d\n",      pboot.track_size );
      printf( "       heads = %d\n",      pboot.heads );
      printf( "      hidden = %d\n",      pboot.hidden );
   }

   // transfer
   g_heads     = pboot.heads;
   g_sectors   = pboot.sector_cnt;
   g_cylinders = 0;
#endif

   // done
   return( 0 );

} /* xdiskparms */


#endif
/*------------------------------------------------------------------------*/


STATIC INT_2 xget_partition( VOID )
{
 INT_2             i;
 INT_2             j;
 INT_U2            tmp1, tmp2;
 INT_2             l_ret         = 0;
 BOOTBLOCK_TYP    *pboot         = (BOOTBLOCK_TYP *) &l_buf[0];

   /* only hard disks have partitioning */
   if ( (g_disk & 0x80) == 0 ) {
      g_p_len       = 1;
      g_p_idx       = 0;
      g_p[0].length = 1440;         /* assume 1.44 floppy drive */
      g_p[0].type   = PART_EXT2;
      goto x_done;
   }

   /* read boot sector */
   l_ret  = readbios_LIBIO( 0, 1, &l_buf );
   if ( l_ret != 0 )
      goto x_done;

   /* validity test */
   if ( pboot->magic != PART_MAGIC ) {
      set_errhdl( l_ret = ERR_IO_PART );
      goto x_done;
   }

   /* load base partition information */
   for( i=0; (i < 4); i++ ) {
      tmp1          = pboot->part[i].start_secthi;
      tmp2          = pboot->part[i].start_sectlo;
      g_p[i].start  = __INT_U4( tmp1, tmp2 );
      tmp1          = pboot->part[i].nr_sectshi;
      tmp2          = pboot->part[i].nr_sectslo;
      g_p[i].length = __INT_U4( tmp1, tmp2 );
      g_p[i].type   = __INT_U2(pboot->part[i].sys_ind);
      tmp1          = __INT_U2(pboot->part[i].boot_ind);
      g_p[i].flags  = (tmp1 == 0x0080) ? PARTF_ACTIVE : 0 ;
   }
   g_p_len = 4;

   /* load extended partion information */
   for( i=0; (i < MAX_PARTITIONS); i++ ) {

      if ( g_p[i].type != PART_EXTENDED )
         continue;

      l_ret = readbios_LIBIO( g_p[i].start, 1, &l_buf );
      if ( l_ret != 0 )
         goto x_done;
      if ( pboot->magic != PART_MAGIC ) {
         set_errhdl( l_ret = ERR_IO_EXTPART );
         goto x_done;
      }

      for( j=0; (j < 4); j++ ) {
         if ( g_p_len == MAX_PARTITIONS ) {
            set_errhdl( l_ret = ERR_IO_MAXPART );
            goto x_done;
         }
         tmp1                = pboot->part[j].start_secthi;
         tmp2                = pboot->part[j].start_sectlo;
         g_p[g_p_len].start  = __INT_U4( tmp1, tmp2 ) + g_p[i].start;
         tmp1                = pboot->part[j].nr_sectshi;
         tmp2                = pboot->part[j].nr_sectslo;
         g_p[g_p_len].length = __INT_U4( tmp1, tmp2 );
         g_p[g_p_len].type   = __INT_U2(pboot->part[j].sys_ind);
         tmp1                = __INT_U2(pboot->part[j].boot_ind);
         g_p[g_p_len].flags  = (tmp1 == 0x0080) ? PARTF_ACTIVE : 0 ;
         g_p_len++;
      }

   } /* end for */

x_done:

   return( l_ret );

} /* xget_partition */


/*------------------------------------------------------------------------*/


