/* inode.c
 * 
 * Inode and Superblock handling for the EFS filesystem
 *
 * (C) 1995,96 Christian Vogelgsang
 */
 
#include <linux/module.h>
#include "efs.h"

/* ----- Define the operations for the Superblock ----- */
void efs_read_inode(struct inode *);
void efs_put_super(struct super_block *);
void efs_statfs(struct super_block *, struct statfs *,int );

static struct super_operations efs_sops = {
  efs_read_inode, 
  NULL, /* notify_change */
  NULL, /* write_inode */
  NULL, /* put_inode */
  efs_put_super, 
  NULL, /* write_super */
  efs_statfs, 
  NULL
};


/* ----- Conversion utilities ----- 
   Get 16- and 32-bit unsigned values in big-endian format from a byte buffer
*/
LONG ConvertLong(BYTE *buf, int offset)
{
  return  (LONG)buf[offset+3] |
    ((LONG)buf[offset+2])<<8 |
      ((LONG)buf[offset+1])<<16 |
	((LONG)buf[offset])<<24;
}

SHORT ConvertShort(BYTE *buf, int offset)
{
  return (SHORT)buf[offset+1] |
    ((SHORT)buf[offset])<<8;
}


/* ----- Install/Remove Module ----- 
 These procedures are used to install/remove our filesystem
 module in the linux kernel
*/

/* describe filesystem */
struct super_block *efs_read_super(struct super_block *s, void *d, int sil);
static struct file_system_type efs_fs_type = {
  efs_read_super, "efs", 1,  NULL 
};

/* install module */
int init_module(void)
{
  int status;

  if((status=register_filesystem(&efs_fs_type))==0)
    register_symtab(0);
  return status;
}

/* remove module */
void cleanup_module(void)
{
  unregister_filesystem(&efs_fs_type);
}


/* ----- efs_checkVolDesc -----
   Analyse the firts block of a CD and check 
   if it's a valid efs volume descriptor

   blk    - buffer with the data of first block
   silent - 0 -> verbose

   return : 0 - error ; >0 - start block of filesystem
*/
static LONG efs_checkVolDesc(BYTE *blk,int silent)
{
  LONG magic;
  BYTE *ptr;
  BYTE name[10];
  LONG pos,len;
  int i;

  /* is the magic cookie here? */
  magic = ConvertLong(blk,0);
  if(magic!=0x0be5a941) {
    printk("EFS: no magic on first block\n");
    return 0;
  }
  
  /* Walk through the entries of the VD */
  /* Quite useless, but gives nice output ;-) */
  ptr = blk + EFS_VD_ENTRYFIRST;
  name[8] = 0;
  while(*ptr) {
    for(i=0;i<8;i++)
      name[i] = ptr[i];

    /* start and length of entry */
    pos = ConvertLong(ptr,EFS_VD_ENTRYPOS);
    len = ConvertLong(ptr,EFS_VD_ENTRYLEN);

    if(!silent) 
      printk("EFS: VolDesc: %8s blk: %08lx len: %08lx\n",
	     name,pos,len);
    
    ptr+=EFS_VD_ENTRYSIZE;
  }

  return ConvertLong(blk,EFS_VD_FS_START);
}


/* ----- efs_checkSuper ----- 
 Check if the given block is a valid EFS-superblock
 
 sbi    - my EFS superblock info
 block  - block that must be examined

 return - 0 ok, -1 error
*/
static int efs_checkSuper(struct efs_sb_info *sbi, BYTE *block,int silent)
{
  LONG    magic;
    
  /* check if the magic cookie is here */
  magic = ConvertLong(block, EFS_SB_MAGIC);
  if((magic!=EFS_MAGIC1)&&(magic!=EFS_MAGIC2))
    return -1;
  
  sbi->total_blocks = ConvertLong(block, EFS_SB_TOTAL);
  sbi->first_block  = ConvertLong(block, EFS_SB_FIRST);
  sbi->group_size   = ConvertLong(block, EFS_SB_GROUP);
  sbi->inode_blocks = ConvertShort(block, EFS_SB_INODE);
  sbi->total_groups = ConvertShort(block, EFS_SB_TOGRP);
    
  return 0;    
}


/* ----- efs_read_super ----- 
   read_super: if the fs gets mounted this procedure is called to
   check if the filesystem is valid and to fill the superblock struct
   
   s     - superblock struct
   d     - options for fs (unused)
   sil   - flag to be silent
   
   return - filled s struct or NULL on error
 */

struct super_block *efs_read_super(struct super_block *s, void *d, int silent)
{
  struct buffer_head *bh;
  struct efs_sb_info *sb_info = (struct efs_sb_info *)&s->u.generic_sbp;
  int error = 0;
  int dev = s->s_dev;

  MOD_INC_USE_COUNT;
  
  /* say hello to my log file! */
  if(!silent)
    printk("EFS: --- Filesystem ---\n");

  /* set blocksize to 512 */
  set_blocksize(dev, EFS_BLOCK_SIZE);
  
  lock_super(s);
  
  /* Read first block of CD: the Volume Descriptor */
  bh = bread(dev, 0, EFS_BLOCK_SIZE);
  if(bh) {
    sb_info->fs_start = efs_checkVolDesc((BYTE *)bh->b_data,silent);
    if(sb_info->fs_start==0) {
      printk("EFS: failed checking Volume Descriptor\n");
      error++;
    }
    brelse(bh);
  } else {
    printk("EFS: cannot read the first block\n");
    error++;
  }

  /* Read the Superblock */
  if(!error) {	
    bh = bread(dev, sb_info->fs_start + EFS_BLK_SUPER,  EFS_BLOCK_SIZE );
    if(bh) {
      if(efs_checkSuper(sb_info, (BYTE *)bh->b_data,silent)) {
	printk("EFS: failed checking Superblock\n");
	error++;
      }
      brelse(bh);
    } else {
      printk("EFS: cannot read the superblock\n");
      error++;
    }
  } 
  
  if(!error) {
    s->s_blocksize = EFS_BLOCK_SIZE;
    s->s_blocksize_bits = EFS_BLOCK_SIZE_BITS;
    
    s->s_magic	= EFS_SUPER_MAGIC;
    s->s_flags	= MS_RDONLY;
    s->s_op	= &efs_sops;
    s->s_mounted= iget(s, EFS_ROOT_INODE);
    s->s_dev	= dev;
  }
  unlock_super(s);
  
  if(!(s->s_mounted)) {
    printk("EFS: Not mounted!\n");
    error++;
  }
  if(check_disk_change(s->s_dev)) {
    printk("EFS: Device changed!\n");
    error++;
  }

  /* We found errors -> say goodbye! */
  if(error) {
    s->s_dev = 0;
    printk("EFS: init failed with %d errors\n", error);
    MOD_DEC_USE_COUNT;
    return NULL;
  } 

  return s;
}


/* ----- efs_put_super ----- 
   put_super: remove the filesystem and the module use count

   s - superblock
*/
void efs_put_super(struct super_block *s)
{
  lock_super(s);
  s->s_dev = 0;
  unlock_super(s);
  MOD_DEC_USE_COUNT;
}


/* ----- efs_statfs ----- 
   statfs: get informatio on the filesystem
   
   s   - superblock of fs
   buf - statfs struct that has to be filled
*/
void efs_statfs(struct super_block *s, struct statfs *buf,int bufsize)
{
  struct efs_sb_info *sbi = (struct efs_sb_info *)&s->u.generic_sbp;
  struct statfs tmp;

  tmp.f_type = EFS_SUPER_MAGIC;
  tmp.f_bsize = EFS_BLOCK_SIZE;
  tmp.f_blocks = sbi->total_blocks;
  tmp.f_bfree = 0;
  tmp.f_bavail = 0;
  tmp.f_files = 100; /* don't know how to calculate the correct value */
  tmp.f_ffree = 0;
  tmp.f_namelen = NAME_MAX;

  memcpy_tofs(buf,&tmp,bufsize);
}


/* ----- efs_read_inode ----- 
   read an inode specified by in->i_ino from disk, fill the inode
   structure and install the correct handler for the file type
   
   in   - inode struct
*/
void efs_read_inode(struct inode *in)
{
  struct buffer_head *bh;
  struct efs_sb_info *sbi = (struct efs_sb_info *)&in->i_sb->u.generic_sbp;
  LONG blk,off;
  int error = 0;
  
  /* Calc the discblock and the offset for inode (4 Nodes fit in one block) */
  blk = in->i_ino >> 2; 
  blk = sbi->fs_start + sbi->first_block + 
    (sbi->group_size * (blk / sbi->inode_blocks)) +
      (blk % sbi->inode_blocks);
  off = (in->i_ino&3)<<7;
  
  /* Read the block with the inode from disk */
  bh = bread(in->i_dev,blk,EFS_BLOCK_SIZE);
  if(bh) {
    BYTE *rawnode = (BYTE *)(bh->b_data + off);
    SHORT numext;
    struct efs_in_info *ini = (struct efs_in_info *)&in->u.generic_ip;
    LONG rdev;
    BYTE *ptr;
    int i;
    
    /* fill in standard inode infos */
    in->i_mtime = ConvertLong(rawnode,EFS_IN_MTIME);
    in->i_ctime = ConvertLong(rawnode,EFS_IN_CTIME);
    in->i_atime = ConvertLong(rawnode,EFS_IN_ATIME);
    in->i_size  = ConvertLong(rawnode,EFS_IN_SIZE);
    in->i_nlink = ConvertShort(rawnode,EFS_IN_LINKS);
    in->i_uid   = ConvertShort(rawnode,EFS_IN_UID);
    in->i_gid   = ConvertShort(rawnode,EFS_IN_GID);
    
    in->i_mode = ConvertShort(rawnode,EFS_IN_MODE);

    /* Special files store their rdev value where the extends of
       a regular file are found */
    rdev = ConvertLong(rawnode,EFS_IN_EXTENTS);

    /* -----------------------------------------------------------------
       The following values are stored in my private part of the Inode.
       They are necessary for further operations with the file */

    /* get the number of extends the inode posseses */
    numext = ConvertShort(rawnode,EFS_IN_NUMEXT);

    /* if this inode has more than EFS_MAX_EXTENDS then the extends are
       stored not directly in the inode but indirect on an extra block.
       The address of the extends-block is stored in the inode */
    if(numext>EFS_MAX_EXTENTS) { 
      struct buffer_head *bh2;

      /* Store the discblock and offset of extend-list in Inode info */
      ini->extblk = sbi->fs_start + ConvertLong(rawnode,EFS_IN_EXTENTS);

      /* read INI_MAX_EXT extents from the indirect block */
      bh2 = bread(in->i_dev,ini->extblk,EFS_BLOCK_SIZE);
      if(bh2) {
	ptr = (BYTE *)bh2->b_data;
	for(i=0;i<EFS_MAX_EXTENTS;i++) {
	  ini->extends[i][0] = ConvertLong(ptr,0);
	  ini->extends[i][1] = ConvertLong(ptr,4);
	  ptr+=8;
	}
	brelse(bh2);
      } else
	printk("efs: failed reading indirect extends!\n");

    } else {
      /* The extends are found in the inode block */
      ini->extblk = blk;
      
      /* copy extends directly from rawinode */
      ptr = rawnode + EFS_IN_EXTENTS;
      for(i=0;i<numext;i++) {
	ini->extends[i][0] = ConvertLong(ptr,0);
	ini->extends[i][1] = ConvertLong(ptr,4);
	ptr+=8;
      }
    }
    ini->tot = numext;
    ini->cur = 0;

    brelse(bh);
    
#ifdef DEBUG
    printk("inode: blk %lx %x off num %d\n",
	   ini->extblk,ini->extoff,ini->extnum);
#endif

    /* Install the filetype Handler */
    switch(in->i_mode & S_IFMT) {
    case S_IFDIR: 
      in->i_op = &efs_dir_in_ops; 
      break;
    case S_IFREG:
      in->i_op = &efs_file_in_ops;
      break;
    case S_IFLNK:
      in->i_op = &efs_symlink_in_ops;
      break;
    case S_IFCHR:
      in->i_rdev = rdev;
      in->i_op = &chrdev_inode_operations; 
      break;
    case S_IFBLK:
      in->i_rdev = rdev;
      in->i_op = &blkdev_inode_operations; 
      break;
    case S_IFIFO:
      init_fifo(in);
      break;    
    default:
      printk("EFS: Unsupported inode Mode %o\n",in->i_mode);
      error++;
      break;
    }
        
  } else {
    printk("EFS: Inode: failed bread!\n");
    error++;
  }
  
  /* failed inode */
  if(error) {
    printk("EFS: read inode failed with %d errors\n",error);
    in->i_mtime = in->i_atime = in->i_ctime = 0;
    in->i_size = 0;
    in->i_nlink = 1;
    in->i_uid = in->i_gid = 0;
    in->i_mode = S_IFREG;
    in->i_op = NULL;   
  }
}
