#include <stdio.h>
#include <unistd.h>

#include "keyb.h"
#include "keyb_internal.h"

typedef int (*k_conv)(unsigned char value, char up_flag,
					  unsigned short *key);
typedef int (k_convfn)(unsigned char value, char up_flag,
					   unsigned short *key);

static k_convfn
  conv_self, conv_fn, conv_spec, conv_pad, conv_cur, conv_shift,
  conv_meta, conv_ascii, conv_lock, conv_slock, conv_ignore;

static k_conv key_convert[16] =
{
  conv_self,		/* KT_LATIN */
	conv_fn,		/* KT_FN */
	conv_spec,		/* KT_SPEC */
	conv_pad,		/* KT_PAD */
	conv_ignore,	/* KT_DEAD */
	conv_ignore,	/* KT_CONS */
	conv_cur,		/* KT_CUR */
	conv_shift,		/* KT_SHIFT */
	conv_meta,		/* KT_META */
	conv_ascii,		/* KT_ASCII */
	conv_lock,		/* KT_LOCK */
	conv_ignore,	/* KT_LETTER -- handled as KT_LATIN */
	conv_slock,		/* KT_SLOCK */
	conv_ignore,	/* ??? */
	conv_ignore,	/* ??? */
	conv_ignore		/* ??? */
};

int keyb_translatekey(unsigned char keycode, int down, keyb_key_t *key)
{
  char up_flag = down ? 0 : 0200;
  unsigned short key1;
  int rc;

  unsigned short keysym;
  unsigned char type;

  int shift_final = shift_state ^ lockstate ^ slockstate;
  unsigned short *key_map = key_maps[shift_final];

  if (key_map != NULL)
  {
	keysym = key_map[keycode];
	type = KTYP(keysym);

	if (type >= 0xf0)
	{
	  type -= 0xf0;
	  if (type == KT_LETTER)
	  {
		type = KT_LATIN;
		if (vc_kbd_led(VC_CAPSLOCK))
		{
		  key_map = key_maps[(shift_final ^ (1 << KG_SHIFT))];
		  if (key_map)
			keysym = key_map[keycode];
		};
	  };
	  if(opts.ctrl_as_state && (shift_final & (1 << KG_CTRL)))
	  {
		key_map = key_maps[(shift_final ^ (1 << KG_CTRL))];
		if (key_map)
		  keysym = key_map[keycode];
	  };
	  rc = (*key_convert[type])(keysym & 0xff, up_flag, &key1);
	  if(rc)
	  {
		*key = (shift_final << 16) | key1;
		return 1;
	  };
	}
#ifdef DEBUG
	else
	{
	  fprintf(stderr, "Wrong type = %i found\n", type);
	};
#endif    /* DEBUG */
  }
#ifdef DEBUG
  else
  {
	fprintf(stderr, "key_map for shift_final = 0x%08x not found\n", 
			shift_final);
  };
#endif   /* DEBUG */
  return 0;
}

static int conv_ignore(unsigned char value, char up_flag, unsigned short *key)
{
  return 0;
}

static int conv_spec(unsigned char value, char up_flag, unsigned short *key)
{
  if(value != KVAL(K_ENTER))
	return 0;
  if(opts.tr_enter)
  {
	*key = K(KT_LATIN, '\n');
  }
  else
  {
	*key = K_ENTER;
  };
  return 1;
}

static int conv_self(unsigned char value, char up_flag, unsigned short *key)
{
  *key = K(KT_LATIN, value);
  return 1;
}

static int conv_fn(unsigned char value, char up_flag, unsigned short *key)
{
  *key = K(KT_FN, value);
  return 1;
}

static int conv_pad(unsigned char value, char up_flag, unsigned short *key)
{
  static const char *pad_chars = "0123456789+-*/\015,.?()";
  int rc;

  /* kludge... shift forces cursor/number keys */
  if(opts.tr_pad && !k_down[KG_SHIFT])
  {
	*key = K(KT_PAD, value);
	return 1;
  };

  if (!vc_kbd_led(VC_NUMLOCK))
  {
	switch (value)
	{
	 case KVAL(K_PCOMMA):
	 case KVAL(K_PDOT):
	  rc = conv_fn(KVAL(K_REMOVE), 0, key);
	  break;
	 case KVAL(K_P0):
	  rc = conv_fn(KVAL(K_INSERT), 0, key);
	  break;
	 case KVAL(K_P1):
	  rc = conv_fn(KVAL(K_SELECT), 0, key);
	  break;
	 case KVAL(K_P2):
	  rc = conv_cur(KVAL(K_DOWN), 0, key);
	  break;
	 case KVAL(K_P3):
	  rc = conv_fn(KVAL(K_PGDN), 0, key);
	  break;
	 case KVAL(K_P4):
	  rc = conv_cur(KVAL(K_LEFT), 0, key);
	  break;
	 case KVAL(K_P6):
	  rc = conv_cur(KVAL(K_RIGHT), 0, key);
	  break;
	 case KVAL(K_P7):
	  rc = conv_fn(KVAL(K_FIND), 0, key);
	  break;
	 case KVAL(K_P8):
	  rc = conv_cur(KVAL(K_UP), 0, key);
	  break;
	 case KVAL(K_P9):
	  rc = conv_fn(KVAL(K_PGUP), 0, key);
	  break;
	 case KVAL(K_P5):   /* ghmm... */
	  rc = 0;
	  break;
	 default:
	  rc = 0;
	  break;
	}
  }
  else
  {
	rc = 1;
	*key = K(KT_LATIN, value + 0x30);
  };

  if(value == KVAL(K_PPLUS) || value == KVAL(K_PMINUS) ||
	 value == KVAL(K_PSTAR) || value == KVAL(K_PSLASH))
  {
	rc = conv_self(pad_chars[value], 0, key);
  };

  return rc;
};

static int conv_cur(unsigned char value, char up_flag, unsigned short *key)
{
  *key = K(KT_CUR, value);
  return 1;
}

static int conv_shift(unsigned char value, char up_flag, unsigned short *key)
{
  /* Mimic typewriter:
   a CapsShift key acts like Shift but undoes CapsLock */
  if (value == KVAL(K_CAPSSHIFT))
  {
	value = KVAL(K_SHIFT);
  }
  *key = K(KT_SHIFT, value);
  return 1;
}

static int conv_meta(unsigned char value, char up_flag, unsigned short *key)
{
  /* we have report alt/meta pressing via another way, so... */
  return conv_self(value, 0, key);
}

static int conv_ascii(unsigned char value, char up_flag, unsigned short *key)
{
  /* any ideas? */
  return 0;
}

static int conv_lock(unsigned char value, char up_flag, unsigned short *key)
{
  if(!opts.tr_lock)
	return 0;
  return conv_shift(value, 0, key);
}

static int conv_slock(unsigned char value, char up_flag, unsigned short *key)
{
  if(!opts.tr_slock)
	return 0;
  return conv_shift(value, 0, key);
}
