#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <linux/kd.h>

#include "keyb.h"
#include "keyb_internal.h"

int __tty_fd = -1;

static struct termios oldkbdtermios, newkbdtermios;
static int oldkbmode = K_XLATE;

keyb_options_t opts =
{
  1,    /* tr_enter */
	0,    /* tr_pad */
	1,    /* tr_lock */
	1,    /* tr_slock */
	1,    /* ctrl_as_state */
	0   /* show_release */
};

static int is_a_console(int fd)
{
  char arg;

  arg = 0;
  return (ioctl(fd, KDGKBTYPE, &arg) == 0
		  && ((arg == KB_101) || (arg == KB_84)));
}

static int open_a_console(char *fnam)
{
  int fd;

  /* try read-write */
  fd = open(fnam, O_RDWR);

  /* if failed, fail */
  if (fd < 0)
	return -1;

  /* if not a console, fail */
  if (! is_a_console(fd))
  {
	close(fd);
	return -1;
  }

  /* success */
  return fd;
}

/*
 * Get an fd for use with kbd/console ioctls.
 * We try several things because opening /dev/console will fail
 * if someone else used X (which does a chown on /dev/console).
 *
 * if tty_name is non-NULL, try this one instead.
 */

static int get_console_fd(char* tty_name)
{
  int fd;

  if (tty_name)
  {
    if (-1 == (fd = open_a_console(tty_name)))
      return -1;
	else
	  return fd;
  }

  fd = open_a_console("/dev/tty");
  if (fd >= 0)
    return fd;

  fd = open_a_console("/dev/console");
  if (fd >= 0)
    return fd;

  for (fd = 0; fd < 3; fd++)
    if (is_a_console(fd))
      return fd;

  return -1;		/* total failure */
}

int keyb_init()
{
  if((__tty_fd = get_console_fd(NULL)) == -1)
  {
#ifdef DEBUG
	fprintf(stderr, "Can`t get_console_fd(NULL)\n");
#endif    /* DEBUG */
	return -1;
  };

  if(ioctl(__tty_fd, KDGKBMODE, &oldkbmode))
  {
#ifdef DEBUG
	fprintf(stderr, "Can`t ioctl(%i, KDGKBMODE, &oldkbmode) %s\n",
			__tty_fd, strerror(errno));
#endif    /* DEBUG */
	return -1;
  };

  tcgetattr(__tty_fd, &oldkbdtermios);
  newkbdtermios = oldkbdtermios;

  newkbdtermios.c_lflag &= ~(ICANON | ECHO | ISIG);
  newkbdtermios.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
  newkbdtermios.c_cc[VMIN] = 0;       /* Making these 0 seems to have the */
  newkbdtermios.c_cc[VTIME] = 0;      /* desired effect. */

  tcsetattr(__tty_fd, TCSAFLUSH, &newkbdtermios);

  ioctl(__tty_fd, KDSKBMODE, K_RAW);

  return 0;
};

#ifdef SVGALIB

extern int __svgalib_tty_fd;
extern int __svgalib_kbd_fd;

int keyb_init_svgalib()
{
  __tty_fd = __svgalib_kbd_fd = __svgalib_tty_fd;

  if(__tty_fd == -1)
  {
# ifdef DEBUG
	fprintf(stderr, "__svgalib_tty_fd == -1\n");
# endif    /* DEBUG */
	return -1;
  };

  if(ioctl(__tty_fd, KDGKBMODE, &oldkbmode))
  {
# ifdef DEBUG
	fprintf(stderr, "Can`t ioctl(%i, KDGKBMODE, &oldkbmode) %s\n",
			__tty_fd, strerror(errno));
# endif    /* DEBUG */
	return -1;
  };

  tcgetattr(__tty_fd, &oldkbdtermios);
  newkbdtermios = oldkbdtermios;

  newkbdtermios.c_lflag &= ~(ICANON | ECHO | ISIG);
  newkbdtermios.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
  newkbdtermios.c_cc[VMIN] = 0;       /* Making these 0 seems to have the */
  newkbdtermios.c_cc[VTIME] = 0;      /* desired effect. */

  tcsetattr(__tty_fd, TCSAFLUSH, &newkbdtermios);

  ioctl(__tty_fd, KDSKBMODE, K_RAW);

  return 0;
};

#endif   /* SVGALIB */

void keyb_close()
{
  ioctl(__tty_fd, KDSKBMODE, oldkbmode);
  tcsetattr(__tty_fd, 0, &oldkbdtermios);

  __tty_fd = -1;
};
