/*
 * symtab.c --
 *
 *      Symbol table implementation.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */


#include "symtab.h"


/*
 * Local Variables
 */
static OS_SYM_TAB_TYPE symtab[MAX_SYMBOLS+1];
static pthread_mutex_t symtab_lock;
static unsigned long   last_id   = SYM_COUNT_BASE;


int get_symtab_base()
{
   return SYM_COUNT_BASE;
}

int get_symtab_size()
{
   return last_id;
}

/*
 *--------------------------------------------------------------
 *
 * add_sym --
 *
 *      Adds a symbol entry into the symtab.
 *
 * Results:
 *      Returns a unique identifier for the symbol if the
 *      function is successful. Otherwise, return BADPRM.
 *
 *--------------------------------------------------------------
 */

uxid_t add_sym(key_t  key,
               void*  data_p)
{
    uxid_t retval = NOERR;
    int    i;
    
    /*
     * Check for duplicate symbols
     */    
    for (i=SYM_COUNT_BASE; i<last_id; i++)
    {        
       if (symtab[i].key == key)
       {
           return BADPRM;
       }
    }    

    if (data_p != NULL)
    {
       int  free = -1;
	
       pthread_mutex_lock(&symtab_lock);

       /*
        * Search for a free slot in the symbol table
	*/
       for (i=SYM_COUNT_BASE; i<MAX_SYMBOLS; i++)
       {
	   if (symtab[i].sym == NULL)
	   {
	       free = i;
	       break;
	   }
       }

       if (free != -1)
       {
	   symtab[free].sym = data_p;
	   symtab[free].key = key;
	   pthread_mutex_init(&symtab[free].sym_lock, NULL);       
	   retval = free;
	   if (last_id <= free)
	       last_id = free+1;
       }
       else
       {
	   retval = QUEFUL;
       }
       
       pthread_mutex_unlock(&symtab_lock);
       
       return retval;
    }

    return BADPRM;
}


/*
 *--------------------------------------------------------------
 *
 * delete_sym --
 *
 *      Deletes the given symbol from the table.
 *
 * Results:
 *      Returns NOERR if the function is successful. Otherwise,
 *      return BADPRM.
 *
 *--------------------------------------------------------------
 */

long delete_sym(uxid_t tid)
{
    if (tid >= last_id || symtab[tid].sym == NULL)
        return BADPRM;

    pthread_mutex_lock(&symtab_lock);
    
    free(symtab[tid].sym);
    symtab[tid].sym = NULL;
    symtab[tid].key = 0;
    pthread_mutex_destroy(&symtab[tid].sym_lock);       

    pthread_mutex_unlock(&symtab_lock);

    return NOERR;
}


/*
 *--------------------------------------------------------------
 *
 * id2sym --
 *
 *      Returns the symbol corresponding to the given identifier.
 *
 * Results:
 *      Returns NULL if the identifier is invalid. Otherwise,
 *      return the symbol.
 *
 *--------------------------------------------------------------
 */

void* id2sym(uxid_t id)
{
    if (id >= last_id)
	return NULL;

    return symtab[id].sym;
}    


/*
 *--------------------------------------------------------------
 *
 * key2sym --
 *
 *      Returns the symbol corresponding to the given key.
 *
 * Results:
 *      Returns NULL if the identifier is invalid. Otherwise,
 *      return the symbol.
 *
 *--------------------------------------------------------------
 */

void* key2sym(key_t key)
{
    int i;
   
    for (i=SYM_COUNT_BASE; i<last_id; i++)
    {        
       if (symtab[i].key == key)
           return symtab[i].sym;
    }

    return NULL;    
}

/*
 *--------------------------------------------------------------
 *
 * key2id --
 *
 *      Returns the identifier corresponding to the given key.
 *
 * Results:
 *      Returns BADPRM if the identifier is invalid. Otherwise,
 *      return the identifier.
 *
 *--------------------------------------------------------------
 */

uxid_t key2id(key_t key)
{
    int i;
   
    for (i=SYM_COUNT_BASE; i<last_id; i++)
    {        
       if (symtab[i].key == key)
           return i;
    }

    return BADPRM;
}


/*
 *--------------------------------------------------------------
 *
 * id2key --
 *
 *      Returns the key corresponding to the given identifier.
 *
 * Results:
 *      Returns BADPRM if the identifier is invalid. Otherwise,
 *      return the key.
 *
 *--------------------------------------------------------------
 */

key_t id2key(uxid_t id)
{
    if (id >= last_id || symtab[id].sym == NULL)
        return BADPRM;

    return symtab[id].key;
}


/*
 *--------------------------------------------------------------
 *
 * lock_sym --
 *
 *      Locks the symbol entry given by the idenitifier id.
 *      This prevents concurrent writes to a symbol entry which may
 *      potentially corrupt the entry data.
 *
 * Results:
 *      None.
 *
 *--------------------------------------------------------------
 */

void lock_sym(uxid_t id)
{
    if (id < last_id)
        pthread_mutex_lock(&symtab[id].sym_lock);
}


/*
 *--------------------------------------------------------------
 *
 * release_sym --
 *
 *      Release the symbol entry given by the idenitifier id.
 *      This prevents concurrent writes to a symbol entry which may
 *      potentially corrupt the entry data.
 *
 * Results:
 *      None.
 *
 *--------------------------------------------------------------
 */

void release_sym(uxid_t id)
{
    if (id < last_id)    
        pthread_mutex_unlock(&symtab[id].sym_lock);       
}

/*
 *--------------------------------------------------------------
 *
 * init_symtab --
 *
 *      Initialise the symbol table. This function must be called
 *      before using any of the services to access the symbol
 *      table.
 *
 * Results:
 *      None.
 *
 *--------------------------------------------------------------
 */

void init_symtab()
{
    pthread_mutex_init(&symtab_lock, NULL);
}
