/*
 * sem.c --
 *
 *      Implements the semaphore services.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */


#include <stdlib.h>
#include <math.h>
#include "sem.h"
#include "os_timer.h"

long wait_sem_cond(pthread_cond_t* cond, long interval);


uxid_t crsem(key_t key)
{
    OS_SEM_TYPE*       sem_p;
    uxid_t             retval;

    /*
     *	If the efg exists, return it's gid (attach).
     */
    if ((retval = key2id(key)) == BADPRM)
    {
    
	sem_p = (OS_SEM_TYPE*) malloc(sizeof(OS_SEM_TYPE));
	sem_p->symtype = sym_sem;    
	sem_p->count = 0;    
	if ((pthread_mutex_init(&sem_p->sem_mutex, NULL) == 0) &&
	    (pthread_cond_init(&sem_p->sem_timer, NULL) == 0))
	{        
	    retval = add_sym(key, sem_p);
	}
	else
	{
	    free(sem_p); 
	    retval = QUEFUL;    
	}
    }
	
    return retval;    
}

long waisem(uxid_t sid, ULONG *stabuf, qual_t qual)
{
    long    retval = NOERR;
    long    period;    
    
    *stabuf = 0L;
    
    /*
     * Only support wait to finish coordination mode
     */
    
    if (CMODE(qual) == WAIFIN)
    {
        OS_SEM_TYPE* sem_p = id2sym(sid);    

        if (sem_p == NULL)
	{
	    *stabuf = BADPRM;
	    
            return BADPRM;
	}
        
        /*
         * Only support infinite or immediate times 
        */
          
        switch (INTERVAL(qual))
        {
            case IMONLY:
                /*
                 *  call getmsn to try to lock
                 */

                if (sem_p->count > 0)
                    sem_p->count--;
                else
                    retval = TIMOUT;
                break;
                
            case NOEND:

                
                /*
                 * Call os_getmsw to wait indefinitely
                 */
                if (sem_p->count == 0)
                    pthread_mutex_lock(&sem_p->sem_mutex);         

                sem_p->count--;                
                break;

            default:

                /*
                 * Implement timeout
                 */
                if (sem_p->count == 0)
                {
                    period = LNUM(qual);
                    if (wait_sem_cond(&sem_p->sem_timer, period) == TIMOUT)
                        break;                    
                }

                sem_p->count--;
                
                break;
        }        
    }
    else
    {
        perror("waisem: Requested modes not supported\n");
        
        retval = BADPRM;
    }

    return retval;
    
}

long rlssem(uxid_t sid)
{
    OS_SEM_TYPE* sem_p  = id2sym(sid);
    long         retval = BADPRM;
    
    if (sem_p != NULL)
    {
        pthread_mutex_unlock(&sem_p->sem_mutex);
        pthread_cond_signal(&sem_p->sem_timer);        
        sem_p->count++;
	
        retval = (sem_p->count) ? NOTFRE : NOERR;
    }
            
    return retval;    
}

long dlsem(uxid_t sid)
{
    OS_SEM_TYPE* sem_p = id2sym(sid);

    if (sem_p->count == 0)
    {	
	pthread_mutex_destroy(&sem_p->sem_mutex);
	pthread_cond_destroy(&sem_p->sem_timer);
	
	return delete_sym(sid);
    }
    else
    {
	return NOTFRE;
    }
}

long wait_sem_cond(pthread_cond_t* cond, long interval)
{
    int             unit  = interval / MS;
    int             val   = interval % MS; 
    long            time;
    struct timeval  curtime;
    struct timespec abstime;
    pthread_mutex_t mutex;
    
    gettimeofday(&curtime, NULL);
    time = val * pow(10, unit-1) * 1000;
    curtime.tv_sec += time / 1000000;
    curtime.tv_usec += time % 1000000;
    TIMEVAL_TO_TIMESPEC(&curtime, &abstime);

    pthread_mutex_init(&mutex, NULL);    
    pthread_mutex_lock(&mutex);    
    if (pthread_cond_timedwait(cond, &mutex, &abstime) != -1) {
        return TIMOUT;
    }
    
    return NOERR;
}
