/*
 * efg.c --
 *
 *      Provides event flag services for the MTOS-UX simulator.
 *
 * Copyright (C) 1995, 1996 Daniel Wu.
 * Distributed under the terms of the GNU Library General Public
 * License.
 *
 * This file is part of SimTos.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation (version 2).
 *
 * This library is distributed "AS IS" in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with Pthreads; see the file COPYING.  If not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
 * 02139, USA.
 *
 * Author: Daniel Wu (dwu@linus.socs.uts.edu.au)
 *
 */


#include <math.h>

#include "os_mtosux.h"
#include "symtab.h"
#include "task.h"
#include "efg.h"


/*
 *--------------------------------------------------------------
 *
 * crefg --
 *
 *      Create or attach a global efg.
 *
 * Results:
 *      If the function is successful, returns the identifier of
 *      the event flag group. Otherwise, QUEFUL.
 *
 *--------------------------------------------------------------
 */

uxid_t	crefg(key_t key)
{
    OS_EVT_FLAG_TYPE*   new_efg;
    uxid_t              gid;
    
    /*
     *	If the efg exists, return it's gid (attach).
     */
    if ((gid = key2id(key)) != BADPRM)
    {
        return gid;
    }

    /*
     *	Make a new efg.
     */
    new_efg = (OS_EVT_FLAG_TYPE*) malloc(sizeof(OS_EVT_FLAG_TYPE));    
    if (new_efg != NULL)
    {
        
        /*
         * Initialize efg fields.
         */
        new_efg->symtype = sym_efg;
        new_efg->gid	= gid;
        new_efg->key	= key;
        new_efg->status	= EFG_NORMAL;
        new_efg->flags  = 0;
        new_efg->xflags  = 0;

        /*
        ** Create system variables
        */
        
        /*
         * Add key to the symbol table.  The symbol entry points
         * to the efg.
         */        
        if (os_init_efg(new_efg) != NOERR)
        {
          return QUEFUL;
        }
                           
        return add_sym(key, new_efg);             
    }
    
    /*
     * Create failed.  Return QUEFUL.
     */
    
    return QUEFUL;
    
}


/*
 *--------------------------------------------------------------
 *
 * dlefg --
 *
 *      Deletes a global efg.
 *
 * Results:
 *      If the function is successful, returns NOERR. Otherwise,
 *      BADPRM.
 *
 *--------------------------------------------------------------
 */

long	dlefg(uxid_t gid)
{

    OS_EVT_FLAG_TYPE*	efg_ptr;
    
    /*
     *	Find efg in symbol table.
     */
    if ((efg_ptr = id2sym(gid)) != NULL)
    {
	if (efg_ptr->status == EFG_NORMAL)
	{
	    pthread_cond_destroy(&(GET_OS_COND((*efg_ptr)))); 
	    pthread_mutex_destroy(&(GET_OS_MUTEX((*efg_ptr))));

	    /*
	     * Delete symbol table entries.
	     */        
	    delete_sym(gid);
	}
	else
	{
	    efg_ptr->status = EFG_DELETED;
	}
	
        return NOERR;

    }

    /*
     * No efg exists for specifed gid.  Return BADPRM.
     */
    
    return BADPRM;
    
}


/*
 *--------------------------------------------------------------
 *
 * waiefg --
 *
 *      Wait until the event flags are set.
 *
 * Results:
 *      If the function is successful, returns the final value
 *      of the event flag group (with the high-order word
 *      cleared). Otherwise, returned BADPRM, QUEFUL or TIMOUT.
 *
 *--------------------------------------------------------------
 */

long waiefg(uxid_t gid,	ULONG opmask, qual_t intrvl)
{
    uxid_t              tid      = gettid(0);
    OS_TSK_TYPE*        sym      = id2sym(tid);
    unsigned short      op       = OP(opmask);        
    unsigned short      mask     = MASK(opmask);
    OS_EVT_FLAG_TYPE*	efg_ptr;
    struct timespec     abstime;
    
    if (gid == 0)
    {
        if (sym != NULL)
        {
            efg_ptr = &sym->efgs;
        }
        else
        {            
            return BADPRM;
        }
    }
    else
    {
	efg_ptr = id2sym(gid);
        if (efg_ptr == NULL || efg_ptr->symtype != sym_efg)
            return BADPRM;
    }

    if ((intrvl%MS) > 0)
    {
	int             unit     = intrvl / MS;
	int             val      = intrvl % MS; 
	long            time;
	struct timeval  curtime;
	
        time = val * pow(10, unit-1) * 1000;
        gettimeofday(&curtime, NULL);
        curtime.tv_sec += time / 1000000L;
        curtime.tv_usec += time % 1000000L;
        if (curtime.tv_usec >= 1000000L)
        {
            curtime.tv_sec++;
            curtime.tv_usec = curtime.tv_usec - 1000000L;
        }
        TIMEVAL_TO_TIMESPEC(&curtime, &abstime);
        
    }
    
    switch (op)
    {
      case OP(EFAND):
	   while ((efg_ptr->flags & mask) != mask)
	   {
	       /*
	       ** Wait for a change indication
	       */
	       pthread_mutex_trylock(&GET_OS_MUTEX((*efg_ptr)));
	       if ((intrvl%MS) > 0)
	       {
		   if (pthread_cond_timedwait(&GET_OS_COND((*efg_ptr)),
					      &GET_OS_MUTEX((*efg_ptr)),
					      &abstime) == -1)
		   {
		       return TIMOUT;
		   }
	       }
	       else
	       {
		   pthread_cond_wait(&GET_OS_COND((*efg_ptr)),
				     &GET_OS_MUTEX((*efg_ptr)));
	       }
	   }
	   break;
	   
      case OP(EFOR):
	  while ((mask != 0) && ((mask & efg_ptr->flags) == 0))
	  {
	      /*
	      ** Wait for a change indication
	      */                
	      pthread_mutex_trylock(&GET_OS_MUTEX((*efg_ptr)));
	      if ((intrvl%MS) > 0)
	      {
		  if (pthread_cond_timedwait(&GET_OS_COND((*efg_ptr)),
					     &GET_OS_MUTEX((*efg_ptr)),
					     &abstime) == -1)
		  {
		      return TIMOUT;
		  }
	      }
	      else
	      {
		  pthread_cond_wait(&GET_OS_COND((*efg_ptr)),
				    &GET_OS_MUTEX((*efg_ptr)));
	      }
	  }
	  break;

      default:
	  return BADPRM;
	  
    }                      
    
    /*
     * Clean up if the EFG is deleted
     */
    if (efg_ptr->status == EFG_DELETED)
    {
	pthread_cond_destroy(&(GET_OS_COND((*efg_ptr)))); 
	pthread_mutex_destroy(&(GET_OS_MUTEX((*efg_ptr))));
	
	/*
	 * Delete symbol table entries.
	 */        
	delete_sym(gid);
    }
    
    return efg_ptr->flags;    
}


/*
 *--------------------------------------------------------------
 *
 * srslef --
 *
 *      Set or reset a set of event flags for the event flag group
 *      associated with the task (local event flag group)
 *
 * Results:
 *      The current value of the event flag
 *
 *--------------------------------------------------------------
 */

long srslef(uxid_t tid, ULONG opmask)
{
    OS_TSK_TYPE*  sym;

    if (tid == 0L)
        tid = gettid(0);

    sym = id2sym(tid);            
    if (OP(opmask) == EFRST)
        sym->efgs.flags &= ~MASK(opmask);
    else
        sym->efgs.flags |= MASK(opmask);

    /*
     * Go through flags and release any semaphores
     */

    /*
     * For all tasks waiting on the event flag
     */
    pthread_cond_broadcast(&GET_OS_COND(sym->efgs));
    pthread_yield(NULL);
           
    return sym->efgs.flags;
}

/*
 *--------------------------------------------------------------
 *
 * srsefg --
 *
 *      Set or reset a set of event flags for the given group.
 *
 * Results:
 *      The current value of the event flag.
 *
 *--------------------------------------------------------------
 */

long srsefg(uxid_t gid, ULONG opmask)
{

    OS_EVT_FLAG_TYPE*  sym;    

    if (gid == 0L)
    {
       return srslef(0L, opmask);        
    }

    sym = id2sym(gid);    
    if (sym == NULL)
        return (BADPRM<<16);
    
    if (OP(opmask) == EFRST)
        sym->flags &= ~(MASK(opmask));
    else
        sym->flags |= MASK(opmask);

    /*
     * Go through flags and release any semaphores
     */

    /*
     * For all tasks waiting on the event flag
     */
    pthread_cond_signal(&GET_OS_COND((*sym)));
    pthread_yield(NULL);

    return sym->flags;    
}


/*
 *--------------------------------------------------------------
 *
 * os_init_efg --
 *
 *      Initilaise the components for the event flag group.
 *
 * Results:
 *      NOERR if successful, otherwise QUEFUL.
 *
 *--------------------------------------------------------------
 */

long os_init_efg(OS_EVT_FLAG_TYPE* efgs)
{
  if ((pthread_cond_init(&(GET_OS_COND((*efgs))), NULL) != 0) ||
      (pthread_mutex_init(&(GET_OS_MUTEX((*efgs))), NULL) != 0))
  {
      return QUEFUL;
  }

  return NOERR;
}


typedef struct efgarg {
    uxid_t         efg_id;
    uxid_t         tsk_id;
    unsigned short mask;
    qual_t         interval;
} OS_EFG_ARG;

static OS_EVT_FLAG_TYPE* os_getsym(uxid_t gid, uxid_t tid)
{
    OS_EVT_FLAG_TYPE* sym = NULL;    
    
    if (gid == 0L)
    {
	OS_TSK_TYPE* tsym;
	
	tsym = id2sym(tid);
	if (tsym != NULL)
	{
	    sym = &tsym->efgs;
	}
    }
    else
    {
	sym = id2sym(gid);
    }

    return sym;
}

static void monitor(UPTR arg)
{
    OS_EFG_ARG*       efgargs     = (OS_EFG_ARG*) arg;
    pthread_t         curr_thread = pthread_self();
    OS_EVT_FLAG_TYPE* efg;
    pthread_cond_t    pcond;
    pthread_mutex_t   pmutex;        
    
    /*
     * Delay setting of the event flag by interval
     */
    pthread_cond_init(&pcond, NULL);        
    pthread_mutex_init(&pmutex, NULL);            
    os_pause(&pcond, &pmutex, efgargs->interval);
    pthread_mutex_destroy(&pmutex);            
    pthread_cond_destroy(&pcond);

    /*
    ** Get the EFG structure
    */
    efg = os_getsym(efgargs->efg_id, efgargs->tsk_id);
    
    if (efg != NULL && efg->symtype == sym_efg)
    {
	short savedmask = efgargs->mask;

	/*
	 * Update mask
	 */
	efgargs->mask &= ~efg->xflags;	
	efg->xflags &= ~savedmask;
	
	/*
	 * Update the efg data structure
	 */
	efg->flags |= MASK(efgargs->mask);
    
	/*
	 * Go through flags and release any semaphores
	 */
	
	/*
	 * For all tasks waiting on the event flag
	 */
	pthread_cond_signal(&GET_OS_COND((*efg)));
	pthread_yield(NULL);

    }
    

    /*
     * Deallocate argument structure
     */
    free(efgargs);
    
    /*
     * Terminate the thread
     */
    pthread_detach(&curr_thread);
}

/*
 *--------------------------------------------------------------
 *
 * sgiefg --
 *
 *      Set or reset a set of event flags for the given group
 *      after an interval.
 *
 * Results:
 *      The current value of the event flag.
 *
 *--------------------------------------------------------------
 */

long sgiefg(		/* set event flag after given interval */
	  uxid_t	gid,	/* group identifier */
	  ULONG		mask,	/* mask to select flags */
	  qual_t	intrvl	/* time interval */
	)
{
    uxid_t             tid      = gettid(0);    
    long               retval   = NOERR;
    OS_EVT_FLAG_TYPE*  sym      = NULL;    
    OS_EFG_ARG*        efgargs;
    pthread_t          mthread;    
    
    /*
     * Retrieve the EFG data structure
     */
    sym = os_getsym(gid, tid);
    if (sym == NULL)
        return (BADPRM<<16);

    /*
    ** Pattern for mask after given interval
    */
    if ((intrvl%MS) != 0L)
    {
	pthread_attr_t tsk_attr;
	
	/*
	 * Create EFG argument for the monitor thread
	 */    
	efgargs = (OS_EFG_ARG*) malloc(sizeof(OS_EFG_ARG));
	efgargs->mask = mask;
	efgargs->efg_id = gid;
        efgargs->tsk_id = tid;
	efgargs->interval = intrvl;
	sym->xflags &= ~mask;
	sym->status = EFG_ACTIVE;
        
	/*
	 * Start Monitor task to set event flag after an interval
	 */
	pthread_create(&mthread, NULL, (pthread_func_t)monitor, efgargs);

	/*
	** Set priority of monitor task to maximum
	*/
	pthread_getschedattr(mthread, &tsk_attr);    
	pthread_attr_setprio(&tsk_attr, 255);
	pthread_setschedattr(mthread, tsk_attr);
		
	pthread_yield(NULL);	
    }
    else
    {
	sym->xflags |= mask;
	retval = TIMCAN;
    }
    
    return retval;
}
