/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* mysql_select and join optimization */
/* Copyright Monty Program Kb & TCX DataKonsult AB */

#include "mysql_priv.h"
#include "procedure.h"
#include "select_q.h"
#include <nisam.h>

#define JOIN_CACHE_SIZE	RECORD_CACHE_SIZE	/* when using full join */
#define PREV_BITS(A)	(ulong) ((1L << (A)) -1)

enum cond_result { COND_UNDEF,COND_OK,COND_TRUE,COND_FALSE };

typedef struct keyuse_t {
  TABLE *table;
  FIELD	*field;				/* Field to compare keypart */
  Item		*val;				/* or value if no field */
  uint		key,keypart;
} KEYUSE;


/*
** CACHE_FIELD and JOIN_CACHE is used on full join to cache records in outer table
*/

typedef struct st_cache_field {
  char *str;
  uint length,blob_length;
  FIELD *blob_field;
  bool strip;
} CACHE_FIELD;


typedef struct st_join_cache {
  uchar *buff,*pos,*end;
  uint records,record_nr,ptr_record,fields,length,blobs;
  CACHE_FIELD *field,**blob_ptr;
  SELECT *select;
} JOIN_CACHE;


/*
** The structs which holds the join connections and join states
*/

enum join_type { JT_UNKNOWN,JT_SYSTEM,JT_CONST,JT_EQ_REF,JT_REF,JT_MAYBE_REF,
		 JT_ALL};

struct st_join;

typedef struct st_join_table {
  TABLE *table;
  uint		keys;				/* all keys with can be used */
  uint		const_keys;			/* Keys with constant part */
  ulong		records,found_records;
  uint		use_quick;
  enum join_type type;
  READ_RECORD	read_record;
  JOIN_CACHE	cache;
  KEYUSE	*keyuse;			/* pointer to first used key */
  SELECT	*select;
  int		(*read_first_record)(struct st_join_table *tab);
  int		(*next_select)(struct st_join *,struct st_join_table *,bool);
  uint		used_fields,used_fieldlength,used_blobs;
} JOIN_TAB;


typedef struct st_position {			/* Used in find_best */
  uint index;
  KEYUSE *key;
  double records_read;
} POSITION;


typedef struct st_join {
  JOIN_TAB *join_tab;
  FORM	   **form;				/* For unireg interfaces */
  TABLE	   **table;				/* array of tables */
  uint	   tables,const_tables;
  uint	   copy_field_count,field_count,sum_func_count,func_count;
  uint	   send_group_parts,group_parts,group_length,sort_by_table;
  ulong    const_bits,send_records;
  bool	   sort_and_group,first_record,quick_group;
  POSITION positions[MAX_REF_REG+1],best_positions[MAX_REF_REG+1];
  uint     index[MAX_REF_REG+1];
  double   best_read;
  List<Item> *fields;
  List<Item> copy_funcs;
  List<Item_buff> group_fields;
  TABLE    *tmp_table;
  THD	   *thd;
  COPY_FIELD *copy_field;
  Item_result_field **funcs;
  Item_sum  **sum_funcs;
  byte	    *group_buff;
  Procedure *procedure;
  Item	    *having;
  uint	    select_options;
  select_result *result;
  MYSQL_LOCK *lock;
} JOIN;


typedef struct st_select_check {
  uint const_ref,reg_ref;
} SELECT_CHECK;


static void make_join_statistics(JOIN *join,tname_t *tables,COND *conds,
				 DYNAMIC_ARRAY *keyuse);
static uint update_ref_and_keys(DYNAMIC_ARRAY *keyuse,JOIN_TAB *stat,
				COND *conds);
static int sort_keyuse(KEYUSE *a,KEYUSE *b);
static void set_position(JOIN *join,uint index,uint table,KEYUSE *key);
static void find_best_combination(JOIN *join);
static void find_best(JOIN *join,ulong rest_tables,uint index,
		      double record_count,double read_time);
static uint cache_record_length(JOIN *join,ulong rest_tables);
static double prev_record_reads(JOIN *join,ulong found_ref);
static void get_best_combination(JOIN *join);
static void store_val_in_field(FIELD *field,Item *val);
static void make_simple_join(JOIN *join,TABLE *tmp_table,SELECT *select);
#ifndef DBUG_OFF
static void TEST_join(JOIN *join);
#endif
static int make_join_select(JOIN *join,SELECT *select);
static void make_join_readinfo(JOIN *join,bool simple_group);
static void join_free(JOIN *join);
static ORDER *remove_const(JOIN *join,ORDER *first_order,uint *simple_order);
static void return_zero_rows(select_result *res,List<Item> &fields);
static COND *optimize_cond(COND *conds,enum cond_result *cond_value);
static void propagate_cond_constants(COND *and_father,COND **cond_ptr);
static void change_cond_ref_to_const(COND *conds,COND **cond_ptr,
				     Item *field,Item *value);
static COND *remove_eq_conds(COND *cond,enum cond_result *cond_value);
static TABLE * create_tmp_table(THD *thd,JOIN *join,List<Item> &fields,
				ORDER *group,
				bool distinct,bool save_sum_fields);
static void free_tmp_table(TABLE *entry);
static int do_select(JOIN *join,List<Item> *fields,TABLE *tmp_table,
		     Procedure *proc);
static int sub_select_cache(JOIN *join,JOIN_TAB *join_tab,bool end_of_records);
static int sub_select(JOIN *join,JOIN_TAB *join_tab,bool end_of_records);
static int flush_cacheed_records(JOIN *join,JOIN_TAB *join_tab,bool skipp_last);
static int end_send(JOIN *join, JOIN_TAB *join_tab, bool end_of_records);
static int end_send_group(JOIN *join, JOIN_TAB *join_tab,bool end_of_records);
static int end_write(JOIN *join, JOIN_TAB *join_tab, bool end_of_records);
static int end_update(JOIN *join, JOIN_TAB *join_tab, bool end_of_records);
static int end_write_group(JOIN *join, JOIN_TAB *join_tab,
			   bool end_of_records);
static void copy_fields(JOIN *join);
static int test_if_group_changed(List<Item_buff> &list);
static int join_read_const_tables(JOIN *join);
static int join_read_key(JOIN_TAB *tab);
static int join_read_always_key(JOIN_TAB *tab);
static int join_no_more_records(READ_RECORD *info);
static int join_read_next(READ_RECORD *info);
static int join_init_quick_read_record(JOIN_TAB *tab);
static int test_if_quick_select(JOIN_TAB *tab);
static int join_init_read_record(JOIN_TAB *tab);
static bool only_one_bit(ulong bits);
static uint max_part_bit(ulong bits);
static uint max_bit(ulong bits);
static SELECT_CHECK check_if_select_neaded(SELECT *select);
static FIELD * part_of_refkey(FORM *form,FIELD *field);
static int create_sort_index(JOIN_TAB *tab,ORDER *order,SELECT *select,
			     ulong select_limit);
static int remove_dupplicates(TABLE *entry);
static SORT_FIELD * make_unireg_sortorder(TABLE *entry,ORDER *order,
					  uint *length);
static int join_init_cache(THD *thd,JOIN_TAB *tables,uint table_count);
static uint used_blob_length(CACHE_FIELD **ptr);
static bool store_record_in_cache(JOIN_CACHE *cache);
static void reset_cache(JOIN_CACHE *cache);
static void read_cacheed_record(JOIN_TAB *tab);
static bool cmp_buffer_with_ref(byte *buffer,REF_FIELD *fields,uint length);
static int setup_order(THD *thd,tname_t *tables,List<Item> &fields,ORDER *order,
		       string where);
static int setup_group(THD *thd,tname_t *tables,List<Item> &fields,
		       ORDER *order);
static bool setup_new_fields(THD *thd,tname_t *tables,List<Item> &fields,
			     ORDER *new_order);
static ORDER *add_all_fields_to_order(JOIN *join,ORDER *order,
				      List<Item> &fields);
static void count_field_types(JOIN *join,List<Item> &fields);
static bool test_if_subpart(ORDER *a,ORDER *b);
static bool test_if_special_field(ORDER *order);
static uint get_sort_by_table(ORDER *a,ORDER *b);
static void calc_group_buffer(JOIN *join,ORDER *group);
static void alloc_group_fields(JOIN *join,ORDER *group);
static void setup_copy_fields(JOIN *join,List<Item> &fields);
static void make_sum_func_list(JOIN *join,List<Item> &fields);
static void change_to_use_tmp_fields(List<Item> &func);
static void change_refs_to_tmp_fields(List<Item> &func);
static void init_tmptable_sum_functions(Item_sum **func);
static void update_tmptable_sum_func(Item_sum **func,TABLE *tmp_table);
static void copy_funcs(Item_result_field **func_ptr);
static void copy_sum_funcs(Item_sum **func_ptr);
static void init_sum_functions(Item_sum **func);
static void update_sum_func(Item_sum **func);

/*****************************************************************************
** check fields, find best join, do the select and output fields.
** mysql_select assumes that all tables are allready opened
*****************************************************************************/

int
mysql_select(THD *thd,tname_t *tables,List<Item> &fields,COND *conds,
	     ORDER *order, ORDER *group,Item *having,ORDER *proc_param,
	     uint select_options,select_result *result)
{
  TABLE		*tmp_table;
  int		error;
  bool		need_tmp;
  uint		simple_order,simple_group,table_count;
  enum cond_result cond_value;
  SELECT	*select,*use_select;
  DYNAMIC_ARRAY keyuse;
  JOIN		join;
  Procedure	*procedure;
  List<Item>	all_fields;
  bool		select_distinct;
  DBUG_ENTER("mysql_select");

  /* Check that all tables, fields, conds and order are ok */

  select_distinct=test(select_options & SELECT_DISTINCT);
  tmp_table=0;
  use_select=select=0;
  bzero(&keyuse,sizeof(keyuse));
  all_fields=fields;
  if (setup_fields(thd,tables,fields,1) ||
      setup_conds(thd,tables,conds) ||
      setup_order(thd,tables,all_fields,order,"order clause") ||
      setup_group(thd,tables,all_fields,group) ||
      (having && having->fix_fields(thd,tables,"having clause")))
    DBUG_RETURN(-1);
  if (!tables)
  {						// Only test of functions
    if (result->prepare(fields))
      DBUG_RETURN(-1);
    result->send_fields(fields,1);
    if (result->send_data(fields))
      result->send_error(NullS);
    else
      result->send_eof();
    DBUG_RETURN(0);
  }
  procedure=setup_procedure(thd,proc_param,result,&error);
  if (error)
    DBUG_RETURN(-1);
  if (procedure)
  {
    if (setup_new_fields(thd,tables,all_fields,procedure->param_fields))
    {
      delete procedure;
      DBUG_RETURN(-1);
    }
    if (procedure->group)
    {
      if (!test_if_subpart(procedure->group,group))
      {
	my_message("Can't handle procedures with differents groups yet",MYF(0));
	delete procedure;
	DBUG_RETURN(-1);
      }
    }
#ifdef NOT_NEEDED
    else if (!group && procedure->flags & PROC_GROUP)
    {
      my_message("Select must have a group with this procedure",MYF(0));
      delete procedure;
      DBUG_RETURN(-1);
    }
#endif
    if (order && (procedure->flags & PROC_NO_SORT))
    {
      my_message("Can't use order with this procedure",MYF(0));
      delete procedure;
      DBUG_RETURN(-1);
    }
  }

  /* Init join struct */
  join.thd=thd;
  join.lock=thd->lock;
  join.join_tab=0;
  join.form=0;
  join.sum_funcs=0;
  join.send_records=0L;
  join.first_record=join.sort_and_group=0;
  join.select_options=select_options;
  join.result=result;
  count_field_types(&join,all_fields);
  join.tables=join.const_tables=0;
  table_count=0;
  for (tname_t *table=tables ; table ; table=table->next)
    table->table->tablenr=table_count++;

#ifdef RESTRICTED_GROUP
  if (join.sum_func_count && !group && (join.func_count || join.field_count))
  {
    my_message(ER(ER_WRONG_SUM_SELECT));
    delete procedure;
    DBUG_RETURN(-1);
  }
#endif
  if (result->prepare(fields))
  {
    delete procedure;
    DBUG_RETURN(-1);
  }

  /* Optimize count(*) */
  if (!tables->next && !conds && !having && all_fields.elements == 1 &&
      all_fields.head()->type() == Item::SUM_FUNC_ITEM &&
      ((Item_sum*) all_fields.head())->sum_func() == Item_sum::COUNT_FUNC &&
      !all_fields.head()->maybe_null)
  {
    result->send_fields(fields,1);
    ha_info(&tables->table->form,0);
    fields.empty();
    fields.push_back((Item*) (new Item_int((long) tables->table->
					   form.keyfile_info.records,11)));
    if (result->send_data(fields))
      result->send_error(NullS);
    else
      result->send_eof();
    delete procedure;
    DBUG_RETURN(0);
  }

  if (group || join.sum_func_count)
    select_distinct=0;
  conds=optimize_cond(conds,&cond_value);
  if (cond_value == COND_FALSE)
  {					/* Impossible cond */
    error=0;
    return_zero_rows(result,fields);
    goto err;
  }

  error = -1;
  join.tables=table_count;
  join.form=  (FORM**) sql_alloc(sizeof(FORM*)*join.tables);
  join.sort_by_table=get_sort_by_table(order,group);

  /* Calculate how to do the join */
  make_join_statistics(&join,tables,conds,&keyuse);
  if (join_read_const_tables(&join))
  {
    error=0;
    return_zero_rows(result,fields);
    goto err;
  }

  mysql_unlock_some_tables(join.table,join.const_tables);
  select=make_unireg_select(join.table,join.tables,
			    (join.const_tables == join.tables ?
			     join.const_tables-1 : join.const_tables),
			    join.const_tables,conds,&error);
  if (error)
  {
    error= -1;
    goto err;
  }
  if (make_join_select(&join,select) ||
      join.tables == join.const_tables && skipp_record(select))
  {
    error=0;
    return_zero_rows(result,fields);
    goto err;
  }

  use_select=select;
  error= -1;					/* if goto err */

  /* Optimize distinct away if possible */
  if (select_distinct && join.tables - join.const_tables == 1)
  {						// No groups and no sum
    select_distinct=0;
    group=add_all_fields_to_order(&join,order,fields);
    order=0;
  }
  order=remove_const(&join,order,&simple_order);
  group=remove_const(&join,group,&simple_group);

  calc_group_buffer(&join,group);
  join.send_group_parts=join.group_parts;	/* Save org parts */
  if (procedure && procedure->group)
  {
    group=procedure->group=remove_const(&join,procedure->group,&simple_group);
    calc_group_buffer(&join,group);
  }

  if (test_if_subpart(group,order) || (!group && join.sum_func_count))
    order=0;

  need_tmp= (join.const_tables != join.tables &&
	     (select_distinct || !simple_order || !simple_group) ||
	     (group && order));
  make_join_readinfo(&join,group && simple_group && !need_tmp);
  DBUG_EXECUTE("info",TEST_join(&join););

  /* Create a tmp table if distinct or if the sort is too complicated */
  if (need_tmp)
  {
    DBUG_PRINT("info",("Createing tmp table"));

    if (!(tmp_table = create_tmp_table(thd,&join,all_fields,
				      ((!simple_group && !procedure &&
					!(test_flags & TEST_NO_KEY_GROUP)) ?
				       group : (ORDER*) 0),
				       select_distinct,
				       group && simple_group)))
      goto err;
    /* if group or order on first table, sort first */
    if (group && simple_group)
    {
      DBUG_PRINT("info",("Sorting for group"));
      if (create_sort_index(&join.join_tab[join.const_tables],group,select,0L))
	goto err;
      make_sum_func_list(&join,all_fields);
      alloc_group_fields(&join,group);
      group=0;
    }
    else
    {
      make_sum_func_list(&join,all_fields);
      if (!group && ! tmp_table->distinct && (order && simple_order))
      {
	DBUG_PRINT("info",("Sorting for order"));
	if (create_sort_index(&join.join_tab[join.const_tables],order,select,
			      0L))
	  goto err;
	order=0;
      }
    }
    if (do_select(&join,(List<Item> *) 0,tmp_table,0))
      goto err;
    use_select=0;			/* All conds done */

    /* Change sum_fields reference to calculated fields in tmp_table */
    if (join.sort_and_group || tmp_table->group)
    {
      change_to_use_tmp_fields(all_fields);
      join.field_count+=join.sum_func_count+join.func_count;
      join.sum_func_count=join.func_count=0;
    }
    else
    {
      change_refs_to_tmp_fields(all_fields);
      join.field_count+=join.func_count;
      join.func_count=0;
    }

    if (tmp_table->group)
    {
      if (!order)
	order=group;			/* Sort by group */
      group=0;
    }

    /*
    ** If we have different sort & group then we must sort the data by group
    ** and copy it to another tmp table
    */

    if ((group && (!test_if_subpart(group,order))) ||
	test_if_special_field(order))
    {					/* Must copy to another table */
      TABLE *tmp_table2;
      DBUG_PRINT("info",("Creating group table"));

      /* Free first data from old join */
      join_free(&join);
      make_simple_join(&join,tmp_table,(SELECT*) 0);
      calc_group_buffer(&join,group);
      count_field_types(&join,all_fields);

      /* group data to new table */
      if (!(tmp_table2 = create_tmp_table(thd,&join,all_fields,(ORDER*) 0,
					  select_distinct,1)))
	goto err;
      if (group)
      {
	if (create_sort_index(join.join_tab,group,select,0L))
	{
	  free_tmp_table(tmp_table2);
	  goto err;
	}
	alloc_group_fields(&join,group);
	group=0;
      }
      make_sum_func_list(&join,all_fields);	// Is this possible ?
      if (do_select(&join,(List<Item> *) 0,tmp_table2,0))
	goto err;

      free_tmp_table(tmp_table);
      join.const_tables=join.tables;		// Mark free for join_free()
      tmp_table=tmp_table2;

      change_to_use_tmp_fields(all_fields);	/* No sum funcs anymore */
      join.field_count+=join.sum_func_count;
      join.sum_func_count=0;
    }

    if (tmp_table->distinct)
      select_distinct=0;			/* Each row is uniq */
    if (select_distinct && ! group)
    {
      if (remove_dupplicates(tmp_table))
	goto err;
      select_distinct=0;
    }
    tmp_table->form.reginfo.update=0;
    join_free(&join);				/* Free quick selects */
    make_simple_join(&join,tmp_table,(SELECT*) 0);
    calc_group_buffer(&join,group);
    count_field_types(&join,all_fields);
    join.thd=thd;				/* For send_fields */
  }
  if (procedure)
  {
    procedure->change_columns(fields);
    count_field_types(&join,fields);
  }
  if (group || join.sum_func_count ||
      (procedure && (procedure->flags & PROC_GROUP)))
  {
    alloc_group_fields(&join,group);
    setup_copy_fields(&join,fields);
    make_sum_func_list(&join,fields);
  }
  if (group || order)
  {
    DBUG_PRINT("info",("Sorting for send_fields"));
    if (create_sort_index(&join.join_tab[join.const_tables],
			  group ? group : order,
			  use_select,
			  (having || join.const_tables != join.tables - 1) ?
			  0L : join.thd->select_limit))
      goto err;
  }
  join.having=having;				// Actually a parameter
  error=do_select(&join,&fields,NULL,procedure);

err:
  join_free(&join);
  if (tmp_table)
    free_tmp_table(tmp_table);
  if (select)
    end_select(select);
  delete_dynamic(&keyuse);
  delete procedure;
  DBUG_RETURN(error);
}

/*****************************************************************************
**	Create JOIN_TABS, make a guess about the table types,
**	Approximate how many records will be used in each table
*****************************************************************************/


static void
make_join_statistics(JOIN *join,tname_t *tables,COND *conds,
		     DYNAMIC_ARRAY *keyuse_array)
{
  int error;
  uint i,index,table_count,const_count,found_ref,refs,const_bits,
       key,const_ref,eq_part;
  TABLE **table_vector,*table;
  FORM *form;
  JOIN_TAB *stat,*stat_end,*s;
  SELECT *select;
  KEYUSE *keyuse,*start_keyuse;
  DBUG_ENTER("make_join_statistics");

  table_count=join->tables;
  stat=(JOIN_TAB*) sql_calloc(sizeof(JOIN_TAB)*table_count);
  table_vector=(TABLE**) sql_alloc(sizeof(TABLE**)*table_count);
  select=0;

  stat_end=stat+table_count;
  const_bits=const_count=0;
  for (s=stat,i=0 ; tables ; s++,tables=tables->next,i++)
  {
    table_vector[i]=s->table=tables->table;
    ha_info(&s->table->form,2);		/* Get record count */
    if (s->table->form.system || s->table->form.keyfile_info.records <= 1L)
    {
      s->type=JT_SYSTEM;
      const_bits|=(1L << s->table->tablenr);
      set_position(join,const_count++,s->table->tablenr,(KEYUSE*) 0);
    }
  }
  VOID(update_ref_and_keys(keyuse_array,stat,conds));

  /* loop until no more const tables are found */
  do
  {
    found_ref=0;
    for (index=0,s=stat ; s < stat_end ; s++,index++)
    {
      keyuse=s->keyuse;
      table=s->table;
      if (const_bits & (1L << index))
	continue;				// Already const table

      /* check if table can be read by key or table only uses const refs */
      if (keyuse)
      {
	s->type= JT_REF;
	while (keyuse->table == table)
	{
	  start_keyuse=keyuse;
	  key=keyuse->key;
	  s->keys|= 1L << key;

	  refs=const_ref=eq_part=0;
	  do
	  {
	    if (!keyuse->field || 
		const_bits & (1L << keyuse->field->table->tablenr))
	      const_ref|= 1L << keyuse->keypart;
	    else
	      refs|=(1L << keyuse->field->table->tablenr);
	    eq_part|= 1L << keyuse->keypart;
	    keyuse++;
	  } while (keyuse->table == table && keyuse->key == key);

	  form= &table->form;
	  if (eq_part == PREV_BITS(form->key_info[key].key_parts) &&
	      !form->key_info[key].dupp_key)
	  {
	    if (const_ref == eq_part)
	    {					// Found everything for ref.
	      s->type=JT_CONST;
	      const_bits|=(1L << table->tablenr);
	      set_position(join,const_count++,table->tablenr,start_keyuse);
	    }
	    else
	      found_ref|= refs;			// Is const is theese are const
	  }
	}
      }
    }
  } while (const_bits & found_ref);

  /* Calc how many (possible) matched records in each table */

  for (s=stat ; s < stat_end ; s++)
  {
    if (s->type == JT_SYSTEM || s->type == JT_CONST)
    {
      s->found_records=s->records=1L;		/* System or ref */
      continue;
    }
    s->records=s->table->form.keyfile_info.records;
    s->found_records=s->records;		/* Here som good aprox */
    /* if (s->type == JT_EQ_REF)
      continue; */
    if (s->const_keys)
    {
      ulong records;
      if (!select)
	select=make_unireg_select(table_vector,table_count,s->table->tablenr,
				  0,conds,&error);
      records=get_quick_record_count(select,s->table->tablenr,
				     s->const_keys);
      if (records != (ulong) ~0L)
	s->found_records=records;
      else
	s->const_keys=0;			/* Don't use const key */
    }
  }
  if (select)
    end_select(select);

  /* Find best combination and return it */
  join->join_tab=stat;
  join->table= table_vector;
  join->const_tables=const_count;
  join->const_bits=const_bits;

  if (join->const_tables != join->tables)
    find_best_combination(join);
  else
    memcpy((gptr) join->best_positions,(gptr) join->positions,
	   sizeof(POSITION)*join->const_tables);
  get_best_combination(join);
  DBUG_VOID_RETURN;
}


/*****************************************************************************
**	check with keys are used and with tables references with tables
**	updates in stat:
**	  keys	     Bitmap of all used keys
**	  const_keys Bitmap of all keys with may be used with quick_select
**	  keyuse     Pointer to possibly keys
*****************************************************************************/

typedef struct key_field_t {		// Used when finding key fields
  FIELD		*field;
  Item		*val;			// May be empty if diff constant
  uint		level,const_level;
  bool		eq_func;
} KEY_FIELD;


/* merge new key definitions to old ones, remove those not used in both */

static KEY_FIELD *
merge_key_fields(KEY_FIELD *start,KEY_FIELD *new_fields,KEY_FIELD *end,
		 uint and_level)
{
  if (start == new_fields)
    return start;				// Impossible or
  if (new_fields == end)
    return start;				// No new fields, skipp all

  KEY_FIELD *first_free=new_fields;

  /* Mark all found fields in old array */
  for (; new_fields != end ; new_fields++)
  {
    for (KEY_FIELD *old=start ; old != first_free ; old++)
    {
      if (start->field == new_fields->field)
      {
	if (new_fields->val->type() == Item::FIELD_ITEM)
	{
	  if (old->val->eq(new_fields->val))
	    old->level=old->const_level=and_level;
	}
	else if (old->val->eq(new_fields->val) && old->eq_func &&
		 new_fields->eq_func)
	{
	  old->level=old->const_level=and_level;
	}
	else
	  old->const_level=and_level;		// Remember comp against const
      }
    }
  }
  /* Remove all not used items */
  for (KEY_FIELD *old=start ; old != first_free ; old++)
  {
    if (old->level != and_level && old->const_level != and_level)
    {						// Not used in all levels
      first_free--;
      if (old != first_free)
      {
	*old= *first_free;			// Remove old value
	continue;
      }
      break;
    }
  }
  return first_free;
}


static void
add_key_field(JOIN_TAB *stat,KEY_FIELD **key_fields,uint and_level,
	      FIELD *field,bool eq_func,Item *value)
{
  if (!(field->flags & PART_KEY_FLAG))
    return;					// Not a key. Skipp it
  // Mark as possible quick key
  uint tablenr=field->table->tablenr;
  stat[tablenr].keys|=field->key_parts;		// Add possible keys
  if (value->type() != Item::FIELD_ITEM &&
      value->type() != Item::STRING_ITEM &&
      value->type() != Item::REAL_ITEM &&
      value->type() != Item::INT_ITEM)
    return;

    /* Save the following cases:
       Field op constant
       Field LIKE constant where constant doesn't start with a wildcard
       field = field2 where field2 is in a different table
       */
  if ((value->type() == Item::FIELD_ITEM &&
       (((Item_field *) value)->field->table->tablenr == tablenr ||
	!eq_func)))
    return;
  (*key_fields)->field=field;
  (*key_fields)->eq_func=eq_func;
  (*key_fields)->val=value;
  (*key_fields)->level=(*key_fields)->const_level=and_level;
  (*key_fields)++;
}


static void
add_key_fields(JOIN_TAB *stat,KEY_FIELD **key_fields,uint *and_level,
	       COND *cond)
{
  if (cond->type() == Item_func::COND_ITEM)
  {
    List_iterator<Item> li(*((Item_cond*) cond)->argument_list());
    KEY_FIELD *org_key_fields= *key_fields;

    if (((Item_cond*) cond)->functype() == Item_func::COND_AND_FUNC)
    {
      Item *item;
      while ((item=li++))
	add_key_fields(stat,key_fields,and_level,item);
      for (; org_key_fields != *key_fields ; org_key_fields++)
      {
	if (org_key_fields->const_level == org_key_fields->level)
	  org_key_fields->const_level=org_key_fields->level= *and_level;
	else
	  org_key_fields->const_level= *and_level;
      }
    }
    else
    {
      (*and_level)++;
      add_key_fields(stat,key_fields,and_level,li++);
      Item *item;
      while ((item=li++))
      {
	KEY_FIELD *start_key_fields= *key_fields;
	(*and_level)++;
	add_key_fields(stat,key_fields,and_level,item);
	*key_fields=merge_key_fields(org_key_fields,start_key_fields,
				     *key_fields,++(*and_level));
      }
    }
    return;
  }
  /* If item is of type 'field op field/constant' add it to key_fields */

  if (cond->type() != Item::FUNC_ITEM)
    return;
  Item_func *cond_func= (Item_func*) cond;
  if (!cond_func->select_optimize())
    return;

  if (cond_func->arguments()[0]->type() == Item::FIELD_ITEM)
  {
    add_key_field(stat,key_fields,*and_level,
		  ((Item_field*) (cond_func->arguments()[0]))->field,
		  cond_func->functype() == Item_func::EQ_FUNC,
		  (cond_func->arguments()[1]));
  }
  if (cond_func->arguments()[1]->type() == Item::FIELD_ITEM &&
      cond_func->functype() != Item_func::LIKE_FUNC)
  {
    add_key_field(stat,key_fields,*and_level,
		  ((Item_field*) (cond_func->arguments()[1]))->field,
		  cond_func->functype() == Item_func::EQ_FUNC,
		  (cond_func->arguments()[0]));
  }
  return;
}

/*
**	Add all keys with uses 'field' for some keypart
**	If field->and_level != and_level then only mark key_part as const_part
*/

static void
add_key_part(JOIN_TAB *stat,DYNAMIC_ARRAY *keyuse_array,KEY_FIELD *key_field,
	     uint and_level)
{
  FIELD *field=key_field->field,*cmp_field;
  FORM *form= &field->table->form;
  KEYUSE keyuse;
  cmp_field= ((key_field->val->type() == Item::FIELD_ITEM) ?
	      ((Item_field*) key_field->val)->field : 0);

  for (uint key=0 ; key < form->keys ; key++)
  {
    bool test_eq=  (key_field->eq_func && key_field->level == and_level);
    if (test_eq || (1L << key) && field->key_parts)
    {
      uint key_parts= test_eq ? (uint) form->key_info[key].key_parts : 1;
      for (uint part=0 ; part <  key_parts ; part++)
      {
	if (field == form->key_info[key].key_part[part].field &&
	    (!cmp_field ||
	     cmp_field->packlength == field->packlength))
	{
	  if (test_eq)
	  {
	    keyuse.table= field->table;
	    keyuse.field= cmp_field;
	    keyuse.val =  key_field->val;
	    keyuse.key =  key;
	    keyuse.keypart=part;
	    VOID(insert_dynamic(keyuse_array,(gptr) &keyuse));
	  }
	  if (!cmp_field && part == 0)
	    stat[field->table->tablenr].const_keys |= (1 << key);
	}
      }
    }
  }
}


static int
sort_keyuse(KEYUSE *a,KEYUSE *b)
{
  if (a->table->tablenr != b->table->tablenr)
    return (int) (a->table->tablenr - b->table->tablenr);
  if (a->key != b->key)
    return (int) (a->key - b->key);
  if (a->keypart != b->keypart)
    return (int) (a->keypart - b->keypart);
  return test(a->field) - test(b->field);	// Place const first
}


static uint
update_ref_and_keys(DYNAMIC_ARRAY *keyuse,JOIN_TAB *stat,COND *cond)
{
  uint	and_level,i,tablenr,found_eq_constant;
  KEY_FIELD *key_fields,*end;

  if (!cond)
    return 0;
  key_fields=(KEY_FIELD*) my_alloca(sizeof(key_fields[0])*
				    current_thd->cond_count*2);
  and_level=0; end=key_fields;
  add_key_fields(stat,&end,&and_level,cond);
  VOID(init_dynamic_array(keyuse,sizeof(KEYUSE),(uint) (end-key_fields),20));

  /* fill keyuse with found key parts */
  for (KEY_FIELD *field=key_fields ; field != end ; field++)
    add_key_part(stat,keyuse,field,and_level);
  my_afree((gptr) key_fields);

  /*
  ** remove ref if there is a keypart which is a ref and a const
  ** remove keyparts without previous keyparts
  */
  if (keyuse->elements)
  {
    KEYUSE end,*prev,*save_pos,*use;

    qsort(keyuse->buffer,keyuse->elements,sizeof(KEYUSE),
	  (qsort_cmp) sort_keyuse);

    bzero(&end,sizeof(end));			/* Add for easy testing */
    VOID(insert_dynamic(keyuse,(gptr) &end));

    use=save_pos=dynamic_element(keyuse,0,KEYUSE*);
    prev=&end;
    found_eq_constant=0;
    for (i=0 ; i < keyuse->elements-1 ; i++,use++)
    {
      if (use->key == prev->key && use->table == prev->table)
      {
	if (prev->keypart+1 < use->keypart ||
	    prev->keypart == use->keypart && found_eq_constant)
	  continue;				/* remove */
      }
      else if (use->keypart != 0)		// First found must be 0
	continue;

      *save_pos= *use;
      prev=use;
      found_eq_constant= !use->field;
      tablenr=use->table->tablenr;
      if (!stat[tablenr].keyuse)
	stat[tablenr].keyuse= save_pos;		/* Save ptr to first use */
      save_pos++;
    }
    i=(uint) (save_pos-(KEYUSE*) keyuse->buffer);
    VOID(set_dynamic(keyuse,(gptr) &end,i));
    keyuse->elements=i;
  }
  return and_level;
}


/*****************************************************************************
**	Go through all combinations of not marked tables and find the one
**	which uses least records
*****************************************************************************/


static void
set_position(JOIN *join,uint index,uint table,KEYUSE *key)
{
  join->positions[index].index=table;
  join->positions[index].key=key;
  join->positions[index].records_read=1.0;	/* This is a const table */
  join->index[table]=index;
}


static void
find_best_combination(JOIN *join)
{
  DBUG_ENTER("find_best_combination");
  join->best_read=DBL_MAX;
  find_best(join,PREV_BITS(join->tables) & ~join->const_bits,
	    join->const_tables,1.0,0.0);
  DBUG_VOID_RETURN;
}


static void
find_best(JOIN *join,ulong rest_tables,uint index,double record_count,
	  double read_time)
{
  uint i,keypart;
  ulong rec,table_bit;
  KEYUSE *keyuse;
  double tmp;

  if (!rest_tables)
  {
    DBUG_PRINT("best",("read_time: %g  record_count: %g",read_time,
		       record_count));

    read_time+=record_count/10.0;		// 10 record reads = 1 compare?
    if (join->sort_by_table != (uint) ~0 &&
	join->positions[join->const_tables].index != join->sort_by_table)
      read_time+=record_count;			// We have to make a temp table
    if (read_time < join->best_read)
    {
      memcpy((gptr) join->best_positions,(gptr) join->positions,
	     sizeof(POSITION)*index);
      join->best_read=read_time;
    }
    return;
  }
  if (read_time+record_count/10.0 >= join->best_read)
    return;					/* Found better before */

  for (i=0,table_bit=1 ; table_bit <= rest_tables ; i++,table_bit<<=1)
  {
    if (rest_tables & table_bit)
    {
      JOIN_TAB *s=join->join_tab+i;
      double best,records;
      best=records=DBL_MAX;
      KEYUSE *best_key=0;

      if (s->keyuse && index != join->const_tables)
      {						/* Use key if possible */
	TABLE *table=s->table;
	KEYUSE *start_key=0;
	double best_records=DBL_MAX;

	/* Test how we can use keys */
	rec= s->records/10;			/* Assume 10 records/key */
	for (keyuse=s->keyuse ; keyuse->table == table ;)
	{
	  ulong found_part=0,found_ref=0;
	  uint key=keyuse->key;
	  KEY *keyinfo=table->form.key_info+key;

	  start_key=keyuse;
	  do
	  {
	    keypart=keyuse->keypart;
	    do
	    {
	      if (!keyuse->field ||
		  !(rest_tables & (1L << keyuse->field->table->tablenr)))
	      {
		found_part|=1L << keypart;
		if (keyuse->field)
		  found_ref|=(1L << join->index[keyuse->field->table->tablenr]);
	      }
	      if (keyuse->field &&
		  rec > keyuse->field->table->form.keyfile_info.records)
		rec=keyuse->field->table->form.keyfile_info.records;
	      keyuse++;
	    } while (keyuse->table == table && keyuse->key == key &&
		     keyuse->keypart == keypart);
	  } while (keyuse->table == table && keyuse->key == key);

	  /*
	  ** Assume that that each key matches a proportional part of table.
	  */
	  if (!found_part)
	    continue;				// Noting usable found
	  if (rec == 0)
	    rec=1L;
	  /*
	  ** Check if we found full key
	  */
	  if (found_part == PREV_BITS(keyinfo->key_parts))
	  {				/* use eq key */
	    if (!keyinfo->dupp_key)
	    {
	      tmp=prev_record_reads(join,found_ref);
	      records=1.0;
	    }
	    else
	    {
	      if (table->form.keyfile_info.rec_per_key &&
		  (records=table->form.keyfile_info.rec_per_key[key]))
	      {
		tmp=(double) (record_count*records);
	      }
	      else
	      {					// Prefere longer keys
		tmp=((double) s->records / (double) rec *
		     (1.0 +((double) table->form.max_key_length -
			    (double) keyinfo->key_length)/10.0));
		tmp=record_count*min((double) s->records,max(tmp,2.0));
		records=s->records/rec+1;
	      }
	    }
	  }
	  else
	  {
	    /*
	    ** Use as much key-parts as possible and a uniq key is better
	    ** than a not unique key
	    ** Set tmp to (previous record count) * (records / combination)
	    */
	    if (found_part & 1)
	    {
	      uint max_parts=keyinfo->key_parts+test(keyinfo->dupp_key);
	      ulong rec_per_key;
	      if (!table->form.keyfile_info.rec_per_key ||
		  !(rec_per_key=table->form.keyfile_info.rec_per_key[key]))
		rec_per_key=s->records/rec+1;

	      tmp=(double) rec_per_key*
		log_10[max_parts-max_part_bit(found_part)];
	      set_if_smaller(tmp,s->records);
	      records=(ulong) records;
	      tmp*=record_count;
	    }
	    else
	      tmp=best;				// Do nothing
	  }
	  if (tmp < best)
	  {
	    best=tmp;
	    best_records=records;
	    best_key=start_key;
	  }
	}
	records=best_records;
      }
      if (records >= s->found_records)
      {						/* Check full join */
	if (s->const_keys)
	  tmp=(double) s->found_records;
	else					/* Read through read_cache */
	  tmp=(double) (1+
			 (ulong) (s->records*
				  s->table->form.keyfile_info.mean_rec_length/
				  IO_SIZE))*2.0;
	tmp*=(1.0+floor((double) cache_record_length(join,rest_tables)*
			record_count/(double) JOIN_CACHE_SIZE));
	if (tmp < best)
	{
	  best=tmp;
	  records=s->found_records;
	  best_key=0;
	}
      }
      join->positions[index].records_read=(double) records;
      join->positions[index].key=best_key;
      join->positions[index].index= i;
      join->index[i]=index;
      find_best(join,rest_tables & ~table_bit,index+1,
		record_count*records,read_time+best);
      if (join->select_options & SELECT_STRAIGHT_JOIN)
	break;					// Don't test all combinations
    }
  }
}


/*
** Find how much space the prevous read not const tables takes in cache
*/

static uint
cache_record_length(JOIN *join,ulong rest_tables)
{
  uint length;
  ulong used_tables;
  JOIN_TAB *join_tab;
  FIELD  *field,*end;
  THD *thd=current_thd;

  length=0;
  if ((used_tables=PREV_BITS(join->tables) &
       ~(rest_tables | join->const_bits)))
  {
    for (join_tab=join->join_tab ; used_tables ; join_tab++, used_tables>>=1)
    {
      if (used_tables & 1)
      {
	if (!join_tab->used_fieldlength)
	{					/* Not calced yet */
	  uint null_fields,blobs,fields,rec_length;
	  null_fields=blobs=fields=rec_length=0;

	  for (field=join_tab->table->field,end=join_tab->table->field_end ;
	       field != end ;
	       field++)
	  {
	    if (field->query_id == thd->query_id)
	    {
	      uint flag=field->pack_flag;
	      fields++;
	      rec_length+=field->packlength;
	      if (f_is_blob(flag))
		blobs++;
	      if (f_maybe_null(flag))
		null_fields++;
	    }
	  }
	  if (null_fields)
	    rec_length+=(join_tab->table->form.null_fields+7)/8;
	  if (blobs)
	  {
	    uint blob_length=(uint) (join_tab->table->form.
				     keyfile_info.mean_rec_length-
				     (join_tab->table->form.reclength-
				      rec_length));
	    rec_length+=(uint) max(4,blob_length);
	  }
	  join_tab->used_fields=fields;
	  join_tab->used_fieldlength=rec_length;
	  join_tab->used_blobs=blobs;
	}
	length+=join_tab->used_fieldlength;
      }
    }
  }
  return length;
}


static double
prev_record_reads(JOIN *join,ulong found_ref)
{
  uint i;
  ulong bit;
  double found=1.0;

  for (i=0,bit=1L ; bit <= found_ref ; i++,bit<<=1)
  {
    if ((found_ref & bit) ||
	(!join->positions[i].key && !(bit & join->const_bits)))
      found*=join->positions[i].records_read;
  }
  return found;
}


/*****************************************************************************
**	Set up join struct according to best position.
**	Change tablenr to new order
*****************************************************************************/

static void
get_best_combination(JOIN *join)
{
  uint i,ref_count,key,length,tablenr;
  ulong used_tables;
  TABLE *table;
  JOIN_TAB *join_tab,*j,*org_join_tab;
  KEYUSE *keyuse;
  KEY *keyinfo;
  FIELD *from_field;
  uint table_count;

  table_count=join->tables;
  org_join_tab=join->join_tab;
  join->join_tab=join_tab=(JOIN_TAB*) sql_alloc(sizeof(JOIN_TAB)*table_count);
  join->const_tables=0;				/* for checking */

  used_tables=0;
  for (j=join_tab, tablenr=0 ; tablenr < table_count ; tablenr++,j++)
  {
    FORM *form;
    *j= org_join_tab[join->best_positions[tablenr].index];
    used_tables|=(1L << j->table->tablenr);
    form=join->form[tablenr]= &j->table->form;
    join->table[tablenr]=j->table;
    form->reginfo.ref_fields=0;
    form->reginfo.ref_key= -1;

    if (j->type == JT_SYSTEM)
    {
      join->const_tables++;
      continue;
    }
    if (!j->keys || !(keyuse= join->best_positions[tablenr].key))
    {
      j->type=JT_ALL;
    }
    else
    {
      uint keyparts;
      /*
      ** Use best key from find_best
      */
      table=j->table;
      key=keyuse->key;

      keyparts=length=0;
      keyinfo=table->form.key_info+key;
      do
      {
	if ((!keyuse->field ||
	     (used_tables & (1L << keyuse->field->table->tablenr))))
	{
	  if (keyparts == keyuse->keypart)
	  {
	    keyparts++;
	    length+=keyinfo->key_part[keyuse->keypart].length;
	  }
	}
	keyuse++;
      } while (keyuse->table == table && keyuse->key == key);


      /* set up fieldref */
      keyinfo=table->form.key_info+key;
      form->reginfo.ref_fields=keyparts;
      form->reginfo.ref_length=length;
      form->reginfo.ref_key=(int) key;
      form->reginfo.key_buff= (char*) sql_calloc(length);
      ref_count=0;
      keyuse=join->best_positions[tablenr].key;
      for (i=0 ; i < keyparts ; keyuse++,i++)
      {
	while (keyuse->keypart != i)
	  keyuse++;				/* Skipp other parts */
	if (!keyuse->field)
	{
	  FIELD *field=keyinfo->key_part[i].field;
	  store_val_in_field(field,keyuse->val);
	  from_field=(FIELD*) sql_memdup((gptr) field,sizeof(FIELD));
	  from_field->table_name=0;		/* Marker for check_select */
	  length=packlength(field);
	  from_field->str=(char*) sql_alloc(length);
	  memcpy(from_field->str,field->str,length);
	}
	else
	{
	  ref_count++;
	  from_field=keyuse->field;
	}
	form->reginfo.ref_field[i].field=from_field;
      }
      if (j->type == JT_CONST)
      {
	join->const_tables++;
      }
      else if (keyinfo->dupp_key || keyparts != keyinfo->key_parts)
	j->type=JT_REF;				/* Must read with repeat */
      else if (!ref_count)
      {						/* Should never be reached */
	j->type=JT_CONST;
	join->const_tables++;
      }
      else
	j->type=JT_EQ_REF;
    }
  }

  for (i=0 ; i < table_count ; i++)
    join->table[i]->tablenr=i;
  return;
}

/*
** This function is only called for const items
*/

static void
store_val_in_field(FIELD *field,Item *val)
{
  if (val->null_value)
    f_fieldinit(field->str,field->pack_flag,field->length);
  else if (val->result_type() == Item::STRING_RESULT)
  {
    String *result=val->str((String*) 0);
    packf(result->ptr(),field);
  }
  else
    packnrf(val->val(),field);
}


static void
make_simple_join(JOIN *join,TABLE *tmp_table,SELECT *select)
{
  FORM **formptr;
  TABLE **tableptr;
  JOIN_TAB *join_tab;

  formptr= (FORM**)  sql_alloc(sizeof(FORM*));
  tableptr=(TABLE**) sql_alloc(sizeof(TABLE*)); 
  join_tab=(JOIN_TAB*) sql_alloc(sizeof(JOIN_TAB));
  join->join_tab=join_tab;
  join->table=tableptr; tableptr[0]=tmp_table;
  join->form=formptr; formptr[0]= &tmp_table->form;
  join->tables=1;
  join->const_tables=0;
  join->copy_field_count=join->field_count=join->sum_func_count=
    join->func_count=0;
  join->first_record=join->sort_and_group=0;
  join->sum_funcs=0;
  join->send_records=0L;

  join_tab->cache.buff=0;			/* No cacheing */
  join_tab->table=tmp_table;
  join_tab->select=select;
  join_tab->type= JT_ALL;			/* Map through all records */
  join_tab->keys= ~0;				/* test everything in quick */
  tmp_table->form.status=0;
  join_tab->read_first_record= join_init_read_record;
}


#ifndef DBUG_OFF

static string join_type_str[]={ "UNKNOWN","system","const","eq_ref","ref",
				"MAYBE_REF","ALL" };
static void
TEST_join(JOIN *join)
{
  uint i,ref,regnr;
  DBUG_ENTER("TEST_join");

  VOID(fputs("\nInfo about JOIN\n",DBUG_FILE));
  for (i=0 ; i < join->tables ; i++)
  {
    FORM *form=join->form[i];

    fprintf(DBUG_FILE,"%-16.16s  type: %-7s  q_keys: %4d  refs: %d  key: %d  len: %d\n",
	    form->reginfo.alias,
	    join_type_str[join->join_tab[i].type],
	    join->join_tab[i].keys,
	    form->reginfo.ref_fields,
	    form->reginfo.ref_key,
	    form->reginfo.ref_length);
    if (join->join_tab[i].select)
    {
      if (join->join_tab[i].use_quick == 2)
	fprintf(DBUG_FILE,
		"                  quick select checked for each record (keys: %d  ref-keys: %d)\n",
		(int) join->join_tab[i].select->quick_keys,
		(int) join->join_tab[i].select->neaded_reg);
      else if (join->join_tab[i].select->quick)
	fprintf(DBUG_FILE,"                  quick select used on key %d\n",
		join->join_tab[i].select->quick->index);
      else
	VOID(fputs("                  select used\n",DBUG_FILE));
    }
    if (form->reginfo.ref_fields)
    {
      VOID(fputs("                  refs: ",DBUG_FILE));
      for (ref=0 ; ref < form->reginfo.ref_fields ; ref++)
      {
	char buff[MAX_FIELD_WIDTH];
	FIELD *field=form->reginfo.ref_field[ref].field;

	if (field->table_name)
	{
	  field_name_or_number(buff,&field->table->form,
			       (uint) (field - field->table->form.field));
	  fprintf(DBUG_FILE,"#%s.%s  ", field->table_name,buff);
	}
	else
	{
	  VOID(r_unpackf_stripp(field,buff));
	  fprintf(DBUG_FILE,"'%s' ",buff);
	}
      }
      VOID(fputc('\n',DBUG_FILE));
    }
  }
  DBUG_VOID_RETURN;
}
#endif


static int
make_join_select(JOIN *join,SELECT *select)
{
  uint i,head=0,ref_key;
  SELECT_CHECK select_neaded;
  DBUG_ENTER("make_join_select");

  if (select)
  {
    head=min(join->const_tables,join->tables-1);
    select->reg_init_read=select->const_reg=PREV_BITS(head);
    select->head=select->forms[head];
    ref_key=(uint) select->head->form.reginfo.ref_key+1;
    if (join->const_tables == join->tables)
      ref_key= (uint) ~0;			/* No quick select */
    if (fix_select(select,ref_key))
      DBUG_RETURN(1);				/* impossible range */
    select_neaded=check_if_select_neaded(select);
    if (select_neaded.reg_ref & (1L << head))
    {
      select->reg_init_read|= (1L << head);
      select->reg_skipped= ~select->reg_init_read;
      join->join_tab[head].select=select;
    }
    for (i=join->const_tables ; i < join->tables ; i++)
    {
      JOIN_TAB *tab=join->join_tab+i;
      if (select_neaded.reg_ref & (1L << i) && i != head)
      {
	SELECT *sel;
	sel=tab->select=(SELECT*) sql_memdup((gptr) select,sizeof(SELECT));
	sel->quick=0;
	sel->head=sel->forms[i];
	sel->reg_init_read= PREV_BITS(i+1);
	if (tab->type == JT_ALL)
	{
	  VOID(test_quick_select(sel,tab->keys));
	  if (sel->quick_keys | sel->neaded_reg)
	  {
	    tab->keys=sel->quick_keys | sel->neaded_reg;
	    tab->use_quick= (sel->neaded_reg &&
			     (!select->quick_keys ||
			      (select->quick->records >= 20L))) ?
	      2 : 1;
	    sel->const_reg=sel->reg_init_read;
	  }
	  if (i != join->const_tables)
	  {					/* Read with cache */
	    if (select_neaded.const_ref & (1L << i))
	    {
	      tab->cache.select=(SELECT*) sql_memdup((gptr) sel,
						    sizeof(SELECT));
	      tab->cache.select->reg_init_read=PREV_BITS(join->const_tables) |
		(1L << i);
	      tab->cache.select->reg_skipped= ~tab->cache.select->reg_init_read;
	    }
	  }
	}
	else
	{
	  sel->const_reg=sel->reg_init_read;
	}
	sel->reg_skipped= ~sel->reg_init_read;
      }
    }
  }
  DBUG_RETURN(0);
}


static void
make_join_readinfo(JOIN *join,bool simple_group)
{
  uint i;
  bool use_cache;

  for (i=join->const_tables ; i < join->tables ; i++)
  {
    JOIN_TAB *tab=join->join_tab+i;
    tab->read_record.form= &tab->table->form;
    tab->next_select=sub_select;		/* normal select */
    switch (tab->type) {
    case JT_SYSTEM: break;
    case JT_CONST: break;
    case JT_EQ_REF:
      tab->table->form.status=STATUS_NO_RECORD;
      tab->read_first_record= join_read_key;
      tab->read_record.read_record= join_no_more_records;
      break;
    case JT_REF:
      tab->table->form.status=STATUS_NO_RECORD;
      tab->read_first_record= join_read_always_key;
      tab->read_record.read_record= join_read_next;
      break;
    case JT_ALL:
      /*
      ** if previous table use cache
      */
      use_cache=0;
      if (i != join->const_tables &&
	  (!simple_group || i != join->const_tables+1))
      {
	if (!join_init_cache(join->thd,join->join_tab+join->const_tables,
			    i-join->const_tables))
	{
	  tab[-1].next_select=sub_select_cache;	/* Patch previous */
	  use_cache=1;
	}
      }
      /* These init changes read_record */
      tab->table->form.status=0;
      if (tab->use_quick == 2 && !use_cache)
	tab->read_first_record= join_init_quick_read_record;
      else
	tab->read_first_record= join_init_read_record;
      break;
    default:
      DBUG_PRINT("error",("Table type %d found",tab->type));
      break;
    case JT_UNKNOWN:
    case JT_MAYBE_REF:
      abort();
    }
  }
  join->join_tab[join->tables-1].next_select=0; /* Set by do_select */
}


static void
join_free(JOIN *join)
{
  JOIN_TAB *tab,*end;

  if (join->tables > join->const_tables)
    free_io_cache(join->table[join->const_tables]); /* only sorted is cacheed */
  for (tab=join->join_tab,end=tab+join->tables ; tab != end ; tab++)
  {
    if (tab->select)
    {
      x_free(tab->select->quick);
      tab->select->quick=0;
    }
    x_free(tab->cache.buff);
    end_read_record(&tab->read_record);
  }
  // We are not using tables anymore
  // Unlock all tables. We may be in a an INSERT .... SELECT statement.
  if (join->lock)
  {
    mysql_unlock_tables(join->lock,1);		// Don't free join->lock
    join->lock=0;
  }
  join->group_fields.delete_elements();
  join->copy_funcs.delete_elements();
}


/*****************************************************************************
** Remove const table fields from sort order
** simple_order is set to 1 if sort_order only uses fields from head table
*****************************************************************************/

static ORDER *
remove_const(JOIN *join,ORDER *first_order,uint *simple_order)
{
  ORDER *order,**prev_ptr;

  *simple_order=1;
  if (join->const_tables == join->tables)
    return 0;					/* No nead to sort */

  prev_ptr=&first_order;
  for (order=first_order; order ; order=order->next)
  {
    if ((*order->item)->type() == Item::FIELD_ITEM)
    {
      FIELD *field=((Item_field*) (*order->item))->field;
      if (field->table->tablenr >= join->const_tables)
      {
	*prev_ptr= order;		/* use this */
	prev_ptr= &order->next;
	if (field->table->tablenr > join->const_tables)
	  *simple_order=0;
      }
    }
    else if (!const_item(*order->item))
    {
      *prev_ptr= order;		/* use this */
      prev_ptr= &order->next;
      *simple_order=0;
    }
  }
  *prev_ptr=0;
  return first_order;
}


static void
return_zero_rows(select_result *result,List<Item> &fields)
{
  DBUG_ENTER("return_zero_rows");
  result->send_fields(fields,1);
  result->send_eof();
  DBUG_VOID_RETURN;
}


/*****************************************************************************
** Make som simple condition optimization:
** If there is a test 'field = const' change all refs to 'field' to 'const'
** Remove all dummy tests 'item = item', 'const op const'
** Return in cond_value FALSE if condition is impossible (1 = 2)
*****************************************************************************/

static COND *
optimize_cond(COND *conds,enum cond_result *cond_value)
{
  uint brace_levels;
  if (!conds)
  {
    *cond_value= COND_TRUE;
    return conds;
  }

  /* change field = field to field = const for each found field = const */
  brace_levels=0;
  (void) count_select_levels(conds,&brace_levels);// Set up markers
  propagate_cond_constants(conds,&conds);
  /*
  ** Remove all instances of item == item
  ** Remove all and-levels where CONST item != CONST item
  */
  conds=remove_eq_conds(conds,cond_value) ;
  DBUG_EXECUTE("info",print_where(conds););
  return conds;
}


static void
propagate_cond_constants(COND *and_father,COND **cond_ref)
{
  Item *cond= *cond_ref;

  if (cond->type() == Item::COND_ITEM)
  {
    bool and_level= ((Item_cond*) cond)->functype() ==
      Item_func::COND_AND_FUNC;
    List_iterator<Item> li(*((Item_cond*) cond)->argument_list());
    Item *item;
    while ((item=li++))
    {
      propagate_cond_constants(and_level ? cond : item, li.ref());
    }
  }
  else if (and_father != cond)
  {
    if (cond->type() == Item::FUNC_ITEM &&
	((Item_func*) cond)->functype() == Item_func::EQ_FUNC &&
	const_item(((Item_func*) cond)->arguments()[1]) &&
	!cond->marker)
      change_cond_ref_to_const(and_father,&and_father,
			       ((Item_func*) cond)->arguments()[0],
			       ((Item_func*) cond)->arguments()[1]);
  }
}


/*
** change field = field to field = const for each found field = const in the
** and_level
*/

static void
change_cond_ref_to_const(COND *and_father,Item **cond_ref,Item *field,
			 Item *value)
{
  Item *cond= *cond_ref;

  if (cond->type() == Item::COND_ITEM)
  {
    bool and_level= ((Item_cond*) cond)->functype() ==
      Item_func::COND_AND_FUNC;
    List_iterator<Item> li(*((Item_cond*) cond)->argument_list());
    Item *item;
    while ((item=li++))
      change_cond_ref_to_const(and_level ? cond : item,li.ref(),field,value);
    return;
  }
  if (cond->type() != Item::FUNC_ITEM ||
      ((Item_func*) cond)->argument_count() != 2)
    return;

  Item *left_item=  ((Item_func*) cond)->arguments()[0];
  Item *right_item= ((Item_func*) cond)->arguments()[1];
  enum Item_func::Functype functype=  ((Item_func*) cond)->functype();

  if (right_item->eq(field) && !left_item->eq(field))
  {
    ((Item_func*) cond)->arguments()[1] = value;
    if (functype == Item_func::EQ_FUNC && !const_item(left_item))
    {
      cond->marker=1;
      change_cond_ref_to_const(and_father,&and_father,left_item,value);
    }
  }
  if (left_item->eq(field) && right_item != value)
  {
#ifdef CANT_BE_DONE
    delete left_item;				// This may be shared
#endif
    if (functype != Item_func::LIKE_FUNC &&
	functype != Item_func::NOTLIKE_FUNC)
    {						/* put const in item->value */
      ((Item_func*) cond)->arguments()[0]=right_item; /* Save compare field */
      ((Item_func*) cond)->arguments()[1]=value; // Change org field to const
      switch (functype) {			// Swap compare op
      case Item_func::LT_FUNC:
	((Item_func*) cond)->remove_arguments(); // Don't delete arguments
	delete cond;
	*cond_ref=new Item_func_gt(right_item,value);
	break;
      case Item_func::LE_FUNC:
	((Item_func*) cond)->remove_arguments(); // Don't delete arguments
	delete cond;
	*cond_ref=new Item_func_ge(right_item,value);
	break;
      case Item_func::GT_FUNC:
	((Item_func*) cond)->remove_arguments(); // Don't delete arguments
	delete cond;
	*cond_ref=new Item_func_lt(right_item,value);
	break;
      case Item_func::GE_FUNC:
	((Item_func*) cond)->remove_arguments(); // Don't delete arguments
	delete cond;
	*cond_ref=new Item_func_le(right_item,value);
	break;
      default:
	break;
      }
      if (functype == Item_func::EQ_FUNC && and_father != cond &&
	  !const_item(right_item))
      {
	cond->marker=1;
	change_cond_ref_to_const(and_father,&and_father,right_item,value);
      }
    }
    else
      ((Item_func*) cond)->arguments()[0]=value;
  }
}


/*
** remove const and eq items. Return new item, or NULL if no condition
** cond_value is set to according:
** COND_OK    query is possible (field = constant)
** COND_TRUE  always true	( 1 = 1 )
** COND_FALSE always false	( 1 = 2 )
*/

static COND *
remove_eq_conds(COND *cond,enum cond_result *cond_value)
{
  if (cond->type() == Item::COND_ITEM)
  {
    bool and_level= ((Item_cond*) cond)->functype()
      == Item_func::COND_AND_FUNC;
    List_iterator<Item> li(*((Item_cond*) cond)->argument_list());
    enum cond_result tmp_cond_value;

    *cond_value=COND_UNDEF;
    Item *item;
    while ((item=li++))
    {
      Item *new_item=remove_eq_conds(item,&tmp_cond_value);
      if (!new_item)
      {
#ifdef CANT_BE_DONE
	delete item;				// This may be shared
#endif
	li.remove();
      }
      else if (item != new_item)
      {
#ifdef CANT_BE_DONE
	delete item;				// This may be shared
#endif
	VOID(li.replace(new_item));
      }
      if (*cond_value == COND_UNDEF)
	*cond_value=tmp_cond_value;
      switch (tmp_cond_value) {
      case COND_OK:				// Not TRUE or FALSE
	if (and_level || *cond_value == COND_FALSE)
	  *cond_value=tmp_cond_value;
	break;
      case COND_FALSE:
	if (and_level)
	{
	  *cond_value=tmp_cond_value;
	  return (COND*) 0;			// Always false
	}
	break;
      case COND_TRUE:
	if (!and_level)
	{
	  *cond_value= tmp_cond_value;
	  return (COND*) 0;			// Always true
	}
	break;
      case COND_UNDEF:				// Impossible
	break;
      }
    }
    if (!((Item_cond*) cond)->argument_list()->elements ||
	*cond_value != COND_OK)
      return (COND*) 0;
    if (((Item_cond*) cond)->argument_list()->elements == 1)
    {						// Remove list
      Item *item= ((Item_cond*) cond)->argument_list()->head();
      ((Item_cond*) cond)->argument_list()->empty();
      return item;
    }
  }
  else
  {
    if (cond->type() == Item_func::FUNC_ITEM &&
	((Item_func*) cond)->argument_count() == 2)
    {
      Item *left_item=	((Item_func*) cond)->arguments()[0];
      Item *right_item= ((Item_func*) cond)->arguments()[1];
      enum Item_func::Functype functype=  ((Item_func*) cond)->functype();

      if (left_item->eq(right_item))
      {					/* op gives truth value */
	*cond_value= (functype == Item_func::EQ_FUNC ||
		      functype == Item_func::LE_FUNC ||
		      functype == Item_func::GE_FUNC) ? COND_TRUE : COND_FALSE;
	return (COND*) 0;
      }
      else if (const_item(left_item) && const_item(right_item))
      {
	*cond_value= eval_const_cond(cond) ? COND_TRUE : COND_FALSE;
	return (COND*) 0;
      }
    }
    *cond_value=COND_OK;
  }
  return cond;				/* Point at next and level */
}


/****************************************************************************
**	Create a temp table according to a field list
**	set distinct if dupplicates could be removed
**	given fields field pointers are changed to point at tmp_table
**	for send_fields
****************************************************************************/

static TABLE *
create_tmp_table(THD *thd,JOIN *join,List<Item> &fields,ORDER *group,
		 bool distinct,bool save_sum_fields)
{
  int error;
  TABLE *table;
  uint	i,field_count,reclength,old_reclength,null_count,blob_count;
  char	*tmpname,*tmpdir,path[FN_REFLEN];
  byte	*pos;
  uchar *null_flags;
  FIELD *reg_field,**from_field;
  COPY_FIELD *copy;
  FORM *form;
  N_RECINFO recinfo[2];
  KEY *keyinfo;
  KEY_PART_INFO *key_part_info;
  Item_result_field **copy_func;
  DBUG_ENTER("create_tmp_table");

  if (!(tmpdir=getenv("TMPDIR")))	/* Use this if possibly */
    tmpdir=P_tmpdir;			/* Use system default */
  sprintf(path,"%s/SQL%lx_%x",tmpdir,thd->query_id,thd->tmp_table++);
  tmpname=sql_strdup(path);
  if (join->group_parts > 16 || join->group_length >= 127 ||
      !join->quick_group)
    group=0;				/* Can't make a enough big key */

  field_count=join->field_count+join->func_count+join->sum_func_count;

  if (!my_multi_malloc(MYF(MY_WME),
		       &table,sizeof(TABLE),
		       &copy,sizeof(COPY_FIELD)*field_count,
		       &reg_field,sizeof(FIELD)*field_count,
		       &from_field,sizeof(FIELD*)*field_count,
		       &copy_func,sizeof(Item**)*(join->func_count+1),
		       &keyinfo,sizeof(KEY),
		       &key_part_info,sizeof(KEY_PART_INFO),
		       NullS))
  {
    DBUG_RETURN(NULL);
  }
  join->copy_field=copy;
  join->funcs=copy_func;
  table->field=reg_field;
  table->tmp_table = 1;				/* Mark as tmp table */
  table->tablenr=0;				/* Always head table */
  table->group=0;
  table->io_cache=0;
  table->distinct=0;

  /* make nisam database according to fields */

  form= &table->form;
  bzero(form,sizeof(*form));
  bzero(reg_field,sizeof(FIELD)*field_count);
  bzero(from_field,sizeof(FIELD*)*field_count);
  form->field=reg_field;
  form->reginfo.name=tmpname;
  table->table_name=form->reginfo.alias=base_name(tmpname);
  form->reginfo.update=1;		/* Will be updated */
  form->db_stat=HA_OPEN_KEYFILE+HA_OPEN_RNDFILE;
  form->db_record_offset=1;
  form->reginfo.ref_key= -1;		/* Not using key read */

  /* Calculate with type of fields we will nead in heap table */

  reclength=blob_count=null_count=field_count=0;

  List_iterator<Item> li(fields);
  while (Item *field=li++)
  {
    enum Item::Type type;
    Item *org_field=field;

    type=field->type();
    if (type == Item::SUM_FUNC_ITEM && ! group && !save_sum_fields)
    {						/* Can't calc group yet */
      ((Item_sum *)field)->result_field=reg_field;/* value is stored here */
      field=((Item_sum*) field)->item;
      type=field->type();
    }
    switch (type) {
    case Item::SUM_FUNC_ITEM:
    {
      Item_sum *item_sum=(Item_sum*) field;

      item_sum->result_field=reg_field;		/* Save result of func here */
      switch (item_sum->sum_func()) {
      case Item_sum::AVG_FUNC:			/* Place for sum & count */
	reg_field->type=FIELD_TYPE_DOUBLE;
	reg_field->pack_flag=0;			/* Alloc as binary */
	reg_field->length=sizeof(double)+sizeof(long);
	break;
      case Item_sum::COUNT_FUNC:
      case Item_sum::UNIQUE_USERS_FUNC:
      case Item_sum::INT_FUNC:
	reg_field->type=FIELD_TYPE_LONG;
	reg_field->pack_flag=FIELDFLAG_NUMBER | f_settype(FIELD_TYPE_LONG);
	reg_field->length=9;
	break;
      default:
	if (item_sum->result_type() == Item::REAL_RESULT)
	{
	  reg_field->type=FIELD_TYPE_DOUBLE;
	  reg_field->pack_flag=FIELDFLAG_DECIMAL | FIELDFLAG_NUMBER |
	    f_settype(FIELD_TYPE_DOUBLE) |
	    (item_sum->decimals << FIELDFLAG_DEC_SHIFT);
	  reg_field->length=(uint8) item_sum->max_length;
	}
	else
	{
	  reg_field->type=FIELD_TYPE_STRING;
	  reg_field->pack_flag=0;
	  reg_field->length= (uint8) item_sum->max_length;
	}
	if (item_sum->maybe_null)
	{
	  reg_field->pack_flag|=FIELDFLAG_MAYBE_NULL;
	  null_count++;
	}
	break;
      }
      reg_field->table_name="";
      reg_field->field_name=field->name;
      reg_field->flags=0;
      break;
    }
    case Item::FIELD_ITEM:
      memcpy(reg_field,((Item_field*) field)->field,sizeof(FIELD));
      ((Item_field*) field)->result_field=reg_field;
      from_field[field_count]=((Item_field*) field)->field;
      if (f_is_blob(reg_field->pack_flag))
	blob_count++;
      break;
    case Item::FUNC_ITEM:
    case Item::FIELD_AVG_ITEM:
      reg_field->table_name="";
      reg_field->field_name=field->name;
      reg_field->flags=0;
      *(copy_func++) = (Item_result_field*) field; // Save for copy_funcs
      ((Item_result_field*) field)->result_field=reg_field;
      reg_field->length=(uint8) (reg_field->length=field->max_length);

      if (field->result_type() == Item::REAL_RESULT)
      {
	reg_field->type=FIELD_TYPE_DOUBLE;
	reg_field->pack_flag=FIELDFLAG_DECIMAL | FIELDFLAG_NUMBER |
	  f_settype(FIELD_TYPE_DOUBLE) |
	  (field->decimals << FIELDFLAG_DEC_SHIFT);
      }
      else
      {
	reg_field->type=FIELD_TYPE_STRING;
	reg_field->pack_flag=0;
	reg_field->length=(uint8) field->max_length;
      }
      if (field->maybe_null)
      {
	reg_field->pack_flag|=FIELDFLAG_MAYBE_NULL;
	null_count++;
      }
      break;
    default:					// Dosen't have to be stored
      if (org_field->type() == Item::SUM_FUNC_ITEM)
	((Item_sum *) org_field)->result_field=0;
      continue;
    }
    reclength+=packlength(reg_field);
    if (f_maybe_null(reg_field->pack_flag))
      null_count++;
    reg_field++;
    field_count++;
  }

  /* If result table is small; use a heap */
  if (blob_count || thd->big_temporaries)
    form->db_type=DB_TYPE_ISAM;
  else
    form->db_type=DB_TYPE_HEAP;
  form->has_blob=test(blob_count);
  if (form->db_type == DB_TYPE_ISAM)
    reclength++;				/* For delete link */
  old_reclength=reclength;
  reclength+=(null_count+7)/8;

  form->fields=field_count;
  form->reclength=reclength;
  form->record[0]= (char *) my_malloc(reclength*3, MYF(MY_FAE));
  form->record[1]= form->record[0]+reclength;
  form->record[2]= form->record[1]+reclength;
  null_flags=(uchar*) form->record[0]+old_reclength;
  copy_func[0]=0;				// End marker

  pos=form->record[0] + test(form->db_type == DB_TYPE_ISAM);
  null_count=0;
  for (i=0,reg_field=form->field; i < field_count; i++,reg_field++)
  {
    uint length=packlength(reg_field);
    reg_field->str=pos;
    reg_field->packlength=length;
    reg_field->table=table;
    f_fieldinit(pos+reclength*2,reg_field->pack_flag,reg_field->length);

    if (reg_field->type == FIELD_TYPE_DOUBLE && reg_field->pack_flag == 0)
    {						/* Change AVG to number */
      reg_field->pack_flag=FIELDFLAG_DECIMAL | FIELDFLAG_NUMBER |
	f_settype(FIELD_TYPE_DOUBLE) | ( 2 << FIELDFLAG_DEC_SHIFT);
      reg_field->length=16;
    }
    reg_field->decimals=f_decimals(reg_field->pack_flag);
    if (from_field[i])
    {						/* Not a formula Item */
      FIELD *field=from_field[i];
      copy->from=field->str;			/* Set up for quick copy */
      copy->to=pos;
      copy->length=length;
      if (f_maybe_null(reg_field->pack_flag))
      {
	copy->from_bytepos=	(uchar*) field->null_pos;
	copy->from_bit=		(uint8)  field->null_bit;
	copy->to_bytepos=	(uchar*) null_flags+null_count/8;
	copy->to_bit=		(uint8)  (1 << (null_count & 7));
      }
      else
	copy->from_bytepos=	0;
      copy++;
    }
    if (f_maybe_null(reg_field->pack_flag))
    {
      reg_field->null_bit= 1 << (null_count & 7);
      reg_field->null_pos= null_flags+null_count/8;
      null_count++;
    }
    else
    {
      reg_field->flags|= NOT_NULL_FLAG;
      reg_field->null_pos=0;
    }
    pos+=length;
  }
  table->field_end=reg_field;
  join->copy_field_count=(uint) (copy - join->copy_field);

  form->record[2][0]=(char) 254;
  bfill(pos+reclength*2,(null_count+7)/8,255);
  recinfo[0].base.type=(int) FIELD_NORMAL;
  recinfo[0].base.length=(uint16) reclength;
  recinfo[1].base.type=(int) FIELD_LAST;

  form->max_records=((ulong) my_default_record_cache_size)*16/form->reclength;

  if (group)
  {
    table->group=group;				/* Table is grouped by key */
    join->group_buff=(char*) sql_alloc(join->group_length);
    key_part_info=(KEY_PART_INFO*) sql_alloc(sizeof(KEY_PART_INFO)*
					     join->group_parts);
    form->keys=1;
    form->key_info=keyinfo;
    keyinfo->key_part=key_part_info;
    keyinfo->dupp_key=0;
    keyinfo->usable_key_parts=keyinfo->key_parts= join->group_parts;
    keyinfo->key_length=0;
    for (; group ; group=group->next,key_part_info++)
    {
      reg_field=(*group->item)->tmp_table_field();
      key_part_info->field=  reg_field;
      key_part_info->offset= (uint16) (reg_field->str - form->record[0]);
      key_part_info->length= (uint) reg_field->packlength;
      key_part_info->key_type=0;		/* Binary keys are simple */
      keyinfo->key_length+=  key_part_info->length;
    }
  }
  if (distinct && !group && form->db_type == DB_TYPE_HEAP && reclength < 256)
  {
    table->distinct=distinct;
    form->keys=1;
    form->key_info=keyinfo;
    keyinfo->key_parts=1;
    keyinfo->dupp_key=0;
    keyinfo->key_length=(uint16) reclength;
    keyinfo->key_part=key_part_info;
    keyinfo->name="tmp";
    key_part_info->field=table->field;
    key_part_info->offset= 0;
    key_part_info->length= reclength;
    key_part_info->key_type=0;
  }
  if (form->db_type == DB_TYPE_ISAM)
  {
    if ((error=ni_create(tmpname,0,(N_KEYDEF*) 0,recinfo,0L,0L,0,0)))
    {
      ha_error(form,error,MYF(0));
      form->db_stat=0;
      free_tmp_table(table);
      DBUG_RETURN(NULL);
    }
  }
  if ((error=ha_open(form,tmpname,O_RDWR,HA_OPEN_WAIT_IF_LOCKED)))
  {
    ha_error(form,error,MYF(0));
    form->db_stat=0;
    free_tmp_table(table);
    DBUG_RETURN(NULL);
  }

  VOID(ha_lock(form,F_WRLCK));		/* Single thread table */
  VOID(ha_extra(form,HA_EXTRA_NO_READCHECK)); /* Not neaded */
  DBUG_RETURN(table);
}


static void
free_tmp_table(TABLE *entry)
{
  FORM *form;
  DBUG_ENTER("free_tmp_table");
  DBUG_PRINT("enter",("table: %s",entry->table_name));

  form= &entry->form;
  if (form->db_stat)
    VOID(ha_close(form));
  VOID(ha_fdelete(form->db_type,form->reginfo.name));
  my_free(form->record[0],MYF(0));
  free_io_cache(entry);
  my_free((gptr) entry,MYF(0));

  DBUG_VOID_RETURN;
}


/*****************************************************************************
**	Make a join of all tables and write it on socket or to table
*****************************************************************************/

static int
do_select(JOIN *join,List<Item> *fields,TABLE *table,Procedure *procedure)
{
  int error;
  JOIN_TAB *join_tab;
  int (*end_select)(struct st_join *,struct st_join_table *,bool);
  DBUG_ENTER("do_select");

  join->procedure=procedure;
  /*
  ** Tell the client how many fields there are in a row
  */
  if (!table)
    join->result->send_fields(*fields,1);
  else
  {
    VOID(ha_extra(&table->form,HA_EXTRA_WRITE_CACHE));
    restore_record(&table->form,2);		/* Make empty */
  }
  join->tmp_table=table;			/* Save for easy recursion */
  join->fields= fields;

  /* Set up select_end */
  if (table)
  {
    if (table->group)
      end_select=end_update;
    else if (join->sort_and_group)
      end_select=end_write_group;
    else
      end_select=end_write;
  }
  else
  {
    if (join->sort_and_group || (join->procedure &&
				 join->procedure->flags & PROC_GROUP))
      end_select=end_send_group;
    else
      end_select=end_send;
  }
  join->join_tab[join->tables-1].next_select=end_select;

  join_tab=join->join_tab+join->const_tables;
  join->send_records=0;
  if (join->tables == join->const_tables)
  {
    if (!(error=(*end_select)(join,join_tab,0)) || error == -3)
      error=(*end_select)(join,join_tab,1);
  }
  else
  {
    error=sub_select(join,join_tab,0);
    if (error >= 0)
      error=sub_select(join,join_tab,1);
    if (error == -3)
      error=0;					/* select_limit used */
  }
  if (!table)
  {
    if (error < 0)
      join->result->send_error(NullS);
    else
      join->result->send_eof();
  }
  if (error >= 0)
  {
    DBUG_PRINT("info",("%ld records output",join->send_records));
  }
  if (table)
    VOID(ha_extra(&table->form,HA_EXTRA_NO_CACHE));
  DBUG_RETURN(error < 0);
}


static int
sub_select_cache(JOIN *join,JOIN_TAB *join_tab,bool end_of_records)
{
  int error;

  if (end_of_records)
  {
    if ((error=flush_cacheed_records(join,join_tab,FALSE)) < 0)
      return error;
    return sub_select(join,join_tab,end_of_records);
  }
  if (join_tab->use_quick != 2 || test_if_quick_select(join_tab) <= 0)
  {
    if (!store_record_in_cache(&join_tab->cache))
      return 0;				/* There is more room in cache */
    return flush_cacheed_records(join,join_tab,FALSE);
  }
  if ((error=flush_cacheed_records(join,join_tab,TRUE)) < 0)
    return error;
  return sub_select(join,join_tab,end_of_records); /* Use ordinary select */
}


static int
sub_select(JOIN *join,JOIN_TAB *join_tab,bool end_of_records)
{
  int error;
  READ_RECORD *info;

  if (end_of_records)
    return (*join_tab->next_select)(join,join_tab+1,end_of_records);

  if ((error=(*join_tab->read_first_record)(join_tab)) < 0)
    return 1;					/* No records */
  info= &join_tab->read_record;
  do
  {
    if (abort_loop)
    {
      my_error(ER_SERVER_SHUTDOWN,MYF(0));
      return -2;				/* Aborted by user */
    }
    if (!error && !(join_tab->select && skipp_record(join_tab->select)))
      if ((error=(*join_tab->next_select)(join,join_tab+1,0)) < 0)
	return error;
  } while ((error=info->read_record(info)) <= 0);
  if (error != HA_ERR_END_OF_FILE)
  {
    ha_error(info->form,error,MYF(0));
    return -1;
  }
  return 0;
}


static int
flush_cacheed_records(JOIN *join,JOIN_TAB *join_tab,bool skipp_last)
{
  int error;
  READ_RECORD *info;

  if (!join_tab->cache.records)
    return 0;				/* Nothing to do */
  if (skipp_last)
    (void) store_record_in_cache(&join_tab->cache); // Must save this for later
  if (join_tab->use_quick == 2)
  {
    if (join_tab->select->quick)
    {					/* Used quick select last. reset it */
      my_free((gptr) join_tab->select->quick,MYF(0));
      join_tab->select->quick=0;
    }
  }
 /* read through all records */
  if ((error=join_init_read_record(join_tab)) < 0)
    return 1;				/* No records (not fatal) */

  info= &join_tab->read_record;
  do
  {
    SELECT *select=join_tab->select;
    if (!error && (!join_tab->cache.select ||
		   !skipp_record(join_tab->cache.select)))
    {
      uint i;
      reset_cache(&join_tab->cache);
      for (i=(join_tab->cache.records- (skipp_last ? 1 : 0)) ; i-- > 0 ;)
      {
	read_cacheed_record(join_tab);
	if (!select || !skipp_record(select))
	  if ((error=(join_tab->next_select)(join,join_tab+1,0)) < 0)
	    return error;
      }
    }
  } while ((error=info->read_record(info)) <= 0);
  if (skipp_last)
    read_cacheed_record(join_tab);	/* Restore current record */
  reset_cache(&join_tab->cache);
  join_tab->cache.records=0; join_tab->cache.ptr_record= (uint) ~0;
  if (error != HA_ERR_END_OF_FILE)
  {
    ha_error(info->form,error,MYF(0));
    return -1;
  }
  return 0;
}


/*****************************************************************************
**	The different ways to read a record
*****************************************************************************/

static int
join_read_const_tables(JOIN *join)
{
  uint i;

  DBUG_ENTER("join_read_const_tables");
  for (i=0 ; i < join->const_tables ; i++)
  {
    FORM *form=join->form[i];

    if (join->join_tab[i].type == JT_SYSTEM)
    {
      if (ha_readfirst(form,form->record[0]))
	DBUG_RETURN(1);
    }
    else
    {
      VOID(cmp_buffer_with_ref(form->reginfo.key_buff,
				  form->reginfo.ref_field,
				  form->reginfo.ref_length)); /* Copy fields */
      if (ha_rkey(form,form->record[0],form->reginfo.ref_key,
		  form->reginfo.key_buff,
		  form->reginfo.ref_length,HA_READ_KEY_EXACT))
	DBUG_RETURN(1);
    }
    form->status=0; form->found=1;
  }
  DBUG_RETURN(0);
}


static int
join_read_key(JOIN_TAB *tab)
{
  FORM *form= &tab->table->form;

  if (cmp_buffer_with_ref(form->reginfo.key_buff,
			  form->reginfo.ref_field,
			  form->reginfo.ref_length) ||
      (form->status & (STATUS_GARBAGE | STATUS_NO_PARENT)))
  {
    if (ha_rkey(form,form->record[0],form->reginfo.ref_key,
		form->reginfo.key_buff,
		form->reginfo.ref_length,HA_READ_KEY_EXACT))
    {
      form->found=0;
      form->status= STATUS_NOT_FOUND;
    }
    else
    {
      form->found=1;
      form->status=0;
    }
  }
  return form->found ? 0 : -1;
}


static int
join_read_always_key(JOIN_TAB *tab)
{
  FORM *form= &tab->table->form;

  VOID(cmp_buffer_with_ref(form->reginfo.key_buff,
			   form->reginfo.ref_field,
			   form->reginfo.ref_length)); /* Copy fields */
  if (ha_rkey(form,form->record[0],form->reginfo.ref_key,
	      form->reginfo.key_buff,
	      form->reginfo.ref_length,HA_READ_KEY_EXACT))
  {
    form->found=0;
    form->status= STATUS_NOT_FOUND;
    return -1;
  }
  else
  {
    form->found=1;
    form->status=0;
    return 0;
  }
}


	/* ARGSUSED */
static int
join_no_more_records(READ_RECORD *info __attribute__((unused)))
{
  return HA_ERR_END_OF_FILE;
}


static int
join_read_next(READ_RECORD *info)
{
  FORM *form= info->form;
  if (ha_rnext(form,form->record[0],form->reginfo.ref_key) ||
      key_cmp(form,form->reginfo.key_buff,(uint) form->reginfo.ref_key,
	      form->reginfo.ref_length))
  {
    form->found=0;
    form->status= STATUS_GARBAGE;
    return HA_ERR_END_OF_FILE;
  }
  return 0;
}


static int
join_init_quick_read_record(JOIN_TAB *tab)
{
  if (test_if_quick_select(tab) == -1)
    return -1;					/* No possible records */
  return join_init_read_record(tab);
}


static int
test_if_quick_select(JOIN_TAB *tab)
{
  if (tab->select->quick)
  {
    my_free((gptr) tab->select->quick,MYF(0));
    tab->select->quick=0;
  }
  return test_quick_select(tab->select,tab->keys);
}


static int
join_init_read_record(JOIN_TAB *tab)
{
  int result;
  if (tab->select && tab->select->quick)
  {
    tab->select->quick->next=0;
    tab->select->quick->pos=tab->select->quick->region;
  }
  init_read_record(&tab->read_record,&tab->table->form,0,tab->table->io_cache,
		   tab->select);
  result=(*tab->read_record.read_record)(&tab->read_record);
  return result == HA_ERR_END_OF_FILE ? -1 : (result ? 1 : 0);
}


/*****************************************************************************
**  The different end of select functions
*****************************************************************************/

	/* ARGSUSED */
static int
end_send(JOIN *join, JOIN_TAB *join_tab __attribute__((unused)),
	 bool end_of_records)
{
  if (!end_of_records)
  {
    int error;
    if (join->procedure)
      error=join->procedure->send_row(*join->fields);
    else
    {
      if (join->having && join->having->val() == 0.0)
	return 0;				// Didn't match having
      error=join->result->send_data(*join->fields);
    }
    if (error)
      return -1;
    if (++join->send_records == join->thd->select_limit)
      return -3;				// Abort nicely
  }
  return 0;
}


	/* ARGSUSED */
static int
end_send_group(JOIN *join, JOIN_TAB *join_tab __attribute__((unused)),
	       bool end_of_records)
{
  int index= -1;

  if (!join->first_record || end_of_records ||
      (index=test_if_group_changed(join->group_fields)) >= 0)
  {
    if (join->first_record)
    {
      if (join->procedure)
	join->procedure->end_group();
      if (index < (int) join->send_group_parts)
      {
	int error;
	if (join->procedure)
	  error=join->procedure->send_row(*join->fields) ? 1 : 0;
	else
	{
	  if (join->having && join->having->val() == 0.0)
	    error= -1;				// Didn't satisfy having
	  else
	    error=join->result->send_data(*join->fields) ? 1 : 0;
	}
	if (error > 0)
	  return -1;
	if (end_of_records)
	  return 0;
	if (!error && ++join->send_records == join->thd->select_limit)
	  return -3;				/* Abort nicely */
      }
    }
    else
    {
      if (end_of_records)
	return 0;
      join->first_record=1;
      VOID(test_if_group_changed(join->group_fields));
    }
    if (index < (int) join->send_group_parts)
    {
      copy_fields(join);
      init_sum_functions(join->sum_funcs);
      if (join->procedure)
	join->procedure->add();
      return(0);
    }
  }
  update_sum_func(join->sum_funcs);
  if (join->procedure)
    join->procedure->add();
  return(0);
}


	/* ARGSUSED */
static int
end_write(JOIN *join, JOIN_TAB *join_tab __attribute__((unused)),
	  bool end_of_records)
{
  TABLE *table=join->tmp_table;
  int error;
  if (!end_of_records)
  {
    copy_fields(join);
    copy_funcs(join->funcs);
    if ((error=ha_write(&table->form,table->form.record[0])))
    {
      if (error != HA_ERR_FOUND_DUPP_KEY || !table->distinct)
      {
	ha_error(&table->form,error,MYF(0));
	return -1;
      }
    }
    else
      join->send_records++;
  }
  return 0;
}

	/* Group by searching after group record and updating it if possible */
	/* ARGSUSED */

static int
end_update(JOIN *join, JOIN_TAB *join_tab __attribute__((unused)),
	   bool end_of_records)
{
  TABLE *table=join->tmp_table;
  ORDER   *group;
  char	  *buff=join->group_buff;
  int	  error;

  if (end_of_records)
    return 0;
  copy_fields(join);
  /* Make a key of group index */
  for (group=table->group ; group ; group=group->next)
  {
    FIELD *result_field=(*group->item)->tmp_table_field();
    char *save=result_field->str;
    result_field->str=buff;
    (void) (*group->item)->save_in_field(result_field);
    result_field->str=save;
    buff+=result_field->packlength;
  }
  if (!ha_rkey(&table->form,table->form.record[1],0,join->group_buff,0,
	       HA_READ_KEY_EXACT))
  {						/* Update old record */
    restore_record(&table->form,1);
    update_tmptable_sum_func(join->sum_funcs,table);
    if ((error=ha_update(&table->form,table->form.record[1],
			 table->form.record[0])))
    {
      ha_error(&table->form,error,MYF(0));
      return -1;
    }
    return 0;
  }
  key_restore(&table->form,join->group_buff,0,join->group_length);
  init_tmptable_sum_functions(join->sum_funcs);
  if ((error=ha_write(&table->form,table->form.record[0])))
  {
    ha_error(&table->form,error,MYF(0));
    return -1;
  }
  join->send_records++;
  return 0;
}


	/* ARGSUSED */
static int
end_write_group(JOIN *join, JOIN_TAB *join_tab __attribute__((unused)),
		bool end_of_records)
{
  TABLE *table=join->tmp_table;
  int	  error;
  int	  index= -1;

  if (!join->first_record || end_of_records ||
      (index=test_if_group_changed(join->group_fields)) >= 0)
  {
    if (join->first_record)
    {
      if (join->procedure)
	join->procedure->end_group();
      if (index < (int) join->send_group_parts)
      {
	copy_sum_funcs(join->sum_funcs);
	if ((error=ha_write(&table->form,table->form.record[0])))
	{
	  ha_error(&table->form,error,MYF(0));
	  return -1;
	}
	join->send_records++;
	if (end_of_records)
	  return 0;
      }
    }
    else
    {
      if (end_of_records)
	return 0;
      join->first_record=1;
      VOID(test_if_group_changed(join->group_fields));
    }
    if (index < (int) join->send_group_parts)
    {
      copy_fields(join);
      copy_funcs(join->funcs);
      init_sum_functions(join->sum_funcs);
      if (join->procedure)
	join->procedure->add();
      return(0);
    }
  }
  update_sum_func(join->sum_funcs);
  if (join->procedure)
    join->procedure->add();
  return(0);
}


/*****************************************************************************
**	Test if there are any nead to call select when some table is read
**	through key.
**	return a bit set for each table which uses a test agains previous
**	table or constant
**	const_select is set for each table which uses tests again constants
*****************************************************************************/

static bool
only_one_bit(ulong bits)
{
  while (bits)
  {
    if (bits & 1)
      return (bits & ~1L) == 0L;
    bits>>=1;
  }
  return FALSE;
}

static uint
max_part_bit(ulong bits)
{
  uint found;
  for (found=0; bits & 1 ; found++,bits>>=1) ;
  return found;
}

	/* Return only the max bit of bits */
static uint
max_bit(ulong bits)
{
  uint found;
  ulong org_bits=bits;
  for (found=0; (bits= bits>>1) ; found++) ;
  return org_bits & (1L << found);
}


static SELECT_CHECK
check_if_select_neaded(SELECT *select)
{
  uint flag,res;
  uint level,regnr,optflag;
  bool const_cmp;
  SELECT_ROW *row;
  FIELD *field,*ref_field;
  SELECT_CHECK *result_stack,result;
  DBUG_ENTER("check_if_select_neaded");

  row=select->row;
  level=0;
  result_stack=(struct st_select_check*)
    my_malloc((select->max_brace_level*2+3)*sizeof(SELECT_CHECK),MYF(MY_FAE));
  select->stack[0].flag=0;
  result_stack[0].const_ref=result_stack[0].reg_ref=0;

  while ((flag=row->flag))
  {
    if (flag & SELECT_FLAG_RIGHT_BRACE)
    {
      if (select->stack[level].flag == SELECT_FLAG_OR)
      {
	level--;
	result_stack[level].const_ref&= result_stack[level+1].const_ref;
	result_stack[level].reg_ref&=	result_stack[level+1].reg_ref;
      }
      if (level)
      {						/* if not lonely ')' */
	if (select->stack[level].flag & SELECT_FLAG_AND)
	{
	  level--;
	  result_stack[level].const_ref|= result_stack[level+1].const_ref;
	  result_stack[level].reg_ref|=   result_stack[level+1].reg_ref;
	}
	else
	  select->stack[level].flag=SELECT_FLAG_OR;
      }
    }
    if (flag & SELECT_FLAG_OR && select->stack[level].flag == SELECT_FLAG_OR)
    {
      level--;
      result_stack[level].const_ref=
	(result_stack[level].const_ref & result_stack[level+1].const_ref) |
	max_bit(result_stack[level].const_ref |
		result_stack[level+1].const_ref);
      result_stack[level].reg_ref=
	(result_stack[level].reg_ref & result_stack[level+1].reg_ref) |
	(max_bit(result_stack[level].reg_ref |
		 result_stack[level+1].reg_ref));
    }

    if (flag & SELECT_FLAG_LEFT_BRACE)
    {						/* '(' found */
      select->stack[++level].flag= flag;
      result_stack[level].const_ref=result_stack[level].reg_ref = 0;
    }

    if ((flag & SELECT_FLAG_NO_FIELDS) == 0)
    {						/* Not Empty level */
      res= 1;
      const_cmp=only_one_bit(row->reg_used & ~select->const_reg);

      optflag= flag & SELECT_FLAG_OP;
      if (row->field_type != SEL_FIELD_FIELD ||
	  (row->diff_type != SEL_FIELD_FIELD &&
	   row->diff_type != SEL_FIELD_CONST_STR &&
	   row->diff_type != SEL_FIELD_CONST_NUM) ||
	  optflag != 1)
	goto skipp;				/* Can't remove test */

      /* Check if this is a reference through key */
      regnr=(uint) row->field.field.regnr;
      field=row->field.field.regfield;
      ref_field=part_of_refkey(&select->forms[regnr]->form,field);
      if (row->diff_type == SEL_FIELD_FIELD)
      {
	if (ref_field == row->diff.field.regfield)
	  res=0;
	else
	{
	  if ((ref_field=part_of_refkey(&select->forms[row->diff.field.regnr]->form,
					row->diff.field.regfield)))
	  {
	    if (ref_field == field)
	      res=0;
	  }
	}
      }
      else
      {
	if (ref_field && !ref_field->table_name) /* flag for const field */
	{
	  if (row->diff_type == SEL_FIELD_CONST_STR)
	  {
	    char buff[MAX_FIELD_WIDTH];
	    VOID(r_unpackf_stripp(ref_field,buff));
	    if (row->cmp_type == SEL_CMP_BINARY_STR)
	    {
	      if (strcmp(buff,row->diff.str))
		goto skipp;
	    }
	    else if (my_strcasecmp(buff,row->diff.str))
	      goto skipp;
	  }
	  else
	  {
	    if (valfield(field) != row->diff.nr)
	      goto skipp;
	  }
	  res=0;				/* Test can be removed */
	}
      }

skipp:						/* Here when wrong fieldref */
      if (flag & (SELECT_FLAG_AND | SELECT_FLAG_LEFT_BRACE))
      {
	if (res)
	{
	  result_stack[level].reg_ref|=     max_bit(row->reg_used);
	  if (const_cmp)
	    result_stack[level].const_ref|= row->reg_used;
	}
      }
      else
      {
	  select->stack[++level].flag= SELECT_FLAG_OR;
	  result_stack[level].reg_ref=	  (res ? max_bit(row->reg_used)
					   : 0L);
	  result_stack[level].const_ref=  (res && const_cmp) ? row->reg_used :
	    0;
      }
    }
    row++;
  }

  while (level-- > 0)				/* Test last levels */
  {						/* Some ')' is not written */
    if (select->stack[level].flag & SELECT_FLAG_AND)
    {
      result_stack[level].const_ref|= result_stack[level+1].const_ref;
      result_stack[level].reg_ref|=   result_stack[level+1].reg_ref;
    }
    else
    {
      result_stack[level].const_ref=
	(result_stack[level].const_ref & result_stack[level+1].const_ref) |
	max_bit(result_stack[level].const_ref |
		result_stack[level+1].const_ref);
      result_stack[level].reg_ref=
	(result_stack[level].reg_ref & result_stack[level+1].reg_ref) |
	max_bit(result_stack[level].reg_ref | result_stack[level+1].reg_ref);
    }
  }
  /* Swap bits to get bit set if select is neaded */
  result=result_stack[0];
  my_free((gptr) result_stack,MYF(0));
  DBUG_PRINT("exit",("nead select for:  %x  %x",result.reg_ref,
		     result.const_ref));
  DBUG_RETURN(result);
}



static FIELD *
part_of_refkey(FORM *form,FIELD *field)
{
  uint key,keypart;

  key=(uint) form->reginfo.ref_key;
  for (keypart=0 ; keypart < form->reginfo.ref_fields ; keypart++)
    if (field == form->key_info[key].key_part[keypart].field)
      return form->reginfo.ref_field[keypart].field;
  return (FIELD*) 0;
}


/*****************************************************************************
** If not selecting by given key, create a index how records should be read
** return: 0  ok
**	  -1 some fatal error
**	   1  no records
*****************************************************************************/

static int
create_sort_index(JOIN_TAB *tab,ORDER *first_order,SELECT *select,
		  ulong select_limit)
{
  SORT_FIELD *sortorder;
  ORDER *order;
  ulong records;
  int ref_key;
  uint length,ref_fields;
  FORM *form;
  KEY_PART_INFO *key_part,*key_part_end;
  TABLE *table=tab->table;
  DBUG_ENTER("create_sort_index");

  form= &table->form;
  ref_key= -1;
  ref_fields=0;
  if (form->reginfo.ref_key >= 0)
  {
    ref_key=form->reginfo.ref_key;
    ref_fields=form->reginfo.ref_fields;
  }
  else if (select && select->quick)
  {
    ref_key=select->quick->index;
    ref_fields=1;
  }
  if (ref_key >= 0)
  {						/* using select by key */
    key_part=form->key_info[ref_key].key_part;
    key_part_end=key_part+ref_fields;

    for (order=first_order; order ; order=order->next)
    {
      if ((*order->item)->type() == Item::FIELD_ITEM)
      {
	FIELD *field=((Item_field*) (*order->item))->field;
	if (key_part != key_part_end &&
	    key_part->field == field &&
	    order->asc == test(key_part->key_part_flag == 0))
	{
	  key_part++;
	  continue;
	}
      }
      break;
    }
    if (!order)
      DBUG_RETURN(0);			/* No nead to sort */
  }

  if (!(sortorder=make_unireg_sortorder(table,first_order,&length)))
    DBUG_RETURN(-1);
  table->io_cache=(IO_CACHE*) my_malloc(sizeof(IO_CACHE),
				      MYF(MY_FAE | MY_ZEROFILL));
  form->status=0;				// May be wrong if quick_select
  records=filesort(&table,sortorder,length,table->io_cache,select,0,0L,
		   select_limit);
  tab->select=0;				// filesort did select
  tab->type=JT_ALL;				// Read with normal read_record
  tab->read_first_record= join_init_read_record;
  DBUG_RETURN(records == (ulong) ~0L);
}


/*****************************************************************************
** Remove dupplicates from tmp table
** Table is a locked single thread table
*****************************************************************************/

static int
remove_dupplicates(TABLE *entry)
{
  READ_RECORD info;
  FORM *form;
  int error;
  DBUG_ENTER("remove_dupplicates");

  form= &entry->form;
  form->reginfo.update=1;
  VOID(ha_extra(form,HA_EXTRA_NO_READCHECK));
  init_read_record(&info,form,0,(IO_CACHE *) 0,(SELECT *) 0);

  while ((error=info.read_record(&info)) <= 0)
  {
    if (error == 0)
    {
      ha_info(form,1);
      store_record(form,1);
      while ((error=info.read_record(&info)) <= 0)
      {
	if (error == 0)
	{
	  if (memcmp(form->record[0],form->record[1],form->reclength) == 0)
	    if ((error=ha_delete(form,form->record[0])))
	    {
	      ha_error(form,error,MYF(0));
	      goto end;
	    }
	}
      }
      if (ha_r_rnd(form,form->record[0],form->keyfile_info.ref.refpos))
      {
	error= -1;
	goto end;
      }
      VOID(ha_extra(form,HA_EXTRA_REINIT_CACHE));
    }
  }
  if (error != HA_ERR_END_OF_FILE)
  {
    ha_error(form,error,MYF(0));
    end_read_record(&info);
    return -1;
  }
  error=0;
end:
  end_read_record(&info);
  DBUG_RETURN(error);
}

/* This must change when we allow filesort on formulas! */

static SORT_FIELD *
make_unireg_sortorder(TABLE *entry,ORDER *order, uint *length)
{
  uint count;
  SORT_FIELD *sort,*pos;
  DBUG_ENTER("make_unireg_sortorder");

  count=0;
  for (ORDER *tmp = order; tmp; tmp=tmp->next)
    count++;
  pos=sort=(SORT_FIELD*) sql_alloc(sizeof(SORT_FIELD)*(count+1));

  for (;order;order=order->next,pos++)
  {
    pos->regnr=0;
    if (order->item[0]->type() == Item::COPY_STR_ITEM)
    {						// Blob patch
      pos->fieldnr= (uint)
	(((Item_field*) (((Item_copy_string*)
			  (*order->item))->item))->field -
	 entry->form.field);
    }
    else if (order->item[0]->type() == Item::SUM_FUNC_ITEM)
    {
      pos->fieldnr= (uint) (((Item_sum*) order->item[0])->tmp_table_field()
			    - entry->form.field);
    }
    else
    {
      pos->fieldnr= (uint) (((Item_field*) (*order->item))->field -
			    entry->form.field);
    }
    pos->reverse=! order->asc;
  }
  *length=count;
  DBUG_RETURN(sort);
}


/*****************************************************************************
**	Fill join cache with packed records
**	Records are stored in tab->cache.buffer and last record in
**	last record is stored with pointers to blobs to support very big
**	records
******************************************************************************/

static int
join_init_cache(THD *thd,JOIN_TAB *tables,uint table_count)
{
  reg1 uint i;
  uint length,blobs,size;
  CACHE_FIELD *copy,**blob_ptr;

  FIELD    *field;
  JOIN_CACHE  *cache;
  DBUG_ENTER("join_init_cache");

  cache= &tables[table_count].cache;
  cache->fields=blobs=0;

  for (i=0 ; i < table_count ; i++)
  {
    cache->fields+=tables[i].used_fields;
    blobs+=tables[i].used_blobs;
  }
  if (!(cache->field=(CACHE_FIELD*)
	sql_alloc(sizeof(CACHE_FIELD)*(cache->fields+table_count)+(blobs+1)*
		  sizeof(CACHE_FIELD*))))
  {
    my_free((gptr) cache->buff,MYF(0));
    cache->buff=0;
    DBUG_RETURN(1);
  }
  copy=cache->field;
  blob_ptr=cache->blob_ptr=(CACHE_FIELD**)
    (cache->field+cache->fields+table_count);

  length=0;
  for (i=0 ; i < table_count ; i++)
  {
    uint null_fields=0,used_fields;
    for (field=tables[i].table->field,used_fields=tables[i].used_fields ;
	 used_fields ;
	 field++)
    {
      if (field->query_id == thd->query_id)
      {
	used_fields--;
	copy->str=field->str;
	copy->length=field->packlength;
	copy->blob_field=0;
	if (f_is_blob(field->pack_flag))
	{
	  (*blob_ptr++)=copy;
	  copy->blob_field=field;
	  copy->strip=0;
	  copy->length-=sizeof(char*);
	  length+=copy->length;
	}
	else if (f_is_alpha(field->pack_flag) &&
		 !f_is_packed(field->pack_flag) &&
		 copy->length > 4)
	{
	  copy->strip=1;				/* Remove end space */
	  length+=copy->length+1;			/* Max length */
	}
	else
	{
	  copy->strip=0;
	  length+=copy->length;
	}
	if (f_maybe_null(field->pack_flag))
	  null_fields++;
	copy++;
      }
    }
    if (null_fields)
    {						/* must copy null bits */
      copy->str=(byte*) tables[i].table->null_flags;
      copy->length=(tables[i].table->form.null_fields+7)/8;
      copy->strip=0;
      copy->blob_field=0;
      copy++;
      cache->fields++;
    }
  }

  cache->records=0; cache->ptr_record= (uint) ~0;
  cache->length=length+blobs*sizeof(char*);
  cache->blobs=blobs;
  *blob_ptr=0;					/* End sequentel */
  size=max(JOIN_CACHE_SIZE,cache->length);
  if (!(cache->buff=(uchar*) my_malloc(size,MYF(0))))
    DBUG_RETURN(1);				/* Don't use cache */
  cache->end=cache->buff+size;
  reset_cache(cache);
  DBUG_RETURN(0);
}


static uint
used_blob_length(CACHE_FIELD **ptr)
{
  uint length,blob_length;
  for (length=0 ; *ptr ; ptr++)
  {
    (*ptr)->blob_length=blob_length=(ulong) valfield_l((*ptr)->blob_field);
    length+=blob_length;
    memcpy((char*) &(*ptr)->str,(*ptr)->blob_field->str+ (*ptr)->length,
	   sizeof(char*));
  }
  return length;
}


static bool
store_record_in_cache(JOIN_CACHE *cache)
{
  uint length;
  uchar *pos;
  CACHE_FIELD *copy,*end_field;
  bool last_record;

  pos=cache->pos;
  end_field=cache->field+cache->fields;

  length=cache->length;
  if (cache->blobs)
    length+=used_blob_length(cache->blob_ptr);
  if ((last_record=(length > (uint) (cache->end - pos))))
    cache->ptr_record=cache->records;

  /*
  ** There is room in cache. Put record there
  */
  cache->records++;
  for (copy=cache->field ; copy < end_field; copy++)
  {
    if (copy->blob_field)
    {
      if (last_record)
      {
	memcpy(pos,copy->blob_field->str,copy->length+sizeof(char*));
	pos+=copy->length+sizeof(char*);
      }
      else
      {
	memcpy(pos,copy->blob_field->str,copy->length);
	memcpy(pos+copy->length,copy->str,copy->blob_length);
	pos+=copy->length+copy->blob_length;
      }
    }
    else
    {
      if (copy->strip)
      {
	char *str,*end;
	for (str=copy->str,end= str+copy->length;
	     end > str && end[-1] == ' ' ;
	     end--) ;
	length=(uint) (end-str);
	memcpy(pos+1,str,length);
	*pos=(uchar) length;
	pos+=length+1;
      }
      else
      {
	memcpy(pos,copy->str,copy->length);
	pos+=copy->length;
      }
    }
  }
  cache->pos=pos;
  return last_record || (uint) (cache->end -pos) < cache->length;
}


static void
reset_cache(JOIN_CACHE *cache)
{
  cache->record_nr=0;
  cache->pos=cache->buff;
}


static void
read_cacheed_record(JOIN_TAB *tab)
{
  uchar *pos;
  uint length;
  bool last_record;
  CACHE_FIELD *copy,*end_field;

  last_record=tab->cache.record_nr++ == tab->cache.ptr_record;
  pos=tab->cache.pos;

  for (copy=tab->cache.field,end_field=copy+tab->cache.fields ;
       copy < end_field;
       copy++)
  {
    if (copy->blob_field)
    {
      char *blob_pos=copy->blob_field[0].str;
      if (last_record)
      {
	memcpy(blob_pos,pos,copy->length+sizeof(char*));
	pos+=copy->length+sizeof(char*);
      }
      else
      {
	memcpy(blob_pos,pos,copy->length);
	pos+=copy->length;
	memcpy(blob_pos+copy->length,(char*) &pos,sizeof(char*));
	pos+=valfield_l(copy->blob_field);
      }
    }
    else
    {
      if (copy->strip)
      {
	memcpy(copy->str,pos+1,length=(uint) *pos);
	memset(copy->str+length,' ',copy->length-length);
	pos+=1+length;
      }
      else
      {
	memcpy(copy->str,pos,copy->length);
	pos+=copy->length;
      }
    }
  }
  tab->cache.pos=pos;
  return;
}


static bool
cmp_buffer_with_ref(byte *buffer,REF_FIELD *fields,uint length)
{
  reg2 uint tmp_length;
  reg3 bool flag;

  for (flag=0 ; length ; fields++, buffer+=tmp_length, length-=tmp_length)
  {
    FIELD *field=fields->field;
    tmp_length=packlength(field);
    if (tmp_length > length)
      tmp_length=length;
    if (flag || memcmp(buffer,field->str,tmp_length))
    {
      flag=1;
      memcpy(buffer,field->str,(size_t) tmp_length);
    }
  }
  return flag;
}


/*****************************************************************************
** Group and order functions
*****************************************************************************/

/*
** Find order/group item in requested columns and change the item to point at it
** If item doesn't exists, add it first in the field list
** Return 0 if ok.
*/

static int
find_order_in_list(THD *thd,tname_t *tables,ORDER *order,List<Item> &fields,
		   string where)
{
  if ((*order->item)->type() == Item::INT_ITEM)
  {						/* Order by position */
    Item *item=0;
    List_iterator<Item> li(fields);

    for (uint count= (uint) ((Item_int*) (*order->item))->value ;
	 count-- && (item=li++) ;) ;
    if (!item)
    {
      my_error(ER_BAD_FIELD_ERROR,MYF(0),(*order->item)->full_name(), where);
      return 1;
    }
    order->item=li.ref();
    return 0;
  }
  Item **item=find_item_in_list(*order->item,fields,NULL);
  if (item)
  {
    order->item=item;				// use it
    return 0;
  }
  if ((*order->item)->fix_fields(thd,tables,where))
    return 1;					// Wrong field
  fields.push_front(*order->item);		// Add new field to field list
  order->item=(Item**) fields.head_ref();
  return 0;
}


/*
** Change order to point at item in select list. If item isn't a number
** and doesn't exits in the select list, add it the the field list.
*/

static int
setup_order(THD *thd,tname_t *tables,List<Item> &fields,ORDER *order,
	    string where)
{
  for (; order; order=order->next)
  {
    if (find_order_in_list(thd,tables,order,fields,where))
      return 1;
  }
  return 0;
}


static int
setup_group(THD *thd,tname_t *tables,List<Item> &fields,ORDER *order)
{
  Item *item;
  if (!order)
    return 0;				/* Everything is ok */

  List_iterator<Item> li(fields);
  while ((item=li++))
    item->marker=0;			/* Marker that field is not used */

  for ( ; order; order=order->next)
  {
    if (find_order_in_list(thd,tables,order,fields,"group statement"))
	return 1;
    (*order->item)->marker=1;		/* Mark found */
    if ((*order->item)->type() == Item::SUM_FUNC_ITEM)
    {
      net_printf(&thd->net,ER(ER_WRONG_GROUP_FIELD),
		 (*order->item)->full_name());
      return 1;
    }
  }
#ifdef RESTRICTED_GROUP
  li.rewind();
  while ((item=li++))
  {
    if (item->type() != Item::SUM_FUNC_ITEM && !item->marker)
    {
      net_printf(&thd->net,ER(ER_WRONG_FIELD_WITH_GROUP),item->full_name());
      return 1;
    }
  }
#endif
  return 0;
}

/*
** Add fields with aren't used at start of field list. Return FALSE if ok
*/

static bool
setup_new_fields(THD *thd,tname_t *tables,List<Item> &fields,ORDER *new_field)
{
  Item	  **item;
  DBUG_ENTER("setup_new_fields");

  thd->set_query_id=1;				// Not really neaded, but...
  for ( ; new_field ; new_field=new_field->next)
  {
    if ((item=find_item_in_list(*new_field->item,fields,NullS)))
      new_field->item=item;			/* Change to shared Item */
    else
    {
      if ((*new_field->item)->fix_fields(thd,tables,"procedure_list"))
	DBUG_RETURN(1);
      new_field->free_me=TRUE;
      fields.push_front(*new_field->item);
    }
  }
  DBUG_RETURN(0);
}

/* Add all fields with dosen't exist in order to order */

static ORDER *
add_all_fields_to_order(JOIN *join,ORDER *order_list,List<Item> &fields)
{
  List_iterator<Item> li(fields);
  Item *item;
  ORDER *order,**prev;

  while ((item=li++))
    item->marker=0;			/* Marker that field is not used */

  prev= &order_list;
  for (order=order_list ; order; order=order->next)
  {
    (*order->item)->marker=1;
    prev= &order->next;
  }

  li.rewind();
  while ((item=li++))
  {
    if (!item->marker && (item->type() != Item::FIELD_ITEM ||
			  ((Item_field*) item)->field->table->tablenr >=
			  join->const_tables))
    {
      ORDER *ord=(ORDER*) sql_alloc(sizeof(ORDER));
      ord->item=li.ref();
      ord->asc=1;
      ord->free_me=0;
      *prev=ord;
      prev= &ord->next;
    }
  }
  *prev=0;
  return order_list;
}


/*****************************************************************************
** Update join with count of the different type of fields
** Change type of sum_functions from MIN or MAX to MIN_STR,or MAX_STR if neaded
*****************************************************************************/


static void
count_field_types(JOIN *join,List<Item> &fields)
{
  List_iterator<Item> li(fields);
  Item *field;

  join->field_count=join->sum_func_count=join->func_count=0;
  join->quick_group=1;
  while ((field=li++))
  {
    if (field->type() == Item::FIELD_ITEM)
      join->field_count++;
    else if (field->type() == Item::FUNC_ITEM ||
	     field->type() == Item::FIELD_AVG_ITEM)
      join->func_count++;
    else if (field->type() == Item::SUM_FUNC_ITEM)
    {
      Item_sum *sum_item=(Item_sum*) field;
      if (!sum_item->quick_group)
	join->quick_group=0;
      join->sum_func_count++;
      if (sum_item->sum_func() == Item_sum::MIN_FUNC &&
	  sum_item->item->result_type() == Item::STRING_RESULT)
      {
	Item *item=sum_item->item;
	sum_item->item=0;			// Don't delete this
	delete li.replace(new Item_sum_min_str(item,sum_item->name));
      }
      else if (sum_item->sum_func() == Item_sum::MAX_FUNC &&
	       sum_item->item->result_type() == Item::STRING_RESULT)
      {
	Item *item=sum_item->item;
	sum_item->item=0;			// Don't delete this
	delete li.replace(new Item_sum_max_str(item,sum_item->name));
      }
      else if (sum_item->item->type() == Item::FUNC_ITEM)
	join->func_count++;
    }
  }
}


/*
  Return 1 if second is a subpart of first argument
  If first parts has different direction, change it to second part
  (group is sorted like order)
*/

static bool
test_if_subpart(ORDER *a,ORDER *b)
{
  for (; a && b; a=a->next,b=b->next)
  {
    if (a->item == b->item)
      a->asc=b->asc;
    else
      return 0;
  }
  return test(!b);
}

/*
  return 1 if some field in order isn't in tmp table
*/

static bool
test_if_special_field(ORDER *order)
{
  for (; order ; order=order->next)
  {
    if (order->item[0]->type() == Item::FIELD_AVG_ITEM)
      return 1;
  }
  return 0;
}

/*
  Return table number if there is only one table in sort order
  and group and order is compatible 
  else return ~0;
*/

static uint
get_sort_by_table(ORDER *a,ORDER *b)
{
  uint tablenr= (uint) ~0;
  if (!a)
    a=b;					// Only one need to be given
  else if (!b)
    b=a;

  for (; a && b; a=a->next,b=b->next)
  {
    if (a->item != b->item)
      return (uint) ~0;
    if ((*a->item)->type() != Item::FIELD_ITEM)
      return (uint) ~0;				// Can't sort functions yet

    FIELD *field=((Item_field*) (*a->item))->field;
    if (tablenr != (uint) ~0 && tablenr != field->table->tablenr)
      return (uint) ~0;
    tablenr=field->table->tablenr;
  }
  return tablenr;
}

	/* calc how big buffer we nead for comparing group entries */

static void
calc_group_buffer(JOIN *join,ORDER *group)
{
  uint key_length=0,parts=0;
  for (; group ; group=group->next)
  {
    FIELD *field=(*group->item)->tmp_table_field();
    if (field)
      key_length+=field->packlength;
    else if ((*group->item)->result_type() == Item::REAL_RESULT)
      key_length+=sizeof(double);
    else
      key_length+=(*group->item)->max_length;
    parts++;
  }
  join->group_length=key_length;
  join->group_parts=parts;
}


/*
** Get a list of buffers for saveing last group
** Groups are saved in reverse order for easyer check loop
*/

static void
alloc_group_fields(JOIN *join,ORDER *group)
{
  if (group)
  {
    for (; group ; group=group->next)
      join->group_fields.push_front(new_Item_buff(*group->item));
  }
  join->sort_and_group=1;			/* Mark for do_select */
}


static int
test_if_group_changed(List<Item_buff> &list)
{
  List_iterator<Item_buff> li(list);
  int index= -1,i;
  Item_buff *buff;

  for (i=(int) list.elements-1 ; (buff=li++) ; i--)
  {
    if (buff->cmp())
      index=i;
  }
  return index;
}



/*
** Setup copy_fields to save fields at start of new group
** Only FIELD_ITEM:s and FUNC_ITEM:s neads to be saved between groups.
** Change old item_field to use a new field with points at saved fieldvalue
** This function is only called before use of send_fields
*/

static void
setup_copy_fields(JOIN *join,List<Item> &fields)
{
  Item *pos;
  uint field_count;
  List_iterator<Item> li(fields);
  COPY_FIELD *copy;

  field_count=join->field_count;
  copy=join->copy_field=(COPY_FIELD*) sql_alloc(sizeof(COPY_FIELD)*
						field_count);
  join->copy_funcs.empty();
  while ((pos=li++))
  {
    if (pos->type() == Item::FIELD_ITEM)
    {
      Item_field *item=(Item_field*) pos;
      if (item->field->flags & BLOB_FLAG)
      {
	pos=new Item_copy_string(pos);
	VOID(li.replace(pos));
	join->copy_funcs.push_back(pos);
	continue;
      }
      FIELD *field= item->field;
      copy->from=field->str;
      copy->to=(char*) sql_alloc(copy->length=field->packlength+1);

      /* change result_field to point at saved value */
      item->result_field=field=
	(FIELD*) sql_memdup((gptr) field,sizeof(*field));
      field->str=copy->to;
      if (f_maybe_null(field->pack_flag))
      {
	copy->from_bytepos=	field->null_pos;
	copy->from_bit=		field->null_bit;
	copy->to_bytepos=	(uchar*) copy->to+copy->length;
	copy->to_bit=		1;
	field->null_pos=	copy->to_bytepos; /* Use new position */
	field->null_bit=	1;
      }
      else
	copy->from_bytepos=	0;
      copy++;
    }
    else if (pos->type() == Item::FUNC_ITEM)	// This dosen't happend yet
    {						// Save for send fields
      pos=new Item_copy_string(pos);
      join->copy_funcs.push_back(pos);
    }
  }
  join->copy_field_count=(uint) (copy - join->copy_field);
}


/*
** Copy fields and null values between two tables
*/

static void
copy_fields(JOIN *join)
{
  COPY_FIELD *ptr=join->copy_field,*end=ptr+join->copy_field_count;

  for ( ; ptr != end; ptr++)
  {
    memcpy(ptr->to,ptr->from,ptr->length);
    if (ptr->from_bytepos)
    {
      if (*ptr->from_bytepos & ptr->from_bit)
	*(ptr->to_bytepos)|= (uchar) ptr->to_bit;
      else
	(*ptr->to_bytepos)&= (uchar) ~ptr->to_bit;
    }
  }

  List_iterator<Item> it(join->copy_funcs);
  Item_copy_string *item;
  while ((item = (Item_copy_string*) it++))
  {
    item->copy();
  }
}


/*****************************************************************************
** Make an array of pointer to sum_functions to speed up sum_func calculation
*****************************************************************************/

static void
make_sum_func_list(JOIN *join,List<Item> &fields)
{
  Item_sum **func = (Item_sum**) sql_alloc(sizeof(Item_sum*)*
					   (join->sum_func_count+1));
  List_iterator<Item> it(fields);
  join->sum_funcs=func;

  Item *field;
  while ((field=it++))
  {
    if (field->type() == Item::SUM_FUNC_ITEM)
    {
      *func++=(Item_sum*) field;
    }
  }
  *func=0;					// End marker
}


/*
** Change all funcs and sum_funcs to fields in tmp table
*/

static void
change_to_use_tmp_fields(List<Item> &items)
{
  List_iterator<Item> it(items);
  Item *item_field,*item;

  while ((item=it++))
  {
    if (item->type() == Item::FIELD_ITEM)
    {
      ((Item_field*) item)->field=
	((Item_field*) item)->result_field;
    }
    else if (item->type() == Item::FUNC_ITEM ||
	     item->type() == Item::SUM_FUNC_ITEM ||
	     item->type() == Item::FIELD_AVG_ITEM)
    {
      FIELD *field=item->tmp_table_field();

      if (item->type() == Item::SUM_FUNC_ITEM &&
	  ((Item_sum*) item)->sum_func() == Item_sum::AVG_FUNC)
	item_field=(Item*) new Item_avg_field((Item_sum_avg*) item);
      else
	item_field=(Item*) new Item_field(field);
      item_field->name=item->name;		/*lint -e613 */
      delete it.replace(item_field);		/*lint -e613 */
    }
  }
}


/*
** Change all sum_func refs to fields to point at fields in tmp table
** Change all funcs to be fields in tmp table
*/

static void
change_refs_to_tmp_fields(List<Item> &items)
{
  List_iterator<Item> it(items);
  Item *item;

  while ((item= it++))
  {
    if (item->type() == Item::FUNC_ITEM)
    {						/* All funcs are stored */
      delete it.replace(new Item_field(((Item_func*) item)->result_field));
    }
    else if (item->type() == Item::FIELD_ITEM)	/* Change refs */
    {
      ((Item_field*)item)->field=((Item_field*) item)->result_field;
    }
    else if (item->type() == Item::SUM_FUNC_ITEM)
    {
      Item_sum *sum_item= (Item_sum*) item;
      if (sum_item->result_field)		// If not a const sum func
      {
	delete sum_item->item;			// This is a nop
	sum_item->item= new Item_field(sum_item->result_field);
      }
    }
  }
}



/******************************************************************************
** code for calculating functions
******************************************************************************/

static void
init_tmptable_sum_functions(Item_sum **func_ptr)
{
  Item_sum *func;
  while ((func= *(func_ptr++)))
    func->reset_field();
}


	/* Update record 0 in tmp_table from record 1 */

static void
update_tmptable_sum_func(Item_sum **func_ptr,
			 TABLE *tmp_table __attribute__((unused)))
{
  Item_sum *func;
  while ((func= *(func_ptr++)))
    func->update_field(0);
}


	/* Copy result of sum functions to record in tmp_table */

static void
copy_sum_funcs(Item_sum **func_ptr)
{
  Item_sum *func;
  for (; (func = *func_ptr) ; func_ptr++)
    (void) func->save_in_field(func->result_field);
  return;
}


static void
init_sum_functions(Item_sum **func_ptr)
{
  Item_sum *func;
  for (; (func= (Item_sum*) *func_ptr) ; func_ptr++)
    func->reset();
}


static void
update_sum_func(Item_sum **func_ptr)
{
  Item_sum *func;
  for (; (func= (Item_sum*) *func_ptr) ; func_ptr++)
    func->add();
}

	/* Copy result of functions to record in tmp_table */

static void
copy_funcs(Item_result_field **func_ptr)
{
  Item_result_field *func;
  for (; (func = *func_ptr) ; func_ptr++)
    (void) func->save_in_field(func->result_field);
  return;
}
