/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

#include "mysql_priv.h"
#include "sql_lex.h"

extern int yyparse(void);

static bool check_access(THD *thd,uint access,char *db=0);
static bool check_dup(char *name,tname_t *tables);

char *command_name[]={
  "Sleep", "Quit", "Init DB", "Query", "Field List", "Create DB",
  "Drop DB", "Reload","Shutdown", "Statistics", "Processes",
  "Connect",
};

/****************************************************************************
	Initialize global thd variables neaded for query
****************************************************************************/

static void
mysql_init_query(THD *thd)
{
  DBUG_ENTER("mysql_init_query");
  thd->net.last_error[0]=0;
  thd->item_list.empty();
  thd->value_list.empty();
  thd->key_list.empty();
  thd->create_list.empty();
  thd->order_list.elements=thd->table_list.elements= thd->group_list.elements=
    thd->create_list.elements=thd->proc_list.elements=0;

  thd->order_list.first=0;
  thd->order_list.next= (byte**) &thd->order_list.first;
  thd->table_list.first=0;
  thd->table_list.next= (byte**) &thd->table_list.first;
  thd->group_list.first=0;
  thd->group_list.next= (byte**) &thd->group_list.first;
  thd->proc_list.first=0;
  thd->proc_list.next= (byte**) &thd->proc_list.first;
  DBUG_VOID_RETURN;
}


void
mysql_parse(THD *thd,char *inBuf,uint length)
{
  DBUG_ENTER("mysql_parse");
  LEX *lex=lex_start((uchar*) inBuf,length);
  mysql_init_query(thd);
  VOID(yyparse());
  lex_end(lex);
  DBUG_VOID_RETURN;
}


inline static void
link_in_list(SQL_LIST *list,byte *element,byte **next)
{
  list->elements++;
  (*list->next)=element;
  list->next=next;
  *next=0;
}


/*****************************************************************************
** Store field definition for create
** Because of the old .frm file format we can't have a null field with
** a default value. (null fields always have a default value of null)
** Return 0 if ok
******************************************************************************/

int add_field_to_list(char *field_name, enum enum_field_types type,
		      char *length, char *decimals,
		      uint type_modifier,int notNull, int unireg_type,
		      int keyinfo, Item *default_value,char *change)
{
  register create_field *new_field;
  THD	*thd=current_thd;
  DBUG_ENTER("add_field_to_list");

  if (strlen(field_name) > NAME_LEN)
  {
    net_printf(&thd->net, ER(ER_TOO_LONG_IDENT), field_name);
    DBUG_RETURN(1);
  }
  if (keyinfo)
  {
    notNull=1;					// Keys can't be NULL in mysql
    if (keyinfo == PRI_KEY_FLAG)		// Compatible with mysql
    {
      current_lex->col_list.push_back(new key_part_spec(field_name,0));
      thd->key_list.push_back(new Key(Key::PRIMARY,NullS,current_lex->col_list));
      current_lex->col_list.empty();
    }
  }

  if ((notNull && default_value && default_value->type() == Item::NULL_ITEM) ||
      (!notNull &&
       (default_value && default_value->type() != Item::NULL_ITEM)))
  {
    net_printf(&thd->net,ER(ER_INVALID_DEFAULT),field_name);
    DBUG_RETURN(1);
  }

  /* change FLOAT(precision) to FLOAT or DOUBLE */
  if (type == FIELD_TYPE_FLOAT && length && !decimals)
  {
    uint tmp_length=atoi(length);
    if (tmp_length > sizeof(double))
    {
      net_printf(&thd->net,ER(ER_WRONG_FIELD_SPEC),field_name);      
      DBUG_RETURN(1);
    }
    else if (tmp_length > sizeof(float))
      type=FIELD_TYPE_DOUBLE;
    length=0;				// Use default format
  }
  if (!notNull)				// NULL fields can't have default
    default_value=0;
  new_field =new create_field();
  new_field->field=0;
  new_field->field_name=field_name;
  new_field->def=default_value;
  new_field->flags= type_modifier;
  new_field->type=type;
  new_field->unireg_type= unireg_type;
  new_field->decimals= decimals ? (uint) atoi(decimals) : 0;
  new_field->length=0;
  new_field->change=change;
  new_field->intervall=0;
  if (notNull)
    new_field->flags|= NOT_NULL_FLAG;
  if (keyinfo)
    new_field->flags|= keyinfo | NOT_NULL_FLAG;	/* Key can't be null */
  if (length)
    if (!(new_field->length= (uint) atoi(length)))
      length=0;
  uint sign_len=type_modifier & UNSIGNED_FLAG ? 1 : 0;
  if (new_field->length && new_field->decimals &&
      new_field->length < new_field->decimals+2)
    new_field->length=new_field->decimals+2;

  switch (type) {
  case FIELD_TYPE_CHAR:
    if (!length) new_field->length=3+sign_len;
    break;
  case FIELD_TYPE_SHORT:
    if (!length) new_field->length=5+sign_len;
    break;
  case FIELD_TYPE_INT24:
    if (!length) new_field->length=8+sign_len;
    break;
  case FIELD_TYPE_LONG:
    if (!length) new_field->length=10+sign_len;
    break;
  case FIELD_TYPE_LONGLONG:
    if (!length) new_field->length=20+sign_len;
    break;
  case FIELD_TYPE_STRING:
  case FIELD_TYPE_VAR_STRING:
  case FIELD_TYPE_DECIMAL:
  case FIELD_TYPE_NULL:
    break;
  case FIELD_TYPE_BLOB:
    new_field->flags|=BLOB_FLAG;
    new_field->type=FIELD_TYPE_SHORT;		// How blob length is packed
    break;
  case FIELD_TYPE_TINY_BLOB:
    new_field->flags|=BLOB_FLAG;
    new_field->type=FIELD_TYPE_CHAR;
    break;
  case FIELD_TYPE_LONG_BLOB:
    new_field->flags|=BLOB_FLAG;
    new_field->type=FIELD_TYPE_LONG;
    break;
  case FIELD_TYPE_MEDIUM_BLOB:
    new_field->flags|=BLOB_FLAG;
    new_field->type=FIELD_TYPE_INT24;
    break;
  case FIELD_TYPE_FLOAT:
    if (!length)
    {
      new_field->length = 8+4;
      new_field->decimals=4;
    }
    break;
  case FIELD_TYPE_DOUBLE:
    if (!length)
    {
      new_field->length = 12+4;
      new_field->decimals=4;
    }
    break;
  case FIELD_TYPE_TIMESTAMP:
    if (!length)
      new_field->length= 14;			// Full date YYYYMMDDHHMMSS
    else
    {
      new_field->length=((new_field->length+1)/2)*2;
      new_field->length= min(new_field->length,14);
    }
    new_field->flags|= ZEROFILL_FLAG | UNSIGNED_FLAG | NOT_NULL_FLAG;
    break;
  case FIELD_TYPE_DATE:
    new_field->length=10;
    break;
  case FIELD_TYPE_TIME:
    new_field->length=8;
    break;
  }
  if (new_field->length >= MAX_FIELD_WIDTH ||
      (!new_field->length && !(new_field->flags & BLOB_FLAG)))
  {
    net_printf(&thd->net,ER(ER_TOO_BIG_FIELDLENGTH),field_name,
	       MAX_FIELD_WIDTH-1);
    DBUG_RETURN(1);
  }
  thd->create_list.push_back(new_field);
  DBUG_RETURN(0);
}


void
add_item_to_list(Item *item)
{
  current_thd->item_list.push_back(item);
}

/* Update item in field struct to point to item for update */

void
add_value_to_list(Item *value)
{
  current_thd->value_list.push_back(value);
}


void
add_proc_to_list(Item *item)
{
  ORDER *order;
  Item	**item_ptr;

  order    = (ORDER *) sql_alloc(sizeof(ORDER)+sizeof(Item*));
  item_ptr = (Item**) (order+1);
  *item_ptr= item;
  order->item=item_ptr;
  order->free_me=0;
  link_in_list(&current_thd->proc_list,(byte*) order,(byte**) &order->next);
}


/* Fix escaping of _, % and \ in database and table names (for ODBC) */

static void remove_escape(char *name)
{
  char *to;
  for (to=name; *name ; name++)
  {
    if (*name == '\\' && name[1])
      name++;					// Skipp '\\'
    *to++= *name;
  }
  *to=0;
}

/****************************************************************************
** save order by and tables in own lists
****************************************************************************/


void add_to_list(SQL_LIST &list,Item *item,bool asc)
{
  ORDER *order;
  Item	**item_ptr;
  DBUG_ENTER("add_to_list");
  order =    (ORDER *) sql_alloc(sizeof(ORDER)+sizeof(Item*));
  item_ptr = (Item**) (order+1);
  *item_ptr=item;
  order->item= item_ptr;
  order->asc = asc;
  order->free_me=0;
  link_in_list(&list,(byte*) order,(byte**) &order->next);
  DBUG_VOID_RETURN;
}


void add_order_to_list(Item *item,bool asc)
{
  add_to_list(current_thd->order_list,item,asc);
}


void add_group_to_list(Item *item,bool asc)
{
  add_to_list(current_thd->group_list,item,asc);
}


tname_t *add_table_to_list(char *name,char *alias)
{
  register tname_t *ptr;
  THD	*thd=current_thd;
  DBUG_ENTER("add_table_to_list");

  ptr = (tname_t *) sql_alloc(sizeof(tname_t));
  ptr->real_name=name;
  ptr->name=alias ? alias : name;
  ptr->table=0;

  /* check that used name is unique */
  for (tname_t *table=(tname_t*) thd->table_list.first ; table ;
       table=table->next)
  {
    if (!strcmp(ptr->name,table->name))
    {
      net_printf(&thd->net,ER(ER_NONUNIQ_TABLE),ptr->name);
      DBUG_RETURN(0);
    }
  }
  link_in_list(&thd->table_list,(byte*) ptr,(byte**) &ptr->next);
  DBUG_RETURN(ptr);
}


/****************************************************************************
** mysql_execute_command
** Execute command saved in thd and current_lex->sql_command
****************************************************************************/

void
mysql_execute_command(void)
{
  int	res=0;
  THD	*thd=current_thd;
  LEX	*lex=current_lex;
  tname_t *tables=(tname_t*) thd->table_list.first;
  DBUG_ENTER("mysql_execute_command");

  switch(lex->sql_command) {
  case SELECT_SYMBOL:
  {
    select_result *result;
    if (check_access(thd,lex->exchange ? SELECT_ACL | FILE_ACL : SELECT_ACL))
    {
      delete lex->having;
      goto error;
    }
    thd->offset_limit=lex->offset_limit;
    thd->select_limit=lex->select_limit+lex->offset_limit;
    if (thd->select_limit < lex->select_limit)
      thd->select_limit=0L;			// Fix if overflow

    if (lex->exchange)
    {
      result=new select_export(lex->exchange);
      thd->big_temporaries=1;
    }
    else
      result=new select_send();
    res=open_and_lock_tables(thd,tables);
    if (!res)
    {
      res=mysql_select(thd,tables,thd->item_list,
		       lex->where,
		       (ORDER*) thd->order_list.first,
		       (ORDER*) thd->group_list.first,
		       lex->having,
		       (ORDER*) thd->proc_list.first,
		       lex->select_options,
		       result);
    }
    delete result;
    delete lex->having;
    thd->big_temporaries=0;
    break;
  }
  case CREATE_TABLE:
    if (check_access(thd,CREATE_ACL))
      goto error;
    res = mysql_create_table(thd,tables->name, thd->create_list,
			     thd->key_list,0);
    if (!res)
      send_ok(&thd->net);
    break;
  case CREATE_INDEX:
    if (check_access(thd,CREATE_ACL))
      goto error;
    res = mysql_create_index(thd,tables->name, lex->col_list,
			     lex->unique_flag);
    break;
  case ALTER:
    if (check_access(thd,DB_ACL))
      goto error;
    res= mysql_alter_table(thd,tables->name, thd->create_list,
			   thd->key_list, lex->drop_list, lex->alter_list,
			   lex->drop_primary, lex->duplicates);
    break;
  case UPDATE_SYMBOL:
    if (check_access(thd,UPDATE_ACL))
      goto error;
    if (thd->item_list.elements != thd->value_list.elements)
    {
      send_error(&thd->net,ER(ER_WRONG_VALUE_COUNT));
      DBUG_VOID_RETURN;
    }
    res = mysql_update(thd,tables->name,
		       thd->item_list,
		       thd->value_list,
		       lex->where);
    break;
  case INSERT:
    if (check_access(thd,INSERT_ACL))
      goto error;
    res = mysql_insert(thd,tables,thd->field_list,thd->value_list);
    break;
  case INSERT_SELECT:
  {
    if (check_access(thd,INSERT_ACL | SELECT_ACL))
    {
      delete lex->having;
      goto error;
    }
    select_result *result;
    thd->offset_limit=lex->offset_limit;
    thd->select_limit=lex->select_limit+lex->offset_limit;
    if (thd->select_limit < lex->select_limit)
      thd->select_limit=0L;			// Fix if overflow

    if (check_dup(tables->real_name,tables->next))
    {
      net_printf(&thd->net,ER(ER_INSERT_TABLE_USED),tables->real_name);
      DBUG_VOID_RETURN;
    }
    if (!(res=open_tables(thd,tables)))
    {
      tables->table->form.reginfo.update=1;
      if (!(res=lock_tables(thd,tables)))
      {
	result=new select_insert(tables->table,&thd->field_list);
	res=mysql_select(thd,tables->next,thd->item_list,
			 lex->where,
			 (ORDER*) thd->order_list.first,
			 (ORDER*) thd->group_list.first,
			 lex->having,
			 (ORDER*) thd->proc_list.first,
			 lex->select_options,
			 result);
	delete result;
      }
    }
    delete lex->having;
    break;
  }
  case DELETE:
    if (check_access(thd,DELETE_ACL))
      goto error;
    res = mysql_delete(thd,tables->name,lex->where);
    break;
  case DROP_TABLE:
    if (check_access(thd,DROP_ACL))
      goto error;
    res = mysql_rm_table(thd,tables);
    break;
  case DROP_INDEX:
    if (check_access(thd,DROP_ACL))
      goto error;
    res=0;					// Always ok
    send_ok(&thd->net);
    break;
  case DATABASES:
    res= mysqld_show_dbs(thd, (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case TABLES:
    if (!lex->db)
    {
      send_error(&thd->net,ER(ER_NO_DB_ERROR));
      goto error;
    }
    remove_escape(lex->db);			// Fix escaped '_'
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error;
    res= mysqld_show_tables(thd,lex->db,
			    (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case COLUMNS:
    if (!lex->db)
    {
      send_error(&thd->net,ER(ER_NO_DB_ERROR));
      goto error;
    }
    remove_escape(lex->db);			// Fix escaped '_'
    remove_escape(tables->name);
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error;
    res= mysqld_show_fields(thd,lex->db,tables->name,
			    (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case KEYS:
    if (!lex->db)
    {
      send_error(&thd->net,ER(ER_NO_DB_ERROR));
      goto error;
    }
    remove_escape(lex->db);			// Fix escaped '_'
    remove_escape(tables->name);
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error;
    res= mysqld_show_keys(thd,lex->db,tables->name);
    break;
  case LOAD:
    if (check_access(thd,SELECT_ACL | FILE_ACL))
      goto error;
    res=mysql_load(thd, lex->exchange, tables, thd->field_list,
		   lex->duplicates, 0L);
    break;
  default:					/* Impossible */
    break;
  }
  if (res < 0)
    send_error(&thd->net,abort_loop ? ER(ER_SERVER_SHUTDOWN) : NullS);

  /* Free strings used by items */
  thd->field_list.delete_elements();
  thd->item_list.delete_elements();
  thd->value_list.delete_elements();
  ORDER *order;
  for (order=(ORDER*) thd->proc_list.first ; order ; order=order->next)
  {
    if (order->free_me)
      delete *order->item;
  }

error:
  DBUG_VOID_RETURN;
}


/* Return true if access denied */

static bool
check_access(THD *thd,uint want_access,char *db)
{
  uint access;
  if ((thd->master_access & want_access) == want_access)
    return FALSE;
  if (!(want_access & DB_ACL))
  {
    send_error(&thd->net,ER(ER_ACCESS_DENIED_ERROR));
    return TRUE;
  }
  if (!db && !thd->db)
  {
    send_error(&thd->net,ER(ER_NO_DB_ERROR));
    return TRUE;
  }
  if (db && (!thd->db || strcmp(db,thd->db)))
    access=acl_get(thd->host, thd->ip, thd->user, db);
  else
    access=thd->db_access;

  if (((access | thd->master_access) & want_access) == want_access)
    return FALSE;				/* Ok */
  send_error(&thd->net,ER(ER_DBACCESS_DENIED_ERROR));
  return TRUE;
}

	/* Check if name is used in table list */

static bool check_dup(char *name,tname_t *tables)
{
  for (; tables ; tables=tables->next)
    if (!strcmp(name,tables->real_name))
      return 1;
  return 0;
}

	/* Execute one command from socket */

bool do_command(THD *thd)
{
  char dbname[MAXREGNAME+1],*packet,*db;
  uint old_timeout,packet_length;
  bool	error=0,tables_used=0;
  NET *net;
  struct sched_param sched_param;
  enum enum_server_command command;
  DBUG_ENTER("do_command");

  init_sql_alloc(&thd->alloc);
  net= &thd->net;
  thd->select_distinct=thd->current_tablenr=0;
  thd->dupp_field=0;
  thd->tmp_table=0;

  packet=0;
  old_timeout=net->timeout;
  net->timeout=8*60*60;				/* Wait max for 8 hours */
  net->last_error[0]=0;				// Clear error message

  net_new_transaction(net);
  if ((packet_length=net_read(net)) == packet_error)
  {
    command = QUIT;
    DBUG_PRINT("general",("Command on sock %d failed",net->fd));
  }
  else
  {
    packet=(char*) net->buff;
    command = (enum enum_server_command) (uchar) packet[0];
    DBUG_PRINT("general",("Command on sock %d = %d (%s)",
			  net->fd, command, command_name[command]));
  }
  net->timeout=old_timeout;				/* Timeout */
  thd->command=command;
  thd->version=reload_version;

  switch(command) {
  case INIT_DB:
    strmake(dbname,(char*) packet+1,sizeof(dbname)-1);
    if (!stripp_sp(dbname))
    {
      send_error(net,ER(ER_NO_DB_ERROR));
      break;
    }
    DBUG_PRINT("general",("Use database: %s", dbname));
    mysql_log.write(command,"%s",dbname);
    thd->db_access= acl_get(thd->host,thd->ip,thd->user,dbname) |
      thd->master_access;
    if (!(thd->db_access & DB_ACL))
    {
      send_error(net,ER(ER_DBACCESS_DENIED_ERROR));
      break;
    }
    if (mysql_init_db(dbname) < 0)
      send_error(net,NULL);
    else
    {
      send_ok(net);
      safeFree(thd->db);
      thd->db=my_strdup(dbname,MYF(MY_FAE));	/* This better work.. */
    }
    break;

  case QUERY:
    tables_used=1;
    sched_param.sched_priority=QUERY_PRIOR;
    VOID(pthread_setschedparam(pthread_self(),SCHED_OTHER,
			       &sched_param));
#ifdef FIX_LOG					// Done in client
    if (opt_log)
    {						// Remove linefeed for log
      for (char *pos=packet+1; pos=strchr(pos,'\n'); pos++)
	*pos=' ';
    }
#endif
    mysql_log.write(command,"%s",packet+1);
    DBUG_PRINT("query",("%s",packet+1));
    mysql_parse(thd,thd->query=sql_memdup(packet+1,packet_length),
		packet_length);

    sched_param.sched_priority=WAIT_PRIOR;
    VOID(pthread_setschedparam(pthread_self(),SCHED_OTHER,
			       &sched_param));
    DBUG_PRINT("info",("query ready"));
    break;
  case FIELD_LIST:				// This isn't actually neaded
  {
    tables_used=1;
    char *table,*fields;
    if (!thd->db)
    {
      send_error(net,ER(ER_NO_DB_ERROR));
      break;
    }
    table=packet+1;
    fields=strend(table)+1;
    mysql_log.write(command,"%s %s",table,fields);
    remove_escape(table);			// This can't have wildcards
    mysqld_list_fields(thd,sql_strdup(table),sql_strdup(fields));
    break;
  }
  case QUIT:
    mysql_log.write(command,NullS);
    net->error=0;				// Don't give 'abort' message
    error=TRUE;					// End server
    break;

  case CREATE_DB:
    db=sql_strdup(packet+1);
    if (check_access(thd,CREATE_ACL,db))
      break;
    mysql_log.write(command,packet+1);
    mysql_createDB(thd,sql_strdup(packet+1));
    break;

  case DROP_DB:
    db=sql_strdup(packet+1);
    if (check_access(thd,DROP_ACL,db))
      break;
    mysql_log.write(command,db);
    mysql_rm_db(thd,db);
    break;

  case RELOAD:
    if (check_access(thd,RELOAD_ACL))
      break;
    tables_used=1;
    mysql_log.write(command,NullS);
    reload_acl_and_cache();
    send_eof(net);
    break;

  case SHUTDOWN:
    if (check_access(thd,SHUTDOWN_ACL))
      break;
    mysql_log.write(command,NullS);
    send_eof(net);
    close_connection(net);
    close_thread_tables(thd);			/* Free before kill */
    sql_free(&thd->alloc);
    VOID(pthread_kill(signal_thread,SIGQUIT));	/* End everything nicely */
    error=TRUE;
    break;

  case STATISTICS:
  {
    mysql_log.write(command,NullS);
    char buff[100];
    sprintf((char*) buff,
	    "Uptime: %ld  Running threads: %d  Questions: %ld  Reloads: %ld  Open tables: %d",
	    (ulong) (time((time_t*) 0) - start_time),
	    (int) thread_count,thd->query_id,reload_version,cached_tables());
    VOID(net_write(net,buff,strlen(buff)));
    VOID(net_flush(net));
    break;
  }
  case PROCESS_INFO:
    if (check_access(thd,PROCESS_ACL,0))
      break;
    mysql_log.write(command,NullS);
    mysqld_list_processes(thd);
    break;
  case SLEEP:
  case CONNECT:					// Impossible here
  default:
    send_error(net, ER(ER_UNKNOWN_COM_ERROR));
    break;
  }
  if (tables_used)
    close_thread_tables(thd);			/* Free tables */
#ifndef DBUG_OFF
  if (test_flags & TEST_PRINT_CACHED_TABLES)
  {
    VOID(pthread_mutex_lock(&LOCK_open));
    print_cached_tables();
    VOID(pthread_mutex_unlock(&LOCK_open));
  }
#endif

  VOID(pthread_mutex_lock(&LOCK_thread_count)); // For process list
  thd->command=SLEEP;
  thd->query=0;
  sql_free(&thd->alloc);
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
  DBUG_RETURN(error);
}


void reload_acl_and_cache(void)
{
    acl_reload();
    reload_version++;				/* Doesn't nead protection */
    select_errors=0;				/* Write if more errors */
    close_cached_tables();
    mysql_log.flush();				// Flush log
}
