/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Functions to handle quick-select of records */
/* The idea is to see if we from a normal select-structure can use a
   handler-key and only use read-next in different ranges */

#include "mysql_priv.h"
#include "select_q.h"
#include <m_ctype.h>
#include <nisam.h>

#ifndef DBUG_OFF
/* #define EXTRA_DEBUG */
#endif

	/* functions defined in this file */

static int NEAR_F parse_quick_select(PARAM *param,SELECT *select,
				     uint fi_nr,uint keynr);
static int add_quick_test(PARAM *param,FIELD *field,SELECT_ROW *row,
			  SELECT_TYPE *diff);
static int NEAR_F get_new_param(PARAM *param);
static int NEAR_F select_and(PARAM *param,int level);
static int NEAR_F do_and(PARAM *param,S_ARG *a, S_ARG *b);
static int NEAR_F do_or(PARAM *param,S_ARG *a, S_ARG *b);
static void NEAR_F get_next_arg(PARAM *param, S_ARG * *first);
static void NEAR_F select_or(PARAM *param,int level);
static SELECT_QUICK * NEAR_F make_quick_select(SELECT *select,PARAM *param,
					       uint index,uint fi_nr);
static void NEAR_F free_param(PARAM *param);
static int NEAR_F qs_cmp(STR_OR_DOUBLE *a,STR_OR_DOUBLE *b,
			 uint type,uint flag);
static int change_field_to_ref_field(TABLE *head,TABLE **first_form,
				     uint regnr,uint fieldnr,
				     uint *out_regnr,uint *out_fieldnr);
#ifdef EXTRA_DEBUG
static void NEAR_F print_param(PARAM *param,int level);
#endif

/*****************************************************************************
**	Test if a key can be used in different ranges
**	Returns:
**	-1 if impossible select
**	0 if can't use quick_select
**	1 if found usably range
*****************************************************************************/

int test_quick_select(SELECT *select, ulong keys_to_use)
{
  reg1 KEY *pos;
  reg2 uint index;
  PARAM param;
  int  error;
  uint basflag,fi_nr,keylength,length;
  ulong records;
  SELECT_QUICK *quick,*new_quick;
  DBUG_ENTER("test_quick_select");

  if (!select || my_b_inited(&select->file) ||
      specialflag & SPECIAL_NO_NEW_FUNC)
    DBUG_RETURN(0);
  select->quick=0;				/* Not quick-select */
  select->neaded_reg=select->quick_keys=0;

  if (!((basflag= select->head->form.db_capabilities) & HA_KEYPOS_TO_RNDPOS) &&
      keys_to_use == (uint) ~0 || !keys_to_use)
    DBUG_RETURN(0);				/* Not smart database */
  if ((records=(select->head->form.keyfile_info.records+
		select->head->form.keyfile_info.deleted)/4) < 2)
    DBUG_RETURN(0);				/* No nead for quick select */

  bzero((byte*) &param,sizeof(param));
  param.pos_level=(S_ARG **) my_malloc((uint) (select->max_brace_level*2+3)*
				       sizeof(S_ARG*),MYF(MY_FAE));
  param.pre_reg_read=select->reg_init_read &
    ~((ulong) 1L << select->head->tablenr) &
    ~select->const_reg;

  pos=select->head[0].form.key_info;
  quick=0;
  for (index=0 ; index < select->head->form.keys ; index++, pos++)
  {
    if (!(keys_to_use & (1L << index)) ||
	pos->key_part[0].key_part_flag & HA_REVERSE_SORT)
      continue;
    if ((fi_nr=find_keyfield(&select->head->form,index)-1) == (uint) -1)
      continue;
    keylength= pos->key_part[0].length;
    if (keylength >= (length=packlength(select->head[0].field+fi_nr)))
    {
      if (! (basflag & HA_READ_ORDER) && keylength > length)
	continue;
      keylength= select->head->form.field[fi_nr].length;
    }
    param.type=select->head->form.field[fi_nr].pack_flag;
    param.use= &param.root;
    param.use->next=0;
    param.keylength=keylength;
    param.baseflag=basflag;
    if ((error=parse_quick_select(&param,select,fi_nr,index)) > 0)
    {						/* quick-select-key found */
      if ((new_quick=make_quick_select(select,&param,index,fi_nr)))
      {
	if (new_quick->records < records)
	{
	  records=new_quick->records;
	  if (quick)
	    my_free((gptr) quick,MYF(0));
	  quick=new_quick;
	}
	else
	  my_free((gptr) new_quick,MYF(0));
      }
    }
    if (param.root.next)
    {
      param.use->next=param.free;		/* block to free-link */
      param.free=param.root.next;
    }
    if (error < 0)
    {
      free_param(&param);
      DBUG_RETURN(-1);				/* Error in select */
    }
    if (quick && (select->head->form.db_type != DB_TYPE_ISAM ||
		  records < 5L))
      break;					/* Found key; Use it */
  }
  select->quick=quick;				/* Use best found range */
  free_param(&param);
  DBUG_RETURN(test(quick));
} /* test_quick_select */



inline static void stack_or(SELECT_STACK *stack)
{
  uint a,b,c;
  if ((a=stack[0].result))
  {
    if ((b=stack[1].result))
    {
      if ((c=((a | b) & (2+4))) & 2)
	c|=8;					/* Mark that ref isn't true */
      stack[0].result= (uint8) ((a & b & 1) | c);
      return;
    }
  }
  stack[0].result=0;
}

	/* test if a key is used through hole select */

static int NEAR_F parse_quick_select(PARAM *param, register SELECT *select, uint fi_nr, uint keynr)
{
  reg2 uint flag;
  uint8 res;
  int level;
  uint regnr,fieldnr,optflag;
  SELECT_TYPE *diff;
  SELECT_STACK *stack;
  S_ARG *use_pos;
  reg3 SELECT_ROW *row;
  FIELD *field;
  DBUG_ENTER("parse_quick_select");

  row=select->row;
  level=0;

  param->pos_level[0]= &param->root;		/* Points at root-block */
  stack=select->stack;
  stack[0].result=0;
  stack[0].flag=0;

  while ((flag=row->flag))
  {
    if (flag & SELECT_FLAG_RIGHT_BRACE)
    {
      if (stack->flag == SELECT_FLAG_OR)
      {
	level--; stack--;
	stack_or(stack);
	select_or(param,level);
      }
      if (level)
      {						/* if not lonely ')' */
	if (stack->flag & SELECT_FLAG_AND)
	{
	  level--; stack--;
	  stack[0].result|=select->stack[1].result;
	  if (select_and(param,level))
	    goto err;
	}
	else
	  stack->flag=SELECT_FLAG_OR;
      }
    }

    if (flag & SELECT_FLAG_OR && stack->flag == SELECT_FLAG_OR)
    {
      level--; stack--;
      stack_or(stack);
      select_or(param,level);
    }

    if (flag & SELECT_FLAG_LEFT_BRACE)
    {						/* '(' found */
      DBUG_PRINT("test",("upp one level to: %d",level+1));
      level++; stack++;
      stack->flag= flag;
      stack->result=0;
      param->pos_level[level]= param->use;
    }

    param->pos_level[level+1]=param->use;
    if ((flag & SELECT_FLAG_NO_FIELDS) == 0)
    {						/* Not Empty level */
      if (get_new_param(param))
	goto err;
      use_pos=param->use;
      res=0;

      if (flag & SELECT_FLAG_ALWAYS_FALSE)
      {
	res=1;					/* same as found key */
	use_pos->min_flag=use_pos->max_flag= IMPOSSIBLE_RANGE;
	goto skipp;
      }
      optflag= flag & SELECT_FLAG_OP;

      if ((row->reg_used & param->pre_reg_read) &&
	  !(row->reg_used & ~select->reg_init_read))
	res=2;					/* Can affect quick test */

      if (flag & SELECT_FLAG_ALWAYS_TRUE ||
	  row->field_type != SEL_FIELD_FIELD ||
	  (row->diff_type != SEL_FIELD_FIELD &&
	   row->diff_type != SEL_FIELD_CONST_STR &&
	   row->diff_type != SEL_FIELD_CONST_NUM) ||
	  optflag == 2 || optflag == 8)
	goto skipp;

      regnr=(uint) row->field.field.regnr;
      field=row->field.field.regfield;
      if (!select->keys_optimized &&
	  row->diff_type != SEL_FIELD_FIELD &&
	  change_field_to_ref_field(select->head,select->forms,regnr,
				    (uint) (field-select->forms[regnr]->field),
				    &regnr,&fieldnr))
	field=select->forms[regnr]->field+fieldnr;

      diff= &row->diff;
      if (field->str != select->head->form.field[fi_nr].str)
      {
	if (row->diff_type != SEL_FIELD_FIELD ||
	    diff->field.regfield->str != select->head->form.field[fi_nr].str)
	  goto skipp;				/* Not key field */
	field= diff->field.regfield;		/* Swap cmp fields */
	diff= &row->field;
      }
      if (row->diff_type == SEL_FIELD_FIELD)
      {
	if (!(select->const_reg & (1L << diff->field.regnr)))
	{
	  if (res == 2)
	    res=2|4;				/* Key ranged by prev table */
	  goto skipp;
	}
      }
      if (add_quick_test(param,field,row,&row->diff))
	goto skipp;

      res=1;					/* We are affecting test */
skipp:						/* Here when wrong fieldref */
      if (!(res & 1))
	use_pos->min_flag=ERROR_RANGE;		/* If wrong field */

      if (flag & (SELECT_FLAG_AND | SELECT_FLAG_LEFT_BRACE))
      {
	stack->result|=res;
	if (select_and(param,level))
	  goto err;
      }
      else
      {
	level++; stack++;
	stack->flag= SELECT_FLAG_OR;
	stack->result=res;
	param->pos_level[level]= use_pos->prev;
      }
    }
    row++;
#ifdef EXTRA_DEBUG
    print_param(param,level);
#endif
  }

  while (level-- > 0)				/* Test last levels */
  {						/* Some ')' is not written */
    stack--;
    if (stack[1].flag & SELECT_FLAG_AND)
    {
      if (select_and(param,level))
	goto err;
      stack[0].result|=stack[1].result;
    }
    else
    {
      select_or(param,level);
      stack_or(stack);
    }
#ifdef EXTRA_DEBUG
    print_param(param,level);
#endif
  }
    /* Test if one can use key if previous tables are read */
  if ((stack[0].result & (1 | 4)) &&
      (test_all_bits(stack[0].result,2|8)))
    select->neaded_reg|=(1L << keynr);
  if (!(stack[0].result & 1))
    DBUG_RETURN(0);				/* select not ranged by key */
  select->quick_keys|=	(1L << keynr);
  if (param->root.next->min_flag & (IMPOSSIBLE_RANGE | ERROR_RANGE))
    DBUG_RETURN(-1);				/* Impossibly select */

	/* Return 1 if one range from min to max */
  DBUG_RETURN(param->use != param->root.next ||
	      param->use->min_flag != NO_MIN_RANGE ||
	      param->use->max_flag != NO_MAX_RANGE);
err:
  DBUG_RETURN(0);				/* Got error */
} /* parse_quick_select */



/*****************************************************************************
**	Add field range to ranges check
**	return 0 if field was added
*****************************************************************************/

static int add_quick_test(PARAM *param, FIELD *field, SELECT_ROW *row, SELECT_TYPE *diff)
{
  int dec;
  uint length,org_length,optflag,flag,keylength;
  char buff[MAX_FIELD_WIDTH],*pos,fill;
  S_ARG *use_pos;
  SELECT_TYPE tmp;
  char wild[4];

  use_pos=param->use;
  keylength=param->keylength;
  flag=row->flag;
  optflag= flag & SELECT_FLAG_OP;
  wild[0]=wild_one; wild[1]=wild_many; wild[2]=wild_prefix; wild[3]=0;

  if (row->diff_type == SEL_FIELD_FIELD)
  {						/* Change field to const */
    if (row->cmp_type == SEL_CMP_NUM)
      tmp.nr= valfield(diff->field.regfield);
    else
    {
      VOID(r_unpackf_stripp(diff->field.regfield,buff));
      tmp.str=buff;
    }
    diff=&tmp;
  }

  if (f_is_alpha(param->type))
  {						/* Stringcompare */
    /* Don't use packed string if not = const */
    if (f_is_packed(field->pack_flag) &&
	(optflag != 1 || flag & SELECT_FLAG_WILD_COMPARE))
      return 1;

    if (flag & SELECT_FLAG_WILD_COMPARE)
    {					/* Wildcompare */
      if ((pos=strcont(diff->str,wild)))
      {
	if ((length=(uint) (pos - diff->str)) == 0)
	  return 1;
      }
      else
	length=strlen(diff->str);
      org_length=length;
      if (length > keylength)
	length=keylength;
      VOID(strmake(buff,diff->str,length));
    }
    else
    {
      if (optflag != 7)
      {					/* Normal stringcompare */
	if (row->cmp_type != SEL_CMP_NUM)
	{
	  org_length=strlen(diff->str);
	  VOID(strmake(buff,diff->str,keylength));
	}
	else
	{
	  packnrf(diff->nr,field);
	  r_unpackf(field,buff);
	  org_length=strlen(buff);
	  buff[keylength]=0;
	}
      }
      else
      {						/* Soundex string */
	optflag=1;				/* Test as == */
	org_length=1;
	buff[0]=diff->str[0];
	buff[1]=0;
      }
    }
    if (((param->baseflag & HA_BINARY_KEYS) &&
	 row->cmp_type != SEL_CMP_BINARY_STR) ||
	(! (param->baseflag & HA_READ_ORDER) &&
	 (keylength != (uint) strlen(buff) || optflag != 1)))
      return 1;

    switch (optflag) {
    case 1:					/* find == key */
      VOID(strmov(use_pos->min_arg.str.str,buff));
      VOID(strmov(use_pos->max_arg.str.str,buff));
      break;
    case 3:					/* < buff */
      if (keylength >= org_length)
	use_pos->max_flag=NEAR_KEY;		/* < max */
    case 4:					/* <= buff */
      use_pos->min_flag=NO_MIN_RANGE;	/* From start */
      VOID(strmov(use_pos->max_arg.str.str,buff));
      break;
    case 5:					/* > buff */
      if (keylength >= org_length)
	use_pos->min_flag=NEAR_KEY;		/* > min */
    case 6:
      use_pos->max_flag=NO_MAX_RANGE;	/* to end */
      VOID(strmov(use_pos->min_arg.str.str,buff));
      break;
    default:					/* Impossible */
      break;
    }
  }
  else					/* We are comparing numbers */
  {
    if (row->cmp_type != SEL_CMP_NUM)
    {
      org_length=strlen(diff->str);
      VOID(strmake(buff,diff->str,keylength));
    }
    else
    {
      packnrf(diff->nr,field);
      r_unpackf(field,buff);
      org_length=strlen(buff);
      buff[keylength]=0;
    }
    if (flag & SELECT_FLAG_WILD_COMPARE && (pos=strcont(buff,wild)))
    {
      if (strchr(buff,wild_many))
	return 1;
      if ((length=(uint) (pos - buff)) == 0 ||
	  (f_is_dec(field->pack_flag) && ! isdigit(pos[-1])))
	return 1;
      DBUG_PRINT("test",("Number with joker.  length: %d  keylength: %d",
			 length,keylength));
    }
    else
      length=strlen(buff);
    if (! (param->baseflag & HA_READ_ORDER) &&
	(keylength != length || optflag != 1))
      return 1;

    if (length < keylength)
    {				/* Find lower and upper boarder */
      fill= '0';
      if (strchr(buff,'-'))
	fill='9';
      VOID(strfill(buff+length,keylength-length,fill));
      dec=f_decimals(field->pack_flag);
      if (dec && (int) field->length - (int) keylength > dec)
	buff[field->length-dec-1] = '.';
      use_pos->min_arg.nr=atod(buff);
      fill^= '9' ^ '0';
      VOID(strfill(buff+length,keylength-length,fill));
      if (dec && (int) field->length - (int) keylength > dec)
	buff[field->length-dec-1] = '.';
      use_pos->max_arg.nr=atod(buff);
    }
    else
      use_pos->min_arg.nr=use_pos->max_arg.nr=atod(buff);

    switch (optflag) {
    case 3:					/* < buff */
      if (keylength >= org_length)
      {
	use_pos->max_flag=NEAR_KEY;		/* < max */
	use_pos->max_arg.nr=use_pos->min_arg.nr;
      }
      /* fall through */
    case 4:					/* <= buff */
      use_pos->min_flag=NO_MIN_RANGE;		/* From start */
      break;
    case 5:					/* > buff */
      if (keylength >= org_length)
      {
	use_pos->min_flag=NEAR_KEY;		/* > min */
	use_pos->min_arg.nr=use_pos->max_arg.nr;
      }
      /* fall through */
    case 6:
      use_pos->max_flag=NO_MAX_RANGE;		/* to end */
      break;
    default:
      break;
    }
  }
  return 0;
}


	/* Get pointer to new param-struct and init it */

static int NEAR_F get_new_param(register PARAM *param)
{
  reg1 S_ARG *new_pos;
  DBUG_ENTER("get_new_param");

  if ((new_pos=param->free))
  {
    param->free=new_pos->next;
  }
  else
  {
    if (! (new_pos= (S_ARG*) my_malloc(sizeof(S_ARG),MYF(MY_WME))))
      DBUG_RETURN(1);
  }
  new_pos->min_flag=new_pos->max_flag=IDENTICAL_KEY;
  new_pos->min_arg.str.str[0]=new_pos->max_arg.str.str[0]=0;
  new_pos->prev=param->use;
  param->use->next = new_pos;
  param->use=new_pos; new_pos->next = 0;

  DBUG_RETURN(0);
} /* get_new_param */


	/* do AND between (two ?) selects-levels */

static int NEAR_F select_and(PARAM *param, int level)
{
  reg1 S_ARG *pos,*next;
  S_ARG *last,*first;
  DBUG_ENTER("select_and");
  DBUG_PRINT("enter",("from: %lx  to: %lx",param->pos_level[level]->next,
		      param->pos_level[level+1]->next));

  last= param->use;
  for (pos=param->pos_level[level]->next ;
       pos != param->pos_level[level+1]->next ;
       pos=pos->next)
  {
    for (next=param->pos_level[level+1]; next != last; )
    {
      next=next->next;
      if (do_and(param,pos,next))
	DBUG_RETURN(1);
    }
  }


  if (last != param->use)
  {						/* More than one param */
	/* free used select-args */
    first=last->next;				/* First of new select-args */
    pos=param->free;
    param->free=param->pos_level[level]->next;
    last->next=pos;

	/* get select-args in linked buffer */
    param->use= param->pos_level[level]; param->use->next=0;

    while (first)
      get_next_arg(param,&first);		/* Put next in param->use */
  }
  param->pos_level[level+1]= param->use;
  DBUG_RETURN(0);
} /* select_and */


	/* Put in new_param AND of a and b */
	/* Returns != 0 if error (out of memory) */

static int NEAR_F do_and(PARAM *param, S_ARG *a, S_ARG *b)
{
  S_ARG *from,*next;
  int ret;

  if (get_new_param(param))
    return 1;					/* End of memory */
  next=param->use;				/* Next pos to use */

	/* IMPOSSIBLE ranges can't be anded */
  if (a->min_flag == IMPOSSIBLE_RANGE || b->min_flag == IMPOSSIBLE_RANGE)
  {
    next->min_flag = IMPOSSIBLE_RANGE;
    return 0;
  }
	/* ERROR ranges can be ANDed and act like a true */
  if (a->min_flag == ERROR_RANGE || b->min_flag == ERROR_RANGE)
  {
    if ((from=a)->min_flag == ERROR_RANGE)
      from=b;
    next->min_flag=from->min_flag; next->max_flag=from->max_flag;
    memcpy((byte*) &next->min_arg,(byte*) &from->min_arg,
	   sizeof(STR_OR_DOUBLE));
    memcpy((byte*) &next->max_arg,(byte*) &from->max_arg,
	   sizeof(STR_OR_DOUBLE));
    return 0;
  }
	/* Put in next max of a->min_arg & b-> min_arg */

  from=a;
  if (b->min_flag != NO_MIN_RANGE &&
      (a->min_flag == NO_MIN_RANGE ||
       (ret=qs_cmp(&a->min_arg,&b->min_arg,param->type,0)) < 0 ||
       (!ret && b->min_flag == NEAR_KEY)))
    from=b;
  next->min_flag = from->min_flag;
  memcpy((byte*) &next->min_arg,(byte*) &from->min_arg,sizeof(STR_OR_DOUBLE));

	/* Put in next min of a->max_arg & b->max_arg */

  from=a;
  if (b->max_flag != NO_MAX_RANGE &&
      (a->max_flag == NO_MAX_RANGE ||
       (ret=qs_cmp(&a->max_arg,&b->max_arg,param->type,1+2)) > 0 ||
       (!ret && b->max_flag == NEAR_KEY)))
    from=b;
  next->max_flag = from->max_flag;
  memcpy((byte*) &next->max_arg,(byte*) &from->max_arg,sizeof(STR_OR_DOUBLE));

	/* test if min_arg > max_arg */

  if (next->min_flag != NO_MIN_RANGE && next->max_flag != NO_MAX_RANGE &&
      ((ret=qs_cmp(&next->min_arg,&next->max_arg,param->type,2)) > 0 ||
       (!ret && (next->min_flag == NEAR_KEY || next->max_flag == NEAR_KEY))))
    next->min_flag = IMPOSSIBLE_RANGE;		/* Select can't be AND:ed */
  return 0;
} /* do_and */


	/* Do or between two select */
	/* Result is put in first struct */
	/* Returns != 0 if second struct can be skipped */

static int NEAR_F do_or(PARAM *param, S_ARG *a, S_ARG *b)
{
  int ret;

	/* ERROR ranges can't be OR ed */
  if (a->min_flag == ERROR_RANGE || b->min_flag == ERROR_RANGE)
  {
    a->min_flag=ERROR_RANGE;			/* Can't OR ERROR_RANGE */
    goto end;					/* Skipp b */
  }
	/* IMPOSSIBLE range act like FALSE when OR ing */
  if (b->min_flag == IMPOSSIBLE_RANGE)
    goto end;					/* Skipp impossible */
  if (a->min_flag == IMPOSSIBLE_RANGE)
  {						/* Move info to a */
    a->min_flag = b->min_flag;
    a->max_flag = b->max_flag;
    memcpy((byte*) &a->min_arg,(byte*) &b->min_arg,sizeof(STR_OR_DOUBLE));
    memcpy((byte*) &a->max_arg,(byte*) &b->max_arg,sizeof(STR_OR_DOUBLE));
    goto end;					/* Skipp b */
  }

	/* test if a->min_arg > b->max_arg  || b->min_arg > a->max_arg */

  if ((a->min_flag != NO_MIN_RANGE && b->max_flag != NO_MAX_RANGE &&
       ((ret=qs_cmp(&a->min_arg,&b->max_arg,param->type,2)) > 0 ||
       (!ret && (a->min_flag == NEAR_KEY || b->max_flag == NEAR_KEY)))) ||
      (a->max_flag != 1 && b->min_flag != 1 &&
       ((ret=qs_cmp(&a->max_arg,&b->min_arg,param->type,1)) < 0 ||
	(!ret && (a->min_flag == NEAR_KEY || b->max_flag == NEAR_KEY)))))
      return 0;				/* Yes, can't do inline or done */

	/* Put in a->min_arg min of (a->min_arg,b->min_arg) */

  if (b->min_flag == NO_MIN_RANGE ||
      (a->min_flag != NO_MIN_RANGE &&
       ((ret=qs_cmp(&a->min_arg,&b->min_arg,param->type,0)) > 0 ||
       (!ret && a->min_flag == NEAR_KEY))))
  {
    a->min_flag = b->min_flag;
    memcpy((byte*) &a->min_arg,(byte*) &b->min_arg,sizeof(STR_OR_DOUBLE));
  }

	/* Put in a->max_arg max of (a->max_arg,b->max_arg) */

  if (b->max_flag == NO_MAX_RANGE ||
      (a->max_flag != NO_MAX_RANGE &&
       ((ret=qs_cmp(&a->max_arg,&b->max_arg,param->type,1+2)) < 0 ||
       (!ret && a->max_flag == NEAR_KEY))))
  {
    a->max_flag = b->max_flag;
    memcpy((byte*) &a->max_arg,(byte*) &b->max_arg,sizeof(STR_OR_DOUBLE));
  }

	/* We don't nead b; remove it */
end:
  b->prev->next=b->next;
  b->next=param->free;
  param->free=b;
  return 1;					/* b removed */
} /* do_or */


	/* Get smallest select-arg from first -- last and put in param->use */
	/* Merge found arg with all rest */
	/* If ERROR_RANGE is found then return it and free everything */

static void NEAR_F get_next_arg(PARAM *param, S_ARG **first)
{
  S_ARG *found,*pos;
  DBUG_ENTER("get_next_arg");

  found= *first;
  for (pos= (*first)->next ; pos ; pos= pos->next)
  {
    if (do_or(param,found,pos))
    {
      pos=pos->prev;				/* Pos removed, get previous */
      continue;
    }
    if (pos->min_flag == NO_MIN_RANGE ||
	(found->min_flag != NO_MIN_RANGE &&
	 qs_cmp(&found->min_arg,&pos->min_arg,param->type,0) > 0))
      found=pos;
  }
  if (found == *first)
    *first= (*first)->next;			/* unlink first */
  else
    found->prev->next=found->next;
  if (found->min_flag == ERROR_RANGE)
  {						/* unlink all */
    while (*first)
    {
      pos= (*first)->next;
      (*first)->next=param->free;
      param->free= *first;
      *first= pos;
    }
  }
  param->use->next=found;			/* Link in after param->use */
  found->prev=param->use;
  param->use=found; found->next=0;

  DBUG_VOID_RETURN;
} /* get_next_arg */


	/* do OR and merge all selects-levels */

static void NEAR_F select_or(PARAM *param, int level)
{
  S_ARG *first;
  DBUG_ENTER("select_or");

	/* free used select-args */
  first=param->pos_level[level]->next;		/* Selects starts here */
  param->use=first->prev; param->use->next=0;

  while (first)
    get_next_arg(param,&first);			/* Put next in param->use */
  param->pos_level[level+1]= param->use;
  DBUG_VOID_RETURN;
} /* select_or */


	/* return a SELECT_QUICK struct to be used by quick_select() */

static SELECT_QUICK * NEAR_F make_quick_select(register SELECT *select, PARAM *param, uint index, uint fi_nr)
{
  reg1 S_ARG *pos;
  SELECT_QUICK_REGION *region;
  uint count,length,min_key_length,max_key_length;
  my_bool is_num;
  FIELD *field;
  SELECT_QUICK *quick;
  char *min_key_buff,*min_key,*max_key;
  ulong records;
  DBUG_ENTER("make_quick_select");

  for (pos=param->root.next,count=0; pos ; pos=pos->next)
    count++;
  if (!(min_key_buff=(char*) my_alloca(ha_max_key_length)))
    DBUG_RETURN((SELECT_QUICK*) 0);
  if (!(my_multi_malloc(MYF(MY_WME),
			&quick,sizeof(*quick),
			&region,sizeof(*region)*count,
			NullS)))
  {
    my_afree(min_key_buff);
    DBUG_RETURN((SELECT_QUICK*) 0);
  }
  quick->region=quick->pos= region;
  quick->end=quick->region+count;
  quick->index=(int) index;
  quick->next=0;
  field=quick->field=select->head->form.field+fi_nr;
  length=packlength(field);
  quick->records=0;
  is_num=f_is_num(field->pack_flag);
  min_key_length=max_key_length=0;

  for (pos=param->root.next, region=quick->region;
       pos ;
       pos=pos->next,region++)
  {
    region->min_flag=pos->min_flag;
    region->max_flag=pos->max_flag;
    memcpy((byte*) &region->min_arg,(byte*) &pos->min_arg,
	   sizeof(STR_OR_DOUBLE));
    memcpy((byte*) &region->max_arg,(byte*) &pos->max_arg,
	   sizeof(STR_OR_DOUBLE));
    if (select->head->form.db_type == DB_TYPE_ISAM)
    {
      min_key=max_key=0;
      if (region->min_flag != NO_MIN_RANGE)
      {
	if (is_num)
	{
	  packnrf(region->min_arg.nr,field);
	  memcpy(min_key_buff,field->str,
		 (size_t) min(length,ha_max_key_length));
	  min_key=min_key_buff;
	  min_key_length=length;
	}
	else
	{
	  packf(region->min_arg.str.str,field);
	  memcpy(min_key_buff,field->str,
		 (size_t) min(length,ha_max_key_length));
	  min_key=min_key_buff;
	  min_key_length=strlen(region->min_arg.str.str);
	}
      }
      if (region->max_flag != NO_MAX_RANGE)
      {
	if (is_num)
	{
	  packnrf(region->max_arg.nr,field);
	  max_key=field->str;
	  max_key_length=length;
	}
	else
	{
	  packf(region->max_arg.str.str,field);
	  max_key=field->str;
	  max_key_length=strlen(region->max_arg.str.str);
	}
      }
      if (min_key && max_key && region->min_flag != NEAR_KEY &&
	  region->max_flag != NEAR_KEY &&
	  select->head->form.key_info[index].key_length == length &&
	  !select->head->form.key_info[index].dupp_key &&
	  !memcmp(min_key,max_key,min_key_length))
	records=1;				/* Assume we find it */
      else
	records=ni_records_in_range((N_INFO*) select->head->form.file,
				    (int) index,
				    min_key,
				    (is_num || f_is_packed(field->pack_flag) ||
				     !min_key || ! min_key_length) ?
				    length : min_key_length,
				    (region->min_flag == NEAR_KEY ?
				     HA_READ_AFTER_KEY : HA_READ_KEY_EXACT),
				    max_key,
				    (is_num || f_is_packed(field->pack_flag) ||
				     !max_key || !max_key_length) ?
				    length : max_key_length,
				    (region->max_flag == NEAR_KEY ?
				     HA_READ_BEFORE_KEY : HA_READ_AFTER_KEY));
      if (records != (ulong) -1L)
	quick->records+=records;
    }
  }
  DBUG_PRINT("exit",("index: %d  records: %ld",index,quick->records));
  my_afree(min_key_buff);
  DBUG_RETURN(quick);
} /* make_quick_select */


	/* free everything with param */

static void NEAR_F free_param(PARAM *param)
{
  S_ARG *pos,*next;

  for (pos=param->free; pos ; pos=next)
  {
    next=pos->next;
    my_free((gptr) pos,MYF(0));
  }
  my_free((gptr) param->pos_level,MYF(0));
} /* free_param */


	/* compare two search-args to to find smallest */

static int NEAR_F qs_cmp(STR_OR_DOUBLE *a, STR_OR_DOUBLE *b, uint type, uint flag)
                    
          				/* Type according to field */
          				/* | 1 if a == maxarg */
					/* | 2 if b == maxarg */
{
  reg1 char *a_pos,*b_pos;
  int cmp;
  double nr;

  if (type & 2)
    return ((nr= a->nr - b->nr) < 0.0 ? -1 : nr > 0.0); /* Float */
  a_pos= &a->str.str[0]; b_pos= &b->str.str[0];
  if (type & 1)
  {					/* Binary keys */
    while ((cmp= (int) (uchar) *a_pos - (int) (uchar) *b_pos) == 0)
    {
      if (*a_pos++ == 0)
	return 0;
      b_pos++;
    }
  }
  else
  {
    while ((cmp= (int) my_sort_order[(uchar) *a_pos] -
	    (int) my_sort_order[(uchar) *b_pos]) == 0)
    {
      if (*a_pos++ == 0)
	return 0;
      b_pos++;
    }
  }
  if (*a_pos == 0)
    return flag & 1 ? 1 : -1;
  if (*b_pos == 0)
    return flag & 2 ? -1 : 1;
  return cmp;
} /* qs_cmp */


	/* Optimize select by substituting use of keys in select with
	   reference fields when comparing to constant
	*/

void substitute_keys(SELECT *select)
{
  reg1 SELECT_ROW *row;
  uint regnr,fieldnr,reg_used,flag;
  DBUG_ENTER("substitue_keys");

  reg_used=0;
  for (row=select->row; (flag=row->flag) ; row++)
  {
    if ((flag & SELECT_FLAG_NO_FIELDS) == 0 &&
	row->field_type == SEL_FIELD_FIELD &&
	(row->diff_type == SEL_FIELD_CONST_STR ||
	 row->diff_type == SEL_FIELD_CONST_NUM))
    {						/* Not Empty level */
      regnr=(uint) row->field.field.regnr;
      if (change_field_to_ref_field(select->head,select->forms,
				    regnr,
				    (uint) (row->field.field.regfield -
					    select->forms[regnr]->field),
				    &regnr,&fieldnr))
      {
	row->field.field.regnr= (int) regnr;
	row->reg_used= (1L << regnr);
	row->field.field.regfield= select->forms[regnr]->field+fieldnr;
      }
    }
    reg_used|=row->reg_used;
  }
  select->reg_used=reg_used;
  select->keys_optimized=1;
  DBUG_PRINT("exit",("reg_used: %x",reg_used));
  DBUG_VOID_RETURN;
} /* substitue_keys */


/*****************************************************************************
**	change a field to a ref-field if possibly
**	ie ; #table1.field = #table2.field -> #table1.field = #table1.field
*****************************************************************************/

static int change_field_to_ref_field(TABLE *head, TABLE **first_table,
				     uint regnr, uint fieldnr, uint *out_regnr,
				     uint *out_fieldnr)
{
  uint ref_fields;
  TABLE *table;

  ref_fields=0;					// Keep gcc happy
  if ((table=first_table[regnr]) != head &&
      (ref_fields=table->form.reginfo.ref_fields))
  {
    reg1 uint i;
    KEY *keyinfo=table->form.key_info+table->form.reginfo.ref_key;
    uint key_parts=(uint) keyinfo->key_parts,length;
    FIELD *field=table->field+fieldnr,*key_field,*ref_field;

    set_if_smaller(ref_fields,key_parts);
    for (i=0 ; i < ref_fields ; i++)
    {
      if (!(fieldnr=keyinfo->key_part[i].fieldnr))
	break;
      key_field= table->field+fieldnr-1;
      ref_field= table->form.reginfo.ref_field[i].field;
      if ((length=packlength(key_field)) != packlength(ref_field))
	break;					/* Key isnt ref-field */
      if (field->str == key_field->str && packlength(field) == length &&
	  ref_field->table_name)
      {
	*out_regnr= ref_field->table->tablenr;
	*out_fieldnr= (uint) (ref_field - first_table[*out_regnr]->field);
	DBUG_PRINT("info",("change #%d.%d to #%d.%d",
			   regnr,fieldnr,*out_regnr,*out_fieldnr));
	return change_field_to_ref_field(head,first_table,*out_regnr,
					 *out_fieldnr,
					 out_regnr,out_fieldnr) | 1;
      }
    }
  }
  return 0;
}


	/* Print param structs on DBUG-FILE */

#ifdef EXTRA_DEBUG

static void NEAR_F print_param(PARAM *param,int level)
{
  reg1 S_ARG *pos,*prev;
  int i;
  DBUG_ENTER("print_param");

  if (! _db_on_)
    DBUG_VOID_RETURN;

  LINT_INIT(prev);
  if (param->root.next)
  {
    for (pos=param->root.next; pos ; pos=pos->next)
    {
      prev=pos;
      if (pos->min_flag & (IMPOSSIBLE_RANGE | ERROR_RANGE))
	fprintf(DBUG_FILE,"pos: %lx  min_flag: %d\n",
		pos,pos->min_flag);
      else
	if (!(param->type & 2))
	  fprintf(DBUG_FILE,"pos: %lx\nmin_flag: %d  min_buff: '%s'\nmax_flag: %d  max_buff: '%s'\n",
		  pos,pos->min_flag,
		  pos->min_arg.str.str ? pos->min_arg.str.str : "(null)",
		  pos->max_flag,
		  pos->max_arg.str.str ? pos->max_arg.str.str : "(null)");
	else
	  fprintf(DBUG_FILE,"pos: %lx\nmin_flag: %d  min_nr: %lg\nmax_flag: %d  max_nr: %lg\n",
		  pos,pos->min_flag,pos->min_arg.nr,
		  pos->max_flag,pos->max_arg.nr);
    }
    if (param->use != prev)
      fprintf(DBUG_FILE,"param->use: %lx\n",param->use);
  }
  if (param->free)
  {
    fprintf(DBUG_FILE,"Free blocks:\n");
    for (pos=param->free; pos ; pos=pos->next)
      fprintf(DBUG_FILE,"%8lx  ",pos);
    fprintf(DBUG_FILE,"\n");
  }

  fprintf(DBUG_FILE,"Level pos (%d):\n",level);
  for (i=0 ; i <= level ; i++)
    fprintf(DBUG_FILE,"%8lx->%8lx  ",param->pos_level[i],
	    param->pos_level[i]->next);
  fprintf(DBUG_FILE,"\n");

  DBUG_VOID_RETURN;
} /* print_param */
#endif
