/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/*
** Calculation functions
** Warning: Some string functions doesn't always put and end-null on a String
** (Shouldn't be neaded)
*/

#pragma implementation

#include "mysql_priv.h"
#include <m_ctype.h>

String empty_string("");

uint nr_of_decimals(char *str)
{
  if ((str=strchr(str,'.')))
  {
    char *start= ++str;
    for ( ; isdigit(*str) ; str++) ;
    return (uint) (str-start);
  }
  return 0;
}


/* return TRUE if item is a constant */

bool
const_item(Item *item)
{
  switch (item->type()) {
  case Item::INT_ITEM:
  case Item::REAL_ITEM:
  case Item::NULL_ITEM:
  case Item::STRING_ITEM:
    return TRUE;
  default:
    return FALSE;
  }
}


bool
eval_const_cond(COND *cond)
{
  if (cond->type() != Item::FUNC_ITEM ||
      ((Item_func*) cond)->functype() == Item_func::UNKNOWN_FUNC)
    return 0;
  return ((Item_func*) cond)->val() ? TRUE : FALSE;
}


Item_func::Item_func(List<Item> &list)
{
  arg_count=list.elements;
  args=(Item**) sql_alloc(sizeof(Item*)*arg_count);

  uint i=0;
  List_iterator<Item> li(list);
  Item *item;

  while ((item=li++))
    args[i++]= item;
  list.empty();					// Fields are used
}

Item_func::~Item_func()
{
  for (uint i=0 ; i < arg_count ; i++)
    delete args[i];
}

bool
Item_func::fix_fields(THD *thd,tname_t *tables,const string where)
{
  maybe_null=0;
  for (uint i=0 ; i < arg_count ; i++)
  {
    if (args[i]->fix_fields(thd,tables,where))
      return 1;
    if (args[i]->maybe_null)
      maybe_null=1;
  }
  fix_length_and_dec();
  return 0;
}

ulong Item_func::used_tables()
{
  ulong tables=0;
  for (uint i=0 ; i < arg_count ; i++)
    tables|=args[i]->used_tables();
  return tables;
}

double Item_str_func::val()
{
  String s,*res;
  res=str(&s);
  return res ? atof(res->c_ptr()) : 0.0;
}

String *Item_num_func::str(String *str)
{
  double nr=val();
  if (null_value)
    return 0;
  else
    str->set(nr,decimals);
  return str;
}

void Item_num_func::fix_length_and_dec()
{
  decimals=0;
  maybe_null=0;
  for (uint i=0 ; i < arg_count ; i++)
  {
    set_if_bigger(decimals,args[i]->decimals);
    maybe_null|= args[i]->maybe_null;
  }
  max_length=float_length(decimals);
}


double Item_func_plus::val()
{
  double val=args[0]->val()+args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val;
}


double Item_func_minus::val()
{
  double val=args[0]->val() - args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val;
}


double Item_func_mul::val()
{
  double val=args[0]->val()*args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val;
}


double Item_func_div::val()
{
  double val=args[0]->val();
  double val2=args[1]->val();
  if ((null_value= val2 == 0.0 || args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val/val2;
}

void Item_func_div::fix_length_and_dec()
{
  decimals=max(args[0]->decimals,args[1]->decimals)+2;
  max_length=args[0]->max_length - args[0]->decimals + decimals;
  uint tmp=float_length(decimals);
  set_if_smaller(max_length,tmp);
}


double Item_func_mod::val()
{
  double val= floor(args[0]->val()+0.5);
  double val2=floor(args[1]->val()+0.5);
  if ((null_value=val2 == 0.0 || args[0]->null_value || args[1]->null_value))
    return 0.0;
  return fmod(val,val2);
}

void Item_func_mod::fix_length_and_dec()
{
  max_length=args[1]->max_length;
  decimals=0;
}


double Item_func_neg::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return -val;
}

void Item_func_neg::fix_length_and_dec()
{
  decimals=args[0]->decimals;
  max_length=args[0]->max_length;
}


double Item_func_abs::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return fabs(val);
}

double Item_func_log::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val <= 0.0)))
    return 0.0;
  return log(val);
}

double Item_func_log10::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val <= 0.0)))
    return 0.0;
  return log10(val);
}

double Item_func_exp::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return exp(val);
}

double Item_func_sqrt::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val < 0)))
    return 0.0;
  return sqrt(val);
}

double Item_func_pow::val()
{
  double val=args[0]->val();
  double val2=args[1]->val();
  if ((null_value=(args[0]->null_value || args[1]->null_value)))
    return 0.0;
  return pow(val,val2);
}

void Item_func_integer::fix_length_and_dec()
{
  max_length=args[0]->max_length - args[0]->decimals+1;
  uint tmp=float_length(decimals);
  set_if_smaller(max_length,tmp);
  decimals=0;
}

double Item_func_ceiling::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return ceil(val);
}

double Item_func_floor::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return floor(val);
}

double Item_func_round::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return rint(val);
}

double Item_func_rand::val()
{
  if (arg_count)
  {					// Only use argument once in query
    randominit(&current_thd->rand,((ulong) args[0]->val())+55555555L);
    delete args[0];
    arg_count=0;
  }
  return rnd(&current_thd->rand);
}

double Item_func_sign::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return val < 0.0 ? -1.0 : (val  > 0 ? 1.0 : 0.0);
}

void Item_func_min_max::fix_length_and_dec()
{
  decimals=0;
  max_length=0;
  maybe_null=1;
  for (uint i=0 ; i < arg_count ; i++)
  {
    if (max_length < args[i]->max_length)
      max_length=args[i]->max_length;
    if (decimals < args[i]->decimals)
      decimals=args[i]->max_length;
    if (!args[i]->maybe_null)
      maybe_null=0;
  }
}


double Item_func_min::val()
{
  double value=0.0;
  null_value=1;
  for (uint i=0; i < arg_count ; i++)
  {
    if (null_value)
    {
      value=args[i]->val();
      null_value=args[i]->null_value;
    }
    else
    {
      double tmp=args[i]->val();
      if (tmp < value)
	value=tmp;
    }
  }
  return value;
}


double Item_func_max::val()
{
  double value=0.0;
  null_value=1;
  for (uint i=0; i < arg_count ; i++)
  {
    if (null_value)
    {
      value=args[i]->val();
      null_value=args[i]->null_value;
    }
    else
    {
      double tmp=args[i]->val();
      if (tmp > value)
	value=tmp;
    }
  }
  return value;
}


/*
** Test functions
** These returns 0.0 if false and 1.0 if true and null if some arg is null
** 'AND' and 'OR' never returns null
*/

double Item_func_not::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return !null_value && val == 0.0 ? 1.0 : 0.0;
}

double Item_func_and::val()
{
  if (!args[0]->val())
    return 0.0;
  if (!args[1]->val())
    return 0.0;
  return 1.0;
}

double Item_func_or::val()
{
  if (args[0]->val())
    return 1.0;
  if (args[1]->val())
    return 1.0;
  return 0.0;
}

int Item_bool_func2::compare()
{
  null_value=0;
  if (args[0]->result_type() == STRING_RESULT &&
      args[1]->result_type() == STRING_RESULT)
  {
    String *res1,*res2;
    res1=args[0]->str(&tmp_value1);
    res2=args[1]->str(&tmp_value2);
    if (!args[0]->null_value && !args[1]->null_value)
      return sortcmp(*res1,*res2);
  }
  else
  {
    double val1=args[0]->val();
    double val2=args[1]->val();

    if (!args[0]->null_value && !args[1]->null_value)
    {
      if (val1 < val2)  return -1;
      if (val1 == val2) return 0;
      return 1;
    }
  }
  null_value=1;
  return 0;
}

double Item_func_eq::val()
{
  int val=compare();
  if (null_value)
  {
    if (args[0]->type() == NULL_ITEM || 
	args[1]->type() == NULL_ITEM)
    {						// Compare with NULL const
      null_value=0;
      return (args[0]->null_value && args[1]->null_value) ? 1.0 : 0.0;
    }
  }
  return val ? 0.0 : 1.0;			// Compare with NULL result
}


double Item_func_ne::val()
{
  int val=compare();
  if (null_value)
  {
    if (args[0]->type() == NULL_ITEM || 
	args[1]->type() == NULL_ITEM)
    {						// Compare with NULL const
      null_value=0;
      return (args[0]->null_value && args[1]->null_value) ? 0.0 : 1.0;
    }
  }
  return val ? 1.0 : 0.0;			// Compare with NULL result
}


double Item_func_ge::val()
{
  int val=compare();
  return val >= 0 && !null_value ? 1.0 : 0.0;
}


double Item_func_gt::val()
{
  int val=compare();
  return val > 0 ? 1.0 : 0.0;
}

double Item_func_le::val()
{
  int val=compare();
  return val <= 0 && !null_value ? 1.0 : 0.0;
}


double Item_func_lt::val()
{
  int val=compare();
  return val < 0 ? 1.0 : 0.0;
}


/*
** Concatinate args with the following premissess
** If only one arg which is ok, return value of arg
** Don't reallocate str() if not absolute nessesarly needed.
*/

String *Item_func_concat::str(String *str)
{
  String *res,*res2,*use_as_buff;
  uint i;

  null_value=0;
  if (!(res=args[0]->str(str)))
    goto null;
  use_as_buff= &tmp_value;
  for (i=1 ; i < arg_count ; i++)
  {
    if (res->length() == 0)
    {
      if (!(res=args[i]->str(str)))
	goto null;
    }
    else
    {
      if (!(res2=args[i]->str(use_as_buff)))
	goto null;
      if (res2->length() == 0)
	continue;
      if (res->alloced_length() >= res->length()+res2->length())
      {						// Use old buffer
	res->append(*res2);
      }
      else if (str->alloced_length() >= res->length()+res2->length())
      {
	str->copy(*res);
	str->append(*res2);
	res=str;
      }
      else if (res == &tmp_value)
      {
	res->append(*res2);			// Must be a blob
      }
      else if (res2 == &tmp_value)
      {						// This can happend only 1 time
	tmp_value.replace(0,0,*res);
	res= &tmp_value;
	use_as_buff=str;			// Put next arg here 
      }
      else
      {						// Two big const strings
	(void) tmp_value.alloc(max_length);
	tmp_value.copy(*res);
	tmp_value.append(*res2);
	res= &tmp_value;
	use_as_buff=str;
      }	
    }
  }
  return res;

null:
  null_value=1;
  return 0;
}


void Item_func_concat::fix_length_and_dec()
{
  max_length=0;
  for (uint i=0 ; i < arg_count ; i++)
    max_length+=args[i]->max_length;
}

/*
** Replace all occurences of string2 in string1 with string3.
** Don't reallocate str() if not neaded
*/

String *Item_func_replace::str(String *str)
{
  String *res,*res2,*res3;
  int offset,extra_length;
  uint from_length,to_length;

  null_value=0;
  res=args[0]->str(str);
  if (args[0]->null_value)
    goto null;
  res2=args[1]->str(&tmp_value);
  if (args[1]->null_value)
    goto null;
  
  if (res2->length() == 0)
    return res;
  if ((offset=res->strstr(*res2)) < 0)
    return res;
  if (!(res3=args[2]->str(&tmp_value2)))
    goto null;
  from_length= res2->length();
  to_length=   res3->length();
  extra_length= (int) (res3->length() - from_length);
  do
  {
    if (extra_length > 0 && res->realloc(res->length()+(uint) extra_length*10))
      return res;			// Actually null, but don't care
    res->replace((uint) offset,from_length,*res3);
    offset+=(int) to_length;
  }
  while ((offset=res->strstr(*res2,(uint) offset)) >0);
  return res;

null:
  null_value=1;
  return 0;
}


void Item_func_replace::fix_length_and_dec()
{
  max_length=args[0]->max_length;
  int diff=(int) (args[2]->max_length - args[1]->max_length);
  if (diff > 0 && args[1]->max_length)
  {						// Calculate of maxreplaces
    max_length= max_length/args[1]->max_length;
    max_length= (max_length+1)*(uint) diff;
  }
}

String *Item_func_insert::str(String *str)
{
  String *res,*res2;
  uint start,length;

  null_value=0;
  res=args[0]->str(str);
  if (args[0]->null_value)
    goto null;
  res2=args[3]->str(&tmp_value);
  if (args[3]->null_value)
    goto null;
  start=(uint) args[1]->val()-1;
  length=(uint) args[2]->val();
  if (args[1]->null_value || args[2]->null_value)
    goto null;
  if (start > res->length()+1)
    return res;					// Wrong param; skipp insert
  if (length > res->length()-start)
    length=res->length()-start;
  res->replace(start,length,*res2);
  return res;
null:
  null_value=1;
  return 0;
}


void Item_func_insert::fix_length_and_dec()
{
  max_length=args[0]->max_length+args[3]->max_length;
}


String *Item_func_lcase::str(String *str)
{
  String *res;
  if (!(res=args[0]->str(str)))  
  {
    null_value=1;
    return 0;
  }
  null_value=0;
  res=copy_if_not_alloced(str_value,*res,res->length());
  res->casedn();
  return res;
}


String *Item_func_ucase::str(String *str)
{
  String *res;
  if (!(res=args[0]->str(str)))  
  {
    null_value=1;
    return 0;
  }
  null_value=0;
  res=copy_if_not_alloced(str_value,*res,res->length());
  res->caseup();
  return res;
}


double Item_func_length::val()
{
  String *res=args[0]->str(&value);
  if (!res)
    return 0.0;
  return res->length();
}

void Item_func_length::fix_length_and_dec()
{
  maybe_null=0;
  max_length=10;
}

double Item_func_strcmp::val()
{
  String *a=args[0]->str(&tmp_value1);
  String *b=args[1]->str(&tmp_value2);
  if (!a || !b)
  {
    null_value=1;
    return 0.0;
  }
  int val=stringcmp(*a,*b);
  return !val ? 0.0 : (val < 0 ? -1.0 : 1.0);
}

void Item_func_strcmp::fix_length_and_dec()
{
  max_length=2;
}


double Item_func_locate::val()
{
  String *a=args[0]->str(&value1);
  String *b=args[1]->str(&value2);
  uint start=0;
  if (!a || !b)
    return 0.0;
  if (arg_count == 3)
  {
    start=(uint) args[2]->val()-1;
    if (start+b->length() > a->length())
      return 0.0;
  }
  if (!b->length())				// Found empty string at start
    return (double) (start+1);
  return (double) (a->strstr(*b,start)+1) ;
}


void Item_func_locate::fix_length_and_dec()
{
  maybe_null=0;
  max_length=11;
}


String *Item_func_left::str(String *str)
{
  String *res  =args[0]->str(str);
  long length  =(long) args[1]->val();

  if ((null_value=args[0]->null_value))
    return 0;
  if (length <= 0)
    return &empty_string;
  if (res->length() > (ulong) length)
    res->length((uint) length);			// Safe even if const arg
  return res;
}


void Item_str_func::left_right_max_length()
{
  max_length=args[0]->max_length;
  if (const_item(args[1]))
  {
    int length=(int) args[1]->val();
    if (length <= 0)
      max_length=0;
    else
      set_if_smaller(max_length,(uint) length);
  }
}


void Item_func_left::fix_length_and_dec()
{
  left_right_max_length();
}


String *Item_func_right::str(String *str)
{
  String *res  =args[0]->str(str);
  long length  =(long) args[1]->val();

  if ((null_value=args[0]->null_value))
    return 0;
  if (length <= 0)
    return &empty_string;
  if (res->length() <= (uint) length)
    return res;
  tmp_value.set(*res,(res->length()- (uint) length),(uint) length);
  return &tmp_value;
}


void Item_func_right::fix_length_and_dec()
{
  left_right_max_length();
}


String *Item_func_substr::str(String *str)
{
  String *res  =args[0]->str(str);
  long start   =(long) args[1]->val();
  long length  =(long) args[2]->val();
  long tmp_length;

  if ((null_value=args[0]->null_value))
    return 0;
  if (start <= 0 || (uint) start > res->length() || length <= 0)
    return &empty_string;

  start--;
  tmp_length=(long) res->length()-start;
  length=min(length,tmp_length);

  if (!start && res->length() == (uint) length)
    return res;
  tmp_value.set(*res,(uint) start,(uint) length);
  return &tmp_value;
}


void Item_func_substr::fix_length_and_dec()
{
  max_length=args[0]->max_length;

  if (const_item(args[1]))
  {
    long start=(long) args[1]->val()-1;
    if (start < 0 || start >= (long) max_length)
      max_length=0;
    else
      max_length-= (uint) start;
  }
  if (const_item(args[2]))
  {
    long length= (long) args[2]->val();
    if (length <= 0)
      max_length=0;
    else
      set_if_smaller(max_length,(uint) length);
  }
}


String *Item_func_ltrim::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;

  char *ptr=(char*) res->ptr();
  char *end=ptr+res->length();
  while (ptr != end && isspace(*ptr)) ptr++;
  if (ptr == res->ptr())
    return res;
  tmp_value.set(*res,(uint) (ptr - res->ptr()),(uint) (end-ptr));
  return &tmp_value;
}


String *Item_func_rtrim::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;

  char *ptr=(char*) res->ptr();
  char *end=ptr+res->length();
  while (ptr != end  && isspace(end[-1])) end--;
  if (end == res->ptr()+res->length())
    return res;
  tmp_value.set(*res,0,(uint) (end-ptr));
  return &tmp_value;
}


String *Item_func_password::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;
  if (res->length() == 0)
    return &empty_string;
  make_scrambled_password(tmp_value,res->c_ptr());
  str->set(tmp_value,8);
  return str;
}


String *Item_func_database::str(String *str)
{
  str->set((const char*) current_thd->db,strlen(current_thd->db));
  return str;
}

String *Item_func_user::str(String *str)
{
  str->set((const char*) current_thd->user,strlen(current_thd->user));
  return str;
}

void Item_func_intervall::fix_length_and_dec()
{
  bool nums=1;
  for (uint i=0 ; i < arg_count ; i++)
  {
    if (args[i]->type() != Item::INT_ITEM &&
	args[i]->type() != Item::REAL_ITEM)
    {
      nums=0;
      break;
    }
  }
  if (nums && arg_count >= 8)
  {
    intervalls=(double*) sql_alloc(sizeof(double)*arg_count);
    for (uint i=0 ; i < arg_count ; i++)
      intervalls[i]=args[i]->val();
  }
  maybe_null=0; max_length=2;
}

/*
  return -1 if null value,
	  0 if lower than lowest
	  1 - arg_count if between args[n] and args[n+1]
	  arg_count+1 if higher than biggest argument
*/

double Item_func_intervall::val()
{
  double value=item->val();
  if (item->null_value)
    return -1;					// -1 if null
  if (intervalls)
  {						// Use binary search to find intervall
    uint start,end;
    start=0; end=arg_count-1;
    while (start != end)
    {
      uint mid=(start+end+1)/2;
      if (intervalls[mid] <= value)
	start=mid;
      else
	end=mid-1;
    }
    return (value < intervalls[start]) ? 0 : start+1;
  }
  if (args[0]->val() > value)
    return 0;
  for (uint i=1 ; i < arg_count ; i++)
  {
    if (args[i]->val() > value)
      return i;
  }
  return arg_count;
}


double Item_func_between::val()
{						// ANSI BETWEEN
  if (args[0]->type() == Item::STRING_RESULT)
  {
    String *value,*a,*b;
    value=args[0]->str(&value0);
    if ((null_value=args[0]->null_value))
      return 0.0;
    a=args[1]->str(&value1);
    b=args[2]->str(&value2);
    if (!args[1]->null_value && !args[2]->null_value)
      return (stringcmp(*value,*a) >= 0 && stringcmp(*value,*b) <= 0) ?
	1.0 : 0.0;
    if (args[1]->null_value && args[2]->null_value)
      null_value=1;
    else if (args[1]->null_value)
    {
      null_value= stringcmp(*value,*b) <= 0;	// not null if false range.
    }
    else
    {
      null_value= stringcmp(*value,*a) >= 0;	// not null if false range.
    }
  }
  else
  {
    double value=args[0]->val(),a,b;
    if ((null_value=args[0]->null_value))
      return 0.0;
    a=args[1]->val();
    b=args[2]->val();
    if (!args[1]->null_value && !args[2]->null_value)
      return (value >= a && value <= b) ? 1.0 : 0.0;
    if (args[1]->null_value && args[2]->null_value)
      null_value=1;
    else if (args[1]->null_value)
    {
      null_value= value <= b;			// not null if false range.
    }
    else
    {
      null_value= value >= a;
    }
  }
  return 0.0;
}


double Item_func_elt::val()
{
  String *field;
  if (!(field=item->str(&value)))
    return 0;					// -1 if null
  for (uint i=0 ; i < arg_count ; i++)
  {
    String *tmp_value=args[i]->str(&tmp);
    if (tmp_value && field->length() == tmp_value->length() &&
	!memcmp(field->ptr(),tmp_value->ptr(),tmp_value->length()))
      return i+1;
  }
  return 0;
}

double Item_func_period_add::val()
{
  ulong period=(ulong) args[0]->val();
  int months=(int) args[1]->val();

  if ((null_value=args[0]->null_value || args[1]->null_value) ||
      period == 0L)
    return 0.0;
  return convert_month_to_period((uint) ((int) convert_period_to_month(period)+
					 months));
}


double Item_func_period_diff::val()
{
  ulong period1=(ulong) args[0]->val();
  ulong period2=(ulong) args[1]->val();

  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return ((long) convert_period_to_month(period1)-
	  (long) convert_period_to_month(period2));
}


double Item_func_to_days::val()
{
  null_value=0;
  if (args[0]->result_type() == Item::STRING_RESULT)
  {
    String *str;
    char *pos,*start;
    uint length,date[3];

    if (!(str=args[0]->str(&value)))
    {
      null_value=1;
      return 0.0;
    }
    // Fix at least the following formats
    // YYMMDD, YYYYMMDD,  YY-MM-DD, YYYY-MM-DD

    for (pos=str->c_ptr(); *pos && !isdigit(*pos) ; pos++) ;
    length=(uint) ((str->ptr()+str->length())-pos);
    for (uint i=0 ; i < 3; i++)
    {
      uint tmp_value=0;
      while (!isdigit(*pos) && *pos)
	pos++;
      start=pos;
      while (isdigit(pos[0]) &&
	     ((pos-start) < 2 || ((pos-start) < 4 && i == 0 && length >= 8)))
      {
	tmp_value=tmp_value*10 + (uint) (uchar) (*pos - '0');
	pos++;
      }
      date[i]=tmp_value;
    }
    return (double) calc_daynr(date[0],date[1],date[2]);
  }
  else
  {
    ulong date=(ulong) args[0]->val();
    if (args[0]->null_value)
    {
      null_value=1;
      return 0.0;
    }
    return (double) calc_daynr(date/10000,(date/100)%100,date%100);
  }
}


double Item_func_weekday::val()
{
  ulong tmp_value=(ulong) args[0]->val();
  if ((null_value=args[0]->null_value) && !tmp_value)
    return 0.0;
  
  return (double) calc_weekday(calc_daynr((uint) (tmp_value/10000L),
					  (uint) (tmp_value/100L)%100,
					  (uint) (tmp_value%100L)));
}


String *Item_date::str(String *str)
{
  ulong value=(ulong) val();
  if (null_value)
    return (String*) 0;
  if (!value)
  {
    return &empty_string;			// zero daynr
  }
  (void) str->alloc(16);
  sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	  (int) (value/10000L),(int) (value/100)%100,(int) (value%100));
  str->length(strlen(str->ptr()));
  return str;
}


double Item_func_from_days::val()
{
  double value=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;

  uint year,month,day;
  get_date_from_daynr((long) value,&year,&month,&day);
  return (double) (year*10000L+month*100+day);
}


void Item_func_curdate::fix_length_and_dec()
{
  time_t skr;
  struct tm tm_tmp,*start_time;

  decimals=0; max_length=10;
  skr=time((time_t*) 0);
  start_time=localtime_r(&skr,&tm_tmp);
  value=(double) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		  ((uint) start_time->tm_mon+1)*100+
		  (uint) start_time->tm_mday);
}


double Item_func_curdate::val()
{
  return value;
}


void Item_func_now::fix_length_and_dec()
{
  time_t skr;
  struct tm tm_tmp,*start_time;

  decimals=0; max_length=19;
  skr=time((time_t*) 0);
  start_time=localtime_r(&skr,&tm_tmp);
  value=((double) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		   (((uint) start_time->tm_mon+1)*100+
		    (uint) start_time->tm_mday))*1000000.0+
	 (double) ((ulong) ((uint) start_time->tm_hour)*10000L+
		   (ulong) (((uint) start_time->tm_min)*100L+
			    (uint) start_time->tm_sec)));
  sprintf(buff,"%04d-%02d-%02d %02d:%02d:%02d",
	  (int) start_time->tm_year+1900,
	  (int) start_time->tm_mon+1,
	  (int) start_time->tm_mday,
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str_value.set((const char*) buff,strlen(buff));
}


double Item_func_now::val()
{
  return value;
}


String *Item_func_now::str(String *str)
{
  return &str_value;
}

double Item_func_bit_or::val()
{
  ulonglong arg1= (ulonglong) args[0]->val();
  if (args[0]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  ulonglong arg2= (ulonglong) args[1]->val();
  if (args[1]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  return (double) (arg1 | arg2);
}


double Item_func_bit_and::val()
{
  ulonglong arg1= (ulonglong) args[0]->val();
  if (args[0]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  ulonglong arg2= (ulonglong) args[1]->val();
  if (args[1]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  return (double) (arg1 & arg2);
}


static char nbits[256] = {
  0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

double Item_func_bit_count::val()
{
  ulonglong value= (ulonglong) args[0]->val();
  if (args[0]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  ulong v1=(ulong) value;
  ulong v2=(value >> 32);
  uint bits= (uint) (uchar) (nbits[(uchar) v1] +
			     nbits[(uchar) (v1 >> 8)] +
			     nbits[(uchar) (v1 >> 16)] +
			     nbits[(uchar) (v1 >> 24)] +
			     nbits[(uchar) v2] +
			     nbits[(uchar) (v2 >> 8)] +
			     nbits[(uchar) (v2 >> 16)] +
			     nbits[(uchar) (v2 >> 24)]);
  return (double) bits;
}

void
Item_func_ifnull::fix_length_and_dec()
{
  maybe_null=args[1]->maybe_null;
  max_length=max(args[0]->max_length,args[1]->max_length);
  decimals=max(args[0]->decimals,args[1]->decimals);
  cached_result_type=args[0]->result_type();
}

double
Item_func_ifnull::val()
{
  double value=args[0]->val();
  null_value=0;
  if (!args[0]->null_value)
    return value;
  value=args[1]->val();
  if ((null_value=args[1]->null_value))
    return 0.0;
  return value;
}

String *
Item_func_ifnull::str(String *str)
{
  String *res  =args[0]->str(str);
  if (!args[0]->null_value)
    return res;
  res  =args[1]->str(str);
  if ((null_value=args[1]->null_value))
    return 0;
  return res;
}

void
Item_func_if::fix_length_and_dec()
{
  maybe_null=args[1]->maybe_null || args[2]->maybe_null;
  max_length=max(args[1]->max_length,args[2]->max_length);
  decimals=max(args[0]->decimals,args[1]->decimals);
  cached_result_type = args[1]->result_type();
}


double
Item_func_if::val()
{
  double value=args[0]->val();
  Item *arg= value ? args[1] : args[2];
  value=arg->val();
  null_value=arg->null_value;
  return value;
}


String *
Item_func_if::str(String *str)
{
  double value=args[0]->val();
  Item *arg= value ? args[1] : args[2];
  String *res=arg->str(str);
  null_value=arg->null_value;
  return res;
}

/*
** COND functions
** add_or_conds && add_and_conds adds item to existing list to get easy
** parsing of and/or levels
*/

Item *add_or_conds(Item *i1,Item *i2)
{
  bool i2_is_cond= (i2->type() == Item::COND_ITEM &&
		    ((Item_cond*) i2)->functype() == Item_func::COND_OR_FUNC);
  if (i1->type() == Item::COND_ITEM &&
      ((Item_cond*) i1)->functype() == Item_func::COND_OR_FUNC)
  {
    if (i2_is_cond)
    {						// Join lists
      List_iterator<Item> li(((Item_cond*) i2)->list);
      Item *item;
      while ((item=li++))
	((Item_cond*) i1)->list.push_back(item);
      ((Item_cond*) i2)->list.empty();
      delete i2;
    }
    else
      ((Item_cond*) i1)->list.push_back(i2);	// Add to list
    return i1;
  }
  if (i2_is_cond)
  {
    ((Item_cond*) i2)->list.push_front(i1);	// Add to list
    return i2;
  }
  return new Item_cond_or(i1,i2);
}


Item *add_and_conds(Item *i1,Item *i2)
{
  bool i2_is_cond= (i2->type() == Item::COND_ITEM &&
		    ((Item_cond*) i2)->functype() == Item_func::COND_AND_FUNC);
  if (i1->type() == Item::COND_ITEM &&
      ((Item_cond*) i1)->functype() == Item_func::COND_AND_FUNC)
  {
    if (i2_is_cond)
    {						// Join lists
      List_iterator<Item> li(((Item_cond*) i2)->list);
      Item *item;
      while ((item=li++))
	((Item_cond*) i1)->list.push_back(item);
      ((Item_cond*) i2)->list.empty();
      delete i2;
    }
    else
      ((Item_cond*) i1)->list.push_back(i2);	// Add to list
    return i1;
  }
  if (i2_is_cond)
  {
    ((Item_cond*) i2)->list.push_front(i1);	// Add to list
    return i2;
  }
  return new Item_cond_and(i1,i2);
};


bool
Item_cond::fix_fields(THD *thd,tname_t *tables,const string where)
{
  List_iterator<Item> li(list);
  Item *item;
  thd->cond_count+=list.elements;
  while ((item=li++))
    if (item->fix_fields(thd,tables,where))
      return 1;
  fix_length_and_dec();
  used_table=(ulong*) sql_calloc(sizeof(ulong)*arg_count);
  return 0;
}


ulong
Item_cond::used_tables()
{						// This caches used_tables
  ulong tables=0;
  for (uint i=0 ; i < arg_count ; i++)
  {
    used_table[i]=args[i]->used_tables();
    tables|=used_table[i];
  }
  return tables;
}


String *Item_cond::str(String *str __attribute__((unused)))
{
  return &empty_string;				// Never used
}

double Item_cond_and::val()
{
  List_iterator<Item> li(list);
  Item *item;
  while ((item=li++))
    if (item->val() == 0.0)
      return 0.0;
  return 1.0;
}

double Item_cond_or::val()
{
  List_iterator<Item> li(list);
  Item *item;
  while ((item=li++))
    if (item->val() != 0.0)
      return 1.0;
  return 0.0;
}

double Item_func_isnull::val()
{
  (void) args[0]->val();
  return (args[0]->null_value) ? 1.0 : 0.0;
}

double Item_func_isnotnull::val()
{
  (void) args[0]->val();
  return !(args[0]->null_value) ? 1.0 : 0.0;
}

static
int wild_case_compare(const char *str,const char *wildstr)
{
  reg3 int flag;
  DBUG_ENTER("wild_compare");

  while (*wildstr)
  {
    while (*wildstr && *wildstr != wild_many && *wildstr != wild_one)
    {
      if (*wildstr == wild_prefix && wildstr[1])
	wildstr++;
      if (toupper(*wildstr++) != toupper(*str++)) DBUG_RETURN(1);
    }
    if (! *wildstr ) DBUG_RETURN (*str != 0);
    if (*wildstr++ == wild_one)
    {
      if (! *str++) DBUG_RETURN (1);	/* One char; skipp */
    }
    else
    {						/* Found '*' */
      if (!*wildstr) DBUG_RETURN(0);		/* '*' as last char: OK */
      flag=(*wildstr != wild_many && *wildstr != wild_one);
      do
      {
	if (flag)
	{
	  char cmp;
	  if ((cmp= *wildstr) == wild_prefix && wildstr[1])
	    cmp=wildstr[1];
	  cmp=toupper(cmp);
	  while (*str && toupper(*str) != cmp)
	    str++;
	  if (!*str) DBUG_RETURN (1);
	}
	if (wild_case_compare(str,wildstr) == 0) DBUG_RETURN (0);
      } while (*str++);
      DBUG_RETURN(1);
    }
  }
  DBUG_RETURN (*str != '\0');
}


double Item_func_like::val()
{
  String *res,*res2;
  res=args[0]->str(&tmp_value1);
  if (args[0]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  res2=args[1]->str(&tmp_value2);
  if (args[1]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  null_value=0;
  return wild_case_compare(res->c_ptr(),res2->c_ptr()) ? 0.0 : 1.0;
}

/* We can optimize a where if first character isn't a wildcard */

bool Item_func_like::select_optimize()
{
  if (args[1]->type() == Item::STRING_ITEM)
  {
    if (((Item_string *) args[1])->str_value[0] != wild_many)
    {
      if ((args[0]->result_type() != STRING_RESULT) ||
	  ((Item_string *) args[1])->str_value[0] != wild_one)
	return 1;
    }
  }
  return 0;
}


double Item_func_nlike::val()
{
  String *res,*res2;
  
  if (!(res=args[0]->str(&tmp_value1)))
  {
    null_value=1;
    return 0.0;
  }
  if (!(res2=args[1]->str(&tmp_value1)))
  {
    null_value=1;
    return 0.0;
  }
  null_value=0;
  return !wild_case_compare(res->c_ptr(),res2->c_ptr()) ? 0.0 : 1.0;
}


/*
** Change a number to format '3,333,333,333.000'
** This should be 'internationalized' sometimes.
*/

Item_func_format::Item_func_format(Item *org,int dec) :Item_str_func(org)
{
  decimals=(uint) set_zone(dec,0,30);
}


String *Item_func_format::str(String *str)
{
  double nr	=args[0]->val();
  uint diff,length,dec;
  if ((null_value=args[0]->null_value))
    return 0;
  dec= decimals ? decimals+1 : 0;
  str->set(nr,decimals);
  length=str->length()+(diff=(str->length()- dec-1)/3);
  if (diff)
  {
    char *tmp,*pos;
    str=copy_if_not_alloced(tmp_str,*str,length);
    str->length(length);
    tmp=(char*) str->ptr()+length - dec-1;
    for (pos=(char*) str->ptr()+length ; pos != tmp; pos--)
      pos[0]=pos[- (int) diff];
    while (diff)
    {
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=',';
      pos--;
      diff--;
    }
  }
  return str;
}


#ifdef USE_REGEX

bool
Item_func_regex::fix_fields(THD *thd,tname_t *tables,const string where)
{
  if (args[0]->fix_fields(thd,tables,where))
    return 1;
  maybe_null=args[0]->maybe_null;
  max_length=1; decimals=0;
  
  String &res=((Item_string*) args[1])->str_value;
  int error;
  if ((error=regcomp(&preg,res.c_ptr(),REG_EXTENDED | REG_NOSUB)))
  {
    char buff[100];
    (void) regerror(error,&preg,buff,sizeof(buff));
    my_printf_error("Got error '%s' from regexp",MYF(0),buff);
    return 1;
  }
  regex_compiled=1;
  return 0;
}


double Item_func_regex::val()
{
  String s,*res;
  res=args[0]->str(&s);
  if (args[0]->null_value)
  {
    null_value=1;
    return 0.0;
  }
  return regexec(&preg,res->c_ptr(),0,(regmatch_t*) 0,0) ? 0.0 : 1.0;
}


Item_func_regex::~Item_func_regex()
{
  if (regex_compiled)
  {
    regfree(&preg);
    regex_compiled=0;
  }
}

#endif /* USE_REGEX */
