/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

#pragma implementation				// gcc: implement item.h

#include "mysql_priv.h"

#define MAX_BLOB_WIDTH	8192			// Define this for now

/*****************************************************************************
** Item functions
*****************************************************************************/

void Item::set_name(char *str,uint length)
{
  if (!length)
    name=str;					// Can use old lex
  else
  {
    if ((name=(char*) sql_alloc(length+1)))
    {
      memcpy(name,str,length); name[length]=0;
    }
  }
}

bool Item::eq(const Item *item) const		// Only doing this on conds
{
  return type() == item->type() && name && item->name &&
    !my_strcasecmp(name,item->name);
}


Item_field::Item_field(FIELD *f) :Item_ident(f->table_name,f->field_name)
{
  set_field(f);
}


void Item_field::set_field(FIELD *field_par)
{
  uint flag=field_par->pack_flag;
  this->field=result_field=field_par;		// for easy coding with fields
  maybe_null=(bool) f_maybe_null(flag);
  max_length=f_is_blob(flag) ? MAX_BLOB_WIDTH : field_par->length;
  decimals=f_is_num(flag) ? f_decimals(flag) : 0;
  table_name=field_par->table_name;
  field_name=field_par->field_name;
}

char *Item_ident::full_name()
{
  if (!table_name)
    return field_name;
  char *tmp=(char*) sql_alloc(strlen(table_name)+strlen(field_name)+2);
  strxmov(tmp,table_name,".",field_name,NullS);
  return tmp;
}


String *field_to_string(String &str,FIELD *field)
{
  if (f_is_blob(field->pack_flag))
  {
    gptr pos=field->str+(field->packlength-sizeof(char*)),blob_pos;
    memcpy((gptr) &blob_pos,pos,sizeof(char*));
    str.set((const char*) blob_pos,(uint) valfield_l(field));
  }
  else if (f_is_packed(field->pack_flag))
  {
    if (!str.alloc(field->length))
      str.length(r_unpackf_stripp(field,(char*) str.ptr()));
  }
  else
  {
    char *start=field->str, *ptr=start+field->length;
    while (ptr > start && ptr[-1] == ' ')
      ptr--;
    str.set((const char*) start,(uint) (ptr - start));
  }
  return &str;
}


/* ARGSUSED */
String *Item_field::str(String *str __attribute__((unused)))
{
  if ((null_value=test_safe_null(field)))
    return 0;
  return field_to_string(str_value,field);
}

double Item_field::val()
{
  if ((null_value=test_safe_null(field)))
    return 0.0;
  return valfield(field);
}

String *Item_field::str_result(String *str __attribute__((unused)))
{
  if ((null_value=test_safe_null(result_field)))
    return 0;
  return field_to_string(str_value,result_field);
}

double Item_field::val_result()
{
  if ((null_value=test_safe_null(result_field)))
    return 0.0;
  return valfield(result_field);
}


bool Item_field::eq(const Item *item) const
{
  return item->type() == type() && ((Item_field*) item)->field == field;
}

FIELD *Item_field::tmp_table_field()
{
  return result_field;
}

ulong Item_field::used_tables()
{
  return (ulong) 1 << field->table->tablenr;
}


String *Item_int::str(String *str)
{
  str->set(value);
  return str;
}


String *Item_real::str(String *str)
{
  str->set(value,decimals);
  return str;
}


bool Item_null::eq(const Item *item) const { return item->type() == type(); }
double Item_null::val() { null_value=1; return 0.0; }
/* ARGSUSED */
String *Item_null::str(String *str __attribute__((unused)))
{ null_value=1; return 0;}


void Item_copy_string::copy()
{
  String *res=item->str(&str_value);
  if (res != &str_value)
    str_value.copy(*res);
  null_value=item->null_value;
}

double Item_copy_string::val()
{
  return 0.0;					// Never done
}

/* ARGSUSED */
String *Item_copy_string::str(String *str __attribute__((unused)))
{
  return &str_value;
}


Item_avg_field::Item_avg_field(Item_sum_avg *item)
{
  name=item->name;
  decimals=item->decimals;
  max_length=item->max_length;
  field=item->result_field;
  maybe_null=1;
}

double Item_avg_field::val()
{
  double nr;
  ulong count;
  doubleget(nr,field->str);
  ulongget(count,(field->str+sizeof(double)));

  if (!count)
  {
    null_value=1;
    return 0.0;
  }
  return nr/(double) count;
}

String *Item_avg_field::str(String *str)
{
  double nr=val();
  if (null_value)
    return 0;
  str->set(nr,decimals);
  return str;
}


/*
** Functions to convert item to field (for send_fields)
*/

/* ARGSUSED */
bool Item::fix_fields(THD *thd __attribute__((unused)),
		      struct tlist_s *list __attribute__((unused)),
		      const string where __attribute__((unused)))
{
  return 0;
}

bool Item_field::fix_fields(THD *thd,tname_t *tables,const string where)
{
  if (!field)
  {
    FIELD *tmp;
    if (!(tmp=find_field_in_tables(thd,this,tables,where)))
      return 1;
    set_field(tmp);
  }
  return 0;
}

FIELD *Item::init_make_field(FIELD *tmp_field,enum enum_field_types type)
{
  tmp_field->table_name="";
  tmp_field->field_name=name;
  tmp_field->flags=maybe_null ? 0 : NOT_NULL_FLAG;
  tmp_field->type=type;
  tmp_field->length=max_length;
  tmp_field->decimals=decimals;
  return tmp_field;
}

/* ARGSUSED */
FIELD *Item_field::make_field(FIELD *tmp_field __attribute__((unused)))
{
  return field;
}

FIELD *Item_int::make_field(FIELD *tmp_field)
{
  return init_make_field(tmp_field,FIELD_TYPE_LONG);
}

FIELD *Item_real::make_field(FIELD *tmp_field)
{
  return init_make_field(tmp_field,FIELD_TYPE_DOUBLE);
}

FIELD *Item_string::make_field(FIELD *tmp_field)
{
  return init_make_field(tmp_field,FIELD_TYPE_STRING);
}

FIELD *Item_null::make_field(FIELD *tmp_field)
{
  (void) init_make_field(tmp_field,FIELD_TYPE_NULL);
  tmp_field->length=4;
  return tmp_field;
}

FIELD *Item_func::make_field(FIELD *tmp_field)
{
  return init_make_field(tmp_field, (result_type() == Item::STRING_RESULT) ?
			 FIELD_TYPE_STRING : FIELD_TYPE_DOUBLE);
}

FIELD *Item_avg_field::make_field(FIELD *tmp_field)
{
  return init_make_field(tmp_field,FIELD_TYPE_DOUBLE);
}

/*
** Set a field:s value from a item
*/

static bool
set_field_to_null(FIELD *field)
{
  if (!f_maybe_null(field->pack_flag))
  {
    if (field == field->table->form.next_number_field)
    {
      packnrf(0.0,field);			// Set 0.0
      return 0;					// field is set in handler.cc
    }
    if (field->type == FIELD_TYPE_TIMESTAMP)
    {
      if (field->str != field->table->form.record[0] + field->table->form.time_stamp-1)
      {						// Not set automaticly
	long skr= (long) time((time_t*) 0);
	longstore(field->str,skr);
      }
      return 0;					// Ok to set time to null
    }
    f_fieldinit(field->str,field->pack_flag,field->length);
    if (current_thd->count_cuted_fields)
    {
      current_thd->cuted_fields++;		// Increment error counter
      return 0;
    }
    my_error(ER_BAD_NULL_ERROR,MYF(0),field->field_name);
    return 1;
  }
  mark_as_null(field);
  f_fieldinit(field->str,field->pack_flag,field->length);
  return 0;
}


bool Item_field::save_in_field(FIELD *to)
{
  if (test_safe_null(field))
    return set_field_to_null(to);
  else
  {
    if (to->null_pos)
      mark_as_not_null(to);
    f_konv(to,field);
  }
  return 0;
}


bool Item_null::save_in_field(FIELD *field)
{
  return set_field_to_null(field);
}


bool Item::save_in_field(FIELD *field)
{
  if (result_type() == Item::STRING_RESULT || f_is_blob(field->pack_flag))
  {
    String *result;
    result=str(&str_value);
    if (null_value)
      return set_field_to_null(field);
    if (field->null_pos)
      mark_as_not_null(field);
    if (!f_is_blob(field->pack_flag))
      store_string(*result,field);
    else
      store_ptr_in_blob(field,result->ptr(),result->length());
  }
  else
  {
    double nr=val();
    if (null_value)
      return set_field_to_null(field);
    if (field->null_pos)
      mark_as_not_null(field);
    packnrf(nr,field);
  }
  return 0;
}

/*
** pack data in buffer for sending
*/

char *Item::send(char *pos)
{
  char buff[MAX_FIELD_WIDTH];
  String s(buff,sizeof(buff)),*res;
  res=str(&s);
  if (!res)
    return net_store_length(pos,NULL_LENGTH);
  return net_store_data(pos,res->ptr(),res->length());
}

char *Item_null::send(char *pos)
{
  return net_store_length(pos,NULL_LENGTH);
}

char *Item_field::send(char *pos)
{
  if (test_safe_null(this->result_field))
    return net_store_length(pos,NULL_LENGTH);
  else if (f_is_blob(result_field->pack_flag))
  {
    long blob_length=valfield_l(result_field);
    byte *blob_ptr=get_blob_ptr(result_field);
    return net_store_data(pos,blob_ptr,(uint) blob_length);
  }
  else
  {
    char buff[MAX_FIELD_WIDTH];
    return net_store_data(pos,buff,r_unpackf_stripp(result_field,buff));
  }
}

/*
  This is used for HAVING clause
  Find field in select list having the same name
 */

bool Item_ref::fix_fields(THD *thd,tname_t *tables,const string where)
{
  if (!(ref=find_item_in_list(this,thd->item_list,where)))
    return 1;
  max_length= (*ref)->max_length;
  maybe_null= (*ref)->maybe_null;
  decimals=   (*ref)->decimals;
  return 0;
}


/*****************************************************************************
** Instansiate templates
*****************************************************************************/

#ifdef __GNUC__
template class List<Item>;
template class List_iterator<Item>;
#endif
