/*
**	mysqldump.c  - Dump a tables contents and format to an ASCII file
**
**
** This software is provided "as is" without any expressed or implied warranty.
**
** The author's original notes follow :-
**
**		******************************************************
**		*						     *
**		* AUTHOR: Igor Romanenko (igor@frog.kiev.ua)	     *
*		* DATE:   December 3, 1994			     *
**		* WARRANTY: None, expressed, impressed, implied      *
**		*	    or other				     *
**		* STATUS: Public domain				     *
**		* Adapted and optimized for mysql by
**		* Michael Widenius				     *
**		*						     *
**		******************************************************
*/

#define DUMP_VERSION "2.8"

#include <global.h>
#include <my_sys.h>
#include <m_string.h>
#include "mysql.h"
#include "version.h"
#include <getopt.h>

/* Exit codes */

#define EX_USAGE 1
#define EX_MYSQLERR 2
#define EX_CONSCHECK 3

/* index into 'show fields from table' */

#define SHOW_FIELDNAME	0
#define SHOW_TYPE	1
#define SHOW_NULL	2
#define SHOW_DEFAULT	4
#define SHOW_EXTRA	5

uint	verbose = 0, tFlag = 0, cFlag = 0,dFlag = 0, quick=0;
MYSQL	mysql,*sock=0;
char	insert_pat[5 * 1024],*password,*current_user=0,*current_host,
  	*current_db,*current_table;

static struct option long_options[] =
{
  {"debug",		optional_argument,	0, '#'},
  {"help",		no_argument,		0, '?'},
  {"complete-insert",	no_argument,		0, 'c'},
  {"host",		required_argument,	0, 'h'},
  {"no-data",		no_argument,		0, 'd'},
  {"no-create-info",	no_argument,		0, 't'},
  {"password",		required_argument, 	0, 'p'},
  {"port",		required_argument,	0, 'P'},
  {"quick",		no_argument,	   	0, 'q'},
  {"socket",		required_argument,	0, 'S'},
#ifdef SAFE_USER
  {"user",		required_argument,	0, 'u'},
#endif
  {"verbose",		no_argument,		0, 'v'},
  {"version",		no_argument,		0, 'V'},
  {0, 0, 0, 0}
};


static void print_version(void)
{
  printf("%s  Ver %s Distrib %s, for %s (%s)\n",my_progname,DUMP_VERSION,
	 SERVER_VERSION,SYSTEM_TYPE,MACHINE_TYPE);
}

static void usage(void)
{
  print_version();
  puts("By Igor Romanenko & Monty. This software is in public Domain");
  puts("This software comes with ABSOLUTELY NO WARRANTY\n");
  puts("Dumping definition and data mysql database or table");
  printf("Usage: %s [OPTIONS] [database [table [field]]]\n",my_progname);
  printf("\n\
  -#, --debug=...       Output debug log. Often this is 'd:t:o,filename`\n\
  -?, --help		Display this help and exit\n\
  -c, --compleat-insert Use complete insert statements\n\
  -h, --host=...	Connect to host\n\
  -d, --no-data		No row information.\n\
  -t, --no-create-info	Don't write table creation info.\n\
  -p, --password=...	Password to use when connecting\n\
  -P, --port=...	Port number to use for connection\n\
  -q, --quick		Don't buffer query, dump directly to stdout\n\
  -S, --socket=...	Socket file to use for connection\n");
#ifdef SAFE_USER
  printf("\
  -u, --user=#		User for login if not current user\n");
#endif
  printf("\
  -v, --verbose		Print info about the various stages\n\
  -V, --version		Output version information and exit\n"
  );
}


static int get_options(int *argc,char ***argv)
{
  int c,option_index;

  while ((c=getopt_long(*argc,*argv,"#::?cdh:p:qtu:P:S:vV?I",long_options,
			&option_index)) != EOF)
  {
    switch(c) {
    case 'h':
      current_host=my_strdup(optarg,MYF(MY_WME));
      break;
#ifdef SAFE_USER
    case 'u':
      current_user=optarg;
      break;
#endif
    case 'p':
      password=my_strdup(optarg,MYF(MY_FAE));
      while (*optarg) *optarg++= 'x';		/* Destroy argument */
      break;
    case 'P':
      mysql_port= (unsigned int) atoi(optarg);
      break;
    case 'S':
      mysql_unix_port= optarg;
      break;
    case '#':
      DBUG_PUSH(optarg ? optarg : "d:t:o");
      break;
    case 'c': cFlag++; break;
    case 'd': dFlag++; break;
    case 'q': quick=1; break;
    case 't': tFlag++;	break;
    case 'v': verbose++; break;
    case 'V': print_version(); exit(0);
    default:
      fprintf(stderr,"Illegal option character '%c'\n",opterr);
      /* Fall throught */
    case 'I':
    case '?':
      usage();
      exit(0);
    }
  }
  (*argc)-=optind;
  (*argv)+=optind;
  if (*argc != 1 && *argc != 2)
  {
    usage();
    return 1;
  }
  current_db= *((*argv)++);
  if (*argc == 2)
    current_table=**argv;
  return(0);
}


/*
** DBerror -- prints mysql error message and exits the program.
*/

static void DBerror(MYSQL *mysql)
{
  my_printf_error("mysql error: %s", MYF(0),mysql_error(mysql));
  mysql_close(mysql);
  exit(EX_MYSQLERR);
}


/*
** dbConnect -- connects to the host and selects DB.
**	      Also checks whether the tablename is a valid table name.
*/

void dbConnect(char *host, char *database,char *user,char *passwd)
{
  if (verbose)
  {
    fprintf(stderr, "Connecting to %s...\n", host ? host : "localhost");
  }
  if (!(sock = mysql_connect(&mysql,host,user,passwd)))
    DBerror(&mysql);
  if ( verbose )
    fprintf(stderr, "Selecting data base %s...\n", database);
  if (mysql_select_db(sock, database) == -1)
    DBerror(&mysql);
}



/*
** dbDisconnect -- disconnects from the host.
*/

void dbDisconnect(char *host)
{
  if (verbose)
    fprintf(stderr, "Disconnecting from %s...\n", host ? host : "localhost");
  mysql_close(sock);
}

static void unescape(char *pos,uint length)
{
  char *end=pos+length;

  putchar('\'');
  for ( ; pos != end ; pos++)
  {
    switch (*pos) {
    case 0:
      putchar('\\');
      putchar('0');
      break;
    case '\n':
      putchar('\\');
      putchar('n');
      break;
    case '\r':
      putchar('\\');
      putchar('r');
      break;
    case '\\':
      putchar('\\');
      putchar('\\');
      break;
    default:
      putchar(*pos);
      break;
    }
  }
  putchar('\'');
}

/*
** getStructure -- retrievs database structure, prints out corresponding
**		   CREATE statement and fills out insert_pat.
*/

uint getTableStructure(char *table)
{
  MYSQL_RES	*tableRes;
  MYSQL_ROW	row;
  uint		init=0, numFields;
  char		*strpos;

  if (verbose)
    fprintf(stderr, "Retrieving table structure for table %s...\n", table);

  sprintf(insert_pat,"show fields from %s",table);
  if (mysql_query(sock,insert_pat) || !(tableRes=mysql_store_result(sock)))
  {
    fprintf(stderr, "mysql error: Can't get info about table: '%s' (%s)\n",
	    table, mysql_error(sock));
    exit(EX_MYSQLERR);
  }

  if (!tFlag)
  {
    printf("\n#\n# Table structure for table '%s'\n#\n",table);
    printf("CREATE TABLE %s (\n", table);
  }
  if (cFlag)
    sprintf(insert_pat, "INSERT INTO %s (", table);
  else
    sprintf(insert_pat, "INSERT INTO %s VALUES (", table);

  strpos=strend(insert_pat);
  while ((row=mysql_fetch_row(tableRes)))
  {
    uint *lengths=mysql_fetch_lengths(tableRes);
    if (init)
    {
      if (!tFlag)
	fputs(",\n",stdout);
      if (cFlag)
	strpos=strmov(strpos,", ");
    }
    init=1;
    if (cFlag)
      strpos=strmov(strpos,row[SHOW_FIELDNAME]);
    if (!tFlag)
    {
      printf("  %s %s", row[SHOW_FIELDNAME],row[SHOW_TYPE]);
      if (!row[SHOW_NULL][0])
	fputs(" NOT NULL",stdout);
      if (row[SHOW_DEFAULT][0])
      {
	fputs(" DEFAULT ",stdout);
	unescape(row[SHOW_DEFAULT],lengths[SHOW_DEFAULT]);
      }
      if (row[SHOW_EXTRA][0])
	printf(" %s",row[SHOW_EXTRA]);
    }
  }
  numFields = mysql_num_rows(tableRes);
  mysql_free_result(tableRes);
  if (!tFlag)
  {
    char buff[20+FN_REFLEN];
    uint keynr,primary_key;
    sprintf(buff,"show keys from %s",table);
    if (mysql_query(sock, buff))
    {
      fprintf(stderr, "mysql error: Can't get keys for table %s (%s)\n", table,
	      mysql_error(sock));
      exit(EX_MYSQLERR);
    }

    tableRes=mysql_store_result(sock);
    /* Find first which key is primary key */
    keynr=0;
    primary_key=INT_MAX;
    while ((row=mysql_fetch_row(tableRes)))
    {
      if (atoi(row[3]) == 1)
      {
	keynr++;
	if (atoi(row[1]) == 0 && primary_key == INT_MAX)
	  primary_key=keynr;
	if (!strcmp(row[2],"PRIMARY"))
	{
	  primary_key=keynr;	  
	  break;
	}
      }
    }
    mysql_data_seek(tableRes,0);
    keynr=0;
    while ((row=mysql_fetch_row(tableRes)))
    {
      if (atoi(row[3]) == 1)
      {
	if (keynr++)
	  putchar(')');
	if (atoi(row[1]))			/* Test if dupplicate key */
	  printf(",\n  KEY %s (",row[2]);	/* Dupplicate allowed */
	else if (keynr == primary_key)
	  fputs(",\n  PRIMARY KEY (",stdout);	/* First UNIQUE is primary */
	else
	  printf(",\n  UNIQUE %s (",row[2]);	/* UNIQUE key */
      }
      else
	putchar(',');
      fputs(row[4],stdout);
      if (row[7])
	printf("(%s)",row[7]);			/* Sub key */
    }
    if (keynr)
      putchar(')');
    fputs("\n);\n\n",stdout);
  }
  if (cFlag)
    strpos=strmov(strpos,") VALUES (");
  return(numFields);
}




/*
** dumpTable saves database contents as a series of INSERT statements.
*/

void dumpTable(uint numFields, char *table)
{
  char query[FN_REFLEN*2+48];
  MYSQL_RES	*res;
  MYSQL_FIELD	*field;
  MYSQL_ROW		row;
  uint		i;
  int		init = 1;

  if (verbose)
    fprintf(stderr, "Sending SELECT query...\n");
  printf("\n#\n# Dumping data for table '%s'\n#\n\n",table);
  sprintf(query, "SELECT * FROM %s", table);
  if (mysql_query(sock, query) == -1)
    DBerror(sock);
  if (quick)
    res=mysql_use_result(sock);
  else
    res=mysql_store_result(sock);
  if (!res)
    DBerror(sock);
  if (verbose)
    fprintf(stderr, "Retrieving rows...\n");
  if (mysql_num_fields(res) != numFields)
  {
    fprintf(stderr,"Error in field count!  Aborting.\n\n");
    exit(EX_CONSCHECK);
  }

  while ((row=mysql_fetch_row(res)))
  {
    uint *lengths=mysql_fetch_lengths(res);

    fputs(insert_pat,stdout);
    init = 1;
    mysql_field_seek(res,0);
    for (i = 0; i < mysql_num_fields(res); i++)
    {
      if (!(field = mysql_fetch_field(res)))
      {
	fprintf(stderr,"Not enough fields! Aborting\n");
	exit(EX_CONSCHECK);
      }
      if (!init )
	printf(",");
      else
	init=0;
      if (row[i])
      {
	if (field->type == FIELD_TYPE_STRING ||
	    field->type == FIELD_TYPE_VAR_STRING ||
	    field->type == FIELD_TYPE_TINY_BLOB ||
	    field->type == FIELD_TYPE_MEDIUM_BLOB ||
	    field->type == FIELD_TYPE_LONG_BLOB ||
	    field->type == FIELD_TYPE_BLOB)
	{
	  unescape(row[i],lengths[i]);
	}
	else
	  fputs(row[i],stdout);
      }
      else
      {
	fputs("NULL",stdout);
      }
    }
    fputs(");\n",stdout);
  }
  if (!mysql_eof(res))
  {
    fprintf(stderr,"Unexpected error: %s",mysql_error(sock));
    exit(EX_CONSCHECK);
  }
  mysql_free_result(res);
}



char *getTableName(void)
{
  static MYSQL_RES *res = NULL;
  MYSQL_ROW		row;

  if (!res)
  {
    res = mysql_list_tables(sock,NullS);
    if (!res)
      return(NULL);
  }
  row = mysql_fetch_row(res);
  if (row)
  {
    return((char*) row[0]);
  }
  else
  {
    mysql_free_result(res);
    return(NULL);
  }
}




int main(int argc, char **argv)
{
  uint	numRows;
  DBUG_ENTER("main");
  DBUG_PROCESS("main");
  MY_INIT(argv[0]);

  /*
  ** Check out the args
  */
  if (get_options(&argc,&argv))
    exit(EX_USAGE);
  puts("# mysql Dump\n#\n");
  printf("# Host: %s    Database: %s\n",
	 current_host ? current_host : "localhost", current_db);
  puts("#--------------------------------------------------------\n");
  dbConnect(current_host,current_db,current_user,password);
  if (current_table)
  {
    numRows = getTableStructure(current_table);
    if (!dFlag)
      dumpTable(numRows,current_table);
  }
  else
  {
    char *table;
    while((table = getTableName()))
    {
      numRows = getTableStructure(table);
      if (!dFlag)
	dumpTable(numRows,table);
    }
  }
  dbDisconnect(current_host);
  puts("");
  my_free(password,MYF(MY_ALLOW_ZERO_PTR));
  my_end(0);
  exit(0);
}
