/* Copyright Abandoned 1996 TCX DataKonsult AB & Monty Program KB & Detron HB
   This file is public domain and comes with NO WARRANTY of any kind */

/* mysql command tool
 * Written by Michael Widenius
 * Commands compatible with mSQL by David J. Hughes
 **/

#include <global.h>
#include <my_sys.h>
#include <m_string.h>
#include "mysql.h"
#include "errmsg.h"
#include <my_dir.h>
#include <getopt.h>
#include "version.h"

gptr sql_alloc(unsigned size);			// Don't use mysqld alloc for theese
void sql_element_free(void *ptr);
#include "sql_string.h"

extern "C" {
#ifdef HAVE_ASM_TERMBITS_H
#include <asm/termbits.h>
#endif
#undef VOID
#include <curses.h>
#undef SYSV					// hack to avoid syntax error
#ifdef HAVE_TERM_H
#include <term.h>
#endif
#undef bcmp					// Fix problem with new readline
#undef bzero
#include <readline/readline.h>
  int vidattr(long unsigned int attrs);		// Missing in sun curses
}

#if defined(__linux__) || !defined(HAVE_VIDATTR)
#undef vidattr
#define vidattr(A) {}				// Can't get this to work
#endif

enum enum_info_type { INFO_INFO,INFO_ERROR,INFO_RESULT};
typedef enum enum_info_type INFO_TYPE;

static MYSQL mysql;			/* The connection */
static bool info_flag=0,silent=0,batch=0,ignore_errors=1,wait_flag=0,quick=0,
	    connected=0,opt_raw_data;
static uint verbose=0;
static char *current_host,*current_db,*current_user=0,*password=0;

/* The names of functions that actually do the manipulation. */
static int get_options(int argc,char **argv);
static int com_quit(String *str,char*), com_go(String *str,char*),
	   com_edit(String *str,char*), com_print(String *str,char*),
	   com_help(String *str,char*), com_clear(String *str,char*),
	   com_connect(String *str,char*);
static int sql_connect(char *host,char *database,char *user,char *password,
		       int silent);
static int put_info(const char *str,INFO_TYPE info);

/* A structure which contains information on the commands this program
   can understand. */

typedef struct {
  char *name;			/* User printable name of the function. */
  char cmd_char;		/* msql command character */
  int (*func)(String *str,char *); /* Function to call to do the job. */
  bool takes_params;		/* Max parameters for command */
  char *doc;			/* Documentation for this function.  */
} COMMAND;

static COMMAND commands[] = {
  { "help", 'h',com_help, 0,"Display this text" },
  { "?",    'h',com_help, 0,"Synonym for `help'" },
  { "clear",'c',com_clear,0,"Clear command"},
  { "quit", 'q',com_quit, 0,"Quit mysql" },
  { "print",'p',com_print,0,"print current command" },
  { "edit", 'e',com_edit, 0,"Edit command with $EDITOR"},
  { "go",   'g',com_go,   0,"Send command to mysql server" },
  { "connect",'r',com_connect,1,"Reconnect if mysql connection was down. Optional args are db and host" },

  { "create table",0,0,0},	/* Get bash expansion for some commmands */
  { "select",0,0,0},
  { "drop",0,0,0},
  { "insert",0,0,0},
  { "update",0,0,0},
  { "delete",0,0,0},
  { "show databases",0,0,0},
  { "show fields",0,0,0},
  { "show tables",0,0,0},
  { (char *)NULL, 0,0,0},
};


#ifdef NOT_NEEDED				// Got warning from SUNPRO_C
extern char *readline(char*);
extern char **completion_matches(char *, char *(*command_generator)(char *text,
								    int state));
#endif
extern "C" void add_history(char *command); /* From readline directory */
extern char *batch_readline(void);
extern void batch_readline_end(void);
extern void batch_readline_command(string str);

static void initialize_readline (char *name);
static COMMAND *find_command (char *name,char cmd_name);
static bool add_line(String &buffer,char *line,char *in_string);
static void remove_cntrl(String &buffer);
static void print_table_data(MYSQL_RES *result);
static void print_tab_data(MYSQL_RES *result);
static uint start_timer(void);
static void end_timer(uint start_time,char *buff);


int main(int argc,char *argv[])
{
  String buffer(512);
  char	*line;
  char	in_string=0;
  COMMAND *com;
  DBUG_ENTER("main");
  DBUG_PROCESS(argv[0]);

  MY_INIT(argv[0]);
  initialize_readline(my_progname);

  if (!isatty(0) || !isatty(1))
  {
    batch=silent=1;
    ignore_errors=0;
  }
  if (get_options(argc,(char **) argv))
    exit(1);

  if (sql_connect(current_host,current_db,current_user,password,silent))
    exit(1);
  /*
  **  Run in interactive mode like the ingres/postgres monitor
  */

  put_info("Welcome to the mysql monitor.  Commands ends with ; or \\g.",
	   INFO_INFO);
  put_info("Type 'help' for help.\n",INFO_INFO);

  for (;;)
  {
    if (batch)
      line=batch_readline();
    else
      line=readline(buffer.is_empty() ? "mysql> " : in_string ?
		    "    '> " : "    -> ");
    if (!line)
      break;					// End of file
    if (!in_string && (line[0] == '#' || 
		       (line[0] == '-' && line[1] == '-') ||
		       line[0] == 0))
      continue;					// Skipp comment lines

    /* Check if line is a mysql command line */
    /* (We want to allow help, print and clear anywhere at line start */
    if ((com=find_command(line,0)))
    {
      if ((*com->func)(&buffer,line) > 0)
	break;
      if (buffer.is_empty())			// If buffer was emptied
	in_string=0;
      if (!batch)
	add_history(line);
      continue;
    }
    if (line[0] && add_line(buffer,line,&in_string))
      break;
  }
  /* if in batch mode, send last query even if it doesn't end with \g or go */

  if (batch)
  {
    remove_cntrl(buffer);
    if (!buffer.is_empty())
      (void) com_go(&buffer,line);
  }

  if (connected)
    mysql_close(&mysql);
  batch_readline_end();
  put_info("\nBye",INFO_RESULT);
  my_end(info_flag ? MY_CHECK_ERROR | MY_GIVE_INFO : 0);
  if (password)
    my_free(password,MYF(0));
  exit(0);
#ifndef _lint
  DBUG_RETURN(0);				/* Keem compiler happy */
#endif
}

static struct option long_options[] =
{
  {"batch",	no_argument,	   0, 'B'},
  {"debug",	optional_argument, 0, '#'},
  {"debug-info",no_argument,	   0, 'T'},
  {"force",	no_argument, 	   0, 'f'},
  {"help",	no_argument,	   0, '?'},
  {"host",	required_argument, 0, 'h'},
  {"password",	required_argument, 0, 'p'},
  {"port",	required_argument, 0, 'P'},
  {"quick",	no_argument,	   0, 'q'},
  {"raw",	no_argument,	   0, 'r'},
  {"silent",	silent,		   0, 's'},
  {"socket",	required_argument, 0, 'S'},
#ifdef SAFE_USER
  {"user",	required_argument, 0, 'u'},
#endif
  {"verbose",	no_argument,	   0, 'v'},
  {"version",	no_argument,	   0, 'V'},
  {"wait",	no_argument,	   0, 'w'},
  {0, 0, 0, 0}
};


static void usage(int version)
{
  printf("%s  Ver 5.4 Distrib %s, for %s (%s)\n",
	 my_progname, SERVER_VERSION, SYSTEM_TYPE, MACHINE_TYPE);
  if (version)
    return;
  puts("By TCX Datakonsult AB, by Monty");
  puts("This software comes with ABSOLUTELY NO WARRANTY.\n");
  printf("Usage: %s [OPTIONS] database\n", my_progname);
  printf("\n\
  -B, --batch		print results with a tab as separator, each row on\
			a new line\n\
  -#, --debug=...       output debug log. Often this is 'd:t:o,filename`\n\
  -T, --debug-info	print some debug info at exit\n\
  -e, --exec=...	execute command and quit.(--batch is implicit)\n\
  -f, --force		continue even if we get an sql error.\n\
  -?, --help		display this help and exit\n\
  -h, --host=...	connect to host\n\
  -p, --password=...	password to use when connecting to server\n\
  -P  --port=...	Port number to use for connection\n\
  -q, --quick		don't cache result, print it row by row. This may\n\
			slow down the server if the output is suspended\n\
  -r, --raw		write fields without conversion. Used with --batch\n\
  -s, --silent		be more silent.\n\
  -S  --socket=...	Socket file to use for connection\n");

#ifdef SAFE_USER
  printf("\
  -u, --user=#		user for login if not current user\n");
#endif
  printf("\
  -v, --verbose		write more (-v -v gives the table output form)\n\
  -V, --version		output version information and exit\n\
  -w, --wait		wait and retry if connection is down\n");
}


static int get_options(int argc,char **argv)
{
  int c,option_index=0;

  while ((c=getopt_long(argc,argv,"e:h:p:u:#::BTf?qrsvVw",long_options,
			&option_index)) != EOF)
  {
    switch(c) {
    case 'e':
      batch=1;
      batch_readline_command(optarg);
      break;
    case 'f':
      ignore_errors=1;
      break;
    case 'h':
      current_host=my_strdup(optarg,MYF(MY_WME));
      break;
#ifdef SAFE_USER
    case 'u':
      current_user=optarg;
      break;
#endif
    case 'p':
      password=my_strdup(optarg,MYF(MY_FAE));
      while (*optarg) *optarg++= 'x';		// Destroy argument
      break;
    case 'r':
      opt_raw_data=1;
      break;
    case '#':
      DBUG_PUSH(optarg ? optarg : "d:t:o");
      info_flag=1;
      break;
    case 'q': quick=1; break;
    case 's': silent=1; break;
    case 'T': info_flag=1; break;
    case 'v': verbose++; break;
    case 'w': wait_flag=1; break;
    case 'B': batch=silent=1; break;
    case 'P':
      mysql_port= (unsigned int) atoi(optarg);
      break;
    case 'S':
      mysql_unix_port= optarg;
      break;
    case 'V': usage(1); exit(0);
    case 'I':
    case '?':
      usage(0);
      exit(0);
    default:
      fprintf(stderr,"illegal option: -%c\n",opterr);
      usage(0);
      return 1;
    }
  }
  argc-=optind;
  argv+=optind;
  if (argc != 1)
  {
    usage(0);
    exit(1);
  }
  current_db= my_strdup(*argv,MYF(MY_WME));
  if (!current_host)
  {	/* If we don't have a hostname have a look at MYSQL_HOST */
    char *tmp=(char *) getenv("MYSQL_HOST");
    if (tmp)
      current_host = my_strdup(tmp,MYF(MY_WME));
  }
  return(0);
}


static COMMAND *find_command (char *name,char cmd_char)
{
  uint len;
  char *end;

  if (!name)
  {
    len=0;
    end=0;
  }
  else
  {
    while (isspace(*name))
      name++;
    if (strchr(name,';') || strinstr(name,"\\g") != 0)
      return ((COMMAND *)NULL);
    if ((end=strcont(name," \t")))
    {
      len=(uint) (end - name);
      while (isspace(*end))
	end++;
      if (!*end)
	end=0;					// no arguments to function
    }
    else
      len=strlen(name);
  }

  for (uint i= 0; commands[i].name; i++)
  {
    if (commands[i].func &&
	((name && !my_casecmp(name,commands[i].name,len) &&
	  !commands[i].name[len] &&
	  (!end || (end && commands[i].takes_params))) ||
	 !name && commands[i].cmd_char == cmd_char))
      return (&commands[i]);
  }
  return ((COMMAND *)NULL);
}


static bool add_line(String &buffer,char *line,char *in_string)
{
  uchar inchar;
  char buff[80],*pos,*out;
  COMMAND *com;

  if (!line[0])
    return 0;
  if (!batch)
    add_history(line);

  for (pos=out=line ; (inchar= (uchar) *pos) ; pos++)
  {
    if (isspace(inchar) && out == line)
      continue;
    if (inchar == '\\')
    {					// Check if msql command
      if (!(inchar = (uchar) *++pos))
	break;				// readline adds one '\'
      if (*in_string)
      {					// Don't allow commands in string
	*out++='\\';
	*out++= (char) inchar;
	continue;
      }
      if ((com=find_command(NullS,(char) inchar)))
      {
	const String tmp(line,(uint) (out-line));
	buffer.append(tmp);
	if ((*com->func)(&buffer,line) > 0)
	  return 1;				// Quit
	out=line;
      }
      else
      {
	sprintf(buff,"Unknown command '\\%c'.",inchar);
	if (put_info(buff,INFO_ERROR))
	  return 1;
	*out++='\\';
	*out++=(char) inchar;
	continue;
      }
    }
    else if (inchar == ';' && !*in_string)
    {						// ';' is end of command
      const String tmp(line,(uint) (out-line));
      buffer.append(tmp);			// Add this line
      if ((com=find_command(buffer.c_ptr(),0)))
      {
	if ((*com->func)(&buffer,buffer.c_ptr()) > 0)
	  return 1;				// Quit
      }
      else
      {
	int error=com_go(&buffer,0);
	if (error)
	{
	  return error < 0 ? 0 : 1;		// < 0 is not fatal
	}
      }
      buffer.length(0);
      out=line;
    }
    else
    {						// Add found char to buffer
      if (inchar == *in_string)
	*in_string=0;
      else if (!*in_string && (inchar == '\'' || inchar == '"'))
	*in_string=(char) inchar;
      *out++ = (char) inchar;
    }
  }
  *out++='\n';
  String tmp(line,(uint) (out-line));
  buffer.append(tmp);
  return 0;
}

/* **************************************************************** */
/*								    */
/*		    Interface to Readline Completion		    */
/*								    */
/* **************************************************************** */

static char *command_generator (char *text,int state);
static char **mysql_completion (char *text,int start, int end);

/* Tell the GNU Readline library how to complete.  We want to try to complete
   on command names if this is the first word in the line, or on filenames
   if not. */

char **no_completion (char *text __attribute__ ((unused)),
		      char *word __attribute__ ((unused)))
{
  return 0;					/* No filename completion */
}

void initialize_readline (char *name)
{
  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = name;

  /* Tell the completer that we want a crack first. */
  rl_attempted_completion_function = (CPPFunction *)mysql_completion;
  rl_completion_entry_function=(Function *) no_completion;
}

/* Attempt to complete on the contents of TEXT.  START and END show the
   region of TEXT that contains the word to complete.  We can use the
   entire line in case we want to do some simple parsing.  Return the
   array of matches, or NULL if there aren't any. */

	/* ARGSUSED */
static char **mysql_completion (char *text,int start,
				int end __attribute__((unused)))
{
  char **matches;

  /* If this word is at the start of the line, then it is a command
     to complete.  Otherwise no compleation */

  if (start == 0)
    matches = completion_matches(text, command_generator);
  else
    matches= NULL;
  return (matches);
}

/* Generator function for command completion.  STATE lets us know whether
   to start from scratch; without any state (i.e. STATE == 0), then we
   start at the top of the list. */

static char *command_generator (char *text,int state)
{
  static uint list_index, len;
  char *name;

  /* If this is a new word to complete, initialize now.  This includes
     saving the length of TEXT for efficiency, and initializing the index
     variable to 0. */
  if (!state)
  {
    list_index = 0;
    len = strlen (text);
  }

  /* Return the next name which partially matches from the command list. */
  while ((name = commands[list_index].name))
  {
    list_index++;
    if (strncmp (name, text, len) == 0)
    {
      uint tmp=strlen(name)+1;
      char *str=(char*) malloc(tmp);
      if (str)
      {
	memcpy(str,name,tmp);
	return str;
      }
    }
    /* If no names matched, then return NULL. */
  }
  return ((char *)NULL);
}


	/* for gnu readline */

#ifndef HAVE_INDEX
#ifdef	__cplusplus
extern "C" {
#endif
extern char *index(const char *,pchar c),*rindex(const char *,pchar);

char *index(const char *s,pchar c)
{
  for (;;)
  {
     if (*s == (char) c) return (char*) s;
     if (!*s++) return NullS;
  }
}

char *rindex(const char *s,pchar c)
{
  reg3 char *t;

  t = NullS;
  do if (*s == (char) c) t = (char*) s; while (*s++);
  return (char*) t;
}
#ifdef	__cplusplus
}
#endif
#endif


/***************************************************************************
 The different commands
***************************************************************************/

static int
com_help (String *buffer __attribute__((unused)),
	  char *line __attribute__((unused)))
{
  reg1 int i;

  put_info("\nMySQL commands",INFO_INFO);
  for (i = 0; commands[i].name; i++)
  {
    if (commands[i].func)
      printf("%s\t(\\%c)\t%s\n", commands[i].name,commands[i].cmd_char,
	     commands[i].doc);
  }
  puts("");
  return 0;
}


	/* ARGSUSED */
static int
com_clear(String *buffer,char *line __attribute__((unused)))
{
  buffer->length(0);
  return 0;
}

/*
** Execute command
** Returns: 0  if ok
** 	    -1 if not fatal error
**	    1  if fatal error
*/


static int
com_go(String *buffer,char *line __attribute__((unused)))
{
  char		buff[160],time_buff[32];
  MYSQL_RES	*result;
  uint		timer;

	/* Remove garbage for nicer messages */
  LINT_INIT(buff[0]);
  remove_cntrl(*buffer);

  if (buffer->is_empty())
  {
    return put_info("No query specified\n",INFO_ERROR) ? 1 : -1;
  }
  if (!connected)
  {
    return put_info("Not connected to a database\n",INFO_ERROR) ? 1 : -1;
  }
  if (verbose)
    (void) com_print(buffer,0);

  timer=start_timer();
  if (mysql_query(&mysql,buffer->c_ptr()))
  {
    buffer->length(0);				// Remove query on error
    return put_info(mysql_error(&mysql),INFO_ERROR) ? 1 : -1;
  }
  buffer->length(0);

  if (quick)
  {
    if (!(result=mysql_use_result(&mysql)) && mysql_num_fields(&mysql))
    {
      return put_info(mysql_error(&mysql),INFO_ERROR) ? 1 : -1;
    }
    end_timer(timer,time_buff);
    if (mysql_affected_rows(&mysql) != (ulong) ~0L)
      sprintf(buff,"Query OK (%s)\n",time_buff);
    else
      sprintf(buff,"Query OK, %ld rows affected (%s)\n",
	      mysql_affected_rows(&mysql),time_buff);
    put_info(buff,INFO_RESULT);
    if (!result)
      return 0;
  }
  else
  {
    if (!(result=mysql_store_result(&mysql)))
    {
      end_timer(timer,time_buff);
      if (mysql_error(&mysql)[0])
      {
	return put_info(mysql_error(&mysql),INFO_ERROR) ? 1 : -1;
      }
      if (mysql_affected_rows(&mysql) == (ulong) ~0L)
	sprintf(buff,"Query OK (%s)",time_buff);
      else
	sprintf(buff,"Query OK, %ld rows affected (%s)",
		mysql_affected_rows(&mysql),time_buff);
      put_info(buff,INFO_RESULT);
      if (mysql_info(&mysql))
	put_info(mysql_info(&mysql),INFO_RESULT);
      put_info("",INFO_RESULT);			// Empty row
      return 0;
    }
    end_timer(timer,time_buff);
    if (!mysql_num_rows(result))
    {
      sprintf(buff,"Empty set (%s)\n",time_buff);
      put_info(buff,INFO_RESULT);
      mysql_free_result(result);
      return 0;
    }
    sprintf(buff,"%ld rows in set (%s)\n",mysql_num_rows(result),time_buff);
    put_info(buff,INFO_RESULT);
  }

  if (silent && verbose <= 2)
    print_tab_data(result);
  else
    print_table_data(result);

  int error=0;
  if (!mysql_eof(result))		/* Something wrong when using quick */
    error=put_info(mysql_error(&mysql),INFO_ERROR);
  mysql_free_result(result);
  return error;				/* New command follows */
}


static void
print_table_data(MYSQL_RES *result)
{
  String separator(256);
  MYSQL_ROW	cur;
  uint		length;
  MYSQL_FIELD	*field;
  bool		*num_flag;

  num_flag=(bool*) my_alloca(sizeof(bool)*mysql_num_fields(result));
  separator.copy("+",1);
  while((field = mysql_fetch_field(result)))
  {
    uint length=strlen(field->name);
    if (quick)
      length=max(length,field->length);
    else
      length=max(length,field->max_length);
    field->max_length=length+1;
    separator.fill(separator.length()+length+2,'-');
    separator.append('+');
  }
  puts(separator.c_ptr());

  mysql_field_seek(result,0);
  (void) fputs("|",stdout);
  for (uint off=0; (field = mysql_fetch_field(result)) ; off++)
  {
    printf(" %-*s|",field->max_length,field->name);
    num_flag[off]= !(field->type == FIELD_TYPE_VAR_STRING ||
		     field->type == FIELD_TYPE_STRING ||
		     (field->flags & BLOB_FLAG) ||
		     field->type == FIELD_TYPE_NULL);
  }
  (void) fputc('\n',stdout);
  puts(separator.c_ptr());

  while ((cur = mysql_fetch_row(result)))
  {
    (void) fputs("|",stdout);
    mysql_field_seek(result,0);
    for (uint off=0 ; off < mysql_num_fields(result); off++)
    {
      field = mysql_fetch_field(result);
      length=field->max_length;
      printf(num_flag[off] ? "%*s |" : " %-*s|",
	     length,cur[off] ? (char*) cur[off] : "NULL");
    }
    (void) fputc('\n',stdout);
  }
  puts(separator.c_ptr());
  puts("");
  my_afree((gptr) num_flag);
}


static void
safe_put_field(byte *pos,uint length)
{
  if (!pos)
    fputs("NULL",stdout);
  else
  {
    if (opt_raw_data)
      fputs(pos,stdout);
    else for (char *end=pos+length ; pos != end ; pos++)
    {
      if (!*pos)
	fputs("\\0",stdout);
      else if (*pos == '\t')
	fputs("\\t",stdout);
      else if (*pos == '\n')
	fputs("\\n",stdout);
      else if (*pos == '\\')
	fputs("\\\\",stdout);
      else
	putchar(*pos);
    }
  }
}


static void
print_tab_data(MYSQL_RES *result)
{
  MYSQL_ROW	cur;
  MYSQL_FIELD	*field;
  uint 		*lengths;

  int first=0;
  while ((field = mysql_fetch_field(result)))
  {
    if (first++)
      (void) fputc('\t',stdout);
    (void) fputs(field->name,stdout);
  }
  (void) fputc('\n',stdout);

  while ((cur = mysql_fetch_row(result)))
  {
    lengths=mysql_fetch_lengths(result);
    safe_put_field(cur[0],lengths[0]);
    for (uint off=1 ; off < mysql_num_fields(result); off++)
    {
      (void) fputc('\t',stdout);
      safe_put_field(cur[off],lengths[off]);
    }
    (void) fputc('\n',stdout);
  }
}


static int
com_edit(String *buffer,char *line __attribute__((unused)))
{
  char	*filename,*editor,buff[160];
  int	fd,tmp;

  filename = my_tempnam(NULL,"sql",MYF(MY_WME));
  if ((fd = my_create(filename,0,O_CREAT | O_WRONLY, MYF(MY_WME))) < 0)
    goto err;
  (void) my_write(fd,buffer->ptr(),buffer->length(),MYF(MY_WME));
  (void) my_close(fd,MYF(0));

  if (!(editor = (char *)getenv("EDITOR")) &&
      !(editor = (char *)getenv("VISUAL")))
    editor = "vi";
  strxmov(buff,editor," ",filename,NULL);
  (void) system(buff);

  MY_STAT stat;
  if (!my_stat(filename,&stat,MYF(MY_WME)))
    goto err;
  if ((fd = my_open(filename,O_RDONLY, MYF(MY_WME))) < 0)
    goto err;
  (void) buffer->alloc((uint) stat.st_size);
  if ((tmp=read(fd,(char*) buffer->ptr(),buffer->alloced_length())) >= 0L)
    buffer->length((uint) tmp);
  else
    buffer->length(0);
  (void) my_close(fd,MYF(0));
  (void) my_delete(filename,MYF(MY_WME));
err:
  my_free(filename,MYF(0));
  return 0;
}


/* If arg is given, exit without errors. This happens on command 'quit' */

static int
com_quit(String *buffer __attribute__((unused)),
	 char *line __attribute__((unused)))
{
  return 1;
}

static int
com_print(String *buffer,char *line __attribute__((unused)))
{
  puts("--------------");
  (void) fputs(buffer->c_ptr(),stdout);
  if (!buffer->length() || (*buffer)[buffer->length()-1] != '\n')
    putchar('\n');
  puts("--------------\n");
  return 0;					/* If empty buffer */
}

	/* ARGSUSED */
static int
com_connect(String *buffer __attribute__((unused)),
	    char *line)
{
  char *tmp,*last;
  last=0;

  while (isspace(*line))
    line++;
  tmp=(char *) strtok_r(line," \t",&last);	// Skipp connect command
  if (tmp && (tmp=(char *) strtok_r(NullS," \t",&last)))
  {
    my_free(current_db,MYF(0));
    current_db=my_strdup(tmp,MYF(MY_WME));
    if ((tmp=(char *) strtok_r(NullS," \t",&last)))
    {
      my_free(current_host,MYF(0));
      current_host=my_strdup(tmp,MYF(MY_WME));
    }
  }
  (void) sql_connect(current_host,current_db,current_user,password,0);
  return 0;
}


static int
sql_real_connect(char *host,char *database,char *user,char *password,
		 int silent)
{
  if (connected)
  {					/* if old is open, close it first */
    mysql_close(&mysql);
    connected= 0;
  }
  if (!mysql_connect(&mysql,host,user,password))
  {
    if (!silent || !is_prefix(mysql_error(&mysql),CONN_HOST_PREFIX_ERROR))
    {
      put_info(mysql_error(&mysql),INFO_ERROR);
      return 1;
    }
    return -1;					// Retryable
  }
  else if (mysql_select_db(&mysql,database) < 0)
  {
    put_info(mysql_error(&mysql),INFO_ERROR);
    mysql_close(&mysql);
    return 1;
  }
  connected=1;
  return 0;
}


static int
sql_connect(char *host,char *database,char *user,char *password,int silent)
{
  bool message=0;
  uint count=0;
  int error;
  for (;;)
  {
    if ((error=sql_real_connect(host,database,user,password,wait_flag)) >= 0)
    {
      if (count)
      {
	fputs("\n",stderr); 
	(void) fflush(stderr);
      }
      return error;
    }
    if (!wait_flag)
      return 1;
    if (!message && !silent)
    {
      message=1;
      fputs("Waiting",stderr); (void) fflush(stderr);
    }
    (void) sleep(5);
    if (!silent)
    {
      putc('.',stderr); (void) fflush(stderr);
      count++;
    }
  }
}



static int
put_info(const char *str,INFO_TYPE info_type)
{
  static int inited=0;

  if (batch)
  {
    if (info_type == INFO_ERROR)
    {
      (void) fflush(stdout);
      fputs("ERROR: ",stderr);
      fputs(str,stderr);
      fputc('\n',stderr);
      (void) fflush(stderr);
      if (!ignore_errors)
	return 1;
    }
    else if (info_type == INFO_RESULT && verbose > 1)
      puts(str);
    return 0;
  }
  if (silent && info_type != INFO_ERROR)
    return 0;
  if (!inited)
  {
    inited=1;
#ifdef HAVE_SETUPTERM
    (void) setupterm((char *)0, 1, (int *) 0);
#endif
  }
  if (info_type == INFO_ERROR)
  {
    putchar('\007');				/* This should make a bell */
    vidattr(A_STANDOUT);
    fputs("ERROR: ",stdout);
  }
  else
    vidattr(A_BOLD);
  (void) puts(str);
  vidattr(A_NORMAL);
  return 0;
}

static void remove_cntrl(String &buffer)
{
  char *start,*end;
  end=(start=(char*) buffer.ptr())+buffer.length();
  while (start < end && !isgraph(end[-1]))
    end--;
  buffer.length((uint) (end-start));
}


#include <sys/times.h>

static uint start_timer(void)
{
  struct tms tms;
  return (uint) times(&tms);
}

static void end_timer(uint start_time,char *buff)
{
  struct tms tms;
  uint now,tmp;
  double sec;

  now=(uint) times(&tms);
  sec=(double) (now-start_time)/sysconf(_SC_CLK_TCK);
  if (sec >= 3600.0*24)
  {
    tmp=(ulong) floor(sec/(3600.0*24));
    sec-=3600.0*24*tmp;
    buff=int2str((long) tmp,buff,10);
    buff=strmov(buff,tmp > 1 ? " days " : " day ");
  }
  if (sec >= 3600.0)
  {
    tmp=(ulong) floor(sec/3600.0);
    sec-=3600.0*tmp;
    buff=int2str((long) tmp,buff,10);
    buff=strmov(buff,tmp > 1 ? " hours " : " hour ");
  }
  if (sec >= 60.0)
  {
    tmp=(ulong) floor(sec/60.0);
    sec-=60.0*tmp;
    buff=int2str((long) tmp,buff,10);
    buff=strmov(buff," min ");
  }
  sprintf(buff,"%.2f sec",sec);
}

/* Keep sql_string library happy */

gptr sql_alloc(unsigned int Size)
{
  return my_malloc(Size,MYF(MY_WME));
}

void sql_element_free(void *ptr)
{
  my_free((gptr) ptr,MYF(0));
}
