#include "math.h"
#include "matrix.h"
#include <kernel.h>
#include "string.h"

#define sin sin2
#define cos cos2


static void printfloat(float a)
{
  if(a<0)
    {
      printf("-");
      a=-a;
    }
  a*=1000;
  unsigned int A=(unsigned int)a;
  int A1=A/1000;
  int A2=A-A1*1000;
  printf("%i.",A1);
  if(A2==0)printf("000");
  else if(A2<10)printf("00%i",A2);
  else if(A2<100)printf("0%i",A2);
  else printf("%i",A2);
}


void setmatrixcolumnsf(fmatrix *m, fvec *u, fvec *v, fvec *w, fvec *p)
{
  m->xu=u->x;m->xv=v->x;m->xw=w->x;m->xx=p->x;
  m->yu=u->y;m->yv=v->y;m->yw=w->y;m->yy=p->y;
  m->zu=u->z;m->zv=v->z;m->zw=w->z;m->zz=p->z;
}


void transposematrixf(fmatrix *m)
{
  fvec x=fvec_(m->xu,m->xv,m->xw);
  fvec y=fvec_(m->yu,m->yv,m->yw);
  fvec z=fvec_(m->zu,m->zv,m->zw);
  fvec p=fvec_(0,0,0);
  setmatrixcolumnsf(m,&x,&y,&z,&p);
}


void screenmatrixf(fmatrix *m, float width, float height, float offsetx, float offsety)
{
  m->xu+=0.5*m->zu;
  m->xv+=0.5*m->zv;
  m->xw+=0.5*m->zw;
  m->xx+=0.5*m->zz;
  m->yu+=0.5*m->zu;
  m->yv+=0.5*m->zv;
  m->yw+=0.5*m->zw;
  m->yy+=0.5*m->zz;
  
  scalematrixf(m,width,height,1);
  
  m->xu+=offsetx*m->zu;
  m->xv+=offsetx*m->zv;
  m->xw+=offsetx*m->zw;
  m->xx+=offsetx*m->zz;
  m->yu+=offsety*m->zu;
  m->yv+=offsety*m->zv;
  m->yw+=offsety*m->zw;
  m->yy+=offsety*m->zz;
}


void resetmatrixf(fmatrix *m)
{
  m->xu=m->yv=m->zw=1;
  m->xv=m->xw=m->xx=m->yu=m->yw=m->yy=m->zu=m->zv=m->zz=0;
  m->dummy1=m->dummy2=m->dummy3=m->dummy4=0;
}


void scalematrixf(fmatrix *m, float x, float y, float z)
{
  m->xu*=x;
  m->xv*=x;
  m->xw*=x;
  m->xx*=x;
  m->yu*=y;
  m->yv*=y;
  m->yw*=y;
  m->yy*=y;
  m->zu*=z;
  m->zv*=z;
  m->zw*=z;
  m->zz*=z;
}


void rotatematrixzf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
  
  s=sin(v);
  c=cos(v);

  t=m->xu;
  m->xu=t*c-m->yu*s;
  m->yu=t*s+m->yu*c;
  
  t=m->xv;
  m->xv=t*c-m->yv*s;
  m->yv=t*s+m->yv*c;

  t=m->xw;
  m->xw=t*c-m->yw*s;
  m->yw=t*s+m->yw*c;
  
  t=m->xx;
  m->xx=t*c-m->yy*s;
  m->yy=t*s+m->yy*c;
}


void rotatematrixyf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
	
  s=sin(v);
  c=cos(v);
  
  t=m->zu;
  m->zu=t*c-m->xu*s;
  m->xu=t*s+m->xu*c;
  
  t=m->zv;
  m->zv=t*c-m->xv*s;
  m->xv=t*s+m->xv*c;
  
  t=m->zw;
  m->zw=t*c-m->xw*s;
  m->xw=t*s+m->xw*c;
  
  t=m->zz;
  m->zz=t*c-m->xx*s;
  m->xx=t*s+m->xx*c;
}


void rotatematrixxf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
  
  s=sin(v);
  c=cos(v);
  
  t=m->yu;
  m->yu=t*c-m->zu*s;
  m->zu=t*s+m->zu*c;
  
  t=m->yv;
  m->yv=t*c-m->zv*s;
  m->zv=t*s+m->zv*c;
  
  t=m->yw;
  m->yw=t*c-m->zw*s;
  m->zw=t*s+m->zw*c;
  
  t=m->yy;
  m->yy=t*c-m->zz*s;
  m->zz=t*s+m->zz*c;
}


void movematrixf(fmatrix *m, float x, float y, float z)
{
  m->xx+=x;
  m->yy+=y;
  m->zz+=z;
}


void fmovematrixf(fmatrix *m, float x, float y, float z)
{
  m->xx+=(x*(m->xu))+(y*(m->xv))+(z*(m->xw));
  m->yy+=x*m->yu+y*m->yv+z*m->yw;
  m->zz+=x*m->zu+y*m->zv+z*m->zw;
}


void frotatematrixzf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
  
  s=sin(v);
  c=cos(v);

  t=m->xu;
  m->xu=t*c+m->xv*s;
  m->xv=-t*s+m->xv*c;
  
  t=m->yu;
  m->yu=t*c+m->yv*s;
  m->yv=-t*s+m->yv*c;
  
  t=m->zu;
  m->zu=t*c+m->zv*s;
  m->zv=-t*s+m->zv*c;
}


void frotatematrixzf2(fmatrix *m, float c, float s)
{
  float	t;
  
  t=m->xu;
  m->xu=t*c+m->xv*s;
  m->xv=-t*s+m->xv*c;
  
  t=m->yu;
  m->yu=t*c+m->yv*s;
  m->yv=-t*s+m->yv*c;
  
  t=m->zu;
  m->zu=t*c+m->zv*s;
  m->zv=-t*s+m->zv*c;
}


void frotatematrixyf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
#if 1
  cossin2(v);

  __asm__ __volatile__(
		       "lqc2	vf9, 0x20(%1)\n"
		       "lqc2	vf10, 0x0(%1)\n"
		       
		       "vmulx.xyz vf7,vf9,vf11x\n"
		       "vmuly.xyz vf5,vf10,vf11y\n"
		       "vmuly.xyz vf6,vf9,vf11y\n"
		       "vmulx.xyz vf4,vf10,vf11x\n"
		       "vadd.xyz vf9,vf7,vf5\n"
		       "vsub.xyz vf10,vf4,vf6\n"
		       "sqc2	vf9,0x20(%1)\n"
		       "sqc2	vf10,0x0(%1)\n"
		       
		       : : "r" (m),"r" (m));
#else
  s=sin(v);
  c=cos(v);
	
  t=m->xv;
  m->xv=t*c+m->xw*s;
  m->xw=-t*s+m->xw*c;
  
  t=m->yv;
  m->yv=t*c+m->yw*s;
  m->yw=-t*s+m->yw*c;

  t=m->zv;
  m->zv=t*c+m->zw*s;
  m->zw=-t*s+m->zw*c;
#endif
}


void frotatematrixyf2(fmatrix *m, float c, float s)
{
  float	t;
  
  t=m->xw;
  m->xw=t*c+m->xu*s;
  m->xu=-t*s+m->xu*c;
  
  t=m->yw;
  m->yw=t*c+m->yu*s;
  m->yu=-t*s+m->yu*c;
  
  t=m->zw;
  m->zw=t*c+m->zu*s;
  m->zu=-t*s+m->zu*c;
}


void frotatematrixxf(fmatrix *m, float v)
{
  float	s,c;
  float	t;
#if 1
  cossin2(v);
  
  __asm__ __volatile__(
		       "lqc2	vf9, 0x10(%1)\n"
		       "lqc2	vf10, 0x20(%1)\n"
		       
		       "vmulx.xyz vf7,vf9,vf11x\n"
		       "vmuly.xyz vf5,vf10,vf11y\n"
		       "vmuly.xyz vf6,vf9,vf11y\n"
		       "vmulx.xyz vf4,vf10,vf11x\n"
		       "vadd.xyz vf9,vf7,vf5\n"
		       "vsub.xyz vf10,vf4,vf6\n"
		       "sqc2	vf9,0x10(%1)\n"
		       "sqc2	vf10,0x20(%1)\n"
		       
		       : : "r" (m),"r" (m));
#else
  s=sin(v);
  c=cos(v);
  
  t=m->xv;
  m->xv=t*c+m->xw*s;
  m->xw=-t*s+m->xw*c;
  
  t=m->yv;
  m->yv=t*c+m->yw*s;
  m->yw=-t*s+m->yw*c;
  
  t=m->zv;
  m->zv=t*c+m->zw*s;
  m->zw=-t*s+m->zw*c;
#endif
}


void frotatematrixxf2(fmatrix *m, float c, float s)
{
  float t;
  
  t=m->xv;
  m->xv=t*c+m->xw*s;
  m->xw=-t*s+m->xw*c;
  
  t=m->yv;
  m->yv=t*c+m->yw*s;
  m->yw=-t*s+m->yw*c;
  
  t=m->zv;
  m->zv=t*c+m->zw*s;
  m->zw=-t*s+m->zw*c;
}


void fscalematrixf(fmatrix *m, float x, float y, float z)
{
  m->xu*=x;
  m->yu*=x;
  m->zu*=x;
  m->xv*=y;
  m->yv*=y;
  m->zv*=y;
  m->xw*=z;
  m->yw*=z;
  m->zw*=z;
}


void fmulmatrixf(fmatrix *m, fmatrix *n)
{
  //m=m*n
  
  fmatrix temp;

  temp=*m;
  
  m->xu=temp.xu*n->xu+temp.xv*n->yu+temp.xw*n->zu;
  m->xv=temp.xu*n->xv+temp.xv*n->yv+temp.xw*n->zv;
  m->xw=temp.xu*n->xw+temp.xv*n->yw+temp.xw*n->zw;
  m->xx=temp.xu*n->xx+temp.xv*n->yy+temp.xw*n->zz+temp.xx;
  
  m->yu=temp.yu*n->xu+temp.yv*n->yu+temp.yw*n->zu;
  m->yv=temp.yu*n->xv+temp.yv*n->yv+temp.yw*n->zv;
  m->yw=temp.yu*n->xw+temp.yv*n->yw+temp.yw*n->zw;
  m->yy=temp.yu*n->xx+temp.yv*n->yy+temp.yw*n->zz+temp.yy;
  
  m->zu=temp.zu*n->xu+temp.zv*n->yu+temp.zw*n->zu;
  m->zv=temp.zu*n->xv+temp.zv*n->yv+temp.zw*n->zv;
  m->zw=temp.zu*n->xw+temp.zv*n->yw+temp.zw*n->zw;
  m->zz=temp.zu*n->xx+temp.zv*n->yy+temp.zw*n->zz+temp.zz;
}


void mulmatrixf(fmatrix *m, fmatrix *n)
{
  //m=n*m

  fmatrix temp;

  temp=*m;
  
  m->xu=n->xu*temp.xu+n->xv*temp.yu+n->xw*temp.zu;
  m->xv=n->xu*temp.xv+n->xv*temp.yv+n->xw*temp.zv;
  m->xw=n->xu*temp.xw+n->xv*temp.yw+n->xw*temp.zw;
  m->xx=n->xu*temp.xx+n->xv*temp.yy+n->xw*temp.zz+n->xx;
  
  m->yu=n->yu*temp.xu+n->yv*temp.yu+n->yw*temp.zu;
  m->yv=n->yu*temp.xv+n->yv*temp.yv+n->yw*temp.zv;
  m->yw=n->yu*temp.xw+n->yv*temp.yw+n->yw*temp.zw;
  m->yy=n->yu*temp.xx+n->yv*temp.yy+n->yw*temp.zz+n->yy;
  
  m->zu=n->zu*temp.xu+n->zv*temp.yu+n->zw*temp.zu;
  m->zv=n->zu*temp.xv+n->zv*temp.yv+n->zw*temp.zv;
  m->zw=n->zu*temp.xw+n->zv*temp.yw+n->zw*temp.zw;
  m->zz=n->zu*temp.xx+n->zv*temp.yy+n->zw*temp.zz+n->zz;
}


void	rotatematrixf(fmatrix *m, float x, float y, float z, float v)
{
  float	i,j;
  
  i=-atan2(y,x);
  j=atan2(z,sqrt(x*x+y*y));
  
  rotatematrixzf(m,i);
  rotatematrixyf(m,j);
  rotatematrixxf(m,v);
  rotatematrixyf(m,-j);
  rotatematrixzf(m,-i);
}


void	skewmatrixxtozf(fmatrix *m, float s)
{
  m->zu+=s*m->xu;
  m->zv+=s*m->xv;
  m->zw+=s*m->xw;
  m->zz+=s*m->xx;
}


void	skewmatrixytozf(fmatrix *m, float s)
{
  m->zu+=s*m->yu;
  m->zv+=s*m->yv;
  m->zw+=s*m->yw;
  m->zz+=s*m->yy;
}


void	fskewmatrixxtozf(fmatrix *m, float s)
{
  m->xu-=s*m->xw;
  m->yu-=s*m->yw;
  m->zu-=s*m->zw;
}


void	fskewmatrixytozf(fmatrix *m, float s)
{
  m->xv-=s*m->xw;
  m->yv-=s*m->yw;
  m->zv-=s*m->zw;
}


void    transformf(fmatrix *m, fvec *s, fvec *d, int n)
{
  int	t;
  
  for(t=0;t<n;t++)
    {
      d[t].x=m->xu*s[t].x+m->xv*s[t].y+m->xw*s[t].z+m->xx;
      d[t].y=m->yu*s[t].x+m->yv*s[t].y+m->yw*s[t].z+m->yy;
      d[t].z=m->zu*s[t].x+m->zv*s[t].y+m->zw*s[t].z+m->zz;
    }
}


void    transformf2(fmatrix *m, fvec *s, fvec *d, int n)
{
  int	t;
  
  __asm__ __volatile__(
		       "lqc2	vf8, 0x0(%2)\n"
		       "lqc2	vf9, 0x10(%2)\n"
		       "lqc2	vf10, 0x20(%2)\n"
		       "lqc2	vf11, 0x30(%2)\n"
		       
		       
		       : : "r" (&(s[t])),"r" (&(d[t])),"r" (m));
  
  for(t=0;t<n;t++)
    {
      __asm__ __volatile__(
			   "lqc2	vf4, 0x0(%0)\n"
			   
			   "vmulax.xyz	ACC,vf8,vf4x\n"
			   "VMADDAy.xyz	ACC,vf9,vf4y\n"
			   "vmaddz.xyz	vf5,vf10,vf4z\n"
			   "vadd	vf5,vf11,vf5\n"
			   
			   "sqc2	vf5,0x0(%1)\n"
			   
			   : : "r" (&(s[t])),"r" (&(d[t])),"r" (m));
    }
}


void printmatrixf(fmatrix *m)
{
  printfloat(m->xu);printf(" ");
  printfloat(m->xv);printf(" ");
  printfloat(m->xw);printf(" ");
  printfloat(m->xx);printf("\n");
  printfloat(m->yu);printf(" ");
  printfloat(m->yv);printf(" ");
  printfloat(m->yw);printf(" ");
  printfloat(m->yy);printf("\n");
  printfloat(m->zu);printf(" ");
  printfloat(m->zv);printf(" ");
  printfloat(m->zw);printf(" ");
  printfloat(m->zz);printf("\n");
}


void printhexvector(ivec v)
{
  char temp[16];

  printf("%8x %8x %8x %8x\n",v.x,v.y,v.z,v.w);
}


void printvector(fvec v)
{
  printfloat(v.x);printf(" ");
  printfloat(v.y);printf(" ");
  printfloat(v.z);printf(" ");
  printfloat(v.w);printf("\n");
}


void lookatmatrixf(fmatrix *m, fvec direction, fvec up)
{
  fvec invX;
  fvec invY;
  fvec invZ=direction;

  resetmatrixf(m);

  invZ=1.0/length(direction)*direction;
  invX=cross(invZ,up);
  invX=1.0/length(invX)*invX;
  invY=cross(invZ,invX);

  m->xu=invX.x;
  m->xv=invX.y;
  m->xw=invX.z;
  m->yu=invY.x;
  m->yv=invY.y;
  m->yw=invY.z;
  m->zu=invZ.x;
  m->zv=invZ.y;
  m->zw=invZ.z;
}
