/*
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "xcha.h"

FD_xcha *fdui;

/*                            dummy jan feb mar apr may jun jul aug sep oct nov dec */               
static int days_in_month[] = {   0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
static char weekday_strings[7][4] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
extern int timer_form_modified_flag;


struct timer *lookup_timer(char *name)
{
struct timer *pa;

/* argument check */
if(! name) return 0;

for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(strcmp(pa -> name, name) == 0) return pa;
	}

return 0; /* not found */
} /* end function lookup_timer */


struct timer *install_timer_at_end_of_list(char *name)
{
struct timer *plast, *pnew;
struct timer *lookup_timer();

if(debug_flag)
	{
	printf("install_timer_at_end_of_list(): arg name=%s\n", name);
	}

/* argument check */
if(! name) return 0;

/* create new structure */
pnew = (struct timer *) calloc(1, sizeof(*pnew) );
if(! pnew) return 0;

pnew -> name = strsave(name);
if(! pnew -> name) return 0;

/* get previous structure */
plast = timertab[1]; /* end list */

/* set new structure pointers */
pnew -> nxtentr = 0; /* new points to zero (is end) */
pnew -> prventr = plast; /* point to previous entry, or 0 if first entry */

/* set previous structure pointers */
if( !timertab[0] ) timertab[0] = pnew; /* first element in list */
else plast -> nxtentr = pnew;

/* set array end pointer */
timertab[1] = pnew;

return pnew;/* pointer to new structure */
}/* end function install_timer_at_end_of_list */


int delete_timer(char *name)/* delete entry from double linked list */
{
struct timer *pa, *pprev, *pdel, *pnext;

if(debug_flag)
	{
	printf("delete_timer(): arg name=%s\n", name);
	}

/* argument check */
if(! name) return 0;

pa = timertab[0];
while(1)
	{
	/* if end list, return not found */
	if(! pa) return 0;

	/* test for match in name */
	if(strcmp(name, pa -> name) != 0) /* no match */
		{
		/* point to next element in list */
		pa = pa -> nxtentr;

		/* loop for next element in list */
		continue;
		}

	/* we now know which struture to delete */
	pdel = pa;

	/* get previous and next structure */
	pnext = pa -> nxtentr;
	pprev = pa -> prventr;

	/* set pointers for previous structure */
	/* if first one, modify timertab[0] */
	if(pprev == 0) timertab[0] = pnext;
	else pprev -> nxtentr = pnext;

	/* set pointers for next structure */
	/* if last one, modify timertab[1] */
	if(pnext == 0) timertab[1] = pprev;
	else pnext -> prventr = pprev;
	
	/* delete structure */	
	/* delete data */
	free(pdel -> name);
	if(pdel -> directory) free(pdel -> directory);
	if(pdel -> filename) free(pdel -> filename);
	free(pdel); /* free structure */

	/* return OK deleted */
	return 1;
	}/* end for all structures */

}/* end function delete_timer */


int delete_all_timers()/* delete all entries from table */
{
struct timer *pa;

if(debug_flag)
	{
	printf("delete_all_timers() arg none\n");
	}

while(1)
	{	
	pa = timertab[0];
	if(! pa) break;

	timertab[0] = pa -> nxtentr;
	free(pa -> name);
	if(pa -> directory) free(pa -> directory);
	if(pa -> filename) free(pa -> filename);
	free(pa);/* free structure */
	}/* end while all structures */

timertab[1] = 0;
return 1;
}/* end function delete_all_timers */


int sort_timers()
/*
sorts the double linked list with as criterium that the lowest ASCII goes
on top.
doing some sort of bubble sort.
*/
{
struct timer *pa;
struct timer *pb;
int swap_flag;
unsigned long long la, lb;

if(debug_flag)
	{
	printf("sort_timers() arg none\n");
	}
		
while(1)/* go through list again and again */
	{
	swap_flag = 0;
	for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
		{
		if(debug_flag)
			{
			printf("sort_timers(): sorting %s pa=%lu\n",\
			pa -> name, pa);		
			}	
	
		pb = pa -> prventr;
		if(debug_flag)
			{
			printf("pb=pa->prventr=%lu\n", pb);
			}
		
		if(pb)
			{
			/* compare */
			if(pa -> start_time < pb -> start_time)
				{
				swap_flag = swap_timer_position(pa , pb);
				/* indicate position was swapped */
				}/* end if strcmp < 0 */
			}/* end if pb */
		}/* end for all entries */

	/* if no more swapping took place, ready, list is sorted */
	if(! swap_flag) break;
	}/* end while go through list again and again */

return 1;
}/* end function sort_timers */


int test_timer_overlap()
{
/* must be called AFTER sort! */
struct timer *pa, *pprev, *pnext;

if(debug_flag)
	{
	printf("test_timer_overlap(): arg none\n");
	}

/* global, prevents user from closing form if set (error) */
timer_overlap_flag = 0;

for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
	{
	pprev = pa -> prventr;
	pnext = pa -> nxtentr;

	/* test for overlap */
	pa -> status &= ~TIMER_OVERLAP;

	if(pprev)
		{
		if(pa -> start_time < pprev -> stop_time)
			{
			pa -> status |= TIMER_OVERLAP;
			timer_overlap_flag = 1;
			}
		}

	if(pnext)
		{
		if(pa -> stop_time > pnext -> start_time)
			{
			pa -> status |= TIMER_OVERLAP;	
			timer_overlap_flag = 1;
			}
		}

	} /* else for al timers */

return 0; /* not found */
} /* end function test_timer_overlap */


int load_timers()
{
int a, i;
FILE *timers_dat_file;
char temp[READSIZE];
char pathfilename[TEMP_SIZE];
struct timer *pa;
char *ptr;
char name[TEMP_SIZE];
char directory[TEMP_SIZE];
char filename[TEMP_SIZE];
time_t start_time;
time_t stop_time;
int start_year, start_month, start_day, start_hour, start_minute;
int start_weekday;
int stop_year, stop_month, stop_day, stop_hour, stop_minute;
int status;
int browser_line;
unsigned long showview;

if(debug_flag)
	{
	printf("load_timers() arg none\n");
	}
	
delete_all_timers();

sprintf(pathfilename, "%s/.xcha/timers.dat", home_dir);
timers_dat_file = fopen(pathfilename, "r");
if(! timers_dat_file)
	{
	fl_show_alert("Could not open file", pathfilename, "for read", 0);
	if(debug_flag)
		{
		printf("could not load file %s\n", pathfilename);
		}

	return 0;
	} 

while(1)
	{
	a = readline(timers_dat_file, temp);
	if(a == EOF) return 1;

	strcpy(name, temp);

	a = readline(timers_dat_file, temp);
	if(a == EOF)
		{
		printf("load_timers(): early EOF\n");

		return 0;
		}

	a = sscanf(temp, "%ld %ld %d %d  %d %d %d  %d %d  %d %d %d  %d %d  %d  %lu",\
	&start_time, &stop_time,\
	&status, &browser_line,\
	&start_year, &start_month, &start_day,\
	&start_hour, &start_minute,\
	&stop_year, &stop_month, &stop_day,\
	&stop_hour, &stop_minute,\
	&start_weekday,\
	&showview); 

	if(a != 16) break; // file format error
 
	a = readline(timers_dat_file, temp);
	if(a == EOF)
		{
		printf("load_timers(): early EOF\n");

		return 0;
		}

	strcpy(directory, temp);

	a = readline(timers_dat_file, temp);
	if(a == EOF)
		{
		printf("load_timers(): early EOF\n");

		return 0;
		}

	strcpy(filename, temp);

	pa = install_timer_at_end_of_list(name);
	if(! pa)
		{
		if(debug_flag)
			{
			printf("load_timers(): cannot install timer %s\n", name);
			}

		fclose(timers_dat_file);
		return 0;
		}

	pa -> directory = strsave(directory);
	if(! pa -> directory)
		{
		printf("load_timers(): strsave directory failed\n");

		fclose(timers_dat_file);
		return 0;
		}

	pa -> filename = strsave(filename);
	if(! pa -> filename)
		{
		printf("load_timers(): strsave filename failed\n");

		fclose(timers_dat_file);
		return 0;
		}

	pa -> start_time = start_time;
	pa -> stop_time = stop_time;
	pa -> start_year = start_year;
	pa -> start_month = start_month;
	pa -> start_day = start_day;
	pa -> start_hour = start_hour;
	pa -> start_minute = start_minute;
	pa -> start_weekday = start_weekday;
	pa -> stop_year = stop_year;
	pa -> stop_month = stop_month;
	pa -> stop_day = stop_day;
	pa -> stop_hour = stop_hour;
	pa -> stop_minute = stop_minute;
	pa -> showview = showview;

	pa -> status = status;
	pa -> browser_line = browser_line;

	if(a == EOF) break;
	}/* end while all lines in timers.dat */

return 1;
}/* end function load_timers */


int save_timers()
{
FILE *timers_dat_file;
char pathfilename[TEMP_SIZE];
char pathfilename2[TEMP_SIZE];
struct timer *pa;

if(debug_flag)
	{
	printf("save_timers() arg none\n");
	}

sprintf(pathfilename, "%s/.xcha/timers.tmp", home_dir);
timers_dat_file = fopen(pathfilename, "w");
if(! timers_dat_file)
	{
	fl_show_alert("Could not open file", pathfilename, "for write", 0);

	if(debug_flag)
		{
		printf("could not open file %s for write\n", pathfilename);
		}
	return 0;
	} 

/* save list of timers */
for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)	
	{
	fprintf(timers_dat_file, "%s\n", pa -> name);
	fprintf(timers_dat_file,\
	"%lu %lu  %d %d  %d %d %d  %d %d  %d %d %d  %d %d  %d  %lu\n",\
	pa -> start_time, pa -> stop_time,\
	pa -> status, pa -> browser_line,\
	pa -> start_year, pa -> start_month, pa -> start_day,\
	pa -> start_hour, pa -> start_minute,\
	pa -> stop_year, pa -> stop_month, pa -> stop_day,\
	pa -> stop_hour, pa -> stop_minute,\
	pa -> start_weekday,\
	pa -> showview);

	fprintf(timers_dat_file, "%s\n", pa -> directory);
	fprintf(timers_dat_file, "%s\n", pa -> filename);
	}/* end while all elements in list */

fclose(timers_dat_file);

/* set some path file names */
sprintf(pathfilename, "%s/.xcha/timers.dat", home_dir);
sprintf(pathfilename2, "%s/.xcha/timers.dat~", home_dir);

/* unlink the old .dat~ */
unlink(pathfilename2);

/* rename .dat to .dat~ */
if( rename(pathfilename, pathfilename2) == -1)
	{
	if(debug_flag)
		{
		printf("save_timers(); rename %s into %s failed\n",\
		pathfilename, pathfilename2);
		}

	/* if first time there is no .dat */
//	return 0;
	}

/* rename .tmp to .dat */
sprintf(pathfilename2, "%s/.xcha/timers.tmp", home_dir);
if( rename(pathfilename2, pathfilename) == -1)
	{
	if(debug_flag)
		{
		printf("save_timers(); rename %s into %s failed\n",\
		pathfilename, pathfilename2);
		}

	return 0;
	}

return 1;/* ok */
}/* end function save_timers */


int show_timer_form()
{
show_timers(SHOW_UNREAD_AT_TOP);

fl_show_form(fdui -> timer_form, FL_PLACE_CENTER, FL_TRANSIENT, "");

timer_form_shown_flag = 1;

return 1;
}/* end function show_timer_form */


int show_timers(int position)
{
char temp[TEMP_SIZE];
struct timer *pa;
char formatstr[20];
int browser_line;
int browser_topline;
int first_new_timer_line;
int find_first_new_timer_flag;
int pa_line;

int maxline;

if(debug_flag)
	{
	printf("show_timers(): arg position=%d\n", position);
	}

sort_timers();

test_timer_overlap();

/* remember vertical slider position */
browser_topline = fl_get_browser_topline(fdui -> timer_form_browser);

fl_freeze_form(fdui -> timer_form);

fl_clear_browser(fdui -> timer_form_browser);

first_new_timer_line = -1;/* keep gcc -Wall from complaining */
find_first_new_timer_flag = 1;
browser_line = 1;
for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)	
	{
	if(pa == selected_timer_pa) pa_line = browser_line;

	if(position == SHOW_UNREAD_AT_TOP)
		{
		/* want to show the first unaccessed at top of display */
		if(find_first_new_timer_flag)
			{
			if(pa -> status == 1)
				{
				first_new_timer_line = browser_line;
				find_first_new_timer_flag = 0;
				}
			}
		}
	
	strcpy(formatstr, "@f");
	if(pa -> status == 0) strcat(formatstr, "@C0");/* black */

	// never visible 
//	if(pa -> status & TIMER_RUNNING) strcat(formatstr, "@C4");/* blue */

	if(pa -> status & TIMER_EXPIRED) strcat(formatstr, "@C2"); /* green */
	if(pa -> status & TIMER_REPEAT_DAILY) strcat(formatstr, "@C24"); /* orange */
	if(pa -> status & TIMER_REPEAT_WORK_DAYS) strcat(formatstr, "@C10"); /* mat blue brown */
	if(pa -> status & TIMER_REPEAT_WEEKLY) strcat(formatstr, "@C21"); /* dark cyan */

	if(pa -> status & TIMER_RUNNING_CANCELLED) strcat(formatstr, "@C4"); /* blue */
	if(pa -> status & TIMER_ERROR) strcat(formatstr, "@C5"); /* magenta */
	if(pa -> status & TIMER_OVERLAP) strcat(formatstr, "@C1");/* red */

	sprintf(temp,\
	"%s %s  from %d-%d-%d %02d:%02d  to %d-%d-%d %02d:%02d file %s/%s",\
	formatstr, pa -> name,\
	pa -> start_day, pa -> start_month, pa -> start_year,\
	pa -> start_hour, pa -> start_minute,\
	pa -> stop_day, pa -> stop_month, pa -> stop_year,\
	pa -> stop_hour, pa -> stop_minute,\
	pa -> directory, pa -> filename);

	fl_add_browser_line(fdui -> timer_form_browser, temp);
	pa -> browser_line = browser_line;/* first one is 1 */
	browser_line++;
	}/* end for all elements in list */
	
/* re adjust browser for same position of top line (vertical slider) */	
if(position == SHOW_UNREAD_AT_TOP)
	{
	fl_set_browser_topline(fdui -> timer_form_browser, first_new_timer_line);
	}
if(position == SHOW_BOTTOM)
	{
	/* make the status of the last timer visible */
	maxline = fl_get_browser_maxline(fdui -> timer_form_browser);
	fl_set_browser_topline(fdui -> timer_form_browser, maxline);
	}
/*
if(position == SHOW_TOP)
	{
	fl_set_browser_topline(fdui -> timer_form_browser, 1);
	}
*/
if(position == SHOW_SAME_POSITION)
	{
	fl_set_browser_topline(fdui -> timer_form_browser, browser_topline);
	}	

fl_unfreeze_form(fdui -> timer_form);

/* selected what was selected */
fl_select_browser_line(fdui -> timer_form_browser, pa_line);

return 1;
}/* end function show_timers */


char *line_to_timer(int line)
{
struct timer *pa;
extern long atol();

if(debug_flag)
	{
	printf("line_to_timer(): arg line=%d\n", line);
	}

/* argument check */
if(line < 0) return 0;

for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)	
	{
	if(pa -> browser_line == line)
		{
		return pa -> name;
		}
	}

/* no save_timers, nothing was changed */
return 0;
}/* end function line_to_timer_id */


int timer_to_line(char *name, int *line)
{
struct timer *pa;

if(debug_flag)
	{
	printf("timer_to_line(): arg name=%s\n", name);
	}

/* argument check */
if(! name) return 0;

pa = lookup_timer(name);
if(! pa) return 0;

*line = pa -> browser_line;

return 1;
}/* end function timer_to_line */


int swap_timer_position(struct timer *ptop, struct timer *pbottom)
{
struct timer *punder;
struct timer *pabove;

if(debug_flag)
	{
	printf("swap_timer_position(): swapping top=%lu bottom=%lu\n",\
	ptop, pbottom);
	}

/* argument check */
if(! ptop) return 0;
if(! pbottom) return 0;

/* get one below the bottom */
punder = pbottom -> prventr;/* could be zero if first entry */
if(debug_flag)
	{
	printf("swap_timer_position(): punder=%lu\n", punder);
	}

/* get the one above the top */
pabove = ptop -> nxtentr;/* could be zero if last entry */
if(debug_flag)
	{
	printf("swap_timer_position(): pabove=%lu\n", pabove);
	}

/* the next pointer in punder (or timertab[0]) must now point to ptop */
if(! punder)
	{
	timertab[0] = ptop; 
	}
else
	{
	punder -> nxtentr = ptop;
	}

/* the prev pointer in in ptop must now point to punder */
ptop -> prventr = punder;/* could be zero if first entry */

/* the next pointer in ptop must now point to pbottom */
ptop -> nxtentr = pbottom;

/* the next pointer in pbottom must now point to pabove */
pbottom -> nxtentr = pabove;

/* mark last one in timertab */
if(! pabove)
	{
	timertab[1] = pbottom;
	}
else
	{
	/* the prev pointer in pabove must now point to pbottom */
	pabove -> prventr = pbottom;
	}

/* the prev pointer in pbottom must now point to ptop */
pbottom -> prventr = ptop;

/* return swapped */
return 1;
}/* end function swap_timer_position */


int add_timer_to_list(int line)
{
char *ptr;
struct timer *pa;
time_t now;
char weekday_str[80];
char month_str[80];
int monthday, hour, minute, second, year;
int timer_line;

if(debug_flag)
	{
	printf("add_timer_to_list(): arg line=%d\n", line);
	}

ptr = get_selected_program_definition(line);
if(! ptr)
	{
	if(debug_flag)
		{
		printf("add_timer_to_list(): cannot get program definition\n");
		}	
	return 0;
	}

if(timer_form_record_filename) free(timer_form_record_filename);
timer_form_record_filename = strsave(ptr);
if(! timer_form_record_filename)
	{
	printf(\
	"add_timer_to_list(): strsave timer_form_record_filename failed\n");

	return 0;
	}

pa = install_timer_at_end_of_list(ptr);
if(! pa)
	{
	fl_show_alert(\
	"add_timer():",\
	"cannot install timer", \
	"command cancelled", 0);
	return 0;
	}

selected_timer_pa = pa;

pa -> status = 0;
pa -> status |= TIMER_REPEAT_NO;
pa -> browser_line = 0;

now = time(NULL);

pa -> start_time = now;
pa -> stop_time = now;

get_this_time(now,\
	weekday_str, month_str, &monthday, &hour, &minute, &second, &year);
pa -> start_year = year;
pa -> start_month = month_to_digit(month_str);
pa -> start_day = monthday;
pa -> start_hour = hour;
pa -> start_minute = minute;
pa -> start_weekday = weekday_to_digit(weekday_str);
pa -> stop_year = year;
pa -> stop_month = pa -> start_month;
pa -> stop_day = monthday;
pa -> stop_hour = hour;
pa -> stop_minute = minute;

pa -> directory = strsave(record_dir);
if(! pa -> directory)
	{
	printf("add_timer_to_list(): strsave pa->record_dir failed\n");
	}

if(! create_filename(timer_form_record_filename, pa) ) return 0;

/* set the input fields */
timer_to_inputs(pa);

timer_form_modified_flag = 1;

show_timer_form();

if(! pa_to_timer_line(pa, &timer_line) ) return 0;

/* mark new entry in browser */
fl_select_browser_line(fdui -> timer_form_browser, timer_line);

/* make visible */
fl_set_browser_topline(fdui -> timer_form_browser, timer_line - 10);

return 1;
} /* end function add_timer_to_list */


int inputs_to_timer(struct timer *pa, int update_length_field_flag)
{
char *ptr;
char temp[TEMP_SIZE];
struct tm *ptime;
time_t start_time;
time_t stop_time;
int status;
int daily, weekly, work_days;
long length;

if(debug_flag)
	{
	printf("inputs_to_timer(): arg pa=%lu update_length_field_flag=%d\n",\
	pa, update_length_field_flag);
	}

/* argument check */
if(! pa) return 0;

ptime = (struct tm *)malloc( sizeof(struct tm) );
if(! ptime)
	{
	printf("inputs_to_timer(): malloc struct ptime failed\n");
	return 0;
	}

ptr = (char *) fl_get_input(fdui -> timer_form_start_year_input_field);
if(! ptr) return 0;
pa -> start_year = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_start_month_input_field);
if(! ptr) return 0;
pa -> start_month = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_start_day_input_field);
if(! ptr) return 0;
pa -> start_day = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_start_hour_input_field);
if(! ptr) return 0;
pa -> start_hour = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_start_minute_input_field);
if(! ptr) return 0;
pa -> start_minute = atoi(ptr);

ptime -> tm_sec = 0;
ptime -> tm_min = pa -> start_minute;
ptime -> tm_hour = pa -> start_hour;
ptime -> tm_mday = pa -> start_day;
ptime -> tm_mon = (pa -> start_month) - 1;
ptime -> tm_year = (pa -> start_year) - 1900;
ptime -> tm_isdst = -1; /* daylight savings time info n.a. */
/*
int tm_isdst
This is a flag that indicates whether Daylight Saving Time is
(or was, or will be) in effect at the time described.
The value is positive if Daylight Saving Time is in effect,
zero if it is not, and negative if the information is not available.
*/
/*
long int tm_gmtoff
This field describes the time zone that was used to compute
this broken-down time value, including any adjustment for
daylight saving; it is the number of seconds that you must
add to UTC to get local time.
You can also think of this as the number of seconds east of UTC.
For example, for U.S. Eastern Standard Time, the value is `-5*60*60'.
The `tm_gmtoff' field is derived from BSD and is a GNU library
extension; it is not visible in a strict ISO C environment.
*/

pa -> start_time = mktime( (struct tm *)ptime);
if(debug_flag)
	{
	printf("inputs_to_timer(): ptime -> tm_isdst=%d\n",\
	ptime -> tm_isdst);
	}

/* calculated by mktime() */
pa -> start_weekday = ptime -> tm_wday;

ptr = (char *) fl_get_input(fdui -> timer_form_stop_year_input_field);
if(! ptr) return 0;
pa -> stop_year = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_stop_month_input_field);
if(! ptr) return 0;
pa -> stop_month = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_stop_day_input_field);
if(! ptr) return 0;
pa -> stop_day = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_stop_hour_input_field);
if(! ptr) return 0;
pa -> stop_hour = atoi(ptr);

ptr = (char *) fl_get_input(fdui -> timer_form_stop_minute_input_field);
if(! ptr) return 0;
pa -> stop_minute = atoi(ptr);

ptime -> tm_sec = 0;
ptime -> tm_min = pa -> stop_minute;
ptime -> tm_hour = pa -> stop_hour;
ptime -> tm_mday = pa -> stop_day;
ptime -> tm_mon = (pa -> stop_month) - 1;
ptime -> tm_year = (pa -> stop_year) - 1900;
ptime -> tm_isdst = -1; /* daylight savings time info n.a. */

pa -> stop_time = mktime( (struct tm *)ptime);

/* free the structure */
free(ptime);

ptr = (char *)fl_get_input(fdui -> timer_form_directory_input_field);
if(! ptr) return 0;

if(pa -> directory) free(pa -> directory);
pa -> directory = strsave(ptr);
if(! pa -> directory)
	{
	printf("inputs_to_timer(): strsave directory failed\n");
	return 0;
	}

if(! create_filename(pa -> name, pa) )
	{
	/*
	allow daily and weekly to be modified if called from timer puldown menu,	
	and timer_form_record_filename is not set.
	*/
//	return 0;
	}

fl_set_object_label(fdui -> timer_form_filename_output_field,\
pa -> filename);

daily = fl_get_button(fdui -> timer_form_daily_button);
weekly = fl_get_button(fdui -> timer_form_weekly_button);
work_days = fl_get_button(fdui -> timer_form_work_days_button);

pa -> status &= ~TIMER_REPEAT_NO;
pa -> status &= ~TIMER_REPEAT_DAILY;
pa -> status &= ~TIMER_REPEAT_WEEKLY;
pa -> status &= ~TIMER_REPEAT_WORK_DAYS;

if(daily) pa -> status |= TIMER_REPEAT_DAILY;
else if(weekly) pa -> status |= TIMER_REPEAT_WEEKLY;
else if(work_days) pa -> status |= TIMER_REPEAT_WORK_DAYS;
else pa -> status |= TIMER_REPEAT_NO;

set_weekday_output(pa);

if(update_length_field_flag)
	{
	length = pa -> stop_time - pa -> start_time;
	set_timer_length_input(length);
	} /* end if update_length_field_flag */

return 1;
} /* end function inputs_to_timer */


int timer_to_inputs(struct timer *pa)
{
char temp[TEMP_SIZE];
long length;

if(debug_flag)
	{
	printf("timer_to_inputs(): arg pa=%lu\n", pa);
	}

/* argument check */
if(! pa) return 0;

fl_set_input(fdui -> timer_form_directory_input_field, pa -> directory);

fl_set_object_label(fdui -> timer_form_filename_output_field,\
pa -> filename);

sprintf(temp, "%d", pa -> start_year);
fl_set_input(fdui -> timer_form_start_year_input_field, temp);

sprintf(temp, "%d", pa -> start_month);
fl_set_input(fdui -> timer_form_start_month_input_field, temp);

sprintf(temp, "%d", pa -> start_day);
fl_set_input(fdui -> timer_form_start_day_input_field, temp);

sprintf(temp, "%d", pa -> start_hour);
fl_set_input(fdui -> timer_form_start_hour_input_field, temp);

sprintf(temp, "%d", pa -> start_minute);
fl_set_input(fdui -> timer_form_start_minute_input_field, temp);

sprintf(temp, "%d", pa -> stop_year);
fl_set_input(fdui -> timer_form_stop_year_input_field, temp);

sprintf(temp, "%d", pa -> stop_month);
fl_set_input(fdui -> timer_form_stop_month_input_field, temp);

sprintf(temp, "%d", pa -> stop_day);
fl_set_input(fdui -> timer_form_stop_day_input_field, temp);

sprintf(temp, "%d", pa -> stop_hour);
fl_set_input(fdui -> timer_form_stop_hour_input_field, temp);

sprintf(temp, "%d", pa -> stop_minute);
fl_set_input(fdui -> timer_form_stop_minute_input_field, temp);

length = pa -> stop_time - pa -> start_time;
set_timer_length_input(length);

if(pa -> status & TIMER_REPEAT_DAILY)
	{
	fl_set_button(fdui -> timer_form_daily_button, 1);
	}
else
	{
	fl_set_button(fdui -> timer_form_daily_button, 0);
	}

if(pa -> status & TIMER_REPEAT_WEEKLY)
	{
	fl_set_button(fdui -> timer_form_weekly_button, 1);
	}	
else
	{
	fl_set_button(fdui -> timer_form_weekly_button, 0);
	}

if(pa -> status &  TIMER_REPEAT_WORK_DAYS)
	{
	fl_set_button(fdui -> timer_form_work_days_button, 1);
	}
else
	{
	fl_set_button(fdui -> timer_form_work_days_button, 0);
	}

set_weekday_output(pa);

if(pa -> showview) sprintf(temp, "%lu", pa -> showview);
else sprintf(temp, "");

fl_set_object_label(fdui -> timer_form_showview_output_field, temp);

set_weekday_output(pa);

XSync(fl_get_display(), 0);

return 1;
} /* end function timer_to_inputs */


int pa_to_timer_line(struct timer *pt, int *timer_line)
{
struct timer *pa;
int line;

if(debug_flag)
	{
	printf("pa_to_timer_line(): arg pt=%lu\n", pt);
	}

/* browser lines start at 1 */
line = 1;
for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(pt == pa)
		{
		*timer_line = line;

		return 1;
		} 

	line++;
	} /* end for all timers in list */

/* not found */
return 0;
} /* end function pa_to_timer_line */


struct timer *timer_line_to_pa(int line)
{
struct timer *pa;

if(debug_flag)
	{
	printf("timer_line_to_pa(): arg line=%d\n", line);
	}

for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(pa -> browser_line == line) return pa;
	} /*end for all timers in list */

/* not found */
return 0;
}/* end function timer_line_to_pa */


int delete_timer_pa(struct timer *pt)
{
struct timer *pa, *pprev, *pdel, *pnext;

if(debug_flag)
	{
	printf("delete_timer_pa(): arg pt=%p\n", pt);
	}

/* argument check */
if(! pt) return 0;

pa = timertab[0];
while(1)
	{
	/* if end list, return not found */
	if(! pa) return 0;

	/* test for match */
	if(pa != pt) /* no match */
		{
		/* point to next element in list */
		pa = pa -> nxtentr;

		/* loop for next element in list */
		continue;
		}

	/* we now know which struture to delete */

	/* last user confirmation */
	if(! fl_show_question("Delete selected timer?", 0) ) /* mouse on no */
		{
		return 0;
		}

	pdel = pa;

	/* get previous and next structure */
	pnext = pa -> nxtentr;
	pprev = pa -> prventr;

	/* set pointers for previous structure */
	/* if first one, modify timertab[0] */
	if(pprev == 0) timertab[0] = pnext;
	else pprev -> nxtentr = pnext;

	/* set pointers for next structure */
	/* if last one, modify timertab[1] */
	if(pnext == 0) timertab[1] = pprev;
	else pnext -> prventr = pprev;
	
	/* delete structure */	
	/* delete data */
	free(pdel -> name);
	if(pdel -> directory) free(pdel -> directory);
	if(pdel -> filename) free(pdel -> filename);
	free(pdel); /* free structure */

	/* return OK deleted */
	return 1;
	}/* end for all structures */

/* not found */
return 0;
} /* end function delete_timer_pa */


int process_timers()
{
struct timer *pa;
time_t now;

if(timer_form_shown_flag) return;

if(debug_flag)
	{
	printf("process_timers(): arg none\n");
	}

for(pa = timertab[0]; pa != 0; pa = pa -> nxtentr)
	{
	now = time(NULL);

//	printf("TIMERS: pa=%p now=%lu diff_start=%lu diff_stop=%lu\n\
//	pa->start_time=%lu pa->stop_time=%lu\n\n",\
//	pa, now, pa -> start_time - now, now - pa -> stop_time,\
//	pa -> start_time, pa -> stop_time);

	/* sanity check */
	if(pa -> stop_time <= pa -> start_time) continue;
	
	if(pa -> status & TIMER_EXPIRED) continue;

	if(pa -> status & TIMER_RUNNING_CANCELLED)
		{
		/* fake a later time, much later... */
		/* 
		Note (from libc.info):
		In the GNU C library, time_t' is equivalent to long int'.
		In other systems, time_t' might be either an integer or
		floating-point type.
		So change LONG_MAX accordingly!
		*/
		now = LONG_MAX;
		} /* end if status cancelled */

	if( (now >= pa -> start_time) && (now < pa -> stop_time) )
		{
		/* if not running start recording */
		if(! (pa -> status & TIMER_RUNNING) )
			{
			/* start recording */			
			if(debug_flag)
				{
				printf("STARTING RECORDING pa->name=%s\n", pa -> name);
				}

			if( timer_start_recording(pa) )
				{
				running_timer = pa;
		 		pa -> status |= TIMER_RUNNING;

				save_timers();

				return 1;
				}
			else
				{
				pa -> status |= TIMER_ERROR;

				save_timers();
				return 0;
				}
			} /* end if timer not running */

		} /* end if >= start_time and < stop_time */

	if(now >= pa -> stop_time)
		{	
		/* if running stop recording */
		if(pa -> status & TIMER_RUNNING)
			{
			/* stop recording */

			if(debug_flag)
				{
				printf("STOP RECORDING pa->name=%s\n", pa -> name);
				}

			if(timer_stop_recording() )
				{
				pa -> status &= ~TIMER_RUNNING;
				pa -> status |= TIMER_EXPIRED;

				running_timer = 0;

				/* handle daily, work days, and weekly by modifying timer */
				if( (pa -> status & TIMER_REPEAT_DAILY) ||\
				(pa -> status & TIMER_REPEAT_WEEKLY) ||\
				(pa -> status & TIMER_REPEAT_WORK_DAYS)	)
					{
					/* leave any error flag */
					pa -> status &= ~TIMER_EXPIRED;

					/* if timer was cancelled manually, allow next run */
					pa -> status &= ~TIMER_RUNNING_CANCELLED;

					/* update time and filename */
					if(! update_time_and_filename(pa) )
						{
						pa -> status |= TIMER_ERROR;
						}
					} /* end if daily or weekly */

				save_timers();
				return 1;
				} /* end if recording stopped OK */
			else
				{
				pa -> status |= TIMER_ERROR;

				save_timers();
				return 0;
				} /* end if recording stopped error */
				
			} /* end if timer running */

		} /* end if >= stop_time */

	} /*end for all timers in list */

/* not active timer found */
return 0;
} /* end function process_timers */


int timer_start_recording(struct timer *pt)
{
struct program *pa;
int power;
char temp[TEMP_SIZE];

if(debug_flag)
	{
	printf("timer_start_recording(): arg pt=%p\n", pa);
	}

if(recorder_mode == RECORD_MODE) return 0;

/* release pause button */
fl_set_button(fdui -> main_form_pause_button, 0);
recorder_pause_flag = 0;

/* release beep button */
fl_set_button(fdui -> main_form_beep_button, 0);
beep_mode = 0;

/* find the entry in the linked list */
for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)
    {
    if(strcmp(pa -> name, pt -> name) == 0)
		{
		clear_status_line();

		set_wait_indicator(1);

		if(recorder_mode == RECORD_MODE)
			{
			stop_recorder();

			/* teletext cannot be running */

			/* stop the pids routine (in a thread) */
			getpids_stop();

			/* if we were recording, receiver is stil active */
			recorder_mode = STOP_MODE;
			} /* end if record was active */
		else if(recorder_mode == PLAYBACK_MODE)
			{
			stop_recorder();

			/*
			if we were playing, re-start receiver and
			tune to selected channel again
			*/

			/* free the tuning devices */
			close_all_devices();

			/* try starting the receiver again */
			if(! init_tuning() )
				{
				printf("timer_start_recording(): Could not init tuning\n");

//				fl_show_alert("Could not init tuning", "aborting", "", 1);

				return 0;
				}

			recorder_mode = STOP_MODE;
			} /* end if play was active */
		else if(recorder_mode == STOP_MODE)
			{
			/* stop the teletext decoder, so the dvr device is free */
			stop_teletext_viewer();
			stop_teletext_decoder();                 
			fl_set_button(fdui -> main_form_teletext_button, 0);
			save_general_settings();
			clear_teletext_dir();

			/* stop the pids routine (in a thread) */
			getpids_stop();
			} /* end if recorder_mode is stop */

		/* 
		recorder is in stop mode, receiver is operational,
		no teletext or CA running.
		*/

		/* no more language */
		language[0] = 0;
	
		used_vpid = used_apid = used_tpid = 0;

		power = 1;
		program_number = pa -> pnr;

		have_pmt = 0;

		error_message_flag = 0;

		/* tune to this channel */
		set_channel(\
			pa -> frequency,\
			pa -> polarization,\
			pa -> disecq,\
			pa -> symbolrate,\
			pa -> vpid,\
			pa -> apid,\
			pa -> tpid,\
			pa -> ca,\
			power
			);

		/* modify selected_program */
		selected_program = pa -> browser_line;
		
		/*
		always read PAT then PMT to get video, audio and telext pid for
		this program number
		*/
		if( pa -> pnr != 0)
//		if(pa -> ca == 1)
			{
			pids_program_number = pa -> pnr;
			strcpy(report_string, "");
			
			/*
			pids routine will, if program number specified, read PAT,\
			then get PMT, and set vpid, apid, tpid
			*/
			getpids_start();
			}
		else
			{
			strcpy(report_string, "");
			pids_program_number = -1;
			}

		set_wait_indicator(0);

		/*
		maintain visible indication that line is selected with single click
		*/
		fl_select_browser_line(fdui -> main_form_browser, selected_program);

		/* set input fields */
		set_inputs(selected_program);

		fl_set_object_color(fdui -> main_form_seek_slider,\
		FL_WHITE, FL_RED);

		sprintf(temp, "%s/%s", pt -> directory, pt -> filename);
		if(record_filename) free(record_filename);
		record_filename = strsave(temp);
		if(! record_filename)
			{
			printf(\
			"timer_start_recording(): strsave record_filename failed\n");

			return 0;
			}

		/* allow ca routine to set pids */
		sleep(1);

		if(debug_flag)
			{
			printf("timer_start_recording():\n\
			record_filename=%s selected_program=%d\n\
			pa->vpid=%d pa->apid=%d\n",\
			record_filename, selected_program,\
			pa -> vpid, pa -> apid);
			}

		recorder_mode = RECORD_MODE;
		start_recorder();

		enable_receiver_controls(0);
		return 1;
		} /* end if entry found */

 	} /* end for all programs in list */

/* program not found  */
return 0;
} /* end function timer_start_recording */


int timer_stop_recording()
{
if(debug_flag)
	{
	printf("timer_stop_recording(): arg none\n");
	}

stop_button_routine();

enable_receiver_controls(1);

return 1;
} /* end function timer_stop_recording */


int clear_timer_form()
{
int total_lines, s_line;

if(debug_flag)
	{
	printf("clear_timer_form(): arg none\n");
	}

/* clear all inputs and deselect all browser lines */
selected_timer_pa = 0;

fl_set_input(fdui -> timer_form_directory_input_field, "");
fl_set_input(fdui -> timer_form_start_year_input_field, "");
fl_set_input(fdui -> timer_form_start_month_input_field, "");
fl_set_input(fdui -> timer_form_start_day_input_field, "");
fl_set_input(fdui -> timer_form_start_hour_input_field, "");
fl_set_input(fdui -> timer_form_start_minute_input_field, "");
fl_set_input(fdui -> timer_form_stop_year_input_field, "");
fl_set_input(fdui -> timer_form_stop_month_input_field, "");
fl_set_input(fdui -> timer_form_stop_day_input_field, "");
fl_set_input(fdui -> timer_form_stop_hour_input_field, "");
fl_set_input(fdui -> timer_form_stop_minute_input_field, "");
fl_set_input(fdui -> timer_form_length_input_field, "");

fl_set_object_label(fdui -> timer_form_filename_output_field, "");
fl_set_object_label(fdui -> timer_form_showview_output_field, "");

fl_set_button(fdui -> timer_form_daily_button, 0);
fl_set_button(fdui -> timer_form_weekly_button, 0);

total_lines = fl_get_browser_maxline(fdui -> timer_form_browser);
for(s_line = 1; s_line <= total_lines; s_line++)
    {
    fl_deselect_browser_line(fdui -> timer_form_browser, s_line);
    }/* end for all lines */

return 1;
} /* end function clear_timer_form */


int create_filename(char *program, struct timer *pa)
{
int c, i, j;
char temp[TEMP_SIZE];
char temp2[TEMP_SIZE];
int space_flag;
int minutes;

if(debug_flag)
	{
	printf("create_filename(): arg program=%s pa=%p\n",\
	program, pa);
	}

/* argument check */
if(! program) return 0;
if(! pa) return 0;

strcpy(temp2, program);
/* break at first ':' and replace spaces with '_' */
i = 0;
j = 0;
space_flag = 0;
while(1)
	{
	c = temp2[i];

	if(c == ':')
		{
		/* test for trailing '_' */
		if(space_flag)
			{
			temp[j - 1] = 0;

			break;
			}

		temp[j] = 0;
		break;
		}

	if(c == ' ')
		{
		if(space_flag)
			{
			i++;
			continue;
			}

		temp[j] = '_';
		space_flag = 1;
		}
	else
		{
		temp[j] = c; 
		space_flag = 0;
		}

	i++;
	j++;
	} /* end while all charactrs in temp */


minutes = (pa -> stop_time - pa -> start_time) / 60;
sprintf(temp2, "%s.%02dh%02d.%d-%d-%d-%dm.pes",\
temp,\
pa -> start_hour, pa -> start_minute,\
pa -> start_day, pa -> start_month, pa -> start_year,\
minutes);

if(pa -> filename) free(pa -> filename);
pa -> filename = strsave(temp2);
if(! pa -> filename)
	{
	printf("inputs_to_timer(): strsave filename failed\n");
	return 0;
	}

fl_set_object_label(fdui -> timer_form_filename_output_field,\
pa -> filename);

return 1;
} /* end function create_filename */


int update_time_and_filename(struct timer *pa)
{
struct tm *ptime;
int days;
time_t recording_length_in_seconds;
char weekday_str[80];
char month_str[80];
int monthday, hour, minute, second, year;

if(debug_flag)
	{
	printf("update_time_and_filename(): arg pa=%p\n", pa);
	}

/* get length recording, this should NOT change! */
recording_length_in_seconds = pa -> stop_time - pa -> start_time; 

/* create structure entry */
ptime = (struct tm *)malloc( sizeof(struct tm) );
if(! ptime)
    {       
    printf("update_time_and_filename(): malloc struct ptime failed\n");
    return 0;
    }

/* set increments */
if(pa -> status & TIMER_REPEAT_DAILY)
	{
	/* increment a day */
	pa -> start_day += 1;
	pa -> stop_day += 1;
	}
else if(pa -> status & TIMER_REPEAT_WORK_DAYS)
	{
	pa -> start_day += 1;
	pa -> stop_day += 1;	
	}
else if(pa -> status & TIMER_REPEAT_WEEKLY)
	{
	/* increment 7 days */
	pa -> start_day += 7;
	pa -> stop_day += 7;
	}
else
	{
	/* not implemented */
	return 0;
	}

/* for start */

while(1)
	{
	/* get days in this month */
	days = days_in_month[pa -> start_month];

	/* test for leap year, february has 29 days, month start at 0! */
	if( ( (pa -> start_year % 4) == 0) && (pa -> start_month == 1) )
		{
		days += 1;
		}

	/* test if past end month */
	if(pa -> start_day > days)
		{
		/* get number of days in next month */
		pa -> start_day -= days;

		/* increment the month */
		pa -> start_month++;

		/* test if past end year */
		if(pa -> start_month > 12)
			{
			pa -> start_month = 1;
			pa -> start_year++;
			}
		} /* end if past end current month */	

	/* calculate seconds since year 1900 */
	ptime -> tm_sec = 0;
	ptime -> tm_min = pa -> start_minute;
	ptime -> tm_hour = pa -> start_hour;
	ptime -> tm_mday = pa -> start_day; /* 1 - 31 !!! */
	ptime -> tm_mon = (pa -> start_month) - 1; /* 0 - 11 */
	ptime -> tm_year = (pa -> start_year) - 1900;
	ptime -> tm_isdst = -1; /* daylight savings time info n.a. */

	pa -> start_time = mktime( (struct tm *)ptime);

	/* calculated by mktime() */
	pa -> start_weekday = ptime -> tm_wday;

	if(pa -> status & TIMER_REPEAT_WORK_DAYS)
		{
		/* sunday */
		if(pa -> start_weekday == 0)
			{
			/* scip sunday */
			pa -> start_day += 1;
			continue;
			}

		/* saturday */
		if(pa -> start_weekday == 6)
			{
			/* scip saturday and sunday */
			pa -> start_day += 2; 
			continue;
			}	
		} /* end if work days timer */

	break;
	} /* end while skip days (in work day mode ) */

if(debug_flag)
	{
	printf("update_time_and_filename(): ptime -> tm_isdst=%d\n",\
	ptime -> tm_isdst);
	}

/* free structure */
free(ptime);

/* for stop */

/* calculate stop time from recording length */
pa -> stop_time = pa -> start_time + recording_length_in_seconds;

get_this_time(pa -> stop_time,\
	weekday_str, month_str, &monthday, &hour, &minute, &second, &year);
pa -> stop_year = year;
pa -> stop_month = month_to_digit(month_str);
pa -> stop_day = monthday;
pa -> stop_hour = hour;
pa -> stop_minute = minute;

/* create a new unique filename */
if(! create_filename(pa -> name, pa) ) return 0;

return 1;
} /* end function update_time_and_filename */


int set_weekday_output(struct timer *pa)
{
char temp[80];

if(debug_flag)
	{
	printf("set_weekday_output(): arg pa=%p\n", pa);
	}

/* argument check */
if(! pa) return 0;

if(pa -> status & TIMER_REPEAT_NO)
	{
	sprintf(temp, "%s", weekday_strings[pa -> start_weekday]);
	}
else if(pa -> status & TIMER_REPEAT_DAILY)
	{
	sprintf(temp, "Daily");
	}
else if(pa -> status & TIMER_REPEAT_WORK_DAYS)
	{
	sprintf(temp, "Workdays");
	}
else if(pa -> status & TIMER_REPEAT_WEEKLY)
	{
	sprintf(temp, "Weekly: %s", weekday_strings[pa -> start_weekday]);
	}

fl_set_object_label(fdui -> timer_form_start_weekday_output_field_text,\
temp);

return 1;
} /* end function set_weekday_output */


int set_timer_length_input(int length)
{
int seconds, minutes, hours;
char temp[80];

hours = length / 3600;
minutes = (length - (hours * 3600) ) / 60;
seconds = length - (hours * 3600) - (minutes * 60);

sprintf(temp, "%02d:%02d:%02d", hours, minutes, seconds);
fl_set_input(fdui -> timer_form_length_input_field, temp);

if(length <= 0)
	{
	fl_set_object_color(fdui -> timer_form_length_input_field,\
	FL_RED,FL_RED);
	}
else
	{
	fl_set_object_color(fdui -> timer_form_length_input_field,\
	FL_COL1, FL_COL1);
	}	


return 1;
} /* end function set_timer_length_input */

