/* 
test_switch.c - Test program for new API

Copyright (C) 2001 Ralph  Metzler <ralph@convergence.de>
                  & Marcus Metzler <marcus@convergence.de>
                      for convergence integrated media GmbH
Modified (c) 2002 by me
I read alt.tv.crypt 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public License
as published by the Free Software Foundation; either version 2.1
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "xcha.h"

/*
Examples from the documention included with driver 0.9.4:

In this section we would like to present some examples for using the
DVB API.

Tuning
We will start with a generic tuning subroutine that uses the frontend
and SEC, as well as the demux devices. The example is given for QPSK
tuners, but can easily be adjusted for QAM.

The program assumes that you are using a universal LNB and a standard
DiSEqC switch with up to 4 addresses. Of course, you could build in
some more checking if tuning was successful and maybe try to repeat
the tuning process. Depending on the external hardware, i.e. LNB and
DiSEqC switch, and weather conditions this may be necessary. 
*/


#include <sys/ioctl.h>
#include <stdio.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>

#include <ost/dmx.h>
#include <ost/frontend.h>
#include <ost/sec.h>
#include <sys/poll.h>

#define DMX "/dev/ost/demux"
#define QPSK "/dev/ost/qpskfe"
#define SEC "/dev/ost/sec"


#ifdef EXAMPLE_CODE
/* routine for checking if we have a signal */
int has_signal(int front)
{
feStatus stat;
int i;

if( front < 0 )
	{
	if( (front = open(QPSK, O_RDWR) ) < 0)
		{
		perror("FRONTEND DEVICE: ");
		return -1;
		}
	}

//FEReadStatus(front, &stat);

//while( !(stat & FE_HAS_SIGNAL) && count < tries)
for(i = 0; i < 10; i++)
	{
	if (ioctl(fd_frontend, FE_READ_STATUS, &stat) < 0)
		perror("has_signal stat");

	if(stat & FE_HAS_SIGNAL) break;
	}

if(stat & FE_HAS_SIGNAL) return 0;
else
	{
	printf("Tuning failed\n");
	return -1;
	}
} /* end function has_signal */
 

/*
tune qpsk
freq: frequency of transponder
vpid, apid, tpid: PIDs of video, audio and teletext TS packets
iseqc: DiSEqC address of the used LNB
pol: Polarisation
srate: Symbol Rate
fec. FEC
lnb_lof1: local frequency of lower LNB band
lnb_lof2: local frequency of upper LNB band
lnb_slof: switch frequency of LNB
*/

int set_qpsk_channel_0_9_4(int freq, int vpid, int apid, int tpid,
                int diseqc, int pol, int srate, int fec, int lnb_lof1, 
                int lnb_lof2, int lnb_slof)
{
        struct secCommand scmd;
        struct secCmdSequence scmds;
        struct dmxPesFilterParams pesFilterParams; 
        struct qpskParameters qpsk;
        struct pollfd pfd[1];
        struct qpskEvent event;
        int front, sec, demux1, demux2, demux3;

        /* Open all the necessary the devices */

        if((front = open(QPSK,O_RDWR)) < 0){
                perror("FRONTEND DEVICE: ");
                return -1;
        }

        if((sec = open(SEC,O_RDWR)) < 0){
                perror("SEC DEVICE: ");
                return -1;
        }

        if ((demux1 = open(DMX, O_RDWR|O_NONBLOCK))  < 0){
                perror("DEMUX DEVICE: ");
                return -1;
        }

        if ((demux2 = open(DMX, O_RDWR|O_NONBLOCK))  < 0){
                perror("DEMUX DEVICE: ");
                return -1;
        }

        if ((demux3 = open(DMX, O_RDWR|O_NONBLOCK))  < 0){
                perror("DEMUX DEVICE: ");
                return -1;
        }

      /* Set the frequency of the transponder, taking into account the
         local frequencies of the LNB */

        if (freq < lnb_slof) {
                qpsk.iFrequency = (freq - lnb_lof1);
                scmds.continuousTone = SEC_TONE_OFF;
        } else {
                qpsk.iFrequency = (freq - lnb_lof2);
                scmds.continuousTone = SEC_TONE_ON;
        }

      /* Set the polarity of the transponder by setting the correct
         voltage on the universal LNB */

        if (pol) scmds.voltage = SEC_VOLTAGE_18;
        else scmds.voltage = SEC_VOLTAGE_13;

      /* In case we have a DiSEqC, set it to the correct address */

        scmd.type=0;
        scmd.u.diseqc.addr=0x10;
        scmd.u.diseqc.cmd=0x38;
        scmd.u.diseqc.numParams=1;
        scmd.u.diseqc.params[0] = 0xF0 | ((diseqc * 4) & 0x0F) | 
                (scmds.continuousTone == SEC_TONE_ON ? 1 : 0) |
                (scmds.voltage==SEC_VOLTAGE_18 ? 2 : 0);

        scmds.miniCommand=SEC_MINI_NONE;
        scmds.numCommands=1;
        scmds.commands=&scmd;

      /* Send the data to the SEC device to prepare the LNB for tuning  */
        if (ioctl(sec, SEC_SEND_SEQUENCE, &scmds) < 0){
                perror("SEC SEND: ");
                return -1;
        }

      /* Set symbol rate and FEC */

        qpsk.SymbolRate = srate;
        qpsk.FEC_inner = fec;

      /* Now send it all to the frontend device */
        if (ioctl(front, QPSK_TUNE, &qpsk) < 0){
                perror("QPSK TUNE: ");
                return -1;
        }
 
      /* poll for QPSK event to check if tuning worked */
        pfd[0].fd = front;
        pfd[0].events = POLLIN;

        if (poll(pfd,1,3000)){
                if (pfd[0].revents & POLLIN){
                        printf("Getting QPSK event\n");
                        if ( ioctl(front, QPSK_GET_EVENT, &event)  

                             == -EBUFFEROVERFLOW){
                                perror("qpsk get event");
                                return -1;
                        }
                        printf("Received ");
                        switch(event.type){
                        case FE_UNEXPECTED_EV:
                                printf("unexpected event\n");
                                return -1;
                        case FE_FAILURE_EV:
                                printf("failure event\n");
                                return -1;
                       case FE_COMPLETION_EV:
                                printf("completion event\n");
                        }
                }
        }
      

      /* Set the filters for video, audio and teletext demuxing */

        pesFilterParams.pid     = vpid;
        pesFilterParams.input   = DMX_IN_FRONTEND; 
        pesFilterParams.output  = DMX_OUT_DECODER; 
        pesFilterParams.pesType = DMX_PES_VIDEO; 
        pesFilterParams.flags   = DMX_IMMEDIATE_START;
        if (ioctl(demux1, DMX_SET_PES_FILTER, &pesFilterParams) < 0){
                perror("set_vpid");
                return -1;
        }

        pesFilterParams.pid     = apid;
        pesFilterParams.input   = DMX_IN_FRONTEND; 
        pesFilterParams.output  = DMX_OUT_DECODER; 
        pesFilterParams.pesType = DMX_PES_AUDIO; 
        pesFilterParams.flags   = DMX_IMMEDIATE_START;
        if (ioctl(demux2, DMX_SET_PES_FILTER, &pesFilterParams) < 0){
                perror("set_apid");
                return -1;
        }

        pesFilterParams.pid     = tpid;
        pesFilterParams.input   = DMX_IN_FRONTEND; 
        pesFilterParams.output  = DMX_OUT_DECODER; 
        pesFilterParams.pesType = DMX_PES_TELETEXT; 
        pesFilterParams.flags   = DMX_IMMEDIATE_START;
        if (ioctl(demux3, DMX_SET_PES_FILTER, &pesFilterParams) < 0){
                perror("set_tpid");
                return -1;
        }

        /* check if we have a signal */
        return has_signal(front);
} /* end function set_qpsk_channel */


int set_qpsk_channel(int freq, int vpid, int apid, int tpid,
	int diseqc, int pol, int srate, int fec, int lnb_lof1, 
	int lnb_lof2, int lnb_slof)
{
struct secCommand scmd;
struct secCmdSequence scmds;
struct dmxPesFilterParams pesFilterParams; 
struct qpskParameters qpsk;

struct pollfd pfd[1];
struct qpskEvent event;

int front, sec, demux1, demux2, demux3;

/* Open all the necessary the devices */

if( (front = open(QPSK, O_RDWR) ) < 0)
	{
	perror("FRONTEND DEVICE: ");
	return -1;
	}

if( (sec = open(SEC, O_RDWR) ) < 0)
	{
	perror("SEC DEVICE: ");
	return -1;
	}	

if( (demux1 = open(DMX, O_RDWR | O_NONBLOCK) ) < 0)
	{
	perror("DEMUX DEVICE: ");
	return -1;
	}

if( (demux2 = open(DMX, O_RDWR | O_NONBLOCK) ) < 0)
	{
	perror("DEMUX DEVICE: ");
	return -1;
	}

if( (demux3 = open(DMX, O_RDWR | O_NONBLOCK) ) < 0)
	{
	perror("DEMUX DEVICE: ");
	return -1;
	}

/* Set the frequency of the transponder, taking into account the
local frequencies of the LNB */

if(freq < lnb_slof)
	{
	qpsk.iFrequency = (freq - lnb_lof1);
	scmds.continuousTone = SEC_TONE_OFF;
	}
else
	{
	qpsk.iFrequency = (freq - lnb_lof2);
	scmds.continuousTone = SEC_TONE_ON;
	}

/* 
Set the polarity of the transponder by setting the correct voltage on
the universal LNB
*/

if(pol) scmds.voltage = SEC_VOLTAGE_18;
else scmds.voltage = SEC_VOLTAGE_13;

/* In case we have a DiSEqC, set it to the correct address */

scmd.type = 0;
scmd.u.diseqc.addr = 0x10;
scmd.u.diseqc.cmd = 0x38;
scmd.u.diseqc.numParams = 1;
scmd.u.diseqc.params[0] = 0xF0 | ((diseqc * 4) & 0x0F) |\
(scmds.continuousTone == SEC_TONE_ON ? 1 : 0) |\
(scmds.voltage == SEC_VOLTAGE_18 ? 2 : 0);

scmds.miniCommand = SEC_MINI_NONE;
scmds.numCommands = 1;
scmds.commands = &scmd;

/* Send the data to the SEC device to prepare the LNB for tuning */
if( ioctl(sec, SEC_SEND_SEQUENCE, &scmds) < 0)
	{
	perror("SEC SEND: ");
	return -1;
	}

/* Set symbol rate and FEC */

qpsk.SymbolRate = srate;
qpsk.FEC_inner = fec;

/* Now send it all to the frontend device */
if( ioctl(front, QPSK_TUNE, &qpsk) < 0)
	{
	perror("QPSK TUNE: ");
	return -1;
	}

/* poll for QPSK event to check if tuning worked */
pfd[0].fd = front;
pfd[0].events = POLLIN;

if( poll(pfd, 1, 3000) )
	{
	if(pfd[0].revents & POLLIN)
		{
		printf("Getting QPSK event\n");
		if( ioctl(front, QPSK_GET_EVENT, &event) == -EBUFFEROVERFLOW)
			{
			perror("qpsk get event");
			return -1;
			}
		printf("Received ");

		switch(event.type)
			{
			case FE_UNEXPECTED_EV:
				printf("unexpected event\n");
				return -1;
			case FE_FAILURE_EV:
				printf("failure event\n");
				return -1;
			case FE_COMPLETION_EV:
				printf("completion event\n");
			} /* end switch */
		} /* end if revents */
	} /* end if poll */

/* Set the filters for video, audio and teletext demuxing */

pesFilterParams.pid = vpid;
pesFilterParams.input = DMX_IN_FRONTEND; 
pesFilterParams.output = DMX_OUT_DECODER; 
pesFilterParams.pesType = DMX_PES_VIDEO; 
pesFilterParams.flags = DMX_IMMEDIATE_START;
if( ioctl(demux1, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
	{
	perror("set_vpid");
	return -1;
	}

pesFilterParams.pid = apid;
pesFilterParams.input = DMX_IN_FRONTEND; 
pesFilterParams.output = DMX_OUT_DECODER; 
pesFilterParams.pesType = DMX_PES_AUDIO; 
pesFilterParams.flags = DMX_IMMEDIATE_START;
if( ioctl(demux2, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
	{
	perror("set_apid");
	return -1;
	}

pesFilterParams.pid = tpid;
pesFilterParams.input = DMX_IN_FRONTEND; 
pesFilterParams.output = DMX_OUT_DECODER; 
pesFilterParams.pesType = DMX_PES_TELETEXT; 
pesFilterParams.flags = DMX_IMMEDIATE_START;
if( ioctl(demux3, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
	{
	perror("set_tpid");
	return -1;
	}

/* check if we have a signal */
return has_signal(front);
} /* end function set_qpsk_channel */

#endif /* EXAMPE_CODE */


#define checkreturn(f); if( ret!=0 ) { printf(f); return ret; }

int fd_sec;
int fd_demuxa;
int fd_demuxv;
int fd_demuxtt;

struct secCommand scmd;
struct secCmdSequence scmds;
struct dmxPesFilterParams pesFilterParamsV; 
struct dmxPesFilterParams pesFilterParamsA; 
struct dmxPesFilterParams pesFilterParamsTT; 
struct qpskParameters qpsk;
int front_type;


set_front(void) 
{
feStatus stat = 0;
int i, freq;
uint32_t soff;
	  
if(debug_flag)
	{
	printf("set_front(): arg none\n");
	}

scmds.miniCommand = SEC_MINI_NONE;
scmds.numCommands = 0; /* WAS 1 for bursts */
scmds.commands = &scmd;
	
soff = qpsk.SymbolRate / 16000;

if (ioctl(fd_sec, SEC_SEND_SEQUENCE, &scmds) < 0)
	perror("setfront sec");

usleep(100000);
		
freq = qpsk.iFrequency;
		
for(i = 2; i < 10; i++)
	{    
	if(debug_flag)
		{
		printf("set_front(): adjacent_flag=%d soff=%lu qpsk.iFrequency=%lu\n",\
		adjacent_flag, soff, qpsk.iFrequency);
		}

//	stat = tune_it(qpsk);
	stat = tune_it();
	if (stat & FE_HAS_SIGNAL)
		{
		if(debug_flag)
			{
			printf("set_front(): stat=%d got signal\n", stat);
			}
		have_signal_flag = 1;

		break;
		}

	if(! adjacent_flag) break;

	if(i & 1) 
		{
		qpsk.iFrequency = freq + soff * (i / 2);
		}
	else
		{
		qpsk.iFrequency = freq - soff * (i / 2);
		}		

	} /* end for */

} /* end function set_front */
	        

set_diseqc_nb(int nr) 
{
scmd.type=0;
scmd.u.diseqc.addr = 0x10;
scmd.u.diseqc.cmd = 0x38;
scmd.u.diseqc.numParams = 1;
scmd.u.diseqc.params[0] = 0xF0 | ((nr * 4) & 0x0F) | 
  (scmds.continuousTone == SEC_TONE_ON ? 1 : 0) |
  (scmds.voltage==SEC_VOLTAGE_18 ? 2 : 0);
} /* end function set_diseqc_nb */


set_tpid(ushort ttpid) 
{  
if (ttpid==0 || ttpid==0xffff || ttpid==0x1fff)
	{
	ioctl(fd_demuxtt, DMX_STOP, 0);
    return;
	}

pesFilterParamsTT.pid     = ttpid;
pesFilterParamsTT.input   = DMX_IN_FRONTEND; 

//pesFilterParamsTT.output  = DMX_OUT_DECODER; 
pesFilterParamsTT.output = DMX_OUT_TS_TAP;
//pesFilterParamsTT.output = DMX_OUT_TAP;

pesFilterParamsTT.pesType = DMX_PES_TELETEXT; 
pesFilterParamsTT.flags   = DMX_IMMEDIATE_START;

if (ioctl(fd_demuxtt, DMX_SET_PES_FILTER, &pesFilterParamsTT) < 0)
	{
	printf("PID=%04x\n", ttpid);
	perror("set_tpid");
	}

used_tpid = ttpid;
} /* end function set_tpid */


set_vpid(ushort vpid) 
{  
if(vpid == 0 || vpid == 0xffff || vpid == 0x1fff)
	{
	ioctl(fd_demuxv, DMX_STOP, 0);
	return;
	}

pesFilterParamsV.pid     = vpid;
pesFilterParamsV.input   = DMX_IN_FRONTEND; 

//pesFilterParamsV.output  = DMX_OUT_DECODER; 
pesFilterParamsV.output = DMX_OUT_TS_TAP;
//pesFilterParamsV.output = DMX_OUT_TAP;

pesFilterParamsV.pesType = DMX_PES_VIDEO; 
pesFilterParamsV.flags   = DMX_IMMEDIATE_START;
if (ioctl(fd_demuxv, DMX_SET_PES_FILTER, &pesFilterParamsV) < 0)
	perror("set_vpid");

used_vpid = vpid;
} /* end function set_vpid */


set_apid(ushort apid) 
{  
if(apid == 0 || apid == 0xffff || apid == 0x1fff)
	{
	ioctl(fd_demuxa, DMX_STOP, apid);
	return;
	}

pesFilterParamsA.pid = apid;
pesFilterParamsA.input = DMX_IN_FRONTEND; 

//pesFilterParamsA.output = DMX_OUT_DECODER; 
pesFilterParamsA.output = DMX_OUT_TS_TAP;
//pesFilterParamsA.output = DMX_OUT_TAP;

pesFilterParamsA.pesType = DMX_PES_AUDIO; 
pesFilterParamsA.flags = DMX_IMMEDIATE_START;

if (ioctl(fd_demuxa, DMX_SET_PES_FILTER, &pesFilterParamsA) < 0)
	perror("set_apid");
used_apid = apid;
} /* end function set_apid */


//tune_it(struct qpskParameters qpsk)
tune_it()
{
feStatus stat = 0;
int res;
struct qpskEvent event;
int count = 0;
int tries;

if(debug_flag)
	{
	printf("tune_it(): arg using struct qpsk=%p\n", qpsk);
	}

//printf("qpsk.SymbolRate=%d qpsk.FEC_inner=%d qpsk.iFrequency=%d\n",\
//qpsk.SymbolRate, qpsk.FEC_inner, qpsk.iFrequency);

//printf("fd_frontend=%d qpsk=%p\n", fd_frontend, qpsk);

if (ioctl(fd_frontend, QPSK_TUNE, &qpsk) < 0)
	perror("setfront front");

if ( (res = ioctl(fd_frontend, QPSK_GET_EVENT, &event)) == -EBUFFEROVERFLOW)

	res = ioctl(fd_frontend, QPSK_GET_EVENT, &event);
if (res < 0)
	perror("qpsk get event");
	
if(adjacent_flag) tries = 10;
else tries = 1;

while( !(stat & FE_HAS_SIGNAL) && count < tries)
	{
	if (ioctl(fd_frontend, FE_READ_STATUS, &stat) < 0)
		perror("set_tp stat");

	count++;
	}

return stat;
} /* end function tune_it */


set_tp(uint *freq, int ttk, int pol, uint srate, int dis) 
{
if(debug_flag)
	{
	printf("set_tp(): arg *freq=%u ttk=%d pol=%d srate=%d dis=%d\n",\
	*freq, ttk, pol, srate, dis);
	}

if (*freq < 11700000)
	{
	qpsk.iFrequency = *freq - 9750000;
	scmds.continuousTone = SEC_TONE_OFF;
	}
else
	{
	qpsk.iFrequency = *freq - 10600000;
	scmds.continuousTone = SEC_TONE_ON;
	}

if (pol) scmds.voltage = SEC_VOLTAGE_18;
else scmds.voltage = SEC_VOLTAGE_13;

//set_diseqc_nb(dis);

qpsk.SymbolRate = srate;
qpsk.FEC_inner = 0;
} /* end function set_tp */


get_front(void) 
{
set_vpid(0);
set_apid(0);
//set_tpid(0);

scmds.voltage = SEC_VOLTAGE_13; 
scmds.miniCommand = SEC_MINI_NONE;
scmds.continuousTone = SEC_TONE_OFF;
scmds.numCommands = 1;
scmds.commands =& scmd;
	
qpsk.iFrequency = 12666000 - 10600000;
qpsk.SymbolRate = 22000000;
qpsk.FEC_inner = 0;
} /* end function get_front */
  

int init_tuning()
{
//fd_demuxtt = open("/dev/ost/demux", O_RDWR | O_NONBLOCK);
//if(! fd_demuxtt) return 0;

fd_frontend = open("/dev/ost/qpskfe", O_RDWR);
if(! fd_frontend) return 0;

fd_sec = open("/dev/ost/sec", O_RDWR);
if(! fd_sec) return 0;

fd_demuxa = open("/dev/ost/demux", O_RDWR | O_NONBLOCK);
if(! fd_demuxa) return 0;

fd_demuxv = open("/dev/ost/demux", O_RDWR | O_NONBLOCK);
if(! fd_demuxv) return 0;

receiver_active_flag = 1;

return 1;
} /* end function init_tuning */


set_ost_power_off()
{
scmds.voltage = SEC_VOLTAGE_OFF;
set_front();
} /* end function set_ost_power_off */


set_pol(int polarity)
{
if (polarity) scmds.voltage = SEC_VOLTAGE_18;
else scmds.voltage = SEC_VOLTAGE_13;
set_front();
} /* end function set_pol */


void set_filt(int fd, unsigned short tt_pid, dmxPesType_t t)
{
size_t bytesRead;
struct dmxPesFilterParams pesFilterParams;

pesFilterParams.pid     = tt_pid;
pesFilterParams.input   = DMX_IN_FRONTEND;
pesFilterParams.output  = DMX_OUT_TAP;
pesFilterParams.pesType = t;
pesFilterParams.flags   = DMX_IMMEDIATE_START;

if(ioctl(fd, DMX_SET_PES_FILTER, &pesFilterParams) < 0)  
	perror("DMX SET PES FILTER:");
} /* end function set_filt */


int set_channel(int freqMHz, char pol, int diseqc, uint srate,\
int vpid, int apid, int tpid, int ca, int power)
{
int polarity;
unsigned int frequency;
unsigned int symbolrate;
int high_band_flag;

if(debug_flag)
	{
	fprintf(stdout,\
	"set_channel(): arg\n\
	FreqMHz=%d Pol=%c diseqc=%d drate=%d\n\
	vpid=%d apid=%d tpid=%d ca=%d power=%d\n",\
	freqMHz, pol, diseqc, srate,\
	vpid, apid, tpid, ca, power);
	}

/* select band */
if(freqMHz < 11700UL) high_band_flag = 0;
else high_band_flag = 1;

if(pol == 'h') polarity = 1;
else polarity = 0;

symbolrate = (unsigned int) srate * 1000;
frequency = (unsigned int) freqMHz * 1000;

get_front();
set_vpid( (unsigned short) vpid);
set_apid( (unsigned short) apid);
//set_tpid( (unsigned short) tpid);

set_tp(&frequency, 1, polarity, symbolrate, diseqc);

#ifdef STRANGE_LNB
if(high_band_flag)
	{
	set_front();
	return 1;
	}
else /* my defective LNB kludge, cycle LNB power off / on */
	{
	set_ost_power_off();
	set_pol(polarity);

	return 1;
	}
#else
	set_front();
#endif /* STRANGE_LNB */

return 1;
} /* end function set_channel */


int close_all_devices()
{
if(debug_flag)
	{
	printf("close_all_devices(): arg none\n");
	}

receiver_active_flag = 0;

close(fd_frontend);
close(fd_sec);
close(fd_demuxv);
close(fd_demuxa);

return 1;
} /* end function close all devices */

