/*
 *
 * Copyright (C) 2000, 2001 Marcus Metzler 
 *            for convergence integrated media GmbH
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 * 

 * The author can be reached at marcus@convergence.de, 

 * the project's page is at http://linuxtv.org/dvb/
 */

#include "dvb_formats.h"

void get_pespts(uint8_t *av_pts,uint8_t *pts)
{
	
	pts[0] = 0x21 | 
		((av_pts[0] & 0xC0) >>5);
	pts[1] = ((av_pts[0] & 0x3F) << 2) |
		((av_pts[1] & 0xC0) >> 6);
	pts[2] = 0x01 | ((av_pts[1] & 0x3F) << 2) |
		((av_pts[2] & 0x80) >> 6);
	pts[3] = ((av_pts[2] & 0x7F) << 1) |
		((av_pts[3] & 0x80) >> 7);
	pts[4] = 0x01 | ((av_pts[3] & 0x7F) << 1);
}


uint16_t get_pid(uint8_t *pid)
{
	uint16_t pp;

	pp = (pid[0] & PID_MASK_HI) << 8;
	pp |= pid[1] &0xFF;
	return pp;
}




void reset_ipack(ipack *p)
{
	p->found = 0;
	p->cid = 0;
	p->plength = 0;
	p->flag1 = 0;
	p->flag2 = 0;
	p->hlength = 0;
	p->mpeg = 0;
	p->check = 0;
	p->which = 0;
	p->done = 0;
	p->count = 0;
}

void init_ipack(ipack *p, int size,
		void (*func)(uint8_t *buf,  int size, void *priv))
{
	if ( !(p->buf = malloc(size)) ){
		printf("Couldn't allocate memroy for ipack\n");
		exit(1);
	}
	p->size = size;
	p->func = func;
	reset_ipack(p);
}


void free_ipack(ipack * p)
{
	if (p->buf) free(p->buf);
}


void send_ipack(ipack *p)
{
	if (p->count < 10) return;
	p->buf[3] = p->cid;
	p->buf[4] = (uint8_t)(((p->count-6) & 0xFF00) >> 8);
	p->buf[5] = (uint8_t)((p->count-6) & 0x00FF);
	p->func(p->buf, p->count, p->data);
	
	switch ( p->mpeg ){
	case 2:		
		p->buf[6] = 0x80;
		p->buf[7] = 0x00;
		p->buf[8] = 0x00;
		p->count = 9;
		break;
	case 1:
		p->buf[6] = 0x0F;
		p->count = 7;
		break;
	}
}


void write_ipack(ipack *p, uint8_t *data, int count)
{
	uint8_t headr[3] = { 0x00, 0x00, 0x01} ;

	if (p->count < 6){
		memcpy(p->buf, headr, 3);
		p->count = 6;
	}

	if (p->count + count < p->size){
		memcpy(p->buf+p->count, data, count);
		p->count += count;
	} else {
		int rest = p->size - p->count;
		memcpy(p->buf+p->count, data, rest);
		p->count += rest;
		send_ipack(p);
		if (count - rest > 0)
			write_ipack(p, data+rest, count-rest);
 	}
}

void instant_repack (uint8_t *buf, int count, ipack *p)
{

	int l;
	unsigned short *pl;
	int c=0;

	while (c < count && (p->mpeg == 0 ||
			     (p->mpeg == 1 && p->found < 7) ||
			     (p->mpeg == 2 && p->found < 9))
	       &&  (p->found < 5 || !p->done)){
		switch ( p->found ){
		case 0:
		case 1:
			if (buf[c] == 0x00) p->found++;
			else p->found = 0;
			c++;
			break;
		case 2:
			if (buf[c] == 0x01) p->found++;
			else if (buf[c] == 0){
				p->found = 2;
			} else p->found = 0;
			c++;
			break;
		case 3:
			p->cid = 0;
			switch (buf[c]){
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
				p->done = 1;
			case PRIVATE_STREAM1:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
				p->found++;
				p->cid = buf[c];
				c++;
				break;
			default:
				p->found = 0;
				break;
			}
			break;
			

		case 4:
			if (count-c > 1){
				pl = (unsigned short *) (buf+c);
				p->plength =  ntohs(*pl);
				p->plen[0] = buf[c];
				c++;
				p->plen[1] = buf[c];
				c++;
				p->found+=2;
			} else {
				p->plen[0] = buf[c];
				p->found++;
				return;
			}
			break;
		case 5:
			p->plen[1] = buf[c];
			c++;
			pl = (unsigned short *) p->plen;
			p->plength = ntohs(*pl);
			p->found++;
			break;


		case 6:
			if (!p->done){
				p->flag1 = buf[c];
				c++;
				p->found++;
				if ( (p->flag1 & 0xC0) == 0x80 ) p->mpeg = 2;
				else {
					p->hlength = 0;
					p->which = 0;
					p->mpeg = 1;
					p->flag2 = 0;
				}
			}
			break;

		case 7:
			if ( !p->done && p->mpeg == 2){
				p->flag2 = buf[c];
				c++;
				p->found++;
			}	
			break;

		case 8:
			if ( !p->done && p->mpeg == 2){
				p->hlength = buf[c];
				c++;
				p->found++;
			}
			break;
			
		default:

			break;
		}
	}

	if (c == count) return;

	if (!p->plength) p->plength = MMAX_PLENGTH-6;


	if ( p->done || ((p->mpeg == 2 && p->found >= 9)  || 
	     (p->mpeg == 1 && p->found >= 7)) ){
		switch (p->cid){
			
		case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		case PRIVATE_STREAM1:
			
			if (p->mpeg == 2 && p->found == 9){
				write_ipack(p, &p->flag1, 1);
				write_ipack(p, &p->flag2, 1);
				write_ipack(p, &p->hlength, 1);
			}

			if (p->mpeg == 1 && p->found == 7){
				write_ipack(p, &p->flag1, 1);
			}


			if (p->mpeg == 2 && (p->flag2 & PTS_ONLY) &&  
			    p->found < 14){
				while (c < count && p->found < 14){
					p->pts[p->found-9] = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
				}
				if (c == count) return;
			}

			if (p->mpeg == 1 && p->which < 2000){

				if (p->found == 7) {
					p->check = p->flag1;
					p->hlength = 1;
				}

				while (!p->which && c < count && 
				       p->check == 0xFF){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
				}

				if ( c == count) return;
				
				if ( (p->check & 0xC0) == 0x40 && !p->which){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;

					p->which = 1;
					if ( c == count) return;
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}

				if (p->which == 1){
					p->check = buf[c];
					write_ipack(p, buf+c, 1);
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}
				
				if ( (p->check & 0x30) && p->check != 0xFF){
					p->flag2 = (p->check & 0xF0) << 2;
					p->pts[0] = p->check;
					p->which = 3;
				} 

				if ( c == count) return;
				if (p->which > 2){
					if ((p->flag2 & PTS_DTS_FLAGS)
					    == PTS_ONLY){
						while (c < count && 
						       p->which < 7){
							p->pts[p->which-2] =
								buf[c];
							write_ipack(p,buf+c,1);
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					} else if ((p->flag2 & PTS_DTS_FLAGS) 
						   == PTS_DTS){
						while (c < count && 
						       p->which< 12){
							if (p->which< 7)
								p->pts[p->which
								      -2] =
									buf[c];
							write_ipack(p,buf+c,1);
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					}
					p->which = 2000;
				}
							
			}

			while (c < count && p->found < p->plength+6){
				l = count -c;
				if (l+p->found > p->plength+6)
					l = p->plength+6-p->found;
				write_ipack(p, buf+c, l);
				p->found += l;
				c += l;
			}	
		
			break;
		}


		if ( p->done ){
			if( p->found + count - c < p->plength+6){
				p->found += count-c;
				c = count;
			} else {
				c += p->plength+6 - p->found;
				p->found = p->plength+6;
			}
		}

		if (p->plength && p->found == p->plength+6) {
			send_ipack(p);
			reset_ipack(p);
			if (c < count)
				instant_repack(buf+c, count-c, p);
		}
	}
	return;
}


int write_ts_header(uint16_t pid, uint8_t *counter, int pes_start, 
		    uint8_t *buf, uint8_t length)
{
	int i;
	int c = 0;
	int fill;
	uint8_t tshead[4] = { 0x47, 0x00, 0x00, 0x10}; 
        

	fill = TS_SIZE-4-length;
        if (pes_start) tshead[1] = 0x40;
	if (fill) tshead[3] = 0x30;
        tshead[1] |= (uint8_t)((pid & 0x1F00) >> 8);
        tshead[2] |= (uint8_t)(pid & 0x00FF);
        tshead[3] |= ((*counter)++ & 0x0F) ;
        memcpy(buf,tshead,4);
	c+=4;


	if (fill){
		buf[4] = fill-1;
		c++;
		if (fill >1){
			buf[5] = 0x00;
			c++;
		}
		for ( i = 6; i < fill+4; i++){
			buf[i] = 0xFF;
			c++;
		}
	}

        return c;
}


int write_pes_header(uint8_t id,int length , long PTS, uint8_t *obuf, 
		     int stuffing)
{
	uint8_t le[2];
	uint8_t dummy[3];
	uint8_t *pts;
	uint8_t ppts[5];
	long lpts;
	int c;
	uint8_t headr[3] = {0x00, 0x00, 0x01};
	
	lpts = htonl(PTS);
	pts = (uint8_t *) &lpts;
	
	get_pespts(pts,ppts);

	c = 0;
	memcpy(obuf+c,headr,3);
	c += 3;
	memcpy(obuf+c,&id,1);
	c++;

	le[0] = 0;
	le[1] = 0;
	length -= 6+stuffing;

	le[0] |= ((uint8_t)(length >> 8) & 0xFF); 
	le[1] |= ((uint8_t)(length) & 0xFF); 
	memcpy(obuf+c,le,2);
	c += 2;

	if (id == PADDING_STREAM){
		memset(obuf+c,0xff,length);
		c+= length;
		return c;
	}

	dummy[0] = 0x80;
	dummy[1] = 0;
	dummy[2] = 0;
	if (PTS){
		dummy[1] |= PTS_ONLY;
		dummy[2] = 5+stuffing;
	}
	memcpy(obuf+c,dummy,3);
	c += 3;
	memset(obuf+c,0xFF,stuffing);

	if (PTS){
		memcpy(obuf+c,ppts,5);
		c += 5;
	}
	
	return c;
}


void init_p2p(p2p *p, void (*func)(uint8_t *buf, int count, void *data), 
	      int repack){
	p->found = 0;
	p->cid = 0;
	p->mpeg = 0;
	p->done = 0;
	p->func = func;
	p->bigend_repack = 0;
	p->repack = 0; 
	if ( repack < MAX_PLENGTH && repack > 265 ){
		p->repack = repack-6;
		p->bigend_repack = (uint16_t)htons((short)((repack-6) & 0xFFFF));
	} 
	
	if( !(p->buf = (char *) malloc(sizeof(char)*MMAX_PLENGTH))){
		fprintf(stderr,"Not enough memory for ts_transform\n");
	}
	if( repack && !(p->buf2 = (char *) malloc(sizeof(char)*p->repack))){
		fprintf(stderr,"Not enough memory for ts_transform\n");
	}

	
	memset(p->buf,0,MMAX_PLENGTH);

}

void free_p2p(p2p *p)
{
	free(p->buf);
	free(p->buf2);
	p->buf = NULL;
	p->buf2 = NULL;
}


void pes_repack(p2p *p)
{
	int count = 0;
	int repack = p->repack;
	int rest = p->plength;
	int bfill = 0;
	int diff;
	uint16_t length;
	uint8_t *buf = p->buf2;
	

	if (rest <= 0) {
		//fprintf(stderr,"Error in repack\n");
		return;
	}
	if (!repack){
		fprintf(stderr,"forgot to set repack size\n");
		return;
	}
	if (p->plength == repack){
		memcpy(p->buf+4,(char *)&p->bigend_repack,2);
		p->func(p->buf, repack+6, p);
		return;
	}
	
	buf[0] = 0x00;
	buf[1] = 0x00;
	buf[2] = 0x01;
	buf[3] = p->cid;
	memcpy(buf+4,(char *)&p->bigend_repack,2);
	memset(buf+6,0, repack-6);

	if (p->mpeg == 2){

		if ( rest > repack){
			memcpy(p->buf+4,(char *)&p->bigend_repack,2);
			p->func(p->buf, repack+6, p->data);
			count += repack+6;
			rest -= repack;
		} else {
			memcpy(buf,p->buf,9+p->hlength);
			bfill = p->hlength;
			count += 9+p->hlength;
			rest -= p->hlength+3;
		}

		while (rest >= repack-3){
			memset(buf+6,0,repack-6);
			buf[6] = 0x80;
			buf[7] = 0x00;
			buf[8] = 0x00;
			memcpy(buf+9,p->buf+count,repack-3);
			rest -= repack-3;
			count += repack-3;
			p->func(buf, repack+6, p->data);
		}
		
		if (rest){
			diff = (repack - 3) - rest - bfill;
			if (!bfill){
				buf[6] = 0x80;
				buf[7] = 0x00;
				buf[8] = 0x00;
			}

			if ( diff < PES_MIN){
				length = repack; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				buf[8] = (uint8_t)(bfill+diff);
				memset(buf+9+bfill,0xFF,diff);
				memcpy(buf+9+bfill+diff,p->buf+count,rest);
			} else {
				length = rest+ bfill+3; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				memcpy(buf+9+bfill,p->buf+count,rest);
				bfill += rest+9;
				write_pes_header( PADDING_STREAM, diff, 0,
						  buf+bfill, 0);
			}
			p->func(buf, repack+6, p->data);
		}
	}	

	if (p->mpeg == 1){

		if ( rest > repack){
			memcpy(p->buf+4,(char *)&p->bigend_repack,2);
			p->func(p->buf, repack+6, p->data);
			count += repack+6;
			rest -= repack;
		} else {
			memcpy(buf,p->buf,6+p->hlength);
			bfill = p->hlength;
			count += 6;
			rest -= p->hlength;
		}

		while (rest >= repack-1){
			memset(buf+6,0,MAX_PLENGTH-6);
			buf[6] = 0x0F;
			memcpy(buf+7,p->buf+count,repack-1);
			rest -= repack-1;
			count += repack-1;
			p->func(buf, repack+6, p->data);
		}
		
		if (rest){
			diff = repack - 1 - rest - bfill;

			if ( diff < PES_MIN){
				length = repack; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				memset(buf+6,0xFF,diff);
				if (!bfill){
					buf[6+diff] = 0x0F;
				}
				memcpy(buf+7+diff,p->buf+count,rest+bfill);
			} else {
				length = rest+ bfill+1; 
				buf[4] = (uint8_t)((length & 0xFF00) >> 8);
				buf[5] = (uint8_t)(length & 0x00FF);
				if (!bfill){
					buf[6] = 0x0F;
					memcpy(buf+7,p->buf+count,rest);
					bfill = rest+7;
				} else {
					memcpy(buf+6,p->buf+count,rest+bfill);
					bfill += rest+6;
				}
				write_pes_header( PADDING_STREAM, diff, 0,
						  buf+bfill, 0);
			}
			p->func(buf, repack+6, p->data);
		}
	}	
}


void get_pes (uint8_t *buf, int count, p2p *p, void (*func)(p2p *p))
{

	int l;
	unsigned short *pl;
	int c=0;

	uint8_t headr[3] = { 0x00, 0x00, 0x01} ;


	while (c < count && (p->mpeg == 0 ||
			     (p->mpeg == 1 && p->found < 7) ||
			     (p->mpeg == 2 && p->found < 9))
	       &&  (p->found < 5 || !p->done)){
		switch ( p->found ){
		case 0:
		case 1:
			if (buf[c] == 0x00) p->found++;
			else p->found = 0;
			c++;
			break;
		case 2:
			if (buf[c] == 0x01) p->found++;
			else if (buf[c] == 0){
				p->found = 2;
			} else p->found = 0;
			c++;
			break;
		case 3:
			p->cid = 0;
			switch (buf[c]){
			case PROG_STREAM_MAP:
			case PRIVATE_STREAM2:
			case PROG_STREAM_DIR:
			case ECM_STREAM     :
			case EMM_STREAM     :
			case PADDING_STREAM :
			case DSM_CC_STREAM  :
			case ISO13522_STREAM:
				p->done = 1;
			case PRIVATE_STREAM1:
			case VIDEO_STREAM_S ... VIDEO_STREAM_E:
			case AUDIO_STREAM_S ... AUDIO_STREAM_E:
				p->found++;
				p->cid = buf[c];
				c++;
				break;
			default:
				p->found = 0;
				break;
			}
			break;
			

		case 4:
			if (count-c > 1){
				pl = (unsigned short *) (buf+c);
				p->plength =  ntohs(*pl);
				p->plen[0] = buf[c];
				c++;
				p->plen[1] = buf[c];
				c++;
				p->found+=2;
			} else {
				p->plen[0] = buf[c];
				p->found++;
				return;
			}
			break;
		case 5:
			p->plen[1] = buf[c];
			c++;
			pl = (unsigned short *) p->plen;
			p->plength = ntohs(*pl);
			p->found++;
			break;


		case 6:
			if (!p->done){
				p->flag1 = buf[c];
				c++;
				p->found++;
				if ( (p->flag1 & 0xC0) == 0x80 ) p->mpeg = 2;
				else {
					p->hlength = 0;
					p->which = 0;
					p->mpeg = 1;
					p->flag2 = 0;
				}
			}
			break;

		case 7:
			if ( !p->done && p->mpeg == 2){
				p->flag2 = buf[c];
				c++;
				p->found++;
			}	
			break;

		case 8:
			if ( !p->done && p->mpeg == 2){
				p->hlength = buf[c];
				c++;
				p->found++;
			}
			break;
			
		default:

			break;
		}
	}

	if (!p->plength) p->plength = MMAX_PLENGTH-6;


	if ( p->done || ((p->mpeg == 2 && p->found >= 9)  || 
	     (p->mpeg == 1 && p->found >= 7)) ){
		switch (p->cid){
			
		case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		case PRIVATE_STREAM1:

			memcpy(p->buf, headr, 3);
			p->buf[3] = p->cid;
			memcpy(p->buf+4,p->plen,2);

			if (p->mpeg == 2 && p->found == 9){
				p->buf[6] = p->flag1;
				p->buf[7] = p->flag2;
				p->buf[8] = p->hlength;
			}

			if (p->mpeg == 1 && p->found == 7){
				p->buf[6] = p->flag1;
			}


			if (p->mpeg == 2 && (p->flag2 & PTS_ONLY) &&  
			    p->found < 14){
				while (c < count && p->found < 14){
					p->pts[p->found-9] = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
				}
				if (c == count) return;
			}

			if (p->mpeg == 1 && p->which < 2000){

				if (p->found == 7) {
					p->check = p->flag1;
					p->hlength = 1;
				}

				while (!p->which && c < count && 
				       p->check == 0xFF){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
				}

				if ( c == count) return;
				
				if ( (p->check & 0xC0) == 0x40 && !p->which){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;

					p->which = 1;
					if ( c == count) return;
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}

				if (p->which == 1){
					p->check = buf[c];
					p->buf[p->found] = buf[c];
					c++;
					p->found++;
					p->hlength++;
					p->which = 2;
					if ( c == count) return;
				}
				
				if ( (p->check & 0x30) && p->check != 0xFF){
					p->flag2 = (p->check & 0xF0) << 2;
					p->pts[0] = p->check;
					p->which = 3;
				} 

				if ( c == count) return;
				if (p->which > 2){
					if ((p->flag2 & PTS_DTS_FLAGS)
					    == PTS_ONLY){
						while (c < count && 
						       p->which < 7){
							p->pts[p->which-2] =
								buf[c];
							p->buf[p->found] = 
								buf[c];
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					} else if ((p->flag2 & PTS_DTS_FLAGS) 
						   == PTS_DTS){
						while (c < count && 
						       p->which< 12){
							if (p->which< 7)
								p->pts[p->which
								      -2] =
									buf[c];
							p->buf[p->found] = 
								buf[c];
							c++;
							p->found++;
							p->which++;
							p->hlength++;
						}
						if ( c == count) return;
					}
					p->which = 2000;
				}
							
			}

			while (c < count && p->found < p->plength+6){
				l = count -c;
				if (l+p->found > p->plength+6)
					l = p->plength+6-p->found;
				memcpy(p->buf+p->found, buf+c, l);
				p->found += l;
				c += l;
			}			
			if(p->found == p->plength+6)
				func(p);
			
			break;
		}


		if ( p->done ){
			if( p->found + count - c < p->plength+6){
				p->found += count-c;
				c = count;
			} else {
				c += p->plength+6 - p->found;
				p->found = p->plength+6;
			}
		}

		if (p->plength && p->found == p->plength+6) {
			p->found = 0;
			p->done = 0;
			p->plength = 0;
			memset(p->buf, 0, MAX_PLENGTH);
			if (c < count)
				get_pes(buf+c, count-c, p, func);
		}
	}
	return;
}


void pes_in_ts(p2p *p)
{
	int l, pes_start;
	uint8_t obuf[TS_SIZE];
	long int c = 0;
	int length = p->plength+6;
	uint16_t pid;
	uint8_t *counter;
	uint8_t tspid0[TS_SIZE] = { 
                0x47, 0x40, 0x00, 0x10, 0x00, 0x00, 0xb0, 0x11, 
                0x00, 0x00, 0xcb, 0x00, 0x00, 0x00, 0x00, 0xe0, 
                0x10, 0x00, 0x01, 0xe4, 0x00, 0x2a, 0xd6, 0x1a, 
                0xfb, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff
        };

	uint8_t tspid1[TS_SIZE] = { 
                0x47, 0x44, 0x00, 0x10, 0x00, 0x02, 0xb0, 0x1c,
                0x00, 0x01, 0xcb, 0x00, 0x00, 0xe0, 0xa0, 0xf0, 
                0x05, 0x48, 0x03, 0x01, 0x00, 0x00, 0x02, 0xe0,
                0xa0, 0xf0, 0x00, 0x03, 0xe0, 0x50, 0xf0, 0x00, 
                0xae, 0xea, 0x4e, 0x48, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                0xff, 0xff, 0xff, 0xff
        };

	pes_start = 1;
	switch ( p->cid ) {
	case AUDIO_STREAM_S ... AUDIO_STREAM_E:			
		pid = p->pida;
		counter = &p->acounter;
		break;
	case VIDEO_STREAM_S ... VIDEO_STREAM_E:
		pid = p->pidv;
		counter = &p->acounter;

		tspid0[3] |= (p->count0++) 
			& 0x0F ;
		tspid1[3] |= (p->count1++) 
			& 0x0F ;
	
		tspid1[24]  = p->pidv;
		tspid1[23] |= (p->pidv >> 8) & 0x3F;
		tspid1[29]  = p->pida;
		tspid1[28] |= (p->pida >> 8) & 0x3F;
		
		p->func(tspid0,188,p->data);
		p->func(tspid1,188,p->data);
		break;
	default:
		return;
	}

	while ( c < length ){
		memset(obuf,0,TS_SIZE);
		if (length - c >= TS_SIZE-4){
			l = write_ts_header(pid, counter, pes_start
					     , obuf, TS_SIZE-4);
			memcpy(obuf+l, p->buf+c, TS_SIZE-l);
			c += TS_SIZE-l;
		} else { 
			l = write_ts_header(pid, counter, pes_start
					     , obuf, length-c);
			memcpy(obuf+l, p->buf+c, TS_SIZE-l);
			c = length;
		}
		p->func(obuf,188,p->data);
		pes_start = 0;
	}
}


void setup_pes2ts( p2p *p, uint16_t pida, uint16_t pidv, 
		   void (*ts_write)(uint8_t *buf, int count, void *data),
		   void *data)
{
	init_p2p( p, ts_write, 2048);
	p->pida = pida;
	p->pidv = pidv;
	p->acounter = 0;
	p->vcounter = 0;
	p->count1 = 0;
	p->count0 = 0;
	p->data = data;
}

void pes_to_ts( p2p *p,uint8_t *buf ,int count )
{
	get_pes(buf,count, p,pes_in_ts);
}


void setup_ts2pes( ipack *pa, ipack *pv, uint16_t pida, uint16_t pidv, 
		   void (*pes_write)(uint8_t *buf, int count, void *data),
		   void *priv)
{
	init_ipack( pa, 2048, pes_write);
	init_ipack( pv, 2048, pes_write);
	pa->pid = pida;
	pv->pid = pidv;
	pa->data = priv;
	pv->data = priv;
}

void ts_to_pes( ipack *p, uint8_t *buf) // don't need count (=188)
{
	uint8_t off = 0;

	if (!buf || !p ){
		return;
	}


	if ( buf[1]&PAY_START) {
		if (p->plength == MMAX_PLENGTH-6 && p->found>6){
			p->plength = p->found-6;
			p->found = 0;
			send_ipack(p);
			reset_ipack(p);
		}
	}

	if ( buf[3] & ADAPT_FIELD) {  // adaptation field?
		off = buf[4] + 1;
		if (off+4 > 187) return;
	}

	instant_repack(buf+4+off, TS_SIZE-4-off, p );
}




unsigned int audioCapabilities(int fd)
{

	int ans;

	if ( ioctl(fd,AUDIO_GET_CAPABILITIES, &ans) < 0 ){
		perror("AUDIO GET CAPABILITIES: ");
		return -1;
	}

	return ans;
}


int audioSetStreamtype(int fd, char *format)
{

	int ans;
	unsigned int cap;
	unsigned int f = 0;

	cap = audioCapabilities(fd);

	if (!strcmp(format, "DTS"))   f = AUDIO_CAP_DTS;
	if (!strcmp(format, "LPCM"))  f = AUDIO_CAP_LPCM;
	if (!strcmp(format, "MP1"))   f = AUDIO_CAP_MP1;
	if (!strcmp(format, "MP2"))   f = AUDIO_CAP_MP2;
	if (!strcmp(format, "MP3"))   f = AUDIO_CAP_MP3;
	if (!strcmp(format, "AAC"))   f = AUDIO_CAP_AAC;
	if (!strcmp(format, "OGG"))   f = AUDIO_CAP_OGG;
	if (!strcmp(format, "SDDS"))  f = AUDIO_CAP_SDDS;
	if (!strcmp(format, "AC3"))   f = AUDIO_CAP_AC3;

	if (!(cap & f)) {
		fprintf(stderr,"Can't set audio format %s, not supported\n",format);
		exit(1);
	}

	if ( ( ans = ioctl(fd,AUDIO_SET_STREAMTYPE, f)) < 0 ){
		perror("AUDIO SET STREAMTYPE: ");
		return -1;
	}

	return ans;
}

unsigned int videoCapabilities(int fd)
{

	int ans;

	if ( ioctl(fd,VIDEO_GET_CAPABILITIES, &ans) < 0 ){
		perror("VIDEO GET CAPABILITIES: ");
		return -1;
	}

	return ans;
}

int videoSetStreamtype(int fd, char *format)
{

	int ans;
	unsigned int cap;
	unsigned int f = 0;

	cap = videoCapabilities(fd);

	if (!strcmp(format, "MPEG1"))   f = VIDEO_CAP_MPEG1;
	if (!strcmp(format, "MPEG2"))  f = VIDEO_CAP_MPEG2;
	if (!strcmp(format, "SYS"))   f = VIDEO_CAP_SYS;
	if (!strcmp(format, "PROG"))   f = VIDEO_CAP_PROG;
	if (!strcmp(format, "SPU"))   f = VIDEO_CAP_SPU;
	if (!strcmp(format, "NAVI"))   f = VIDEO_CAP_NAVI;
	if (!strcmp(format, "CSS"))   f = VIDEO_CAP_CSS;

	if (!(cap & f)) {
		fprintf(stderr,"Can't set audio format %s, not supported\n",format);
		exit(1);
	}

	if ( (ans = ioctl(fd,VIDEO_SET_STREAMTYPE, f)) < 0 ){
		perror("VIDEO SET STREAMTYPE: ");
		return -1;
	}

	return ans;
}


int videoStop(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_STOP,0) < 0)){
		perror("VIDEO STOP: ");
		return -1;
	}

	return 0;
}

int videoPlay(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_PLAY) < 0)){
		perror("VIDEO PLAY: ");
		return -1;
	}

	return 0;
}


int videoFreeze(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_FREEZE) < 0)){
		perror("VIDEO FREEZE: ");
		return -1;
	}

	return 0;
}


int videoContinue(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_CONTINUE) < 0)){
		perror("VIDEO CONTINUE: ");
		return -1;
	}

	return 0;
}

int videoSelectSource(int fd, videoStreamSource_t source)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SELECT_SOURCE, source) < 0)){
		perror("VIDEO SELECT SOURCE: ");
		return -1;
	}

	return 0;
}



int videoSetBlank(int fd, boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SET_BLANK, state) < 0)){
		perror("VIDEO SET BLANK: ");
		return -1;
	}

	return 0;
}

int videoFastForward(int fd,int nframes)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_FAST_FORWARD, nframes) < 0)){
		perror("VIDEO FAST FORWARD: ");
		return -1;
	}

	return 0;
}

int videoSlowMotion(int fd,int nframes)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_SLOWMOTION, nframes) < 0)){
		perror("VIDEO SLOWMOTION: ");
		return -1;
	}

	return 0;
}

int videoGetStatus(int fd)
{
	struct videoStatus stat;
	int ans;

	if ( (ans = ioctl(fd,VIDEO_GET_STATUS, &stat) < 0)){
		perror("VIDEO GET STATUS: ");
		return -1;
	}

	printf("Video Status:\n");
	printf("  Blank State          : %s\n",
	       (stat.videoBlank ? "BLANK" : "STILL"));
	printf("  Play State           : ");
	switch ((int)stat.playState){
	case VIDEO_STOPPED:
		printf("STOPPED (%d)\n",stat.playState);
		break;
	case VIDEO_PLAYING:
		printf("PLAYING (%d)\n",stat.playState);
		break;
	case VIDEO_FREEZED:
		printf("FREEZED (%d)\n",stat.playState);
		break;
	default:
		printf("unknown (%d)\n",stat.playState);
		break;
	}
	
	printf("  Stream Source        : ");
	switch((int)stat.streamSource){
	case VIDEO_SOURCE_DEMUX:
		printf("DEMUX (%d)\n",stat.streamSource);
		break;
	case VIDEO_SOURCE_MEMORY:
		printf("MEMORY (%d)\n",stat.streamSource);
		break;
	default:
		printf("unknown (%d)\n",stat.streamSource);
		break;
	}

	printf("  Format (Aspect Ratio): ");
	switch((int)stat.videoFormat){
	case VIDEO_FORMAT_4_3:
		printf("4:3 (%d)\n",stat.videoFormat);
		break;
	case VIDEO_FORMAT_16_9:
		printf("16:9 (%d)\n",stat.videoFormat);
		break;
	default:
		printf("unknown (%d)\n",stat.videoFormat);
		break;
	}

	printf("  Display Format       : ");
	switch((int)stat.displayFormat){
	case VIDEO_PAN_SCAN:
		printf("Pan&Scan (%d)\n",stat.displayFormat);
		break;
	case VIDEO_LETTER_BOX:
		printf("Letterbox (%d)\n",stat.displayFormat);
		break;
	case VIDEO_CENTER_CUT_OUT:
		printf("Center cutout (%d)\n",stat.displayFormat);
		break;
	default:
		printf("unknown (%d)\n",stat.displayFormat);
		break;
	}
	return 0;
}

int videoStillPicture(int fd, struct videoDisplayStillPicture *sp)
{
	int ans;

	if ( (ans = ioctl(fd,VIDEO_STILLPICTURE, sp) < 0)){
		perror("VIDEO STILLPICTURE: ");
		return -1;
	}

	return 0;
}

int audioStop(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_STOP,0) < 0)){
		perror("AUDIO STOP: ");
		return -1;
	}

	return 0;
}

int audioPlay(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_PLAY) < 0)){
		perror("AUDIO PLAY: ");
		return -1;
	}

	return 0;
}


int audioPause(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_PAUSE) < 0)){
		perror("AUDIO PAUSE: ");
		return -1;
	}

	return 0;
}


int audioContinue(int fd)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_CONTINUE) < 0)){
		perror("AUDIO CONTINUE: ");
		return -1;
	}

	return 0;
}

int audioSelectSource(int fd, audioStreamSource_t source)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SELECT_SOURCE, source) < 0)){
		perror("AUDIO SELECT SOURCE: ");
		return -1;
	}

	return 0;
}



int audioSetMute(int fd, boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_MUTE, state) < 0)){
		perror("AUDIO SET MUTE: ");
		return -1;
	}

	return 0;
}

int audioSetAVSync(int fd,boolean state)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_AV_SYNC, state) < 0)){
		perror("AUDIO SET AV SYNC: ");
		return -1;
	}

	return 0;
}

int audioSetBypassMode(int fd,boolean mode)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_SET_BYPASS_MODE, mode) < 0)){
		perror("AUDIO SET BYPASS MODE: ");
		return -1;
	}

	return 0;
}


int audioChannelSelect(int fd, audioChannelSelect_t select)
{
	int ans;

	if ( (ans = ioctl(fd,AUDIO_CHANNEL_SELECT, select) < 0)){
		perror("AUDIO CHANNEL SELECT: ");
		return -1;
	}

	return 0;
}

int audioGetStatus(int fd)
{
	struct audioStatus stat;
	int ans;

	if ( (ans = ioctl(fd,AUDIO_GET_STATUS, &stat) < 0)){
		perror("AUDIO GET STATUS: ");
		return -1;
	}

	printf("Audio Status:\n");
	printf("  Sync State          : %s\n",
	       (stat.AVSyncState ? "SYNC" : "NO SYNC"));
	printf("  Mute State          : %s\n",
	       (stat.muteState ? "muted" : "not muted"));
	printf("  Play State          : ");
	switch ((int)stat.playState){
	case AUDIO_STOPPED:
		printf("STOPPED (%d)\n",stat.playState);
		break;
	case AUDIO_PLAYING:
		printf("PLAYING (%d)\n",stat.playState);
		break;
	case AUDIO_PAUSED:
		printf("PAUSED (%d)\n",stat.playState);
		break;
	default:
		printf("unknown (%d)\n",stat.playState);
		break;
	}
	
	printf("  Stream Source       : ");
	switch((int)stat.streamSource){
	case AUDIO_SOURCE_DEMUX:
		printf("DEMUX (%d)\n",stat.streamSource);
		break;
	case AUDIO_SOURCE_MEMORY:
		printf("MEMORY (%d)\n",stat.streamSource);
		break;
	default:
		printf("unknown (%d)\n",stat.streamSource);
		break;
	}

	printf("  Channel Select      : ");
	switch((int)stat.channelSelect){
	case AUDIO_STEREO:
		printf("Stereo (%d)\n",stat.channelSelect);
		break;
	case AUDIO_MONO_LEFT:
		printf("Mono left(%d)\n",stat.channelSelect);
		break;
	case AUDIO_MONO_RIGHT:
		printf("Mono right (%d)\n",stat.channelSelect);
		break;
	default:
		printf("unknown (%d)\n",stat.channelSelect);
		break;
	}
	printf("  Bypass Mode         : %s\n",
	       (stat.bypassMode ? "ON" : "OFF"));

	return 0;

}



void play_file_video_stdin(int fd, int fda)
{
	char buf[BUFFY];
	int count, c;
	int written;
	struct pollfd pfd[NFD];
	int stop = 0;

	pfd[0].fd = STDIN_FILENO;
	pfd[0].events = POLLIN;
	
	pfd[1].fd = fd;
	pfd[1].events = POLLOUT;

	if (fda) audioSetAVSync(fda, true);
	videoSelectSource(fd,VIDEO_SOURCE_MEMORY);
	videoPlay(fd);
	
	written = 0;
	count = 0;
	while ( 1 ){
		if (stop >1000) break;
		if (poll(pfd,NFD,1)){
			if ( pfd[1].revents & POLLOUT){
				if ( written < count){
					c = write(fd,buf+written,
						  count-written);
					if (c > 0) written += c;
				} else stop++;
				
			}
			if (written == count &&
			    pfd[0].revents & POLLIN){
				if ( (count = 
				      read(STDIN_FILENO,buf,BUFFY)) 
				     < 0){
					perror("AUDIO DEVICE: ");
					exit(1);
				}
				written = 0;
				if (count>0) stop = 0;
			}
		}
	}
}




void load_iframe(int filefd, int fd)
{
	struct stat st;
	struct videoDisplayStillPicture sp;

	fstat(filefd, &st);
	
	sp.iFrame = (char *) malloc(st.st_size);
	sp.size = st.st_size;
	printf("I-frame size: %d\n", sp.size);
	
	if(!sp.iFrame) {
		printf("No memory for I-Frame\n");
		return;
	}

	printf("read: %d bytes\n",read(filefd,sp.iFrame,sp.size));
	videoStillPicture(fd,&sp);

	sleep(3);
	videoPlay(fd);
}

int open_av(int *fdv,int *fda, int dev)
{
	char devnamea[80];
	char devnamev[80];
	
	sprintf(devnamea,"/dev/ost/audio%d",dev);
	sprintf(devnamev,"/dev/ost/video%d",dev);
	
	if((*fdv = open(devnamev,O_RDWR|O_NONBLOCK)) < 0){
		perror("VIDEO DEVICE: ");
		return -1;
	}
	    
	if((*fda = open(devnamea,O_RDWR|O_NONBLOCK)) < 0){
		perror("AUDIO DEVICE: ");
		return -1;
	}
	


	videoSetBlank(*fdv,false);
	audioSelectSource(*fda,AUDIO_SOURCE_MEMORY);
	audioSetAVSync(*fda, true);
	audioPlay(*fda);
	return 0;
}



void set_pat_filt(int fd)
{
	struct dmxSctFilterParams sctFilterParams;


	memset(&sctFilterParams.filter, 0, sizeof(struct dmxFilter));
	sctFilterParams.pid                       = 0;
	sctFilterParams.filter.filter[0]          = 0x00;
	sctFilterParams.filter.mask[0]            = 0x00;
	sctFilterParams.timeout                   = 1000;
	sctFilterParams.flags                     = DMX_IMMEDIATE_START;

	if (ioctl(fd, DMX_SET_FILTER, &sctFilterParams) < 0)  
		perror("DMX SET FILTER:");

}

void set_pmt_filt(int fd,uint16_t ppid)
{
	struct dmxSctFilterParams sctFilterParams;


	memset(&sctFilterParams.filter, 0, sizeof(struct dmxFilter));
	sctFilterParams.pid                       = ppid;
	sctFilterParams.filter.filter[0]          = 0x02;
	sctFilterParams.filter.mask[0]            = 0xFF;
	sctFilterParams.timeout                   = 1000;
	sctFilterParams.flags                     = DMX_IMMEDIATE_START;

	if (ioctl(fd, DMX_SET_FILTER, &sctFilterParams) < 0)  
		perror("DMX SET FILTER:");

}

void set_av_filts(int vfd,int afd,uint16_t vpid,uint16_t apid)
{
	struct dmxPesFilterParams pesFilterParams; 


	if (ioctl(vfd, DMX_SET_BUFFER_SIZE, 64*1024) < 0)  
		perror("DMX SET BUFFER:");
	pesFilterParams.pid = vpid; 
	pesFilterParams.input = DMX_IN_DVR; 
	pesFilterParams.output = DMX_OUT_DECODER; 
	pesFilterParams.pesType = DMX_PES_VIDEO; 
	pesFilterParams.flags = DMX_IMMEDIATE_START;
	if (ioctl(vfd, DMX_SET_PES_FILTER, &pesFilterParams) < 0) 
		perror("DMX SET FILTER:");
  
	if (ioctl(afd, DMX_SET_BUFFER_SIZE, 64*1024) < 0) 
		perror("DMX SET BUFFER:");
	pesFilterParams.pid = apid;
	pesFilterParams.input = DMX_IN_DVR; 
	pesFilterParams.output = DMX_OUT_DECODER; 
	pesFilterParams.pesType = DMX_PES_AUDIO; 
	pesFilterParams.flags = DMX_IMMEDIATE_START;
  
	if (ioctl(afd, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
		perror("DMX SET FILTER:");
}


uint16_t get_pmt_pid(int fd)
{
	u_char sec[MAX_SECTION_SIZE];
	int len, i;
	uint16_t cpid = 0;
	uint16_t length;
	len=read(fd, sec, 4096);
	
	if (len <= 0) return 0;
	
	length  = (sec[1]& 0x0F)<<8;
	length |= (sec[2]& 0xFF);
	
	for (i = 8; i< length-1 && cpid == 0; i+=4){
		if (sec[i] != 0 || sec[i+1] !=0){
			cpid = get_pid(sec+i+2);
			printf("TS: PMT PID: %04x\n",cpid);
		}
	}
	return cpid;
}


void get_av_pids(int fd, uint16_t *vpid, uint16_t *apid)
{		
	u_char sec[MAX_SECTION_SIZE];
	int len, i, ilength;
	uint16_t length;
	len=read(fd, sec, 4096);
	
	if (len <= 0) return;

	length  = (sec[1]& 0x0F)<<8;
	length |= (sec[2]& 0xFF);
        
	ilength = (unsigned short)
		((sec[10]&3)<<8);
	ilength |= (sec[11]&0xFF);
	for (i = 12+ilength; i< length-1; i+=5){
		if (sec[i] == 0x02){
			*vpid = get_pid(sec+i+1);
			printf("TS: VIDEO PID: %d\n",*vpid);
		}
		if (sec[i] == 0x03 || sec[i] == 0x04){
			*apid = get_pid(sec+i+1);
			printf("TS: AUDIO PID: %d\n",*apid);
		}
		if (*vpid && *apid) break;
		i+=((sec[i+3]&15)<<8)|sec[i+4];
	}
}





