/*
 * unrm files stored in wastebasket
 *   Copyright (C) 1998 by Michael Meskes
 *   PLaced under LGPL
 * 
 */
#include <dirent.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <utime.h>
#include <fcntl.h>
#include <libgen.h>
#include <malloc.h>
#include <time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <pwd.h>

#include "libwastebasket.h"

#define BUFSIZE		8192

extern int real_unlink(const char *);

/*
 * Write all of the supplied buffer out to a file.
 * This does multiple writes as necessary.
 * Returns the amount written, or -1 on an error.
 */
static int fullWrite(int fd, const char *buf, int len)
{
    int cc;
    int total;

    total = 0;

    while (len > 0) {
	cc = write(fd, buf, len);

	if (cc < 0)
	    return -1;

	buf += cc;
	total += cc;
	len -= cc;
    }

    return total;
}

/*
 * undelete a file
 */
int undelete(const char *filename, const char *timestamp)
{
    char *destfile, *wb_file, buf[BUFSIZE], *wb_dir, *exists = NULL, *restore_file;
    struct stat statbuf, filestatbuf;
    int rfd, wfd, rcc;
    struct utimbuf times;
    DIR *           dirp;
    struct dirent * dp;

    /* check for user's wastebasket */
    if ((wb_dir = check_for_wastebasket(NULL)) == NULL)
	return (-1);

    free(wb_dir);
    
    if ((destfile = full_src_name(filename)) == NULL)
    	return(-1);
    
    /* check the wb_dir for the file */
    if ((wb_dir = dirname(strdup(full_waste_name(filename)))) == NULL) {
	free(destfile);
	return(-1);
    }

    if ((dirp = opendir(wb_dir)) == NULL) {
	free(destfile);
	free(wb_dir);
	return(-1);
    }
    
    wb_file = strdup(filename);
    restore_file = basename(wb_file);
    
    while ((dp = readdir(dirp)) != NULL) {
    	if (strncmp(dp->d_name, restore_file, strlen(restore_file)) == 0) {
		/* okay found one possibilty */
		/* is it the first one? the name has to be unique unless we have a time stamp */
		if (exists != NULL)  {
			free(wb_file);
			free(destfile);
			free(wb_dir);
			return(-2);
		}
		
		/* if we have a timestamp check it */
		/* note that the highest digits might either be '0' or ' ' in timesvtamp */
		if (timestamp != NULL) {
			int i, j = strlen(dp->d_name) - strlen(timestamp);
			
			if (strncmp(dp->d_name + j, timestamp, strlen(timestamp)) != 0)
				continue;
			
			/* make sure the remaining characters are "_0*" */
			i = strlen(restore_file);
			if ((dp->d_name)[i] != '_')
				continue;
			
			for (i++; i < j && (dp->d_name)[i] == '0'; i++);
			if (i < j)
				continue;
		}

		/* timestamp is okay */
		if ((exists = strdup(dp->d_name)) == NULL) {
			free(wb_file);
			free(destfile);
			free(wb_dir);
			return(-1);
		}
		
		/* if we have a timestamp we can stop here */
		if (timestamp != NULL)
			break;
	}
    }

    closedir(dirp);

    /* Did we find one? */
    if (exists == NULL) { 
        free(destfile);
	free(wb_dir);
	free(exists);
	errno = ENOENT;
	return (-1);
    }
    
    free(wb_file);
    
    /* create wb_file name */
    if ((wb_file = malloc(strlen(wb_dir) + strlen(exists) + 2)) == NULL) {
	free(destfile);
	free(wb_dir);
	free(exists);
	return (-1);
    }
    
    strcpy(wb_file, wb_dir);
    strcat(wb_file, "/");
    strcat(wb_file, exists);
    
    free(wb_dir);
    free(exists);

    /* stat the file to see we can access it */
    if (stat(wb_file, &statbuf) != 0) {
	free(wb_file);
	free(destfile);
	return (-1);
    }
    
    /* unrm must not restore directories */
    if (S_ISDIR(filestatbuf.st_mode)) {
	free(wb_file);
	free(destfile);
	errno = EPERM;
	return (-1);
    }
    
    /* finally start moving the file */
    if (rename(wb_file, destfile) != 0) {
	if (errno == EXDEV) {
	    /* we have to do a copy */
	    /* so make sure we can read the file */
	    if (chmod(wb_file, filestatbuf.st_mode | S_IREAD) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    /* and open it */
	    rfd = open(wb_file, O_RDONLY);
	    if (rfd < 0) {
		chmod(wb_file, filestatbuf.st_mode);
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    wfd = creat(destfile, filestatbuf.st_mode);
	    if (wfd < 0) {
		chmod(wb_file, filestatbuf.st_mode);
		free(destfile);
		free(wb_file);
		close(rfd);
		return (-1);
	    }
	    
	    while ((rcc = read(rfd, buf, sizeof(buf))) > 0) {
		if (fullWrite(wfd, buf, rcc) < 0) {
		    chmod(wb_file, filestatbuf.st_mode);
		    free(destfile);
		    free(wb_file);
		    close(rfd);
		    close(wfd);
		    return (-1);
		}
	    }

	    if (rcc < 0) {
		chmod(wb_file, filestatbuf.st_mode);
		free(destfile);
		free(wb_file);
		close(rfd);
		close(wfd);
		return (-1);
	    }
	    close(rfd);

	    /* restore original permissions */
	    /* just in case we have an error exit later */
	    if (chmod(destfile, filestatbuf.st_mode) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    if (close(wfd) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    /* set filestamps accordingly */
	    if (chmod(destfile, filestatbuf.st_mode) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    if (chown(destfile, filestatbuf.st_uid, statbuf.st_gid) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    times.actime = filestatbuf.st_atime;
	    times.modtime = filestatbuf.st_mtime;
	    if (utime(destfile, &times) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	    
	    /* finally remove the original file */
	    if (real_unlink(wb_file) != 0) {
		free(destfile);
		free(wb_file);
		return (-1);
	    }
	} else {
	    free(destfile);
	    free(wb_file);
	    return (-1);
	}
    }
    free(destfile);
    free(wb_file);
    return (0);
}

static void usage(const char *progname)
{
    fprintf(stderr, "Usage: %s filename [timestamp]\n", progname);
    exit(-1);
}

int main(int argc, char **argv)
{
    int ret;
    char *progname = basename(argv[0]);

    if (argc < 2 || argc > 3)
    	usage(progname);    
    	
    ret = undelete(argv[1], argv[2]);
    switch(ret) {
    	case -1:
		perror(progname);
		break;
	case -2:
		fprintf(stderr, "%s: Filename is not unique!\n", progname);
		break;
	default:
		break;
    }
    exit(ret);
}
