/*
 * unlink replacement for usage with generic wastebasket
 *   Copyright (C) 1998 by Michael Meskes
 *   PLaced under LGPL
 * 
 * This Linux shared library moves files into a wastebasket instead of really
 * unlinking them.
 * 
 */
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <utime.h>
#include <malloc.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <libgen.h>
#include <time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <pwd.h>

#include "libwastebasket.h"

#define	BUFSIZE		8192

/*
 * Write all of the supplied buffer out to a file.
 * This does multiple writes as necessary.
 * Returns the amount written, or -1 on an error.
 */
static int fullWrite(int fd, const char *buf, int len)
{
    int cc;
    int total;

    total = 0;

    while (len > 0) {
	cc = write(fd, buf, len);

	if (cc < 0)
	    return -1;

	buf += cc;
	total += cc;
	len -= cc;
    }

    return total;
}

/* 
 * call the real unlink()
 */
int real_unlink(const char *filename)
{
    int (*fptr)(const char *);
    
    if ((fptr = (int (*)(const char *)) dlsym(RTLD_NEXT, "unlink")) == NULL)
    {
	    errno = ENOSYS;
	    return errno;
    }
    return ((*fptr)(filename));
}

/*
 * the unlink replacement
 * note, that this function might call the real unlink system call
 */
int unlink(const char *filename)
{
    char *dir, *srcfile, *wastefile, *wastedir, buf[BUFSIZE];
    struct stat statbuf, filestatbuf;
    int i, rfd, wfd, rcc, size;
    struct utimbuf times;
    time_t now = time(NULL);

    /* stat the file to see we can access it */
    if (lstat(filename, &filestatbuf) != 0)
	return (-1);

    /* unlink must not delete directories */
    if (S_ISDIR(filestatbuf.st_mode)) {
	errno = EPERM;
	return (-1);
    }

    /* check for the user's wastebasket */
    if ((dir = check_for_wastebasket(NULL)) == NULL)
	/* user has no wastebasket or problem accessing it */
	return (real_unlink(filename));

    if ((srcfile = full_src_name(filename)) == NULL)
    	return(-1);
    	
    /* make sure we do not try to move already deleted files */
    /* also do not protect /tmp, /usr/tmp and /var/tmp */
    if (strncmp(srcfile, dir, strlen(dir)) == 0 ||
	strncmp(srcfile, "/tmp", strlen("/tmp")) == 0 ||
	strncmp(srcfile, "/usr/tmp", strlen("/usr/tmp")) == 0 ||
	strncmp(srcfile, "/var/tmp", strlen("/var/tmp")) == 0) {

	/* oops, call real unlink instead */
	free(srcfile);
	free(dir);
	return (real_unlink(filename));
    }
    
    if ((wastefile = (char *) malloc(strlen(srcfile) + strlen(dir) + 12)) == NULL) {
	free(srcfile);
	return (-1);
    }

    sprintf(wastefile, "%s%s_%010lu", dir, srcfile, now);
    
    if ((wastedir = (char *) malloc(size = strlen(wastefile))) == NULL) {
	free(srcfile);
	free(wastefile);
	return (-1);
    }
    
    i = strlen(dir) + 1;
    free(dir);

    while (wastefile[i] != '\0') {
	for (; wastefile[i] != '\0' && wastefile[i] != '/'; i++);
	if (wastefile[i] != '\0') {
	    /* if this directory does not exist, create it */
	    memset(wastedir, '\0', size);
	    strncpy(wastedir, wastefile, i);
	    if (stat(wastedir, &statbuf) != 0) {
		if (errno == ENOENT) {
		    if (mkdir(wastedir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != 0) {
			free(wastefile);
			free(srcfile);
			free(wastedir);
			return (-1);
		    }
	    	} else {
		    	free(wastefile);
			free(srcfile);
			free(wastedir);
			return (-1);
		}
	    }
	    
	    i++;
	}
    }
    
    free(wastedir);
    /* finally start moving the file */
    if (rename(srcfile, wastefile) != 0) {
	if (errno == EXDEV) {
	    /* we have to do a copy */
	    /* so make sure we can read the file */
	    if (chmod(srcfile, filestatbuf.st_mode | S_IREAD) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    /* and open it */
	    rfd = open(srcfile, O_RDONLY);
	    if (rfd < 0) {
		chmod(srcfile, filestatbuf.st_mode);
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    wfd = creat(wastefile, filestatbuf.st_mode);
	    if (wfd < 0) {
		chmod(srcfile, filestatbuf.st_mode);
		free(srcfile);
		free(wastefile);
		close(rfd);
		return (-1);
	    }
	    while ((rcc = read(rfd, buf, sizeof(buf))) > 0) {
		if (fullWrite(wfd, buf, rcc) < 0) {
		    chmod(srcfile, filestatbuf.st_mode);
		    free(srcfile);
		    free(wastefile);
		    close(rfd);
		    close(wfd);
		    return (-1);
		}
	    }

	    if (rcc < 0) {
		chmod(srcfile, filestatbuf.st_mode);
		free(srcfile);
		free(wastefile);
		close(rfd);
		close(wfd);
		return (-1);
	    }
	    close(rfd);

	    /* restore original permissions */
	    /* just in case we have an error exit later */
	    if (chmod(srcfile, filestatbuf.st_mode) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    if (close(wfd) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    /* set filestamps accordingly */
	    if (chmod(wastefile, filestatbuf.st_mode) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    if (chown(wastefile, filestatbuf.st_uid, statbuf.st_gid) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    times.actime = filestatbuf.st_atime;
	    times.modtime = filestatbuf.st_mtime;
	    if (utime(wastefile, &times) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	    /* finally remove the original file */
	    if (real_unlink(srcfile) != 0) {
		free(srcfile);
		free(wastefile);
		return (-1);
	    }
	} else {
	    free(srcfile);
	    free(wastefile);
	    return (-1);
	}
    }
    free(srcfile);
    free(wastefile);
    return (0);
}
