/*
 * Some helpful functions
 *   Copyright (C) 1998 by Michael Meskes
 *   PlLaced under LGPL
 * 
 * This Linux shared library moves files into a wastebasket instead of really
 * unlinking them.
 * 
 */
#include <errno.h>
#include <malloc.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <pwd.h>
#include <sys/stat.h>
#include <fcntl.h>

#if !defined(WASTEBASKET)
#define WASTEBASKET        ".wastebasket"
#endif

#if !defined(WBREADME)
#define WBREADME     "README.wastebasket"
#endif

#if !defined(WBROOT)
#define WBROOT	"waste"
#endif

char *full_src_name(const char *filename)
{
    char *dir, *pwd, *file, *basefile;
    static char *srcfile;

    /* we need the full path of the file */
    if (strcmp(filename, "/") != 0) {
	    /* first store the actual dir */
	    if ((pwd = getcwd(NULL, 0)) == NULL)
		return (NULL);

	    /* note that dirname changes its argument */
	    if ((dir = strdup(filename)) == NULL) {
		free(pwd);
		return (NULL);
	    }

	    if (chdir(dirname(dir)) != 0) {
		free(dir);
		free(pwd);
		return (NULL);
	    }
	    free(dir);

	    /* now get the current working directory */
	    if ((dir = getcwd(NULL, 0)) == NULL) {
		free(pwd);
		return (NULL);
	    }

	    /* and add the filename */
	    file = strdup(filename);
	    basefile = basename(file);
	    if ((srcfile = (char *) malloc(strlen(dir) + strlen(basefile) + 2)) == NULL) {
		free(dir);
		free(pwd);
		free(file);
		return (NULL);
	    }

	    strcpy(srcfile, dir);
	    if (strcmp(dir, "/") != 0)
 		   strcat(srcfile, "/");
	    strcat(srcfile, basefile);
	    free(dir);
	    free(file);

	    /* and go back to where we started */
	    if (chdir(pwd) != 0) {
		free(pwd);
		free(srcfile);
		return (NULL);
	    }

	    free(pwd);
    } else {
    	    if ((srcfile = strdup(filename)) == NULL)
    	    	return(NULL);
    }
    return (srcfile);
}

char *full_waste_name(const char *filename)
{
    char *file;
    static char *srcfile;
    struct passwd *passwd;
    struct stat statbuf;

    /* we need to canonify the path of the file */
    if ((file = full_src_name(filename)) == NULL)
	return (NULL);

    /* get the user's home dir */
    if ((passwd = getpwuid(geteuid())) == NULL) {
	free(file);
	return (NULL);
    }

    /* is it a directory? */
    /* otherwise there is no wastedir */
    if (stat(passwd->pw_dir, &statbuf) != 0) {
	free(file);
	return (NULL);
    }

    if (!S_ISDIR(statbuf.st_mode)) {
	free(file);
	errno = ENOENT;
	return (NULL);
    }

    /* and add the filename */
    if ((srcfile = (char *) malloc(strlen(passwd->pw_dir) + strlen(WASTEBASKET) + strlen(WBROOT) + strlen(file) + 3)) == NULL) {
	free(file);
	return (NULL);
    }

    strcpy(srcfile, passwd->pw_dir);
    strcat(srcfile, "/");
    strcat(srcfile, WASTEBASKET);
    strcat(srcfile, "/");
    strcat(srcfile, WBROOT);
    strcat(srcfile, file);
    return (srcfile);
}

/* check if the user has a wastebasket */
char *check_for_wastebasket(int *wastedir_prefix)
{
    struct stat statbuf;
    char *dir, *readme;
    struct passwd *passwd;

    /* get the user's home dir */
    if ((passwd = getpwuid(geteuid())) == NULL)
	return (NULL);

    /* is it a directory? */
    if (stat(passwd->pw_dir, &statbuf) != 0)
	return (NULL);

    if (!S_ISDIR(statbuf.st_mode)) {
	/* user has no wastebasket */
	errno = ENOENT;
	return (NULL);
    }
    
    /* do we have a wastebasket directory in the user's home? */
    if ((dir = (char *) malloc(strlen(passwd->pw_dir) + strlen(WASTEBASKET) + strlen(WBROOT) + 3)) == NULL)
	return (NULL);

    strcpy(dir, passwd->pw_dir);
    strcat(dir, "/");
    strcat(dir, WASTEBASKET);

    if (stat(dir, &statbuf) != 0) {
	if (errno == ENOENT) {
	    /* create wastebasket dir */
	    if (mkdir(dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != 0) {
		free(dir);
		return (NULL);
	    }
	} else {
	    free(dir);
	    return (NULL);
	}
    }
    
    /* check if we also have the README.wastebasket file */
    if ((readme = (char *) malloc(strlen(dir) + strlen(WBREADME) + 2)) == NULL) {
    	free(dir);
	return (NULL);
    }

    strcpy(readme, dir);
    strcat(readme, "/");
    strcat(readme, WBREADME);

    if (stat(readme, &statbuf) != 0) {
	if (errno == ENOENT) {
	    int fd; 
	    char *text = "This is the wastebasket directory!\n\nIt is completely managed by the wastebasket utilities.\n\nUse purge to delete files!\n";
	    
	    /* create wastebasket readme */
	    if ((fd = creat(readme, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) == -1) {
		free(dir);
		free(readme);
		return (NULL);
	    }
	    
	    if (write(fd, text, strlen(text)) != strlen(text)) {
	    	free(dir);
	    	free(readme);
	    	return(NULL);
	    }	
	    
	    if (close(fd) != 0) {
	    	free(dir);
	    	free(readme);
	    	return(NULL);
	    }
	} else {
	    free(dir);
	    free(readme);
	    return (NULL);
	}
    }
    
    free(readme);
    
    /* check for the root dir in the wastebasket */
    strcat(dir, "/");
    strcat(dir, WBROOT);

    if (stat(dir, &statbuf) != 0) {
	if (errno == ENOENT) {
	    /* create root dir */
	    if (mkdir(dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != 0) {
		free(dir);
		return (NULL);
	    }
	} else {
	    free(dir);
	    return (NULL);
	}
    }
    
    if (wastedir_prefix)
    	*wastedir_prefix = strlen(dir);
    	
    return (dir);
}
