/* undelete.c
   Undeletes the files saved by the safedelete
   program.

   Uses ncurses to perform terminal I/O to the
   Linux console or xterm window.

   Usage:  undelete [ -i | --info ]
                    [ -l | --list ]
                    [ -f | --files ]
                    [ -r | --replace ]
                    [ -s | --subdir ]
                    [ -a | --all]
                    [ -g | --grunge ] <- for use by undeltk!
                                         used when multiple safedeleted files
                                         have the same filename.
                    [ filename ]

   If filename is specified, the most recent
   copy of that file is restored to its original
   location.  The filename can consist of
   the complete path+filename, partial path
   with filename, or just the filename.  If
   the complete path is not specified, it
   is assumed to be relative to the current
   directory (where the undelete command
   was issued).

   If no filename is specified, the list of 
   files in the .safedelete.log is shown as
   a scrollable list allowing the users to 
   choose which file(s) to undelete.

   If the file to be undeleted already exists
   the user is prompted for permission to
   overlay the existing file.

   If the -i or --info options are specified,
   only information on the deleted file is given.
   If no filename is given with the -i/--info
   options, information on all files is displayed.
*/

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <utime.h>
#include <getopt.h>

#ifndef NO_CURSES
#include <curses.h>
#define BACK_ROW 0
#define BACK_COL (int)((COLS-80)/2)
#define LIST_ROW BACK_ROW+3
#define LIST_COL BACK_COL+1
#define LIST_BEG 0
/*
#ifdef HPUX
chtype __acs_map[];
#endif
*/
#endif

#include "safedelete.h"

/* Global variables */

int  ListEnd = 17;
int  PartialLastLine = 0;
char FileNamePattern[256];
char SafeDelLockFile[DEL_LOG_LEN];
char CmdName[16];
int  PatternType = ALLFILES;
int  PatternPrefixLen = 0;
int  PatternSuffixLen = 0;
int  SubdirFlag = 0;
int  ReplaceFlag = 0;

/* ================================================================ */

/* PatternMatch

   This routine compares the passed filename to
   either the default filename pattern or the 
   pattern specified by the user.

   Passed: pointer to filename

   Returns: 0 - filename matches pattern
            1 - filename does not match pattern

   Rules:
   1 If pattern begins with '/' then only that
     directory is checked for matches.  Only
     if SubdirFlag is set (using -s option) will
     we descend the subdirectories.
   2 If the pattern does not begin with '/' then
     we assume it's a filename and check only the
     filename level for a match, not any part of 
     the directory path.
   3 If the pattern is a prefix (is "something*")
     and begins with '/' we follow rule 1 otherwise
     we assume it's a filename and follow rule 2.
   4 If the pattern is a suffix (is "*something")
     we scan the lowest level of the filename for
     a match.
   5 If the pattern is infix (is "something*something")
     we follow rules 1 and 2 for the each part of the
     pattern.
   6 If the pattern is just '*' then we obviously match
     on all filenames.
*/
int  PatternMatch(FileName)
  char *FileName;
{
char *wrkptr;

  switch(PatternType)
  {
    case PREFIX   : if(*FileNamePattern != '/')
                      wrkptr=FileName+ParseFileName(FileName)+1;
                    else
                    {
                      wrkptr=FileName;
                      if(! SubdirFlag && strchr(FileName+PatternPrefixLen, '/'))
                        return 1;
                    }
                    if(strncmp(wrkptr, FileNamePattern, PatternPrefixLen) == 0)
                      return 0;
                    else
                      return 1;

    case SUFFIX   : if(strncmp(FileName+strlen(FileName)-PatternSuffixLen,
                              FileNamePattern+1, PatternSuffixLen) == 0)
                      return 0;
                    else
                      return 1;

    case INFIX    : if(*FileNamePattern != '/')
                      wrkptr=FileName+ParseFileName(FileName)+1;
                    else
                      wrkptr=FileName;
                    if(strncmp(wrkptr, FileNamePattern, PatternPrefixLen) == 0)
                    {
                      if(! SubdirFlag && strchr(wrkptr+PatternPrefixLen, '/'))
                        return 1;
                      if(strcmp(FileName+strlen(FileName)-PatternSuffixLen,
                                FileNamePattern+PatternPrefixLen+1) == 0)
                        return 0;
                      else
                        return 1;
                    }
                    else
                      return 1;

    case FULLNAME : if(*FileNamePattern != '/')
                      wrkptr=FileName+ParseFileName(FileName)+1;
                    else
                      wrkptr=FileName;
                    if(strcmp(wrkptr, FileNamePattern) == 0)
                      return 0;
                    else
                      return 1;

    case ALLFILES : return 0;
  }
  return 1;
}

/* ================================================================ */

/* UndeleteFile
   Scans the users .safedelete.log looking for the fully
   qualified filename.  Once found, the grunged filename
   is obtained and the file is tar'ed back to its original
   location.

   Passed: pointer to fully qualified filename
           pointer to name of safedelete directory
	   pointer to path of user's .safedelete.log
           pointer to path of user's .Safedelrc
           pointer to new filename or NULL
	   pointer to existing file information or NULL
           relative record number to undelete or -1 if last occurence of file
           undelete flag - TRUE => undelete the file, FALSE => remove log entry only
           pointer to grunged name or NULL (used primarily by undeltk to
                                            specify which file to undelete)
	   
   Returns: 0 - file successfully undeleted.
            1 - filename not found in .safedelete.log
            2 - saved file not found in safedelete directory
            3 - undelete failed, reason unknown
*/
int  UndeleteFile(FileName, SafeDir, SafeDelLog, SafeDelRC,
                  NewName, CurrInfo,
                  FileNum, UndelFlag, GrungeName)
  char *FileName, *SafeDir, *SafeDelLog, *SafeDelRC, *NewName, *GrungeName;
  struct stat *CurrInfo;
  int  FileNum, UndelFlag;
{
char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];

char *LogList, *palloc, *wrkptr, *wrkptr2;

char *GrungedFN;
char *UncompressCmd;
char *FileSuffix;

int  i, j, k;
long LogSize;
time_t NewestMtime;
FILE *flog, *flink;
   
mode_t  Mode;
uid_t   Uid;
gid_t   Gid;
off_t   Size;
time_t  Atime;
time_t  Mtime;
   
struct stat FileInfo;
fileflag_struct *FileFlags;
struct utimbuf origtime;

/* Open the user's .safedelete.log and read the whole
   thing into storage where we can easily manipulate it.
*/

  wrkptr=NULL;
  j=k=0;
  stat(SafeDelLog, &FileInfo);
  LogSize=FileInfo.st_size;
  LogList=palloc=malloc(LogSize);
  flog=fopen(SafeDelLog, "r");
  fread(LogList, LogSize, 1, flog);
  fclose(flog); 

/* Now search the whole list looking for the most recent
   occurence of the file only if the relative file number is -1.
*/

  if(FileNum == -1)
  {
    j=0;
    NewestMtime=-1;
    while(j < LogSize)
    {
      if(strcmp(FileName, LogList+sizeof(fileflag_struct)) == 0)
      {
        if(GrungeName)
        {
          wrkptr2=LogList+sizeof(fileflag_struct); /* point past file flags to orig filename */
          wrkptr2+=strlen(wrkptr2)+1;  /* point past orig filename to grunged name */
          if(strcmp(GrungeName, wrkptr2))
          {
            k=RecLen(LogList);
            j+=k;
            LogList+=k;
            continue;
          }
          else
          {
            wrkptr=LogList;
            break;
          }
        }
        else
        { 
          wrkptr2=LogList+sizeof(fileflag_struct); /* point past file flags to orig filename */
  	  for(k=0; k < 7; k++)
            wrkptr2+=strlen(wrkptr2)+1;        /* point to original st_mtime */
  	  Mtime=atol(wrkptr2);
          if(Mtime > NewestMtime)
          {
            NewestMtime=Mtime;
            wrkptr=LogList;
          }
        }
      }
      k=RecLen(LogList);
      j+=k;
      LogList+=k;
    }

    if(wrkptr == NULL)
    {
      free(palloc);
      return 1;
    }
  }
  else
  {

/* Find the record number specified by FileNum */

    for(j=0; j < FileNum; j++)
      LogList+=RecLen(LogList);
    wrkptr=LogList;

/* Make sure this is really the filename we're looking for */

    if(strcmp(FileName, wrkptr+sizeof(fileflag_struct)) != 0)
    {
      free(palloc);
      return 1;
    }
  }

/* Set up some work pointers */

  LogList=wrkptr;
  FileFlags=(fileflag_struct*)LogList;
  wrkptr+=sizeof(fileflag_struct);
  wrkptr+=strlen(wrkptr)+1;    /* point to start of grunged filename */
  GrungedFN=wrkptr;
   
  wrkptr2=wrkptr+strlen(wrkptr)+1;  /* start of original inode info */
  Mode=atoi(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;
  Uid=atoi(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;
  Gid=atoi(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;
  Size=atol(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;
  Atime=atol(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;
  Mtime=atol(wrkptr2);
  wrkptr2+=strlen(wrkptr2)+1;  /* leave wrkptr2 pointing past inode info */
   
  if(S_ISREG(Mode) && FileFlags->NoCompressFlag == FLAGOFF)
  {

/* Normal file and it was compressed */

    UncompressCmd=wrkptr2;
    wrkptr2+=strlen(wrkptr2)+1;
    FileSuffix=wrkptr2;
  }
  else
  {

/* File was not normal or it was not compressed */

    UncompressCmd=NULL;
    FileSuffix=NULL;
  }

  if(UndelFlag)
  {

/* We found the record we're looking for.  First make
   sure the grunged file really exists.
*/

    sprintf(WorkBuf, "%s/%s", SafeDir, GrungedFN);
    if(stat(WorkBuf, &FileInfo) != 0)
    {
      free(palloc);
      return 2;
    }

/* If we have an old format entry just tar it back */

    if(FileFlags->Version == '0')
    {
      sprintf(WorkBuf, "tar -xzf '%s/%s' > /dev/null", SafeDir, GrungedFN);
      getcwd(WorkBuf2, WORK_BUF_LEN);
      chdir("/");
      system(WorkBuf);
      chdir(WorkBuf2);
    }
    else
    {

/* Rename the file to its original name */

      i=ParseFileName(LogList+sizeof(fileflag_struct));
      sprintf(WorkBuf2, "%s/%s", SafeDir, LogList+sizeof(fileflag_struct)+i+1);
      rename(WorkBuf, WorkBuf2);

/* If this was a regular file and it was compressed
   uncompress it.
*/

      if(UncompressCmd)
      {
        sprintf(WorkBuf, "%s%s", WorkBuf2, FileSuffix);
        rename(WorkBuf2, WorkBuf);
        sprintf(WorkBuf, "%s '%s%s' > /dev/null 2>&1", UncompressCmd, WorkBuf2, FileSuffix);
        system(WorkBuf);
      }

/* If this is a regular file, copy it to its ultimate
   destination, even if that is a new directory
   or a new filename.
*/
      if(S_ISREG(Mode))
      {
        if(NewName == NULL)
        {
          sprintf(WorkBuf, "cp '%s' '%s' > /dev/null 2>&1", WorkBuf2,
		  LogList+sizeof(fileflag_struct));
          system(WorkBuf);
          strcpy(FileName, LogList+sizeof(fileflag_struct));
        }
        else
        {
          sprintf(WorkBuf, "cp '%s' '%s' > /dev/null 2>&1", WorkBuf2, NewName);
          system(WorkBuf);
          strcpy(FileName, NewName);
        }

/* Reset the permissions, uid, gid, etc of
   the new file.
*/

        chmod(FileName, Mode);
        chown(FileName, Uid, Gid);
        origtime.actime=Atime;
        origtime.modtime=Mtime;
        utime(FileName, &origtime);
      }

/* This is a link, create a new one */

      else
      {
        flink=fopen(WorkBuf2, "r");
        fgets(WorkBuf, WORK_BUF_LEN, flink);
        fclose(flink);

/* If user wants link to have a new name, give it one */

        if(NewName == NULL)
        {
	  unlink(LogList+sizeof(fileflag_struct));   /* make sure there is no existing link */
          symlink(WorkBuf, LogList+sizeof(fileflag_struct));
          strcpy(FileName, LogList+sizeof(fileflag_struct));
        }
        else
        {
	  unlink(NewName);   /* make sure there is no existing link */
          symlink(WorkBuf, NewName);
          strcpy(FileName, NewName);
        } 
      }
    }

/* Make sure that the file was really
   undeleted.
*/

    if(lstat(FileName, &FileInfo) &&
      stat(FileName, &FileInfo))
    {
        free(palloc);
        return 3;
    }

/* If we just overlaid the file, compare
   the size and time stamps to see if
   we really undeleted the file.  If they
   are equal we have to assume that the
   undelete failed for some reason.
*/

    if(CurrInfo)
    {
      if(FileInfo.st_size == CurrInfo->st_size &&
         FileInfo.st_mtime == CurrInfo->st_mtime)
      {
        free(palloc);
        return 3;
      }
    }
    strcpy(WorkBuf2, SafeDir);
    strcat(WorkBuf2, "/");
    j=ParseFileName(LogList+sizeof(fileflag_struct));
    strcat(WorkBuf2, LogList+sizeof(fileflag_struct)+j+1);
  }
  else
  {
    strcpy(WorkBuf2, SafeDir);
    strcat(WorkBuf2, "/");
    strcat(WorkBuf2, GrungedFN);
  }

/* Now that the file is undeleted we need
   to do some cleanup.  Delete the saved
   file from the safedelete directory and
   remove the entry in the user's safedelete
   log.
*/

  unlink(WorkBuf2);

  *(LogList+sizeof(fileflag_struct))='$';   /* Show this file is undeleted */

/* Write the user's log back, minus the
   entry for the file we just undeleted.
*/

  flog=fopen(SafeDelLog, "w");
  j=0;
  LogList=palloc;
  while(j < LogSize)
  {
    k=RecLen(LogList);
    if(*(LogList+sizeof(fileflag_struct)) == '/')
      fwrite(LogList, k, 1, flog);
    j+=k;
    LogList+=k;
  }
  fclose(flog);
  free(palloc);
  return 0;
}

/* ======================================================== */

#ifndef NO_CURSES

/* FillListWin
   Fills in the window containing the list of files currently deleted and
   the date each was deleted.

   Passed: pointer to the list window
           pointer to the file pointer array
           number of files in the FileMatch array
           number of line currently containing selection indicator '>'
           pointer to array of file line numbers
           pointer to array of file numbers matching pattern
           number of the file which occupies the first line in the window

   Returns: number of files listed in the window
*/

void FillListWin(ListWin, FilePtr,
                NumFiles, SelFile, FileLine,
                FileMatch, FirstFile)
  WINDOW *ListWin;
  char *FilePtr[];
  int  NumFiles, *SelFile, FileLine[], FileMatch[], *FirstFile;
{
char WorkBuf[WORK_BUF_LEN];
char *wrkptr, *wrkptr2;
int  i, j, k;
int  FilesInList;

/* Do some initialization work */

  werase(ListWin);
  PartialLastLine=0;
  FilesInList=0;

/* If there are no files to display, just return */

  if(NumFiles <= 0)
    return;

/* FileLine contains the line number in the window where
   a particular filename starts (e.g. FileLine[4]=2 means
   that the 5th filename begins on line 2 in the ListWin
   window).  We use this array for moving the '>' indicator
   up and down.  We initialize the entire array to some
   arbitrary large limit so we can tell when we get to the
   bottom of the window.  This just makes coding the
   scrolling functions easier.
*/

  for(i=0; i < MAX_FILES; i++)
    FileLine[i]=BIG_NUM;

/* If the last file in the list is also in the first line
   of the window and has been undeleted, restart list
   from the beginning.
*/

  if(*FirstFile >= NumFiles)
    *SelFile=*FirstFile=0;

/* Look at each filename and put it into the screen buffer.
   Also split filenames that are longer than what will fit
   on one line.
*/

  j=LIST_BEG;
  for(i=*FirstFile; i < NumFiles; i++)
  {
    FileLine[FileMatch[i]]=j;
    wrkptr=FilePtr[FileMatch[i]]+sizeof(fileflag_struct);
    k=strlen(wrkptr);
    wrkptr2=wrkptr+k+1;
    if(k > 60)
    {

/* Filename longer than one line, write as many full lines
   as we can.
*/

      for(k/=60; k > 0; k--)
      {
        mvwaddnstr(ListWin, j, 3, wrkptr, 60);
        j++;
        if(j > ListEnd)
	{
          PartialLastLine=1;
          break;
	}
        wrkptr+=60;
      }

/* Write the last line of the filename, if it will fit in the window */

      if(j > ListEnd)
        break;
      k=strlen(FilePtr[FileMatch[i]]+sizeof(fileflag_struct))%60;
      mvwaddnstr(ListWin, j, 3, wrkptr, k);
    }

/* Full filename fits on one line, place it there */

    else
      mvwaddnstr(ListWin, j, 3, wrkptr, k);

/* Add the date stamp following the filename */

    strncpy(WorkBuf, wrkptr2+1, 2);
    WorkBuf[2]='/';
    strncpy(WorkBuf+3, wrkptr2+3, 2);
    WorkBuf[5]='/';
    strncpy(WorkBuf+6, wrkptr2+5, 2);
    mvwaddnstr(ListWin, j, 67, WorkBuf, 8);
    j++;
    if(j > ListEnd)
      break;
  }

/* Add a file selection indicator that will move vertically
   along the left side of the file names, provided there are
   any files to display.
*/

  mvwaddch(ListWin, FileLine[FileMatch[*SelFile]], 1, '>');

  return;
}

/* ================================================================ */

/* GetInput

   Reads input from a specified window and places
   the characters entered into the area provided.

   Passed: pointer to the window
           pointer to an area to receive the input
           length of the area
           starting line number of cursor
           starting column number of cursor
           length of line on the screen
           flag indicating whether or not to clear the window
            before displaying it

   Returns: length of the area
*/

int GetInput(window, Area, AreaLen, LineNum, ColNum, LineLen,
             RefreshFlag)
  WINDOW *window;
  char *Area;
  int  AreaLen, LineNum, ColNum, LineLen;
  bool RefreshFlag;
{
static int i = 0;
static int WinLine = 0, WinCol = 0;
int j = 0;
int k;
int MaxCol, MaxChars;
char c;

/* Set maximum column number in the window */

  MaxChars=(AreaLen <= LineLen ? AreaLen : LineLen);
  MaxCol=ColNum+MaxChars-1; 

/* Clear input window to blanks, if needed */

  if(RefreshFlag)
  {
    memset(Area, ' ', AreaLen);
    for(j=ColNum; j <= MaxCol; j++)
      mvwaddch(window, LineNum, j, ' ');

    WinLine=LineNum;
    WinCol=ColNum;
    i=0;
  }

/* Display the window */

  touchwin(window);
  wrefresh(window);

/* Show the cursor */

  curs_set(TRUE);

/* Get input from user */

  while(i <= MaxChars)
  {
    while((j=mvwgetch(window, WinLine, WinCol)) == ERR);

    if(j < KEY_MIN || j > KEY_MAX)
      j&=0xff;

    if(j == 27 || j == '\n')
      break;        /* Leave "while" loop if user hit enter or escape */

    switch(j)
    {
    case KEY_BACKSPACE :
    case '\b' : 
                if(i == 0)
                  continue;

                WinCol--;
                i--;

    case KEY_DC : 
                if(i >= MaxChars-1)
                  continue;
                for(k=i; k < MaxChars-1; k++)
                {
                  c=*(Area+k+1);
                  mvwaddch(window, WinLine, WinCol+(k-i), c);
                  *(Area+k)=c;
                }
                mvwaddch(window, WinLine, MaxCol, ' ');
                *(Area+MaxChars-1)=' ';
                wrefresh(window);
                break;

    case KEY_LEFT :
                if(i == 0)
                  continue;
                WinCol--;
                i--;
                break;

    case KEY_RIGHT :
                if(i >= MaxChars-1)
                  continue;
                WinCol++;
                i++;
                break;
 
    default:
                if(! isprint(j))
		{
                  beep();
                  break; 
                }

                if(*(Area+MaxChars-1) == ' ')
                {
		  for(k=MaxChars-1; k > i; k--)
		  {
		    c=*(Area+k-1);
		    mvwaddch(window, WinLine, WinCol+(k-i), c);
		    *(Area+k)=c;
		  }
                  *(Area+i)=j;
		  mvwaddch(window, WinLine, WinCol, j);
		  wrefresh(window);
                  WinCol++;
                  i++;
                }
                else
                 beep();
                break;
    }
  }

/* Hide the cursor */

  curs_set(FALSE);

  if(j == '\n')
  {
    *(Area+i)='\0';
    return i;
  }
  else
  {
    *Area='\0';
    return 0;
  }
}

#endif

/* ================================================================ */

/* GetFileInfo

   This routine scans an entry from the .safedelete.log
   and creates a series of null-delimited strings 
   containing the information for the file.

   Passed: pointer .safedelete.log entry
           pointer to area to receive strings

   returns: nothing
*/

void GetFileInfo(LogEntry, StringArea, DelDate)
  char *LogEntry, *StringArea;
  int  DelDate;
{
struct passwd *UidInfo;
struct group *GidInfo;
struct tm *tm;

int i, k;
char *wrkptr;
char TimeBuf[24];

char Permissions[8][4];

fileflag_struct *FileFlags;
mode_t Mode;
uid_t  Uid;
gid_t  Gid;
off_t  Size;
time_t Atime;
time_t Mtime;

/* Do some initialization */

  strcpy(Permissions[0], "---");
  strcpy(Permissions[1], "--x");
  strcpy(Permissions[2], "-w-");
  strcpy(Permissions[3], "-w-");
  strcpy(Permissions[4], "r--");
  strcpy(Permissions[5], "r-x");
  strcpy(Permissions[6], "rw-");
  strcpy(Permissions[7], "rwx");

  i=0;
  
/* Get file permissions */

  FileFlags=(fileflag_struct *)LogEntry;
  
  wrkptr=LogEntry+sizeof(fileflag_struct);
  wrkptr+=strlen(wrkptr)+1;       /* Point to grunged filename */
  if(DelDate)
  {
    if(*(wrkptr+5) == '9')
      sprintf(StringArea+i, "Date deleted: %c%c/%c%c/19%c%c", *(wrkptr+1),
	    *(wrkptr+2), *(wrkptr+3), *(wrkptr+4), *(wrkptr+5), *(wrkptr+6));
    else
      sprintf(StringArea+i, "Date deleted: %c%c/%c%c/20%c%c", *(wrkptr+1),
	    *(wrkptr+2), *(wrkptr+3), *(wrkptr+4), *(wrkptr+5), *(wrkptr+6));
    i+=strlen(StringArea+i)+1;
  }
  sprintf(StringArea+i, "Time deleted: %c%c:%c%c:%c%c", *(wrkptr+9),
          *(wrkptr+10), *(wrkptr+11), *(wrkptr+12), *(wrkptr+13),
          *(wrkptr+14));
  i+=strlen(StringArea+i)+1;

  wrkptr+=strlen(wrkptr)+1;       /* Point to original file info */
  
  Mode=atoi(wrkptr);
  wrkptr+=strlen(wrkptr)+1;
  Uid=atoi(wrkptr);
  wrkptr+=strlen(wrkptr)+1;
  Gid=atoi(wrkptr);
  wrkptr+=strlen(wrkptr)+1;
  Size=atol(wrkptr);
  wrkptr+=strlen(wrkptr)+1;
  Atime=atol(wrkptr);
  wrkptr+=strlen(wrkptr)+1;
  Mtime=atol(wrkptr);
  wrkptr+=strlen(wrkptr)+1;  /* Leave wrkptr pointing past file info */
  
  if(FileFlags->Version == '0')
  {
    strcpy(StringArea+i, "Permissions: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Owners login id: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Owners group id: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Size: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Last accessed: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Last modified: (unknown)");
    i+=strlen(StringArea+i)+1;
    *(StringArea+i)='\0';
  }
  else
  {
    strcpy(StringArea+i, "Permissions: -");
    if(S_ISDIR(Mode))
      *(StringArea+i+13)='d';
    if(S_ISLNK(Mode))
      *(StringArea+i+13)='l';
    if(S_ISCHR(Mode))
      *(StringArea+i+13)='c';
    if(S_ISBLK(Mode))
      *(StringArea+i+13)='b';
    if(S_ISFIFO(Mode))
      *(StringArea+i+13)='f';
    if(S_ISSOCK(Mode))
      *(StringArea+i+13)='s';
    k=(Mode>>6)&0x0007;
    strcat(StringArea+i, Permissions[k]);
    k=(Mode>>3)&0x0007;
    strcat(StringArea+i, Permissions[k]);
    k=Mode&0x0007;
    strcat(StringArea+i, Permissions[k]);
    i+=strlen(StringArea+i)+1;

/* Get owner info */

    if((UidInfo=getpwuid(Uid)) == NULL)
      sprintf(StringArea+i, "Owners login id: %d", Uid);
    else
      sprintf(StringArea+i, "Owners login name: %s", UidInfo->pw_name);
    i+=strlen(StringArea+i)+1;

    if((GidInfo=getgrgid(Gid)) == NULL)
      sprintf(StringArea+i, "Owners group id: %d", Gid);
    else
      sprintf(StringArea+i, "Owners group name: %s", GidInfo->gr_name);
    i+=strlen(StringArea+i)+1;

/* Show size of file */

    sprintf(StringArea+i, "Size: %li", Size);
    i+=strlen(StringArea+i)+1;

/* Show the date and time stamps */

    tm=localtime(&Atime);
    strftime(TimeBuf, 18, "%m/%d/%Y %H:%M", tm);
    sprintf(StringArea+i, "Last accessed: %s", TimeBuf);
    i+=strlen(StringArea+i)+1;

    tm=localtime(&Mtime);
    strftime(TimeBuf, 18, "%m/%d/%Y %H:%M", tm);
    sprintf(StringArea+i, "Last modified: %s", TimeBuf);
    i+=strlen(StringArea+i)+1;

/* Get symbolic link info, if needed */

    if(FileFlags->LinkFlag == FLAGON)
    {
      strcpy(StringArea+i, "Link target: ");
      strcat(StringArea+i, wrkptr);
      i+=strlen(StringArea+i)+1;
    }

/* Add terminating null */
 
    *(StringArea+i)='\0';
  }
}

/* ================================================================ */

/* ShowFileInfo
   This routine goes through the user's .safedelete.log looking
   for the specified filename.  When found, a variety of file information
   is displayed.

   Passed: pointer to the user's .safedelete.log
           pointer to the filename or NULL
           pointer to the grunged name or NULL

   Returns: nothing
*/

void ShowFileInfo(SafeDelLog, FileName, GrungeName)
  char *SafeDelLog, *FileName, *GrungeName;
{
struct stat FileInfo;

FILE *flog;

char *LogList, *palloc, *wrkptr1, *wrkptr2;
char WorkBuf[WORK_BUF_LEN];

int  k, GotFile;
long j;

/* First we read the entire .safedelete.log into memory so
   it's easy to manipulate.
*/

  stat(SafeDelLog, &FileInfo);
  LogList=palloc=malloc(FileInfo.st_size);
  flog=fopen(SafeDelLog, "r");
  fread(LogList, FileInfo.st_size, 1, flog);
  fclose(flog);

/* Now that we've got the entire log in memory, just
   go through it looking for the file the user wants.
*/

  j=GotFile=0;
  while(j < FileInfo.st_size)
  {
    k=RecLen(LogList);    /* get length of this record */
    wrkptr1=LogList+sizeof(fileflag_struct);

    if((FileName == NULL && PatternMatch(wrkptr1) == 0) || 
       strcmp(FileName, wrkptr1) == 0)
    {

/* If a grunged filename was passed make sure we get the log
   entry which contains it */

      if(GrungeName)
      {
        wrkptr2=wrkptr1+strlen(wrkptr1)+1;
        if(strcmp(GrungeName, wrkptr2))
        {     
          j+=k;
          LogList+=k;
          continue;
        }
      }

      GotFile=1;

/* Show file name */

      printf("\n%s\n", wrkptr1);
      
      GetFileInfo(LogList, WorkBuf, YES_MSG);

/* Show file information */

      wrkptr2=WorkBuf;
      while(*wrkptr2)
      {
        if(strlen(wrkptr2) > 0)
	{
          printf("\t");
          printf(wrkptr2);
          printf("\n");
	}
        wrkptr2+=strlen(wrkptr2)+1;
      }
    }
    j+=k;
    LogList+=k;
  }

/* Issue message if file wasn't found */

  if(! GotFile)
    fprintf(stderr, "undelete: %s: not found in .safedelete.log\n",
            FileName);

  free(palloc);
}

/* ================================================================ */

/* ListFiles

   This routine reads the .safedelete.log and displays
   the names of the files that are currently safely deleted.
   If a particular filename is given, only those entries that
   match it are listed.  If FileFlag is non-zero, the
   safedeleted filename is also listed.

   Passed: pointer to the SafeDelete log filename
           pointer to the filename to look for or NULL
           flag - 1=list safedeleted filename  0=list original filename only
           size of SafeDelete log (in bytes)
*/

void ListFiles(SafeDelLog, FileName, FileFlag, FileSize)
  char *SafeDelLog, *FileName;
  int  FileFlag;
  off_t FileSize;
{
char *wrkptr, *wrkptr2, *palloc;
int  j, k, GotIt;
FILE *flog;

  GotIt=0;

/* Read the .safedelete.log and list the entries */

  wrkptr=palloc=malloc(FileSize);
  flog=fopen(SafeDelLog, "r");
  fread(wrkptr, FileSize, 1, flog);
  fclose(flog);
  j=k=0;
  while(j < FileSize)
  {
    k=RecLen(wrkptr);

/* If filename passed, we need an exact match on it */

    if(FileName != NULL)
    {
      if(strcmp(FileName, wrkptr+sizeof(fileflag_struct)) != 0)
	goto NextFile;
    }
    else
    {

/* Check the filename pattern instead */

      if(PatternMatch(wrkptr+sizeof(fileflag_struct)))
        goto NextFile;
    }
    GotIt=1;
    printf("%s", wrkptr+sizeof(fileflag_struct));
    if(FileFlag)
    {
      wrkptr2=wrkptr+sizeof(fileflag_struct);
      wrkptr2+=strlen(wrkptr2)+1;
      printf("\t%s", wrkptr2);
    }
    printf("\n");
  NextFile:
    wrkptr+=k;
    j+=k;
  }
  free(palloc);
  if(! GotIt)
  {
    if(FileName)
      printf("undelete: %s: not found in .safedelete.log\n",
             FileName);
    else
      printf("undelete: no files found matching the pattern\n");
  }
}

/* ================================================================ */

/* SortLog

   This routine performs a quick sort of the
   FilePtr array.  I had to write my own because
   the qsort function apparently doesn't understand
   the concept of indirection and can't sort strings
   by just swapping pointers in an array.
*/

void SortLog(FilePtr, left, right)
  char *FilePtr[];
  int left, right;
{
int i, j;
char *wrkptr, *temp;

  i=left;
  j=right;
  wrkptr=FilePtr[(left+right)/2]+sizeof(fileflag_struct);

  do
  {
    while(strcmp(FilePtr[i]+sizeof(fileflag_struct), wrkptr) < 0 && i < right) i++;
    while(strcmp(FilePtr[j]+sizeof(fileflag_struct), wrkptr) > 0 && j > left) j--;
    if(i <= j)
    {
      temp=FilePtr[i];
      FilePtr[i]=FilePtr[j];
      FilePtr[j]=temp;
      i++;
      j--;
    }
  } while(i <= j);

  if(left < j)
    SortLog(FilePtr, left, j);

  if(i < right)
    SortLog(FilePtr, i, right);
}

/* =========================================================== */

/* ProcessFile

   This routine just checks the permissions of the
   directory and the file and undeletes it if everything
   checks out OK.

   Passed: pointer to filename
           pointer to safedelete directory
           pointer to safedelete log
           pointer to safedelete rc file
           pointer to grunged name or NULL

   Returns: 0 - file undeleted
            1 - file not undeleted
*/

int  ProcessFile(FileName, SafeDir, SafeDelLog, SafeDelRC, GrungeName)
  char *FileName, *SafeDir, *SafeDelLog, *SafeDelRC, *GrungeName;
{
struct stat FileInfo;
int i, j, ccode;

  ccode=3;   /* Set default return code */

/* Make sure we have write access to the directory. */

  j=ParseFileName(FileName);
  FileName[j]='\0';

  if(CheckPerms(FileName, NULL, "undelete", YES_MSG) != 0)
      return 1;
  else
    FileName[j]='/';

/* Now see if the file exists.  If it does, ask the user if they 
   want to overlay it.
*/

  if(lstat(FileName, &FileInfo) == 0 ||
     stat(FileName, &FileInfo) == 0)
  {
    if(! ReplaceFlag)
    {
      printf("undelete: %s: file already exists, overlay it? (y|n) ",FileName);
      while((i=getchar()) == '\n');
      if(i != 'y' && i != 'Y')
        return 1;
    }
    ccode=UndeleteFile(FileName, SafeDir, SafeDelLog, SafeDelRC, NULL, &FileInfo,
                       -1, TRUE, GrungeName);
  }
  else
    ccode=UndeleteFile(FileName, SafeDir, SafeDelLog, SafeDelRC, NULL, NULL,
                       -1, TRUE, GrungeName);

  switch(ccode)
  {
    case 0 : printf("undelete: %s: successfully undeleted\n",
                   FileName);
             break;

    case 1 : fprintf(stderr,
             "undelete: %s: not found in .safedelete.log\n",
             FileName);
             break;

    case 2 : fprintf(stderr,
             "undelete: safedelete file containing %s not found\n",
             FileName);
             break;

    case 3 : fprintf(stderr,
	     "undelete: undelete failed, reason unknown\n");
	     break;

    case 4 : fprintf(stderr,
 	     "undelete: file %s is busy, try again later\n", SafeDelLog);
             break;
		    
    default : break;
  }
  return (ccode == 0 ? 0 : 1);  
}

/* ================================================================ */

/* ResetMatches

   This routine resets the FileMatch array
   by matching each file in the FilePtr array
   to the file pattern.

   Passed: pointer to the file pointer array
           pointer to the array of file numbers matching the pattern
           total number of files in the file pointer array

   Returns: number of files matching the pattern
*/

int  ResetMatches(FilePtr, FileMatch, TotalFiles)
  char *FilePtr[];
  int  FileMatch[], TotalFiles;
{
int  i, j;
/* Find which files match the pattern */

  for(i=0, j=0; i < TotalFiles; i++)
  {
    FileMatch[j]=BIG_NUM;
    if(PatternMatch(FilePtr[i]+sizeof(fileflag_struct)) == 0)
      FileMatch[j++]=i;
  }
  return j;
}

/* ================================================================ */

/* Main processing routine

   This routine parses the command
   line looking for any options or
   arguments.  If any are found, they
   are processed accordingly.
*/
   
int  main(argc, argv)
  int argc;
  char **argv;
{
char FileName[FILE_NAME_LEN];
char GrungeName[FILE_NAME_LEN];
char SafeDir[SAFE_DIR_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDelRC[SAFE_RC_LEN];

struct passwd *UserInfo;


int  i, j, k, ccode;
char *LoginName, *HomeDir, *wrkptr;
char *palloc = NULL;


int  InfoFlag, ListFlag, FileFlag, BadOptFlag;
int  AllFlag, PatternFlag, GrungeFlag;

struct stat FileInfo;
FILE *flog;


static struct option LongOpts[] =
{
    {"info", no_argument, NULL, 'i'},
    {"list", no_argument, NULL, 'l'},
    {"file", no_argument, NULL, 'f'},
    {"all",  no_argument, NULL, 'a'},
    {"replace", no_argument, NULL, 'r'},
    {"grunge", no_argument, NULL, 'g'},
    {"pattern", required_argument, NULL, 'p'},
    {"subdir", no_argument, NULL, 's'},
    {0     , 0          , 0   ,   0}
};

#ifndef NO_CURSES

fileflag_struct *FileFlags;

char NewFileName[FILE_NAME_LEN];
char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];
char *FilePtr[MAX_FILES+1];
char c;

int  FileLine[MAX_FILES+1];
int  FileMatch[MAX_FILES+1];
int  TotalFiles, NumFiles, SelFile, SelLimit, FirstFile;

struct tm *tm;
struct stat NewInfo;

bool NotDone = TRUE;
bool RefreshWin;

/* ListWin is the window containing the scrollable
   list of files.  BackWin is the background window.
*/

WINDOW *MainWin;
WINDOW *ListWin, *BackWin, *VerifyWin;
WINDOW *InfoWin, *BackInfoWin;
WINDOW *WarnWin, *NewnameWin;

#endif

/* Do some preliminary initializtion */

  strcpy(FileNamePattern, "*");
  strcpy(CmdName, "undelete");
  
  HomeDir=getenv("HOME");
  if(HomeDir == NULL) 
  {
    LoginName=getlogin();
    UserInfo=getpwnam(LoginName);
    HomeDir=UserInfo->pw_dir;
  }
  strcpy(SafeDelLog, HomeDir);
  strcat(SafeDelLog, LOG_NAME);
  strcpy(SafeDir, HomeDir);
  strcat(SafeDir, SAFE_DIR_NAME);
  strcpy(SafeDelLockFile, HomeDir);
  strcat(SafeDelLockFile, LOCK_NAME);

  if((wrkptr=getenv("SAFEDELRC")) != NULL)
    strcpy(SafeDelRC, wrkptr);
  else
  {
    strcpy(SafeDelRC, HomeDir);
    strcat(SafeDelRC, RC_NAME);
  }

/* Check permissions of safedelete directory and log */

  if(CheckPerms(SafeDir, SafeDelLog, CmdName, YES_MSG) != 0)
    return 1;

/* Get lock on .safedelete.log */
  
  if((i=AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, 0)))
  {
    fprintf(stderr, "%s: lock file is busy, try again later\n", CmdName);
    return 1;
  }
  
/* Get info on .safedelete.log, we'll need it later */

  ccode=stat(SafeDelLog, &FileInfo);
  if(ccode == 0 && FileInfo.st_size > 0)
  {
    palloc=malloc(FileInfo.st_size);
    flog=fopen(SafeDelLog, "r");
    fread(palloc, FileInfo.st_size, 1, flog);
    fclose(flog);
    if(! isdigit(*palloc))
    {
      ReleaseLock();
      printf("%s: your .safedelete.log is still in the old format\n",
             CmdName);
      printf("%s: it will be converted using the safecnvt command\n",
             CmdName);
      system("safecnvt");

/* Now that it's converted, let's just double check */

      if((i=AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, 0)))
      {
        fprintf(stderr, "%s: lock file is busy, try again later\n", CmdName);
	free(palloc);
        return 1;
      }
      
      flog=fopen(SafeDelLog, "r");
      fread(palloc, FileInfo.st_size, 1, flog);
      fclose(flog);
      if(! isdigit(*palloc))
      {
        printf("%s: conversion failed, cannot undelete any files\n",
               CmdName);
	ReleaseLock();
        free(palloc);
        return 1;
      }
      else
        printf("%s: your .safedelete.log was successfully converted\n",
               CmdName);
    }
    free(palloc);
  }

/* Process any options found on the command line */

  InfoFlag=0;
  ListFlag=0;
  FileFlag=0;
  AllFlag=0;
  BadOptFlag=0;
  PatternFlag=0;
  GrungeFlag=0;

  while((i=getopt_long(argc, argv, "ilfarg:p:s", LongOpts, &j)) != -1)
  {
    switch(i)
    {
      case 'i': InfoFlag=1;
                break;

      case 'l' : ListFlag=1;
                 break;

      case 'f' : FileFlag=1;
                 break;

      case 'a' : AllFlag=1;
                 break;

      case 'r' : ReplaceFlag=1;
                 break;

      case 's' : SubdirFlag=1;
                 break;

      case 'g' : GrungeFlag=1;
                 strcpy(GrungeName, optarg);
                 break;

      case 'p' : PatternFlag=1;
                 if((wrkptr=strchr(optarg, '*')))
		 {
                   if(strchr(wrkptr+1, '*'))
                   {
                     printf("%s: invalid pattern specified, pattern ignored\n",
                            CmdName);
                     break;              /* ignore if more than one wildcard char */
                   }
		 }

                 if(strlen(optarg) > 255)
                 {
                   printf("%s: pattern exceeds max length, pattern ignore\n",
                          CmdName);
                   break;                /* ignore it if it's too long */
                 }

/* If user specified relative directory, make it
   relative to current directory.
*/

                 if(strchr(optarg, '/') && *optarg != '/')
                 {
                   if(! getcwd(FileNamePattern, 256))
                   {
                     printf("%s: unable to get current directory, pattern ignored\n",
                            CmdName);
                     break;
                   }
                   if(strlen(FileNamePattern)+strlen(optarg)+2 > 255)
                   {
                     printf("%s: pattern exceeds max length, pattern ignored\n",
                            CmdName);
                     strcpy(FileNamePattern, "*");
                     break;              /* ignore if it's too long */
                   }
                   if(strncmp(optarg, "./", 2) == 0)
                     strcat(FileNamePattern, optarg+1);
                   else
                   {
                     strcat(FileNamePattern, "/");
                     strcat(FileNamePattern, optarg);
                   }
                 }
                 else
                   strcpy(FileNamePattern, optarg);

/* Parse the pattern looking for any wildcard chars */

                 if((wrkptr=strchr(FileNamePattern, '*')))
                 {
                   if(*FileNamePattern == '*')
                   {
                     PatternType=SUFFIX;
                     PatternSuffixLen=strlen(FileNamePattern)-1;
                     break;
                   }
                   if(*(FileNamePattern+strlen(FileNamePattern)-1) == '*')
                   {
                     PatternType=PREFIX;
                     PatternPrefixLen=strlen(FileNamePattern)-1;
                     break;
                   }
                   PatternType=INFIX;
                   PatternPrefixLen=(int)(wrkptr-FileNamePattern);
                   PatternSuffixLen=strlen(FileNamePattern)-PatternPrefixLen-1;

/* If there is a slash ('/') in the second part
   of the pattern (following the '*') turn on
   subdirectory checking.
*/

                   if(strchr(FileNamePattern+PatternPrefixLen+1, '/'))
                     SubdirFlag=1;
                   break;
                 }
                 else
                   PatternType=FULLNAME;
                 break;
                 
      case '?' :
      default  : BadOptFlag=1;
                 break;
    }
  }

/* If a bad option was found, don't go any further */

  if(BadOptFlag)
  {
    ReleaseLock();
    return 1;
  }

/* Make sure we have compatible options */

  if((AllFlag || SubdirFlag) && ! PatternFlag)
  {
    fprintf(stderr, "%s: a pattern is needed with the -a and -s options\n",
            CmdName);
    ReleaseLock();
    return 1;
  }
  
  if(FileFlag && ! ListFlag)
  {
    fprintf(stderr, "%s: must use the -l option with -f\n", CmdName);
    ReleaseLock();
    return 1;
  }

/* The following code is done only when no
   filename is given on the command line.
*/

  if(argc == optind)
  {

/* See if user is requesting information on all files
   (user issued:  undelete -i)
*/

    if(InfoFlag)
    {
      if(FileInfo.st_size < 30)
      {
        fprintf(stderr, "%s: no files to undelete\n", CmdName);
        ReleaseLock();
        return 1;
      }
      ShowFileInfo(SafeDelLog, NULL, NULL);
      ReleaseLock();
      return 0;
    }

/* See if user just wants a list of the files displayed
   (user issued: undelete -l or undelete -lf)
*/

    if(ListFlag)
    {
      if(FileInfo.st_size == 0)
      {
        fprintf(stderr, "%s: no files to undelete\n", CmdName);
        ReleaseLock();
        return 1;
      }
      ListFiles(SafeDelLog, NULL, FileFlag, FileInfo.st_size); 
      ReleaseLock();
      return 0;
    }

/* See if user wants all files matching the pattern
   are to be undeleted.
*/

    if(AllFlag)
    {
      if(! PatternFlag)
      {
        fprintf(stderr, "%s: -a option used without pattern\n", CmdName);
        ReleaseLock();
        return 1;
      }
      wrkptr=palloc=malloc(FileInfo.st_size);
      flog=fopen(SafeDelLog, "r");
      fread(palloc, FileInfo.st_size, 1, flog);
      fclose(flog);
      i=j=k=0;
      while(j < FileInfo.st_size)
      {
	k=RecLen(wrkptr);
        if(PatternMatch(wrkptr+sizeof(fileflag_struct)) == 0)
        {
          i++;
          if(ProcessFile(wrkptr+sizeof(fileflag_struct), SafeDir, SafeDelLog, SafeDelRC, NULL))
	    break;
        }
        j+=k;
        wrkptr+=k;
      }
      free(palloc);
      if(! i)
      {
        fprintf(stderr, "%s: no files found matching the pattern\n",
                CmdName);
        ReleaseLock();
        return 1;
      }
      else
      {
        ReleaseLock();
        return 0;
      }
    }
  }

/* See if a filename was specified on the command line.
   If so, see if it is an absolute path or a relative path.  If
   relative, prefix it with the current directory path.
*/

  if(argc > optind)
  {
    if(PatternFlag)
    {
      fprintf(stderr, "%s: filename not allowed with pattern\n",
              CmdName);
      ReleaseLock();
      return 1;
    }

    wrkptr=argv[optind];

    if(*wrkptr == '/')
      strcpy(FileName, wrkptr);
    else
    {
      getcwd(FileName, 256);
      if(*wrkptr == '.')
      {
        while(*wrkptr == '.')
	{

/* See if user specified ./ */

           if(strncmp(wrkptr, "./", 2) == 0)
	   {
              wrkptr+=2;
              continue;
	   }

/* See if user specified ../ */

           if(strncmp(wrkptr, "../", 3) == 0)
	   {
             wrkptr+=3;
             j=ParseFileName(FileName);
             if(j == 0)
               strcpy(FileName, "/");
             else
               FileName[j]='\0';
             continue;
	   }

/* None of the above, must be a hidden file */

           break;
	}
        strcat(FileName, "/");
        strcat(FileName, wrkptr);
      }
      else
      {
        strcat(FileName, "/");
        strcat(FileName, wrkptr);
      }
    }

/* At this point we have a fully qualified path and filename
   (whether provided by the user or built above).   See if the
   user is only requesting information on it.
*/

    if(InfoFlag)
    {
      if(GrungeFlag)
        ShowFileInfo(SafeDelLog, FileName, GrungeName);
      else
        ShowFileInfo(SafeDelLog, FileName, NULL);
      ReleaseLock();
      return 0;
    }

/* See if the user just wants the entries listed */

    if(ListFlag)
    {
      ListFiles(SafeDelLog, FileName, FileFlag, FileInfo.st_size);
      ReleaseLock();
      return 0;
    }

/* Process the filename and undelete it if we can */

    if(GrungeFlag)
      ccode=ProcessFile(FileName, SafeDir, SafeDelLog, SafeDelRC, GrungeName);
    else
      ccode=ProcessFile(FileName, SafeDir, SafeDelLog, SafeDelRC, NULL);
    ReleaseLock();
    return ccode;

  }
  else
  {

#ifdef NO_CURSES

/* No file name was provided so we can't do anything */

    printf("%s: no filename specified\n", CmdName);
    ReleaseLock();
    return 1;
  }

#else

/* No file name was provided on the command line.
   First see if any files can be undeleted.  If there are,
   show them on the screen and let the user choose
   which file(s) to undelete.
*/

    ccode=stat(SafeDelLog, &FileInfo);
    if(ccode != 0 || FileInfo.st_size == 0)
    {
      fprintf(stderr, "%s: no files to undelete\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* To make the list of files easier to read, we
   first sort them into alphabetic sequence.
   Then we copy the sorted entries back into the
   original log file.
*/

    printf("\nSorting entries, please wait...\n");
    wrkptr=palloc=malloc(FileInfo.st_size);
    flog=fopen(SafeDelLog, "r");
    fread(wrkptr, FileInfo.st_size, 1, flog);
    fclose(flog);

    TotalFiles=j=0;
    while(j < FileInfo.st_size)
    {
      k=RecLen(wrkptr);  /* get length of this record */
      FilePtr[TotalFiles++]=wrkptr;
      if(TotalFiles > MAX_FILES)
      {
        fprintf(stderr, "\nMore than %d entries found, unable to perform sort\n", 
                MAX_FILES);
        fprintf(stderr, "Press any key to continue...\n");
        getchar();
        break;
      }
      j+=k;
      wrkptr+=k;
    }

/* If we have more than one record, sort them 
   and rewrite the .safedelete.log in sorted order
*/
    
    if(TotalFiles > 1 && TotalFiles <= MAX_FILES)
    {
      SortLog(FilePtr, 0, TotalFiles-1);

      flog=fopen(SafeDelLog, "w");
      for(i=0; i < TotalFiles; i++)
      {
	k=RecLen(FilePtr[i]);
        fwrite(FilePtr[i], k, 1, flog);
      }
      fclose(flog);
    }

    if((NumFiles=ResetMatches(FilePtr, FileMatch, TotalFiles)) == 0)
    {
      fprintf(stderr, "%s: no files found matching the pattern\n",
              CmdName);
      goto AllDone;
    }

/* Initialize the ncurses screen */

    if(!(MainWin=initscr()))
    {
      fprintf(stderr, "%s: initscr failed\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* We need at least a 24x80 screen to do our thing */

    if((LINES < 24) || (COLS < 80))
    {
      endwin();
      fprintf(stderr,
              "%s: screen too small, need 24x80, this screen only %dx%d\n",
              CmdName, LINES, COLS);
      ReleaseLock();
      return 1;
    }

/* Set the end of the file list window */

    ListEnd=LINES-8;

/* Create the background window */

    if(! (BackWin=subwin(MainWin, LINES-1, 80, BACK_ROW, BACK_COL)))
    {
      endwin();
      fprintf(stderr, "%s: can't create background window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* Create the file list window */

    if(! (ListWin=subwin(BackWin, LINES-7, 78, LIST_ROW, LIST_COL)))
    {
      endwin();
      fprintf(stderr, "%s: can't create list window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* Create the verify window */

    if(! (VerifyWin=subwin(MainWin, 1, 80, BACK_ROW+(LINES-1), BACK_COL)))
    {
      endwin();
      fprintf(stderr, "%s: can't create verify window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* Create the information background window */

    if(! (BackInfoWin=newwin(11, 42, BACK_ROW+7, BACK_COL+19)))
    {
      endwin();
      fprintf(stderr, "%s: can't create info background window\n", CmdName); 
      ReleaseLock();
      return 1;
    }

/* Create informataion window */

    if(! (InfoWin=subwin(BackInfoWin, 9, 40, BACK_ROW+8, BACK_COL+20)))
    {
      endwin();
      fprintf(stderr, "%s: can't create information window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* Create warning window */

    if(! (WarnWin=newwin(3, 64, BACK_ROW+10, BACK_COL+8)))
    {
      endwin();
      fprintf(stderr, "%s: can't create warning window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* Create new name window */

    if(! (NewnameWin=newwin(5, 64, BACK_ROW+10, BACK_COL+8)))
    {
      endwin();
      fprintf(stderr, "%s: can't create newname window\n", CmdName);
      ReleaseLock();
      return 1;
    }

/* See if we can use colors.  If so, set them up.
   If not, then it could be a vt100 or xterm so we
   use black & white with various attributes.
*/

    if(has_colors())
    {
      start_color();
      init_pair(1, COLOR_WHITE, COLOR_BLUE);
      init_pair(2, COLOR_WHITE, COLOR_BLACK);
      init_pair(3, COLOR_WHITE, COLOR_RED);
      wbkgdset(BackWin, COLOR_PAIR(1));
      wattrset(BackWin, COLOR_PAIR(1));
      wbkgdset(ListWin, COLOR_PAIR(2));
      wattrset(ListWin, COLOR_PAIR(2));
      wbkgdset(VerifyWin, COLOR_PAIR(3));
      wattrset(VerifyWin, COLOR_PAIR(3));
      wbkgdset(BackInfoWin, COLOR_PAIR(1));
      wattrset(BackInfoWin, COLOR_PAIR(1));
      wbkgdset(InfoWin, COLOR_PAIR(1));
      wattrset(InfoWin, COLOR_PAIR(1));
      wbkgdset(WarnWin, COLOR_PAIR(3));
      wattrset(WarnWin, COLOR_PAIR(3));
      wbkgdset(NewnameWin, COLOR_PAIR(1));
      wattrset(NewnameWin, COLOR_PAIR(1));
    }
    else
    {
      wattrset(BackWin, A_NORMAL);
      wattrset(ListWin, A_BOLD);
      wattrset(VerifyWin, A_BOLD);
      wattrset(BackInfoWin, A_NORMAL);
      wattrset(InfoWin, A_BOLD);
      wattrset(WarnWin, A_BOLD);
      wattrset(NewnameWin, A_NORMAL);
    }

/* Clear the windows to blanks */

    werase(BackWin);
    werase(ListWin);
    werase(VerifyWin);
    werase(BackInfoWin);
    werase(NewnameWin);

/* Draw some borders and lines in the various windows */

    wborder(BackWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);

    wmove(BackWin, 2, 0);
    whline(BackWin, ACS_HLINE, 80);
    mvwaddch(BackWin, 2, 0, ACS_LTEE);
    mvwaddch(BackWin, 2, 79, ACS_RTEE);

    wmove(BackWin, LINES-4, 0);
    whline(BackWin, ACS_HLINE, 80);
    mvwaddch(BackWin, LINES-4, 0, ACS_LTEE);
    mvwaddch(BackWin, LINES-4, 79, ACS_RTEE);

    wborder(BackInfoWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
    mvwaddstr(BackInfoWin, 10, 7, " Press any key to return ");

    wborder(WarnWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);

    wborder(NewnameWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
    mvwaddstr(NewnameWin, 1, 2, "Enter new file/path name:");
    mvwaddstr(NewnameWin, 4, 11, " Press Enter to accept or Esc to return ");


/* Now add the headings */

    mvwaddstr(BackWin, 1, 4, "File Name");
    mvwaddstr(BackWin, 1, 66, "Date Deleted");
    wmove(BackWin, LINES-3, 4);
    waddch(BackWin, 'Q'|A_BOLD);
    waddstr(BackWin, "uit     ");
    waddch(BackWin, 'U'|A_BOLD);
    waddstr(BackWin, "p     ");
    waddch(BackWin, 'D'|A_BOLD);
    waddstr(BackWin, "own     ");
    waddch(BackWin, 'S'|A_BOLD);
    waddstr(BackWin, "elect     ");
    waddch(BackWin, 'R'|A_BOLD);
    waddstr(BackWin, "emove     ");
    waddch(BackWin, 'A'|A_BOLD);
    waddstr(BackWin, "ll     ");
    waddch(BackWin, 'I'|A_BOLD);
    waddstr(BackWin, "nfo     ");
    waddch(BackWin, 'N'|A_BOLD);
    waddstr(BackWin, "ewname");

/* Put the .safedelete.log entries in the ListWin window */

    SelFile=FirstFile=0;
    FillListWin(ListWin, FilePtr, NumFiles, &SelFile, FileLine,
                FileMatch, &FirstFile);

/* Terminal options:
   don't echo the user-typed characters, don't wait for
   input, make cursor invisible, don't wait for carriage
   return while waiting for input and enable the keypad.
*/

    noecho();
    nodelay(MainWin, FALSE);
    cbreak();
    curs_set(FALSE);
    keypad(MainWin, TRUE);
    keypad(InfoWin, TRUE);
    keypad(WarnWin, TRUE);
    keypad(NewnameWin, TRUE);
    wtimeout(InfoWin, 0);
    wtimeout(NewnameWin, 0);

/* Display the screen images */

    move(0, 0);
    touchwin(BackWin);
    wnoutrefresh(BackWin);
    touchwin(ListWin);
    wnoutrefresh(ListWin);
    touchwin(VerifyWin);
    wnoutrefresh(VerifyWin);
    doupdate(); 

/* Get input from the user and process it */

    while(NotDone)
    {
      i=getch();
      if(i < KEY_MIN || i > KEY_MAX)
        i&=255;
      switch(i)
      {
       case KEY_UP :
       case 'u' :
       case 'U' : werase(VerifyWin);
                 move(0, 0);
                 if(AllFlag)
		 {
                   AllFlag=0;
                   mvwaddstr(VerifyWin, 0, 5, 
                             "Mass mode OFF");
                 }
                 wrefresh(VerifyWin);
                 if(SelFile <= 0)
                   break;
                 mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, ' ');
                 SelFile--;
                 if(SelFile < FirstFile)
		 {
                   FirstFile--;
                   FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                               FileLine, FileMatch, &FirstFile);
		 }
                 else
		   mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, '>');
                 move(0, 0);
                 wrefresh(ListWin);
                 break;

       case KEY_DOWN :
       case 'd' :
       case 'D' : werase(VerifyWin);
                 move(0, 0);
                 if(AllFlag)
		 {
                   AllFlag=0;
                   mvwaddstr(VerifyWin, 0, 5, 
                             "Mass mode OFF");
                 }
                 wrefresh(VerifyWin);

/* If the last line in the window contains a partial filename,
   we just scroll down 1 filename to show the complete
   filename of that file.  We leave the '>' on the same filename.
   We do this only for the last filename in the list.
*/

                  if(SelFile == NumFiles-1)
		  {
                    while(PartialLastLine)
		    {
                      mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, ' ');
                      FirstFile++;
                      FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                  FileLine, FileMatch, &FirstFile);
                      mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, '>');
		    }
		  }
                  else
		  {
                    mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, ' ');
                    SelFile++;
                    if(FileLine[FileMatch[SelFile]] > ListEnd)
		    {
                      while(PartialLastLine)
		      {
                        FirstFile++;
                        FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                    FileLine, FileMatch, &FirstFile);
                        mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, ' ');
		      }
                      FirstFile++;
                      FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                  FileLine, FileMatch, &FirstFile);
		    }
                    else
                      mvwaddch(ListWin, FileLine[FileMatch[SelFile]], 1, '>');
		  }
                  move(0, 0);
                  wrefresh(ListWin);
                  break;

       case 'a' :
       case 'A' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;
                  if(AllFlag)
                  {
                    AllFlag=0;
                    mvwaddstr(VerifyWin, 0, 5, 
                              "Mass mode OFF");
                  }
                  else
                  {
                   AllFlag=1;
                   mvwaddstr(VerifyWin, 0, 5,
                             "Mass mode ON!!  Select or Remove ALL listed files");
                  }
                  move(0, 0);
                  wrefresh(VerifyWin);
                  break;

       case 's' :
       case 'S' : if(! AllFlag)
                  {
                    werase(VerifyWin);
                    move(0, 0);
                    wrefresh(VerifyWin);
                  }
                  if(SelFile < 0)
                    break;

/* If user is undeleting everything, make sure this is
   what they want
*/

                  if(AllFlag)
		  {
                    touchwin(WarnWin);
                    mvwaddstr(WarnWin, 1, 1,
                    " You are about to undelete all listed files, continue? (y|n)  ");
                    move(0, 0);
                    wrefresh(WarnWin);

                    c=getch();

                    move(0, 0);
                    touchwin(ListWin);
                    wrefresh(ListWin);

                    if(c != 'y' && c != 'Y')
                      break;

                    SelFile=0;
                    SelLimit=NumFiles;
		  }
                  else
                    SelLimit=1;

	          k=0;
                  for(i=0; i < SelLimit; i++)
		  {
                    k=0;
                    ccode=0;

/* wrkptr points to the fully qualified filename.
   Make sure we have write access to the directory
   where the file is going to be placed in.
*/

                    wrkptr=FilePtr[FileMatch[SelFile]]+sizeof(fileflag_struct);
                    j=ParseFileName(wrkptr);
                    *(wrkptr+j)='\0';
                    if(CheckPerms(wrkptr, NULL, CmdName, NO_MSG) != 0)
                    {
                      sprintf(WorkBuf, "Cannot write to directory %s", wrkptr);
                      *(WorkBuf+75)='\0';
                      mvwaddstr(VerifyWin, 0, 5, WorkBuf);
                      move(0, 0);
                      wrefresh(VerifyWin);
                      *(wrkptr+j)='/';
                      k++;
                      break;
                    }
                    else
                     *(wrkptr+j)='/';

/* See if the file already exists.  If so, see
   if the user wants to overlay it provided they
   didn't also specify the -r or --replace options.
*/

                    if(lstat(wrkptr, &FileInfo) == 0 ||
                       stat(wrkptr, &FileInfo) == 0)
	  	    {
                      if(! ReplaceFlag)
                      {
                        FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                    FileLine, FileMatch, &FirstFile);
			wnoutrefresh(ListWin);
			
			werase(VerifyWin);
                        sprintf(WorkBuf, "Processing %s", wrkptr);
                        *(WorkBuf+75)='\0';
                        mvwaddstr(VerifyWin, 0, 5, WorkBuf);
                        wnoutrefresh(VerifyWin);
 
                        touchwin(WarnWin);
                        mvwaddstr(WarnWin, 1, 1,
                        "            File already exists, overlay it? (y|n)            ");
                        move(0, 0);
                        wrefresh(WarnWin);

                        c=getch();

                        move(0, 0);
                        touchwin(ListWin);
                        wrefresh(ListWin);

                        if(c != 'y' && c != 'Y')
			{
			  if(AllFlag)
			  {
			    SelFile++;
			    continue;
			  }
                          k++;
                          break;
			}
                      }

    		      ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                         NULL, &FileInfo,
                                         FileMatch[SelFile], TRUE, NULL);
  		    }
                    else
                      ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                         NULL, NULL,
                                         FileMatch[SelFile], TRUE, NULL);

                    if(AllFlag)
		    {

/* If an error occurs while in "mass mode" we want to process the
   return code from UndeleteFile so just leave the loop after
   redrawing the screen
*/

		      if(ccode)
		      {
                        FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                    FileLine, FileMatch, &FirstFile);
                        move(0, 0);
                        wrefresh(ListWin);
                        break;
		      }
		      
                      for(j=FileMatch[SelFile]; j < TotalFiles; j++)
                        FilePtr[j]=FilePtr[j+1];

                      NumFiles=ResetMatches(FilePtr, FileMatch, --TotalFiles);
		    }
		  }

                  if(k)
		  {
                    FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                FileLine, FileMatch, &FirstFile);
                    move(0, 0);
                    wrefresh(ListWin);
                    break;   /* just skip the rest if k > 0 */
		  }

                  if(AllFlag && ccode == 0)
	          {
                    werase(VerifyWin);
		    if(SelFile)
		    {
		      mvwaddstr(VerifyWin, 0, 5,
				"Selected files were successfully undeleted");
		      SelFile=FirstFile=0;
		    }
		    else
		    {
                      mvwaddstr(VerifyWin, 0, 5,
                                "All files were successfully undeleted");
                      SelFile=NumFiles=FirstFile=0;
		    }
                    wnoutrefresh(VerifyWin);
                    FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                FileLine, FileMatch, &FirstFile);
                    move(0, 0);
                    wrefresh(ListWin);
		    AllFlag=0;
                    break;
		  }

	          werase(VerifyWin);
                  switch(ccode)
		  {
                   case 0 : if(lstat(wrkptr, &FileInfo) || ! S_ISLNK(FileInfo.st_mode))
                              stat(wrkptr, &FileInfo);
                            tm=localtime(&FileInfo.st_mtime);
                            strftime(WorkBuf+80, 11, "%m/%d/%Y", tm);
                            sprintf(WorkBuf,
                            "File undeleted   size: %li   last modified: %s",
                            FileInfo.st_size, WorkBuf+80);
                            mvwaddstr(VerifyWin, 0, 5, WorkBuf);

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                            for(i=FileMatch[SelFile]; i < TotalFiles; i++)
                              FilePtr[i]=FilePtr[i+1];

                            ResetMatches(FilePtr, FileMatch, --TotalFiles);

                            NumFiles--;
                            if(SelFile >= NumFiles)
                              SelFile--;

/* Now refresh the list window with the new list */

                            FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                        FileLine, FileMatch, &FirstFile);
                            move(0, 0);
                            wrefresh(ListWin);
		            AllFlag=0;
                            break;

		   case 1:  mvwaddstr(VerifyWin, 0, 5,
                                      "File not found in .safedelete.log");
                            AllFlag=0;		            
                            break;

		   case 2 : mvwaddstr(VerifyWin, 0, 5,
                                      "Safedelete file not found, undelete failed");
                            AllFlag=0;		            
                            break;

		   case 3 : mvwaddstr(VerifyWin, 0, 5,
			              "Unknown error, file not undeleted");
                            AllFlag=0;		            
		            break;

                  default : break;

		  }
                  move(0, 0);
                  wrefresh(VerifyWin);
                  break;

       case 'n' :
       case 'N' : werase(VerifyWin);
                  move(0, 0);
                  if(AllFlag)
                  {
                    AllFlag=0;
                    mvwaddstr(VerifyWin, 0, 5, 
                              "Mass mode OFF");
                  }
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;

	          FileFlags=(fileflag_struct *)FilePtr[FileMatch[SelFile]];
                  wrkptr=FilePtr[FileMatch[SelFile]]+sizeof(fileflag_struct);
                  if(FileFlags->Version == '0')
		  {
                    touchwin(WarnWin);
                    mvwaddstr(WarnWin, 1, 1,
                    " Entry in old format, cannot rename (press any key to return) ");
                    move(0, 0);
                    wrefresh(WarnWin);

                    c=getch();

                    move(0, 0);
                    touchwin(ListWin);
                    wrefresh(ListWin);
                    break;
		  }

                  RefreshWin=TRUE;

/* Loop till user enters valid filename or presses F3 */

                  c='n';
                  while(c != 'y')
		  {

/* Get new filename or path(+filename) from user */

                    j=GetInput(NewnameWin, WorkBuf2, FILE_NAME_LEN, 3, 2, 60,
                               RefreshWin);

                    move(0, 0);
                    touchwin(ListWin);
                    wnoutrefresh(ListWin);
                    werase(VerifyWin);
                    doupdate();
                    RefreshWin=FALSE;

                    strcpy(NewFileName, WorkBuf2);

                    if(j > 0)
                    {

/* See if user specified ~/ for directory */

                      if(*NewFileName == '~')
                      {
                        strcpy(NewFileName, HomeDir);
                        strcat(NewFileName, WorkBuf2+1);
                      }

/* If user didn't enter full path, prefix it with original directory path */

                      if(*NewFileName != '/')
                      {
                        j=ParseFileName(wrkptr);
                        strncpy(NewFileName, wrkptr, j+1);
                        strcpy(NewFileName+j+1, WorkBuf2);
                      }

/* Get status of file */

                      if(lstat(NewFileName, &NewInfo) == 0 ||
                         stat(NewFileName, &NewInfo) == 0)
		      {
		      
/* If it's really a directory, append the original filename to it */
		      
                        if(S_ISDIR(NewInfo.st_mode))
                        {
                          j=ParseFileName(wrkptr);
                          strcat(NewFileName, wrkptr+j);
                        }
		      }
			
/* Check permissions of directory path */
			
                      j=ParseFileName(NewFileName);
                      *(NewFileName+j)='\0';
                      ccode=CheckPerms(NewFileName, NULL, CmdName, NO_MSG);
		      switch(ccode)
		      {
			  case CP_DIR_NF : 
 		 	  case CP_NOT_DIR :
		                          mvwaddstr(VerifyWin, 0, 5,
                                                    "Invalid directory specified");
                                          move(0, 0);
                                          wrefresh(VerifyWin);
                                          break;
			  
			  case CP_DIR_NOWRITE :
                                          mvwaddstr(VerifyWin, 0, 5,
                                           "Error getting write access to the directory");
                                          move(0, 0);
                                          wrefresh(VerifyWin);
			                  break;
			  
 			  default : break;
	  	      }
			
                      *(NewFileName+j)='/';

		      if(ccode) continue;

/* See if new file already exists */

                      if(lstat(NewFileName, &FileInfo) == 0 ||
                         stat(NewFileName, &FileInfo) == 0)
  		      {
                        touchwin(WarnWin);
                        mvwaddstr(WarnWin, 1, 1,
                        "          New file already exists, overlay it? (y|n)          ");
                        move(0, 0);
                        wrefresh(WarnWin);

                        c=wgetch(WarnWin);
 
                        move(0, 0);
                        touchwin(ListWin);
                        wrefresh(ListWin);

                        if(c != 'y' && c != 'Y')
                          continue;

		        ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                           NewFileName, &FileInfo,
                                           FileMatch[SelFile], TRUE, NULL);
		      }
                      else
		      {
                        c='y';
                        ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                           NewFileName, NULL,
                                           FileMatch[SelFile], TRUE, NULL);
		      }

		      werase(VerifyWin);
                      switch(ccode)
                      {
                        case 0 : if(lstat(NewFileName, &FileInfo) || ! S_ISLNK(FileInfo.st_mode))
                                   stat(NewFileName, &FileInfo);
                                 tm=localtime(&FileInfo.st_mtime);
                                 strftime(WorkBuf+80, 11, "%m/%d/%Y", tm);
                                 sprintf(WorkBuf,
                                 "File undeleted   size: %li   last modified: %s",
                                 FileInfo.st_size, WorkBuf+80);
                                 mvwaddstr(VerifyWin, 0, 5, WorkBuf);

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                                 for(i=FileMatch[SelFile]; i < TotalFiles; i++)
                                   FilePtr[i]=FilePtr[i+1];

                                 ResetMatches(FilePtr, FileMatch, --TotalFiles);

                                 NumFiles--;
                                 if(SelFile >= NumFiles)
                                   SelFile--;

/* Now refresh the list window with the new list */

                                 FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                            FileLine, FileMatch, &FirstFile);
                                 move(0, 0);
                                 wrefresh(ListWin);
                                 break;

  		        case 1 : mvwaddstr(VerifyWin, 0, 5,
                                           "File not found in .safedelete.log");
                                 break;

  		        case 2 : mvwaddstr(VerifyWin, 0, 5,
                                           "Safedelete file not found, undelete failed");
                                 break;

		        case 3 : mvwaddstr(VerifyWin, 0, 5,
			                   "Unknown error, file not undeleted");
		                 break;

		        case 4 : mvwaddstr(VerifyWin, 0, 5,
					   "Log file busy, try again later");
			         break;
			
                       default : break;

		      }
                      move(0, 0);
                      wrefresh(VerifyWin);
		    }
                    else
                      c='y';
		  }
                  break;

       case 'r' :
       case 'R' : if(! AllFlag)
                  {
                    werase(VerifyWin);
                    move(0, 0);
                    wrefresh(VerifyWin);
                  }
                  if(SelFile < 0)
                    break;

/* Make sure the user REALLY wants to remove this entry */

                  mvwaddstr(WarnWin, 1, 1,
                  " Cannot undelete file after entry is removed, proceed? (y|n)  ");

                  move(0, 0);
                  touchwin(WarnWin);
                  wrefresh(WarnWin);

                  c=getch();

                  move(0, 0);
                  touchwin(ListWin);
                  wrefresh(ListWin);

                  if(c != 'y' && c != 'Y')
                    break;

/* Remove all entries if "all mode" is on */

                  if(AllFlag)
                  {
                    for(j=0; j < NumFiles; j++)
                    {
                      wrkptr=FilePtr[FileMatch[0]]+sizeof(fileflag_struct);
                      UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC, NULL,
                                   NULL, FileMatch[0], FALSE, NULL);
/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                       for(i=FileMatch[0]; i < TotalFiles; i++)
                         FilePtr[i]=FilePtr[i+1];

                       ResetMatches(FilePtr, FileMatch, --TotalFiles);
                    }
                    SelFile=NumFiles=FirstFile=0;
                    FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                FileLine, FileMatch, &FirstFile);
                    move(0, 0);
                    wnoutrefresh(ListWin);
                    werase(VerifyWin);
                    mvwaddstr(VerifyWin, 0, 5,
                              "Entries removed from .safedelete.log");
                    move(0, 0);
                    wrefresh(VerifyWin);
                    break;
                  }
                  else
                  {
                    wrkptr=FilePtr[FileMatch[SelFile]]+sizeof(fileflag_struct);

/* Call UndeleteFile to just remove the entry without undeleting the file */

                    ccode=UndeleteFile(wrkptr, 
				       SafeDir, SafeDelLog, SafeDelRC, NULL,
                                       NULL, FileMatch[SelFile], FALSE, NULL);
                  }

                  switch(ccode)
		  {
		    case 0 : mvwaddstr(VerifyWin, 0, 5,
                                      "Entry removed from .safedelete.log");

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                            for(i=FileMatch[SelFile]; i < TotalFiles; i++)
                              FilePtr[i]=FilePtr[i+1];

                            ResetMatches(FilePtr, FileMatch, --TotalFiles);

                            NumFiles--;
                            if(SelFile >= NumFiles)
                              SelFile--;

/* Now refresh the list window with the new list */

                            FillListWin(ListWin, FilePtr, NumFiles, &SelFile,
                                        FileLine, FileMatch, &FirstFile);
                            move(0, 0);
                            wrefresh(ListWin);
                            break;

	            case 1: mvwaddstr(VerifyWin, 0, 5,
                                      "File not found in .safedelete.log");
                           break;

		   case 4 : mvwaddstr(VerifyWin, 0, 5,
				      "Log file busy, try again later");
		            break;

		  default : werase(VerifyWin);
                           break;
		  }
                  move(0, 0);
                  wrefresh(VerifyWin);
                  break;

	case 'i' :
       	case 'I' : werase(VerifyWin);
                  move(0, 0);
                  if(AllFlag)
                  {
                    AllFlag=0;
                    mvwaddstr(VerifyWin, 0, 5, 
                              "Mass mode OFF");
                  }
                  wrefresh(VerifyWin);
                  AllFlag=0;
                  if(SelFile < 0)
                    break;

                  GetFileInfo(FilePtr[FileMatch[SelFile]], WorkBuf, NO_MSG);

                  werase(InfoWin);

                  wrkptr=WorkBuf;
                  for(i=1; i < 8; i++)
                  {
                    if(strlen(wrkptr) > 38)
		    {
                      *(wrkptr+38)='\0';
                      mvwaddstr(InfoWin, i, 1, wrkptr);
                      *(wrkptr+38)='a';
		    }
                    else
                      mvwaddstr(InfoWin, i, 1, wrkptr);
                    wrkptr+=strlen(wrkptr)+1;
                  }

                  move(0, 0);
                  touchwin(BackInfoWin);
                  wnoutrefresh(BackInfoWin);
                  wnoutrefresh(InfoWin);
                  doupdate();

/*
                  do
		  {
                    j=getch();
                    if(j < KEY_MIN || j > KEY_MAX)
                      j&=255;
		  }
                  while(j != KEY_F(3) && j != 27);
 */
                  getch();

                  move(0, 0);
                  touchwin(ListWin);
                  wrefresh(ListWin);
                  break;

        case 'q' :
        case 'Q' : NotDone=FALSE;
                  break;

        default : break;
      }
    }

/* Clear the screen and delete the windows */

    wattrset(ListWin, A_NORMAL);
    wattrset(BackWin, A_NORMAL);
    wattrset(VerifyWin, A_NORMAL);

    werase(ListWin);
    werase(BackWin);
    werase(VerifyWin);

    wrefresh(ListWin);
    wrefresh(BackWin);
    wrefresh(VerifyWin);

    delwin(ListWin);
    delwin(BackWin);
    delwin(VerifyWin);
    delwin(NewnameWin);
    delwin(WarnWin);
    delwin(InfoWin);
    delwin(BackInfoWin);

    curs_set(TRUE);
    endwin();
   
    system("clear");

  AllDone:
    free(palloc);
  }

  ReleaseLock();
  return 0;

#endif /* NO_CURSES */

}
