/* safedelete.c
  This program will make a compressed copy of a file and save it
  in a user-specified location and then remove it from the original
  location.  This will allow a file to be "undeleted" if the user so
  wishes.
*/

#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <time.h>
#include <stdio.h>
#include <pwd.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <dirent.h>
#include <errno.h>
#include "safedelete.h"

/* main routine

   Parses the command line parameters and safedeletes
   all the files listed.

   Passed: list of filenames

   Returns: 0 - all files safedelete successfully
            1 - one or more files not safedeleted
*/

int  main(argc, argv)
  int argc;
  char **argv;
{
int  i, j, k, randnum, ccode;
int  SafeDays, RetCode, RecursionDone;
int  MsgInd, DontSave;
int  BaseEntityIsDir;

char *EnvValue, *LoginName, *HomeDir, c;
char *wrkptr, *palloc;

char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];
char LogRec[WORK_BUF_LEN];
char FileName[FILE_NAME_LEN];
char GrungedFileName[GRUNGED_FILE_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDir[SAFE_DIR_LEN];
char SafeDelRC[SAFE_RC_LEN];
char SafeDelLockFile[DEL_LOG_LEN];

char CmdName[16];

char CompressCmd[41], UncompressCmd[41];
char FileSuff[9];

char *InfoLoc;

FILE *gfp;

time_t currtime;
struct tm *tm;
struct passwd *UserInfo;
struct stat DirInfo, FileInfo, GrungeInfo;

rcstruct *CompRClist, *SuffRClist, *DaysRClist;
safedays_struct *days;
compression_struct *compress;
suffix_struct *suffix;
fileflag_struct FileFlags;   
struct dir_struct StartDir, *CurrentDir, *PreviousDir;

int flog;

static int RecursiveFlag, InteractiveFlag, VerboseFlag, ForceFlag;

static struct option LongOpts[] =
{
#ifdef LINUX
    {"interactive", no_argument, &InteractiveFlag,   1},
    {"verbose"    , no_argument, &VerboseFlag    ,   1},
    {"force"      , no_argument, NULL            , 'f'},
    {"directory"  , no_argument, NULL            , 'd'},
    {"recursive"  , no_argument, NULL            , 'r'},
    {"help"       , no_argument, NULL            ,  98},
    {"version"    , no_argument, NULL            ,  99},
    {0            , 0          , 0               ,   0}
#endif
#ifdef AIX
    {"version"    , no_argument, NULL            ,  99},
    {0            , 0          , 0               ,   0}
#endif
#ifdef HPUX
    {"version"    , no_argument, NULL            ,  99},
    {0            , 0          , 0               ,   0}
#endif
#ifdef SUNOS 
    {"version"    , no_argument, NULL            ,  99},
    {0            , 0          , 0               ,   0}
#endif
};

/* Initialize some variables and see
   if user specified any options.
*/

  strcpy(CmdName, "safedelete");
  RecursiveFlag=InteractiveFlag=VerboseFlag=0;
  ForceFlag=0;
  RetCode=ccode=0;
  MsgInd=1;
  DaysRClist=NULL;
  CompRClist=NULL;
  SuffRClist=NULL;
  days=NULL;
  compress=NULL;
  suffix=NULL;
  WorkBuf2[0]='\0';
  CurrentDir=PreviousDir=NULL;

#ifdef LINUX
  while((i=getopt_long(argc, argv, "dfivrR", LongOpts, &j)) != -1)
#endif
#ifdef AIX
  while((i=getopt_long(argc, argv, "firRe", LongOpts, &j)) != -1)
#endif
#ifdef HPUX
  while((i=getopt_long(argc, argv, "firR", LongOpts, &j)) != -1)
#endif
#ifdef SUNOS
  while((i=getopt_long(argc, argv, "fir", LongOpts, &j)) != -1)
#endif
  {
    switch(i)
    {
#ifdef LINUX
	case 98  : printf("\nSafeDelete accepts the same options as the rm command\n");
                   printf("See the following for further details\n\n");
                   ccode=system("/bin/rm --help");
                   return ccode;
#endif

	case 99  : printf("SafeDelete version 1.3b\n");
#ifdef LINUX
                   ccode=system("/bin/rm --version");
                   return ccode;
#else
                   return 0;
#endif

#ifdef LINUX
	case 'd' :
#endif
#ifdef SUNOS
        case 'r' : RecursiveFlag=1;
                   break; 
#else
	case 'r' :
	case 'R' : RecursiveFlag=1;
                   break;
#endif

	case 'i' : InteractiveFlag=1;
                   break;

	case 'f' : ForceFlag=1;
                   MsgInd=0;
                   break;

#ifdef AIX
	case 'e' : VerboseFlag=1;
		   break;
#endif
#ifdef LINUX
	case 'v' : VerboseFlag=1;
                   break;
#endif

	case '?' : if(isprint(optopt))
                     fprintf(stderr, "%s: unknown option '%c'\n",
                             CmdName, optopt);
                   else
                     fprintf(stderr, "%s: unknown option '\\x%x'\n",
                             CmdName, optopt);
                   return 1;
	 default : break;
    }
  }

/* See if user specified a filename */

  if(optind == argc)
  {
    printf("%s: no filename specified\n", CmdName);
    return 1;
  }

/* See if the variable $SAFEDAYS exists.  If not, default it to MAX_SAFE_DAYS.
   If it does exists, and it is zero, just call /bin/rm to delete the file.
*/

  EnvValue=getenv("SAFEDAYS");
  if(EnvValue == NULL)
    SafeDays=MAX_SAFE_DAYS;
  else
    SafeDays=atoi(EnvValue);

  if(SafeDays == 0)
  {
    strcpy(WorkBuf, "/bin/rm ");
    for(i=1; i < argc; i++)
    {
      strcat(WorkBuf, argv[i]);
      strcat(WorkBuf, " ");
    }
    ccode=system(WorkBuf);
    return ccode;
  }

/* Do some more initialization work
   before going any further
*/

  HomeDir=getenv("HOME");
  if(HomeDir == NULL)
  {
    LoginName=getlogin();
    UserInfo=getpwnam(LoginName);
    HomeDir=UserInfo->pw_dir;
  }
  strcpy(SafeDelLog, HomeDir);
  strcat(SafeDelLog, LOG_NAME);

  if((wrkptr=getenv("SAFEDELRC")) != NULL)
    strcpy(SafeDelRC, wrkptr);
  else
  {
    strcpy(SafeDelRC, HomeDir);
    strcat(SafeDelRC, RC_NAME);
  }

  strcpy(SafeDelLockFile, HomeDir);
  strcat(SafeDelLockFile, LOCK_NAME);

/* If the .safedelete.log already exists, make 
   sure we can write to it.
*/

  if(stat(SafeDelLog, &DirInfo) == 0)
    if(CheckPerms(NULL, SafeDelLog, CmdName, YES_MSG) != 0)
      return 1;

/* Initialize the random number generator */

  srand(time(0)+getpid());

/* set up directory for deleted files */

  strcpy(SafeDir, HomeDir);
  strcat(SafeDir, SAFE_DIR_NAME);
  
/* Now see if the actual directory exists and create
   it if needed.
*/

  if((ccode=CheckPerms(SafeDir, NULL, CmdName, YES_MSG)))
  {
    if(ccode == CP_DIR_NF)
    {     
      if(mkdir(SafeDir, S_IRWXU))
      {
        fprintf(stderr, "%s: error creating %s\n", CmdName, SafeDir);
        return 1;
      }
    }
    else
      return 1;
  }

/* Get lock on .safedelete.log */

  if(AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, 0))
  {
    printf("%s: log file %s is busy, try again later\n", CmdName, SafeDelLog);
    return 1;
  }
  
/* If the users .safedelete.log has data in it, see
   if it's already in the format we need.  If it's not,
   invoke savecnvt to convert it.
*/

  ccode=stat(SafeDelLog, &FileInfo);
  if(ccode == 0 && FileInfo.st_size > 0)
  {
    palloc=malloc(FileInfo.st_size);
    flog=open(SafeDelLog, O_RDONLY);
    read(flog, palloc, FileInfo.st_size);
    close(flog);
    
    if(! isdigit(*palloc))
    {
      ReleaseLock();
      
      printf("%s: your .safedelete.log is still in the old format\n", CmdName);
      printf("%s: it will be converted using the safecnvt command\n", CmdName);
      system("safecnvt");

/* Now that it's converted, let's just double check */

      if(AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, 0))
      {
        printf("%s: log file %s is busy, try again later\n", CmdName, SafeDelLog);
	free(palloc);
        return 1;
      }
      
      flog=open(SafeDelLog, O_RDONLY);
      read(flog, palloc, FileInfo.st_size);
      close(flog);
      
      if(! isdigit(*palloc))
      {
        printf("%s: conversion failed, cannot safely delete any files\n", CmdName);
        ReleaseLock();    
        free(palloc);
        return 1;
      }
      else
        printf("%s: your .safedelete.log was successfully converted\n",
               CmdName);
    }
    free(palloc);
  }
  else
  {
    
/* Create the user's .safedelete.log */
    
    if((flog=creat(SafeDelLog, S_IRUSR | S_IWUSR)) == -1)
    {
      sprintf(WorkBuf, "%s: unable to create %s, errno=%d", 
	      CmdName, SafeDelLog, errno);
      perror(WorkBuf);
      ReleaseLock();
      return 1;
    }
    close(flog);
  }
  
/* Read the users .Safedelrc file */

  DaysRClist=(rcstruct*)ReadRC(SafeDelRC, "safedays");
  CompRClist=(rcstruct*)ReadRC(SafeDelRC, "compression");
  SuffRClist=(rcstruct*)ReadRC(SafeDelRC, "suffix");


/* Open the user's log file so we can append to it */
  
  if((flog=open(SafeDelLog, O_WRONLY | O_APPEND)) == -1)
  {
    sprintf(WorkBuf, "%s: error opening %s", CmdName, SafeDelLog);
    perror(WorkBuf);
    ReleaseLock();
    return 1;
  }

/* At this point we have the number of days to
   keep each file and the directory where they are
   to be kept.  Now go through the filenames on
   the command line and process them.
*/

  for(i=optind; i < argc; i++)
  {

/* Set up .Safedelrc structure pointers */

     if(DaysRClist)
       days=DaysRClist->DaysStruct;
     if(CompRClist)
       compress=CompRClist->CompStruct;
     if(SuffRClist)
       suffix=SuffRClist->SuffStruct;

/* See if user specified absolute path name or relative path name.
   If relative, prefix the user-specified path name with $PWD.
*/

     if(argv[i][0] == '/')
      strcpy(FileName, argv[i]);
     else 
     { 

/* Filename does not start with /, build fully qualified filename */

       getcwd(FileName, 256);
       if(argv[i][0] == '.')
       {
         wrkptr=argv[i];
         while(*wrkptr == '.')
         {

/* See if user specified ./ */
	   
           if(strncmp(wrkptr, "./", 2) == 0)
	   {
             wrkptr+=2;
             continue;
	   }
	   
/* See if user specified ../ */

	   if(strncmp(wrkptr, "../", 3) == 0)
	   {
	     wrkptr+=3;
             j=ParseFileName(FileName);
             if(j == 0)
	       strcpy(FileName, "/");
	     else
	       FileName[j]='\0';
             continue;
           }

/* None of the above, must be a hidden file */

           break;
         }
         strcat(FileName, "/");
	 strcat(FileName, wrkptr);
       }
       else
       {
         strcat(FileName, "/");
         strcat(FileName, argv[i]);
       }
     }

/* See if user specified '/' as the last character
   and delete it
*/

     if(*(FileName+strlen(FileName)-1) == '/')
       *(FileName+strlen(FileName)-1)='\0';

/* Set up structures for recursive deletes */

     StartDir.BaseDir=1;
     StartDir.FilesLeft=0;
     StartDir.PrevDir=NULL;
     strcpy(StartDir.DirName, FileName);
     PreviousDir=CurrentDir=&StartDir;

/* If user specified -r/-R/-d, we need to descend
   if the current entity is a directory
*/

     RecursionDone=1;
     BaseEntityIsDir=0;

     if(lstat(FileName, &FileInfo) || ! S_ISLNK(FileInfo.st_mode))
       stat(FileName, &FileInfo);
     
     if(RecursiveFlag)
     {
       if(S_ISDIR(FileInfo.st_mode))
       {
         BaseEntityIsDir=1;
         RecursionDone=0;

/* We have a directory, open it */

         StartDir.dirptr=opendir(FileName);
       }
     }

/* Now process the command line entries */

    do
    {
       DontSave=0;
       if(RecursiveFlag && BaseEntityIsDir)
       {

/* Read an entry from the directory */

         if((CurrentDir->dirent=readdir(CurrentDir->dirptr)) != NULL)
         {
           if(strcmp(CurrentDir->dirent->d_name, ".") == 0 ||
             strcmp(CurrentDir->dirent->d_name, "..") == 0)
             continue;
           strcpy(FileName, CurrentDir->DirName);
           strcat(FileName, "/");
           strcat(FileName, CurrentDir->dirent->d_name);
         }
         else
         {

/* Readdir returned null, this directory is empty.
   Close it then remove it if there are no files remaining in it.
*/

           closedir(CurrentDir->dirptr);
           if(! CurrentDir->FilesLeft)
	   {
	     j=ParseFileName(CurrentDir->DirName);
	     *(CurrentDir->DirName+j)='\0';
	     if(CheckPerms(CurrentDir->DirName, NULL, CmdName, MsgInd) == 0)
	     {
	       *(CurrentDir->DirName+j)='/';
               rmdir(CurrentDir->DirName);
	     }
	   }
           else
             fprintf(stderr, "%s: %s: not empty, not deleted\n",
                     CmdName, CurrentDir->DirName);

/* If we are at the initial directory, don't  free this area of memory */

           if(CurrentDir->BaseDir == 1)
             RecursionDone=1;
           else
	   {

/* Chain back to previous directory and free this directory structure */

             PreviousDir=CurrentDir->PrevDir;
             free((char *)CurrentDir);
             CurrentDir=PreviousDir;
	   }
           continue;
	 }

/* Check permissions if it's a directory */

         if(lstat(FileName, &FileInfo) || ! S_ISLNK(FileInfo.st_mode))
           stat(FileName, &FileInfo);

         if(S_ISDIR(FileInfo.st_mode))
         {
           if(CheckPerms(FileName, NULL, CmdName, MsgInd) == 0)
           {

/* Get a new directory structure and fill it in */

             CurrentDir=(struct dir_struct *)malloc(sizeof(StartDir));
             CurrentDir->BaseDir=0;
             CurrentDir->FilesLeft=0;
             CurrentDir->PrevDir=PreviousDir;
             PreviousDir=CurrentDir;
             strcpy(CurrentDir->DirName, FileName);

/* Open the directory */

             CurrentDir->dirptr=opendir(FileName);
           }
	    
/* We don't have write access to the directory, skip it */

           else
           {
             RetCode=1;
             CurrentDir->FilesLeft++;
           }
           continue;
	 }
       }

/* At this point FileName contains a fully qualified file name */

/* Check permissions of file */

      if((ccode=CheckPerms(NULL, FileName, CmdName, MsgInd)) != 0)
      {

/* If we don't have read access to the file, we can still 
   remove it, but won't have a safedeleted copy of it.
*/

        if(ccode == CP_FILE_NOREAD)
	{
          DontSave=1;
          RetCode=1;
        }
	else
        {
	  RetCode=1;
	  continue;
	}     
      }

/* Now check for write access to the file, if we don't have
   it examine ForceFlag and issue a verification message if
   the flag is off
*/

       if(S_ISREG(FileInfo.st_mode) && access(FileName, W_OK) != 0 && ! ForceFlag)
       {
         sprintf(WorkBuf2, "%o", FileInfo.st_mode);
         sprintf(WorkBuf, "%s: %s: override mode '%s'?",
                 CmdName, FileName, WorkBuf2+2);
         printf(WorkBuf);

         while((c=getchar()) == '\n');

         if(c != 'y' && c != 'Y')
         {
           CurrentDir->FilesLeft++;
           continue;
         }
       }

/* Check permissions of directory */

       j=ParseFileName(FileName);
       *(FileName+j)='\0';

       if(CheckPerms(FileName, NULL, CmdName, MsgInd) != 0)
       {
         RetCode=1;
         CurrentDir->FilesLeft++;
         continue;
       }

       *(FileName+j)='/';


/* If the user specified interactive mode, ask for permission
   to safedelete the file.
*/

       if(InteractiveFlag)
       {
         printf("%s: remove '%s'? ", CmdName, FileName);
         c=getchar();                 /* get first character of response */
         while(getchar() != '\n');  /* ignore everything else except the enter key */
         if(c != 'y' && c != 'Y')
           continue;
       }

/* See if this file should be safedeleted by examining
   the SafeDays section of  the users .Safedelrc file.
   If the return code is 0, indicating the number of safe
   days for this file is 0, we just delete the file and go
   on to the next one.
*/

       k=SafeDays;
       if(DaysRClist && (days=DaysRClist->DaysStruct))
       { 
         for(j=0; j < DaysRClist->NumDaysEnts; j++)
         {
           if((k=CheckFileDays(FileName, days)) == -1)
             days++;
           else
             break;
         }
       }
       if(k == 0)
       {
         if(VerboseFlag)
           printf("%s: %s\n", CmdName, FileName);
         unlink(FileName);
         continue;
       }

/* Create a "grunged" file name with the following format:
   d<gregorian date>.t<time of day>.r<random number>

   Example:  d040595.t112715.r882719
*/

       currtime=time(0);
       tm=localtime(&currtime);
       strftime(WorkBuf, 7, "%m%d%y", tm);
       strcpy(GrungedFileName, "d");
       strcat(GrungedFileName, WorkBuf);
       strftime(WorkBuf, 7, "%H%M%S", tm);
       strcat(GrungedFileName, ".t");
       strcat(GrungedFileName, WorkBuf);
       strcat(GrungedFileName, ".r");

/* Eliminate the possibility of two files having the same 
   grunged filename.
*/
      
       ccode=0;
       while(! ccode)
       {
         randnum=rand();
         sprintf(WorkBuf, "%d", randnum);
         WorkBuf[6]='\0';
         strcpy(GrungedFileName+17, WorkBuf);
	 sprintf(WorkBuf, "%s/%s", SafeDir, GrungedFileName);
	 ccode=stat(WorkBuf, &GrungeInfo);
       }

/* Update the users $HOME/.safedelete.log with the following info:

       Version of safedelete that processed this file
       Flags describing file (16 flags total)
       Original file name (fully qualified)
       Grunged file name (contains date/time file was deleted)
       File permissions
       UID of owner
       GID of owner
       size of file (in bytes)
       date/time file was last accessed
       date/time file was last modified
       contents of symbolic link, if file is a soft link
       uncompress command, if file was compressed
       compress command file suffix, if file was compressed

   We do this so the undelete command will work.  Without the
   log file, nothing can be undeleted.

   All of the above fields are character.  This is to
   make it easier to use the log file on multiple platforms.
   It results in a larger log file, but greater portability.
   Each item is a null terminated field.
*/

       InfoLoc=LogRec+sizeof(FileFlags);
       strcpy(InfoLoc, FileName);
       InfoLoc+=strlen(InfoLoc)+1;
       strcpy(InfoLoc, GrungedFileName);
       InfoLoc+=strlen(InfoLoc)+1;

/* Before going any further, we need to get some information
   from the file's inode.

   We only work with regular files and links.  Anything
   else we leave to the usual /bin/rm command.
*/

       if(! S_ISREG(FileInfo.st_mode) && ! S_ISLNK(FileInfo.st_mode))
       {
         if(VerboseFlag)
	 {
           fprintf(stderr, "%s: only regular files and symlinks can be safedeleted\n",
                   CmdName);
           fprintf(stderr, "%s: file %s skipped\n", CmdName, FileName);
	 }
         CurrentDir->FilesLeft++;
         continue;
       }

/* Save original file inode information */

       for(wrkptr=(char *)&FileFlags, k=0; k < sizeof(fileflag_struct); k++)
         *(wrkptr+k)=FLAGOFF;
       FileFlags.Version=VERSION;
       
       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%d", FileInfo.st_mode);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;

       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%d", FileInfo.st_uid);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;
	
       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%d", FileInfo.st_gid);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;
	
       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%ld", FileInfo.st_size);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;
	
       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%ld", FileInfo.st_atime);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;
	
       memset(WorkBuf, '\0', 16);
       sprintf(WorkBuf, "%ld", FileInfo.st_mtime);
       strcpy(InfoLoc, WorkBuf);
       InfoLoc+=strlen(InfoLoc)+1;

/* If file is really a symbolic link, read it and save for use by undelete */

       if(S_ISLNK(FileInfo.st_mode))
       {
	 FileFlags.LinkFlag=FLAGON;
         if((j=readlink(FileName, WorkBuf2, FileInfo.st_size)) > 0)
         {
           *(WorkBuf2+j)='\0';

/* Write out the contents of the symbolic link as the file */

           k=ParseFileName(FileName);
           sprintf(WorkBuf, "%s/%s", SafeDir, FileName+k+1);
           if(! DontSave)
           {
             gfp=fopen(WorkBuf, "w");
             fwrite(WorkBuf2, j, 1, gfp);
             fclose(gfp);
           }
         }
	 else
	 {
	   if(MsgInd)
	      fprintf(stderr, "%s: %s: unable to read symbolic link\n", CmdName, FileName);
           RetCode=1;
	   CurrentDir->FilesLeft++;
	   continue;
         }
       }

/* Only regular files are compressed (not symbolic links).
   Use gzip for the default compress command
   and .gz as the default suffix.
*/

       if(S_ISREG(FileInfo.st_mode))
       {
	 
/* Set up the defaults */
	 
         strcpy(CompressCmd, "gzip");
         strcpy(UncompressCmd, "gunzip");
         strcpy(FileSuff, ".gz");

/* Check the filename's pattern against what the
   user specified in the .Safedelrc file, if it exists.
*/

         if(CompRClist && (compress=CompRClist->CompStruct))
         {
           for(j=0; j < CompRClist->NumCompressEnts; j++)
           {

/* See if this file should use a particular compression routine */

             if(CheckFilePattern(FileName, compress) == 0)
             {
               strcpy(CompressCmd, compress->CompCmd);
               strcpy(UncompressCmd, compress->UncompCmd);
               if(strcmp(CompressCmd, "none") != 0)
               {
                 if(SuffRClist && (suffix=SuffRClist->SuffStruct))
                 {
                   for(k=0; k < SuffRClist->NumSuffixes; k++)
                   {

/* See what suffix the compression routine appends to the filename */

                     if(strncmp(CompressCmd, suffix->CmdName, strlen(suffix->CmdName)) == 0)
                     {
                       strcpy(FileSuff, suffix->Suffix);
                       break;
                     }
                     suffix++;
                   }
		   
/* If no suffix for this command was found, use the defaults */
		   
		   if(k >= SuffRClist->NumSuffixes)
		   {
		     strcpy(CompressCmd, "gzip");
		     strcpy(UncompressCmd, "gunzip");
		   }
                 }

/* If there was no suffix section then we use our original defaults */
		 
		 else
		 {
		   strcpy(CompressCmd, "gzip");
		   strcpy(UncompressCmd, "gunzip");
		 }
               }
               break;
             }
             compress++;
           }
         }
       }
       else
         strcpy(CompressCmd, "none");

/* If user specified verbose mode, let'em know we're
   gettin' rid of this file.
*/

       if(VerboseFlag)
          printf("%s: %s\n", CmdName, FileName);

/* Copy the file to the safedelete directory */

       if(! S_ISLNK(FileInfo.st_mode) && ! DontSave)
       {
         sprintf(WorkBuf, "cp '%s' '%s'", FileName, SafeDir);
         system(WorkBuf);
       }

/* Compress the file, if we're supposed to */

       j=ParseFileName(FileName);
       if(strcmp(CompressCmd, "none") != 0 && ! DontSave)
       {
         sprintf(WorkBuf, "%s '%s/%s' 2>/dev/null", CompressCmd, SafeDir, FileName+j+1);
         system(WorkBuf);

/* Append the suffix, added by the compression command,
   then rename the file to the grunged file name.
*/

         sprintf(WorkBuf, "%s/%s%s", SafeDir, FileName+j+1, FileSuff);

/* If the compress failed, indicate this then use
   the original filename for the rename step.
*/

         if(stat(WorkBuf, &FileInfo) == 0)
         {
           memcpy(LogRec, &FileFlags, sizeof(FileFlags));
           strcpy(InfoLoc, UncompressCmd);
           InfoLoc+=strlen(InfoLoc)+1;
           strcpy(InfoLoc, FileSuff);
           InfoLoc+=strlen(InfoLoc)+1;
         }
         else
         {
           FileFlags.NoCompressFlag=FLAGON;
           memcpy(LogRec, &FileFlags, sizeof(FileFlags));
           sprintf(WorkBuf, "%s/%s", SafeDir, FileName+j+1);
         }
       }
       else
       {
         FileFlags.NoCompressFlag=FLAGON;
         memcpy(LogRec, &FileFlags, sizeof(FileFlags));
         if(FileFlags.LinkFlag == FLAGON)
         {
           strcpy(InfoLoc, WorkBuf2);
           InfoLoc+=strlen(InfoLoc)+1;
         }
         sprintf(WorkBuf, "%s/%s", SafeDir, FileName+j+1);
       }

       *InfoLoc='\n';
       InfoLoc++;
	  
       sprintf(WorkBuf2, "%s/%s", SafeDir, GrungedFileName);

       if(! DontSave)
       {
         if(rename(WorkBuf, WorkBuf2))
	 {
           sprintf(WorkBuf, "%s: error renaming saved file", CmdName);
           perror(WorkBuf);
           RetCode=1;
	   CurrentDir->FilesLeft++;
           continue;
	 }

/* Now set up the permissions so that only the owner can read
   the files.
*/

         chmod(WorkBuf2, S_IRUSR);

/* Update the users .safedelete.log with the new entry */

         j=(int)(InfoLoc-LogRec);
         if(write(flog, LogRec, j) == -1)
	 {
	   sprintf(WorkBuf, "%s: error writing to %s", CmdName, SafeDelLog);
	   perror(WorkBuf);
	   break;
	 }
       }

/* Now that we're all done, really delete the original file */

       if(unlink(FileName))
       {
	  sprintf(WorkBuf, "%s: error occured removing %s\n", CmdName, FileName);
	  perror(WorkBuf);
       }
    } while(! RecursionDone);
  }

/* Now we sync the disk file with the in-core buffers 
   for the log file then close it
*/

  if(fsync(flog))
  {
    sprintf(WorkBuf, "%s: error syncing %s", CmdName, SafeDelLog);
    perror(WorkBuf);
  }
  close(flog);
  chmod(SafeDelLog, S_IRUSR | S_IWUSR);

/* Release the log file lock */

  ReleaseLock();

/* Free the RC list arrays */

  if(DaysRClist && DaysRClist->DaysStruct)
    free(DaysRClist->DaysStruct);
  if(SuffRClist && SuffRClist->SuffStruct)
    free(SuffRClist->SuffStruct);
  if(CompRClist && CompRClist->CompStruct)
    free(CompRClist->CompStruct);

/* All done deleting files */

  return RetCode;
}
