/* safedelchk.c
   This scans through the systems safedelete
   directory looking for files older than
   the number of days specified and deleting
   them.

   Usage:  safedelchk [-d nn] [-u] [-v]

           -d : number of days after which
                the saved files will be
                deleted. If not specified
		the maximum set at install
		time will be used.

           -u : force user mode when running
                as root.

           -v : verbose mode - spit out messages while we work.

   This command is meant to be run in 2
   distinct environments:
   - from each users .bash_profile
   - from a nightly scheduled crontab as root
   It can also be run on an ad hoc
   basis, if needed.
   
   Only root can run safedelchk with the
   -d and -u options.  All other users are
   forced to use either the $SAFEDAYS option
   or the installation-defined default.

   When examining the date of each file
   in the safedelete directory, the date
   embedded in the file name is used instead
   of the last-updated date in the directory.
   This allows the files to be deleted even
   though someone may "touch" every file in
   their subdirectory.
*/

#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <dirent.h>
#include <pwd.h>
#include <ctype.h>
#include <getopt.h>
#include "safedelete.h"

/* Global variables */

int VerboseFlag;

char CmdName[16];
char SafeDelLockFile[DEL_LOG_LEN];

/* ================================================= */

/* ComputeDate
   Compute the oldest date a safedeleted file can remain on
   the system.

   Passed: number of days
           pointer to stucture of type tm

   Returns: the correct date in the tm structure

   Example: passed NumDays=4  tm -> 05/12/95
            returns tm -> 05/08/95
*/

void ComputeDate(NumDays, tm)
  int NumDays;
  struct tm *tm;
{
int  DaysPerMonth[12];

/* Do some initializing */

  DaysPerMonth[0]=31;
  DaysPerMonth[1]=28;
  DaysPerMonth[2]=31;
  DaysPerMonth[3]=30;
  DaysPerMonth[4]=31;
  DaysPerMonth[5]=30;
  DaysPerMonth[6]=31;
  DaysPerMonth[7]=31;
  DaysPerMonth[8]=30;
  DaysPerMonth[9]=31;
  DaysPerMonth[10]=30;
  DaysPerMonth[11]=31;

/* Allow for leap years */

  if(tm->tm_year%4 == 0 && tm->tm_year%100 != 0)
    DaysPerMonth[1]=29;
  if(tm->tm_year%100 == 0 && tm->tm_year%400 == 0)
    DaysPerMonth[1]=29;

/* See if we go past a month/year boundary */

  tm->tm_mday-=NumDays;
  if(tm->tm_mday <= 0)
  {
    tm->tm_mon=(tm->tm_mon == 0 ? 11 : --tm->tm_mon);
    tm->tm_mday+=DaysPerMonth[tm->tm_mon];
    if(tm->tm_mon == 11)
      tm->tm_year=(tm->tm_year == 0 ? 99 : --tm->tm_year);
  }

}

/* ============================================================= */

/* PurgeFiles
   Purge all the files in the specified directory which are
   older than the provided date.

   Passed: pointer to safedelete directory name
           pointer to the high level qualifier containing the
           date of the oldest files  (e.g. "d041395")
           pointer to the user's safedelete log file name to update
           flag indicating mode (0=user  1=administrator)

   Returns: 0 if purge went OK
            1 if purge failed
*/

int PurgeFiles(DirName, OldestDate, SafeDelLog, SafeDelRC, rc, Mode)
  char *DirName, *OldestDate, *SafeDelLog, *SafeDelRC;
  rcstruct *rc;
  int  Mode;
{
DIR *dir;
struct dirent *dirent;
struct stat FileInfo;
struct tm *tm;
time_t currtime;
int  i, j, k, retcode;
int  GotMatch, NumDels;
FILE *flog;
char *LogList, *palloc, *wrkptr;
char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];

safedays_struct *days;

/* Check permissions to the safedelete directory and
   to the user's log file.
*/

  if(CheckPerms(DirName, SafeDelLog, CmdName, YES_MSG) != 0)
    return 1;

  retcode=0;

/* Read in the entire safedelete log so it's easier
   to scan and update.  The format will be a null
   terminated entry for each line with a null
   delimiter to indicate the end of list.
*/

  if(AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, Mode))
  {
    printf("%s: file %s is busy, try again later\n", CmdName, SafeDelLog);
    return 1;
  }

  stat(SafeDelLog, &FileInfo);

  LogList=palloc=malloc(FileInfo.st_size);

  flog=fopen(SafeDelLog, "r");
  fread(LogList, FileInfo.st_size, 1, flog);
  fclose(flog);

/* First make sure the .safedelete.log is in the
   format we require.
*/

  if(FileInfo.st_size > 0 && ! isdigit(*LogList))
  {
    free(palloc);
    ReleaseLock();
    
    printf("%s: %s is still in the old format\n", CmdName, SafeDelLog);
    printf("%s: it will be converted using the safecnvt command\n", CmdName);
    system("safecnvt");

/* Now that it's converted, let's just double check */

    stat(SafeDelLog, &FileInfo);
    
    if(AcquireLock(SafeDelLockFile, CmdName, SafeDelRC, Mode))
    {
      printf("%s: file %s is busy, try again later\n", CmdName, SafeDelLog);
     return 1;
    }
    
    LogList=palloc=malloc(FileInfo.st_size);

    flog=fopen(SafeDelLog, "r");
    fread(LogList, FileInfo.st_size, 1, flog);
    fclose(flog);

    if(FileInfo.st_size > 0 && ! isdigit(*LogList))
    {
      printf("%s: conversion failed, cannot continue\n", CmdName);
      free(palloc);
      ReleaseLock();  
      return 1;
    }
    else
      printf("%s: %s was successfully converted\n", CmdName, SafeDelLog);
  }

/* Go through the .safedelete.log and flag all entries
   which are to be removed.
*/

  LogList=palloc;
  NumDels=0;
  j=k=0;
  while(j < FileInfo.st_size)
  {
    k=RecLen(LogList);
    wrkptr=LogList+sizeof(fileflag_struct);
    wrkptr+=strlen(wrkptr)+1;   /* point to grunged filename */
    GotMatch=0;

/* If user has a .Safedelrc file, scan the Safedays section */

    if(rc && (days=rc->DaysStruct))
    {
      for(i=0; i < rc->NumDaysEnts; i++)
      {
        if(CheckFileDays(LogList+sizeof(fileflag_struct), days) >= 0)
        {
          GotMatch=1;
          currtime=time(0);
          tm=localtime(&currtime);
          ComputeDate(days->Days, tm);
          strftime(WorkBuf, 9, "%Y%m%d", tm);
	  if(*(wrkptr+5) == '9')
	    sprintf(WorkBuf2, "19%c%c%c%c%c%c", *(wrkptr+5), *(wrkptr+6),
		*(wrkptr+1), *(wrkptr+2), *(wrkptr+3), *(wrkptr+4));
	  else
	    sprintf(WorkBuf2, "20%c%c%c%c%c%c", *(wrkptr+5), *(wrkptr+6),
		*(wrkptr+1), *(wrkptr+2), *(wrkptr+3), *(wrkptr+4));
          if(strncmp(WorkBuf2, WorkBuf, 10) <= 0)
          {
	    if(VerboseFlag)
              printf("%s: Removing %s safedeleted on %c%c/%c%c/%c%c%c%c - remove after %d days\n", 
		CmdName, LogList+sizeof(fileflag_struct), *(WorkBuf2+4), 
		*(WorkBuf2+5), *(WorkBuf2+6), *(WorkBuf2+7), *(WorkBuf2), 
		*(WorkBuf2+1), *(WorkBuf2+2), *(WorkBuf2+3), days->Days);
            *(LogList+sizeof(fileflag_struct))='$';
            sprintf(WorkBuf, "%s/%s", DirName, wrkptr);
            unlink(WorkBuf);
            NumDels++;
          }
          break;   
        }
        days++;
      }
    }

    if(! GotMatch)
    {
      if(*(wrkptr+5) == '9')
        sprintf(WorkBuf2, "19%c%c%c%c%c%c", *(wrkptr+5), *(wrkptr+6),
		*(wrkptr+1), *(wrkptr+2), *(wrkptr+3), *(wrkptr+4));
      else
        sprintf(WorkBuf2, "20%c%c%c%c%c%c", *(wrkptr+5), *(wrkptr+6),
		*(wrkptr+1), *(wrkptr+2), *(wrkptr+3), *(wrkptr+4));
      if(strncmp(WorkBuf2, OldestDate, 9) < 0)
      {
        if(VerboseFlag)
        printf("%s: Removing %s safedeleted on %c%c/%c%c/%c%c%c%c - remove files safedeleted before %c%c/%c%c/%c%c%c%c\n", 
		CmdName, LogList+sizeof(fileflag_struct), *(WorkBuf2+4), 
		*(WorkBuf2+5), *(WorkBuf2+6), *(WorkBuf2+7), *(WorkBuf2), 
		*(WorkBuf2+1), *(WorkBuf2+2), *(WorkBuf2+3),
		*(OldestDate+4), *(OldestDate+5), *(OldestDate+6),
		*(OldestDate+7), *(OldestDate), *(OldestDate+1),
		*(OldestDate+2), *(OldestDate+3));
        *(LogList+sizeof(fileflag_struct))='$';
        sprintf(WorkBuf, "%s/%s", DirName, wrkptr);
        unlink(WorkBuf);
        NumDels++;
      }
    }
    j+=k;
    LogList+=k;
  }
 
  printf("%s: %d file(s) removed\n", CmdName, NumDels);

/* Read the safedelete directory and delete the old files.
   Also, make sure any remaining files are listed in the
   log.
*/

  dir=opendir(DirName);
  while((dirent=readdir(dir)))
  {
    if(dirent->d_name[0] == '.')
      continue;

/* Here we compare the filename in the directory against
   the in-storage list read from the .safedelete.log.  If an
   entry in the log was deleted (first byte = "$") then
   it is ignored.  If the filename from the directory matches
   an entry in the log, we flag it by setting the first byte
   to "f".  We can get away with this because each of the
   entries is a fully qualified path beginning with "/".  We'll
   restore the "/" just before we rewrite the log.
*/

    else
    {
      i=j=k=0;
      LogList=palloc;
      while(j < FileInfo.st_size)
      {
	k=RecLen(LogList);
        if(*(LogList+sizeof(fileflag_struct)) == '/')
        {
          wrkptr=LogList+sizeof(fileflag_struct);
	  wrkptr+=strlen(wrkptr)+1;
          if(strncmp(wrkptr, dirent->d_name, strlen(wrkptr)) == 0)
          {
            i=1;
            *(LogList+sizeof(fileflag_struct))='f';
            break;
          }
        }
        j+=k;
        LogList+=k;
      }
      if(! i)
      {
        fprintf(stderr,
                "%s: .safedelete.log has no entry for saved file %s\n",
                CmdName, dirent->d_name);
        retcode=1;
      } 
    }
  }
  closedir(dir);

/* Now we go through the in-storage log list
   looking for any entries that were not found
   in the safedelete directory.  We also reset the
   found flag (first char = "f") back to "/".  Again,
   we ignore any files that were deleted (first
   char = "$");
*/

  j=k=0;
  LogList=palloc;
  while(j < FileInfo.st_size)
  {
    k=RecLen(LogList);
    if(*(LogList+sizeof(fileflag_struct)) == 'f')
      *(LogList+sizeof(fileflag_struct))='/';
    else
    {
      if(*(LogList+sizeof(fileflag_struct)) != '$')
      {
        fprintf(stderr,
                "%s: safedelete file for %s not found in %s\n",
                CmdName, LogList+sizeof(fileflag_struct), DirName);
        retcode=1;
      }
    }
    j+=k;
    LogList+=k;
  }

/* Finally, we reset the file pointer to the beginning
   of the file and write out the remaining log entries.
*/

  flog=fopen(SafeDelLog, "w");
  j=k=0;
  LogList=palloc;
  while(j < FileInfo.st_size)
  {
    k=RecLen(LogList);
    if(*(LogList+sizeof(fileflag_struct)) != '$')
      fwrite(LogList, k, 1, flog);
    j+=k;
    LogList+=k;
  }
  fclose(flog);
  
/* Release lock */
  
  ReleaseLock();
  
  free(palloc);
  return retcode;
}

   
/* ============================================================= */  

int main(argc, argv)
  int argc;
  char **argv;
{
int  c, ccode, NumDays, retcode;
char *EnvValue, *HomeDir, *LoginName = NULL;
char SafeDir[SAFE_DIR_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDelRC[SAFE_RC_LEN];
char OldestDate[10];

int  UserMode;

time_t currtime;
struct tm *tm;
struct passwd *UserInfo;
struct stat DirInfo;
uid_t uid;

rcstruct *DaysRClist;

/* Do some initializtion work */

  strcpy(CmdName, "safedelchk");
  
  currtime=time(0);
  tm=localtime(&currtime);
  retcode=0;
  NumDays=MAX_SAFE_DAYS;
  VerboseFlag=0;       /* This is a global variable */

  printf("%s: scan beginning\n", CmdName);

/* First we get the uid of the person running
   the command.  If it's superuser, then we
   look for any options.  If it's Joe User,
   we look for the $SAFEDAYS variable.
*/

  uid=getuid();
  if(uid == 0)
  {
    
/* See if superuser specified any options */

    UserMode=0;

    while((c=getopt(argc, argv, "d:uv")) != -1)
    {
      switch(c)
      {
        case 'd' : NumDays=atoi(optarg);
                   break;

        case 'u' : UserMode=1;
                   break;

        case 'v' : VerboseFlag=1;
                   break;

        default  : fprintf(stderr, "%s: invalid option %c\n", CmdName, c);
                   return 1;
      }
    }
  }
  else
  {

/* Normal user -- just look for $SAFEDAYS */
    
    while((c=getopt(argc, argv, "v")) != -1)
    {
      switch(c)
      {
        case 'v' : VerboseFlag=1;
                   break;

        default  : fprintf(stderr, "%s: invalid option %c\n", CmdName, c);
                   return 1;
      }
    }

    UserMode=1;
    if((EnvValue=getenv("SAFEDAYS")))
      NumDays=atoi(EnvValue);
  }

  NumDays=(NumDays <= MAX_SAFE_DAYS ? NumDays : MAX_SAFE_DAYS);

/* Now get the date of NumDays ago */

  ComputeDate(NumDays, tm);

/* And save it for later */

  strftime(OldestDate, 9, "%Y%m%d", tm);


/* Now we're ready to start deleting files and
   log entries.  If running in user mode, everything
   is all set up.  If running administrator mode,
   we go through the /etc/passwd file and examine 
   the directories and logs for each user with
   safedeleted files.
*/
  if(UserMode)
  {

/* Finish setting up things for
   user mode processing
*/

    HomeDir=getenv("HOME");
    if(HomeDir == NULL)
    {
      LoginName=getlogin();
      UserInfo=getpwnam(LoginName);
      HomeDir=UserInfo->pw_dir;
    }
    strcpy(SafeDelLog, HomeDir);
    strcat(SafeDelLog, LOG_NAME);
    strcpy(SafeDir, HomeDir);
    strcat(SafeDir, SAFE_DIR_NAME);
    strcpy(SafeDelLockFile, HomeDir);
    strcat(SafeDelLockFile, LOCK_NAME);

    if((EnvValue=getenv("SAFEDELRC")) != NULL)
      strcpy(SafeDelRC, EnvValue);
    else
    {
      strcpy(SafeDelRC, HomeDir);
      strcat(SafeDelRC, RC_NAME);
    }

    DaysRClist=(rcstruct*)ReadRC(SafeDelRC, "safedays");

    if(DaysRClist && DaysRClist->DaysStruct && DaysRClist->NumDaysEnts > 0)
      printf("%s: using safedays values from %s\n", CmdName, SafeDelRC);
    else
      printf("%s: removing files safedeleted before %c%c/%c%c/%c%c%c%c\n",
	   CmdName,
           OldestDate[4], OldestDate[5], OldestDate[6], OldestDate[7],
           OldestDate[0], OldestDate[1], OldestDate[2], OldestDate[3]);


/* Purge the files and prune the log for the user */

    ccode=PurgeFiles(SafeDir, OldestDate, SafeDelLog, SafeDelRC, DaysRClist, 0);

/* Release the memory obtained by ReadRC */

    if(DaysRClist && DaysRClist->DaysStruct)
      free(DaysRClist->DaysStruct);

    if(ccode != 0)
      retcode=1;
  }
  else
  {
    
/* Administrator mode -- check all users */

    setpwent();
    while((UserInfo=getpwent()) != NULL)
    {
      strcpy(SafeDir, UserInfo->pw_dir);
      strcat(SafeDir, SAFE_DIR_NAME);
      if(stat(SafeDir, &DirInfo) == 0)
      {
        printf("%s: examining files for user %s\n",
               CmdName, UserInfo->pw_name);

        strcpy(SafeDelLog, UserInfo->pw_dir);
        strcat(SafeDelLog, LOG_NAME);
        strcpy(SafeDelRC, UserInfo->pw_dir);
        strcat(SafeDelRC, RC_NAME);
	strcpy(SafeDelLockFile, UserInfo->pw_dir);
	strcat(SafeDelLockFile, LOCK_NAME);

        DaysRClist=(rcstruct*)ReadRC(SafeDelRC, "safedays");

        ccode=PurgeFiles(SafeDir, OldestDate, SafeDelLog, SafeDelRC, DaysRClist, 1);

        if(DaysRClist && DaysRClist->DaysStruct)
          free(DaysRClist->DaysStruct);

        if(ccode != 0)
          retcode=1;
      }
      else
      {
        printf("%s: No safedeleted files for user %s, skipping\n",
		CmdName, UserInfo->pw_name);
      }
    }
    endpwent();
  }   

  printf("%s: scan completed\n", CmdName);
  return retcode;
}
    

