/* qprompt.c

   version 2.3  Oct, 1994
   by Andreas_Bagge@maush2.han.de

   version 2  May, 1994
   originally by Alex Kent ... alex@nmt.edu

     Feel free to modify this code to your hearts content; 
     trash it, thrash, rewrite it.
     Maybe some others will find it as useful and easy to use 
     as I have.
*/


/****************************************************************/

#include <stdio.h>
#include <termios.h>
#include <sys/fcntl.h>
#include <signal.h>
#include <sys/time.h>
#include <string.h>

/* special exit status */
#define KEYRETURN	100
#define ANYKEY		101
#define TIMEOUT		102
#define ABORT		103
#define USAGEERROR	255

#define BELL		'\007'

#ifndef VERSION
#define VERSION "2.x"
#endif

struct termios org_tty;		/* original line settings */
int oflag = 0;			/* output flag */
int fd = 0;			/* input device, default is stdin */
const char *prg="qprompt";	/* program name */
const char *version=VERSION;	/* program version */

void resetline(void);
void timed_out(void);
int checkit(const char c, char *const argv[]);
void usage(const int exitstatus);

/****************************************************************/
int main(int argc, char *argv[])
{
    int i, maxtime = 0;
    char c;
    int exitstatus = USAGEERROR;
    struct termios buf;
    int rflag = 0;
    int aflag = 0;
    int eflag = 0;
    int Pflag = 0;
    int tflag = 0;
    int arg = 0;

    prg=argv[arg];
    arg++;
    if (argc < 2)
	usage(USAGEERROR);
    while (argv[arg][0] == '-') {
	switch (argv[arg][1]) {
	case 'p':		/* print prompt */
	    if (argv[arg][2])
		fputs(&argv[arg][2], stderr), fflush(stderr);
	    else if (argv[arg + 1])
		fputs(argv[++arg], stderr), fflush(stderr);
	    else {
		fprintf(stderr, "%s: option -p requires an argument\n",prg);
		usage(USAGEERROR);
	    }
	    break;
	case 'b':		/* ring bell */
	    fputc(BELL, stderr);
	    break;
	case 'i':		/* select input device */
	    if (argv[arg][2])
		fd = open(&argv[arg][2], O_RDONLY);
	    else if (argv[arg + 1])
		fd = open(argv[++arg], O_RDONLY);
	    else {
		fprintf(stderr, "%s: option -i requires an argument\n",prg);
		usage(USAGEERROR);
	    }
	    break;
	case 'P':		/* print default prompt */
	    Pflag = 1;
	    break;
	case 'e':		/* echo typed character to stdout */
	    eflag = 1;
	    break;
	case 'o':		/* write exit status to stdout */
	    oflag = 1;
	    break;
	case 'r':		/* disable return key processing */
	    rflag = 1;
	    break;
	case 'a':		/* allow any key */
	    aflag = 1;
	    break;
	case 't':		/* timeout behaviour */
	    if (argv[arg][2])
		tflag = argv[arg][2];
	    else if (argv[arg + 1])
		tflag = argv[++arg][0];
	    else {
		fprintf(stderr, "%s: option -t requires an argument\n",prg);
		usage(USAGEERROR);
	    }
	    switch (tflag) {
	    case '0':
		tflag = 0;	/* stop timer after first key pres */
		break;
	    case '1':
		tflag = 1;	/* restart timer after key press */
		break;
	    case '2':
		tflag = 2;	/* timer ignores any invalid keys */
		break;
	    default:
		fprintf(stderr, "%s: invalid argument to -t option: %c\n",prg, (char) tflag);
		usage(USAGEERROR);
	    }
	    break;
	case '0':		/* enable timeout */
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	    maxtime = atoi(&argv[arg][1]);
	    break;
	case 'd':		/* allow default without usage error */
	    break;
	default:
	    fprintf(stderr, "%s: invalid option: %s\n",prg, argv[arg]);
	    usage(USAGEERROR);
	}
	arg++;
    }

    if (!aflag && rflag && !argv[arg]) {
	fprintf(stderr, "%s: you must allow input with -a or args if option -r is used\n",prg);
	usage(USAGEERROR);
    }

    /* print default prompt */
    if (Pflag) {
	if (argv[arg]) {
	    fprintf(stderr, "hit one key of [");
	    for (i = arg; i < argc; i++)
		fprintf(stderr, argv[i]);
	    fprintf(stderr, "] ");
	    if (aflag)
		fprintf(stderr, "or any other key ");
	    else if (!rflag)
		fprintf(stderr, "or RETURN ");
	} else if (aflag) {
	    fprintf(stderr, "hit any key to continue ");
	} else if (!rflag) {
	    fprintf(stderr, "hit RETURN to continue ");
	}
	if (maxtime > 0)
	    fprintf(stderr, "(timeout in %d sec) ", maxtime);
	fflush(stderr);		/* should be obsolete for stderr */
    }

    /* save old line settings */
    tcgetattr(fd, &org_tty);

    /* modify line setting parameters */
    buf = org_tty;
    buf.c_lflag &= ~(ECHO | ICANON | IEXTEN);
    buf.c_oflag &= ~(OPOST);
    buf.c_cc[VMIN] = 1;
    buf.c_cc[VTIME] = 0;

    /* set new line settings */
    tcsetattr(fd, TCSAFLUSH, &buf);

    signal(SIGHUP, (void *) resetline);
    signal(SIGINT, (void *) resetline);
    signal(SIGQUIT, (void *) resetline);
    signal(SIGTERM, (void *) resetline);

    /* initialize timer */
    if (maxtime > 0) {
	signal(SIGALRM, (void *) timed_out);
	alarm(maxtime);
    }
   
    /* wait for terminal input */
    while (read(fd, &c, 1)) {
	int k;
	if (!rflag && c == '\n') {
	    exitstatus = KEYRETURN;	/* return RETURNKEY */
	    break;
	} else if (k = checkit(c, &argv[arg])) {
    	    if(eflag) fprintf(stderr,"%c",c);
	    exitstatus = k;		/* valid key .. return arg number */
	    break;
	} else if (aflag) {	/* any key is OK ... return ANYKEY */
    	    if(eflag) fprintf(stderr,"%c",c);
	    exitstatus = ANYKEY;
	    break;
	} else if (maxtime > 0) {	/* check for timeout */
	    switch (tflag) {
	    case 0:
		signal(SIGALRM, SIG_IGN);	/* stop timer */
		break;
	    case 1:
		alarm(maxtime);	/* restart timer */
		break;
	    case 2:		/* timer continous */
		break;
	    }
	}
    }

    /* restore original line settings */
    tcsetattr(fd, TCSAFLUSH, &org_tty);

    fprintf(stderr, "\n");
    if (oflag)
	printf("%d\n", exitstatus);
    return(exitstatus);
}

/****************************************************************/
void resetline(void)
{
    tcsetattr(fd, TCSAFLUSH, &org_tty);
    fprintf(stderr, "\n");
    if (oflag)
	printf("%d\n", ABORT);
    exit(ABORT);
}

/****************************************************************/
void timed_out(void)
{
    tcsetattr(fd, TCSAFLUSH, &org_tty);
    fprintf(stderr, "\n");
    if (oflag)
	printf("%d\n", TIMEOUT);
    exit(TIMEOUT);
}

/****************************************************************/
int checkit(const char c, char *const argv[])
{
    int i;
    for (i = 0; argv[i]; i++) {
	if (strchr(argv[i], c))
	    return i + 1;
    }
    return 0;			/* no valid char pressed */
}

/****************************************************************/
void usage(int exitstatus)
{
    fprintf(stderr, "qprompt %s  by Alex Kent, modified by A. Bagge\n",version);
    fprintf(stderr, "Usage: %s [options] [arg1] ...\n",prg);
    fprintf(stderr, "options:\n");
    fprintf(stderr, "  -a          allow any key\n");
    fprintf(stderr, "  -r          disable special processing of RETURN key\n");
    fprintf(stderr, "  -P          print default prompt\n");
    fprintf(stderr, "  -p <prompt> print <prompt>\n");
    fprintf(stderr, "  -b          ring bell to get attention\n");
    fprintf(stderr, "  -<timeout>  timeout of <timeout> seconds\n");
    fprintf(stderr, "  -t <x>      modify timer behaviour\n");
    fprintf(stderr, "               x=0  stop timeout after first key pressed (default)\n");
    fprintf(stderr, "                 1  restart timer after key pressed\n");
    fprintf(stderr, "                 2  run timer continously\n");
    fprintf(stderr, "  -i <dev>    read from <dev> instead of from stdin\n");
    fprintf(stderr, "  -o          write exit status to stdout\n");
    fprintf(stderr, "  -e          echo the typed character to stderr\n");
    fprintf(stderr, "  arg1        valid input characters (all in one arg accepted as equal)\n");
    fprintf(stderr, "exit status:\n");
    fprintf(stderr, "   1,2,..	key of arg1,arg2,.. pressed\n");
    fprintf(stderr, "   %d   	return key pressed\n", KEYRETURN);
    fprintf(stderr, "   %d   	any key pressed\n", ANYKEY);
    fprintf(stderr, "   %d   	timeout\n", TIMEOUT);
    fprintf(stderr, "   %d   	aborted by signal\n", ABORT);
    fprintf(stderr, "   %d   	usage error\n", USAGEERROR);
    if (oflag)
	printf("%d\n", exitstatus);
    exit(exitstatus);
}
