/* PROGRAM paint_changed_characters

Stephen O. Lidie.  September 1991.
Lehigh University Computing Center
Bethlehem, PA  18015
(215)758-3982
Stephen.O.Lidie@CDC1.CC.Lehigh.EDU
lusol@Lehigh.edu, 93/06/08

Copyright (C) 1991 - 1996, Lehigh University.  All rights reserved.

paint_changed_characters

        This command turns the static output from one or more Unix
        commands into a dynamic, pageable, real-time display.  You
        specify the command(s) and the delay period between screen
        refreshes and Paint Changed Characters does the rest, including
        optimizations to minimize cursor movement and the number of
        transmitted characters.  (Patterned after the CYBIL version
        for NOS/VE.)

        The environment variable SHELL determines which shell will
        execute the command(s).  If it is not defined /bin/sh is
        assumed.

        The environment variable D_PAICC_C can be defined to change
        the default command "ps -el".  

          Example:

            paicc                        # execute ps -el every second,
                                         # (or the default command)

            paicc -c "hostname>o;ps -ef>>o;cat o" -mui 3000
                                         # execute ps -ef every 3 seconds
                                         # with hostname on first line

Parameters:

-help, disci, full_help: Display Command Information
-command, c: string = D_PAICC_C, "ps -el"
-millisecond_update_interval, mui: integer = D_PAICC_MUI, 1000
-title, t: boolean = D_PAICC_T, TRUE
-clear_screen_at_end, csae: boolean = D_PAICC_CSAE, TRUE


This program consists of three tasks:  the parent that executes the user's
command(s) and makes calls to curses routines to manage the display and then
delays waiting for the update interval to elapse or the arrival of an 'input
available' signal, an asynchronous task that waits for user input characters
and forwards them via a pipe to the parent, and an asynchronous task that
actually execs the user's command(s) and redirects standard output to the
parent's other input pipe.

September, 1996 - Update paicc/C for Linux 1.2.13; use evap 2.3.
September, 1991 - C version for Unix.
October, 1990   - CYBIL version for NOS/VE.
July, 1985      - RATFOR version for NOS.

*/

#include <stdio.h>
#include <curses.h>
#include <signal.h>
#include <time.h>

#include "evap/evap.h"
#include "evap/paicc_pdt_out"
#include "patchlevel.h"

#define MAXIMUM_BUFFER_SIZE 4096
#define MAXIMUM_LINE_LENGTH 512

#define P_READ  0		/* pipe read */
#define P_WRITE 1		/* pipe write */

int  evap();			/* evaluate parameters */
void fini();			/* finish Paint Changed Characters */
int  getc_from_pipe();		/* get character via a File Descriptor */
char *getenv();			/* get environment variable */
int  gets_from_pipe();		/* get string via a File Descriptor */
void ini();			/* initialize Paint Changed Characters */
void paicc();			/* Paint Changed Characters */ 
void process_user_input();            
void window_ini();		/* curses window initialization */
void window_resize();		/* initialize after a re-size signal */
int  main();			/* PAICC main entry point */ 

int  bl = MAXIMUM_BUFFER_SIZE;	/* read buffer limit */
int  bp = MAXIMUM_BUFFER_SIZE;	/* read buffer pointer */
int  in_pid;			/* PID of input child */
int  in_pipes[2];		/* input child pipes */
char line[MAXIMUM_LINE_LENGTH+1]; /* general purpose line buffer */
char screen_title[80];		/* built at runtime to incorporate version */

char *shell;			/* user's preferred shell */
int  x, y;			/* character coordinates */
int  x_bias;			/* window scroll bias */
int  y_bias;			/* window scroll bias */

/*
  Global signal-processing procedures.  These must appear physically before
  the call to signal.
*/

void fini(sig)
  int sig;
{

  if (sig != -99) {
    if ( pvt[P_clear_screen_at_end].value.boolean_value )
      erase();
    refresh();
  }
  endwin();			/* end curses activity */

  kill(in_pid, SIGKILL);	/* terminate the input task */

  exit(0);

} /* end fini */

void process_user_input (sig)
  int sig;

/*
  This procedure is called when the asynchronous input task raises a SIGUSR1
  signal indicating his blocking read has completed and a user input character
  is available.  Re-establish the condition handler and process the input.
*/

{

  char ch[MAXIMUM_LINE_LENGTH];	/* input buffer */
  int i;			/* counter */
  int nc;			/* number of characters */

  signal(SIGUSR1, SIG_IGN);

  nc = read(in_pipes[P_READ], ch, MAXIMUM_LINE_LENGTH);

  for( i = 0; i < nc; i++){

    switch (ch[i]) {
      
    case '?':			/* ? */
      erase();
      mvaddstr( 0, 0, " Paint Changed Characters Help");
      mvaddstr( 2, 0, "   control/F = page forward");
      mvaddstr( 3, 0, "           + = page forward 1 line");
      mvaddstr( 5, 0, "   control/B = page backward");
      mvaddstr( 6, 0, "           - = page backward 1 line");
      mvaddstr( 8, 0, "   control/R = shift right");
      mvaddstr( 9, 0, "           > = shift right 1 column");
      mvaddstr(11, 0, "   control/L = shift left");
      mvaddstr(12, 0, "           < = shift left 1 column");
      mvaddstr(14, 0, "   control/D = quit (or `q')");
      mvaddstr(15, 0, "   control/U = undo all paging/shifting");
      mvaddstr(16, 0, "   control/W = repaint screen");
      mvaddstr(18, 0, " <cr> to continue ... ");
      refresh();
      signal(SIGUSR1, process_user_input);
      getch();			/* let her read the help display */
      erase();
      break;
      
    case 2:			/* control/B */
      y_bias -= (LINES / 2);
      if (y_bias < 1)
	y_bias = 1;
      erase();
      break;
      
    case 4: case 'q': case 'Q':	/* control/D */
      fini(0);
      break;
      
    case 6:			/* control/F */
      y_bias += (LINES / 2);
      erase();
      break;

    case 12:			/* control/L */
      x_bias -= (COLS / 2);
      if (x_bias < 1)
	x_bias = 1;
      erase();
      break;

    case 18:			/* control/R */
      x_bias += (COLS / 2);
      erase();
      break;

    case 21:			/* control/U */
      x_bias = 1;
      y_bias = 1;
      erase();
      break;
      
    case 23:			/* control/W */
      erase();
      refresh();
      break;
      
    case '+':			/* + */
      y_bias += 1;
      erase();
      break;
      
    case '-':			/* - */
      y_bias -= 1;
      if (y_bias < 1)
	y_bias = 1;
      erase();
      break;
      
    case '<':			/* < */
      x_bias -= 1;
      if (x_bias < 1)
	x_bias = 1;
      erase();
      break;
      
    case '>':			/* > */
      x_bias += 1;
      erase();
      break;
      
    default:
      /*      beep(); # not in Linux!*/
      break;
      
    } /* casend */
    
  } /* forend */

  signal(SIGUSR1, process_user_input);

} /* end process_user_input */

void window_resize(sig)
  int sig;
{

  endwin();
  signal(SIGWINCH, window_resize); /* handle changing window sizes */
  window_ini();    
  erase();
  /*  beep(); # not in Linux!*/
  refresh();

} /* end window_resize */

/*
  Global procedures.
*/

int getc_from_pipe(fd)
  int fd;
{

  char buf[MAXIMUM_BUFFER_SIZE];

  if (bp >= bl) {		/* fill buffer */
    if ( (bl = read(fd, buf, MAXIMUM_BUFFER_SIZE)) == 0)
      return(EOF);		/* signal end-of-file */
    bp = 0;                           
  }

  return(buf[bp++]);		/* return current character */

} /* end getc_from_pipe */

int gets_from_pipe(line, ml, fd)
  char line[];
  int ml;
  int fd;
{

  int ch;
  int lp = 0;

  while (--ml > 0 && (ch = getc_from_pipe(fd)) != EOF && ch != '\n') {
    if ( ch == 7 ) {
      printf( "\007" );
    } else 
      line[lp++] = ch;  
  }

  if (ch == '\n') 
    line[lp++] = ch;           

  line[lp] = '\0';

  return(lp);			/* return length of line */

} /* end gets_from_pipe */

void ini ()

/*
  Various initialization.  Spin off an asynchronous task to do blocking curses
  reads.  When a character is read, the input child writes it to his output
  pipe and uses SIGUSR1 to signal the parent (us).  We then process the user
  input.  This allows immediate user input even if the sleep period is long.
*/

{

  char ch;			/* an input character */
  int p_pid;			/* PID of child's parent */
  
  sprintf(screen_title, "Paint Changed Characters %s    (? for help, O=",
	  VERSION);

  signal(SIGINT, fini);		/* trap control/C */
  signal(SIGSEGV, fini);             

  signal(SIGWINCH, window_resize); /* handle changing window sizes */

  if ( (shell = getenv("SHELL")) == NULL)
    shell = "/bin/sh";

  window_ini();			/* initialize */

  x = 1;			/* start at column 1 */
  x_bias = 1;			/* start at left edge */

  y = 1;			/* start at row 1 */
  y_bias = 1;			/* start at top edge */

  if (pipe(in_pipes) != 0) {
    perror("paicc - Cannot create my input child pipes!");
    fini(-99);
  }

  if ((in_pid = fork()) < 0) {
    perror("paicc - Cannot fork the input child!");
    fini(-99);
  }

  if (in_pid == 0) {		/* if input child */

    /*
      We are the input child process - the asynchronous task that does
      blocking curses reads waiting for user input characeters.  When a
      character arrives, write it to the input pipe's queue and signal
      our parent to read the input pipe's queue for the user's input.
    */
      
    close(in_pipes[P_READ]);
    p_pid = getppid();		/* get parent PID */
    
    while (1) {

      ch = getch();		/* look for user input */
      write(in_pipes[P_WRITE], &ch, 1);
      kill(p_pid, SIGUSR1);	/* signal parent there is user input */

    } /* whilend */

  } /* ifend input child */
	
  /*
    Back to the parent.  Initialize the SIGUSR1 condition handler and return
    to the parent main-loop code.
  */

  close(in_pipes[P_WRITE]);
  signal(SIGUSR1, process_user_input);
      
} /* end ini */

void paicc ()

/*
  This is the Paint Changed Characters parent task.  Use pipe/fork/exec and
  friends to start a periodic asynchronous task to execute the user's
  command(s), read standard input and have curses display changed characters.
*/

{

  char *atime;			/* ASCII format time */
  int  child_status;		/* status of child termination */
  long etime;			/* 'epoch' time */
  int  line_count;		/* input pipe line count */
  int  pid;			/* process ID of child */
  int  pipes[2];		/* pipe I/O file descriptors */
  int  xc;			/* transfer count */

  while (TRUE) {		/* until aborted */

    y = 1;			/* row 1 again */

    if (pvt[P_title].value.switch_value) {
      time(&etime);		/* get second of U*X epoch */
      atime = ctime(&etime);	/* convert to ASCII */
      atime[11+8] = '\0';	/* we want only the clock */
      sprintf(line, "%s%d,%d)%n", screen_title, x_bias, y_bias, &xc);
      sprintf(line+xc, "%*s", 80-xc, atime+11);
      mvaddstr(y-1, x-1, line);	/* move and add string to stdscr */
      y++;			/* now to row 2 */
    } /* ifend */

    /*
      Now fork and let the child exec the user's command.  Meanwhile the
      parent will get the command's standard output from his input pipe.
      Let curses manipulate the screen for us.
    */

    if(pipe(pipes) != 0) {	/* create the I/O pipes */
      perror("paicc - Cannot create my pipes!");
      fini(-99);
    }

    if((pid = fork()) < 0) {
      perror("paicc - Cannot fork myself!");
      fini(-99);
    }

    if(pid == 0){		/* child */

      close(1);			/* close stdout to free it's FD */
      dup(pipes[P_WRITE]);	/* link write end of pipe to stdout */
      close(pipes[P_WRITE]);	/* do not need write end of pipe now */
      close(pipes[P_READ]);	/* child doesn't need read end of pipe */

      execl(shell, shell, "-c", pvt[P_command].value.string_value, (char *)0);
      _exit(1);			/* don't close files on errors */

    } else {			/* parent */

      close(pipes[P_WRITE]);	/* do not need write end of pipe */
      line_count = 0;

      for(; y<=y_bias+LINES; y++) { /* display screen */

        if ((xc = gets_from_pipe(line, MAXIMUM_LINE_LENGTH, pipes[P_READ]))
             == 0)
          break; /* for */

        line_count++;
        if (line_count >= y_bias) {
          mvaddstr(y-y_bias, x-1, line-1+x_bias); /* move and add next row */
	  move(y-y_bias, xc-x_bias);
	  clrtoeol();
        }

      }	/* forend */

      kill(pid, SIGKILL);	/* don't need any more standard output */
      waitpid(pid, &child_status, 0); /* wait for child to complete */

      close(pipes[P_READ]);	/* close read end of pipe */
      bp = MAXIMUM_BUFFER_SIZE;	/* empty buffer for next loop */

      if ( (child_status != 0) && (child_status != SIGKILL) ) {
	printf("\npaicc - Error in command execution; code = %d!\n",
              child_status);
	fini(-99);
      }

      clrtobot();		/* clear to end-of-screen */

      refresh();		/* update stdscr */

      sleep(pvt[P_millisecond_update_interval].value.integer_value / 1000);

    } /* ifend parent */

  } /* whilend */

} /* end paicc */

void window_ini()
{

  initscr();			/* initialize curses */
  cbreak();			/* enable hot-keys */
  noecho();			/* suppress echo-plex */

} /* end window_ini */

main(argc, argv)
  int argc;
  char *argv[];
{

  evap(&argc, &argv, pdt, NULL, pvt); /* evaluate parameters */

  ini();			/* initialize paicc */

  paicc();			/* paint changed characters */

  fini(0);			/* finish paicc */

} /* end paint_changed_characters */
