#
# evap.pl - evaluate_parameters 2.0 for Perl
#
# lusol@lehigh.EDU, 93/11/07
#
# Made to conform, as much as possible, to the C function evap. Both the
# C and Perl versions of evap are patterned after the Control Data procedure
# CLP$EVALUATE_PARAMETERS for the NOS/VE operating system, although neither
# approaches the richness of CDC's implementation.
#
# The C and Perl versions of evap are available via anonymous FTP from
# ftp.Lehigh.EDU (128.180.2.4) in the directory pub/evap.  
#
# Stephen O. Lidie, Lehigh University Computing Center.
#
# Copyright (C) 1993 - 1994 by Stephen O. Lidie.  All rights reserved.
#
# For related information see the evap/C header file evap.h.
#
#
#                                Introduction
#
# Function evaluate_parameters parses a Unix command line in a simple and
# consistent manner, performs type checking of parameter values, and provides
# the user with first-level help.  evaluate_parameters handles command lines
# in the following format:
#
#   command [-parameters] [file_list]
#
# where parameters and file_list are all optional.  A typical example is the
# C compiler:
#
#   cc -O -o chunk chunk.c
#
# In this case there are two parameters and a file_list consisting of a
# single file name for the cc command.
#
#
#                                  Usage
#
# Usage is similar to getopt/getopts/newgetopt:  define a Parameter Description
# Table declaring a list of command line parameters, their aliases, types and
# default values.  The command line parameter `-help' (alias `-h') is auto-
# matically included by evaluate_parameters.  After the evaluation the values
# of the command line parameters are stored in variable names of the form
# `$opt_parameter', except for lists which are returned as `@opt_parameter',
# where 'parameter' is the full spelling of the command line parameter.  
# NOTE:  values are also returned in the associative array %options, with
# lists being passed as:
#
#   Perl 4: a single string `joined' with the multi-dimensional array
#           emulation character ($;)
#   Perl 5: a reference to a list
# 
# An optional PDT line can be included that tells evaluate_parameters whether
# or not trailing file names can appear on the command line after all the
# parameters.  It can read `no_file_list, `optional_file_list' or
# `required_file_list' and, if not specified, defaults to optional.  Although
# placement is not important, this line is by convention the last line of the
# PDT declaration.
#
# Additionally a Message Module is declared that describes the command and
# provides examples.  Following the main help text an optional series of
# help text messages can be specified for individual command line parameters.
# In the following  sample program all the parameters have this additional
# text which describes that parameter's type.  The leadin character is a dot
# in column one followed by the full spelling of the command line parameter.
# Use `-full_help' rather than `-help' to see this supplemental information.
# 
# This sample program illustrates the various types and how to use evap.  The
# `key' type is a special type that enumerates valid values for the command
# line parameter.  The boolean type may be specified as TRUE/FALSE, YES/NO or
# 1/0.  For Unix, parameters of type `file' have ~ and $HOME expanded, and
# default values `stdin' and `stdout' converted to `-' and `>-', respectively.
# Of special note is the default value '$required':  when specified,
# evaluate_parameters will ensure a value is specified for that command line
# parameter.
#
# All types except `switch' may be `list of', like the `tty' parameter below.
# NOTE: in general you should ALWAYS quote components of your lists, even if
# they're not type string, since evaluate_parameters uses eval to parse them.
# Doing this prevents eval from evaluating expressions that it shouldn't, such
# as file name shortcuts like $HOME, and backticked items like `hostname`.
# Although the resulting PDT looks cluttered, evaluate_parameters knows what
# to do and eliminates superfluous quotes appropriately.
# 
# Finally, you can specify a default value via an environment variable.  If
# a command line parameter is not specified and there is a corresponding
# environment variable defined then evaluate_parameters will use the value
# of the environment variable.  Examine the `command' parameter for the syntax.
# With this feature users can easily customize command parameters to their
# liking.   Although the name of the environment variable can be whatever you
# choose,  the following scheme is suggested for consistency and to avoid
# conflicts in names:  
#
#   . Use all uppercase characters.
#   . Begin the variable name with D_, to suggest a default variable.
#   . Continue with the name of the command or its alias followed by
#     an underscore.
#   . Complete the variable name with the name of the parameter or its
#     alias.
#
# So, for example, D_DISCI_DO would name a default variable for the
# display_option (do) parameter of the display_command_information
# (disci) command.  Works for MS-DOS and Unix.
#
#  
#                               Sample Program
#
#------------------------------ Start of sample1.pl --------------------------
##!/usr/local/bin/perl
##
## evap - essentially identical to the C version of sample!
##        Same as sample.pl except we get our command line
##        parameters from the associative array %options.
##        Also knows about Perl 4 versus Perl 5.
#     
#require "evap.pl";
#
#eval 'BEGIN {}';
#$Perl_version_major = $@ ? 4 : 5;
#
#$PDT =<<'end-of-PDT';
#PDT (./samplemm.a/sample.mm) sample
#  verbose, v: switch
#  command, c: string = D_SAMPLE_COMMAND, "ps -el"
#  scale_factor, sf: real = 1.2340896e-1
#  millisecond_update_interval, mui: integer = $required
#  ignore_output_file_column_one, iofco: boolean = TRUE
#  output, o: file = stdout
#  queue, q: key plotter, postscript, text, printer, keyend = printer
#  destination, d: application = `hostname`
#  tty, t: list of name = ("/dev/console", "/dev/tty0", "/dev/tty1")
#PDTEND optional_file_list
#end-of-PDT
#
#$MM =<<'end-of-MM';
#sample
#
#        A sample program demonstrating typical evaluate_parameters
#        usage.
#
#        Examples:
#
#          sample
#          sample -usage_help
#          sample -help
#          sample -full_help
#          sample -mui 1234
#.verbose
#        A 'switch' type parameter emulates a typical standalone
#        switch. If the switch is specified evaluate_parameters
#        returns a '1'.
#.command
#        A 'string' type parameter is just a list of characters,
#        which must be quoted if it contains whitespace. 
#        NOTE:  for this parameter you can also create and
#        initialize the environment variable D_SAMPLE_COMMAND to
#        override the standard default value for this command
#        line parameter.  All types except 'switch' may have a
#        default environment variable for easy user customization.
#.scale_factor
#        A 'real' type parameter must be a real number that may
#        contain a leading sign, a decimal point and an exponent.
#.millisecond_update_interval
#        An 'integer' type parameter must consist of all digits
#        with an optional leading sign.  NOTE: this parameter's
#        default value is '$required', meaning that
#        evaluate_parameters ensures that this parameter is
#        specified and given a valid value.  All types except
#        'switch' may have a default value of '$required'.
#.ignore_output_file_column_one
#        A 'boolean' type parameter may be TRUE/YES/1 or
#        FALSE/NO/O, either upper or lower case.  If TRUE,
#        evaluate_parameters returns a value of '1', else '0'.
#.output
#        A 'file' type parameter expects a filename.  For Unix
#        $HOME and ~ are expanded.  For evap/Perl 'stdin' and
#        'stdout' are converted to '-' and '>-' so they can be
#        used in a Perl 'open' function.
#.queue
#        A 'key' type parameter enumerates valid values.  Only the
#        specified keywords can be entered on the command line.
#.destination
#        An 'application' type parameter is not type-checked in
#        any - the treatment of this type of parameters is
#        application specific.  NOTE:  this parameter' default
#        value is enclosed in grave accents (or "backticks").
#        evaluate_parameters executes the command and uses it's
#        standard output as the default value for the parameter.
#.tty
#        A 'name' type parameter is similar to a string except
#        that embedded white-space is not allowed.  NOTE: this
#        parameter is also a LIST, meaning that it can be
#        specified multiple times and that each value is pushed
#        onto a Perl LIST variable.  In general you should quote
#        all list elements.  All types except 'switch' may be
#        'list of'.
#end-of-MM
#
#@PDT = split( /\n/, $PDT );
#@MM = split( /\n/, $MM );
#
#&evap( *PDT, *MM );		# evaluate parameters
#print "\nProgram name:\n  $options{'help'}\n";
#
#print "\nverbose = $options{'verbose'}\n";
#print "command = \"$options{'command'}\"\n";
#print "scale_factor  = $options{'scale_factor'}\n";
#print "millisecond_update_interval = $options{'millisecond_update_interval'}\n";
#print "ignore_output_file_column_one = $options{'ignore_output_file_column_one'}\n";
#print "output = $options{'output'}\n";
#print "queue = $options{'queue'}\n";
#print "destination = $options{'destination'}\n";
#if ( $Perl_version_major == 5 ) {
#    eval q[$tty_list = $options{'tty'}; print "Perl 5 'list of' tty = \"", join( '", "', @$tty_list ), "\"\n";];
#} else {
#    print "Perl 4 'list of' tty = \"", join( '", "', split( $;, $options{'tty'} ) ), "\"\n";
#}
#
#print "\nFile names:\n  ", join( ' ', @ARGV ), "\n" if @ARGV;
#------------------------------- End of sample1.pl ---------------------------
#  
#
#                          Human Interface Guidlines
#
# To make evaluate_parameters successful, you, the application developer, must
# follow certain conventions when choosing parameter names and aliases.
#
# Parameter names consist of one or more words, separated by underscores, and
# describe the parameter (for example, OUTPUT and TERMINAL_MODEL).
#
# You can abbreviate parameters:  use the first letter of each word in the
# parameter name.  Do not use underscores.  For example, you can abbreviate
# Command as C and Delay_Period as DP.
#
# There are exceptions to this standard:
#
#   - PASSWORD is abbreviated PW.
#   - The words MINIMUM and MAXIMUM are abbreviated MIN and MAX.  So, the
#     abbreviation for the parameter maximum_byte_count is maxbc.
#     
# 
#                           Revision History 
#
# lusol@Lehigh.EDU 93/05/03 (PDT version 1.2)  Version 1.6
#   . Original release - similar to version 1.6 of the C function evap.
#     Differences: support for `list of'; no support for types `application'
#     and `name'; no suport for default environment variables.
#   . For MS-DOS beasts set the internal variable $evap'DOS = 1.
#
# lusol@Lehigh.EDU 93/08/24 (PDT version 2.0)  Version 2.0
#   . The syntax for initializing the default value(s) for a `list of' command
#     line parameter has been defined and implemented:  ("val 1", "val 2"),
#     strangely enough the same syntax that Perl uses (-:, since that and eval
#     made my life much easier.  NOTE: in general you should quote components
#     of your lists, even if they're not type string, so that Perl/eval can
#     properly parse the list for evaluate_parameters.
#   . Essentially unnoticeable bug fix concerning `optional_file_list'.
#   . In a keyword type defintion 'key key1, key2, keyend' a space character
#     after the comma is no longer required.
#   . Various bug fixes when displaying -help information:  always surround
#     strings with double quotes, separate list with commas, convert boolean
#     values to TRUE or FALSE.
#   . Empty a `list of' variable the first time it's specified on the command
#     line.  Previously, list values specified on the command line were simply
#     pushed onto the list of PDT default values.
#   . Fix expansion of $HOME and ~ for file types that broke when I added the
#     stdin/stdout mod at the last second (-:!  Sigh, my QA was done on DOS.
#   . Handle default environment variables just like evap/C.  (DOS and Unix!)
#   . Similarly to evap/C, return program name in $opt_help.
#   . PDT lines beginning with # are considered comments and are ignored.
#   . In addition to returning command line values in scalar/list variables of
#     the form $opt_parameter and @opt_parameter, return them in the 
#     associative array %options as:
#
#       Perl 4: a single string `joined' with the multi-dimensional array
#               emulation character ($;)
#       Perl 5: a reference to a list
#
#     and indexed by the name of the parameter's full-spelling.
#   . Defer evaluation of file, boolean and backticked items if -help is
#     requested so that the unevaluated PDT values are displayed.
#




sub evap {
    
    package evap;
    
    local( *PDT, *MM ) = @_;	# Parameter Description Table, Message Module
    
    $DOS = 0 unless defined( $DOS  ); # 1 iff MS-DOS, else Unix

    eval 'BEGIN {}';
    $Perl_version_major = $@ ? 4 : 5;

    local( $pdt_reg_exp1 ) = '^(.)(.)(.?)$';
    local( $pdt_reg_exp2 ) = '^TRUE$|^YES$|^1$|^true$|^yes$';
    local( $pdt_reg_exp3 ) = '^FALSE$|^NO$|^0$|^false$|^no$';
    local( $pdt_reg_exp4 ) = '^\s*no_file_list\s*$';
    local( $pdt_reg_exp5 ) = '^\s*optional_file_list\s*$';
    local( $pdt_reg_exp6 ) = '^\s*required_file_list\s*$';
    local( $full_help ) = 0;
    local( $usage_help ) = 0;
    local( $file_list ) = 'optional_file_list';
    local( $error ) = 0;
    local( $pkg ) = (caller)[0];
    local( $value, $rt, $type, $required, @P_PARAMETER, %P_INFO, %P_ALIAS, @P_REQUIRED, %P_VALID_VALUES, %P_ENV );
    local( $option, $default_value, $list, $parameter, $alias, @keys0, @keys, $found, $length, %P_EVALUATE, %P_DEFAULT_VALUE );
    
    
    
    #
    # Verify correctness of the PDT.  Check for duplicate parameter names
    # and aliases.  Extract default values and possible keywords.  Decode
    # the user syntax and convert into a simpler form (ala NGetOpt) for
    # internal use.  Handle 'file list' too.
    #
    
    unshift( @PDT, 'help, disci: switch' );	# supply -help automatically
    @P_PARAMETER = ();		# no parameter names
    %P_INFO = ();		# no encoded parameter information
    %P_ALIAS = ();		# no aliases
    @P_REQUIRED = ();		# no required parameters
    %P_VALID_VALUES = ();	# no keywords
    %P_ENV = ();		# no default environment variables
    %P_EVALUATE = ();		# no PDT values evaluated yet
    %P_DEFAULT_VALUE = ();	# no default values yet

  OPTIONS:
    foreach $option ( @PDT ) {

	next OPTIONS if $option =~ /^#.*|PDT\s+|pdt\s+|PDT$|pdt$/;
	$option =~ s/\s*PDTEND|\s*pdtend//;
	next OPTIONS if $option =~ /^ ?$/;
	
	if ( $option =~ /$pdt_reg_exp4|$pdt_reg_exp5|$pdt_reg_exp6/ ) {
	    $file_list = $option; # remember user specified file_list
	    next OPTIONS;
	}
	
        ($parameter, $alias, $_) = ($option =~ /^\s*(\S*)\s*,\s*(\S*)\s*:\s*(.*)$/);
	&evap_PDT_error( "Error in an evaluate_parameters 'parameter, alias: type' option specification:  \"$option\".\n" )
	    unless defined( $parameter ) && defined( $alias ) && defined( $_ );
	&evap_PDT_error( "Duplicate parameter $parameter:  \"$option\".\n" ) if defined( $P_INFO{$parameter} );
	push( @P_PARAMETER, $parameter ); # update the ordered list of parameter names

        /(\bswitch\b|\binteger\b|\bstring\b|\breal\b|\bfile\b|\bboolean\b|\bkey\b|\bname\b|\bapplication\b)/; # type/list
        ($list, $type, $_)=($`, $1, $');
	&evap_PDT_error( "Parameter $parameter has an undefined type:  \"$option\".\n" ) unless defined( $type );
	&evap_PDT_error( "Expecting 'list of', found:  \"$list\".\n") if $list ne '' && $list !~ /\s*list\s+of\s+/;
        $list = '1' if $list;	# list state = 1, possible default PDT values
        $type = 'w' if $type =~ /^switch$/;
	$type = substr( $type, 0, 1 );

        ($_, $default_value) = /\s*=\s*/ ? ($`, $') : ('', ''); # get possible default value
	if ( $default_value =~ /^([^\(]{1})(\w*)\s*,\s*(.*)/ ) { # if environment variable AND not a list
	    $default_value = $3;
	    $P_ENV{$parameter} = $1 . $2;
	}
        $required = ($default_value eq '$required') ? 'R' : 'O';
        $P_INFO{$parameter} = defined $type ? $required . $type . $list : "";
	push( @P_REQUIRED, $parameter ) if $required =~ /^R$/; # update the list of $required parameters

        if ( $type =~ /^k$/ ) {
	    $_ =~ s/,/ /g;
	    @keys = split( ' ' );
	    pop( @keys );	# remove 'keyend'
	    $P_VALID_VALUES{$parameter} = join( ' ', @keys );
        } #ifend keyword type
	
	foreach $value (keys %P_ALIAS) {
	    &evap_PDT_error( "Duplicate alias $alias:  \"$option\".\n" ) if $alias eq $P_ALIAS{$value};
	}
	$P_ALIAS{$parameter} = $alias; # remember alias

	&evap_PDT_error( "Cannot have 'list of switch':  \"$option\".\n" ) if $P_INFO{$parameter} =~ /^.w1$/;

        if ( $default_value ne '' && $default_value ne '$required' ) {
	    $default_value = $ENV{$P_ENV{$parameter}} if $P_ENV{$parameter} && $ENV{$P_ENV{$parameter}};
	    $P_DEFAULT_VALUE{$parameter} = $default_value;
	    $variable_name_old = "${pkg}\'opt_${parameter}";
	    $variable_name_new = "${pkg}\'options";
            &evap_set_value(0,  $type, $list, $default_value, $variable_name_old, $variable_name_new, $parameter ); # initialize
        }
	
    } # forend OPTIONS

    if ( $error ) {
        print STDERR "Inspect the file \"evap.pl\" for details on PDT syntax.\n";
        exit( 1 );
    }

    # Process arguments from the command line, stopping at the first
    # parameter without a leading dash, or a --.  Convert a parameter
    # alias into its full form, type-check parameter values and store
    # the value into global variables for use by the caller.  When
    # complete call evap_fin to perform final processing.
    
  ARGUMENTS:
    while ( $#ARGV >= $[) {
	
	$option = shift (@ARGV); # get next command line parameter
	$value = undef;		# assume no value
	
	$full_help = 1 if $option =~ /^-full_help$/;
	$usage_help = 1 if $option =~ /^-usage_help$/;
	$option = '-help' if $full_help || $usage_help;
	
	if ( $option =~ /^(--|-)/ ) { # check for end of parameters
	    if ( $option eq '--' ) {
		&evap_fin;
		return( 1 );
	    }
	    $option = $';	# option name without dash
	} else {		# not an option, push it back on the list
	    unshift (@ARGV, $option);
	    &evap_fin;
	    return( 1 );
	}
	
	foreach $alias (keys %P_ALIAS) { # replace any alias with the full parameter spelling
	    $option = $alias if $option eq $P_ALIAS{$alias};
	}
	
	unless  ( defined ( $rt = $P_INFO{$option} ) ) {
	    $found = 0;
	    $length = length( $option );
	    foreach $key (keys %P_INFO) { # try substring match
		if ( $option eq substr( $key, $[, $length ) ) {
		    if ( $found ) {
			print STDERR "Ambiguous parameter: -$option.\n";
			$error++;
			last;
		    }
		    $found = $key; # remember full spelling
		}
	    } # forend
	    $option = $found ? $found : $option;
	    unless  ( defined ( $rt = $P_INFO{$option} ) ) {
		print STDERR "Invalid parameter: -$option.\n";
		$error++;
		next ARGUMENTS;
	    }
	} # unlessend non-substring match
	
	($required, $type, $list) = ( $rt =~ /$pdt_reg_exp1/ ); # unpack encoded information
	
	if ( $type =~ /^w$/ ) {	# a switch type, we're almost finished with it
	    $value = 1;		# supply explicit value
	    next ARGUMENTS;
	}
	
	if ( $#ARGV < $[ ) {	# if argument list is exhausted
	    if ( $type ne 'w' ) { # all but switch type must have a value
		print STDERR ("Value required for parameter -$option.\n");
		$error++;
	    }
	    next ARGUMENTS;
	}
	
	$value = shift (@ARGV);	# get argument value, then perform type-checking
	
	if ( $type =~ /^i$/ ) { # integer
	    if ( $value !~ /^[+-]?[0-9]+$/ ) {
		print STDERR "Expecting integer reference, found \"$value\" for parameter -$option.\n";
		$error++;
		undef $value;
	    }
	    next ARGUMENTS;
	} elsif ( $type =~ /^r$/ ) { # real number, int is also ok
	    if ( $value !~ /^\s*[+-]?(\d+(\.\d*)?|\.\d+)([eE][+-]?\d+)?\s*$/ ) {
		print STDERR "Expecting real reference, found \"$value\" for parameter -$option.\n";
		$error++;
		undef $value;
	    }
	    next ARGUMENTS;
	} elsif ( $type =~ /^s$|^n$|^a$/ ) { # string (or name or application for evap/C compatibility)
	    next ARGUMENTS;
	} elsif ( $type =~ /^f$/ ) { # file
	    if ( length( $value ) > 255 ) {
		print STDERR "Expecting file reference, found \"$value\" for parameter -$option.\n";
		$error++;
		undef $value;
	    }
	    next ARGUMENTS;
	} elsif ( $type =~ /^b$/ ) { # boolean
	    $value =~ tr/a-z/A-Z/;
	    if ( $value !~ /$pdt_reg_exp2|$pdt_reg_exp3/ ) {
		print STDERR "Expecting boolean reference, found \"$value\" for parameter -$option.\n";
		$error++;
		undef $value;
            }
	    next ARGUMENTS;
	} elsif ( $type =~ /^k$/ ) { # keyword
	    #
	    # First try exact match, then substring match.
	    #
	    $found = 0;
	    @keys0 = split( ' ', $P_VALID_VALUES{$option} );
	    @keys = @keys0;
	    while ( ($key = shift( @keys )) && ! $found ) {
		$found = 1 if $value eq $key;
	    } # whilend
	    unless ( $found ) {	# try substring match
		@keys = @keys0;
		$length = length( $value );
		while ( $key = shift( @keys ) ) {
		    if ( $value eq substr( $key, $[, $length ) ) {
			if ( $found ) {
			    print STDERR "Ambiguous keyword for parameter -$option: $value.\n";
			    $error++;
			    last;
			}
			$found = $key; # remember full spelling
		    }
		} # whilend
		$value = $found ? $found : $value;
	    } # unlessend
	    unless ( $found ) {
		print STDERR "\"$value\" is not a valid value for the parameter -$option.\n";
		$error++;
		undef $value;
	    }
	    next ARGUMENTS;
	} # ifend type-check
	
    } continue { # while - update value
    	$list = '2' if $list =~ /^1$/; # list state = 2, empty list this time only
        $variable_name_old = "${pkg}\'opt_${option}";
        $variable_name_new = "${pkg}\'options";
	&evap_set_value(1,  $type, $list, $value, $variable_name_old, $variable_name_new, $option ) if defined( $value );
	@P_REQUIRED = grep( $option ne $_, @P_REQUIRED ); # remove from $required list if specified
	$P_INFO{$option} = $required . $type . '3' if $list; # list state = 3, don't empty list from now on

    } # whilend ARGUMENTS
    
    &evap_fin;
    return( 1 );
    
} # end evap




sub evap'evap_fin {

    #
    # Finish up evaluate_parameters processing:
    #
    # If -usage_help, -help or -full_help was requested then do it and exit.  Else,
    #   
    #  . Store program name in `help' variables.
    #  . Perform deferred evaluations.
    #  . Ensure all $required parameters have been given a value.
    #  . Ensure the validity of the trailing file list.
    #  . Exit with a Unix return code of 1 if there were errors, else return to
    #    the calling Perl program with a return code of 1, indicating success.
    #    (Stupid Unix, always reversing return codes!  Perl is the nuts.)
    #
    
    package evap;
    
    local( $m, $p, $required, $type, $list, $def, $rt, $def, $element, $is_string );

    $variable_name = "\$${pkg}\'opt_help";

    if ( eval( $variable_name ) ) { # see if help was requested
	
	local( $optional );
	local( %parameter_help ) = ();
	local( $parameter_help_in_progress ) = 0;
	local( %type_list ) = (
			       'w', 'switch',
			       'i', 'integer',
			       's', 'string',
			       'r', 'real',
			       'f', 'file',
			       'b', 'boolean',
			       'k', 'key',
			       'n', 'name',
			       'a', 'application',
			       );
	
	print "Command Source:  $0\n\n\n\n" if $full_help;

	# Print the message module text and save any full help.  The key
	# is the parameter name and the value is a list of strings with
	# the newline as a separator.  If there is no message module or
	# it's empty then display an abbreviated usage message.
	
        if ( $usage_help || ! defined( @MM ) || ($#MM < $[) ) {
	    
	    print "\nUsage: $0";
	    $optional = '';
	    foreach $p (@P_PARAMETER) {
		if ( $P_INFO{$p} =~ /^R..?$/ ) { # if $required
		    print " -$P_ALIAS{$p}";
		} else {
		    $optional .= " -$P_ALIAS{$p}";
		}
	    } # forend
	    print " [$optional]" if $optional;
	    print " [file(s)]" if $file_list =~ /$pdt_reg_exp5/;
	    print " file(s)" if $file_list =~ /$pdt_reg_exp6/;
	    print "\n";
	    
        } else {
	    
	  MESSAGE_LINE:
	    foreach $m (@MM) {
		
		if ( $m =~ /^\.(.*)$/ ) { # look for 'dot' leadin character
		    $p = $1; # full spelling of parameter
		    $parameter_help_in_progress = 1;
		    $parameter_help{$p} = "\n";
		    next MESSAGE_LINE;
		} # ifend start of help text for a new parameter
		if ( $parameter_help_in_progress ) { 
		    $parameter_help{$p} .=  $m . "\n";
		} else {
		    print $m, "\n";
		}
		
	    } # forend MESSAGE_LINE
	    
	} # ifend usage_help

	# Pass through the PDT list printing a standard evap help summary.

        print"\nParameters:\n\n";
	
      ALL_PARAMETERS:
        foreach $p (@P_PARAMETER) {
	    
	    if ( $p =~ /^help$/ ) {
		print "-$p, $P_ALIAS{$p}, usage_help, full_help: Display Command Information\n";
                if ( $full_help ) {
         	    print "\n\tDisplay information about this command, which includes\n";
		    print "\ta command description with examples, plus a synopsis of\n";
		    print "\tthe command line parameters.  If you specify -full_help\n";
		    print "\trather than -help complete parameter help is displayed\n";
		    print "\tif it's available.\n\n";
	        }
		next ALL_PARAMETERS;
	    }
	    
	    $rt = $P_INFO{$p};	# get encoded required/type information
	    ($required, $type, $list) = ($rt =~ /$pdt_reg_exp1/); # unpack
	    $type = $type_list{$type};
	    $is_string = ( $type =~ /^string$/ );
	    
	    print "-$p, $P_ALIAS{$p}: ", $list ? 'list of ' : '', $type; 
	    
	    print " ", join( ', ', split( ' ', $P_VALID_VALUES{$p} ) ), ", keyend" if $type =~ /^key$/;
	    
            local( *glob ) = "$pkg".'\'opt_'.$p;
	    if ( $list ) {
                $def = defined( @glob ) ? 1 : 0;
	    } else {
                $def = defined( $glob ) ? 1 : 0;
            }
    
	    if ( ($required =~ /^O$/) || ( $def == 1) ) { # if $optional or defined
		
                if ( $def == 0 ) { # undefined and $optional
    		    print "\n";
                } else {	# defined (either $optional or $required), display the default value(s)
                    if ( $list ) {
			print $P_ENV{$p} ? " = $P_ENV{$p}, " : " = ";
			print $is_string ? "(\"" : "(", $is_string ? join( '", "', @glob ) : join( ', ', @glob),
			      $is_string ? "\")\n" : ")\n";
                    } else {	# not 'list of'
			print $P_ENV{$p} ? " = $P_ENV{$p}, " : " = ";
			print $is_string ? "\"" : "", $glob, $is_string ? "\"\n" : "\n";
                    } # ifend 'list of'
                } # ifend
		
	    } elsif ( $required =~ /R/ ) {
		print $P_ENV{$p} ? " = $P_ENV{$p}, " : " = ";
		print "\$required\n";
	    } else {
		print "\n";
	    } # ifend $optional or defined parameter
	    
            if ( $full_help && defined( $parameter_help{$p} ) ) {
		print "$parameter_help{$p}\n";
	    } elsif ( $full_help ) {
		print "\n\n";
	    }
	    
	} # forend ALL_PARAMETERS
	
	print "\n[file(s)] optionally required by this command\n" if $file_list =~ /$pdt_reg_exp5/;
	print "\nfile(s) required by this command\n" if $file_list =~ /$pdt_reg_exp6/;
	
	print "\n";
	exit( 0 );
	
    } # ifend help requested

    # Evaluate remaining unspecified command line parameters.  This has been deferred until now so that
    # if -help was requested the user sees unevaluated boolean, file and backticked values.

    foreach $parameter (@P_PARAMETER) {
	if ( ! $P_EVALUATE{$parameter} && $P_DEFAULT_VALUE{$parameter} ) {
	    ($required, $type, $list) = ($P_INFO{$parameter} =~ /$pdt_reg_exp1/);
	    if ( $type ne 'w' ) {
		$variable_name_old = "${pkg}\'opt_${parameter}";
		$variable_name_new = "${pkg}\'options";
		$list = 2 if $list; # force re-initialization of the list
		&evap_set_value(1, $type, $list, $P_DEFAULT_VALUE{$parameter}, $variable_name_old, $variable_name_new,
				$parameter ); # final evaluation
	    } # ifend non-switch
	} # ifend not specified
    } # forend all PDT parameters

    # Store program name for caller.

    $variable_name_old = "${pkg}\'opt_help";
    $variable_name_new = "${pkg}\'options";
    &evap_set_value(0,  'w', '', $0, $variable_name_old, $variable_name_new, 'help' ); # initialize program name
    
    # Ensure all $required parameters have been specified on the command line.

    foreach $p (@P_REQUIRED) {
	print STDERR "Parameter $p is required but was omitted.\n";
	$error++;
    } # forend
    
    # Ensure any required files follow, or none do if that is the case.

    if ( $file_list =~ /$pdt_reg_exp4/ && $#ARGV > $[ - 1 ) {
        print STDERR "Trailing file name(s) not permitted.\n";
        $error++;
    } elsif ( $file_list =~ /$pdt_reg_exp6/ && $#ARGV == $[ - 1 ) {
        print STDERR "Trailing file name(s) required.\n";
        $error++;
    }
    
    print STDERR "Type $0 -help for command line parameter information.\n" if $error;

    exit( 1 ) if $error;
    return( 1 );
    
} # end evap_fin




sub evap'evap_PDT_error {

    #
    # Inform the application developer that they've screwed up!
    #

    package evap;

    local( $msg ) = @_;

    print STDERR "$msg";
    $error++;
    next OPTIONS;

} # end evap_PDT_error
   



sub evap'evap_set_value {
    
    #
    # Store a parameter's value; some parameter types require special type
    # conversion.  Store values the old way in scalar/list variables of the
    # for $opt_parameter and @opt_parameter, as well as the new way in an
    # associative array named %options.  In Perl 5 'list of' paramaters are
    # returned as a reference in %options; in Perl 4 these parameters are
    # returned as a string 'joined' with the multi-dimensional array
    # emulation character ($;).
    #
    # Evaluate items in grave accents (backticks), boolean and files if
    # `evaluate' is TRUE.
    #
    # Handle list syntax (item1, item2, ...) for 'list of' types.
    #
    # Lists are a little weird as they may already have default values from the
    # PDT declaration. The first time a list parameter is specified on the
    # command line we must first empty the list of its default values.  The
    # P_INFO list flag thus can be in one of three states: 1 = the list has
    # possible default values from the PDT, 2 = first time for this command
    # line parameter so empty the list and THEN push the parameter's value, and
    # 3 = from now just keep pushing new command line values on the list.
    #
    
    package evap;    
    
    local( $evaluate, $type, $list, $v, *parameter_old, *parameter_new, $parameter_new_hash ) = @_;
    local( $value, @values );

    if ( $list =~ /^2$/ ) {	# empty list of default values
	@parameter_old = ();
	if ( $Perl_version_major == 5 ) {
            eval "\$parameter_new{\$parameter_new_hash} = \\\@parameter_old";
        } else {
            $parameter_new{$parameter_new_hash} = '';
        }
    }

    if ( $list && $v =~ /^\(+[^\)]*\)+$/ ) { # check for list
	@values = eval "$v"; # let Perl do the walking
    } else {
	$v =~ s/["|'](.*)["|']/$1/; # remove any bounding superfluous quotes
	@values = $v;		# a simple scalar	
    } # ifend initialize list of values

    foreach $value (@values) {

        if ( $evaluate ) {
            $P_EVALUATE{$parameter_new_hash} = 'evaluated';
            $value =~ /^(`*)([^`]*)(`*)$/; # check for backticks
	    chop( $value = `$2` ) if $1 eq '`' && $3 eq '`';
	    if ( ! $DOS && $type =~ /^f$/ ) {
                local( @path ) = split( /\//, $value );
	        if ( $value =~ /^stdin$/ ) {
                    $value = '-';
                } elsif ( $value =~ /^stdout$/ ) {
                    $value = '>-';
                } elsif ( $path[$[] =~ /(^~$|^\$HOME$)/ ) {
		    $path[$[] = $ENV{'HOME'};
                    $value = join( '/', @path );
                }
            } # ifend file type

            if ( $type =~ /^b$/ ) {
	        $value = '1' if $value =~ /$evap'pdt_reg_exp2/;
	        $value = '0' if $value =~ /$evap'pdt_reg_exp3/;
            } # ifend boolean type
        } # ifend evaluate

        if ( $list ) {		# extend list with new value
	    push( @parameter_old, $value );
	    if ( $Perl_version_major == 5 ) {
                eval "\$parameter_new{\$parameter_new_hash} = \\\@parameter_old";
            } else {
	        $parameter_new{$parameter_new_hash} = '' unless defined( $parameter_new{$parameter_new_hash} );
                $parameter_new{$parameter_new_hash} .= ($parameter_new{$parameter_new_hash} ? $; : '') . $value;
            }
        } else {		# store scalar value
	    $parameter_old = $value;
            $parameter_new{$parameter_new_hash} = $value;
        }

    } # forend
	
} # end evap_set_value




1;
