#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include "ncd.h"

/*************************************************************************/
/* basename must end with a slash. If {baseNode} is NULL, it is also
   created based on {baseName} */

DirNode *readDirsInNodes(char *baseName, DirNode * baseNode)
{
	char fullName[PATH_MAX];
	DIR *dir;
	int testproc;
	DirNode *lastUp;
	static int i;
	static DirNode *newNode;
	static struct dirent *entry;
	static char linkName[PATH_MAX];
	static struct stat entryStat;

	if ((_verbose)&&(!_cursesOn))
		fprintf(stderr, "%s\n", baseName);

	if (baseNode == NULL) {
		baseNode = malloc(sizeof(DirNode));
		if (baseNode == NULL) {
			cleanUp();
			fprintf(stderr, "error: no memory\n");
			exit(1);
		}
		baseNode->left = baseNode->right = baseNode->up =
			baseNode->down = NULL;
		baseNode->lname = NULL;
		baseNode->name = getDirName(baseName);
	}
	dir = opendir(baseName);
	testproc = (strcmp(baseName, "/") == 0);
	lastUp = NULL;
	while ((entry = readdir(dir)) != NULL) {
		if ((strcmp(entry->d_name, ".") != 0) &&
			(strcmp(entry->d_name, "..") != 0) /* &&
			((!testproc) || (strcmp(entry->d_name, "proc") != 0)) */) {
			sprintf(fullName, "%s%s", baseName, entry->d_name);
			i = stat(fullName, &entryStat);
			if (S_ISDIR(entryStat.st_mode)) {
				newNode = malloc(sizeof(DirNode));
				if (newNode == NULL) {
					cleanUp();
					fprintf(stderr, "error: no memory\n");
					exit(1);
				}
				newNode->name = strdup(entry->d_name);
				if (newNode->name == NULL) {
					cleanUp();
					fprintf(stderr, "error: no memory\n");
					exit(1);
				}
				newNode->left = baseNode;
				newNode->right = NULL;
				newNode->up = lastUp;
				newNode->down = NULL;
				if (baseNode->right == NULL)
					baseNode->right = newNode;
				if (lastUp != NULL)
					lastUp->down = newNode;
				lastUp = newNode;
				i = lstat(fullName, &entryStat);
				if (S_ISLNK(entryStat.st_mode)) {
					i = readlink(fullName, linkName, PATH_MAX);
					linkName[(i < 0) ? 0 : i] = '\0';
					if ((_verbose)&&(!_cursesOn))
						fprintf(stderr, "%s -> %s\n", fullName, linkName);
					newNode->lname = strdup(linkName);
					if (newNode->lname == NULL) {
						cleanUp();
						fprintf(stderr, "error: no memory\n");
						exit(1);
					}
				}
				else {	/* not a link */
					newNode->lname = NULL;
					addSlash(fullName);
					if ((!testproc) || (strcmp(entry->d_name, "proc") != 0))
						readDirsInNodes(fullName, newNode);
				}
			}
		}
	}
	closedir(dir);

	return baseNode;
}

/*************************************************************************/

DirNode *readNodesFromFile(char *fname)
{
	char line[PATH_MAX + 3];
	DirNode *node, *first, *oldNode;
	FILE *f;
	size_t len;

	f = fopen(fname, "r");
	if (f == NULL) {
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "warning: can't read file %s\n", fname);
		return NULL;
	}
	node = first = oldNode = NULL;
	while (fgets(line, PATH_MAX + 3, f) != NULL) {
		len = strlen(line);
		if (len < 2)
			continue;
		line[len - 1] = 0;	/* strip \n */
		switch (line[0]) {
		case '>':	/* son */
			node = malloc(sizeof(DirNode));
			if (node == NULL) {
				cleanUp();
				fprintf(stderr, "error: no memory\n");
				exit(1);
			}
			node->name = strdup(line + 2);
			if (node->name == NULL) {
				cleanUp();
				fprintf(stderr, "error: no memory\n");
				exit(1);
			}
			node->lname = NULL;
			node->left = oldNode;
			node->right = node->up = node->down = NULL;
			if (oldNode == NULL)
				first = node;
			else
				oldNode->right = node;

			oldNode = node;
			break;
		case '<':	/* one up */
			if ((oldNode == NULL)) {
				cleanUp();
				fprintf(stderr, "error: invalid file %s\n", fname);
				exit(1);
			}
			oldNode = oldNode->left;
			break;
		case 'V':	/* brother */
			if ((oldNode == NULL) || (oldNode->left == NULL)) {
				cleanUp();
				fprintf(stderr, "error: invalid file %s\n", fname);
				exit(1);
			}
			node = malloc(sizeof(DirNode));
			if (node == NULL) {
				cleanUp();
				fprintf(stderr, "error: no memory\n");
				exit(1);
			}
			node->name = strdup(line + 2);
			if (node->name == NULL) {
				cleanUp();
				fprintf(stderr, "error: no memory\n");
				exit(1);
			}
			node->lname = NULL;
			node->left = oldNode->left;
			node->up = oldNode;
			node->down = node->right = NULL;
			oldNode->down = node;
			oldNode = node;
			break;
		case 'L':
			if (oldNode == NULL) {
				cleanUp();
				fprintf(stderr, "error: invalid file %s\n", fname);
				exit(1);
			}
			oldNode->lname = strdup(line + 2);
			if (oldNode->lname == NULL) {
				cleanUp();
				fprintf(stderr, "error: no memory\n");
				exit(1);
			}
			break;
		default:
			cleanUp();
			fprintf(stderr, "error: invalid file %s\n", fname);
			exit(1);
		}
	}
	if ((oldNode != NULL) && (oldNode->left != NULL) && ((_verbose)&&(!_cursesOn)))
		fprintf(stderr, "warning: possibly truncated file %s\n", fname);

	fclose(f);
	return first;
}

/*************************************************************************/

void writeSubNodeTreeToFile(FILE * f, DirNode * node)
{
	fprintf(f, "%s\n", node->name);
	if (node->lname != NULL)
		fprintf(f, "L %s\n", node->lname);
	if (node->right) {
		fprintf(f, "> ");
		writeSubNodeTreeToFile(f, node->right);
	}
	if (node->down) {
		fprintf(f, "V ");
		writeSubNodeTreeToFile(f, node->down);
	}
	else
		fprintf(f, "< \n");
}

/*************************************************************************/

void writeNodesToFile(char *fname, DirNode * baseNode)
{
	FILE *f;

	f = fopen(fname, "w");
	if (f == NULL) {
		if ((_verbose)&&(!_cursesOn))
			fprintf(stderr, "warning: can't write file %s\n", fname);
		return;
	}
	if (baseNode != NULL) {
		fprintf(f, "> ");
		writeSubNodeTreeToFile(f, baseNode);
	}
	fclose(f);
}

/*************************************************************************/

void delNodesFromMemory(DirNode * baseNode)
{
	if (baseNode == NULL)
		return;

	delNodesFromMemory(baseNode->right);
	delNodesFromMemory(baseNode->down);
	if (baseNode->name)
		free(baseNode->name);
	if (baseNode->lname)
		free(baseNode->lname);
}

/*************************************************************************/
/* showlink==0 no   ==1 ->link  ==-1 @ */

char *getNodeName(DirNode * node, int showlink)
{
	static char s[PATH_MAX * 2];

	s[0] = '\0';
	strcat(s, " ");
	strcat(s, node->name);
	if (node->lname != NULL) {
		if (showlink==1) {
			strcat(s, " -> ");
			strcat(s, node->lname);
			strcat(s, "/");
		}
		else if (showlink==-1)
			strcat(s, "@");
	}
	strcat(s, " ");
	return s;
}

/*************************************************************************/
/* withlink==0 no   ==1 ->link  ==-1 @ */

char *getNodeFullPath(DirNode * node, int withlink, int relative, DirNode * relNode, int use_xroot )
{
	char s[PATH_MAX * 2];
	char *sn;

	sn = getNodeName(node, withlink);
	if ((relative) || (node->left != NULL)) {
		strcpy(s, sn + 1);
		strcpy(sn, s);
		sn[strlen(sn) - 1] = '\0';
	}
	else
		sn[0] = 0;

	s[0] = 0;

	while (node->left != NULL) {
		if ((relative) && (node == relNode))
			break;

		node = node->left;

		if ((relative) || (node->left != NULL)) {
			strcpy(s, node->name);
			addSlash(s);
			strcat(s, sn);
			strcpy(sn, s);
		}
	}
	if (!relative) {
		if (use_xroot)
			strcpy(s, _xroot);
		else
			strcpy(s, _root);
		addSlash(s);
		strcat(s, sn);
		strcpy(sn, s);
	}

	return sn;
}

/*************************************************************************/

int numerateNodeTree(DirNode * rootNode, int x, int y)
{
	int len;

	if (rootNode == NULL)
		return y;

	rootNode->x = x;
	rootNode->y = y;
	len = strlen(rootNode->name);
	if (rootNode->lname != NULL)
		len++;

	if (rootNode->right)
		y = numerateNodeTree(rootNode->right, x + len + 2 + 3, y);
	if (rootNode->down)
		y = numerateNodeTree(rootNode->down, x, y + 1);

	return y;
}

/*************************************************************************/

DirNode *followLink(DirNode * rootNode, DirNode * node, int isLast )
{
	DirNode *dn;
	char s[PATH_MAX * 2];
	char * ss, * st;

	if (node == NULL)
		return NULL;
	if (node->lname == NULL)
		return NULL;
	_nLinks++;
	if (_nLinks>MAX_LINKS)
		return NULL;

	if (node->lname[0]=='/') {  /* absolute */
		stripSlash(_root);
		if (strstr(node->lname,_root)==node->lname) {
			strcpy(s,_xroot);
			strcat(s,(node->lname)+strlen(_root));
		}
		else
			strcpy(s,node->lname);
		addSlash(_root);
	}
	else {  /* relative */
		strcpy(s,getNodeFullPath(node, 0, 0, NULL, 1));
		addSlash(s);
		strcat(s,"../");
		strcat(s,node->lname);
	}
	while (1) {
		compactAbsDir(s);
		dn = locatePathOrSo(s,&ss);

		if (dn->lname!=NULL) {
			if ((isLast)&&((ss[0]=='\0')||((ss[0]=='/')&&(ss[1]=='\0'))))
				return dn;
			dn = followLink(rootNode, dn, 0);
			if (dn==NULL) 
				return NULL;
			else  {
				st = getNodeFullPath(dn, 0, 0, NULL, 1);
				addSlash(st);
				strcat(st,ss);
				strcpy(s,st);
			}
		}
		else {
			if ((ss[0]=='\0')||((ss[0]=='/')&&(ss[1]=='\0'))) 
				return dn;
			else 
				return NULL;
		}
	}

	return NULL;
}

/*************************************************************************/

DirNode *getNodeFollowLink(DirNode * rootNode, DirNode * node)
{
	DirNode *dn;

	if (node == NULL)
		return NULL;
	if (node->lname == NULL)
		return node;
		
	_nLinks = 0;
	dn = followLink(rootNode, node, 1);
	
	if (dn==NULL) 
		return node;
	else 
		return dn;
}

/*************************************************************************/

char *getTreeLine(DirNode * node, int lineart, int marksel)
{
#define MAXLEN (PATH_MAX*5)		/* (PATH_MAX/2)*6+PATH_MAX + some */

	static char line[MAXLEN];
	char tline[MAXLEN];
	size_t len, i;

	line[0] = '\0';

	if (node == NULL)
		return NULL;

	do {
		tline[0] = '\0';
		if (node->left != NULL) {
			if (node->up == NULL) {
				if (node->down == NULL) {
					if (lineart) {
						tline[0] = tline[1] = tline[2] = __ACS_HL;
						tline[3] = 0;

					}
					else
						strcpy(tline, "---");
				}
				else {
					if (lineart) {
						tline[0] = tline[2] = __ACS_HL;
						tline[1] = __ACS_TT;
						tline[3] = 0;
					}
					else
						strcpy(tline, "-+-");
				}
			}
			else {
				if (node->down == NULL) {
					if (lineart) {
						tline[0] = ' ';
						tline[1] = __ACS_LLC;
						tline[2] = __ACS_HL;
						tline[3] = 0;
					}
					else
						strcpy(tline, " `-");
				}
				else {
					if (lineart) {
						tline[0] = ' ';
						tline[1] = __ACS_LT;
						tline[2] = __ACS_HL;
						tline[3] = 0;
					}
					else
						strcpy(tline, " |-");
				}
			}
		}
		if (_curNode == node) {
			strcat(tline, " ");
			tline[strlen(tline) - 1] = __SEL_ON;
		}
		strcat(tline, getNodeName(node, (_showlink?1:-1)));
		if (_curNode == node) {
			strcat(tline, " ");
			tline[strlen(tline) - 1] = __SEL_OFF;
		}
		strcat(tline, line);
		strcpy(line, tline);

		if (node->up != NULL) {
			node = node->left;
			break;
		}

		node = node->left;
	} while (node != NULL);

	while (node != NULL) {
		tline[0] = '\0';
		if (node->left != NULL) {
			if (node->down != NULL) {
				if (lineart) {
					strcat(tline, "   ");
					tline[strlen(tline) - 2] = __ACS_VL;
				}
				else
					strcat(tline, " | ");
			}
			else
				strcat(tline, "   ");
		}
		len = strlen(node->name) + ((node->lname != NULL) ? 1 : 0) + 2;
		for (i = 0; i < len; i++)
			strcat(tline, " ");
		strcat(tline, line);
		strcpy(line, tline);
		node = node->left;
	}

	return line;
}

/*************************************************************************/
