#include <stdio.h>
#include <ncurses.h>

#include "ncd.h"

/*************************************************************************/

void ioInitialize(void)
{
	if (!(initscr())) {
		cleanUp();
		fprintf(stderr, "error: can't initialize screen\n");
		exit(1);
	}

	_cursesOn = 1;
	
	_lines = LINES;
	_cols = COLS;
	if (_lines < MIN_LINES) {
		cleanUp();
		fprintf(stderr, "error: not enought screen lines\n");
		exit(1);
	}
	if (_cols < MIN_COLS) {
		cleanUp();
		fprintf(stderr, "error: not enought screen columns\n");
		exit(1);
	}

	nonl();
/* intrflush(stdscr,FALSE); */

	cbreak();
	noecho();
/* 	halfdelay(10); */
	keypad(stdscr, TRUE);
	meta(stdscr, TRUE); 
/*	notimeout(stdscr, TRUE); */

	prepareColorModes();
	attrset(getColorMode(0));
	clear();
}

/*************************************************************************/

void ioTerminate(void)
{
	nl();
	nocbreak();
	echo();

	standend();
	clear();
	endwin();
	refresh();
	
	_cursesOn = 0;
}

/*************************************************************************/

void prepareColorModes(void)
{
	start_color();
	_use_color = has_colors();
	if (_use_color) {
		init_pair(1, COLOR_WHITE, COLOR_BLUE);
		init_pair(2, COLOR_BLACK, COLOR_CYAN);
	}

	if (_lineart == 2)
		_lineart = _use_color;

}

/*************************************************************************/
/* 0 = normal/normalcolor, 1 = reverse/reversecolor */

int getColorMode(int c)
{
	switch (c) {
	case 1:
		if (_use_color)
			return COLOR_PAIR(2);
		else
			return A_REVERSE;
	case 0:
	default:
		if (_use_color)
			return COLOR_PAIR(1);
		else
			return A_NORMAL;
	}
}

/*************************************************************************/
/* getKey / kbHit variables */

int __gotKey;   /* != 0 if kbHit got a key.  I use this instead 
                  of calling ungetch(), because that does not works well */
int __theKey;   /* the key that kbHit() has got */

/*************************************************************************/
/* delay=0 -> no delay. <0 -> block */


int getKey(int delay)
{
	if (__gotKey) {
		__gotKey = 0;
		return __theKey;
	}
	
	timeout(delay); 
	return getch();
}

/*************************************************************************/

int kbHit(void)
{
	int ch;

	if (__gotKey)
		return 1;
		
	ch = getKey(0);
	if (ch != ERR) {
		__gotKey = 1;
		__theKey = ch;
		
		return 1;
	}

	return 0;
}

/*************************************************************************/

void showHelp()
{
	if ((_lines<HLP_LINES)||(_cols<HLP_COLS))
		showHelpNoSpace();
	else
		showHelpScreen(); 
		
	getKey(-1);
}

/*************************************************************************/
/* quit = 2 -> accept
quit = 1 -> cancel
quit = 3 -> rescan 
quit = 4 -> toggle scope */


DirNode * selectANode( int * quit )
{
#define CTRL(n)  (n-'a'+1)

	int ch;

	*quit = 0;

	if (!_cursesOn)
		ioInitialize();

	_curNode = locatePathOrSo(_cwd,NULL);
	_x0 = 0;
	if (_lastNode->y < (_lines-5))
		_y0 = 0;
	else {
		_y0 = _curNode->y - ((_lines - 6) / 2);
		if (_lastNode->y-_y0<_lines-6)
			_y0 = _lastNode->y-(_lines-6);
	}
	_searchPath[0] = 0;
	
	paintLayout();
	dataRefresh(_x0, _y0, _curNode);

	do {
		ch = getKey(-1);

		switch (ch) {

		case KEY_DOWN:
		case CTRL('n'):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, getNodeCursDown(_curNode));
			break;
		case KEY_UP:
		case CTRL('p'):
			_searchPath[0] = 0; 
			dataRefresh(_x0, _y0, getNodeCursUp(_curNode));
			break;
		case KEY_LEFT:
		case CTRL('b'):
			_searchPath[0] = 0; 
			dataRefresh(_x0, _y0, getNodeCursLeft(_curNode));
			break;
		case KEY_RIGHT:
		case CTRL('f'):
			_searchPath[0] = 0; 
			dataRefresh(_x0, _y0, getNodeCursRight(_curNode));
			break;
		case KEY_HOME:
		case CTRL('a'):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, _rootNode);
			break;
		case KEY_END:
		case CTRL('e'):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, getLastDescendant(_curNode));
			break;
		case KEY_PPAGE:
		case CTRL('u'):
			_searchPath[0] = 0;
			if (_curNode->y < (_lines - 6))
				dataRefresh(0, 0, _rootNode );
			else
				dataRefresh(_x0, _y0 - (_lines - 6), getFirstNodeInLevel(_rootNode, _curNode->y - (_lines - 6)));
			break;
		case KEY_NPAGE:
		case CTRL('v'):
			_searchPath[0] = 0;
			if (_curNode->y + (_lines - 6) >=_lastNode->y)
				dataRefresh(_x0, _y0, _lastNode );
			else {
				if (_y0+(_lines-6)*2<_lastNode->y)
					dataRefresh(_x0, _y0 + (_lines - 6), getFirstNodeInLevel(_rootNode, _curNode->y + (_lines - 6)));
				else
					dataRefresh(_x0, _lastNode->y-(_lines-6), getFirstNodeInLevel(_rootNode, _curNode->y + (_lines - 6)));		
			}
			break;
		case CTRL('o'):
		case KEY_F(7):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, getNodeLnUp(_curNode));
			break;
		case CTRL('j'):
		case KEY_F(8):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, getNodeLnDn(_curNode));
			break;			
		case CTRL('t'):
			_searchPath[0] = 0; 
			dataRefresh(_x0, _y0, getNodePrev(_curNode));
			break;			 
		case CTRL('x'):
		case KEY_F(10):
			*quit = 1;
			break;
		case CTRL('k'):
		case KEY_F(9):
			editSearch(0);
			break;
		case KEY_BACKSPACE:
			editSearch(8);
			break;				
		case CTRL('i'):
		case KEY_F(1):
			showHelp();
			paintLayout();
			dataRefresh(_x0, _y0, _curNode);
			break;
		case KEY_F(2):
		case CTRL('r'): 
			*quit = 3;
			break;
		case KEY_F(3):
		case CTRL('y'):
			*quit = 4;
			strcpy(_cwd,getNodeFullPath(_curNode, 0, 0, NULL, 1 ));
			break;
		case CTRL('g'):
		case KEY_F(6):
			_showlink = !_showlink;
			paintLayout();
			dataRefresh(_x0, _y0, _curNode);
			break;
		case CTRL('l'):			
		case KEY_F(5):
			_searchPath[0] = 0;
			dataRefresh(_x0, _y0, getNodeFollowLink(_rootNode, _curNode));
			break;
		case KEY_DL:
		case KEY_EOL:
		case CTRL('d'):
		case KEY_DC:
			_searchPath[0] = 0; 
			dataRefresh(_x0, _y0, _curNode);
			break;
		case CTRL('w'):
		case KEY_F(4):
			clear();
			paintLayout();
			dataRefresh(_x0, _y0, _curNode);
			break;
		case KEY_ENTER:
		case CTRL('m'):
			*quit = 2;
			break;
		case 27:
			*quit = 1;
			while ((ch = getKey(100)) != ERR)
				*quit = (ch == 'x'); 
			break;
		default:
			if ((ch > 0) && (ch <= 255))
				editSearch(ch);
			else {
				dataRefresh(_x0, _y0, _curNode);
				beep();
			}
			break;
		}
	} while (!*quit);

	if (*quit<=2)	
		ioTerminate();
	
	if (*quit==2)
		return _curNode;
	else
		return NULL;
}

/*************************************************************************/
