/*
*  gcal_hdy.c:  Manage the eternal holiday list.
*
*
*  Copyright (C) 1994, 1995, 1996 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for ANY damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*
*    Free Software Foundation, Inc.
*    59 Temple Place - Suite 330
*    Boston, MA 02111-1307,  USA
*/



#ifdef RCSID
static char rcsid[]="$Id: gcal_hdy.c 1.00 1996/03/01 01:00:00 tom Exp $";
#endif



/*
*  Include header files.
*/
#include "gcal_tai.h"
#include "gcal.h"



/*
*  Function prototypes.
*/
#if __cplusplus
extern "C"
{
#endif
/*
************************************************** Defined in `gcal_tty.c'.
*/
IMPORT void
print_text __P_((FILE *fp,
                 char *text_line));
/*
************************************************** Defined in `gcal_utl.c'.
*/
IMPORT VOID_PTR
my_malloc __P_((const int   amount,
                const int   exit_status,
                const char *module_name,
                const long  module_line,
                const char *var_name,
                const int   var_contents));
IMPORT int
my_atoi __P_((const char *string));
IMPORT int
asc_sort __P_((const char **a,
               const char **b));
IMPORT int
des_sort __P_((const char **a,
               const char **b));
IMPORT Bool
is_presorted __P_((char **table,
                   int   elems));
IMPORT void
resort __P_((      char **table,
             const int    elems));
#if USE_GER
IMPORT const char *
short_day_name __P_((const int day));
#else /* !USE_GER */
IMPORT const char *
day_suffix __P_((int day));
IMPORT const char *
short3_day_name __P_((const int day));
#endif /* !USE_GER */
IMPORT const char *
short_month_name __P_((const int month));
IMPORT Bool
doy2date __P_((      int  doy,
               const int  is_leap_year,
                     int *day,
                     int *month));
IMPORT int
weekday_of_date __P_((const int day,
                      const int month,
                      const int year));
IMPORT int
day_of_year __P_((const int day,
                  const int month,
                  const int year));
IMPORT int
days_of_february __P_((const int year));
IMPORT int
knuth_easter_formula __P_((const int year));
/*
************************************************** Defined in `gcal_hdy.c'.
*/
EXPORT int
eval_holiday __P_((      int  day,
                   const int  month,
                   const int  year,
                   const int  wd,
                   const Bool forwards));
EXPORT void
print_all_holidays __P_((      Bool init_data,
                         const Bool detect));
LOCAL void
print_single_holiday __P_((      Bool  init_data,
                           const Bool  detect,
                           const char *holiday_name,
                           const char *holiday_prefix,
                                 int   day,
                                 int   month,
                           const int   year,
                                 int  *hd_elems,
                           const int   fday,
                           const int   count));
#if __cplusplus
}
#endif



/*
*  Declare public(extern) variables.
*/
IMPORT const int   dvec[MONTH_MAX];            /* Amount of days in months */
IMPORT Hls_struct  ehls1s;                     /* Effective hls 1 start (current day) */
IMPORT Hls_struct  ehls1e;                     /* Effective hls 1 end (current day) */
IMPORT Hls_struct  ehls2s;                     /* Effective hls 2 start (holiday) */
IMPORT Hls_struct  ehls2e;                     /* Effective hls 2 end (holiday) */
IMPORT int         len_year_max;               /* String length of the maximum year able to compute */
IMPORT int         warning_level;              /* --debug[=0...WARN_LVL_MAX] */
IMPORT int         month;                      /* Current month */
IMPORT int         year;                       /* Current year */
IMPORT int         act_day;                    /* Actual day */
IMPORT int         act_month;                  /* Actual month */
IMPORT int         act_year;                   /* Actual year */
IMPORT int         fiscal_month;               /* Starting month of a fiscal year */
IMPORT int         is_leap_year;               /* Is current year a leap year? */
IMPORT int         holiday_vector[MONTH_MAX][MONTH_MAX];  /* Stores the holiday dates */
IMPORT char       *s1;                         /* General purpose text buffer */
IMPORT char       *s2;                         /* General purpose text buffer */
IMPORT Bool        emu_hls;                    /* Must we emulate the highlighting sequences? */
#if USE_RC
IMPORT Bool        rc_enable_hda_flag;         /* [-c]e */
IMPORT Bool        rc_enable_hdl_flag;         /* [-c]E */
IMPORT Bool        is_1month_mode;             /* [-c]<n>w and complete week is in month */
IMPORT Bool        is_2month_mode;             /* [-c]<n>w and only part of week is in month */
#endif
IMPORT Bool        cal_special_flag;           /* -j */
IMPORT Bool        holiday_flag;               /* -n|N */
IMPORT Bool        hd_legal_days_only;         /* -N */
IMPORT Bool        hd_sort_des_flag;           /* [-n|N]- */
IMPORT Bool        hd_special_flag;            /* -jn */
IMPORT Bool        hd_both_dates_flag;         /* -jnb */
IMPORT Bool        hd_title_flag;              /* -X */
IMPORT Bool        is_fiscal_year;             /* `:' char found in argument (`mm':`yyyy') */
IMPORT Bool        is_3month_mode;             /* Argument is "." or ".+" or ".-" */
IMPORT Bool        is_3month_mode2;            /* Argument is ".." -> current quarter of actual year */



/*
   Define public(extern) variables.
*/
PUBLIC char       *hd_table[HD_ELEMS_MAX];     /* Stores the fixed eternal holiday list texts */



#ifdef ANSI_PROTO
PUBLIC int
eval_holiday (      int  day,
              const int  month,
              const int  year,
              const int  wd,
              const Bool forwards)
#else /* !ANSI_PROTO */
   PUBLIC int
eval_holiday (day, month, year, wd, forwards)
         int  day;
   const int  month;
   const int  year;
   const int  wd;
   const Bool forwards;
#endif /* !ANSI_PROTO */
/*
   Detects first (`forwards'==TRUE) or last (`forwards'==!TRUE) appearance
     of weekday `wd' in given date starting the search at `day'
     and returns the day number of `wd' in month.
*/
{
   LOOP
    {
      if (weekday_of_date (day, month, year) == wd)
        break;
      else
        (forwards) ? day++ : day--;
    }
   /*
      Correction for missing days during Gregorian reformation.
   */
   if (   (year == GREG_YEAR)
       && (month == GREG_MONTH)
       && (day >= GREG_F_DAY)
       && (day <= GREG_L_DAY))
     day = (forwards)
       ? eval_holiday (GREG_L_DAY+1, month, year, wd, forwards)
       : eval_holiday (GREG_F_DAY-1, month, year, wd, forwards);

   return day;
}



#ifdef ANSI_PROTO
PUBLIC void
print_all_holidays (      Bool init_data,
                    const Bool detect)
#else /* !ANSI_PROTO */
   PUBLIC void
print_all_holidays (init_data, detect)
         Bool init_data;
   const Bool detect;
#endif /* !ANSI_PROTO */
/*
   Displays all generated holidays (global `hd_table') in a sorted manner.
*/
{
   register int  easter;
   register int  i=0;
   register int  j=0;
   register int  day;
   register int  fday=0;
   register int  count=(fiscal_month>MONTH_MIN) ? 2 : 1;
   auto     int  hd_elems=0;


   if (   init_data
       || detect)
    {
      if (init_data)
        for (i=0 ; i < MONTH_MAX ; i++)
          for (j=0 ; j < MONTH_MAX ; j++)
            holiday_vector[i][j] = 0;
      if (fiscal_month > MONTH_MIN)
       {
         fday = day_of_year (DAY_MIN, fiscal_month, year);
         j = is_leap_year;
         i = year;
       }
      do
       {
         count--;
         easter = knuth_easter_formula (year);
#if USE_GER
         print_single_holiday (init_data, detect, "Neujahr",
                               "", DAY_MIN, MONTH_MIN, year, &hd_elems, fday, count);
         if (init_data)
           init_data = FALSE;
         print_single_holiday (init_data, detect, "St. Valentinstag",
                               DIS_HLS_PREF, 14, 2, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Rosenmontag",
                               DIS_HLS_PREF, easter-48, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Fasching/Fastnacht",
                               DIS_HLS_PREF, easter-47, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Aschermittwoch",
                               DIS_HLS_PREF, easter-46, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "1'ter April",
                               DIS_HLS_PREF, DAY_MIN, 4, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Karfreitag",
                               "", easter-2, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Ostersonntag",
                               "", easter, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Ostermontag",
                               "", easter+1, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Maifeiertag (Tag der Arbeit)",
                               "#", DAY_MIN, 5, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 5, year, DAY_MAX, TRUE);
         if (day_of_year (day, 5, year) != easter+49)
           day += DAY_MAX;
         print_single_holiday (init_data, detect, "Muttertag",
                               DIS_HLS_PREF, day, 5, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Christi Himmelfahrt/Vatertag",
                               "#", easter+39, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Pfingstsonntag",
                               "", easter+49, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Pfingstmontag",
                               "", easter+50, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Fronleichnam",
                               "#", easter+60, 0, year, &hd_elems, fday, count);
         if (year >= 1990)
           print_single_holiday (init_data, detect, "Tag der Deutschen Einheit",
                                 "#", 3, 10, year, &hd_elems, fday, count);
         else
           if (year > 1953)
             print_single_holiday (init_data, detect, "Tag der Deutschen Einheit",
                                   "#", 17, 6, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 10, year, DAY_MAX, TRUE);
         print_single_holiday (init_data, detect, "Erntedankfest",
                               DIS_HLS_PREF, day, 10, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Reformationstag",
                               DIS_HLS_PREF, dvec[10-1], 10, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Allerheiligen",
                               "#", DAY_MIN, 11, year, &hd_elems, fday, count);
         day = weekday_of_date (DAY_MIN, 11, year);
         day = (day<3) ? 18-day : 25-day;
         print_single_holiday (init_data, detect, "Bu"SZ"- und Bettag",
                               (year>1994) ? DIS_HLS_PREF: "#", day, 11, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "St. Nikolaustag",
                               DIS_HLS_PREF, 6, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Heiligabend",
                               DIS_HLS_PREF, 24, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "1'ter Weihnachtstag",
                               "", 25, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "2'ter Weihnachtstag",
                               "", 26, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Silvester",
                               DIS_HLS_PREF, dvec[MONTH_MAX-1], MONTH_MAX, year, &hd_elems, fday, count);
#  if USE_CATH
/*
         print_single_holiday (init_data, detect, "Mari"AE" Lichtmess",
                               DIS_HLS_PREF, 2, 2, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Verk"UE"ndigung",
                               DIS_HLS_PREF, 25, 3, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Heimsuchung",
                               DIS_HLS_PREF, 2, 7, year, &hd_elems, fday, count);
*/
         print_single_holiday (init_data, detect, "Mari"AE" Himmelfahrt",
                               DIS_HLS_PREF, 15, 8, year, &hd_elems, fday, count);
/*
         print_single_holiday (init_data, detect, "Mari"AE" Geburt",
                               DIS_HLS_PREF, 8, 9, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Namen",
                               DIS_HLS_PREF, 12, 9, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Mutterschaft",
                               DIS_HLS_PREF, 11, 10, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Opferung",
                               DIS_HLS_PREF, 21, 11, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Empf"AE"ngnis",
                               DIS_HLS_PREF, 8, 12, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Mari"AE" Erwartung",
                               DIS_HLS_PREF, 18, 12, year, &hd_elems, fday, count);
*/
         print_single_holiday (init_data, detect, "Heilige Drei K"OE"nige",
                               DIS_HLS_PREF, 6, MONTH_MIN, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Palmsonntag",
                               DIS_HLS_PREF, easter-DAY_MAX, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Gr"UE"ndonnerstag",
                               DIS_HLS_PREF, easter-3, 0, year, &hd_elems, fday, count);
         day = eval_holiday (24, MONTH_MAX, year, DAY_MAX, FALSE);
         print_single_holiday (init_data, detect, "4'ter Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "3'ter Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "2'ter Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         if (day > 0)
           print_single_holiday (init_data, detect, "1'ter Advent",
                                 DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         else
           print_single_holiday (init_data, detect, "1'ter Advent",
                                 DIS_HLS_PREF, dvec[11-1]+day, 11, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "Totensonntag",
                               DIS_HLS_PREF, dvec[11-1]+day, 11, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "Volkstrauertag",
                               DIS_HLS_PREF, dvec[11-1]+day, 11, year, &hd_elems, fday, count);
#  endif /* USE_CATH */
#else /* !USE_GER */
         print_single_holiday (init_data, detect, "New Year's Day",
                               "", DAY_MIN, MONTH_MIN, year, &hd_elems, fday, count);
         if (init_data)
           init_data = FALSE;
         print_single_holiday (init_data, detect, "Ash Wednesday",
                               DIS_HLS_PREF, easter-46, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "All Fool's Day",
                               DIS_HLS_PREF, DAY_MIN, 4, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Good Friday",
                               "", easter-2, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Easter Sunday",
                               "", easter, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Easter Monday",
                               "", easter+1, 0, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 5, year, DAY_MAX, TRUE);
         if (day_of_year (day, 5, year) != easter+49)
           day += DAY_MAX;
         print_single_holiday (init_data, detect, "Mother's Day",
                               DIS_HLS_PREF, day, 5, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Whitsunday",
                               "", easter+49, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Whit Monday",
                               "", easter+50, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Feast of Corpus Christi",
                               "#", easter+60, 0, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 6, year, DAY_MAX, TRUE);
         print_single_holiday (init_data, detect, "Father's Day",
                               DIS_HLS_PREF, day+(2*DAY_MAX), 6, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Reformation Day",
                               DIS_HLS_PREF, dvec[10-1], 10, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "All Saints' Day",
                               "#", DAY_MIN, 11, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "St. Nicholas' Day",
                               DIS_HLS_PREF, 6, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Christmas Eve",
                               DIS_HLS_PREF, 24, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Christmas Day",
                               "", 25, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Boxing Day",
                               "", 26, MONTH_MAX, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Sylvester",
                               DIS_HLS_PREF, dvec[MONTH_MAX-1], MONTH_MAX, year, &hd_elems, fday, count);
#  if USE_CATH
         print_single_holiday (init_data, detect, "Three King's Day",
                               DIS_HLS_PREF, 6, MONTH_MIN, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Palm Sunday",
                               DIS_HLS_PREF, easter-DAY_MAX, 0, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Green Thursday",
                               DIS_HLS_PREF, easter-3, 0, year, &hd_elems, fday, count);
         day = eval_holiday (24, MONTH_MAX, year, DAY_MAX, FALSE);
         print_single_holiday (init_data, detect, "4'th Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "3'rd Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         print_single_holiday (init_data, detect, "2'nd Advent",
                               DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         day -= DAY_MAX;
         if (day > 0)
           print_single_holiday (init_data, detect, "1'st Advent",
                                 DIS_HLS_PREF, day, MONTH_MAX, year, &hd_elems, fday, count);
         else
           print_single_holiday (init_data, detect, "1'st Advent",
                                 DIS_HLS_PREF, dvec[11-1]+day, 11, year, &hd_elems, fday, count);
#  endif /* USE_CATH */
#  if USE_USHDY
         day = eval_holiday (DAY_MIN, MONTH_MIN, year, DAY_MIN, TRUE);
         print_single_holiday (init_data, detect, "Martin L. King's Day",
                               DIS_HLS_PREF, day+(2*DAY_MAX), MONTH_MIN, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "St. Valentine's Day",
                               DIS_HLS_PREF, 14, 2, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 2, year, DAY_MIN, TRUE);
         print_single_holiday (init_data, detect, "President's Day",
                               "#", day+(2*DAY_MAX), 2, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "St. Patrick's Day",
                               "#", 17, 3, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 5, year, 6, TRUE);
         print_single_holiday (init_data, detect, "Armed Forces' Day",
                               DIS_HLS_PREF, day+(2*DAY_MAX), 5, year, &hd_elems, fday, count);
         day = eval_holiday (dvec[5-1], 5, year, DAY_MIN, FALSE);
         print_single_holiday (init_data, detect, "Memorial Day",
                               "#", day, 5, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Flag Day",
                               DIS_HLS_PREF, 14, 6, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Independence Day",
                               "#", 4, 7, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 9, year, DAY_MIN, TRUE);
         print_single_holiday (init_data, detect, "Labour Day",
                               "#", day, 9, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 9, year, 5, TRUE);
         print_single_holiday (init_data, detect, "American Indian Day",
                               DIS_HLS_PREF, day+(3*DAY_MAX), 9, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 10, year, DAY_MIN, TRUE);
         print_single_holiday (init_data, detect, "Columbus Day",
                               DIS_HLS_PREF, day+DAY_MAX, 10, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Halloween",
                               DIS_HLS_PREF, dvec[10-1], 10, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 11, year, 2, TRUE);
         if (day == DAY_MIN)
           day += DAY_MAX;
         print_single_holiday (init_data, detect, "Election Day",
                               "#", day, 11, year, &hd_elems, fday, count);
         print_single_holiday (init_data, detect, "Veterans' Day",
                               DIS_HLS_PREF, 11, 11, year, &hd_elems, fday, count);
         day = eval_holiday (DAY_MIN, 11, year, 4, TRUE);
         print_single_holiday (init_data, detect, "Thanksgiving",
                               DIS_HLS_PREF, day+(3*DAY_MAX), 11, year, &hd_elems, fday, count);
#  endif /* USE_USHDY */
#endif /* !USE_GER */
         if (fiscal_month > MONTH_MIN)
          {
            year++;
            is_leap_year = (days_of_february (year) == 29);
          }
       } while (count);
      if (fiscal_month > MONTH_MIN)
       {
         is_leap_year = j;
         year = i;
       }
    }
   if (   hd_elems
       && holiday_flag)
    {
      /*
         At last, sort the eternal holiday dates only if they are not presorted.
      */
      if (hd_elems > 1)
       {
         if (!is_presorted (hd_table, hd_elems))
          {
            if (hd_sort_des_flag)
              qsort((VOID_PTR)hd_table, hd_elems, sizeof *hd_table, (Func_cmp)des_sort);
            else
              qsort((VOID_PTR)hd_table, hd_elems, sizeof *hd_table, (Func_cmp)asc_sort);
          }
         else
           if (hd_sort_des_flag)
             /*
                `hd_table[]' is presorted and must be shown in descending sort order,
                  rearrange its internal sort order from ascending to descending sort order.
             */
             resort (hd_table, hd_elems);
       }
    }
   else
     if (!detect)
      {
        register int   m2=fiscal_month+1;
        register int   m3=fiscal_month+2;
        register int   tmp_month=month;
        auto     Bool  hd_title_shown=FALSE;
        auto     Bool  tmp_ify=is_fiscal_year;


        /*
           Display leading title text for a current year.
        */
        if (   hd_title_flag
#if USE_RC
            && !is_1month_mode
            && !is_2month_mode
#endif
            && (   (   !is_3month_mode
                    && !is_3month_mode2
                    && (   (   !is_fiscal_year
                            && !month)
                        || (   is_fiscal_year
                            && (fiscal_month == MONTH_MIN))))
                || is_3month_mode2
                || (   is_3month_mode
                    && (fiscal_month < MONTH_MAX-1))))
         {
           /*
              Now display a leading NEWLINE character before the title text.
           */
           *s1 = '\0';
           print_text (stdout, s1);
           if (hd_title_flag)
            {
              /*
                 Now display the leading title text only if not suppressed
                   by an explicitly given -X command line option.
              */
#if USE_GER
              sprintf(s1, HD_LIST_TITLE":            "
                      "Das Jahr %0*d ist %sEIN Schaltjahr",
                      len_year_max, year, (is_leap_year) ? "" : "K");
#else /* !USE_GER */
              sprintf(s1, HD_LIST_TITLE":            "
                      "The year %0*d is %s leap year",
                      len_year_max, year, (is_leap_year) ? "A" : "NO");
#endif /* !USE_GER */
              print_text (stdout, s1);
              print_text (stdout, s1);
            }
           hd_title_shown = TRUE;
         }
        /*
           Now display the (complete) eternal holiday list.
        */
        if (   is_3month_mode
#if USE_RC
            || is_1month_mode
            || is_2month_mode
#endif
            || is_3month_mode2)
         {
           /*
              Set the range of the 2|3 months', which have to be respected.
           */
           if (fiscal_month >= MONTH_MAX-1)
            {
              m3 = MONTH_MIN;
              if (fiscal_month == MONTH_MAX)
              {
                m2 = MONTH_MIN;
                m3++;
              }
            }
           is_fiscal_year = FALSE;
           month = fiscal_month;
         }
        i=j = 0;
        while (   (hd_table[i] != (char *)NULL)
               && (i < HD_ELEMS_MAX))
         {
           /*
              One month resp., 2|3-month calendar mode:
                Get holiday month of holiday text.
           */
           if (   is_3month_mode
#if USE_RC
               || is_1month_mode
               || is_2month_mode
#endif
               || is_3month_mode2
               || (   month
                   && !is_fiscal_year))
            {
              s1[0] = hd_table[i][len_year_max];
              s1[1] = hd_table[i][len_year_max+1];
              s1[2] = '\0';
              j = my_atoi (s1);
            }
           /*
              Display dates of eternal holiday list:
                Year calendar  --> Display all eternal holidays.
                Month calendar --> Display only those eternal holidays
                                     which refer to current month.
           */
           if (   is_fiscal_year
               || !month
               || month == j
#if USE_RC
               || (   is_1month_mode
                   && (month == j))
               || (   is_2month_mode
                   && (j == m2))
#endif
               || (   (   is_3month_mode
                       || is_3month_mode2)
                   && (   j == m2
                       || j == m3)))
            {
              if (   !hd_legal_days_only
                  || (   hd_legal_days_only
                      && (*(*(hd_table+i)+LEN_HD_NAME+len_year_max+4+2) != *DIS_HLS_PREF)))
               {
                 /*
                    Month calendar mode:
                      Display a leading NEWLINE character only in case
                        an eternal holiday refers to the current month.
                 */
                 if (!hd_title_shown)
                  {
                    /*
                       Now display a leading NEWLINE character before the title text.
                    */
                    *s1 = '\0';
                    print_text (stdout, s1);
                    if (hd_title_flag)
                     {
                       /*
                          Now display the leading title text only if not
                            explicitly suppressed by a given -X command line option.
                       */
                       sprintf(s1, "%s:", HD_LIST_TITLE);
                       print_text (stdout, s1);
                       print_text (stdout, s1);
                     }
                    hd_title_shown = TRUE;
                  }
                 /*
                    Display the eternal holiday text.
                 */
                 sprintf(s1, "%s", *(hd_table+i)+len_year_max+4+1);
                 print_text (stdout, s1);
               }
            }
           free(hd_table[i]);
           hd_table[i++] = (char *)NULL;
         }
        if (   is_3month_mode
#if USE_RC
            || is_1month_mode
            || is_2month_mode
#endif
            || is_3month_mode2)
         {
           month = tmp_month;
           is_fiscal_year = tmp_ify;
         }
      }
}



#ifdef ANSI_PROTO
LOCAL void
print_single_holiday (      Bool  init_data,
                      const Bool  detect,
                      const char *holiday_name,
                      const char *holiday_prefix,
                            int   day,
                            int   month,
                      const int   year,
                            int  *hd_elems,
                      const int   fday,
                      const int   count)
#else /* !ANSI_PROTO */
   LOCAL void
print_single_holiday (init_data, detect, holiday_name, holiday_prefix,
                      day, month, year, hd_elems, fday, count)
         Bool  init_data;
   const Bool  detect;
   const char *holiday_name;
   const char *holiday_prefix;
         int   day;
         int   month;
   const int   year;
         int  *hd_elems;
   const int   fday;
   const int   count;
#endif /* !ANSI_PROTO */
/*
   Generates a single holiday date text in formatted manner and stores the
     result into global `hd_table[]'.
     ONLY if `holiday_prefix' starts with a '*' (ASTERISK == DIS_HLS_PREF)
     character, this holiday is not stored into global `holiday_vector'
     and will not be highlighted in the generated list!
     If `month' is set to zero, it's assumed that the delivered date is a
     date relative to the Easter Sunday's date (`day' contains the according
     day number of year), otherwise the date is an absolute date of the `year'.
*/
{
   register int  i=day;
   register int  hd;
   static   int  hcount[MONTH_MAX];


   if (!month)
     doy2date (i, is_leap_year, &day, &month);
   hd = day_of_year (day, month, year);
   if (   (year >= EASTER_MIN)
       && (year <= EASTER_MAX)
       && (day >= DAY_MIN)
       && (day <= dvec[month-1])
       && (   !fday
           || (   count
               && (hd >= fday))
           || (   !count
               && (hd < fday-(days_of_february (year-1)==29)))))
    {
      if (   holiday_flag
#if USE_RC
          || rc_enable_hda_flag
          || rc_enable_hdl_flag
#endif
         )
       {
         register int  len=(hd_special_flag&&!hd_both_dates_flag) ? 3 : 2;
         register int  d=day;


         i = hd - ((cal_special_flag) ? act_day : day_of_year (act_day, act_month, act_year));
         if (   hd_special_flag
             && !hd_both_dates_flag)
           d = hd;
         strncpy(s2, holiday_name, LEN_HD_NAME);
         s2[LEN_HD_NAME-1] = '\0';
         sprintf(s1, "%0*d%02d%02d %-*s  %1s  ",
                 len_year_max, year, month, day, LEN_HD_NAME-1, s2, holiday_prefix);
         strcpy(s2, s1);
#if USE_GER
         sprintf(s1, "ist %s, der ", short_day_name (weekday_of_date (day, month, year)));
         strcat(s2, s1);
         if (   !i
             && (year == act_year))
          {
            if (hd_both_dates_flag)
              sprintf(s1, "%*s%s%2d(%3d)%s%*s", !emu_hls, "", ehls1s.seq,
                      d, hd, ehls1e.seq, !emu_hls, "");
            else
              sprintf(s1, "%*s%s%*d%s%*s", !emu_hls, "", ehls1s.seq,
                      len, d, ehls1e.seq, !emu_hls, "");
          }
         else
           if (*holiday_prefix != *DIS_HLS_PREF)
            {
              if (hd_both_dates_flag)
                sprintf(s1, "%*s%s%2d(%3d)%s%*s", !emu_hls, "", ehls2s.seq,
                        d, hd, ehls2e.seq, !emu_hls, "");
              else
                sprintf(s1, "%*s%s%*d%s%*s", !emu_hls, "", ehls2s.seq,
                        len, d, ehls2e.seq, !emu_hls, "");
            }
           else
            {
              if (hd_both_dates_flag)
                sprintf(s1, NO_HLS"%2d(%3d)"NO_HLS, d, hd);
              else
                sprintf(s1, NO_HLS"%*d"NO_HLS, len, d);
            }
         strcat(s2, s1);
         sprintf(s1, "%s %0*d",
                 (hd_special_flag&&!hd_both_dates_flag)
                 ? "Tag" : (char *)short_month_name (month),
                 len_year_max, year);
#else /* !USE_GER */
         sprintf(s1, "is %s, %s%s", short3_day_name (weekday_of_date (day, month, year)),
                 (hd_special_flag&&!hd_both_dates_flag)
                 ? "" : (char *)short_month_name (month),
                 (hd_special_flag&&!hd_both_dates_flag) ? "" : " ");
         strcat(s2, s1);
         if (   !i
             && (year == act_year))
          {
            if (hd_both_dates_flag)
              sprintf(s1, "%*s%s%2d%s(%3d)%s%*s", !emu_hls, "", ehls1s.seq,
                      d, day_suffix (d), hd, ehls1e.seq, !emu_hls, "");
            else
              sprintf(s1, "%*s%s%*d%s%s%*s", !emu_hls, "", ehls1s.seq,
                      len, d, day_suffix (d), ehls1e.seq, !emu_hls, "");
          }
         else
           if (*holiday_prefix != *DIS_HLS_PREF)
            {
              if (hd_both_dates_flag)
                sprintf(s1, "%*s%s%2d%s(%3d)%s%*s", !emu_hls, "", ehls2s.seq,
                        d, day_suffix (d), hd, ehls2e.seq, !emu_hls, "");
              else
                sprintf(s1, "%*s%s%*d%s%s%*s", !emu_hls, "", ehls2s.seq,
                        len, d, day_suffix (d), ehls2e.seq, !emu_hls, "");
            }
           else
            {
              if (hd_both_dates_flag)
                sprintf(s1, NO_HLS"%2d%s(%3d)"NO_HLS, d, day_suffix (d), hd);
              else
                sprintf(s1, NO_HLS"%*d%s"NO_HLS, len, d, day_suffix (d));
            }
         strcat(s2, s1);
         sprintf(s1, "%s%0*d",
                 (hd_special_flag&&!hd_both_dates_flag) ? "day " : "",
                 len_year_max, year);
#endif /* !USE_GER */
         strcat(s2, s1);
         /*
            Add the day displacement, which is relative to the "actual date"
              to the eternal holiday text.
         */
         if (   (   fday
                 && (   (   (year == act_year-1)
                         && (month >= fiscal_month))
                     || (   (year == act_year+1)
                         && (month < fiscal_month))))
             || year == act_year)
          {
            if (   fday
                && (year != act_year))
             {
               if (year == act_year-1)
                 /*
                    Eternal holiday to compute is in "actual year - 1".
                 */
                 i = hd - (DAY_LAST + is_leap_year
                     + ((cal_special_flag) ? act_day : day_of_year (act_day, act_month, act_year)));
               else
                 /*
                    Eternal holiday to compute is in "actual year + 1".
                 */
                 i = hd + (DAY_LAST + (days_of_february (act_year) == 29)
                     - ((cal_special_flag) ? act_day : day_of_year (act_day, act_month, act_year)));
             }
            if (i)
             {
#if USE_GER
               sprintf(s1, "  =  %+4d Tag%c", i, (abs(i)!=1) ? 'e' : '\0');
#else /* !USE_GER */
               sprintf(s1, "  =  %+4d day%c", i, (abs(i)!=1) ? 's' : '\0');
#endif /* !USE_GER */
               strcat(s2, s1);
             }
          }
         /*
            Store the constructed "raw" line in `hd_table[]'.
         */
         if (*hd_elems < HD_ELEMS_MAX)
          {
            hd_table[*hd_elems] = (char *)my_malloc (strlen(s2)+1,
                                                     124, __FILE__, ((long)__LINE__)-1,
                                                     "hd_table[hd_elems]", *hd_elems);
            strcpy(hd_table[(*hd_elems)++], s2);
          }
       }
      /*
         This case can occur only, if the code was extended by more
           eternal holidays and the table size for `hd_table[]', namely
           HD_ELEMS_MAX, isn't increased/adjusted correctly so this
           warning/informational is for maintainers only!!
      */
      if (   (warning_level >= 0)
          && (*hd_elems >= HD_ELEMS_MAX))
       {
#  if USE_GER
         sprintf(s1, "%s"HD_LIST_TITLE" hat nun %d Maximaleintr"AE"ge!%s",
                 (ehls1s.len!=1) ? ehls1s.seq : "", HD_ELEMS_MAX,
                 (ehls1s.len!=1) ? ehls1e.seq : "");
#  else /* !USE_GER */
         sprintf(s1, "%s"HD_LIST_TITLE" contains %d maximum entries now!%s",
                 (ehls1s.len!=1) ? ehls1s.seq : "", HD_ELEMS_MAX,
                 (ehls1s.len!=1) ? ehls1e.seq : "");
#  endif /* !USE_GER */
         print_text (stderr, s1);
       }
      if (init_data)
        for (i=0 ; i < MONTH_MAX ; i++)
          hcount[i] = 0;
      if (   detect
          && (*holiday_prefix != *DIS_HLS_PREF))
        holiday_vector[month-1][hcount[month-1]++] = (cal_special_flag) ? hd : day;
    }
   else
     if (init_data)
       for (i=0 ; i < MONTH_MAX ; i++)
         hcount[i] = 0;
}
