/************************************************************************
 * calc.c								*
 *									*
 * A little RPN (Reverse Polish Notation) calculator,                   *
 * rudimentary emulating a HP 28S. 					*
 * 								        *
 * calc is (c) David Frey, 1993, 1994 					*
 *								        *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *									*
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *									*
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *									*
 ************************************************************************/

/* calc.c,v
 * Revision 1.0  1994/02/20  22:32:19  david
 * Initial revision
 *
 * R1.1 1996/02/26 Chris Bagwell (cbagwell@aud.alcatel.com)
 * Updated to compile on a Sun as well as Linux and added support
 * for AND'ing and OR'ing two values together.  Also changed to
 * always output both Hex and Dec values.  Makes it more handy
 * as a quick loading programming calc.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <sys/errno.h>
#include <ctype.h>
#include <signal.h>
#include "calc.h"

char *progname;

double stack[STACKSIZE];
short int tos;					 /* top of stack */

char line[MAXLINELENGTH];			 /* entire line  */

short int quit;					 /* should we quit ? */
short int hexflag;				 /* are we in hex mode ? */
short int i;

char *rest;                                      /* rest of the line */


/* Push an element on the stack */
void push(double elem)
{
  if (tos > 0) stack[--tos] = elem;
  else         fprintf(stderr, "push: sorry, stack full!\n");
}

/* Pop an element from the stack */
double pop(void)
{
  if (tos < STACKSIZE) return stack[tos++];
  else
  {
    fprintf(stderr, "pop : stack empty.\n"); return 0;
  }
}

/* Returns the top element */
double peep(void)
{
  return stack[tos];
}

/* Returns the elm-th element from the stack */
double pick(short int elm)
{
  if ((tos + elm) < STACKSIZE) return stack[tos + elm];
  else
  {
    fprintf(stderr, "pick: element not available.\n"); return 0;
  }
}

/* Drop top element */
void drop(void)
{
  if (tos < STACKSIZE) tos++;
  else                 fprintf(stderr, "drop: stack empty.\n");
}

/* Duplicate top element */
void dup(void)
{
  push(peep());
}

/* Push 2nd-top element on the stack */
void over(void)
{
  push(pick(1));
}

/* How many elements are on the stack ? */
int depth(void)
{
  return (STACKSIZE - tos);
}

/* Swap the 1st and 2nd element on stack */
void swap(void)
{
  double tmp1, tmp2;

  tmp1 = pop(); tmp2 = pop();

  push(tmp1); push(tmp2);
}

/* Roll top three elements: 1 2 3 -> 3 1 2 */
void roll(void)
{
  double tmp1, tmp2, tmp3;

  tmp1 = pop(); tmp2 = pop(); tmp3 = pop();

  push(tmp1); push(tmp3); push(tmp2);
}

/*
 * Show the contents of the stack. Depending on the hexflag the stack
 * contents will be shown in hex notation (integers only) or in the normal
 * way: Show integers as ii and floats as ff.ffff
 *
 * Infinite results and NaN (not a number) will be displayed by using
 * isinf() and isnan().
 */
void showstack(void)
{
    short int i;

    if (tos < STACKSIZE)
    {
	for (i = STACKSIZE - 1; i >= tos; i--)
	{
	    if (hexflag)
	    {
		printf("%2i: 0x%X ( %d ", i - tos + 1, (int) stack[i],
		       (int) stack[i]);
		if ((isascii((long) stack[i])) && (isalnum((char) stack[i])))
		    printf("'%c' ",(char) stack[i]);
		printf(")\n");
	    }
	    else
	    {
		if (isinf(stack[i]))
		    printf("%2i: %cinfinity\n",
			   i - tos + 1, isinf(stack[i]) == 1 ? '+' : '-');
		else
		    if (isnan(stack[i]))
		    {
			printf("%2i: NaN\n", i - tos + 1);
		    }
		    else
			if ((floor(stack[i]) == stack[i]) &&
			    (stack[i] <= LONG_MAX))
			{
			    printf("%2i: %ld ( 0x%lX ", i - tos + 1,
				   (long int)stack[i], (long int)stack[i]);
			    if ((isascii((long)stack[i])) &&
				 (isalnum((char)stack[i])))
				printf("'%c' ",(char) stack[i]);
			    printf(")\n");
			}
			else
			    printf("%2i: %G\n", i - tos + 1, stack[i]);
	    }
	}
    }
}

/* Stein's Greatest Common Divisor (GCD) algorithm */
long int stein(long int n1, long int n2)
{
  long int d;				 /* difference */
  long int c;				 /* c: shift-count (see below) */
  long int t;				 /* temporary */

  c = 0;
  /* both integers are even; shift them until one gets odd. */
  while (((n1 & 1) == 0) && ((n2 & 1) == 0))
  {
    n1 >>= 1; n2 >>= 1;
    c++;
  }

  do
  {
    if ((n2 & 1) == 0)
    {
      t = n1; n1 = n2; n2 = t;
    }
    while ((n1 & 1) == 0)
      n1 >>= 1;

    /*
     * shift n1 until it gets odd.
     */
    d = n1 - n2;
    if (d < 0)
    {
      n2 = n1;
      d = -d;
    }
    n1 = d >> 1;
  }
  while (n1 > 0);

  return (n2 * (1 << c));
}

/* Evaluate the command line. The work is done in this procedure ! */
int eval(char line[])
{
  short int i;
  short int quit, pushtostack;   /* flags */
  char *cmd;			 /* one command line */
  double op1, op2, res;		 /* operands */

  cmd = strtok(line, delimiters);
  res = 0;
  do
  {
    quit = (cmd == NULL) || (strncmp(cmd, "quit", 4) == 0) ||
	   (strncmp(cmd,"q",1) == 0) ||
           (strncmp(cmd, "exit", 4) == 0);

    if (!quit)
    {
      i = 0;
      while ((i <= NROFCOMMANDS) && (strcmp(cmd, Command[i]) != 0)) i++;
      switch (i)
      {
	/********* help & warranty **********************************************/
	case HELP1: case HELP2:
  	  printf("The following operations will be emulated:\n\n");
	  printf("push\tpop\tpick\tswap\tover\troll\n");
	  printf("dup\tdupn\tdrop\tdropn\tdepth\n");
	  printf("chs\t+\t-\t*\t/\t&\n|\t^\tinv\tsqrt\tsqr\n");
	  printf("sin\tcos\ttan\tasin\tacos\tatan\tatan2\n");
	  printf("sinh\tcosh\ttanh\tasinh\tacosh\tatanh\n");
	  printf("ln\tlog\tld\texp\talog\tshl\n");
	  printf("j0\tj1\tjn\ty0\ty1\tyn\n");
	  printf("erf\terfc\tlgamma\n");
	  printf("abs\tceil\tfact\tmod\tgcd\n");
	  printf("hex\tdec\n\n");
	  printf("The following constants will be recognised:\n\n");
	  printf("pi=3.14159265... and e=2.71828182...\n\n");
	  printf("Delimiters are ',', <space>, <tab> and <newline>.\n");
	  printf("Maximum stacksize are %d elements, "\
		 "maximum line length is %d characters.\n",
		 STACKSIZE, MAXLINELENGTH);
	  printf("Case is significant. Use prefix 0x for hexadecimal "\
		 "and 0 for octal constants.\n\n");
	  break;
       case WARRANTY:
	  printf("\n");
	  printf("This program is free software; you can redistribute it and/or modify\n");
	  printf("it under the terms of the GNU General Public License as published by\n");
	  printf("the Free Software Foundation; either version 2 of the License, or\n");
	  printf("(at your option) any later version.\n");
	  printf("\n");
	  printf("This program is distributed in the hope that it will be useful,\n");
	  printf("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
	  printf("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
	  printf("GNU General Public License for more details.\n");
	  printf("\n");
	  printf("You should have received a copy of the GNU General Public License\n");
	  printf("along with this program; if not, write to the Free Software\n");
	  printf("Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n");
	  printf("\n");
	  break;
      /********* control *****************************************************/

	case HEX:   hexflag = TRUE;                           break;
	case DEC:   hexflag = FALSE;                          break;

      /********* stack oriented operations ***********************************/
	case SHOW:  showstack();                              break;
	case POP:   printf("num returned=%G\n", pop());       break;
	case PEEP:  printf("num on stack=%G\n", peep());      break;
	case PICK:  printf("num returned=%G\n", pick(pop()-1)); break;
	case DROP:  drop();                                   break;
	case DUP:   dup();                                    break;
	case OVER:  over();                                   break;
	case SWAP:  swap();                                   break;
	case ROLL:  roll();                                   break;
	case DEPTH: printf("depth of stack: %i\n",depth());   break;
	case DUPN:  op1 = pop(); op2 = pop();
	            for (i = 0; i <= op1; i++) push(op2);
	            break;
	case DROPN: op1 = pop();
	            for (i = 1; i <= op1; i++) drop();
	            break;
	case CLEAR: tos = STACKSIZE; showstack();             break;

      /********* constants       ********************************************/
	case PUSHPI: push(M_PI);                              break;
	case PUSHE:  push(M_E);                               break;

      /********* unary operators ***********************************
       *
       * Now, the mathematical functions follow.
       * NOTE: The Error Handling is done by the functions itself,
       *       if a function fails, the variable errno is set and an
       *       appropriate message will be dumped to stderr.
       *
       */
	case CHS:
	case SQR:   case SQRT:  case INV:
	case SIN:   case COS:   case TAN:
	case ASIN:  case ACOS:  case ATAN:
	case SINH:  case COSH:  case TANH:
	case ASINH: case ACOSH: case ATANH:
	case EXP:   case LN:    case ALOG:   case LOG:    case LD:
	case ERF:   case ERFC:  case LGAMMA:
	case J0:    case J1:    case Y0:     case Y1:
	case SHL:   case ABS:   case CEIL:   case FACT:
	{
	  op1 = pop(); errno = 0; pushtostack = TRUE;
	  switch (i)
	  {
	    case CHS:    res = -op1;                            break;
	    case SQR:    res = op1 * op1;                       break;

	    case SQRT:   res = sqrt(op1); 		        break;
	    case INV:    if (op1 != 0)
	                   res = 1 / op1;
	                 else
	                   { res = HUGE_VAL; errno = ERANGE; }
	                 break;
	    case SIN:    res = sin(op1);  		        break;
	    case COS:    res = cos(op1);  		        break;
	    case TAN:    res = tan(op1);  		        break;

	    case ASIN:   res = asin(op1); 		        break;
	    case ACOS:   res = acos(op1); 		        break;
	    case ATAN:   res = atan(op1); 		        break;

	    case SINH:   res = sinh(op1); 		        break;
	    case COSH:   res = cosh(op1); 		        break;
	    case TANH:   res = tanh(op1); 		        break;

	    case ASINH:  res = asinh(op1);                      break;
	    case ACOSH:  res = acosh(op1);                      break;
	    case ATANH:  res = atanh(op1); 		        break;

	    case EXP:    res = exp(op1); 	                break;
	    case LN:     res = log(op1);                        break;
	    case ALOG:   res = pow(op1,10);                     break;
	    case LOG:    res = log10(op1);                      break;
	    case LD:     res = log(op1) / M_LN2;  	        break;

	    case ERF:    res = erf(op1);                        break;
	    case ERFC:   res = erfc(op1);                       break;
	    case LGAMMA: res = lgamma(op1);                     break;

	    case J0:     res = j0(op1);                         break;
	    case J1:     res = j1(op1);                         break;
	    case Y0:     res = y0(op1);                         break;
	    case Y1:     res = y1(op1);                         break;

	    case SHL:    res = pow(op1,2);  	                break;
	    case ABS:    res = fabs(op1); 	                break;
	    case CEIL:   res = ceil(op1); 	                break;
	    case FACT:
	    if (op1 < 0)
	    {
	      fprintf(stderr, "defined only for arguments >= 0.\n");
	      pushtostack = FALSE;
	    }
	    else
	      if ((op1 == 0) || (op1 == 1))
  	        res = 1;
	      else
	      {
	        res = 1;
	        for (i = op1; i > 1; i--)
	        res *= i;
	      }
	    break;
	  }
	  if (errno != 0) perror(NULL);
	  if (pushtostack) push(res);

	  break;
	}
	/********* binary operators ***********************************/
	case PLUS:  case MINUS: case TIMES: case DIVIDE: case AND:
        case OR:    case POWER:
	case ATAN2:
	case JN:    case YN:
	case MOD:   case GCD:
	{
	  op2 = pop(); op1 = pop(); errno = 0; pushtostack = TRUE;
	  switch (i)
	  {
	    case PLUS:   res = op1 + op2;                    break;
	    case MINUS:  res = op1 - op2;                    break;
	    case TIMES:  res = op1 * op2;                    break;
	    case DIVIDE: if (op2 != 0)
	                  res = op1 / op2;
	                 else
	                  { res = HUGE_VAL; errno = ERANGE; }
	                 break;
	    case AND:    res = (long)op1 & (long)op2;        break;
	    case OR:     res = (long)op1 | (long)op2;	     break;
	    case ATAN2:  res = atan2(op2, op1);              break;
	    case JN:     res = jn(op1,op2);                  break;
	    case YN:     res = yn(op1,op2);                  break;
	    case POWER:  res = pow(op1, op2);                break;
	    case MOD:    if ((op1 <= LONG_MAX) && (op2 <= LONG_MAX))
	                   if (op2 != 0)
	                   {
			     res = fmod(op1,op2);			break;
			   }
    	                   else
                           { res = HUGE_VAL; errno = ERANGE; }
	                 else
	                   {
	                     errno = ERANGE;	pushtostack = FALSE;
			   }
	                 break;
	    case GCD:    if ((op1 <= LONG_MAX) && (op2 <= LONG_MAX))
	                   if (op2 != 0) res = stein(op1,op2);
    	                   else        { res = HUGE_VAL; errno = ERANGE; }
	                 else
	                   {
	                     errno = ERANGE;	pushtostack = FALSE;
			   }
	                 break;
	  }

	  if (errno != 0) perror(NULL);
	  if (pushtostack) push(res);

	  break;
	}

	default:
	case PUSH:
	{
	  /* First assume it's a double */
	  op1 = strtod(cmd, &rest);
	  /* Was it a integer ? */
	  if (floor(op1) == op1) op1 = strtol(cmd, &rest, 0);

	  /*
	   * If yes: converting according to the rules of the C programming
	   * language
	   */
	  if ((strcmp(rest, "") == 0) || (i == PUSH))
  	    push(op1);
	  else
	    if (rest != NULL)
  	      printf("`%s': unknown command.\n", rest);
	  break;
	}
      }
      cmd = strtok(NULL, delimiters);
    }
  }
  while ((!quit) && (cmd != NULL));

  return quit;
}

/* print a short usage statement. Indicate that the argument must be quoted. */
void usage(void)
{
  fprintf(stderr, "usage: %s [\"expression\"]\n\nRun program with no expression for interactive version and more help.\n\n", progname);
}

int main(int argc, char *argv[])
{
  progname = argv[0];
  if ((argc > 1) &&
      ((strcmp(argv[1],"-h") == 0) || (strcmp(argv[1],"-?") ==0)))
  {
      usage();
      return(0);
  }

  signal(SIGFPE,SIG_IGN); /* ignore floating point exceptions. */

  if (argc == 1)
  {
      fprintf(stderr,"%s version 1.1. Copyright (c) 1993, 1994 David Frey.\n",
	      progname);
      fprintf(stderr,"This is free software with ABSOLUTELY NO WARRANTY.\n");
      fprintf(stderr,"For details, type `warranty'.\n");
  }

  tos = STACKSIZE;
  if (argc > 1)
  {
    strcpy(line, argv[1]);
    for (i = 2; i < argc; i++)
    {
      strcat(line, " "); strcat(line, argv[i]);
    }

    eval(line);
    showstack();
  }
  else
  {
    fprintf(stderr,"Type `quit' to quit and `?' to get a summary of the operations available.\n\n");
    do
    {
      showstack();
      if (!gets(line))
	  break;
      quit = eval(line);
    }
    while (!quit);
  }
  return 0;
}						 /* main */
