/*
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "dvb_no_i2c.h"
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

/* this how many stars maximum in display */
#define MAX_SCALE	50

/* the values of front.agc for zero and 10dB signal strenth for my card */ 
#define ZERO_DB		65280
#define TEN_DB		19000


#define BEEP_OFF	0
#define BEEP_ON		1

/* the sound if beeping (command line option -b) */
#define BEEP_MINIMUM_FREQ	200
#define BEEP_MAXIMUM_FREQ	2000


#define MAXDVBAPI 4
#define VIDEODEV "/dev/video"


/* global vars */
int debug_flag;

main(int argc, char **argv)
{
int a, i, f, r;
struct video_capability cap;
char temp[1024];
int indicator_mode;
int have_device_flag;

iopl(3);

have_device_flag = 0;
indicator_mode = BEEP_OFF;
debug_flag = 0;
while(1)
	{
	a = getopt(argc, argv, "bdh");
	if(a == -1) break;
	switch(a)
		{
		case 'b': /* beep */
			indicator_mode |= BEEP_ON;
			break;			
//		case 'c':
//			sscanf(optarg, "%500s", channel_name);
//			break;
        case 'd': /* not to conflict with -d display option for x */
			debug_flag = 1;
			break;
		case 'h':
			fprintf(stdout,\
"Usage:\n\
sdb -b  [-d] [-h]\n\
\t-b  beep.\n\
\t-d  debug mode.\n\
\t-h  help (this help).\n\
");
			exit(0);	
			break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stdout, "Unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stdout, "Unknown option character `\\x%x'.\n",\
				optopt);
				}
			exit(1);
			break;			
		default:
			break;
		}/* end switch a */
	}/* end while getopt() */


fprintf(stdout,\
"\nPanteltje (c) signal_strength meter sdb-0.2 (c) 2000 (-h for help)\n\n");

for(i = 0; i < MAXDVBAPI; i++)
	{
	sprintf(temp, "%s%d", VIDEODEV, i);
	if(access(temp, F_OK | R_OK | W_OK) != 0)
		{
		fprintf(stdout, "main(): could not access device %s\n", temp);
		continue;

//		perror("access");
//		exit(1);
		}
	have_device_flag = 1;

	f = open(temp, O_RDWR);
	if(f < 0) 
		{
		fprintf(stdout, "Could not open %s for r/w\n", temp);
		continue;
		}

	r = ioctl(f, VIDIOCGCAP, &cap);

	if(r == 0 && (cap.type & VID_TYPE_DVB) )
		{
		fprintf(stdout, "Opened %s\n", temp);
		fprintf(stdout, "Exit with ctrl C.\n");
		if(indicator_mode & BEEP_ON)
			{
			fprintf(stdout, "After exit use  backspace to stop beep.\n");
			}

		print_signal_strength(f, indicator_mode);

		/* close video device */
		close(f);
		}/* end if f OK */
	else
		{
		fprintf(stdout, "Could not do ioctl to %s\n", temp);
		continue;
		}
	}/* end for i */

if(! have_device_flag)
	{
	fprintf(stdout, "No available video device found\n");
	fprintf(stdout, "dvb server running?\n");
	}
fprintf(stdout, "No free video device found, close gVideo or vdr?\n");

exit(0);
}/* end function main */


int print_signal_strength(int videoDev, int mode)
{
int a, c, i;
int top;
struct frontend front;
double dstep, deebee;
int beep_range;
int beep_frequency;
double dpart;

/*
the agc value is now in front.agc (16 bits integer),
the stronger the signal, the more the agc (automatic gain control)
drops.

65280 = 0dB no signal	  0 % scale
19000 = 10 dB
	0 = ?? dB			100 % scale
*/

/*
calculate size of one step
if 19000 = 10dB and 65280 = 0dB then 1 dB =
*/
dstep = (ZERO_DB - TEN_DB) / 10.0;

beep_range = BEEP_MAXIMUM_FREQ - BEEP_MINIMUM_FREQ;

if(mode & BEEP_ON) beep_on();

while(1)
	{
	/* get the values for the receiver front end */
	ioctl(videoDev, VIDIOCGFRONTEND, &front);  

	/*
	The higher the signal strenth, the lower the agc voltage is.
	This is a feedback system, roughly the agc will decrease log with
	the signal strength.
	*/
	/* the top of the indicator bar we are going to draw */
	top = 65535 - front.agc;
	
	top = top * (MAX_SCALE / 65536.0);

	// scale is linear due to agc log
	deebee =  (65535 - front.agc) / dstep ;  

	/* create a beep sound, higher frequecy is more signal */
	if(mode & BEEP_ON)
		{
		(double)dpart = (double)top / (double)MAX_SCALE;

		beep_frequency = BEEP_MINIMUM_FREQ + dpart * beep_range;

		if(debug_flag)
			{
			fprintf(stdout, "WAS BEEP FREQ=%d\n", beep_frequency);
			}

		beep_freq(beep_frequency);
		}

	/* display the analog bar indicator */
	for(i = 0; i < MAX_SCALE; i++)
		{
		/* display the characters ins the analog bar */
		if(i < top) fprintf(stdout, "*");

		/* display the decimal value */
		else if(i == top) fprintf(stdout, "%2.1f dB", deebee);

		/* pad the rest opf the line with spaces */
		else fprintf(stdout, " ");
		}

	/* return to beginning of line to avoid scrolling */
	fprintf(stdout, "\r");

	}/* end while display signal strength loop */

return 1;
}/* end function print signal_strength */

