/*GPL*START*
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#ifndef _tobject_h_
#define _tobject_h_

#include "atypes.h"
#include "string.h"
#include <stdio.h>

// TObject is the base class of all T... classes
// (string is not since it is too 'small')

// some declaration macros
#define CLASSNAME(classname) virtual string className() const {return classname;}
#define FULLCLASSNAME(from)  virtual string fullClassName() const {return from::fullClassName() + "::" + className();}
#define INSTANCENAME(instance) {instance.setInstanceName(#instance);}
#define INSTANCENAMEP(instance) {instance->setInstanceName(#instance);}


// all classes derived from TObject should override:
// className() (see the CLASSNAME macro): return the class name
// fullClassName(): return the class name with all base classes
// dumpImp(FILE *f): ascii dump to file f 


// all classes derived from TObject *can* override:
// InstanceName: return the instance name

// root object for memoryfence:
#define ROOT_OBJECT TObject

class TObject {
 public: 
   // public constants
   const char DEFAULT_INSTANCE_NAME[] = "*anonymous*";   
   const char COPY_INSTANCE_NAME[] = "*copy_cons*";   
   const uint MAGIC1 = ('N')|('a'<<8)|('l'<<16)|('a'<<24);
   const uint MAGIC2 = ('R')|('u'<<8)|('l'<<16)|('z'<<24);
   enum STATUS {CONSTRUCTED=0x7c0bad00|'C', DESTRUCTED=0x7debea00|'D', INVALID=0};
   
   // ctor, dtor
   TObject(): magic1(MAGIC1), magic2(MAGIC2), status(CONSTRUCTED), 
     instance_name(DEFAULT_INSTANCE_NAME) {}
   virtual ~TObject() {status = DESTRUCTED;}
   
   // type info
   // virtual string className() const {return "TObject";}
   CLASSNAME("TObject");
   virtual string fullClassName() const {return className();}
   
   // instance info
   virtual string instanceName() const {return instance_name;}
   bool hasInstanceName() const;
   void setInstanceName(const char *instance);
   void setApplicationName(const string& name) const;
   
   // global methods
   string classAndInstanceName() const;
#ifndef DEBUG_PRINT_FORMAT
   void fatalError(const char *message, ...) const __attribute__ ((noreturn));
   void debugMessage(const char *message, ...) const __attribute__ ((__format__ (printf,2,3) ));
   void warning(const char *message, ...) const;
#endif
   void dump(FILE *f = stdout) const;
   STATUS objectStatus() const;

   
   // class specific implementation of dump() 
   virtual void dumpImp(FILE *f) const;

   
 private: 
   // private data
   uint magic1;  
   uint magic2;  
   STATUS status;
   const char *instance_name;   

 public:
   // default copy (by copy)
   TObject(const TObject&):magic1(MAGIC1), magic2(MAGIC2), 
   status(CONSTRUCTED), instance_name(COPY_INSTANCE_NAME) {}   
   const TObject& operator= (const TObject&) {/* no copy */return *this;}
};


// global dump implementation
inline void dump(const TObject& obj) {obj.dump();}
inline void dump(FILE *f, const TObject& obj) {obj.dump(f);}

// global type info
inline string className(const TObject& obj) {return obj.className();}

// global error
void userWarning(const char *message, ...) __attribute__ ((format(printf,1,2)));
void userError(const char *message, ...) __attribute__ ((noreturn,format(printf,1,2)));
void fatalError(const char *message, ...) __attribute__ ((noreturn,format(printf,1,2)));
void prepareForEarlyExit();


#ifdef DEBUG_PRINT_FORMAT
# define FatalError UserError
# define DebugMessage UserWarning
# define Warning UserWarning
#endif



#endif /* _tobject_h_ */

