/*GPL*START*
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#include "tobject.h"
#include "memoryfence.h"
#include <stdarg.h>
#include <stdio.h>

// global application name
static string application_name;

void TObject::setApplicationName(const string& name) const {
   if(className() != "TAppConfig") 
     fatalError("only a 'TAppConfig' object can set the application name!\n");     
   if(application_name) 
     fatalError("can set application name only once!\n");
   application_name = name;
}

void TObject::setInstanceName(const char *instance) {
   if(hasInstanceName()) {
      if(strcmp(instance, instance_name)) {
	 fatalError("Can't change instance name from '%s' to '%s'! Can only be set once (uniqe)!\n",
		    instance_name, instance);
      }
   }
   instance_name = instance;
}

bool TObject::hasInstanceName() const {
   if(instanceName() != DEFAULT_INSTANCE_NAME) return true;
   else return false;
}

TObject::STATUS TObject::objectStatus() const {
   if((magic1==MAGIC1) && (magic2==MAGIC2)) 
     if((status==CONSTRUCTED) || (status==DESTRUCTED)) return status;
   return INVALID;
}

string TObject::classAndInstanceName() const {
   if(hasInstanceName()) 
     return className() + " '" + instanceName() + "'";
   else 
     return "a " + className();
}

#ifndef DEBUG_PRINT_FORMAT
void TObject::debugMessage(const char *message, ...) const {
   va_list ap;
   
   fprintf(stderr, "Debug message from %s: ", *classAndInstanceName());
   va_start(ap, message);
   vfprintf(stderr, message, ap);
   va_end(ap);
}

void TObject::warning(const char *message, ...) const {
   va_list ap;
   
   fprintf(stderr, "Warning from %s: ", *classAndInstanceName());
   va_start(ap, message);
   vfprintf(stderr, message, ap);
   va_end(ap);
}

void TObject::fatalError(const char *message, ...) const {
   va_list ap;
   
   fprintf(stderr, "Fatal error in %s: ", *classAndInstanceName());
   va_start(ap, message);
   vfprintf(stderr, message, ap);
   va_end(ap);
   prepareForEarlyExit();
   exit(1);
}

void userWarning(const char *message, ...) {
   va_list ap;
   
   va_start(ap, message);
   string mes(string("warning: ") + message);
   if(application_name) mes = application_name + " " + mes;
   vfprintf(stderr, *mes, ap);
   va_end(ap);
}

void userError(const char *message, ...) {
   va_list ap;
   
   va_start(ap, message);
   string mes(message);
   if(application_name) mes = "\r" + application_name + ": " + mes;
   else mes = "\r" + mes;
   vfprintf(stderr, *mes, ap);
   va_end(ap);
   prepareForEarlyExit();
   exit(1);
}
#endif

void fatalError(const char *message, ...) {
   va_list ap;
   
   fprintf(stderr, "Fatal error outside tobject: ");
   va_start(ap, message);
   vfprintf(stderr, message, ap);
   va_end(ap);
   prepareForEarlyExit();
   exit(1);
}

void prepareForEarlyExit() {
   memoryfence_disable_check = true; // disable memoryfence, makes no sense now anyway
}

void TObject::dump(FILE *f) const {
   fprintf(f, "Dump of %s: ", *classAndInstanceName());
   dumpImp(f);
}

void TObject::dumpImp(FILE *f) const {
   fprintf(f, "<dump not supported by this class>\n");
}
