/*GPL*START*
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#ifndef _tarray_h_
#define _tarray_h_

#include <string.h>
#include "tobject.h"
#include "tgenericclassdump.h"
#include "tgenericclassname.h"
#include "minmax.h"


/**@name dynamic array with range checking */
/*@{*/
/**
 The TArray template class implements dynamic arrays with range checking.
 Fixed sized and automatic sized (dynamic) arrays are supported.
 No iterator class is needed: Just use an integer, num() and operator[].
 All types are T allowed, but every class T needs: default ctor, operator== and operator=.
 @memo dynamic array with range checking
 */
template<class T>
class TArray: public TObject {
   // friends: comparison operators:
#ifdef EGCS
   friend bool operator == <> (const TArray<T>& a, const TArray<T>& b);
   friend bool operator != <> (const TArray<T>& a, const TArray<T>& b);
#else
   friend bool operator ==    (const TArray<T>& a, const TArray<T>& b);
   friend bool operator !=    (const TArray<T>& a, const TArray<T>& b);
#endif
 public:
   
   /**@name constructor & destructor */
   /*@{*/
   /// construct automatic sized array
   TArray(); 
   /// construct fixed sized array
   TArray(int size); 
   /// copy construction
   TArray(const TArray& a);
   /// single element to array conversion, does not work for int 
   TArray(const T&);
   /// destructor
   ~TArray()  {delete[] array;}   
   /*@}*/
   
   /**@name type info and dump */
   /*@{*/   
   /// return class name
   virtual string className() const;
   /// return class name preceeded by all superclasses
   virtual string fullClassName() const;
   /// dump implementation for tobject
   virtual void dumpImp(FILE *f) const;
   /*@}*/

   /**@name size and emptyness */
   /*@{*/
   /// get current number of elements
   int num() const {return _num;}
   /// return true == array is empty
   bool isEmpty() const {return _num == 0;}
   /// return true == array is not empty
   bool isNotEmpty() const {return _num != 0;}
   /// return true == array is not empty
   operator bool() const {return _num != 0;}
   /*@}*/

   /**@name main interface */
   /*@{*/
   /// clear array (num=0)
   void empty() {_num=0;}
   // implicit c-style array conversion
   // operator const T * () const {return array;}
   /// explicit c-style array conversion, dangerous! use with caution!
   T * data() const {return array;}
   /// safe access operator, automatically increases size if automatic array
   T& operator[](int i) {return array[createIndex(i)];}
   /// const/readonly access (does not change size)
   const T& operator[](int i) const {return array[checkIndex(i)];}
   /// forced read only access (does not change size)
   const T& operator()(int i) const {return array[checkIndex(i)];}
   /// canon assignement (by copy)
   const TArray& operator = (const TArray& a);
   /// concat (append another array to the end)
   const TArray& operator += (const TArray& a);
   /// append an element to the end
   const TArray& operator += (const T& a);
   /*@}*/
   
   /**@name misc/aux interface */
   /*@{*/
   /// return index of element t if found, else -1 (or FatalError, if fatal_if_not_found == true (== MUST_EXIST))
   int find(const T& t, bool fatal_if_not_found = false) const;
   /// constant for 'find'
   const bool MUST_EXIST = true;
   /// resize the array, this will either truncate the array or append default constructed elements
   void resize(int size); 
   /// switch to automatic allocation mode
   void automaticSize() {automatic = true;}
   /// switch to fixed size
   void fixedSize() {automatic = false;}
   /// garbage collection (internally release unused elements)
   void garbageCollection() {resize(_num);}
   /// copy the last element to index i and decrease size by 1 element (this effectively removes the element i from the array, fast)
   void shrinkRemove(int i);
   /// remove element i (copy all following one backward, slow)
   void slowRemove(int i);
   /// return the last element in the array
   T& lastElement();
   /// remove the last element
   void killLastElement();
   /*@}*/
   
 
 private: 
   // private methods
   
   // create index i if it does not exist
   int createIndex(int i);
   // check if i is a valid index for the size
   int checkIndex(int i) const;
   // copy _num elements from shrink to dst
   void arrayCopy(T *dst, T *src, int _num); 
   // resize the private memory area to size
   void resizePrivate(int size);
   
   
   // private data

   // initial size of array
   const int INITIAL_SIZE = 1; // must be >= 1
   // the array itself
   T *array;
   // current number of elements in the array
   int _num;
   // maximal number of elements in the array
   int max;
   // mode of operation
   bool automatic;
};

// inline implementation 

template<class T>
inline TArray<T>::TArray(): array(new T[INITIAL_SIZE]), 
                            _num(0), max(INITIAL_SIZE), automatic(true) {}

template<class T>
inline TArray<T>::TArray(int size): array(0), _num(size), 
max(0), automatic(false) 
{
   if(size < 0) fatalError("Size must be non-negative! (was %d)\n", size);
   max       = size?size:INITIAL_SIZE;
   array     = new T[max];
}

template<class T>
inline int TArray<T>::createIndex(int i) {
   if((i>=0) && (i<_num)) return i;
   
   // index out of bounds
   if(automatic) {
      if(i >= max) resize(tMax(i+1, max+max/3 + 2));
      _num = i+1;
      return i;
   }
   else fatalError("createIndex: index out of range! (should be 0 <= %d < %d)\n", i, _num);
}

template<class T>
inline int TArray<T>::checkIndex(int i) const {
   if((i>=0) && (i<_num)) return i;
   else fatalError("checkIndex: index out of range! (should be 0 <= %d < %d)\n", i, _num);
}

template<class T>
inline TArray<T>::TArray(const T& t): _num(1), max(1), automatic(false) {
   array = new T[max];
   array[0] = t;
}

template<class T>
inline const TArray<T>& TArray<T>::operator += (const T& elem) {
   if(_num==max) resizePrivate(max+max/3 + 2);
   array[_num++] = elem;
   return *this;
}

template<class T>
inline void TArray<T>::resize(int size) { // resize the arrays size, perhaps truncate
   if(size<0) fatalError("Resize: size must be >=0! (was %d)\n", size);
   resizePrivate(size);
   _num = size;
}

template<class T>
inline void TArray<T>::shrinkRemove(int i) {
   array[checkIndex(i)] = array[_num-1];
   _num--;
}




// non inline inplementation

template<class T>
void TArray<T>::slowRemove(int i) {
   checkIndex(i);
   arrayCopy(&array[i], &array[i+1], _num-i-1);
   _num--;
}


template<class T>
TArray<T>::TArray(const TArray<T>& a): TObject(), array(0), _num(a._num), 
max(a._num?a._num:INITIAL_SIZE), automatic(a.automatic) {
   array = new T[max];
   arrayCopy(array, a.array, _num);
}
 
template<class T>
string TArray<T>::className() const {
   T t;
   return "TArray<" + ::className(t) + '>';
}
 
template<class T>
string TArray<T>::fullClassName() const {
   return TObject::fullClassName() + "::" + className();
}

template<class T>
const TArray<T>& TArray<T>::operator = (const TArray<T>& a) {
   if(&a == this) return *this;
   delete[] array;
   _num       = a._num;
   max       = _num?_num:INITIAL_SIZE;
   automatic = a.automatic;
   array     = new T[max];
   arrayCopy(array, a.array, _num);
   return *this;
}

template<class T>
const TArray<T>& TArray<T>::operator += (const TArray<T>& a) {
   int old = _num;   
   resize(_num + a._num);
   arrayCopy(&array[old], a.array, a._num);
   return *this;
}

template<class T>
int TArray<T>::find(const T& t, bool fatal_if_not_found) const {
   for(int i=0; i<_num; i++) if(array[i] == t) return i;

   // not found
   if(fatal_if_not_found) { // fatal error
      ::dump(t); 
      printf("\n");
      fatalError("Find: this element not found!\n");
   }
   else return -1; // return: not found
}

template<class T>
void TArray<T>::dumpImp(FILE *f) const {
   fprintf(f, "%s size %d (max=%d)\n", automatic?"automatic":"fixed", _num, max);
   for(int i=0; i<_num; i++) {
      fprintf(f, "[%d]={", i);
      ::dump(f, array[i]);
      fprintf(f, "} ");	 
   }
   fprintf(f, "\n");	 
}

template<class T>
void TArray<T>::resizePrivate(int size) {
   if(size < _num) _num = size; // truncate
   max = size?size:INITIAL_SIZE;
   T* newarray = new T[max];
   arrayCopy(newarray, array, _num);
   delete[] array;
   array = newarray;
}

template<class T>
void TArray<T>::arrayCopy(T *dst, T *src, int n) {
   for(int i=0; i<n; i++) dst[i] = src[i];
}

template<class T>
T& TArray<T>::lastElement() {
   if(_num) return array[_num-1];
   else fatalError("lastElement: array is empty!\n");
}

template<class T>
void TArray<T>::killLastElement() {
   if(_num) _num--;
   else fatalError("killLastElement: array is empty!\n");
}



/**@name friends and nonmemebrs */
/*@{*/
/// compare (per element)
template<class T>
bool operator == (const TArray<T>& a, const TArray<T>& b) {
   if(a._num != b._num) return false;
   for(int i=0; i<a._num; i++) if(a[i] != b[i]) return false;
   return true;
}
/// compare (per element)
template<class T>
inline bool operator != (const TArray<T>& a, const TArray<T>& b) {
   return !(a==b);
}
/// concat two arrays
template<class T>
inline TArray<T> operator + (const TArray<T>& a, const TArray<T>& b) {
   TArray<T> r(a);
   r += b;
   return r;
}
/*@}*/
/*@}*/

#endif

