/*GPL*START*
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#include "tappconfig.h"
#include "str2value.h"
#include "file_tools.h"
#include <ctype.h>
#include <limits.h>
#include <values.h>
#include <stdlib.h>
#include <stdarg.h>

// config:

// global email of author:
const char * TAPPCONFIG_EMAIL = "overmann@iname.com";
// gpl message
const char * TAPPCONFIG_GPL = "this program is distributed under the terms of the GNU General Public License";



// 1997
// 01:45 Jun 11 stop on '--' feature added (929 lines)
// 02:00 Jul  3 string once/append feature added (968 lines)
// 12:00 Aug  7 application set/get support started (1103 lines)
// 13:00 Aug  7 application set/get support finished (untested) (1164 lines)
// 13:00 Aug  7 autosave support started
// 21:00 Aug  7 autosave canceled: save support instead
// 23:30 Aug  7 save support finished (1327 lines)
// ???          get default/ranges
// 1998
// 17:00 Jan 10 improved rc file parsing (string scan tolls)
// 18:47 Aug 04 fixed %s bug (printf->puts) 
// 22:48 Aug 05 added %e for authors email



/// global TAppConfig pointer to the only instance TAppConfig
TAppConfig *tApp = 0;


static const char *self_conflist[] = {
   // standard options
   "type=switch, name=help, char=h, onlycl, help='print this help message, then exit'",
   "type=switch, name=version, char=V, onlycl, help='print version, then exit'",
   
   // configuration options
   "type=switch, name=verbose-config, onlycl, configopt, help='print all options, values and where they were specified, then exit'",
   "type=string, name=rc-file, onlycl, configopt, param=FILE, help='use FILE as rcfile'",
   "type=string, name=create-rc-file, onlycl, configopt, param=FILE, help='create a FILE with default values for all options'",
   
   // pseudovariables only visible to the application, not to the user
   "type=string, name=application-version, onlyapp",
   "type=string, name=application-name, onlyapp",
   "EOL"
};



// TAppConfigItem implementation

TAppConfigItem::TAppConfigItem():
must_have(false),
should_have(false),
only_cl(false),
configopt(false),
only_app(false),
save(false),
type(NONE),
set_in(NEVER),
string_mode(OVERRIDE),
string_sep(""),
double_value(0), double_upper(0), double_lower(0), double_default(0),
int_value(0), int_upper(0), int_lower(0), int_default(0),
bool_value(false), bool_default(false),
printed(false),
name(),
context(),
help(),
headline(),
char_name(),
par(),
alias(),
type_str(),
upper(), lower(), def(),
string_value(), string_default()
{}


TAppConfigItem::TAppConfigItem(const char *str, const char *line_context, 
			       bool privat):
must_have(false),
should_have(false),
only_cl(false),
configopt(false),
only_app(false),
save(false),
type(NONE),
set_in(NEVER),
string_mode(OVERRIDE),
string_sep(""),
double_value(0), double_upper(0), double_lower(0), double_default(0),
int_value(0), int_upper(0), int_lower(0), int_default(0),
bool_value(false), bool_default(false),
printed(false),
name(),
context(line_context),
help(),
headline(),
char_name(),
par(),
alias(),
type_str(),
upper(), lower(), def(),
string_value(), string_default()
{
   TArray<string> comp = split(str, ",", true);
   for(int i=0; i<comp.num(); i++) {
      setComp(split(comp[i], "=", true, true), privat);
   }
   validate(context);
}


inline static bool isalphaorul(char c) {
   if(isalpha(c) || c=='_') return true;
   else return false;
}


static string Range2Str(int lower, int upper) {
   if((lower!=INT_MIN)||(upper!=INT_MAX)) {
      string lstr;
      string ustr;
      if(lower!=INT_MIN) lstr = lower;
      if(upper!=INT_MAX) ustr = upper;
      return "[" + lstr + ".." + ustr + "]";
   } else {
      return "";
   }
}

static string Range2Str(double lower, double upper) {
   if((lower!=-MAXDOUBLE) || (upper!=MAXDOUBLE)) {
      string lstr;
      string ustr;
      if(lower!=-MAXDOUBLE) lstr = lower;
      if(upper!=MAXDOUBLE) ustr = upper;
      return "[" + lstr + ".." + ustr + "]";
   } else {
      return "";
   }
}

void TAppConfigItem::validate(const char *line_context) {
   // name
   if(name.len()<2) 
     fatalError("%s: name too short! (was %d, must have a least two chars)\n", line_context, name.len());
   if(!isalphaorul(name[0])) 
     fatalError("%s: name should begin with alpha char or '_'! (was %c, must have a least two chars)\n", line_context, name[0]);
   if(char_name.len() > 1) 
     fatalError("%s: only one character may be specified as char name!\n", line_context);

   // help, alias and flags
   if((help=="")&&(!only_app)) 
     fatalError("%s: you must give a help for each option that is not onlyapp!\n", line_context);
   for(int i=0; i < alias.num(); i++) 
     if(alias[i].len()<1)
       fatalError("%s: alias must not be empty!\n", line_context);
   if(only_app&&only_cl) 
     fatalError("%s: only one out of [onlyapp,onlycl] may be specified!\n", line_context);
   if(should_have&&must_have) 
     fatalError("%s: only one out of [shouldhave,musthave] may be specified!\n", line_context);
   if(def&&must_have&&only_cl)
     fatalError("%s: default value for reqired command line option makes no sense!\n", line_context);

   // type
   if(type_str=="bool") type=BOOL;
   else if(type_str=="int") type=INT;
   else if(type_str=="switch") type=SWITCH;
   else if(type_str=="double") type=DOUBLE;
   else if(type_str=="string") type=STRING;
   else fatalError("%s: illegal/unknown type '%s'!\n", line_context, *type_str);

   // string mode
   if((string_mode!=OVERRIDE) && (type!=STRING))
     fatalError("%s: string-mode-... makes only sense with strings!\n", line_context);
   if((string_sep) && (type!=STRING))
     fatalError("%s: string-append-separator makes only sense with strings!\n", line_context);
   if((string_sep) && (string_mode!=APPEND))
     fatalError("%s: string-append-separator makes only sense with string-mode-append!\n", line_context);
        
   // each type
   bool err(false);
   switch(type) {
    case SWITCH:
      if(must_have||should_have) fatalError("%s: switch can't be reqired/recommended!\n", line_context);
      if(def) fatalError("%s: switch can't have a defaut (is always false)!\n", line_context);
      if(upper) fatalError("%s: switch can't have an upper limit!\n", line_context);
      if(lower) fatalError("%s: switch can't have a lower limit!\n", line_context);
      bool_value = false;
      break;
    case BOOL:
      if(def) {
	 bool_default = str2getBool(def, err);
	 if(err) fatalError("%s: illegal/unknown bool value for default '%s'! (can be [true|yes|on|t|1|false|no|off|f|0])\n", line_context, *def);
      }
      else bool_default = false;
      bool_value = bool_default;
      if(upper) fatalError("%s: bool can't have an upper limit!\n", line_context);
      if(lower) fatalError("%s: bool can't have a lower limit!\n", line_context);
      break;      
    case INT:
      if(def) {
	 int_default = str2getInt(def, err);
	 if(err) fatalError("%s: illegal chars for default value in '%s'!\n", line_context, *def);
      }
      else int_default = 0;
      int_value = int_default;
      if(upper) {
	 int_upper = str2getInt(upper, err);
	 if(err) fatalError("%s: illegal chars for upper bound in '%s'!\n", line_context, *upper);
      }
      else int_upper = INT_MAX;
      if(lower) {
	 int_lower = str2getInt(lower, err);
	 if(err) fatalError("%s: illegal chars for lower bound in '%s'!\n", line_context, *lower);
      }
      else int_lower = INT_MIN;
      if(outOfRange(int_default, int_lower, int_upper)) 
	fatalError("%s: default value out of range! (%d not in %s)!\n", line_context, int_default, *Range2Str(int_lower, int_upper));
      break;
    case DOUBLE:
      if(def) {
	 double_default = str2Double(def, err);
	 if(err) fatalError("%s: illegal chars for default value in '%s'!\n", line_context, *def);
      }
      else double_default = 0.0;
      double_value = double_default;
      if(upper) {
	 double_upper = str2Double(upper, err);
	 if(err) fatalError("%s: illegal chars for upper bound in '%s'!\n", line_context, *upper);
      }
      else double_upper = MAXDOUBLE;
      if(lower) {
	 double_lower = str2Double(lower, err);
	 if(err) fatalError("%s: illegal chars for lower bound in '%s'!\n", line_context, *lower);
      }
      else double_lower = -MAXDOUBLE;
      if(outOfRange(double_default, double_lower, double_upper)) 
	fatalError("%s: default value out of range! (%g not in %s)!\n", line_context, double_default, *Range2Str(double_lower, double_upper));
      break;
     case STRING: 
      // all strings are valid: most generic type!
      string_default = def;
      string_value = string_default;
      break;
    case NONE:
      fatalError("internal error! (invalid type id)\n");
   }
   set_in = DEFAULT;
}

void TAppConfigItem::setComp(const TArray<string>& a, bool privat) {
   if((a.num()==1) && (a[0].consistsOfSpace())) return;
   string comp = a[0];
   string param;
   if(a.num()>1) {
      param=a[1];
      param.unquote();
   }
   if(a.num()>2) 
     fatalError("%s: syntax error near component '%s'! (too many '=')\n", 
		*context, *join(a,'='));
   if(comp=="name") { name = param;
   } else if(comp=="help") { help = param;
   } else if(comp=="headline") { headline = param;
   } else if(comp=="param") { par = param;
   } else if(comp=="char") { char_name = param; 
   } else if(comp=="alias") { alias = split(param, "; ,", true, true);
   } else if(comp=="type") { type_str = param;
   } else if(comp=="default") { def = param;
   } else if(comp=="upper") { upper = param; 
   } else if(comp=="lower") { lower = param;
   } else if(comp=="musthave") { must_have = true;
   } else if(comp=="shouldhave") { should_have = true;
   } else if(comp=="string-mode-append") { string_mode = APPEND;
   } else if(comp=="string-mode-once") { string_mode = ONCE;
   } else if(comp=="string-append-separator") { string_sep = param;
   } else if(privat && comp=="configopt") { configopt = true;
   } else if(comp=="save") { save = true;
   } else if(comp=="onlycl") { only_cl = true;
   } else if(privat && comp=="onlyapp") { only_app = true;
   } else fatalError("%s: unknown component '%s'!\n", *context, *comp);
}

string TAppConfigItem::getWasSetStr(TACO_SET_IN setin, const string& env, const string& rcfile) const {
   switch(setin) {
    case DEFAULT:
      return "by default";
      break;
    case COMMAND_LINE:
      return "on command line";
      break;
    case ENVIRONMENT:
      return "in environment variable '" + env + "'";
      break;
    case RC_FILE:
      return "in rc file '" + rcfile + "'";
      break;
    default:
      fatalError("illegal value %d in setin! (0==NEVER)\n", setin);
   }
}

string TAppConfigItem::getWasSetStr(const string& env, const string& rcfile) const {
   return getWasSetStr(set_in, env, rcfile);
}

void TAppConfigItem::printValue(const string& env, const string& rcfile) const {
   switch(type) {
    case SWITCH:
      if(bool_value) 
	printf("switch %s was set %s\n", *name, *getWasSetStr(env, rcfile));
      else
	printf("switch %s was not set\n", *name);	
      break;
      
    case BOOL:
      printf("bool   %s was set to %s %s\n", *name, bool_value?"true":"false", *getWasSetStr(env, rcfile));
      break;
      
    case INT:
      printf("int    %s was set to %d %s\n", *name, int_value, *getWasSetStr(env, rcfile));
      break;
      
    case DOUBLE:
      printf("double %s was set to %g %s\n", *name, double_value, *getWasSetStr(env, rcfile));
      break;
      
    case STRING:
      {
	 string s(string_value);
	 s.expandUnprintable();
	 printf("string %s was set to \"%s\" %s\n", *name, *s, *getWasSetStr(env, rcfile));
      }
      break;
      
    default:
      fatalError("printValue(): Illegal value %d in type! (0==NONE)\n", type);
   }
}

string TAppConfigItem::getParamStr() const {
   if(par) return par;
   switch(type) {
    case BOOL:
      return "<bool>";
    case INT:
      return "<int>";
    case DOUBLE:
      return "<double>";
    case STRING:
      return "<string>";
    case SWITCH:
      return "";
    default:
      fatalError("getParamStr(): Illegal value %d in type! (0==NONE)\n", type);
   }
}


string TAppConfigItem::getTypeStr() const {
   switch(type) {
    case BOOL:
      return "bool";
    case INT:
      return "int";
    case DOUBLE:
      return "double";
    case STRING:
      return "string";
    case SWITCH:
      return "switch";
    default:
      fatalError("getTypeStr(): Illegal value %d in type! (0==NONE)\n", type);
   }
}

int TAppConfigItem::getOptLen() const {
   int l = getParamStr().len();
   if(l) l++;
   return name.len() + l;
}

string TAppConfigItem::getFlagsStr(const string& optprefix, 
				   bool globalonlycl) const 
{
   string h;   
   string range;
   switch(type) {
    case DOUBLE:
      range = Range2Str(double_lower, double_upper);
      break;
    case INT:
      range = Range2Str(int_lower, int_upper);
      break;
    default:
      break;
   }
   string defval;
   string string_mod;
   switch(type) {
    case DOUBLE:
      if(double_default!=0.0) defval = double_default;
      break;
    case INT:
      if(int_default!=0) defval = int_default;
      break;
    case STRING:
      {
	 string s(string_default);
	 s.expandUnprintable();
	 if(s) defval = "\"" + s + "\"";
      }
      switch(string_mode) {
       case APPEND: string_mod = "multiple allowed"; break;
       case ONCE:   string_mod = "only once"; break;
       default: break;
      }
      break;
    case BOOL:
      if(def) defval = def; // take string to allow: yes true on 1 ...
      break;
    default:
      break;
   }				    
   if((only_cl && (!globalonlycl)) || must_have || should_have || 
      (alias.num()>0) || range || defval || string_mod) {
      h += " (";
      if(only_cl && (!globalonlycl)) h += "only command line";
      if(must_have) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "required";
      }	
      if(should_have) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "recommended";
      }	
      if(alias.num()==1) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "alias=" + optprefix + alias[0];
      }
      if(alias.num()>1) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "aliases={" + optprefix + join(alias, ", " + optprefix) + "}";
      }
      if(range) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "range=" + range;
      }
      if(defval) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += "default=" + defval;
      }
      if(string_mod) {
	 if(h.lastChar()!='(') h+= ", ";
	 h += string_mod;
      }
      h += ")";
   }
   return h;
}
   
void TAppConfigItem::printItemToFile(FILE *f) const {
   if(headline) {    // print headline
      fprintf(f, "\n# %.76s\n# %.76s\n\n", *headline, *string('=', headline.len()));
   }
   string h(help + "\n");
   if(type == SWITCH) {
      h += "uncomment next line to activate switch";
   } else {
      h += "parameter is of type " + getTypeStr();
   }  
   h += " " + getFlagsStr("", false);
   while(h) {
      fprintf(f, "# %s\n", *h.getFitWords(80 - 2 - 1));
   }
   string defval;
   switch(type) {
    case DOUBLE:
      defval = double_default;
      break;
    case INT:
      defval = int_default;
      break;
    case STRING:
      {
	 string s(string_default);
	 s.expandUnprintable();
	 defval = "\"" + s + "\"";
      }
      break;
    case BOOL:
      if(def) defval = def; // take string to allow yes true on 1 ...
      else defval = bool_default?"true":"false";
      break;
    default:
      break;
   }
   if(type == SWITCH) {
      fprintf(f, "#%s\n", *name);
   } else {
      fprintf(f, "%s = %s\n", *name, *defval);
   }     
   fprintf(f, "\n");
}


void TAppConfigItem::printCurItemToFile(FILE *f, bool simple) const {
   if(!simple) {
      if(headline) {    // print headline
	 fprintf(f, "\n# %.76s\n# %.76s\n\n", *headline, *string('=', headline.len()));
      }
      string h(help + "\n");
      if(type==SWITCH) h += "parameter is a switch";
      else {
	 h += "parameter is of type " + getTypeStr();
	 h += " " + getFlagsStr("", false);
      }
      while(h) {
	 fprintf(f, "# %s\n", *h.getFitWords(80 - 2 - 1));
      }
   }
   string val;
   switch(type) {
    case DOUBLE:
      val = double_value;
      break;
    case INT:
      val = int_value;
      break;
    case STRING:
      {
	 string s(string_value);
	 s.expandUnprintable();
	 val = "\"" + s + "\"";
      }
      break;
    case BOOL:
      val = bool_value?"true":"false";
      break;
    default:
      break;
   }
   if(type == SWITCH) {
      if(bool_value)
	fprintf(f, "%s\n", *name);
      else
	fprintf(f, "#%s\n", *name);
   } else {
      fprintf(f, "%s = %s\n", *name, *val);
   }     
   if(!simple) fprintf(f, "\n");
}


void TAppConfigItem::printHelp(int max_optlen, bool globalonlycl) const {
   int hcol = max_optlen + 5 + 2;
   char buf[256];
   if(headline) {
      printf("\n%s\n", *headline);
   }
   sprintf(buf, "%c%c --%s%s", char_name?'-':' ', char_name?char_name[0]:' ', 
	   *name, type==SWITCH?"":*("=" + getParamStr()));
   string h(help);
   h += " " + getFlagsStr("--", globalonlycl);
   printf("%*s%s\n", -hcol, buf, *h.getFitWords(80 - hcol - 1));
   while(h) {
      printf("%*s%s\n", -hcol, "", *h.getFitWords(80 - hcol - 1));
   } 
}


void TAppConfigItem::setValue(const string& parameter, TACO_SET_IN setin, 
			      bool verbose, const string& env, 
			      const string& rcfile, const string& line_context) {
   if(only_app)       
     userError("internal parameter name '%s' is private to the application may not be set by the user!\n", *name);
   if(only_cl && (!setin==COMMAND_LINE))       
     userError("parameter name '%s' is only available on the command line!\n", *name);
   
   // prepare option string
   string option;
   if(setin==COMMAND_LINE) {
      option = "option -" + char_name + ", --" + name;
   } else {
      option = "option '" + name + "'";
   }
  
   if(set_in!=DEFAULT) {
      if(type!=STRING) {	    
	 if(verbose) 
	   printf("warning: %s (set %s) is overridden %s\n", 
		  *option, *getWasSetStr(setin, env, rcfile),
		  *getWasSetStr(env, rcfile));
	 return;
      }
   }
  
   bool err=false;
   switch(type) {
    case SWITCH:
      if(parameter) 
	userError("%s: %s is a switch and takes no parameter!\n", *line_context, *option);
      bool_value = true;
      break;
      
    case BOOL:
      bool_value = str2getBool(parameter, err);
      if(err) userError("%s: illegal/unknown bool value '%s' for %s!\n(can be [true|yes|on|t|1|false|no|off|f|0])\n", *line_context, *parameter, *option);
      break;

    case INT:
      int_value = str2getInt(parameter, err);
      if(err) userError("%s: illegal int value format for %s in '%s'!\n", *line_context, *option, *parameter);
      if(outOfRange(int_value, int_lower, int_upper))
	userError("%s: value out of range for %s! (%d not in %s)!\n", *line_context, *option, int_value, *Range2Str(int_lower, int_upper));
      break;
      
    case DOUBLE:
      double_value = str2Double(parameter, err);
      if(err) userError("%s: illegal double value format for %s in '%s'!\n", *line_context, *option, *parameter);
      if(outOfRange(double_value, double_lower, double_upper))
	userError("%s: value out of range for %s! (%g not in %s)!\n", *line_context, *option, double_value, *Range2Str(double_lower, double_upper));
      break;
      
    case STRING:
      switch(string_mode) {
       case ONCE:
	 if(set_in != DEFAULT) 
	   userError("%s: %s may only be set once! (was already set to '%s' and was tried to set to '%s')\n", *line_context, *option, *string_value, *parameter);
	 string_value = parameter;
	 break;
       case APPEND:
	 if((set_in != DEFAULT)||(string_value))
	   string_value += string_sep;
	 string_value += parameter;
	 break;
       case OVERRIDE:
	 string_value = parameter; // overwrite old value, the default
	 break;
      }
      break;
      
    default:
      fatalError("setValue(): Illegal value %d in type! (0==NONE)\n", type);
   }
   set_in = setin;
}


void TAppConfigItem::setValueFromApp(bool parameter) {
   if(only_app)       
     fatalError("internal parameter name '%s' is private to tapplication may not be set by the user!\n", *name);
   if(only_cl)       
     fatalError("parameter name '%s' is only available on the command line!\n", *name);
   set_in = APPLICATION;
   if((type==SWITCH)||(type==BOOL)) {
      bool_value=parameter;
   } else {
      fatalError("setValueFromApp(bool/switch): type mismatch: type was %s\n", *getTypeStr());
   }
}


// returns true if value is valid, else false
bool TAppConfigItem::setValueFromApp(int i) {
   if(only_app)       
     fatalError("internal parameter name '%s' is private to tapplication may not be set by the user!\n", *name);
   if(only_cl)       
     fatalError("parameter name '%s' is only available on the command line!\n", *name);
   set_in = APPLICATION;
   if(type==INT) {
      if(i > int_upper) {
	 int_value = int_upper;
	 return false;
      }
      if(i < int_lower) {
	 int_value = int_lower;
	 return false;
      }
      int_value = i;
      return true;
   } else {
      fatalError("setValueFromApp(int): type mismatch: type was %s\n", *getTypeStr());
   }
}


// returns true if value is valid, else false
bool TAppConfigItem::setValueFromApp(double d) {
   if(only_app)       
     fatalError("internal parameter name '%s' is private to tapplication may not be set by the user!\n", *name);
   if(only_cl)       
     fatalError("parameter name '%s' is only available on the command line!\n", *name);
   set_in = APPLICATION;
   if(type==DOUBLE) {
      if(d > double_upper) {
	 double_value = double_upper;
	 return false;
      }
      if(d < double_lower) {
	 double_value = double_lower;
	 return false;
      }
      double_value = d;
      return true;
   } else {
      fatalError("setValueFromApp(double): type mismatch: type was %s\n", *getTypeStr());
   }
}


void TAppConfigItem::setValueFromApp(const string& str) {
   if(only_app)       
     fatalError("internal parameter name '%s' is private to tapplication may not be set by the user!\n", *name);
   if(only_cl)       
     fatalError("parameter name '%s' is only available on the command line!\n", *name);
   set_in = APPLICATION;
   if(type==STRING) {
      string_value = str;
   } else {
      fatalError("setValueFromApp(string): type mismatch: type was %s\n", *getTypeStr());
   }
}


// returns true if value is valid, else false
// sets value from string according to any type (switch == bool here)
bool TAppConfigItem::setValueFromAppFromStr(const string& parameter) {
   if(only_app)       
     fatalError("internal parameter name '%s' is private to tapplication may not be set by the user!\n", *name);
   if(only_cl)       
     fatalError("parameter name '%s' is only available on the command line!\n", *name);
   set_in = APPLICATION;
   bool err=false;
   switch(type) {
    case SWITCH:
    case BOOL:
      bool_value = str2getBool(parameter, err);
      if(err) fatalError("SetValueFromAppFromStr: illegal/unknown bool value '%s' for %s!\n(can be [true|yes|on|t|1|false|no|off|f|0])\n", *parameter, *name);
      return true;

    case INT:
      int_value = str2getInt(parameter, err);
      if(err) fatalError("SetValueFromAppFromStr: illegal int value format for %s in '%s'!\n", *name, *parameter);
      return setValueFromApp(int_value);
      
    case DOUBLE:
      double_value = str2Double(parameter, err);
      if(err) fatalError("SetValueFromAppFromStr: illegal double value format for %s in '%s'!\n", *name, *parameter);
      return setValueFromApp(double_value);
      
    case STRING:
      string_value = parameter;
      return true;
      
    default:
      fatalError("SetValueFromStrFromApp(): Illegal value %d in type! (0==NONE)\n", type);
   }
}



// TAppConfig implementation

TAppConfig::TAppConfig(const char *conflist[], const char *listname,
		       int argc, char *argv[],
		       const char *envstrname,
		       const char *rcname,
		       const string& version): 
_params(),
name(),
opt(),
alias(),
envname(),
rc_name(),
rc_str(rcname?rcname:""),
verbose_conf(false),
onlycl(false),
stopatdd(false),
usage(),
trailer(),
commonhead()
{
   if(tApp)
     fatalError("only exactly one instance of TAppConfig is allowd!\n");
   
   int i;
   
   string appname(argv[0]);
   appname.extractFilename();   
   setApplicationName(appname);
   for(i=0; i<256; i++) char2index[i] = -1;
   stopatdd = onlycl = verbose_conf = false;
   addConfigItems(conflist, listname, false);
   addConfigItems(self_conflist, "self_conflist", true);
   opt.fixedSize();
   doCommandLine(argc, argv, version);
   if(getBool("help")) printHelp();
   if(getBool("version")) printf("%s version %s\n", *getString("application-name"), 
			      *getString("application-version"));
   string creatercfile;
   if(!onlycl) {
      creatercfile = getString("create-rc-file");
      verbose_conf = getBool("verbose-config");
      if(creatercfile) createRCFile(creatercfile, rcname);
      doEnvironVar(envstrname);
      doRCFile(rcname, getString("rc-file"));
      if(verbose_conf) printValues();   
   }
   if(verbose_conf || getBool("help") || getBool("version") || creatercfile) {
      prepareForEarlyExit();
      exit(0);
   }
   for(i=0; i<opt.num(); i++) {
      if(opt[i].must_have && opt[i].set_in==DEFAULT) 
	userError("error: option '%s' must be specified somewhere!\n", 
		  *opt[i].name);
      if(opt[i].should_have && opt[i].set_in==DEFAULT) 
	printf("warning: option '%s' should be specified somewhere\n", 
	       *opt[i].name);
   }
   tApp = this;
}


TAppConfig::~TAppConfig() {
   tApp = 0; // avoid dangling pointer to global application configuration;
}


// save some items
// return rcname in rc_name_out
bool TAppConfig::save(string& rc_name_out) {
   rc_name_out = "$(HOME)/." + rc_str;
   char *home = getenv("HOME");
   if(home==0) {
      fprintf(stderr, "can't save config: no HOME variable defined!\n");
      return false;
   }
   string realname(string(home) + "/." + rc_str);
   rc_name_out = realname;
   string tmpname(realname + ".tmp");
   FILE *tmpf = fopen(tmpname, "w");
   if(tmpf==0) {
      fprintf(stderr, "can't save config: can't open '%s' for writing!\n", *tmpname);
      return false;
   }

   // init
   int i, j;
   for(i=0; i<opt.num(); i++) opt[i].printed = false;
   int pri=0;
   
   // update old config
   FILE *oldf = fopen(realname, "r");
   if(oldf) {
      if(verbose_conf) 
	printf("updating items in '%s'\n", *realname);
      char buf[1024];
      while(fgets(buf, sizeof(buf), oldf)) {
	 string line(buf);
	 if(strchr(";\\!", buf[0]) || line.consistsOfSpace()) { // copy comment
	    fputs(buf, tmpf);
	 } else if(buf[0]=='#') { // check for commented out switches
	    string sw(buf+1);
	    sw.cropSpace();
	    if(alias.contains(sw)||name.contains(sw)) {
	       line = sw;
	       goto upd;
	    } else {
	       fputs(buf, tmpf);
	    }
	 } else { // change parameter
	    j = line.firstOccurence('=');
	    if(j < 0) {
	       line.cropSpace();
	       for(j=0; j<line.len() && (!isspace(line[j])); j++);      
	    }
	    line.truncate(j);
	    upd:
	    line.cropSpace();
	    if(alias.contains(line)) line = opt[alias[line]].name;
	    if(name.contains(line)) { // valid name found
	       TAppConfigItem& a(opt[name(line)]);
	       if((!a.only_app)&&(!a.only_cl)) {
		  if(a.save) {
		     a.printCurItemToFile(tmpf, true);
		     a.printed = true;
		  } else {
		     fputs(buf, tmpf);
		  }
	       }
	    } else {
	       fprintf(stderr, "discarding invalid line from file '%s' :%s", *realname, buf);
	    }
	 }	 
	 pri++;
      }
      fclose(oldf);
   }
   
   // write any remaining items?
   int rem=0;
   for(i=0; i < opt.num(); i++) { 
      const TAppConfigItem& a(opt(i));
      if(a.only_app) continue;
      if(a.only_cl) continue;
      if(a.printed) continue;
      if(!a.save) continue;
      rem++;
   }
   if(rem) {
      if(verbose_conf) 
	printf("writing %d remaining items to '%s'\n", rem, *realname);
      if(pri) {
	 fprintf(tmpf, "\n\n\n# the following items have been appended by '%s' V%s\n", *getString("application-name"), *getString("application-version"));
      } else {
	 fprintf(tmpf, "# this file was created by '%s' V%s\n", *getString("application-name"), *getString("application-version"));
	 fprintf(tmpf, "# feel free to edit this file\n");
      }
      fprintf(tmpf, "\n\n");
      for(i=0; i < opt.num(); i++) { 
	 const TAppConfigItem& a(opt(i));
	 if(a.only_app) continue;
	 if(a.only_cl) continue;
	 if(a.printed) continue;
	 if(!a.save) continue;
	 a.printCurItemToFile(tmpf, false);
      }
   }
   
   // finalize
   fclose(tmpf);
   if(rename(tmpname, realname)) {
      fprintf(stderr, "can't save config: can't move '%s' to '%s'!\n", *tmpname, *realname);
      return false;
   }
   return true;
}


void TAppConfig::createRCFile(const string& fname, const string& rcname) const {
   FILE *f;
   
   f = fopen(fname, "w");
   if(f==NULL) 
     userError("can't open '%s' for writing!\n", *fname);
   fprintf(f, "# this is a machine genreated rcfile for %s version %s\n",
	   *getString("application-name"), *getString("application-version"));
   fprintf(f, "# delete these lines and edit this file as soon as possible since all options\n");
   fprintf(f, "# are set to default values and all switches are disabled (commented out)\n\n\n\n");
   for(int i=0; i < opt.num(); i++) { 
      const TAppConfigItem& a(opt(i));
      if(a.only_app) continue;
      if(a.only_cl) continue;
      a.printItemToFile(f);
   }
   fclose(f);
   string globalname = "/etc/" + rcname;
   string localname;
   char *home = getenv("HOME");
   if(home) {
      localname = string(home) + "/." + rcname;
   } else {
      localname = "($HOME)/." + rcname;
   }
   printf("you should now customize the rcfile '%s' and\nsave it as '%s' (or '%s' as root)\n",
	  *fname, *localname, *globalname);
}


string TAppConfig::getName(const string& str, const string& context1, 
			   const string& optprefix) const {
   string context(context1);
   if(context) context += ": ";
   if(str.len()==0) 
     fatalError("%sempty option name!\n", *context);
   if(alias.contains(str)) return opt[alias[str]].name; // alias match
   if((str.len()==1)&&(optprefix!="--")) { // char name
      if(char2index[str[0]] < 0)
	userError("%sunknown char option name '%s'! (try --help)\n", *context, *str);
      return opt[char2index[str[0]]].name;
   }     
   if(name.contains(str)) return str; // exact match
   TArray<string> found; 
   for(int i=0; i<opt.num(); i++) { // search for prefixes
      if(opt[i].name.hasPrefix(str) && (!opt[i].only_app)) 
	found += opt[i].name;
   }
   if(found.num()==0) 
     userError("%sunknown option '%s'! (try --help)\n", *context, *(optprefix+str));
   if(found.num()==1) return found[0]; // uniq abbrevation
   string option = optprefix + str;
   string list = optprefix + join(found, (", "+optprefix));
   userError("%sambiguous option '%s' matches '%s'!\n", *context, *option, *list);
}


bool TAppConfig::getBool(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if((a.type==BOOL) || (a.type==SWITCH)) return a.bool_value;
      else fatalError("type mismatch in call for %s '%s' as bool!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


void TAppConfig::setValue(const string &n, bool b) {
   if(name.contains(n)) {
      opt[name(n)].setValueFromApp(b);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


void TAppConfig::setValue(const string &n, const string& str) {
   if(name.contains(n)) {
      opt[name(n)].setValueFromApp(str);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


bool TAppConfig::setValue(const string &n, int i) {
   if(name.contains(n)) {
      return opt[name(n)].setValueFromApp(i);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


bool TAppConfig::setValue(const string &n, double d) {
   if(name.contains(n)) {
      return opt[name(n)].setValueFromApp(d);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


bool TAppConfig::setValueFromStr(const string &n, const string& str) {
   if(name.contains(n)) {
      return opt[name(n)].setValueFromAppFromStr(str);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


const string& TAppConfig::getString(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==STRING) return a.string_value;
      else fatalError("type mismatch in call for %s '%s' as string!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


int TAppConfig::getInt(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==INT) return a.int_value;
      else fatalError("type mismatch in call for %s '%s' as int!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


int TAppConfig::intUpper(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==INT) return a.int_upper;
      else fatalError("type mismatch in call for %s '%s' as int!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


int TAppConfig::intLower(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==INT) return a.int_lower;
      else fatalError("type mismatch in call for %s '%s' as int!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


int TAppConfig::intDefault(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==INT) return a.int_default;
      else fatalError("type mismatch in call for %s '%s' as int!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


double TAppConfig::getDouble(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==DOUBLE) return a.double_value;
      else fatalError("type mismatch in call for %s '%s' as double!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


double TAppConfig::doubleUpper(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==DOUBLE) return a.double_upper;
      else fatalError("type mismatch in call for %s '%s' as double!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


double TAppConfig::doubleLower(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==DOUBLE) return a.double_lower;
      else fatalError("type mismatch in call for %s '%s' as double!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


double TAppConfig::doubleDefault(const string &n) const {
   if(name.contains(n)) {
      const TAppConfigItem& a(opt(name(n)));
      if(a.type==DOUBLE) return a.double_default;
      else fatalError("type mismatch in call for %s '%s' as double!\n", *a.getTypeStr(), *n);
   } else fatalError("unknown parameter name '%s'\n", *n);
}


string TAppConfig::stringDefault(const string &n) const {
  if(name.contains(n)) {
	 const TAppConfigItem& a(opt(name(n)));
	 if(a.type==STRING) return a.string_default;
	 else fatalError("type mismatch in call for %s '%s' as string!\n", *a.getTypeStr(), *n);
  } else fatalError("unknown parameter name '%s'\n", *n);
}


bool TAppConfig::boolDefault(const string &n) const {
  if(name.contains(n)) {
	 const TAppConfigItem& a(opt(name(n)));
	 if(a.type==BOOL) return a.bool_default;
	 else fatalError("type mismatch in call for %s '%s' as bool!\n", *a.getTypeStr(), *n);
  } else fatalError("unknown parameter name '%s'\n", *n);
}


bool TAppConfig::getSwitch(const string &n) const {
   return getBool(n);
}


bool TAppConfig::operator() (const string &n) const {
   return getBool(n);
}


void TAppConfig::setFromStr(const string& str, const string& context, 
			    TACO_SET_IN setin) {
   // prepare name n and parameter par
   string n;
   string par;
   int scanner=0;
   str.skipSpace(scanner);
   n = str.scanUpTo(scanner, " \t\n=");
   str.skipSpace(scanner);
   str.perhapsSkipOneChar(scanner, '=');
   str.skipSpace(scanner);
   par = str.scanRest(scanner);
   par.cropSpace();
   
   // set value
   if(n) {
      n = getName(n, context); // get name
      TAppConfigItem& a(opt[name(n)]);
      if(a.type==SWITCH) { // need no param
	 if(strchr(str, '=')) par = "t"; // force error
      } else { // get param
	 if(par.consistsOfSpace()) 
	   userError("%s: missing parameter for option '%s'!\n",
		     *context, *n);
      }		     	 
      par.unquote();
      par.compileCString();
      a.setValue(par, setin, verbose_conf, envname, rc_name, context);
   } else {
      userError("%s: syntax error, missing option name in '%s'!\n", 
		*context, *str);
   }
}

void TAppConfig::doEnvironVar(const char *env) {
   envname = env;
   char *p = getenv(env);
   if(p) {
      if(string(p).consistsOfSpace()) {
	 if(verbose_conf)
	   printf("environment variable '%s' is empty\n", env);
	 return;
      }
      if(verbose_conf)
	printf("parsing environment variable '%s'\n", env);
      TArray<string> a = split(p, ",", true, true);
      for(int i=0; i<a.num(); i++) {
	 setFromStr(a[i], "environment '" + string(env) + "'", ENVIRONMENT);
      }
   } else {
      if(verbose_conf)
	printf("environment variable '%s' not found\n", env);
   }
}

void TAppConfig::doRCFile(const string& rcfile, const string& clrcfile) {
   TArray<string> a;
   
   // which file ?
   if(clrcfile) {
      if(fisregular(clrcfile)) {
	 if(verbose_conf)
	   printf("parsing rc-file '%s'\n", *clrcfile);
	 rc_name = clrcfile;
      } else {
	 userError("can't load rc-file '%s'!\n", *clrcfile);
      }
   } else {
      string globalname = "/etc/" + rcfile;
      string localname;
      char *home = getenv("HOME");
      if(home) {
	 localname = string(home) + "/." + rcfile;
      }
      if(fisregular(localname)) {
	 if(verbose_conf)
	   printf("parsing local rc-file '%s'\n", *localname);
	 rc_name = localname;
      } else if(fisregular(globalname)) {
	 if(verbose_conf)
	   printf("parsing global rc-file '%s'\n", *globalname);
	 rc_name = globalname;
      } else {
	 if(verbose_conf)
	   printf("neither '%s' nor '%s' exist: no rc-file found\n",
		  *localname, *globalname);
	 return;
      }
   }
   
   // load file
   a = loadTextFile(rc_name);
   
   // parse rc-file
   for(int i=0; i<a.num(); i++) {
      if(!strchr("#;!/", a[i][0])) { // ignore comment
	 if(!a[i].consistsOfSpace()) 
	   setFromStr(a[i], "rc-file '" + rc_name + "' (line " + string(i+1) + ")", RC_FILE);
      }
   }
}

void TAppConfig::doCommandLine(int ac, char *av[], const string& version) {
   // set version and name
   opt[name("application-name")].string_value = av[0];
   opt[name("application-name")].string_value.extractFilename();
   opt[name("application-version")].string_value = version;
   
   // parse command line
   bool nomoreoptions = false; // -- reached = false 
   for(int i=1; i<ac; i++) {
      if((av[i][0]=='-') && (!nomoreoptions)) {
	 switch(av[i][1]) {
	  case 0: // simple parameter '-'
	    _params += av[i];
	    break;
	    
	  case '-': // long name parameter
	    if(av[i][2]==0) { // simple parameter '--'
	       _params += av[i];
	       if(stopatdd) nomoreoptions = true; // suppress option scanning
	    } else {
	       char *p = strchr(av[i], '=');	       
	       string n(&av[i][2]);
	       if(p) n.truncate(p - &av[i][2]);
	       if(n) {
		  n = getName(n, onlycl?"":"command line", "--");
		  TAppConfigItem& a(opt[name(n)]);
		  string par;
		  if(a.type==SWITCH) { // need no param
		     if(p) par = "t"; // force error
		  } else { // get param
		     if(p) par = &av[i][p-av[i]+1];
		     else {
			i++;
			if(i<ac) par = av[i];
			else userError("missing parameter for long option '--%s'! (try --help)\n", *n);
		     }
		  }		     
		  a.setValue(par, COMMAND_LINE);
	       } else {
		  userError("syntax error: missing name in long option '%s'!\n", av[i]);
	       }
	    }
	    break;
	    
	  default:  // short name parameter
	    for(const char *p = &av[i][1]; *p; p++) {
	       if(char2index[*p] >= 0) { // valid short option
		  TAppConfigItem& a(opt[char2index[*p]]);
		  string par;
		  if(a.type != SWITCH) { // get param
		     if(p[1]) {
			par = &p[1];
		     } else {
			i++;
			if(i<ac) par = av[i];
			else userError("%smissing parameter for option '-%s'! (try --help)\n",
				       onlycl?"":"command line: ", p);
		     }
		  } 
		  a.setValue(par, COMMAND_LINE);
		  if(a.type != SWITCH) break;
	       } else {
		  userError("%sunknown option '-%c'! (try --help)\n", 
			    onlycl?"":"command line: ", *p);
	       }
	    }
	    break;
	 }
      } else {
	 _params += av[i]; // add simple parameter
      }
   }
}

int TAppConfig::getMaxOptLen() const {
   int max = 0;
   for(int i=0; i<opt.num(); i++) {
     if(!opt[i].only_app) {
	int len = opt[i].getOptLen();
	if(len>max) max = len;
     }
   }
   return max;
}


void TAppConfig::printHelp() const {
   string ausage(usage);
   string atrailer(trailer);
   ausage.searchReplace("%n", getString("application-name"));
   ausage.searchReplace("%v", getString("application-version"));
   ausage.searchReplace("%e", TAPPCONFIG_EMAIL);
   ausage.searchReplace("%gpl", TAPPCONFIG_GPL);
   atrailer.searchReplace("%n", getString("application-name"));
   atrailer.searchReplace("%v", getString("application-version"));
   atrailer.searchReplace("%e", TAPPCONFIG_EMAIL);
   atrailer.searchReplace("%gpl", TAPPCONFIG_GPL);
   puts(ausage);
   int max = getMaxOptLen();
   for(int i=0; i<opt.num(); i++)
     if((!opt[i].only_app) && (!(opt[i].configopt && onlycl)))
       opt[i].printHelp(max, onlycl);
   puts(atrailer);
}


void TAppConfig::printValues() const {
   for(int i=0; i<opt.num(); i++) 
     if((!opt[i].only_app) && (!opt[i].only_cl))
       opt[i].printValue(envname, rc_name);
}


void TAppConfig::setComp(const TArray<string>& a, const string& context) {
   if((a.num()==1) && (a[0].consistsOfSpace())) return;
   string comp = a[0];
   string param;
   if(a.num()>1) {
      param=a[1];
      param.unquote();
   }
   if(a.num()>2) {
      string s = join(a,'=');
      s.expandUnprintable();
      fatalError("%s: syntax error near component '%s'! (too many '=')\n", *context, *s);
   }
   if(comp=="usage") { usage = param;
   } else if(comp=="commonheadline") { commonhead = param;
   } else if(comp=="trailer") { trailer = param;
   } else if(comp=="onlycl") { onlycl = true;
   } else if(comp=="stopat--") { stopatdd = true;
   } else fatalError("%s: unknown component '%s'!\n", *context, *comp);
}


void TAppConfig::doMetaChar(const string& str, const string& context) {
   TArray<string> a = split(str, ",", true);
   for(int i=0; i < a.num(); i++) {
      setComp(split(a[i], "=", true, true), context);
   }   
}

void TAppConfig::addConfigItems(const char **list, const char *listname,
				bool privat) {
   TAppConfigItem a;
   
   for(int line=1; ; line++, list++) {
      // context string for errors
      string context("conflist " + string(listname) + " (line " + string(line) + ")");

      // test for end of list (EOL)
      if(*list == 0) 
	fatalError("%s: list not terminated! (0 pointer, should be \"EOL\")\n", *context);
      if(strcmp(*list, "EOL")==0) break;
      if(strcmp(*list, "")==0) break;
      if(strlen(*list)<7) 
	fatalError("%s: line corrupt (too short) perhaps listitem or list not terminated!\n (was '%s') check listitem (must have at least type and name)\nor terminate list with \"EOL\"\n",
		   *context, *list);
      
      // meta char?
      if(*list[0] == '#') {
	 doMetaChar((*list)+1, context);
	 continue;
      }
      
      // get config item
      a = TAppConfigItem(*list, context, privat);
      
      // skip useless options
      if(onlycl && a.configopt) continue;

      // register name
      if(name.contains(a.name)) 
	fatalError("%s: duplicate name '%s' (previoulsy defined in %s)!\n",
		   *context, *a.name, *opt(name(a.name)).context);
      name[a.name] = opt.num();

      // register char name
      if(a.char_name) {
	 if(char2index[(uint)a.char_name[0]]>=0)
	   fatalError("%s: duplicate char name '%c' (previoulsy defined in %s)!\n",
		      *context, a.char_name[0], *opt(char2index[(uchar)a.char_name[0]]).context);
	 char2index[(uint)a.char_name[0]] = opt.num();
      }

      // register aliases 
      for(int i=0; i < a.alias.num(); i++) {
	 if(alias.contains(a.alias[i]))
	   fatalError("%s: duplicate alias '%s'!\n", *context, *a.alias[i]);
	 alias[a.alias[i]] = opt.num();
      }

      // set headline for help option
      if(a.name == "help") 
	a.headline = commonhead;
      
      // add config item
      opt += a;
   }
}





