/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <gpg-error.h>
#include <errno.h>
#include <glib.h>
#include "pwmd_error.h"
#include "mutex.h"
#include "common.h"
#include "rcfile.h"

gboolean do_cache_push(const gchar *filename, const void *password, gsize);

void clear_rcfile_keys()
{
    gsize n;
    gchar **groups;
    gchar **p;

    groups = g_key_file_get_groups(keyfileh, &n);

    for (p = groups; *p; p++) {
	if (g_key_file_has_key(keyfileh, *p, "key", NULL) == TRUE)
	     g_key_file_set_string(keyfileh, *p, "key", "");
    }

    g_strfreev(groups);
}

static gpg_error_t _getline(struct crypto_s *crypto, const gchar *file)
{
    gint fd;
    struct stat st;
    gpg_error_t rc = 0;

    if (stat(file, &st) == -1)
	return gpg_error_from_syserror();

    crypto->tkey_len = st.st_size;
    crypto->tkey = gcry_malloc(crypto->tkey_len);

    if (!crypto->tkey)
	return GPG_ERR_ENOMEM;

    fd = open(file, O_RDONLY);

    if (fd == -1)
	return gpg_error_from_syserror();

    gsize len = read(fd, crypto->tkey, crypto->tkey_len);

    if (len != crypto->tkey_len) 
	rc = GPG_ERR_INCOMPLETE_LINE;

    close(fd);

    // Converting.
    if (crypto->fh && (crypto->fh->v1 || crypto->fh->ver.fh2.version <= 0x213)) {
	unsigned char *p = crypto->tkey;
	unsigned i;

	for (i = 0; i < crypto->tkey_len; i++) {
	    if (p[i] == '\0' || p[i] == '\n')
		break;
	}

	if (i != crypto->tkey_len)
	    log_write(N_("%s: warning: key file data truncated at byte %i. Wanted key data length is %i bytes."), file, i, crypto->tkey_len);

	crypto->tkey_len = i;
    }

    return rc;
}

gpg_error_t parse_rcfile_keyfile(struct crypto_s *crypto, const gchar *filename,
	gboolean import)
{
    GError *rv = NULL;
    gchar *t, *file = NULL;
    gpg_error_t rc = 0;

    // parse_rcfile_keys()
    if (import == FALSE) {
	if (g_key_file_has_key(keyfileh, filename, "key_file", &rv) == TRUE) {
	    file = g_key_file_get_string(keyfileh, filename, "key_file", &rv);

	    if (!file) {
		if (rv) {
		    log_write("%s: key_file: %s", rcfile, rv->message);
		    g_clear_error(&rv);
		}

		return GPG_ERR_ENFILE;
	    }

	    t = expand_homedir(file);

	    if (!t) {
		g_free(file);
		log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
		return GPG_ERR_ENOMEM;
	    }

	    g_free(file);
	    file = t;
	}

	if (rv) {
	    log_write("%s: key_file: %s", rcfile, rv->message);

	    if (file)
		g_free(file);

	    g_clear_error(&rv);
	    return GPG_ERR_UNKNOWN_ERRNO;
	}
    }
    else {
	/* -I or -C. The filename is a key file. */
	file = g_strdup(filename);

	if (!file) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    return GPG_ERR_ENOMEM;
	}
    }

    if (!file)
	return GPG_ERR_ENFILE;

    rc = _getline(crypto, file);

    if (rc)
	log_write("%s: %s: %s", filename, file, pwmd_strerror(rc));

    g_free(file);
    return rc;
}

gchar *get_key_file_string(const gchar *section, const gchar *what)
{
    gchar *val = NULL;
    GError *grc = NULL;

    MUTEX_LOCK(&rcfile_mutex);

    if (g_key_file_has_key(keyfileh, section, what, NULL) == TRUE) {
	val = g_key_file_get_string(keyfileh, section, what, &grc);

	if (grc) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
	    g_clear_error(&grc);
	}
    }
    else {
	if (g_key_file_has_key(keyfileh, "global", what, NULL) == TRUE) {
	    val = g_key_file_get_string(keyfileh, "global", what, &grc);

	    if (grc) {
		log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
		g_clear_error(&grc);
	    }
	}
    }

    MUTEX_UNLOCK(&rcfile_mutex);
    return val;
}

gint get_key_file_integer(const gchar *section, const gchar *what)
{
    gint val = -1;
    GError *grc = NULL;

    MUTEX_LOCK(&rcfile_mutex);

    if (g_key_file_has_key(keyfileh, section ? section : "global", what, NULL) == TRUE) {
	val = g_key_file_get_integer(keyfileh, section ? section : "global", what, &grc);

	if (grc) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
	    g_clear_error(&grc);
	}
    }
    else {
	if (g_key_file_has_key(keyfileh, "global", what, NULL) == TRUE) {
	    val = g_key_file_get_integer(keyfileh, "global", what, &grc);

	    if (grc) {
		log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
		g_clear_error(&grc);
	    }
	}
    }

    MUTEX_UNLOCK(&rcfile_mutex);
    return val;
}

gdouble get_key_file_double(const gchar *section, const gchar *what)
{
    gdouble val = -1;
    GError *grc = NULL;

    MUTEX_LOCK(&rcfile_mutex);

    if (g_key_file_has_key(keyfileh, section ? section : "global", what, NULL) == TRUE) {
	val = g_key_file_get_double(keyfileh, section ? section : "global", what, &grc);

	if (grc) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
	    g_clear_error(&grc);
	}
    }
    else {
	if (g_key_file_has_key(keyfileh, "global", what, NULL) == TRUE) {
	    val = g_key_file_get_double(keyfileh, "global", what, &grc);

	    if (grc) {
		log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
		g_clear_error(&grc);
	    }
	}
    }

    MUTEX_UNLOCK(&rcfile_mutex);
    return val;
}

gboolean get_key_file_boolean(const gchar *section, const gchar *what)
{
    gboolean val = FALSE;
    GError *grc = NULL;

    MUTEX_LOCK(&rcfile_mutex);

    if (g_key_file_has_key(keyfileh, section ? section : "global", what, NULL)
	    == TRUE) {
	val = g_key_file_get_boolean(keyfileh, section ? section : "global",
		what, &grc);

	if (grc) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
	    g_clear_error(&grc);
	}
    }
    else {
	if (g_key_file_has_key(keyfileh, "global", what, NULL) == TRUE) {
	    val = g_key_file_get_boolean(keyfileh, "global", what, &grc);

	    if (grc) {
		log_write("%s(%i): %s", __FILE__, __LINE__, grc->message);
		g_clear_error(&grc);
	    }
	}
    }

    MUTEX_UNLOCK(&rcfile_mutex);
    return val;
}

gboolean parse_rcfile_keys()
{
    gsize n;
    gchar **groups;
    gchar **p;
    gchar *str;

    groups = g_key_file_get_groups(keyfileh, &n);

    for (p = groups; *p; p++) {
	GError *rv = NULL;

	if (!strcmp(*p, "global"))
	    continue;

	if (g_key_file_has_key(keyfileh, *p, "key", &rv) == TRUE) {
	    str = g_key_file_get_string(keyfileh, *p, "key", &rv);

	    if (!str) {
		if (rv) {
		    log_write("%s: key: %s", rcfile, rv->message);
		    g_clear_error(&rv);
		}

		continue;
	    }

	    do_cache_push(*p, str, strlen(str));
	    g_free(str);
	    continue;
	}

	if (rv) {
	    log_write("%s: key: %s", rcfile, rv->message);
	    g_clear_error(&rv);
	    continue;
	}

	struct crypto_s *crypto = init_client_crypto();
	gpg_error_t rc;

	if (!crypto)
	    return FALSE;

	crypto->fh = read_file_header(*p, FALSE, &rc);

	if (!crypto->fh) {
	    log_write("%s: %s", *p, pwmd_strerror(rc));
	    cleanup_crypto(&crypto);
	    continue;
	}

	rc = parse_rcfile_keyfile(crypto, *p, FALSE);

	if (rc) {
	    cleanup_crypto(&crypto);
	    continue;
	}

	do_cache_push(*p, crypto->tkey, crypto->tkey_len);
	cleanup_crypto(&crypto);
    }

    g_strfreev(groups);
    return TRUE;
}

GKeyFile *parse_rcfile(gboolean specified)
{
    GKeyFile *kf = g_key_file_new();
    GError *rc = NULL;

    g_key_file_set_list_separator(kf, ',');

    if (g_key_file_load_from_file(kf, rcfile, G_KEY_FILE_NONE, &rc) == FALSE) {
	log_write("%s: %s", rcfile, rc->message);

	if (cmdline && specified) {
	    g_clear_error(&rc);
	    return NULL;
	}

	if (rc->code && rc->code != G_FILE_ERROR_NOENT) {
	    g_clear_error(&rc);
	    return NULL;
	}
    }

    set_rcfile_defaults(kf);
    return kf;
}

void setup_logging(GKeyFile *kf)
{
    gboolean n = g_key_file_get_boolean(kf, "global", "enable_logging", NULL);

    if (n == TRUE) {
	gchar *p = g_key_file_get_string(kf, "global", "log_path", NULL);

	logfile = expand_homedir(p);
	g_free(p);
    }

    log_syslog = g_key_file_get_boolean(kf, "global", "syslog", NULL);
}

/*
 * Make sure all settings are set to either the specified setting or a
 * default.
 */
void set_rcfile_defaults(GKeyFile *kf)
{
    gchar buf[PATH_MAX];

    if (g_key_file_has_key(kf, "global", "socket_path", NULL) == FALSE) {
	g_snprintf(buf, sizeof(buf), "~/.pwmd/socket");
	g_key_file_set_string(kf, "global", "socket_path", buf);
    }

    if (g_key_file_has_key(kf, "global", "data_directory", NULL) == FALSE) {
	g_snprintf(buf, sizeof(buf), "~/.pwmd/data");
	g_key_file_set_string(kf, "global", "data_directory", buf);
    }

    if (g_key_file_has_key(kf, "global", "backup", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "backup", TRUE);

    if (g_key_file_has_key(kf, "global", "log_path", NULL) == FALSE) {
	g_snprintf(buf, sizeof(buf), "~/.pwmd/log");
	g_key_file_set_string(kf, "global", "log_path", buf);
    }

    if (g_key_file_has_key(kf, "global", "enable_logging", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "enable_logging", FALSE);

#ifdef HAVE_MLOCKALL
    if (g_key_file_has_key(kf, "global", "disable_mlockall", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "disable_mlockall", TRUE);
#endif

    if (g_key_file_has_key(kf, "global", "cache_timeout", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "cache_timeout", -1);

    if (g_key_file_has_key(kf, "global", "iterations", NULL) == FALSE ||
	    g_key_file_get_double(kf, "global", "iterations", 0) < 0L)
	g_key_file_set_double(kf, "global", "iterations", 1UL);

    if (g_key_file_has_key(kf, "global", "disable_list_and_dump", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "disable_list_and_dump", FALSE);

    if (g_key_file_has_key(kf, "global", "iteration_progress", NULL) == FALSE)
	g_key_file_set_double(kf, "global", "iteration_progress", 1000ULL);

    if (g_key_file_has_key(kf, "global", "compression_level", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "compression_level", 6);

    if (g_key_file_has_key(kf, "global", "recursion_depth", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "recursion_depth", DEFAULT_RECURSION_DEPTH);

    if (g_key_file_has_key(kf, "global", "zlib_bufsize", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "zlib_bufsize", DEFAULT_ZLIB_BUFSIZE);

    zlib_bufsize = (guint)g_key_file_get_integer(kf, "global", "zlib_bufsize", NULL);

    max_recursion_depth = g_key_file_get_integer(kf, "global", "recursion_depth", NULL);
    disable_list_and_dump = g_key_file_get_boolean(kf, "global", "disable_list_and_dump", NULL);

#ifdef HAVE_MLOCKALL
    disable_mlock = g_key_file_get_boolean(kf, "global", "disable_mlockall", NULL);
#endif

    if (g_key_file_has_key(kf, "global", "syslog", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "syslog", FALSE);

    if (g_key_file_has_key(kf, "global", "keepalive", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "keepalive", DEFAULT_KEEPALIVE_TO);

#ifdef WITH_PINENTRY
    if (g_key_file_has_key(kf, "global", "enable_pinentry", NULL) == FALSE)
	g_key_file_set_boolean(kf, "global", "enable_pinentry", TRUE);

    if (g_key_file_has_key(kf, "global", "pinentry_timeout", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "pinentry_timeout",
		DEFAULT_PIN_TIMEOUT);

    if (g_key_file_has_key(kf, "global", "pinentry_path", NULL) == FALSE)
	g_key_file_set_string(kf, "global", "pinentry_path", PINENTRY_PATH);
#endif

    if (g_key_file_has_key(kf, "global", "xfer_progress", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "xfer_progress", 8196);

    if (g_key_file_has_key(kf, "global", "cipher", NULL) == FALSE)
	g_key_file_set_string(kf, "global", "cipher", "AES256");

    if (g_key_file_has_key(kf, "global", "log_level", NULL) == FALSE)
	g_key_file_set_integer(kf, "global", "log_level", 0);

    if (!g_key_file_has_key(kf, "global", "allowed", NULL)) {
	const gchar *users[] = { g_get_user_name(), NULL};

	g_key_file_set_string_list(kf, "global", "allowed", users, 1);
    }

    setup_logging(kf);
}
