/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <ctype.h>
#include <string.h>
#include <sys/un.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <sys/mman.h>
#include <termios.h>
#include <assert.h>
#include <syslog.h>
#include <zlib.h>
#include <gcrypt.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/resource.h>

#ifdef HAVE_GETOPT_LONG
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#else
#include "getopt_long.h"
#endif

#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif

#include "mem.h"
#include "xml.h"
#include "common.h"

#ifdef WITH_PINENTRY
#include "pinentry.h"
#endif

#include "commands.h"
#include "pwmd_error.h"
#include "cache.h"
#include "misc.h"
#include "pwmd.h"
#include "mutex.h"
#include "rcfile.h"

GCRY_THREAD_OPTION_PTH_IMPL;

static void *reload_rcfile_thread(void *arg)
{
    gboolean b = disable_list_and_dump;
    gchar **users = g_key_file_get_string_list(keyfileh, "global", "allowed",
	    NULL, NULL);
    GKeyFile *k;
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);
    MUTEX_LOCK(&rcfile_mutex);
    log_write(N_("reloading configuration file '%s'"), rcfile);
    k = parse_rcfile(FALSE);

    if (!k)
	goto done;

    g_key_file_free(keyfileh);
    keyfileh = k;
    parse_rcfile_keys();
    clear_rcfile_keys();
    startStopKeepAlive(FALSE);
    send_status_all(STATUS_CONFIG);
done:
    disable_list_and_dump = !disable_list_and_dump ? b : TRUE;
    g_key_file_set_string_list(keyfileh, "global", "allowed",
	    (const gchar **)users, g_strv_length(users));
    g_strfreev(users);
    MUTEX_UNLOCK(&rcfile_mutex);
    return NULL;
}

static void reload_rcfile()
{
    pth_t tid;
    pth_attr_t attr = pth_attr_new();
    gpg_error_t rc;

    pth_attr_init(attr);
    pth_attr_set(attr, PTH_ATTR_JOINABLE, 0);
    tid = pth_spawn(attr, reload_rcfile_thread, NULL);
    rc = gpg_error_from_syserror();
    pth_attr_destroy(attr);

    if (!tid)
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		pwmd_strerror(rc));
}

gpg_error_t send_error(assuan_context_t ctx, gpg_error_t e)
{
    gpg_err_code_t n = gpg_err_code(e);
    gpg_error_t code = gpg_err_make(PWMD_ERR_SOURCE, n);
    struct client_s *client = assuan_get_pointer(ctx);

    if (client)
	client->last_rc = e;

    if (!e)
	return assuan_process_done(ctx, 0);

    if (!ctx) {
	log_write("%s", pwmd_strerror(e));
	return e;
    }

    if (n == EPWMD_LIBXML_ERROR) {
	xmlErrorPtr xe = client->xml_error;

	if (!xe)
	    xe = xmlGetLastError();

	e = assuan_process_done(ctx, assuan_set_error(ctx, code, xe->message));
	log_write("%s", xe->message);

	if (xe == client->xml_error)
	    xmlResetError(xe);
	else
	    xmlResetLastError();

	client->xml_error = NULL;
	return e;
    }

    return assuan_process_done(ctx, assuan_set_error(ctx, code, pwmd_strerror(e)));
}

void log_write(const gchar *fmt, ...)
{
    gchar *args, *line;
    va_list ap;
    struct tm *tm;
    time_t now;
    gchar tbuf[21];
    gint fd = -1;
    gchar *name;
    gchar buf[255];
    pth_t tid = pth_self();
    pth_attr_t attr;

    if ((!logfile && !isatty(STDERR_FILENO) && log_syslog == FALSE) || !fmt)
	return;

    if (!cmdline && logfile) {
	if ((fd = open(logfile, O_WRONLY|O_CREAT|O_APPEND, 0600)) == -1) {
	    warn("%s", logfile);

	    if (!log_syslog)
		return;
	}
    }

    va_start(ap, fmt);

    if (g_vasprintf(&args, fmt, ap) == -1) {
	if (logfile && fd != -1)
	    close(fd);

	va_end(ap);
	return;
    }

    va_end(ap);

    if (cmdline) {
	fprintf(stderr, "%s\n", args);
	fflush(stderr);
	g_free(args);
	return;
    }

    attr = pth_attr_of(tid);

    if (pth_attr_get(attr, PTH_ATTR_NAME, &name) == FALSE)
	name = "unknown";

    pth_attr_destroy(attr);
    name = print_fmt(buf, sizeof(buf), "%s(%p): ", name, tid);

    if (!cmdline && log_syslog == TRUE)
	syslog(LOG_INFO, "%s%s", name, args);

    time(&now);
    tm = localtime(&now);
    strftime(tbuf, sizeof(tbuf), "%b %d %Y %H:%M:%S ", tm);
    tbuf[sizeof(tbuf) - 1] = 0;

    if (args[strlen(args)-1] == '\n')
	args[strlen(args)-1] = 0;

    line = g_strdup_printf("%s %i %s%s\n", tbuf, getpid(), name, args);
    g_free(args);

    if (!line) {
	if (logfile && fd != -1)
	    close(fd);

	return;
    }

    if (logfile && fd != -1) {
	pth_write(fd, line, strlen(line));
	fsync(fd);
	close(fd);
    }

    if (isatty(STDERR_FILENO)) {
	fprintf(stderr, "%s", line);
	fflush(stderr);
    }

    g_free(line);
}

static void usage(gchar *pn, gint rc)
{
    g_fprintf(rc == EXIT_FAILURE ? stderr : stdout, N_(
	    "Usage: %s [options] [file1] [...]\n"
	    "    --no-fork/-n\n"
	    "          run as a foreground process\n"
	    "    --rcfile/-f <filename>\n"
	    "          load the specified rcfile (~/.pwmd/config)\n"
	    "    --convert/-C <filename>\n"
	    "          convert a version 1 data file to version 2\n"
	    "    --import/-I <filename>\n"
	    "          import an XML file\n"
	    "    --iterations/-i\n"
	    "          encrypt with the specified number of iterations when importing\n"
	    "          (default is in the rcfile \"global\" section)\n"
	    "    --key-file/-k <filename>\n"
	    "          obtain the key from the specified file when importing or converting\n"
	    "    --outfile/-o <filename>\n"
	    "          output file to use when importing or converting (- for stdout)\n"
	    "    --disable-dump/-D\n"
	    "          disable use of the LIST, XPATH and DUMP commands\n"
	    "    --no-pinentry/-P\n"
	    "          disable use of pinentry\n"
	    "    --version\n"
	    "    --help\n"
	    ), pn);
    exit(rc);
}

static void setup_gcrypt()
{
    gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_pth);

    if (!gcry_check_version(GCRYPT_VERSION))
	errx(EXIT_FAILURE, N_("gcry_check_version(): Incompatible libgcrypt. Wanted %s, got %s."), GCRYPT_VERSION, gcry_check_version(NULL));

    gcry_set_allocation_handler(xmalloc, xmalloc, NULL, xrealloc, xfree);
}

static gpg_error_t validate_peer(struct client_s *cl)
{
    gchar **users;
    gboolean allowed = FALSE;
    pid_t pid;
    gid_t gid;
    uid_t uid;
    gpg_error_t rc = assuan_get_peercred(cl->ctx, &pid, &uid, &gid);

    if (rc)
	return rc;

    users = g_key_file_get_string_list(keyfileh, "global", "allowed", NULL, NULL);

    if (users) {
	for (gchar **p = users; *p; p++) {
	    struct passwd pw, *result;
	    struct group gr, *gresult;
	    char *buf;

	    if (*(*p) == '@') {
		size_t len = sysconf(_SC_GETGR_R_SIZE_MAX);

		if (len == -1)
		    len = 16384;

		buf = g_malloc(len);

		if (!buf) {
		    g_strfreev(users);
		    return GPG_ERR_ENOMEM;
		}

		if (!getgrnam_r(*(p)+1, &gr, buf, len, &gresult) && gresult) {
		    if (gresult->gr_gid == gid) {
			g_free(buf);
			allowed = TRUE;
			break;
		    }

		    len = sysconf(_SC_GETPW_R_SIZE_MAX);

		    if (len == -1)
			len = 16384;

		    gchar *tbuf = g_malloc(len);

		    for (gchar **t = gresult->gr_mem; *t; t++) {
			if (!getpwnam_r(*t, &pw, tbuf, len, &result) && result) {
			    if (result->pw_uid == uid) {
				g_free(buf);
				allowed = TRUE;
				break;
			    }
			}
		    }

		    g_free(tbuf);

		    if (allowed)
			break;
		}
	    }
	    else {
		size_t len = sysconf(_SC_GETPW_R_SIZE_MAX);

		if (len == -1)
		    len = 16384;

		buf = g_malloc(len);

		if (!buf) {
		    g_strfreev(users);
		    return GPG_ERR_ENOMEM;
		}

		if (!getpwnam_r(*p, &pw, buf, len, &result) && result) {
		    if (result->pw_uid == uid) {
			g_free(buf);
			allowed = TRUE;
			break;
		    }
		}
	    }

	    g_free(buf);
	}

	g_strfreev(users);
    }

    log_write("peer %s: uid=%i, gid=%i, pid=%i",
	    allowed ? N_("accepted") : N_("rejected"), uid, gid,pid);
    return allowed ? 0 : GPG_ERR_INV_USER_ID;
}

static gboolean new_connection(struct client_s *cl)
{
    gpg_error_t rc;
    gchar *ver;
    gchar *str;

    rc = assuan_init_socket_server_ext(&cl->ctx, cl->thd->fd, 2);

    if (rc)
	goto fail;

    assuan_set_pointer(cl->ctx, cl);
    ver = g_strdup_printf("%s", PACKAGE_STRING);
    assuan_set_hello_line(cl->ctx, ver);
    g_free(ver);
    rc = register_commands(cl->ctx);

    if (rc)
	goto fail;

    rc = assuan_accept(cl->ctx);

    if (rc)
	goto fail;

    rc = validate_peer(cl);

    /* May not be implemented on all platforms. */
    if (rc && gpg_err_code(rc) != GPG_ERR_ASS_GENERAL)
	goto fail;

    str = get_key_file_string("global", "debug_file");

    if (debugfp && str)
	assuan_set_log_stream(cl->ctx, debugfp);

    if (str)
	g_free(str);

    return TRUE;

fail:
    log_write("%s", pwmd_strerror(rc));
    return FALSE;
}

static void xml_error_cb(void *data, xmlErrorPtr e)
{
    struct client_s *client = data;

    /*
     * Keep the first reported error as the one to show in the error
     * description. Reset in send_error().
     */
    if (client->xml_error)
	return;

    xmlCopyError(e, client->xml_error);
}

void close_file_header(file_header_internal_t *fh)
{
    if (!fh)
	return;

    if (fh->fd != -1 || (cmdline == TRUE && fh->fd != STDOUT_FILENO))
	close(fh->fd);

    if (fh->doc)
	gcry_free(fh->doc);

    g_free(fh);
}

void cleanup_crypto(struct crypto_s **c)
{
    struct crypto_s *cr = *c;

    if (!cr)
	return;

    if (cr->iv) {
	gcry_free(cr->iv);
	cr->iv = NULL;
    }

    if (cr->key) {
	gcry_free(cr->key);
	cr->key = NULL;
    }

    if (cr->tkey) {
	gcry_free(cr->tkey);
	cr->tkey = NULL;
    }

    if (cr->tkey2) {
	gcry_free(cr->tkey2);
	cr->tkey2 = NULL;
    }

    if (cr->inbuf) {
	gcry_free(cr->inbuf);
	cr->inbuf = NULL;
    }

    if (cr->outbuf) {
	gcry_free(cr->outbuf);
	cr->outbuf = NULL;
    }

    close_file_header(cr->fh);
    cr->fh = NULL;

    if (cr->gh)
	gcry_cipher_close(cr->gh);

    cr->gh = NULL;
    g_free(cr);
    *c = NULL;
}

/*
 * This is called after a client_thread terminates. Set with
 * pth_cleanup_push().
 */
static void cleanup_cb(void *arg)
{
    struct client_thread_s *cn = arg;
    struct client_s *cl = cn->cl;

    MUTEX_LOCK(&cn_mutex);
    cn_thread_list = g_slist_remove(cn_thread_list, cn);
    MUTEX_UNLOCK(&cn_mutex);

    if (cn->msg_tid) {
	MUTEX_LOCK(&cn->mp_mutex);
	pth_cancel(cn->msg_tid);
	MUTEX_UNLOCK(&cn->mp_mutex);
    }

    if (cn->mp) {
	while (pth_msgport_pending(cn->mp)) {
	    pth_message_t *msg = pth_msgport_get(cn->mp);

	    g_free(msg->m_data);
	    g_free(msg);
	}

	pth_msgport_destroy(cn->mp);
    }

    if (!cl) {
	if (cn->fd != -1)
	    close(cn->fd);

	goto done;
    }

    if (!cl->freed)
	cleanup_client(cl);

    if (cl->ctx)
	assuan_deinit_server(cl->ctx);
    else if (cl->thd && cl->thd->fd != -1)
	close(cl->thd->fd);

#ifdef WITH_PINENTRY
    if (cl->pinentry)
	cleanup_pinentry(cl->pinentry);
#endif

    if (cl->crypto)
	cleanup_crypto(&cl->crypto);

    g_free(cl);
done:
    log_write(N_("exiting, fd=%i"), cn->fd);
    g_free(cn);
    send_status_all(STATUS_CLIENTS);
}

/* 
 * Called every time a connection is made from init_new_connection(). This is
 * the thread entry point.
 */
static void *client_thread(void *data)
{
    struct client_thread_s *thd = data;
    struct client_s *cl = g_malloc0(sizeof(struct client_s));
    gpg_error_t rc;
    pth_attr_t attr;

    /*
     * Prevent a race condition with init_new_connection() if this thread
     * fails (returns) for some reason before init_new_connection() releases
     * the cn_mutex.
     */
    MUTEX_LOCK(&cn_mutex);
    MUTEX_UNLOCK(&cn_mutex);
    pth_cleanup_push(cleanup_cb, thd);

    if (!cl) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

    attr = pth_attr_of(pth_self());
    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);
    thd->cl = cl;
    cl->thd = thd;

    if (!new_connection(cl))
	goto fail;

#ifdef WITH_PINENTRY
    cl->pinentry = pinentry_init();

    if (!cl->pinentry) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }
#endif

    thd->mp = pth_msgport_create(NULL);
    pth_mutex_init(&thd->mp_mutex);
    thd->msg_tid = pth_spawn(NULL, client_msg_thread, thd);
    rc = gpg_error_from_syserror();

    if (!thd->msg_tid) {
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		pwmd_strerror(rc));
	goto fail;
    }

    pth_yield(thd->msg_tid);
    rc = send_status(cl->ctx, STATUS_CACHE, NULL);

    if (rc) {
	log_write("%s", pwmd_strerror(rc));
	goto fail;
    }

    send_status_all(STATUS_CLIENTS);
    xmlSetStructuredErrorFunc(cl, xml_error_cb);

    for (;;) {
#ifdef WITH_PINENTRY
	pth_event_t pev = NULL;
#endif
	pth_status_t st, wst;
	pth_event_t wev = NULL;
	pth_event_t rev = pth_event(PTH_EVENT_FD|PTH_UNTIL_FD_READABLE,
		cl->thd->fd);
	pth_event_t ev = rev;

#ifdef WITH_PINENTRY
	if (cl->pinentry->status == PINENTRY_RUNNING) {
	    pev = pth_event(PTH_EVENT_FD|PTH_UNTIL_FD_READABLE, cl->pinentry->fd);
	    ev = pth_event_concat(ev, pev, NULL);
	}
#endif

	if (cl->inquire_status == INQUIRE_BUSY) {
	    wev = pth_event(PTH_EVENT_FD|PTH_UNTIL_FD_WRITEABLE, cl->thd->fd);
	    ev = pth_event_concat(ev, wev, NULL);
	}

	pth_cleanup_push(cleanup_ev_cb, ev);
	pth_wait(ev);
	st = pth_event_status(rev);
	wst = pth_event_status(wev);

	if (st == PTH_STATUS_OCCURRED || wst == PTH_STATUS_OCCURRED) {
	    rc = assuan_process_next(cl->ctx);

	    if (rc) {
		cl->inquire_status = INQUIRE_INIT;
		pth_cleanup_pop(1);

		if (gpg_err_code(rc) == GPG_ERR_EOF)
		    goto done;

		log_write("assuan_process_next(): %s", pwmd_strerror(rc));
		rc = send_error(cl->ctx, gpg_err_make(PWMD_ERR_SOURCE, rc));

		if (rc) {
		    log_write("assuan_process_done(): %s", pwmd_strerror(rc));
		    goto done;
		}
	    }
	    else {
#ifdef WITH_PINENTRY
		if (cl->pinentry->pid && cl->pinentry->status == PINENTRY_INIT)
		    cl->pinentry->status = PINENTRY_RUNNING;
#endif

		switch (cl->inquire_status) {
		    case INQUIRE_BUSY:
		    case INQUIRE_INIT:
			break;
		    case INQUIRE_DONE:
			cl->inquire_status = INQUIRE_INIT;
			rc = assuan_process_done(cl->ctx, 0);
			break;
		}
	    }
	}

#ifdef WITH_PINENTRY
	if (pev)
	    st = pth_event_status(pev);

	rc = pinentry_iterate(cl,
		pev && cl->pinentry->fd != -1 && st == PTH_STATUS_OCCURRED);
#endif
	pth_cleanup_pop(1);
    }

    /*
     * Client cleanup (including XML data) is done in cleanup_cb(). 
     */
done:
fail:
    pth_exit(PTH_CANCELED);
    return NULL;
}

static gchar *do_read_password(const gchar *prompt)
{
    gchar buf[LINE_MAX] = {0}, *p;
    struct termios told, tnew;
    gchar *key;

    if (tcgetattr(STDIN_FILENO, &told) == -1) {
	log_write("tcgetattr(): %s", pwmd_strerror(gpg_error_from_syserror()));
	return NULL;
    }

    memcpy(&tnew, &told, sizeof(struct termios));
    tnew.c_lflag &= ~(ECHO);
    tnew.c_lflag |= ICANON|ECHONL;

    if (tcsetattr(STDIN_FILENO, TCSANOW, &tnew) == -1) {
	log_write("tcsetattr(): %s", pwmd_strerror(gpg_error_from_syserror()));
	tcsetattr(STDIN_FILENO, TCSANOW, &told);
	return NULL;
    }

    fprintf(stderr, "%s", prompt);

    if ((p = fgets(buf, sizeof(buf), stdin)) == NULL) {
	tcsetattr(STDIN_FILENO, TCSANOW, &told);
	return NULL;
    }

    tcsetattr(STDIN_FILENO, TCSANOW, &told);
    p[strlen(p) - 1] = 0;

    if (!buf[0]) {
	key = gcry_malloc(1);
	key[0] = 0;
    }
    else {
	key = gcry_malloc(strlen(p) + 1);
	sprintf(key, "%s", p);
    }

    memset(&buf, 0, sizeof(buf));
    return key;
}

/* Only used when "enable_pinentry" is "false" or -P. */
static gpg_error_t read_password(const gchar *filename,
	struct crypto_s *crypto, guchar *key, pinentry_cmd_t which)
{
    gchar *prompt;

    if (which == PINENTRY_SAVE) {
	prompt = g_strdup_printf(N_("New passphrase for file %s: "), filename);
	crypto->tkey = do_read_password(prompt);
	g_free(prompt);

	if (!crypto->tkey) {
	    log_write(N_("%s: Skipping file"), filename);
	    return GPG_ERR_BAD_PASSPHRASE;
	}

	prompt = g_strdup_printf(N_("Repeat passphrase: "));
	crypto->tkey2 = do_read_password(prompt);
	g_free(prompt);

	if (!crypto->tkey2) {
	    log_write(N_("%s: Skipping file"), filename);
	    return GPG_ERR_BAD_PASSPHRASE;
	}

	if (strcmp(crypto->tkey, crypto->tkey2)) {
	    log_write(N_("%s: Passphrase mismatch"), filename);
	    return GPG_ERR_INV_PASSPHRASE;
	}

	gcry_md_hash_buffer(GCRY_MD_SHA256, key, crypto->tkey,
		strlen(crypto->tkey) ? strlen(crypto->tkey) : 1);
	return 0;
    }

    prompt = g_strdup_printf(N_("Passphrase required for %s: "), filename);

    if ((crypto->tkey = do_read_password(prompt)) == NULL) {
	log_write(N_("%s: Skipping file"), filename);
	g_free(prompt);
	return GPG_ERR_BAD_PASSPHRASE;
    }

    gcry_md_hash_buffer(GCRY_MD_SHA256, key, crypto->tkey,
	    strlen(crypto->tkey) ? strlen(crypto->tkey) : 1);
    g_free(prompt);
    return 0;
}

/*
 * inbuf must have been allocated with gcry_malloc().
 */
gpg_error_t export_common(const gchar *filename, struct crypto_s *crypto,
	gpointer inbuf, gulong insize)
{
    gpg_error_t rc;
    gint level;
    gulong outsize;
    gpointer outbuf;

    rc = update_save_flags(NULL, crypto);

    if (rc)
	return rc;

    level = get_key_file_integer(filename, "compression_level");

    if (level < 0)
	level = 0;

    rc = do_compress(NULL, level, inbuf, insize, &outbuf, &outsize);

    if (rc)
	return rc;

    crypto->inbuf = outbuf;
    crypto->insize = outsize;
    rc = do_xml_encrypt(NULL, crypto, filename);
    return rc;
}

static gpg_error_t get_password(const gchar *filename,
	struct crypto_s *crypto, guchar *md5file, guchar *key,
	pinentry_cmd_t which)
{
#ifdef WITH_PINENTRY
    gpg_error_t rc = 0;

    if (g_key_file_get_boolean(keyfileh, "global", "enable_pinentry", NULL)
	    == FALSE) {
#endif
	return read_password(filename, crypto, key, which);
#ifdef WITH_PINENTRY
    }
    else {
	gchar *result = NULL;
	struct pinentry_s *pin = g_malloc0(sizeof(struct pinentry_s));

	pth_mutex_init(&pin->status_mutex);
	set_pinentry_defaults(pin);
	pin->which = which;
	pin->filename = g_strdup(filename);
	rc = pinentry_getpin(pin, &result);

	if (rc) {
	    xfree(result);
	    goto done;
	}
	
	gcry_md_hash_buffer(GCRY_MD_SHA256, key, result, result ? strlen(result) : 1);
	xfree(result);
	cleanup_pinentry(pin);
    }

done:
    return rc;
#endif
}

static gboolean xml_import(const gchar *filename, const gchar *outfile, 
	const gchar *keyfile, guint64 iter)
{
    xmlDocPtr doc;
    gint fd;
    struct stat st;
    gint len;
    xmlChar *xmlbuf;
    xmlChar *xml;
    gpg_error_t rc;
    struct crypto_s *crypto;
    guint hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    if (stat(filename, &st) == -1) {
	log_write("%s: %s", filename, pwmd_strerror(gpg_error_from_syserror()));
	return FALSE;
    }

    crypto = init_client_crypto();

    if (!crypto)
	return FALSE;

    crypto->key = gcry_malloc(hashlen);
    memset(crypto->key, 0, hashlen);

    if (!crypto->key) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

    log_write(N_("Importing XML from '%s'. Output will be written to '%s' ..."),
	    filename, outfile);

    if (iter && keyfile) {
	rc = parse_rcfile_keyfile(crypto, keyfile, TRUE);

	if (rc)
	    goto fail;

	gcry_md_hash_buffer(GCRY_MD_SHA256, crypto->key, crypto->tkey,
		crypto->tkey_len);
    }
    else if (iter) {
	rc = get_password(outfile, crypto, NULL, crypto->key, PINENTRY_SAVE);

	if (rc == GPG_ERR_ASSUAN_SERVER_FAULT) {
	    log_write(N_("%s. Maybe disabling pinentry (-P) will help?"),
		    pwmd_strerror(rc));
	    goto fail;
	}
	else if (rc) {
	    log_write("%s", pwmd_strerror(rc));
	    goto fail;
	}
    }

    if ((fd = open(filename, O_RDONLY)) == -1) {
	log_write("%s: %s", filename, pwmd_strerror(gpg_error_from_syserror()));
	goto fail;
    }

    if ((xmlbuf = gcry_malloc(st.st_size+1)) == NULL) {
	close(fd);
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

    if (pth_read(fd, xmlbuf, st.st_size) == -1) {
	rc = gpg_error_from_syserror();
	close(fd);
	log_write("%s: %s", filename, pwmd_strerror(rc));
	goto fail;
    }

    close(fd);
    xmlbuf[st.st_size] = 0;

    /*
     * Make sure the document validates.
     */
    if ((doc = xmlReadDoc(xmlbuf, NULL, "UTF-8", XML_PARSE_NOBLANKS)) == NULL) {
	log_write("xmlReadDoc() failed");
	gcry_free(xmlbuf);
	goto fail;
    }

    gcry_free(xmlbuf);
    xmlNodePtr n = xmlDocGetRootElement(doc);
    rc = validate_import(n ? n->children : n);

    if (rc) {
	log_write("%s", pwmd_strerror(rc));
	xmlFreeDoc(doc);
	goto fail;
    }

    xmlDocDumpMemory(doc, &xml, &len);
    xmlFreeDoc(doc);

    if (!iter)
	memset(crypto->key, '!', hashlen);

    crypto->fh = g_malloc0(sizeof(file_header_internal_t));

    if (!crypto->fh) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

    crypto->fh->ver.fh2.iter = iter;
    rc = export_common(outfile, crypto, xml, len);
    xmlFree(xml);

    if (rc) {
	send_error(NULL, rc);
	goto fail;
    }

    cleanup_crypto(&crypto);
    return TRUE;

fail:
    cleanup_crypto(&crypto);
    return FALSE;
}

gboolean do_cache_push(const gchar *filename, const void *password,
	gsize len)
{
    guchar md5file[16];
    gint timeout;
    const gchar *p = filename;
    struct crypto_s *crypto;
    gpg_error_t rc;
    guint hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    while (isspace(*p))
	p++;

    if (!*p)
	return FALSE;

    if (valid_filename(p) == FALSE) {
	log_write(N_("%s: Invalid characters in filename"), p);
	return FALSE;
    }

    gcry_md_hash_buffer(GCRY_MD_MD5, md5file, p, strlen(p));

    if (cache_iscached(md5file)) {
	log_write(N_("%s: already cached, skipping"), p);
	return FALSE;
    }

    if (access(p, R_OK|W_OK) != 0) {
	log_write("%s: %s", p, pwmd_strerror(gpg_error_from_syserror()));
	return FALSE;
    }

    crypto = init_client_crypto();

    if (!crypto)
	return FALSE;

    crypto->fh = read_file_header(filename, FALSE, &rc); 

    if (!crypto->fh) {
	log_write("%s: %s", p, pwmd_strerror(rc));
	cleanup_crypto(&crypto);
	return FALSE;
    }

    crypto->key = gcry_malloc(hashlen);

    if (!crypto->key) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	cleanup_crypto(&crypto);
	return FALSE;
    }

    log_write(N_("Adding '%s' to the file cache ..."), filename);

    if (crypto->fh->ver.fh2.iter <= 0ULL) {
	memset(crypto->key, '!', hashlen);
	goto try_decrypt;
    }

    if (!password) {
	rc = get_password(p, crypto, md5file, crypto->key, PINENTRY_OPEN);

	if (rc) {
	    send_error(NULL, rc);
	    cleanup_crypto(&crypto);
	    return FALSE;
	}

	gcry_free(crypto->fh->doc);
	crypto->fh->doc = NULL;
    }
    else
	gcry_md_hash_buffer(GCRY_MD_SHA256, crypto->key, password, len);

try_decrypt:
    rc = init_client_crypto2(filename, crypto);

    if (rc) {
	send_error(NULL, rc);
	cleanup_crypto(&crypto);
	return FALSE;
    }

    rc = try_xml_decrypt(NULL, crypto, NULL, NULL);

    if (rc) {
	log_write("%s: %s", filename, pwmd_strerror(rc));
	cleanup_crypto(&crypto);
	return FALSE;
    }

    if (cache_update_key(md5file, crypto->key) == FALSE) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	cleanup_crypto(&crypto);
	return FALSE;
    }

    timeout = get_key_file_integer(p, "cache_timeout");
    cache_set_timeout(md5file, timeout);
    log_write(N_("File '%s' now cached"), filename);
    cleanup_crypto(&crypto);
    return TRUE;
}

static void init_new_connection(gint fd)
{
    pth_attr_t attr;
    struct client_thread_s *new;
    gpg_error_t rc;

    new = g_malloc0(sizeof(struct client_thread_s));

    if (!new) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	close(fd);
	return;
    }

    MUTEX_LOCK(&cn_mutex);
    new->fd = fd;
    attr = pth_attr_new();
    pth_attr_init(attr);
    pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
    new->tid = pth_spawn(attr, client_thread, new);
    rc = gpg_error_from_syserror();
    pth_attr_destroy(attr);

    if (!new->tid) {
	g_free(new);
	close(fd);
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		pwmd_strerror(rc));
	MUTEX_UNLOCK(&cn_mutex);
	return;
    }

    cn_thread_list = g_slist_append(cn_thread_list, new);
    MUTEX_UNLOCK(&cn_mutex);
    log_write(N_("new connection: tid=%p, fd=%i"), new->tid, fd);
}

static void *accept_thread(void *arg)
{
    gint sockfd = *(gint *)arg;
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);

    for (;;) {
	socklen_t slen = sizeof(struct sockaddr_un);
	struct sockaddr_un raddr;
	gint fd = -1;

	fd = pth_accept(sockfd, (struct sockaddr *)&raddr, &slen);

	if (fd == -1) {
	    if (errno != EAGAIN) {
		if (!quit) // probably EBADF
		    log_write("accept(): %s", pwmd_strerror(gpg_error_from_syserror()));

		break;
	    }

	    continue;
	}

	init_new_connection(fd);
    }

    /* Just in case accept() failed for some reason other than EBADF */
    quit = 1;
    pth_exit(PTH_CANCELED);
    return NULL;
}

static void *adjust_cache_timer_thread(void *arg)
{
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);

    for (;;) {
	pth_sleep(1);
	CACHE_LOCK(NULL);
	cache_adjust_timer();
	CACHE_UNLOCK;
    }

    return NULL;
}

void cleanup_mutex_cb(void *arg)
{
    pth_mutex_t *m = arg;

    MUTEX_UNLOCK(m);
}

void cleanup_ev_cb(void *arg)
{
    pth_event_t ev = arg;

    pth_event_free(ev, PTH_FREE_ALL);
}

void cleanup_fd_cb(void *arg)
{
    gint fd = *(gint *)arg;

    close(fd);
}

void cleanup_unlink_cb(void *arg)
{
    gchar *file = arg;

    unlink(file);
}

void cleanup_cancel_cb(void *arg)
{
    pth_t tid = arg;
    pth_attr_t attr;
    gint join;

    attr = pth_attr_of(tid);
    pth_attr_get(attr, PTH_ATTR_JOINABLE, &join);
    pth_cancel(tid);

    if (join) {
	void *p;

	pth_join(tid, &p);
	g_free(p);
    }
}

static void *keepalive_thread(void *arg)
{
    gint to = *(gint *)arg;
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);

    for (;;) {
	pth_event_t ev = pth_event(PTH_EVENT_TIME, pth_timeout(to, 0));

	pth_cleanup_push(cleanup_ev_cb, ev);
	pth_wait(ev);
	send_status_all(STATUS_KEEPALIVE);
	pth_cleanup_pop(1);
    }

    return NULL;
}

static void startStopKeepAlive(gboolean term)
{
    gint n = get_key_file_integer("global", "keepalive");

    if (keepalive_tid)
	pth_cancel(keepalive_tid);

    keepalive_tid = NULL;

    if (term)
	return;

    if (n > 0) {
	pth_attr_t attr = pth_attr_new();
	gpg_error_t rc;

	pth_attr_init(attr);
	pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
	keepalive_tid = pth_spawn(attr, keepalive_thread, &n);
	rc = gpg_error_from_syserror();
	pth_attr_destroy(attr);

	if (!keepalive_tid) {
	    log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		    pwmd_strerror(rc));
	    return;
	}

	pth_yield(keepalive_tid);
    }
}

static gboolean waiting_for_exit()
{
    guint i, t;
    pth_event_t evs = NULL;

    MUTEX_LOCK(&cn_mutex);

    for (t = g_slist_length(cn_thread_list), i = 0;  i < t; i++) {
	struct client_thread_s *thd = g_slist_nth_data(cn_thread_list, i);
	pth_event_t ev = pth_event(PTH_EVENT_TID|PTH_UNTIL_TID_DEAD, thd->tid);

	if (evs)
	    evs = pth_event_concat(evs, ev, NULL);
	else
	    evs = ev;
    }

    MUTEX_UNLOCK(&cn_mutex);

    if (!evs)
	return FALSE;

    pth_wait(evs);
    MUTEX_LOCK(&cn_mutex);
    i = g_slist_length(cn_thread_list);
    MUTEX_UNLOCK(&cn_mutex);
    pth_event_free(evs, PTH_FREE_ALL);
    return i ? TRUE : FALSE;
}

static void catch_sigabrt(int sig)
{
    cache_clear(NULL, 2);
#ifndef MEM_DEBUG
    xpanic();
#endif
}

static void server_loop(gint sockfd, gchar **socketpath)
{
    pth_t accept_tid;
    guint n;
    sigset_t sigset;
    pth_attr_t attr;
    pth_t cache_timeout_tid;
    gpg_error_t rc;

    sigemptyset(&sigset);

    /* Termination */
    sigaddset(&sigset, SIGTERM);
    sigaddset(&sigset, SIGINT);

    /* Clears the file cache. */
    sigaddset(&sigset, SIGUSR1);

    /* Configuration file reloading. */
    sigaddset(&sigset, SIGHUP);

    /* Clears the cache and exits when something bad happens. */
    signal(SIGABRT, catch_sigabrt);
    sigaddset(&sigset, SIGABRT);

    /* Ignored everywhere. When a client disconnects abnormally this signal
     * gets raised. It isn't needed though because client_thread() will check
     * for rcs even after the client disconnects. */
    signal(SIGPIPE, SIG_IGN);
    sigprocmask(SIG_BLOCK, &sigset, NULL);

    log_write(N_("%s started for user %s"), PACKAGE_STRING, g_get_user_name());
    log_write(N_("Listening on %s"), *socketpath);
#ifndef HAVE_SO_PEERCRED
    log_write(N_("Peer credential checking is NOT supported on this OS."));
#endif
    attr = pth_attr_new();
    pth_attr_init(attr);
    accept_tid = pth_spawn(attr, accept_thread, &sockfd);
    rc = gpg_error_from_syserror();

    if (!accept_tid) {
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		pwmd_strerror(rc));
	pth_attr_destroy(attr);
	goto done;
    }

    pth_yield(accept_tid);
    startStopKeepAlive(FALSE);
    pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
    cache_timeout_tid = pth_spawn(attr, adjust_cache_timer_thread, NULL);
    rc = gpg_error_from_syserror();

    if (!cache_timeout_tid) {
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
		pwmd_strerror(rc));
	pth_attr_destroy(attr);
	goto done;
    }

    pth_yield(cache_timeout_tid);
    pth_attr_destroy(attr);

    {
	gchar *str = get_key_file_string("global", "debug_file");

	if (str) {
	    gchar *f = expand_homedir(str);

	    g_free(str);
	    debugfp = fopen(f, "w");

	    if (!debugfp)
		log_write("%s: %s", f, pwmd_strerror(gpg_error_from_syserror()));
	    else
		assuan_set_assuan_log_stream(debugfp);

	    g_free(f);
	}
    }

    do {
	gint sig;

	pth_sigwait(&sigset, &sig);
	log_write(N_("caught signal %i (%s)"), sig, strsignal(sig));

	/* Caught a signal. */
	switch (sig) {
	    case SIGHUP:
		reload_rcfile();
		break;
	    case SIGABRT:
		// not really handled here.
		catch_sigabrt(SIGABRT);
		break;
	    case SIGUSR1:
		CACHE_LOCK(NULL);
		log_write(N_("clearing file cache"));
		cache_clear(NULL, 2);
		CACHE_UNLOCK;
		break;
	    default:
		quit = 1;
		break;
	}
    } while (!quit);

done:
    /*
     * We're out of the main server loop. This happens when a signal was sent
     * to terminate the daemon. We'll wait for all clients to disconnect
     * before exiting and ignore any following signals.
     */
    shutdown(sockfd, SHUT_RDWR);
    close(sockfd);
    pth_cancel(accept_tid);
    pth_join(accept_tid, NULL);
    unlink(*socketpath);
    g_free(*socketpath);
    *socketpath = NULL;
    MUTEX_LOCK(&cn_mutex);
    n = g_slist_length(cn_thread_list);
    MUTEX_UNLOCK(&cn_mutex);

    if (n) {
	log_write(N_("waiting for all clients to disconnect"));

	do {
	    MUTEX_LOCK(&cn_mutex);
	    n = g_slist_length(cn_thread_list);
	    MUTEX_UNLOCK(&cn_mutex);
	    log_write(N_("%i clients remain"), n);
	} while (waiting_for_exit());
    }

    startStopKeepAlive(TRUE);
    pth_cancel(cache_timeout_tid);
    cache_free();
}

/*
 * Only called from pinentry_fork() in the child process.
 */
void free_client_list()
{
    gint i, t = g_slist_length(cn_thread_list);

    for (i = 0; i < t; i++) {
	struct client_thread_s *cn = g_slist_nth_data(cn_thread_list, i);

	free_client(cn->cl);
    }

    cache_free();
}

static guint pwmd_cipher_to_gcrypt(guint64 flags)
{
    if (flags & PWMD_CIPHER_AES128)
	return GCRY_CIPHER_AES128;
    else if (flags & PWMD_CIPHER_AES192)
	return GCRY_CIPHER_AES192;
    else if (flags & PWMD_CIPHER_AES256)
	return GCRY_CIPHER_AES256;
    else if (flags & PWMD_CIPHER_SERPENT128)
	return GCRY_CIPHER_SERPENT128;
    else if (flags & PWMD_CIPHER_SERPENT192)
	return GCRY_CIPHER_SERPENT192;
    else if (flags & PWMD_CIPHER_SERPENT256)
	return GCRY_CIPHER_SERPENT256;
    else if (flags & PWMD_CIPHER_CAMELLIA128)
	return GCRY_CIPHER_CAMELLIA128;
    else if (flags & PWMD_CIPHER_CAMELLIA192)
	return GCRY_CIPHER_CAMELLIA192;
    else if (flags & PWMD_CIPHER_CAMELLIA256)
	return GCRY_CIPHER_CAMELLIA256;
    else if (flags & PWMD_CIPHER_BLOWFISH)
	return GCRY_CIPHER_BLOWFISH;
    else if (flags & PWMD_CIPHER_3DES)
	return GCRY_CIPHER_3DES;
    else if (flags & PWMD_CIPHER_CAST5)
	return GCRY_CIPHER_CAST5;
    else if (flags & PWMD_CIPHER_TWOFISH)
	return GCRY_CIPHER_TWOFISH;
    else if (flags & PWMD_CIPHER_TWOFISH128)
	return GCRY_CIPHER_TWOFISH128;

    /* For backwards compatibility (no flags). */
    return GCRY_CIPHER_AES256;
}

guint pwmd_cipher_str_to_cipher(const gchar *str)
{
    guint64 flags = 0;

    if (!g_strcasecmp(str, "aes128"))
	flags = PWMD_CIPHER_AES128;
    else if (!g_strcasecmp(str, "aes192"))
	flags = PWMD_CIPHER_AES192;
    else if (!g_strcasecmp(str, "aes256"))
	flags = PWMD_CIPHER_AES256;
    if (!g_strcasecmp(str, "serpent128"))
	flags = PWMD_CIPHER_SERPENT128;
    else if (!g_strcasecmp(str, "serpent192"))
	flags = PWMD_CIPHER_SERPENT192;
    else if (!g_strcasecmp(str, "serpent256"))
	flags = PWMD_CIPHER_SERPENT256;
    if (!g_strcasecmp(str, "camellia128"))
	flags = PWMD_CIPHER_CAMELLIA128;
    else if (!g_strcasecmp(str, "camellia192"))
	flags = PWMD_CIPHER_CAMELLIA192;
    else if (!g_strcasecmp(str, "camellia256"))
	flags = PWMD_CIPHER_CAMELLIA256;
    else if (!g_strcasecmp(str, "blowfish"))
	flags = PWMD_CIPHER_BLOWFISH;
    else if (!g_strcasecmp(str, "cast5"))
	flags = PWMD_CIPHER_CAST5;
    else if (!g_strcasecmp(str, "3des"))
	flags = PWMD_CIPHER_3DES;
    else if (!g_strcasecmp(str, "twofish256"))
	flags = PWMD_CIPHER_TWOFISH;
    else if (!g_strcasecmp(str, "twofish128"))
	flags = PWMD_CIPHER_TWOFISH128;

    return flags;
}

const gchar *pwmd_cipher_to_str(guint64 flags)
{
    if (flags & PWMD_CIPHER_AES128)
	return "aes128";
    else if (flags & PWMD_CIPHER_AES192)
	return "aes192";
    else if (flags & PWMD_CIPHER_AES256)
	return "aes256";
    else if (flags & PWMD_CIPHER_SERPENT128)
	return "serpent128";
    else if (flags & PWMD_CIPHER_SERPENT192)
	return "serpent192";
    else if (flags & PWMD_CIPHER_SERPENT256)
	return "serpent256";
    else if (flags & PWMD_CIPHER_CAMELLIA128)
	return "camellia128";
    else if (flags & PWMD_CIPHER_CAMELLIA192)
	return "camellia192";
    else if (flags & PWMD_CIPHER_CAMELLIA256)
	return "camellia256";
    else if (flags & PWMD_CIPHER_BLOWFISH)
	return "blowfish";
    else if (flags & PWMD_CIPHER_CAST5)
	return "cast5";
    else if (flags & PWMD_CIPHER_3DES)
	return "3des";
    else if (flags & PWMD_CIPHER_TWOFISH)
	return "twofish256";
    else if (flags & PWMD_CIPHER_TWOFISH128)
	return "twofish128";

    return NULL;
}

/* To be called after read_file_header(). This sets the wanted algorithm from
 * .flags */
gpg_error_t init_client_crypto2(const char *filename,
	struct crypto_s *crypto)
{
    gpg_error_t rc;
    guint algo;

    /* New file or conversion. */
    if (crypto->fh->v1)
	algo = pwmd_cipher_to_gcrypt(PWMD_CIPHER_AES256);
    else if (!crypto->fh->ver.fh2.flags) {
	gchar *tmp = get_key_file_string(filename ? filename : "global", "cipher");
	guint64 flags;

	flags = pwmd_cipher_str_to_cipher(tmp);
	g_free(tmp);
	algo = pwmd_cipher_to_gcrypt(flags);
	crypto->fh->ver.fh2.flags = flags;
    }
    else
	algo = pwmd_cipher_to_gcrypt(crypto->fh->ver.fh2.flags);

    rc = gcry_cipher_algo_info(algo, GCRYCTL_TEST_ALGO, NULL, NULL);

    if (rc)
	return rc;

    rc = gcry_cipher_algo_info(algo, GCRYCTL_GET_KEYLEN, NULL,
	    &crypto->keysize);

    if (rc)
	return rc;

    rc = gcry_cipher_algo_info(algo, GCRYCTL_GET_BLKLEN, NULL,
	    &crypto->blocksize);

    if (rc)
	return rc;

    if (crypto->gh)
	gcry_cipher_close(crypto->gh);

    return gcry_cipher_open(&crypto->gh, algo, GCRY_CIPHER_MODE_CBC, 0);
}

struct crypto_s *init_client_crypto()
{
    struct crypto_s *new = g_malloc0(sizeof(struct crypto_s));

    if (!new) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return NULL;
    }

    return new;
}

static gpg_error_t convert_file(const gchar *filename, const gchar *keyfile,
	const gchar *outfile)
{
    gpg_error_t rc;
    guchar md5file[gcry_md_get_algo_dlen(GCRY_MD_MD5)];
    guint64 iter;
    struct crypto_s *crypto = init_client_crypto();
    guint hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    if (!crypto)
	return GPG_ERR_ENOMEM;

    crypto->key = gcry_malloc(hashlen);

    if (!crypto->key) {
	cleanup_crypto(&crypto);
	return GPG_ERR_ENOMEM;
    }

    log_write(N_("Converting version 1 data file '%s' to version 2 ..."),
	    filename);
    crypto->fh = read_file_header(filename, TRUE, &rc);

    if (!crypto->fh)
	goto done;

    gcry_md_hash_buffer(GCRY_MD_MD5, md5file, filename, strlen(filename));

    /* The header in version 1 had a bug where the iterations were off-by-one.
     * So 0 iterations was really -1 in the header. This is fixed in v2 of the
     * header.
     */
    if (crypto->fh->ver.fh1.iter != -1) {
	if (keyfile) {
	     rc = parse_rcfile_keyfile(crypto, keyfile, TRUE);

	    if (rc)
		goto done;

	    gcry_md_hash_buffer(GCRY_MD_SHA256, crypto->key, crypto->tkey,
		    crypto->tkey_len);
	    gcry_free(crypto->tkey);
	    crypto->tkey = NULL;
	}
	else {
	    rc = get_password(filename, crypto, md5file, crypto->key,
		    PINENTRY_OPEN);

	    if (rc)
		goto done;
	}
    }

    rc = init_client_crypto2(NULL, crypto);

    if (rc)
	goto done;

    rc = try_xml_decrypt(NULL, crypto, &crypto->fh->doc, &crypto->fh->len);

    if (rc)
	goto done;

    rc = convert_xml((gchar **)&crypto->fh->doc, &crypto->fh->len);

    if (rc) {
	log_write("%s: %s", filename, pwmd_strerror(rc));
	goto done;
    }

    crypto->fh->v1 = FALSE;

    iter = crypto->fh->ver.fh1.iter+1;
    memset(&crypto->fh->ver.fh2, 0, sizeof(crypto->fh->ver.fh2));
    /* Keep the iterations and key from the original file. */
    crypto->fh->ver.fh2.iter = iter;
    rc = export_common(outfile, crypto, crypto->fh->doc, crypto->fh->len);

done:
    if (rc)
	send_error(NULL, rc);

    /* fh->doc is freed from do_xml_decrypt() via the inbuf pointer. */
    cleanup_crypto(&crypto);
    return rc;
}

int main(int argc, char *argv[])
{
    gint opt;
    struct sockaddr_un addr;
    gchar buf[PATH_MAX];
    gchar *socketpath = NULL, *socketdir, *socketname = NULL;
    gchar *socketarg = NULL;
    gchar *datadir = NULL;
    gboolean n;
    gint x;
    gchar *p;
    gchar **cache_push = NULL;
    gchar *import = NULL, *keyfile = NULL;
    guint64 cmd_iterations = 0UL;
    gboolean iterations_arg = FALSE;
    gint default_timeout;
    gboolean rcfile_spec = FALSE;
    gint estatus = EXIT_FAILURE;
    gint sockfd;
    gchar *outfile = NULL;
    GMemVTable mtable = { xmalloc, xrealloc, xfree, xcalloc, NULL, NULL };
    gint do_unlink = 1;
    gboolean secure = FALSE;
    gint background = 1;
    gchar *convert = NULL;
    gint show_version = 0;
    gint show_help = 0;
#ifdef WITH_PINENTRY
    gboolean disable_pinentry = FALSE;
#endif
    gint opt_index;
    const struct option long_opts[] = {
#ifdef WITH_PINENTRY
	{ "no-pinentry", 0, 0, 'P' },
#endif
	{ "outfile", 1, 0, 'o' },
	{ "convert", 1, 0, 'C' },
	{ "no-fork", 0, 0, 'n' },
	{ "disable-dump", 0, 0, 'D' },
	{ "import", 1, 0, 'I' },
	{ "iterations", 1, 0, 'i' },
	{ "key-file", 1, 0, 'k' },
	{ "rcfile", 1, 0, 'f' },
	{ "version", 0, &show_version, 1 },
	{ "help", 0, &show_help, 1 },
	{ 0, 0, 0, 0}
    };
#ifdef WITH_PINENTRY
    const gchar *optstring = "Po:C:nDI:i:k:f:";
#else
    const gchar *optstring = "o:C:nDI:i:k:f:";
#endif
#ifndef DEBUG
#ifdef HAVE_SETRLIMIT
    struct rlimit rl;

    rl.rlim_cur = rl.rlim_max = 0;

    if (setrlimit(RLIMIT_CORE, &rl) != 0)
	err(EXIT_FAILURE, "setrlimit()");
#endif
#endif

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain("pwmd", LOCALEDIR);
    textdomain("pwmd");
#endif

#ifndef MEM_DEBUG
    xmem_init();
#endif
    setup_gcrypt();
    gpg_err_init();
    assuan_set_assuan_err_source(GPG_ERR_SOURCE_DEFAULT);
    g_mem_set_vtable(&mtable);
    assuan_set_malloc_hooks(xmalloc, xrealloc, xfree);
    xmlMemSetup(xfree, xmalloc, xrealloc, xstrdup);
    xmlInitMemory();
    xmlInitGlobals();
    xmlInitParser();
    xmlXPathInit();
    g_snprintf(buf, sizeof(buf), "%s/.pwmd", g_get_home_dir());

    if (mkdir(buf, 0700) == -1 && errno != EEXIST)
	err(EXIT_FAILURE, "%s", buf);

    g_snprintf(buf, sizeof(buf), "%s/.pwmd/data", g_get_home_dir());

    if (mkdir(buf, 0700) == -1 && errno != EEXIST)
	err(EXIT_FAILURE, "%s", buf);

    cmdline = TRUE;

    while ((opt = getopt_long(argc, argv, optstring, long_opts, &opt_index)) != -1) {
	switch (opt) {
	    case 0:
		if (show_help)
		    usage(argv[0], EXIT_SUCCESS);

		if (show_version) {
		    printf(N_("%s\nCopyright (C) 2010 %s\nReleased under the terms of the GPL v2. Use at your own risk.\n\nCompile time features:\n%s"), PACKAGE_STRING,
			    PACKAGE_BUGREPORT,
#ifdef WITH_PINENTRY
			    "+WITH_PINENTRY\n"
#else
			    "-WITH_PINENTRY\n"
#endif
#ifdef WITH_QUALITY
			    "+WITH_QUALITY\n"
#else
			    "-WITH_QUALITY\n"
#endif
#ifdef WITH_LIBACL
			    "+WITH_LIBACL\n"
#else
			    "-WITH_LIBACL\n"
#endif
#ifdef DEBUG
			    "+DEBUG\n"
#else
			    "-DEBUG\n"
#endif
#ifdef MEM_DEBUG
			    "+MEM_DEBUG\n"
#else
			    "-MEM_DEBUG\n"
#endif
#ifdef MUTEX_DEBUG
			    "+MUTEX_DEBUG\n"
#else
			    "-MUTEX_DEBUG\n"
#endif
			    );
		    exit(EXIT_SUCCESS);
		}
		break;
#ifdef WITH_PINENTRY
	    case 'P':
		disable_pinentry = TRUE;
		break;
#endif
	    case 'o':
		outfile = optarg;
		break;
	    case 'C':
		convert = optarg;
		break;
	    case 'n':
		background = 0;
		break;
	    case 'D':
		secure = TRUE;
		break;
	    case 'I':
		import = optarg;
		break;
	    case 'i':
		cmd_iterations = strtoul(optarg, NULL, 10);

		if (cmd_iterations == G_MAXULONG) {
		    log_write("%s", N_("invalid iteration count"));
		    usage(argv[0], EXIT_FAILURE);
		}

		iterations_arg = TRUE;
		break;
	    case 'k':
		keyfile = optarg;
		break;
	    case 'f':
		rcfile = g_strdup(optarg);
		rcfile_spec = TRUE;
		break;
	    default:
		usage(argv[0], EXIT_FAILURE);
	}
    }

    pth_mutex_init(&cn_mutex);
    pth_mutex_init(&cache_mutex);
    pth_mutex_init(&rcfile_mutex);
#ifdef WITH_PINENTRY
    pth_mutex_init(&pin_mutex);
#endif

    if (!rcfile)
	rcfile = g_strdup_printf("%s/.pwmd/config", g_get_home_dir());

    if ((keyfileh = parse_rcfile(rcfile_spec)) == NULL)
	exit(EXIT_FAILURE);

#ifdef WITH_PINENTRY
    if (disable_pinentry == TRUE)
	g_key_file_set_boolean(keyfileh, "global", "enable_pinentry", FALSE);
#endif

    if (g_key_file_has_key(keyfileh, "global", "syslog", NULL) == TRUE)
	log_syslog = g_key_file_get_boolean(keyfileh, "global", "syslog", NULL);

    if (log_syslog == TRUE)
	openlog("pwmd", LOG_NDELAY|LOG_PID, LOG_DAEMON);

    if (g_key_file_has_key(keyfileh, "global", "priority", NULL)) {
	x = g_key_file_get_integer(keyfileh, "global", "priority", NULL);
	errno = 0;

	if (setpriority(PRIO_PROCESS, 0, x) == -1) {
	    log_write("setpriority(): %s", pwmd_strerror(gpg_error_from_syserror()));
	    goto do_exit;
	}
    }

#ifdef HAVE_MLOCKALL
    if (disable_mlock == FALSE && mlockall(MCL_CURRENT|MCL_FUTURE) == -1) {
	log_write("mlockall(): %s", pwmd_strerror(gpg_error_from_syserror()));
	goto do_exit;
    }
#endif

    if (convert) {
	if (!outfile)
	    usage(argv[0], EXIT_FAILURE);

	opt = convert_file(convert, keyfile, outfile);
	g_key_file_free(keyfileh);
	g_free(rcfile);
	exit(opt ? EXIT_FAILURE : EXIT_SUCCESS);
    }

    if (import) {
	if (!outfile)
	    usage(argv[0], EXIT_FAILURE);

	if (!iterations_arg)
	    cmd_iterations = (guint64)get_key_file_double("global", "iterations");

	opt = xml_import(import, outfile, keyfile, cmd_iterations);
	g_key_file_free(keyfileh);
	g_free(rcfile);
	exit(opt == FALSE ? EXIT_FAILURE : EXIT_SUCCESS);
    }

    if ((p = g_key_file_get_string(keyfileh, "global", "socket_path", NULL)) == NULL)
	errx(EXIT_FAILURE, N_("%s: socket_path not defined"), rcfile);

    socketarg = expand_homedir(p);
    g_free(p);

    if ((p = g_key_file_get_string(keyfileh, "global", "data_directory", NULL)) == NULL)
	errx(EXIT_FAILURE, N_("%s: data_directory not defined"), rcfile);

    datadir = expand_homedir(p);
    g_free(p);

    if (secure == FALSE && g_key_file_has_key(keyfileh, "global", "disable_list_and_dump", NULL) == TRUE) {
	n = g_key_file_get_boolean(keyfileh, "global", "disable_list_and_dump", NULL);
	disable_list_and_dump = n;
    }
    else
	disable_list_and_dump = secure;

    if (g_key_file_has_key(keyfileh, "global", "cache_timeout", NULL) == TRUE)
	default_timeout = g_key_file_get_integer(keyfileh, "global", "cache_timeout", NULL);
    else
	default_timeout = -1;

    setup_logging(keyfileh);

    if (g_key_file_has_key(keyfileh, "global", "cache_push", NULL) == TRUE)
	cache_push = g_key_file_get_string_list(keyfileh, "global", "cache_push", NULL, NULL);

    if (argc != optind) {
	for (; optind < argc; optind++) {
	    if (strv_printf(&cache_push, "%s", argv[optind]) == FALSE)
		errx(EXIT_FAILURE, "%s", pwmd_strerror(GPG_ERR_ENOMEM));
	}
    }

    if (strchr(socketarg, '/') == NULL) {
	socketdir = g_get_current_dir();
	socketname = g_strdup(socketarg);
	socketpath = g_strdup_printf("%s/%s", socketdir, socketname);
    }
    else {
	socketname = g_strdup(strrchr(socketarg, '/'));
	socketname++;
	socketarg[strlen(socketarg) - strlen(socketname) -1] = 0;
	socketdir = g_strdup(socketarg);
	socketpath = g_strdup_printf("%s/%s", socketdir, socketname);
    }

    if (chdir(datadir)) {
	log_write("%s: %s", datadir, pwmd_strerror(gpg_error_from_syserror()));
	unlink(socketpath);
	goto do_exit;
    }

    if (parse_rcfile_keys() == FALSE)
	goto do_exit;

    clear_rcfile_keys();

    /*
     * Set the cache entry for a file. Prompts for the password.
     */
    if (cache_push) {
	for (opt = 0; cache_push[opt]; opt++)
	    do_cache_push(cache_push[opt], NULL, 0);

	g_strfreev(cache_push);
	log_write(background ? N_("Done. Daemonizing...") : N_("Done. Waiting for connections..."));
    }

    /*
     * bind() doesn't like the full pathname of the socket or any non alphanum
     * characters so change to the directory where the socket is wanted then
     * create it then change to datadir.
     */
    if (chdir(socketdir)) {
	log_write("%s: %s", socketdir, pwmd_strerror(gpg_error_from_syserror()));
	goto do_exit;
    }

    g_free(socketdir);

    if ((sockfd = socket(PF_UNIX, SOCK_STREAM, 0)) == -1) {
	log_write("socket(): %s", pwmd_strerror(gpg_error_from_syserror()));
	goto do_exit;
    }

    addr.sun_family = AF_UNIX;
    g_snprintf(addr.sun_path, sizeof(addr.sun_path), "%s", socketname);

    if (bind(sockfd, (struct sockaddr *)&addr, sizeof(struct sockaddr)) == -1) {
	log_write("bind(): %s", pwmd_strerror(gpg_error_from_syserror()));

	if (errno == EADDRINUSE)
	    log_write(N_("Either there is another pwmd running or '%s' is a \n"
		    "stale socket. Please remove it manually."), socketpath);

	do_unlink = 0;
	goto do_exit;
    }

    if (g_key_file_has_key(keyfileh, "global", "socket_perms", NULL) == TRUE) {
	gchar *t = g_key_file_get_string(keyfileh, "global", "socket_perms", NULL);
	mode_t mode = strtol(t, NULL, 8);
	mode_t mask = umask(0);

	g_free(t);

	if (chmod(socketname, mode) == -1) {
	    log_write("%s: %s", socketname, pwmd_strerror(gpg_error_from_syserror()));
	    close(sockfd);
	    unlink(socketpath);
	    umask(mask);
	    goto do_exit;
	}

	umask(mask);
    }

    g_free(--socketname);

    if (chdir(datadir)) {
	log_write("%s: %s", datadir, pwmd_strerror(gpg_error_from_syserror()));
	close(sockfd);
	unlink(socketpath);
	goto do_exit;
    }

    g_free(datadir);

    if (listen(sockfd, 0) == -1) {
	log_write("listen(): %s", pwmd_strerror(gpg_error_from_syserror()));
	goto do_exit;
    }

    cmdline = FALSE;
    unsetenv("DISPLAY");
    unsetenv("TERM");

    if (background) {
	switch (fork()) {
	    case -1:
		log_write("fork(): %s", pwmd_strerror(gpg_error_from_syserror()));
		goto do_exit;
	    case 0:
		close(0);
		close(1);
		close(2);
		setsid();
		break;
	    default:
		exit(EXIT_SUCCESS);
	}
    }

    server_loop(sockfd, &socketpath);
    estatus = EXIT_SUCCESS;

do_exit:
    if (socketpath && do_unlink) {
	unlink(socketpath);
	g_free(socketpath);
    }

    g_free(socketarg);
    g_key_file_free(keyfileh);
    g_free(rcfile);
    xmlCleanupParser();
    xmlCleanupGlobals();

    if (estatus == EXIT_SUCCESS)
	log_write(N_("pwmd exiting normally"));

#if defined(DEBUG) && !defined(MEM_DEBUG)
    xdump();
#endif
    exit(estatus);
}
