/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2007-2008 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <gcrypt.h>
#include <glib.h>
#include <errno.h>
#include <pth.h>
#include <pwd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "mem.h"
#include "common.h"
#include "commands.h"
#include "pinentry.h"
#include "misc.h"
#include "pwmd_error.h"
#include "mutex.h"
#include "rcfile.h"

#ifdef __FreeBSD__
#include <sys/types.h>
#include <signal.h>
#endif

#ifdef WITH_QUALITY
#include <crack.h>
#include "misc.h"
#endif

extern void free_client_list();
static gpg_error_t set_pinentry_strings(struct pinentry_s *pin, gint which);

static assuan_error_t mem_realloc_cb(void *data, const void *buffer, size_t len)
{
    membuf_t *mem = (membuf_t *)data;
    void *p;

    if (!buffer)
	return 0;

    if ((p = xrealloc(mem->buf, mem->len + len)) == NULL)
	return 1;

    mem->buf = p;
    memcpy((char *)mem->buf + mem->len, buffer, len);
    mem->len += len;
    return 0;
}

#ifdef WITH_QUALITY
static gpg_error_t quality_cb(void *data, const gchar *line)
{
    struct pinentry_s *pin = data;
    const gchar *tmp;
    gint score = 0;
    gchar buf[5];

    if (strncmp(line, "QUALITY ", 8) != 0)
	return GPG_ERR_INV_ARG;

    if (!(tmp = FascistCheck(line+8, CRACKLIB_DICT)))
	return assuan_send_data(pin->ctx, "100", 3);

    if (!strcmp(tmp, N_("it's WAY too short")))
	score = 10;
    else if (!strcmp(tmp, N_("it is too short")))
	score = 20;
    else if (!strcmp(tmp, N_("it is all whitespace")))
	score = 25;
    else if (!strcmp(tmp, N_("it is based on your username")))
	score = 30;
    else if (!strcmp(tmp, N_("it is based on a dictionary word")))
	score = 40;
    else if (!strcmp(tmp, N_("it is based upon your password entry")))
	score = 50;
    else if (!strcmp(tmp, N_("it's derived from your password entry")))
	score = 50;
    else if (!strcmp(tmp, N_("it is based on a (reversed) dictionary word")))
	score = 60;
    else if (!strcmp(tmp, N_("it is derivable from your password entry")))
	score = 70;
    else if (!strcmp(tmp, N_("it does not contain enough DIFFERENT characters")))
	score = 80;
    else if (!strcmp(tmp, N_("it is too simplistic/systematic")))
	score = 90;
    else
	score = 0;

    tmp = (const gchar *)print_fmt(buf, sizeof(buf), "%i", score);
    return assuan_send_data(pin->ctx, tmp, strlen(tmp));
}
#endif

static gpg_error_t assuan_command(struct pinentry_s *pin, gchar **result,
	const gchar *cmd)
{
    gpg_error_t rc;

    pin->data.len = 0;
    pin->data.buf = NULL;

    rc = assuan_transact(pin->ctx, cmd, mem_realloc_cb, &pin->data,
	    pin->inquire_cb, pin->inquire_data, NULL, NULL);

    if (rc) {
	if (pin->data.buf) {
	    xfree(pin->data.buf);
	    pin->data.buf = NULL;
	}
    }
    else {
	if (pin->data.buf) {
	    mem_realloc_cb(&pin->data, "", 1);
	    *result = (gchar *)pin->data.buf;
	}
    }

    return rc;
}

static gpg_error_t set_pinentry_options(struct pinentry_s *pin)
{
    gchar *display = getenv("DISPLAY");
    gint have_display = 0;
    gchar *tty = NULL, *ttytype = NULL;
    gchar *opt, *val;
    gpg_error_t rc;
    gchar *result = NULL;
    gchar cmd[ASSUAN_LINELENGTH];

    if (pin->display || display)
	have_display = 1;
    else {
	tty = pin->ttyname ? pin->ttyname : ttyname(STDOUT_FILENO);

	if (!tty)
	    return GPG_ERR_ASSUAN_SERVER_FAULT;
    }

    if (!have_display && !tty)
	return GPG_ERR_CANCELED;

    if (!have_display) {
	gchar *p = getenv("TERM");

	ttytype = pin->ttytype ? pin->ttytype : p;

	if (!ttytype)
	    return GPG_ERR_ASSUAN_SERVER_FAULT;
    }

    opt = have_display ? "DISPLAY" : "TTYNAME";
    val = have_display ? pin->display ? pin->display : display : tty;
    g_snprintf(cmd, sizeof(cmd), "OPTION %s=%s", g_ascii_strdown(opt, strlen(opt)), val);
    rc = assuan_command(pin, &result, cmd);

    if (rc)
	return rc;

    if (!have_display) {
	g_snprintf(cmd, sizeof(cmd), "OPTION ttytype=%s", ttytype);
	rc = assuan_command(pin, &result, cmd);
    }

    return rc;
}

static gpg_error_t launch_pinentry(struct pinentry_s *pin)
{
    gpg_error_t rc;
    assuan_context_t ctx;
    gint child_list[] = {-1};
    const gchar *argv[8];
    const gchar **p = argv;
    gchar *tmp;

    *p++ = "pinentry";

    if (pin->display) {
	*p++ = "--display";
	*p++ = pin->display;
    }

    if (pin->lcctype) {
	*p++ = "--lc-ctype";
	*p++ = pin->lcctype;
    }

    if (pin->lcmessages) {
	*p++ = "--lc-messages";
	*p++ = pin->lcmessages;
    }

    *p = NULL;
    tmp = get_key_file_string("global", "pinentry_path");
    rc = assuan_pipe_connect(&ctx, pin->path ? pin->path : tmp, argv,
	    child_list);
    g_free(tmp);

    if (rc)
	return rc;

    pin->pid = assuan_get_pid(ctx);
    pin->ctx = ctx;
    rc = set_pinentry_options(pin);
    return rc ? rc : set_pinentry_strings(pin, 0);
}

static gpg_error_t pinentry_command(struct pinentry_s *pin, gchar **result,
	const gchar *cmd)
{
    gpg_error_t rc = 0;

    if (!pin->ctx)
	rc = launch_pinentry(pin);

    return rc ? rc : assuan_command(pin, result, cmd);
}

static gpg_error_t set_pinentry_strings(struct pinentry_s *pin, gint which)
{
    gchar *buf;
    gpg_error_t rc;
    gchar *title = NULL;

#ifdef WITH_QUALITY
    if (pin->which == PINENTRY_SAVE && which != 2) {
	rc = pinentry_command(pin, NULL, "SETQUALITYBAR");

	if (rc)
	    goto done;

	pin->inquire_cb = quality_cb;
	pin->inquire_data = pin;
    }
#endif

    if (which == 1)
	title = g_strdup(N_("Passphrase mismatch, please try again."));
    else if (!pin->title)
	title = pin->title = g_strdup_printf(N_("Password Manager Daemon%s%s"),
		pin->name ? ": " : "", pin->name ? pin->name : "");
    else
	title = pin->title;

    if (!pin->prompt)
	pin->prompt = g_strdup(N_("Passphrase:"));

    if (!pin->desc && !which)
	pin->desc = g_strdup_printf(pin->which == PINENTRY_OPEN ?
		N_("A passphrase is required to open the file \"%s\". Please enter the passphrase below.") :
		N_("A passphrase is required to save to the file \"%s\". Please enter the passphrase below."),
		pin->filename);

    if (which == 2)
	buf = g_strdup_printf("SETERROR %s", N_("Please enter the passphrase again for confirmation."));
    else
	buf = g_strdup_printf("SETERROR %s", pin->desc);

    rc = pinentry_command(pin, NULL, buf);
    g_free(buf);

    if (rc)
	goto done;

    buf = g_strdup_printf("SETPROMPT %s", pin->prompt);
    rc = pinentry_command(pin, NULL, buf);
    g_free(buf);

    if (rc)
	goto done;

    buf = g_strdup_printf("SETDESC %s", title);
    rc = pinentry_command(pin, NULL, buf);
    g_free(buf);

done:
    if (which == 1)
	g_free(title);

    return rc;
}

static void pinentry_disconnect(struct pinentry_s *pin)
{
    if (!pin)
	return;

    if (pin->ctx)
	assuan_disconnect(pin->ctx);

    pin->ctx = NULL;
    pin->pid = 0;
}

static gpg_error_t do_getpin(struct pinentry_s *pin, gchar **result)
{
    gpg_error_t rc;

    *result = NULL;
    rc = pinentry_command(pin, result, "GETPIN");

    if (!*result)
	*result = xstrdup("");

    return rc;
}

gpg_error_t pinentry_getpin(struct pinentry_s *pin, gchar **result)
{
    gint which = 0;
    gpg_error_t rc = set_pinentry_strings(pin, which);
    gchar *result1 = NULL;

    if (rc)
	goto done;

again:
    rc = do_getpin(pin, result);

    if (rc)
	goto done;

    if (pin->which == PINENTRY_SAVE) {
	if (!result1) {
	    rc = set_pinentry_strings(pin, 2);

	    if (rc)
		goto done;

	    result1 = g_strdup(*result);
	    goto again;
	}

	if (strcmp(result1, *result)) {
	    g_free(result1);
	    xfree(*result);
	    result1 = *result = NULL;
	    rc = set_pinentry_strings(pin, 1);

	    if (rc)
		goto done;

	    goto again;
	}
    }

done:
    g_free(result1);
    pinentry_disconnect(pin);
    return rc;
}

static gint write_result(gint fd, pinentry_key_s *pk, gchar *result)
{
    gsize len;

    if (pk->error) {
	/*
	 * libassuan handles GPG_ERR_EOF in assuan_process_done() and
	 * will disconnect the client even if the error isn't related
	 * to it. Use GPG_ERR_CANCELED instead.
	 */
	if (gpg_err_code(pk->error) == GPG_ERR_EOF)
	    pk->error = GPG_ERR_CANCELED;

	len = pth_write(fd, pk, sizeof(pinentry_key_s));
	close(fd);

	if (len != sizeof(pinentry_key_s))
	    log_write("%s(%i): write: len != sizeof(pk)", __FUNCTION__, __LINE__);

	return 1;
    }

    if (pk->status == PINENTRY_PID)
	pk->what.pid = atoi(result);
    else
	g_strlcpy(pk->what.key, result, sizeof(pk->what.key));

    xfree(result);
    len = pth_write(fd, pk, sizeof(pinentry_key_s));

    if (len != sizeof(pinentry_key_s)) {
	memset(pk, 0, sizeof(pinentry_key_s));
	log_write("%s(%i): write: len != sizeof(pk)", __FUNCTION__, __LINE__);
	close(fd);
	return 1;
    }

    if (pk->status != PINENTRY_PID)
	close(fd);

    memset(pk, 0, sizeof(pinentry_key_s));
    return 0;
}

static void reset(struct pinentry_s *pin)
{
    gint status;

    if (pin->pid)
	waitpid(pin->pid, &status, 0);

    if (pin->fd != -1)
	close(pin->fd);

    pin->fd = -1;
    pin->pid = pin->pin_pid = 0;
    pin->tid = 0;
    pin->status = PINENTRY_NONE;
}

/* pin->status_mutex should be locked before calling this function. */
static void kill_pinentry(struct pinentry_s *pin)
{
    if (pin->pin_pid == 0)
	return;

    if (kill(pin->pin_pid, 0) == 0)
	if (kill(pin->pin_pid, SIGTERM) == 0)
	    if (kill(pin->pin_pid, 0) == 0)
		kill(pin->pin_pid, SIGKILL);

    pin->pin_pid = 0;
}

static void timeout_cleanup(void *arg)
{
    pth_event_t ev = arg;

    pth_event_free(ev, PTH_FREE_ALL);
}

static void *timeout_thread(void *arg)
{
    struct pinentry_s *pin = arg;
    pth_event_t ev;
    pth_attr_t attr = pth_attr_of(pth_self());

    pth_attr_set(attr, PTH_ATTR_NAME, __FUNCTION__);
    pth_attr_destroy(attr);
    ev = pth_event(PTH_EVENT_TIME, pth_timeout(pin->timeout, 0));
    MUTEX_LOCK(&pin->cond_mutex);
    pth_cleanup_push(timeout_cleanup, ev);
    pth_cond_await(&pin->cond, &pin->cond_mutex, ev);
    pth_cleanup_pop(1);

    /* pth_cond_notify() was called from pinentry_iterate() (we have a
     * key). */
    if (pin->status == PINENTRY_NONE) {
	MUTEX_UNLOCK(&pin->cond_mutex);
	pth_exit(PTH_CANCELED);
	return NULL;
    }

    MUTEX_LOCK(&pin->status_mutex);
    kill_pinentry(pin);
    pin->status = PINENTRY_TIMEOUT;
    MUTEX_UNLOCK(&pin->cond_mutex);
    MUTEX_UNLOCK(&pin->status_mutex);
    pth_exit(PTH_CANCELED);
    return NULL;
}

gpg_error_t pinentry_fork(assuan_context_t ctx)
{
    struct client_s *client = assuan_get_pointer(ctx);
    struct pinentry_s *pin = client->pinentry;
    gpg_error_t rc;
    gint p[2];
    pid_t pid;
    pinentry_key_s pk;
    gchar *result = NULL;

    if (pipe(p) == -1)
	return gpg_error_from_syserror();

    reset_pin_defaults(pin);
    pid = pth_fork();

    switch (pid) {
	case -1:
	    rc = gpg_error_from_syserror();
	    close(p[0]);
	    close(p[1]);
	    return rc;
	case 0:
	    close(p[0]);

	    if (pin->timeout > 0 && pin->which == PINENTRY_OPEN) {
		/*
		 * Send the pid of the pinentry process back to pwmd so it can
		 * handle the pinentry timeout properly.
		 */
		pk.status = PINENTRY_PID;
		pk.error = pinentry_command(pin, &result, "GETINFO pid");

		if (write_result(p[1], &pk, result)) {
		    free_client_list();
		    _exit(EXIT_FAILURE);
		}
	    }

	    pk.status = PINENTRY_RUNNING;
	    pk.error = pinentry_getpin(pin, &result);

	    if (write_result(p[1], &pk, result)) {
		free_client_list();
		_exit(EXIT_FAILURE);
	    }

	    free_client_list();
	    _exit(EXIT_SUCCESS);
	default:
	    close(p[1]);
	    client->pinentry->fd = p[0];
	    client->pinentry->pid = pid;
	    client->pinentry->status = PINENTRY_INIT;
	    break;
    }

    /*
     * Don't call assuan_process_done() here. That should be done in
     * the callback function which is called after the key has been read()
     * in pinentry_iterate().
     */
    return 0;
}

gpg_error_t lock_pin_mutex(struct client_s *client)
{
    gpg_error_t rc = 0;

    MUTEX_TRYLOCK(client, &pin_mutex, rc);

    if (!rc)
	client->pinentry->has_lock = TRUE;

    return rc;
}

void unlock_pin_mutex(struct pinentry_s *pin)
{
    if (pin->has_lock == FALSE)
	return;

    MUTEX_UNLOCK(&pin_mutex);
    pin->has_lock = FALSE;
}

void cleanup_pinentry(struct pinentry_s *pin)
{
    if (!pin)
	return;

    if (pin->ctx && pin->pid)
	pinentry_disconnect(pin);

    if (!pth_mutex_acquire(&pin->status_mutex, TRUE, NULL) && errno == EBUSY) {
	MUTEX_UNLOCK(&pin->status_mutex);
    }
    else
	pth_mutex_release(&pin->status_mutex);

    MUTEX_LOCK(&pin->status_mutex);
    kill_pinentry(pin);
    MUTEX_UNLOCK(&pin->status_mutex);
    unlock_pin_mutex(pin);

    if (!pth_mutex_acquire(&pin->cond_mutex, TRUE, NULL) && errno == EBUSY) {
	MUTEX_UNLOCK(&pin->cond_mutex);
    }
    else
	pth_mutex_release(&pin->cond_mutex);

    if (pin->name)
	g_free(pin->name);

    if (pin->ttyname)
	g_free(pin->ttyname);

    if (pin->ttytype)
	g_free(pin->ttytype);

    if (pin->desc)
	g_free(pin->desc);

    if (pin->title)
	g_free(pin->title);

    if (pin->prompt)
	g_free(pin->prompt);

    if (pin->path)
	g_free(pin->path);

    if (pin->display)
	g_free(pin->display);

    if (pin->filename)
	g_free(pin->filename);

    if (pin->lcctype)
	g_free(pin->lcctype);

    if (pin->lcmessages)
	g_free(pin->lcmessages);

    g_free(pin);
}

void reset_pin_defaults(struct pinentry_s *pin)
{
    pin->enable = -1;
    pin->timeout = get_key_file_integer(pin->filename, "pinentry_timeout");
}

void set_pinentry_defaults(struct pinentry_s *pin)
{
    FILE *fp;
    gchar buf[PATH_MAX];
    gchar *p;

    g_snprintf(buf, sizeof(buf), "%s/.pwmd/pinentry.conf", g_get_home_dir());
    fp = fopen(buf, "r");

    if (fp) {
	while ((p = fgets(buf, sizeof(buf), fp)) != NULL) {
	    gchar name[32] = {0}, value[256] = {0};

	    if (*p == '#')
		continue;

	    if (p[strlen(p)-1] == '\n')
		p[strlen(p)-1] = 0;

	    if (sscanf(p, " %31[a-zA-Z] = %255s", name, value) != 2)
		continue;

	    if (g_strcasecmp("TTYNAME", name) == 0) {
		if (pin->ttyname)
		    g_free(pin->ttyname);

		pin->ttyname = g_strdup(value);
	    }
	    else if (g_strcasecmp("TTYTYPE", name) == 0) {
		if (pin->ttytype)
		    g_free(pin->ttytype);

		pin->ttytype = g_strdup(value);
	    }
	    else if (g_strcasecmp("DISPLAY", name) == 0) {
		if (pin->display)
		    g_free(pin->display);

		pin->display = g_strdup(value);
	    }
	    else if (g_strcasecmp("PATH", name) == 0) {
		if (pin->path)
		    g_free(pin->path);

		pin->path = g_strdup(value);
	    }
	    else if (g_strcasecmp("LC_TYPE", name) == 0) {
		if (pin->lcctype)
		    g_free(pin->lcctype);

		pin->lcctype = g_strdup(value);
	    }
	    else if (g_strcasecmp("LC_MESSAGES", name) == 0) {
		if (pin->lcmessages)
		    g_free(pin->lcmessages);

		pin->lcmessages = g_strdup(value);
	    }
	}

	fclose(fp);
    }

    reset_pin_defaults(pin);
}

int pinentry_iterate(struct client_s *cl, gboolean read_ready)
{
    gpg_error_t rc;

    MUTEX_LOCK(&cl->pinentry->status_mutex);

    /* Set from pinentry_timeout_thread(). */
    if (cl->pinentry->status == PINENTRY_TIMEOUT) {
	cl->pinentry->status = PINENTRY_NONE;
	rc = send_error(cl->ctx, GPG_ERR_TIMEOUT);
	cleanup_client(cl);
	reset(cl->pinentry);
	unlock_pin_mutex(cl->pinentry);
    }

    if (cl->pinentry->status == PINENTRY_RUNNING) {
	if (read_ready) {
	    guchar *shakey;
	    pinentry_key_s pk;
	    gsize len;

	    memset(&pk, 0, sizeof(pk));
	    len = pth_read(cl->pinentry->fd, &pk, sizeof(pk));

	    if (len == sizeof(pk)) {
		guint hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

		if (pk.error) {
		    if (cl->pinentry->status == PINENTRY_TIMEOUT)
			pk.error = GPG_ERR_TIMEOUT;

		    cleanup_crypto(&cl->crypto);
		    rc = send_error(cl->ctx, pk.error);
		}
		else if (pk.status == PINENTRY_PID) {
		    /*
		     * Start the timeout thread for the pinentry process
		     * now that we know the pid of it.
		     */
		    pth_attr_t attr = pth_attr_new();
		    pth_attr_init(attr);
		    pth_attr_set(attr, PTH_ATTR_JOINABLE, FALSE);
		    cl->pinentry->pin_pid = pk.what.pid;
		    cl->pinentry->tid = pth_spawn(attr, timeout_thread,
			    cl->pinentry);
		    rc = gpg_error_from_syserror();
		    pth_attr_destroy(attr);

		    if (!cl->pinentry->tid) {
			log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__,
				pwmd_strerror(rc));
			pk.error = rc;
		    }
		}
		else {
		    if (cl->pinentry->tid) {
			cl->pinentry->status = PINENTRY_NONE;
			pth_cond_notify(&cl->pinentry->cond, FALSE);
		    }

		    shakey = gcry_malloc(hashlen);

		    if (!shakey)
			pk.error = GPG_ERR_ENOMEM;
		    else {
			gcry_md_hash_buffer(GCRY_MD_SHA256, shakey,
				pk.what.key,
				strlen(pk.what.key) ? strlen(pk.what.key) : 1);

			if (cl->crypto->key)
			    gcry_free(cl->crypto->key);

			cl->crypto->key = shakey;
			rc = cl->pinentry->cb(cl->ctx, cl->crypto->key, FALSE);
		    }
		}
	    }
	    else if (len == -1) {
		if (cl->pinentry->status == PINENTRY_TIMEOUT)
		    pk.error = GPG_ERR_TIMEOUT;

		pk.error = gpg_err_code_from_syserror();
		log_write("%s", pwmd_strerror(pk.error));
	    }
	    else if (len == 0) {
		if (cl->pinentry->status == PINENTRY_TIMEOUT)
		    pk.error = GPG_ERR_TIMEOUT;

		pk.error = GPG_ERR_EOF;
		log_write("%s", pwmd_strerror(pk.error));
	    }
	    else
		log_write(N_("read(): short byte count"));

	    if (pk.error) {
		cl->pinentry->status = PINENTRY_NONE;

		if (cl->pinentry->tid)
		    pth_cond_notify(&cl->pinentry->cond, FALSE);

		if (cl->pinentry->which == PINENTRY_OPEN)
		    cleanup_client(cl);
	    }

	    if (pk.error || pk.status == PINENTRY_RUNNING) {
		reset(cl->pinentry);
		unlock_pin_mutex(cl->pinentry);
	    }

	    memset(&pk, 0, sizeof(pk));
	}
    }

    MUTEX_UNLOCK(&cl->pinentry->status_mutex);
    return 0;
}

struct pinentry_s *pinentry_init()
{
    struct pinentry_s *pin = g_malloc0(sizeof(struct pinentry_s));

    if (!pin)
	return NULL;

    pth_mutex_init(&pin->status_mutex);
    pth_mutex_init(&pin->cond_mutex);
    pth_cond_init(&pin->cond);
    set_pinentry_defaults(pin);
    return pin;
}
