/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <glib.h>
#include <gcrypt.h>
#include <zlib.h>
#include <dirent.h>

#ifdef WITH_LIBACL
#include <sys/acl.h>
#endif

#include "mem.h"
#include "xml.h"
#include "common.h"

#ifdef WITH_PINENTRY
#include "pinentry.h"
#endif

#include "pwmd_error.h"
#include "cache.h"
#include "misc.h"
#include "commands.h"
#include "mutex.h"

struct gz_s {
    z_stream z;
    gpointer out;
    gboolean done;
    status_msg_t which;
};

static guchar crypto_magic[5] = "\177PWMD";

static gpg_error_t do_lock_command(struct client_s *client);

static void *z_alloc(void *data, unsigned items, unsigned size)
{
    return gcry_calloc(items, size);
}

static void z_free(void *data, void *p)
{
    gcry_free(p);
}

static gpg_error_t file_modified(struct client_s *client)
{
    struct stat st;
    gpg_error_t rc;

    if (client->state != STATE_OPEN)
	return EPWMD_NO_FILE;

    rc = lock_file_mutex(client);

    if (rc)
	return rc;

    if (lstat(client->filename, &st) == 0 && client->mtime) {
	if (client->mtime != st.st_mtime)
	    return EPWMD_FILE_MODIFIED;
    }

    pth_cancel_point();
    return 0;
}

static gpg_error_t parse_xml(assuan_context_t ctx)
{
    struct client_s *client = assuan_get_pointer(ctx);

    client->doc = xmlReadMemory(client->xml, client->len, NULL, "UTF-8", XML_PARSE_NOBLANKS);

    if (!client->doc)
	return EPWMD_LIBXML_ERROR;

    if (!client->crypto->fh || client->crypto->fh->ver.fh2.version >= 0x212)
	return 0;

    return convert_elements(client->doc);
}

void unlock_file_mutex(struct client_s *client)
{
    pth_mutex_t *m;

    if (client->has_lock == FALSE)
	return;

    CACHE_LOCK(client->ctx);

    if (cache_get_mutex(client->md5file, &m) == FALSE) {
	CACHE_UNLOCK;
	return;
    }

    CACHE_UNLOCK;
    MUTEX_UNLOCK(m);
    client->has_lock = client->is_lock_cmd = FALSE;
}

gpg_error_t lock_file_mutex(struct client_s *client)
{
    pth_mutex_t *m;
    gpg_error_t rc = 0;

    if (client->has_lock == TRUE)
	return 0;

    CACHE_LOCK(client->ctx);

    if (cache_get_mutex(client->md5file, &m) == FALSE) {
	CACHE_UNLOCK;
	return 0;
    }

    CACHE_UNLOCK;

    if (client->rc_on_locked) {
	if (!pth_mutex_acquire(m, TRUE, NULL))
	    return GPG_ERR_LOCKED;
#ifdef MUTEX_DEBUG
	MUTEX_LOCK_DEBUG(m);
#endif
    }
    else
	MUTEX_TRYLOCK(client, m, rc);

    if (!rc)
	client->has_lock = TRUE;

    return rc;
}

void free_client(struct client_s *client)
{
    if (client->doc)
	xmlFreeDoc(client->doc);

    if (client->xml)
	gcry_free(client->xml);

    if (client->filename)
	g_free(client->filename);

    if (client->crypto)
	cleanup_crypto(&client->crypto);

    if (client->xml_error)
	xmlResetError(client->xml_error);
}

void cleanup_client(struct client_s *client)
{
    assuan_context_t ctx = client->ctx;
    struct client_thread_s *thd = client->thd;
    gboolean rc_on_locked = client->rc_on_locked;
    gboolean lock_on_open = client->lock_on_open;
#ifdef WITH_PINENTRY
    struct pinentry_s *pin = client->pinentry;
#endif

    unlock_file_mutex(client);
    CACHE_LOCK(client->ctx);
    cache_decr_refcount(client->md5file);

    /*
     * This may be a new file so don't use a cache slot. save_command() will
     * set this to FALSE on success.
     */
    if (client->new == TRUE)
	cache_clear(client->md5file, 1);

    CACHE_UNLOCK;
    free_client(client);
    memset(client, 0, sizeof(struct client_s));
    client->state = STATE_CONNECTED;
    client->ctx = ctx;
    client->thd = thd;
    client->freed = TRUE;
#ifdef WITH_PINENTRY
    client->pinentry = pin;
#endif
    client->rc_on_locked = rc_on_locked;
    client->lock_on_open = lock_on_open;
}

static void gz_cleanup(void *arg)
{
    struct gz_s **gz = (struct gz_s **)arg;

    if (!gz)
	return;

    if (!(*gz)->done && (*gz)->out)
	gcry_free((*gz)->out);

    if ((*gz)->which == STATUS_COMPRESS) {
	if ((*gz)->z.zalloc)
	    deflateEnd(&(*gz)->z);
    }
    else {
	if ((*gz)->z.zalloc)
	    inflateEnd(&(*gz)->z);
    }

    g_free(*gz);
    *gz = NULL;
}

gpg_error_t do_decompress(assuan_context_t ctx, gpointer in, gulong insize,
	gpointer *out, gulong *outsize)
{
    struct gz_s *gz;
    gz_header h;
    gchar buf[17];
    gpg_error_t rc;
    gint zrc;

    gz = g_malloc0(sizeof(struct gz_s));

    if (!gz)
	return GPG_ERR_ENOMEM;

    pth_cleanup_push(gz_cleanup, &gz);
    gz->which = STATUS_DECOMPRESS;
    gz->z.zalloc = z_alloc;
    gz->z.zfree = z_free;
    gz->z.next_in = in;
    gz->z.avail_in = (uInt)insize;
    gz->z.avail_out = zlib_bufsize;
    gz->z.next_out = gz->out = gcry_malloc(zlib_bufsize);

    if (!gz->out) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	pth_cleanup_pop(1);
	return GPG_ERR_ENOMEM;
    }

    zrc = inflateInit2(&gz->z, 47);

    if (zrc != Z_OK) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
	pth_cleanup_pop(1);
	return zrc == Z_MEM_ERROR ? GPG_ERR_ENOMEM : GPG_ERR_COMPR_ALGO;
    }

    memset(&h, 0, sizeof(gz_header));
    h.comment = (guchar *)buf;
    h.comm_max = sizeof(buf);
    zrc = inflateGetHeader(&gz->z, &h);

    if (zrc != Z_OK) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
	pth_cleanup_pop(1);
	return zrc == Z_MEM_ERROR ? GPG_ERR_ENOMEM : GPG_ERR_COMPR_ALGO;
    }

    zrc = inflate(&gz->z, Z_BLOCK);

    if (zrc != Z_OK) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
	pth_cleanup_pop(1);
	return zrc == Z_MEM_ERROR ? GPG_ERR_ENOMEM : GPG_ERR_COMPR_ALGO;
    }

    if (h.comment)
	insize = (gulong)strtol((gchar *)h.comment, NULL, 10);

    do {
	gpointer p;

	zrc = inflate(&gz->z, Z_FINISH);

	switch (zrc) {
	    case Z_OK:
		break;
	    case Z_BUF_ERROR:
		if (!gz->z.avail_out) {
		    p = gcry_realloc(gz->out, gz->z.total_out + zlib_bufsize);

		    if (!p) {
			log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
			rc = GPG_ERR_ENOMEM;
			goto fail;
		    }

		    gz->out = p;
		    gz->z.next_out = (guchar *)gz->out + gz->z.total_out;
		    gz->z.avail_out = zlib_bufsize;
		    rc = send_status(ctx, STATUS_DECOMPRESS, "%li %li",
			    gz->z.total_out, insize);

		    if (rc)
			goto fail;
		}
		break;
	    case Z_STREAM_END:
		break;
	    default:
		log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
		rc = GPG_ERR_COMPR_ALGO;
		goto fail;
		break;
	}
    } while (zrc != Z_STREAM_END);

    rc = send_status(ctx, STATUS_DECOMPRESS, "%li %li", gz->z.total_out,
	    insize);

    if (rc)
	goto fail;

    *out = gz->out;
    *outsize = gz->z.total_out;
    gz->done = TRUE;
    pth_cleanup_pop(1);
    return 0;

fail:
    pth_cleanup_pop(1);
    return rc;
}

file_header_internal_t *read_file_header(const gchar *filename, gboolean v1,
	gpg_error_t *rc)
{
    gint fd;
    gsize len;
    file_header_internal_t *fh = g_malloc0(sizeof(file_header_internal_t));
    gsize fh_size;
    gpointer p;

    *rc = 0;

    if (!fh) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	*rc = GPG_ERR_ENOMEM;
	return NULL;
    }

    pth_cleanup_push(g_free, fh);
    fh_size = v1 ? sizeof(fh->ver.fh1) : sizeof(fh->ver.fh2);

    if (lstat(filename, &fh->st) == -1) {
	*rc = gpg_error_from_syserror();
	pth_cleanup_pop(1);
	return NULL;
    }

    if (!S_ISREG(fh->st.st_mode)) {
	*rc = GPG_ERR_ENOANO;
	pth_cleanup_pop(1);
	return NULL;
    }

    fd = open(filename, O_RDONLY);

    if (fd == -1) {
	*rc = gpg_error_from_syserror();
	pth_cleanup_pop(1);
	return NULL;
    }

    pth_cleanup_push(cleanup_fd_cb, &fd);
    p = v1 ? (void *)&fh->ver.fh1 : (void *)&fh->ver.fh2;
    len = pth_read(fd, p, fh_size);

    if (len != fh_size) {
	*rc = gpg_error_from_syserror();
	pth_cleanup_pop(1);
	pth_cleanup_pop(1);
	return NULL;
    }

    fh->v1 = v1;
    fh->fd = fd;
    pth_cleanup_pop(0);
    pth_cleanup_pop(0);
    return fh;
}

static gpg_error_t open_command_finalize(assuan_context_t ctx, guchar *unused,
	gboolean cached)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    gint timeout;
    guchar *key = client->crypto->key;

    /* New file. */
    if (!client->crypto->fh) {
	if (key[0])
	    goto update_cache;

	goto done;
    }

    rc = init_client_crypto2(client->filename, client->crypto);

    if (rc) {
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

    rc = try_xml_decrypt(ctx, client->crypto, NULL, NULL);

    if (rc) {
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

update_cache:
    CACHE_LOCK(client->ctx);

    if (cached == FALSE) {
	if (cache_update_key(client->md5file, key) == FALSE) {
	    cleanup_client(client);
	    CACHE_UNLOCK;
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}

	timeout = get_key_file_integer(client->filename, "cache_timeout");
	cache_reset_timeout(client->md5file, timeout);
    }
    else
	cache_set_timeout(client->md5file, -2);

    CACHE_UNLOCK;

done:
    rc = parse_xml(ctx);

    if (client->xml) {
	gcry_free(client->xml);
	client->xml = NULL;
    }

    if (!rc) {
	if (client->new == FALSE)
	    send_status_all(STATUS_CACHE);

	client->state = STATE_OPEN;
    }

    if (!rc && client->new == FALSE &&
	    client->crypto->fh->ver.fh2.iter != (guint64)get_key_file_double(client->filename, "iterations")) {
	MUTEX_LOCK(&rcfile_mutex);
	g_key_file_set_double(keyfileh, client->filename, "iterations",
		client->crypto->fh->ver.fh2.iter);
	MUTEX_UNLOCK(&rcfile_mutex);
	send_status_all(STATUS_CONFIG);
    }

    cleanup_crypto(&client->crypto);

    if (!rc && client->lock_on_open)
	return do_lock_command(client);

    return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
}

static void req_cleanup(void *arg)
{
    if (!arg)
	return;

    g_strfreev((gchar **)arg);
}

static gpg_error_t parse_open_opt_lock(gpointer data, gpointer value)
{
    struct client_s *client = data;
    const gchar *p = value;

    if (p && *p) {
	long l = strtol(p, NULL, 10);

	if (l < 0 || l > 1)
	    return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);

	client->lock_on_open = l ? TRUE : FALSE;
    }
    else if ((!p || !*p) && (client->opts & OPT_LOCK))
	client->lock_on_open = FALSE;
    else
	client->lock_on_open = TRUE;

    return 0;
}

static gpg_error_t parse_opt_pinentry(gpointer data, gpointer value)
{
#ifdef WITH_PINENTRY
    struct client_s *client = data;
    gchar *p = NULL;
    gchar *str = value;
    gint n;

    if (!str || !*str) {
	client->pinentry->enable = -1;
	client->opts &= ~(OPT_PINENTRY);
	return 0;
    }

    n = strtol(str, &p, 10);

    if (*p || n < 0 || n > 1)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);

    client->pinentry->enable = n ? TRUE : FALSE;
    client->opts |= OPT_PINENTRY;
    return 0;
#else
    return GPG_ERR_NOT_IMPLEMENTED;
#endif
}

static gpg_error_t parse_opt_inquire(gpointer data, gpointer value)
{
    struct client_s *client = data;

    (void)value;
    client->opts |= OPT_INQUIRE;
    return 0;
}

static gpg_error_t parse_opt_base64(gpointer data, gpointer value)
{
    struct client_s *client = data;

    (void)value;
    client->opts |= OPT_BASE64;
    return 0;
}

static gpg_error_t hash_key(struct client_s *client, const gchar *key)
{
    guchar *tmp;
    gsize len;

    if (client->opts & OPT_BASE64)
	tmp = g_base64_decode(key, &len);
    else {
	tmp = (guchar *)g_strdup(key);
	len = strlen(key);
    }

    if (!tmp)
	return GPG_ERR_ENOMEM;

    gcry_md_hash_buffer(GCRY_MD_SHA256, client->crypto->key, tmp, len);
    g_free(tmp);
    return 0;
}

static gpg_error_t open_command_common(assuan_context_t ctx, gchar *line)
{
    gboolean cached = FALSE;
    gpg_error_t rc;
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    gchar *filename = NULL;
    gsize hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    if ((req = split_input_line(line, " ", 2)) != NULL)
	filename = req[0];

    pth_cleanup_push(req_cleanup, req);

    if (!filename || !*filename) {
	pth_cleanup_pop(1);
	return client->opts & OPT_INQUIRE ? GPG_ERR_SYNTAX : send_error(ctx, GPG_ERR_SYNTAX);
    }

    log_write2("ARGS=\"%s\" %s", filename, req[1] ? "<passphrase>" : "");

    if (valid_filename(filename) == FALSE) {
	pth_cleanup_pop(1);
	return client->opts & OPT_INQUIRE ? GPG_ERR_INV_VALUE : send_error(ctx, GPG_ERR_INV_VALUE);
    }

    gcry_md_hash_buffer(GCRY_MD_MD5, client->md5file, filename, strlen(filename));
    CACHE_LOCK(client->ctx);

    if (cache_has_file(client->md5file) == FALSE) {
	if (cache_add_file(client->md5file, NULL) == FALSE) {
	    pth_cleanup_pop(1);
	    CACHE_UNLOCK;
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}
    }

    CACHE_UNLOCK;
    rc = lock_file_mutex(client);

    if (rc) {
	pth_cleanup_pop(1);
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

    client->freed = FALSE;
    CACHE_LOCK(client->ctx);
    cache_incr_refcount(client->md5file);
    CACHE_UNLOCK;
    client->crypto = init_client_crypto();

    if (!client->crypto) {
	pth_cleanup_pop(1);
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }

    client->crypto->key = gcry_malloc(hashlen);

    if (!client->crypto->key) {
	pth_cleanup_pop(1);
	log_write("%s(%i): %s", __FUNCTION__, __LINE__,
		GPG_ERR_ENOMEM);
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }

    memset(client->crypto->key, 0, hashlen);
    client->crypto->fh = read_file_header(filename, FALSE, &rc);

    if (!client->crypto->fh) {
	if (gpg_err_code_to_errno(rc) != ENOENT) {
	    log_write("%s: %s", filename, pwmd_strerror(rc));
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	}

	/*
	 * New files don't need a key.
	 */
	if ((client->xml = new_document()) == NULL) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}

	client->len = xmlStrlen(client->xml);
	client->new = TRUE;
	client->filename = g_strdup(filename);

	if (!client->filename) {
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}

	if (req[1] && *req[1]) {
	    rc = hash_key(client, req[1]);

	    if (rc) {
		pth_cleanup_pop(1);
		cleanup_client(client);
		return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	    }
	}

	pth_cleanup_pop(1);
#ifdef WITH_PINENTRY
	client->pinentry->filename = g_strdup(client->filename);

	if (!client->pinentry->filename) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    cleanup_client(client);
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}
#endif
	return open_command_finalize(ctx, NULL, cached);
    }
    else {
	if (!(client->opts & OPT_CIPHER))
	    g_key_file_set_string(keyfileh, filename, "cipher",
		    pwmd_cipher_to_str(client->crypto->fh->ver.fh2.flags));

	client->mtime = client->crypto->fh->st.st_mtime;
    }

    client->filename = g_strdup(filename);

    if (!client->filename) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	pth_cleanup_pop(1);
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }

#ifdef WITH_PINENTRY
    if (client->pinentry->filename)
	g_free(client->pinentry->filename);

    client->pinentry->filename = g_strdup(client->filename);

    if (!client->pinentry->filename) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	pth_cleanup_pop(1);
	cleanup_client(client);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }
#endif

    if (client->crypto->fh->ver.fh2.iter <= 0ULL)
	goto done;

    CACHE_LOCK(client->ctx);
    cached = cache_get_key(client->md5file, client->crypto->key);
    CACHE_UNLOCK;

    if (cached == FALSE) {
	gchar *tmp = get_key_file_string(filename, "key_file");

	if (tmp && !(client->opts & OPT_INQUIRE)) {
	    g_free(tmp);
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    return send_error(ctx, GPG_ERR_WRONG_KEY_USAGE);
	}

	if (tmp)
	    g_free(tmp);

	/*
	 * No key specified and no matching filename found in the cache. Use
	 * pinentry to retrieve the key. Cannot return assuan_process_done()
	 * here otherwise the command will be interrupted. The event loop in
	 * client_thread() will poll the file descriptor waiting for it to
	 * become ready to read a pinentry_key_s which will contain the
	 * entered key or an error code. It will then call
	 * open_command_finalize() to to finish the command.
	 */
	if (!req[1] || !*req[1]) {
#ifdef WITH_PINENTRY
	    gboolean b = get_key_file_boolean(filename, "enable_pinentry");

	    /* From set_pinentry_defaults(). */
	    if (client->opts & OPT_INQUIRE ||
		    client->pinentry->enable == FALSE ||
		    (client->pinentry->enable == -1 && b == FALSE)) {
		gcry_md_hash_buffer(GCRY_MD_SHA256, client->crypto->key, "", 1);
		goto done;
	    }

	    pth_cleanup_pop(1);
	    rc = lock_pin_mutex(client);

	    if (rc) {
		unlock_pin_mutex(client->pinentry);
		cleanup_client(client);
		return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	    }

	    client->pinentry->which = PINENTRY_OPEN;
	    rc = pinentry_fork(ctx);

	    if (rc) {
		unlock_pin_mutex(client->pinentry);
		cleanup_client(client);
		return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	    }

	    // Called from pinentry iterate.
	    client->pinentry->cb = open_command_finalize;
	    client->pinentry->status = PINENTRY_INIT;
	    return 0;
#else
	    gcry_md_hash_buffer(GCRY_MD_SHA256, client->crypto->key, "", 1);
	    goto done;
#endif
	}

	rc = hash_key(client, req[1]);

	if (rc) {
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	}
    }
    else if (req && req[1] && *req[1]) {
	rc = hash_key(client, req[1]);

	if (rc) {
	    pth_cleanup_pop(1);
	    cleanup_client(client);
	    return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	}
    }

done:
    pth_cleanup_pop(1);
    return open_command_finalize(ctx, NULL, cached);
}

static gint open_command_inquire_finalize(gpointer data, gint assuan_rc,
	guchar *line, gsize len)
{
    assuan_context_t ctx = data;
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc = file_modified(client);

    if (assuan_rc || (rc && rc != EPWMD_NO_FILE)) {
	if (line)
	    xfree(line);

	return assuan_rc ? assuan_rc : rc;
    }

    rc = open_command_common(ctx, (gchar *)line);

    if (line)
	xfree(line);

    client->inquire_status = INQUIRE_DONE;
    return rc;
}

static gint open_command(assuan_context_t ctx, gchar *line)
{
    gpg_error_t rc;
    struct client_s *client = assuan_get_pointer(ctx);
    struct argv_s *args[] = {
	&(struct argv_s) { "lock", OPTION_TYPE_NOARG, parse_open_opt_lock },
	&(struct argv_s) { "pinentry", OPTION_TYPE_OPTARG, parse_opt_pinentry },
	&(struct argv_s) { "inquire", OPTION_TYPE_NOARG, parse_opt_inquire },
	&(struct argv_s) { "base64", OPTION_TYPE_NOARG, parse_opt_base64 },
	NULL
    };

    if (client->state == STATE_OPEN)
	cleanup_client(client);

    if (!(client->opts & OPT_LOCK))
	client->lock_on_open = FALSE;

    rc = parse_options(&line, args, client);

    if (rc)
	return send_error(ctx, rc);

    if ((client->opts & OPT_INQUIRE)) {
	rc = assuan_inquire_ext(ctx, "OPEN", 0, open_command_inquire_finalize,
		ctx);

	if (rc)
	    return send_error(ctx, rc);

	/* Don't return with assuan_process_done() here. This is an INQUIRE. */
	client->inquire_status = INQUIRE_BUSY;
	return 0;
    }

    return open_command_common(ctx, line);
}

gpg_error_t do_compress(assuan_context_t ctx, gint level, gpointer data,
	guint size, gpointer *out, gulong *outsize)
{
    struct gz_s *gz;
    gz_header h;
    gchar buf[17];
    gint cmd = Z_NO_FLUSH;
    gint zrc;
    gpg_error_t rc;

    gz = g_malloc0(sizeof(struct gz_s));

    if (!gz)
	return GPG_ERR_ENOMEM;

    pth_cleanup_push(gz_cleanup, &gz);
    gz->which = STATUS_COMPRESS;
    gz->z.zalloc = z_alloc;
    gz->z.zfree = z_free;
    gz->z.next_in = data;
    gz->z.avail_in = size < zlib_bufsize ? (uInt)size : (uInt)zlib_bufsize;
    gz->z.avail_out = (uInt)zlib_bufsize;
    gz->z.next_out = gz->out = gcry_malloc(zlib_bufsize);

    if (!gz->out) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	pth_cleanup_pop(1);
	return GPG_ERR_ENOMEM;
    }

    zrc = deflateInit2(&gz->z, level, Z_DEFLATED, 31, 8, Z_DEFAULT_STRATEGY);

    if (zrc != Z_OK) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
	pth_cleanup_pop(1);
	return GPG_ERR_COMPR_ALGO;
    }

    /* Rather than store the size of the uncompressed data in the file header,
     * store it in the comment field of the gzip header. Don't give anyone too
     * much information. Not sure why really, but it seems the right way. :)
     */
    memset(&h, 0, sizeof(gz_header));
    g_snprintf(buf, sizeof(buf), "%u", size);
    h.comment = (guchar *)buf;
    zrc = deflateSetHeader(&gz->z, &h);

    if (zrc != Z_OK) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
	pth_cleanup_pop(1);
	return GPG_ERR_COMPR_ALGO;
    }

    do {
	gpointer p;

	zrc = deflate(&gz->z, cmd);

	switch (zrc) {
	    case Z_OK:
		break;
	    case Z_BUF_ERROR:
		if (!gz->z.avail_out) {
		    p = gcry_realloc(gz->out, gz->z.total_out + zlib_bufsize);

		    if (!p) {
			log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
			rc = GPG_ERR_ENOMEM;
			goto fail;
		    }

		    gz->out = p;
		    gz->z.next_out = (guchar *)gz->out + gz->z.total_out;
		    gz->z.avail_out = zlib_bufsize;
		}

		if (!gz->z.avail_in && gz->z.total_in < size) {
		    if (gz->z.total_in + zlib_bufsize > size)
			gz->z.avail_in = size - gz->z.total_in;
		    else
			gz->z.avail_in = zlib_bufsize;

		    rc = send_status(ctx, STATUS_COMPRESS, "%li %u",
			    gz->z.total_in, size);

		    if (rc)
			goto fail;
		}

		if (gz->z.total_in >= size)
		    cmd = Z_FINISH;

		break;
	    case Z_STREAM_END:
		break;
	    default:
		log_write("%s(%i): %s", __FUNCTION__, __LINE__, gz->z.msg);
		rc = GPG_ERR_COMPR_ALGO;
		goto fail;
	}
    } while (zrc != Z_STREAM_END);

    rc = send_status(ctx, STATUS_COMPRESS, "%li %u", gz->z.total_in, size);

    if (rc)
	goto fail;

    *out = gz->out;
    *outsize = gz->z.total_out;
    gz->done = TRUE;
    pth_cleanup_pop(1);
    return 0;

fail:
    pth_cleanup_pop(1);
    return rc;
}

#define CRYPTO_BLOCKSIZE(c) (c->blocksize * 1024)

/*
 * Useful for a large amount of data. Rather than doing all of the data in one
 * iteration do it in chunks.  This lets the command be cancelable rather than
 * waiting for it to complete.
 */
static gpg_error_t iterate_crypto_once(struct client_s *client,
	struct crypto_s *crypto, status_msg_t which)
{
    gpg_error_t rc = 0;
    goffset len = CRYPTO_BLOCKSIZE(crypto);
    gpointer p = gcry_malloc(len);
    goffset total = 0;
    gpointer inbuf;

    if (!p)
	return GPG_ERR_ENOMEM;

    if (crypto->insize < CRYPTO_BLOCKSIZE(crypto))
	len = crypto->insize;

    pth_cleanup_push(gcry_free, p);

    for (;;) {
	inbuf = (guchar *)crypto->inbuf + total;
	guchar *tmp;

	if (len + total > crypto->insize)
	    len = crypto->blocksize;

	if (which == STATUS_ENCRYPT)
	    rc = gcry_cipher_encrypt(crypto->gh, p, len, inbuf, len);
	else
	    rc = gcry_cipher_decrypt(crypto->gh, p, len, inbuf, len);
	
	if (rc)
	    goto done;

	tmp = (guchar *)crypto->inbuf + total;
	memmove(tmp, p, len);
	total += len;

	if (total >= crypto->insize)
	    break;

	pth_cancel_point();
    }

done:
    pth_cleanup_pop(1);
    return rc;
}

/* The crypto struct must be setup for iterations and .key. */
gpg_error_t do_xml_encrypt(struct client_s *client,
	struct crypto_s *crypto, const gchar *filename)
{
    goffset len = crypto->insize;
    gpointer inbuf;
    gchar *p;
    gpg_error_t rc;
    guint64 iter_progress = 0ULL, n_iter = 0ULL, xiter = 0ULL;
    gchar tmp[FILENAME_MAX];
    struct stat st;
    mode_t mode = 0;
    gsize hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    if (!crypto->fh->ver.fh2.iter) {
	/*
	 * cache_file_count() needs both .used == TRUE and a valid key in
	 * order for it to count as a used cache entry. Fixes CACHE status
	 * messages.
	 */
	memset(crypto->key, '!', hashlen);
	goto write_file;
    }

    /*
     * Resize the existing xml buffer to the block size required by gcrypt
     * rather than duplicating it and wasting memory.
     */
    crypto->insize += sizeof(crypto_magic);
    len = (crypto->insize / crypto->blocksize) * crypto->blocksize;

    if (crypto->insize % crypto->blocksize)
	len += crypto->blocksize;

    inbuf = gcry_realloc(crypto->inbuf, len);

    if (!inbuf) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    guchar *tmpbuf = inbuf;
    memmove(&tmpbuf[sizeof(crypto_magic)], tmpbuf, len-sizeof(crypto_magic));
    memcpy(tmpbuf, crypto_magic, sizeof(crypto_magic));
    crypto->inbuf = tmpbuf;
    crypto->insize = len;
    gcry_create_nonce(crypto->fh->ver.fh2.iv, crypto->blocksize);

    if (crypto->tkey)
	gcry_free(crypto->tkey);

    crypto->tkey = gcry_malloc(hashlen);

    if (!crypto->tkey) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    memcpy(crypto->tkey, crypto->key, hashlen);
    guchar *tkey = crypto->tkey;
    tkey[0] ^= 1;

    if ((rc = gcry_cipher_setkey(crypto->gh, crypto->tkey, crypto->keysize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    iter_progress = (guint64)get_key_file_double(
	    client ? client->filename : "global", "iteration_progress");

    if (iter_progress && crypto->fh->ver.fh2.iter >= iter_progress) {
	rc = send_status(client ? client->ctx : NULL, STATUS_ENCRYPT,
		"0 %llu", crypto->fh->ver.fh2.iter);

	if (rc)
	    return rc;
    }

    while (xiter < crypto->fh->ver.fh2.iter-1) {
	if (iter_progress > 0ULL && xiter >= iter_progress) {
	    if (!(xiter % iter_progress)) {
		rc = send_status(client ? client->ctx : NULL, STATUS_ENCRYPT,
			"%llu %llu", ++n_iter * iter_progress,
			crypto->fh->ver.fh2.iter);

		if (rc)
		    return rc;
	    }
	}

	if ((rc = gcry_cipher_setiv(crypto->gh, crypto->fh->ver.fh2.iv,
			crypto->blocksize))) {
	    log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	    return rc;
	}

	rc = iterate_crypto_once(client, crypto, STATUS_ENCRYPT);

	if (rc) {
	    log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	    return rc;
	}

	xiter++;
    }

    if ((rc = gcry_cipher_setiv(crypto->gh, crypto->fh->ver.fh2.iv,
		crypto->blocksize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    if ((rc = gcry_cipher_setkey(crypto->gh, crypto->key, crypto->keysize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    rc = iterate_crypto_once(client, crypto, STATUS_ENCRYPT);

    if (rc)
	return rc;

    if (iter_progress && crypto->fh->ver.fh2.iter >= iter_progress) {
	rc = send_status(client ? client->ctx : NULL, STATUS_ENCRYPT,
		"%llu %llu", crypto->fh->ver.fh2.iter, crypto->fh->ver.fh2.iter);

	if (rc)
	    return rc;
    }

write_file:
    tmp[0] = 0;

    if (filename) {
	if (!client && !g_ascii_strcasecmp(filename, "-")) {
	    crypto->fh->fd = STDOUT_FILENO;
	    goto do_write_file;
	}

	if (lstat(filename, &st) == 0) {
	    mode = st.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO);

	    if (!(mode & S_IWUSR))
		return GPG_ERR_EACCES;
	}
	else if (errno != ENOENT)
	    return gpg_error_from_syserror();

	g_snprintf(tmp, sizeof(tmp), "%s.XXXXXX", filename);
	crypto->fh->fd = mkstemp(tmp);

	if (crypto->fh->fd == -1) {
	    rc = gpg_error_from_syserror();
	    p = strrchr(tmp, '/');
	    p++;
	    log_write("%s: %s", p, pwmd_strerror(rc));
	    return rc;
	}

	pth_cleanup_push(cleanup_unlink_cb, tmp);
    }
    else
	/*
	 * xml_import() or convert_file() from command line.
	 */
	crypto->fh->fd = STDOUT_FILENO;

do_write_file:
    crypto->fh->ver.fh2.magic[0] = '\177';
    crypto->fh->ver.fh2.magic[1] = 'P';
    crypto->fh->ver.fh2.magic[2] = 'W';
    crypto->fh->ver.fh2.magic[3] = 'M';
    crypto->fh->ver.fh2.magic[4] = 'D';
    crypto->fh->ver.fh2.version = VERSION_HEX;
    len = pth_write(crypto->fh->fd, &crypto->fh->ver.fh2, sizeof(crypto->fh->ver.fh2));

    if (len != sizeof(crypto->fh->ver.fh2)) {
	rc = gpg_error_from_syserror();

	if (tmp[0])
	    pth_cleanup_pop(1);

	return rc;
    }

    len = pth_write(crypto->fh->fd, crypto->inbuf, crypto->insize);

    if (len != crypto->insize) {
	rc = gpg_error_from_syserror();

	if (tmp[0])
	    pth_cleanup_pop(1);

	return rc;
    }

    if (fsync(crypto->fh->fd) == -1) {
	rc = gpg_error_from_syserror();

	if (tmp[0])
	    pth_cleanup_pop(1);

	return rc;
    }

    if (tmp[0]) {
#ifdef WITH_LIBACL
	acl_t acl;
#endif
	if (mode && get_key_file_boolean(filename, "backup") == TRUE) {
	    gchar tmp2[FILENAME_MAX];

	    g_snprintf(tmp2, sizeof(tmp2), "%s.backup", filename);
#ifdef WITH_LIBACL
	    acl = acl_get_file(filename, ACL_TYPE_ACCESS);

	    if (!acl)
		log_write("ACL: %s: %s", filename, pwmd_strerror(gpg_error_from_syserror()));
#endif

	    if (rename(filename, tmp2) == -1) {
		rc = gpg_error_from_syserror();
		pth_cleanup_pop(1);
#ifdef WITH_LIBACL
		if (acl)
		    acl_free(acl);
#endif
		return rc;
	    }
	}
#ifdef WITH_LIBACL
	else {
	    acl = acl_get_file(".", ACL_TYPE_DEFAULT);

	    if (!acl)
		log_write("ACL: %s: %s", filename, pwmd_strerror(gpg_error_from_syserror()));
	}
#endif

	if (rename(tmp, filename) == -1) {
	    rc = gpg_error_from_syserror();
	    pth_cleanup_pop(1);
#ifdef WITH_LIBACL
	    if (acl)
		acl_free(acl);
#endif
	    return rc;
	}

	pth_cleanup_pop(0);

	if (mode)
	    chmod(filename, mode);

#ifdef WITH_LIBACL
	if (acl && acl_set_file(filename, ACL_TYPE_ACCESS, acl))
	    log_write("ACL: %s: %s", filename, pwmd_strerror(gpg_error_from_syserror()));

	if (acl)
	    acl_free(acl);
#endif
    }

    if (client && lstat(filename, &st) == 0)
	client->mtime = st.st_mtime;

    return 0;
}

gpg_error_t update_save_flags(const gchar *filename,
	struct crypto_s *crypto)
{
    gpg_error_t rc;
    guint64 iter;

    /* New file? */
    if (!crypto->fh) {
	crypto->fh = g_malloc0(sizeof(file_header_internal_t));
	
	if (!crypto->fh)
	    return GPG_ERR_ENOMEM;
    }

    rc = init_client_crypto2(filename, crypto);

    if (rc)
	return rc;

    if (filename && !crypto->fh->v1) {
	iter = (guint64)get_key_file_double(filename, "iterations");
	crypto->fh->ver.fh2.iter = iter < 0L ? 0UL : iter;
    }

    return 0;
}

static gpg_error_t save_command_finalize(assuan_context_t ctx, guchar *key,
	gboolean cached)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpointer xmlbuf;
    gulong outsize = 0;
    guint len;
    gint clevel;
    gint timeout;
    gpointer outbuf;
    gpg_error_t rc;

    if (client->crypto->key && client->crypto->key != key)
	gcry_free(client->crypto->key);

    client->crypto->key = key;
    rc = update_element_mtime(xmlDocGetRootElement(client->doc));

    if (rc) {
	cleanup_crypto(&client->crypto);
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

    xmlDocDumpFormatMemory(client->doc, (xmlChar **)&xmlbuf, (gint *)&len, 0);
    pth_cleanup_push(xmlFree, xmlbuf);
    clevel = get_key_file_integer(client->filename, "compression_level");

    if (clevel < 0)
	clevel = 0;

    rc = do_compress(ctx, clevel, xmlbuf, len, &outbuf, &outsize);

    if (rc) {
	pth_cleanup_pop(1);
	cleanup_crypto(&client->crypto);

	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }
    else {
	pth_cleanup_pop(1);
	xmlbuf = outbuf;
	len = outsize;
    }

    client->crypto->inbuf = xmlbuf;
    client->crypto->insize = len;
    rc = update_save_flags(client->filename, client->crypto);

    if (rc) {
	cleanup_crypto(&client->crypto);
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(rc));
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

    rc = do_xml_encrypt(client, client->crypto, client->filename);

    if (rc) {
	cleanup_crypto(&client->crypto);
	return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
    }

    timeout = get_key_file_integer(client->filename, "cache_timeout");
    CACHE_LOCK(client->ctx);

    if (cached) {
	cache_reset_timeout(client->md5file, timeout);
	CACHE_UNLOCK;

	if (client->new == TRUE)
	    send_status_all(STATUS_CACHE);

	client->new = FALSE;
	cleanup_crypto(&client->crypto);
	return client->opts & OPT_INQUIRE ? 0 : send_error(ctx, 0);
    }

    if (cache_update_key(client->md5file, client->crypto->key) == FALSE) {
	CACHE_UNLOCK;
	cleanup_crypto(&client->crypto);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }

    client->new = FALSE;
    cache_reset_timeout(client->md5file, timeout);
    CACHE_UNLOCK;
    send_status_all(STATUS_CACHE);
    cleanup_crypto(&client->crypto);
    return client->opts & OPT_INQUIRE ? 0 : send_error(ctx, 0);
}

static gpg_error_t parse_save_opt_iterations(gpointer data, gpointer v)
{
    struct client_s *client = data;
    guint64 n;
    gchar *value = v;
    gchar *p = NULL;

    if (!client->filename)
	return EPWMD_NO_FILE;

    if (!value || !*value)
	return 0;

    errno = 0;
    n = strtoul(value, &p, 10);

    if (errno || (p && *p) || n == G_MAXULONG)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_ERANGE);

    MUTEX_LOCK(&rcfile_mutex);
    g_key_file_set_double(keyfileh,
	    client->filename ? client->filename : "global", "iterations", n);
    MUTEX_UNLOCK(&rcfile_mutex);

    if (client->filename)
	client->opts |= OPT_ITERATIONS;

    return 0;
}

static gpg_error_t parse_save_opt_cipher(gpointer data, gpointer value)
{
    struct client_s *client = data;
    const gchar *p = value;
    guint64 flags;

    if (!client->filename)
	return EPWMD_NO_FILE;

    if (!p || !*p)
	return 0;

    flags = pwmd_cipher_str_to_cipher(p);

    if (!flags)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);

    MUTEX_LOCK(&rcfile_mutex);
    g_key_file_set_string(keyfileh, client->filename, "cipher", p);
    MUTEX_UNLOCK(&rcfile_mutex);

    if (!value)
	g_free((gchar *)p);

    return 0;
}

static gpg_error_t parse_save_opt_reset(gpointer data, gpointer value)
{
    struct client_s *client = data;

    CACHE_LOCK(client->ctx);
    cache_clear(client->md5file, 1);
    CACHE_UNLOCK;
    return 0;
}

static gpg_error_t save_command_common(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gboolean cached = FALSE;
    guint hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    CACHE_LOCK(ctx);
    cached = cache_iscached(client->md5file);
    CACHE_UNLOCK;

    /*
     * If a cache entry doesn't exist for this file and the file has a
     * "key_file" or "key" parameter, then it's an error. The reason is that
     * cache expiration would be useless. Unless this is an inquire, then its
     * fine.
     */
    if (cached == FALSE) {
	gchar *tmp = get_key_file_string(client->filename, "key_file");

	if (tmp && !(client->opts & OPT_INQUIRE)) {
	    g_free(tmp);
	    return send_error(ctx, GPG_ERR_WRONG_KEY_USAGE);
	}

	if (tmp)
	    g_free(tmp);
    }

    cached = FALSE;

    /* New file? */
    if (!client->crypto) {
	client->crypto = init_client_crypto();

	if (!client->crypto) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
	}
    }

    client->crypto->key = gcry_malloc(hashlen);

    if (!client->crypto->key) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	cleanup_crypto(&client->crypto);
	return client->opts & OPT_INQUIRE ? GPG_ERR_ENOMEM : send_error(ctx, GPG_ERR_ENOMEM);
    }

    memset(client->crypto->key, '!', hashlen);

    if (get_key_file_double(client->filename, "iterations") <= 0L &&
	    (!line || !*line))
	goto done;

    if (!line || !*line) {
	/* It doesn't make sense to use an --inquire with an empty
	 * passphrase. This will prevent a pinentry dialog. */
	if (client->opts & OPT_INQUIRE) {
	    cleanup_crypto(&client->crypto);
	    return GPG_ERR_WRONG_KEY_USAGE;
	}

	client->crypto->tkey = gcry_malloc(hashlen);

	if (!client->crypto->tkey) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    cleanup_crypto(&client->crypto);
	    return send_error(ctx, GPG_ERR_ENOMEM);
	}

	memset(client->crypto->tkey, '!', hashlen);
	CACHE_LOCK(ctx);

	if (cache_get_key(client->md5file, client->crypto->key) == FALSE ||
		!memcmp(client->crypto->key, client->crypto->tkey, hashlen)) {
	    CACHE_UNLOCK;

#ifdef WITH_PINENTRY
	    gpg_error_t rc;

	    if (client->pinentry->enable == FALSE ||
		    get_key_file_boolean(client->filename, "enable_pinentry") == FALSE) {
		/* Empty keys are allowed. */
		gcry_md_hash_buffer(GCRY_MD_SHA256, client->crypto->key, "", 1);
		goto done;
	    }

	    lock_pin_mutex(client);
	    client->pinentry->which = PINENTRY_SAVE;
	    rc = pinentry_fork(ctx);

	    if (rc) {
		unlock_pin_mutex(client->pinentry);
		cleanup_crypto(&client->crypto);
		return send_error(ctx, rc);
	    }

	    client->pinentry->cb = save_command_finalize;
	    client->pinentry->status = PINENTRY_INIT;
	    return 0;
#else
	    /* Empty keys are allowed. */
	    gcry_md_hash_buffer(GCRY_MD_SHA256, client->crypto->key, "", 1);
	    goto done;
#endif
	}
	else {
	    CACHE_UNLOCK;
	    cached = TRUE;
	}
    }
    else {
	gpg_error_t rc;

	if (get_key_file_double(client->filename, "iterations") <= 0L) {
	    guint64 iter = (guint64)get_key_file_double(NULL, "iterations");

	    if (iter <= 0L)
		iter = 1;

	    MUTEX_LOCK(&rcfile_mutex);
	    g_key_file_set_double(keyfileh, client->filename, "iterations", iter);
	    MUTEX_UNLOCK(&rcfile_mutex);
	    client->opts |= OPT_ITERATIONS;
	    rc = send_status(ctx, STATUS_CONFIG, NULL);

	    if (rc) {
		cleanup_crypto(&client->crypto);
		return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	    }
	}

	rc = hash_key(client, line);

	if (rc) {
	    cleanup_crypto(&client->crypto);
	    return client->opts & OPT_INQUIRE ? rc : send_error(ctx, rc);
	}
    }

done:
    return save_command_finalize(ctx, client->crypto->key, cached);
}

static gint save_command_inquire_finalize(gpointer data, gint assuan_rc,
	guchar *line, gsize len)
{
    assuan_context_t ctx = data;
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc = file_modified(client);

    if (assuan_rc || rc) {
	if (line)
	    xfree(line);

	return assuan_rc ? assuan_rc : rc;
    }

    rc = save_command_common(ctx, (gchar *)line);

    if (line)
	xfree(line);

    client->inquire_status = INQUIRE_DONE;
    return rc;
}

static gint save_command(assuan_context_t ctx, gchar *line)
{
    struct stat st;
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    struct argv_s *args[] = {
	&(struct argv_s) { "iterations", OPTION_TYPE_OPTARG, parse_save_opt_iterations },
	&(struct argv_s) { "cipher", OPTION_TYPE_OPTARG, parse_save_opt_cipher },
	&(struct argv_s) { "pinentry", OPTION_TYPE_OPTARG, parse_opt_pinentry },
	&(struct argv_s) { "reset", OPTION_TYPE_NOARG, parse_save_opt_reset },
	&(struct argv_s) { "inquire", OPTION_TYPE_NOARG, parse_opt_inquire },
	&(struct argv_s) { "base64", OPTION_TYPE_NOARG, parse_opt_base64 },
	NULL
    };

    rc = parse_options(&line, args, client);

    if (rc)
	return send_error(ctx, rc);

    if (lstat(client->filename, &st) == -1 && errno != ENOENT)
	return send_error(ctx, gpg_error_from_syserror());

    if (errno != ENOENT && !S_ISREG(st.st_mode)) {
	log_write("%s: %s", client->filename, pwmd_strerror(GPG_ERR_ENOANO));
	return send_error(ctx, GPG_ERR_ENOANO);
    }

    if ((client->opts & OPT_INQUIRE)) {
	rc = assuan_inquire_ext(ctx, "SAVE", 0, save_command_inquire_finalize,
		ctx);

	if (rc)
	    return send_error(ctx, rc);

	/* Don't return with assuan_process_done() here. This is an INQUIRE. */
	client->inquire_status = INQUIRE_BUSY;
	return 0;
    }

    if (line && *line)
	log_write2("ARGS=%s", "<passphrase>");

    return save_command_common(ctx, line);
}

static gint delete_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    gpg_error_t rc;
    xmlNodePtr n;

    log_write2("ARGS=\"%s\"", line);

    if (strchr(line, '\t'))
	req = split_input_line(line, "\t", -1);
    else
	req = split_input_line(line, " ", -1);

    if (!req || !*req)
	return send_error(ctx, GPG_ERR_SYNTAX);

    n = find_root_element(client->doc, &req, &rc, NULL, 0, FALSE);

    if (!n) {
	g_strfreev(req);
	return send_error(ctx, rc);
    }

    /*
     * No sub-node defined. Remove the entire node (root element).
     */
    if (!req[1]) {
	if (n) {
	    rc = unlink_node(n);
	    xmlFreeNode(n);
	}

	g_strfreev(req);
	return send_error(ctx, rc);
    }

    n = find_elements(client->doc, n->children, req+1, &rc, NULL, NULL, NULL, FALSE, 0, NULL, FALSE);
    g_strfreev(req);

    if (!n)
	return send_error(ctx, rc);

    if (n) {
	rc = unlink_node(n);
	xmlFreeNode(n);
    }

    return send_error(ctx, rc);
}

/*
 * Don't return with assuan_process_done() here. This has been called from
 * assuan_process_next() and the command should be finished in
 * client_thread().
 */
static gint store_command_finalize(gpointer data, gint assuan_rc, guchar *line,
	gsize len)
{
    assuan_context_t ctx = data;
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    xmlNodePtr n;
    gpg_error_t rc = file_modified(client);

    if (assuan_rc || rc) {
	if (line)
	    xfree(line);
	return assuan_rc ? assuan_rc : rc;
    }

    req = split_input_line((gchar *)line, "\t", 0);
    xfree(line);

    if (!req || !*req)
	return GPG_ERR_SYNTAX;

again:
    n = find_root_element(client->doc, &req, &rc, NULL, 0, FALSE);

    if (rc && rc == GPG_ERR_ELEMENT_NOT_FOUND) {
	rc = new_root_element(client->doc, *req);

	if (rc) {
	    g_strfreev(req);
	    return rc;
	}

	goto again;
    }

    if (!n) {
	g_strfreev(req);
	return rc;
    }

    if (req[1]) {
	if (!n->children)
	    n = create_elements_cb(n, req+1, &rc, NULL);
	else
	    n = find_elements(client->doc, n->children, req+1, &rc,
		    NULL, NULL, create_elements_cb, FALSE, 0, NULL, FALSE);
    }

    g_strfreev(req);
    client->inquire_status = INQUIRE_DONE;

    if (!rc)
	rc = update_element_mtime(n);

    return rc;
}

static gint store_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;

    rc = assuan_inquire_ext(ctx, "STORE", 0, store_command_finalize, ctx);

    if (rc)
	return send_error(ctx, rc);

    /* Don't return with assuan_process_done() here. This is an INQUIRE. */
    client->inquire_status = INQUIRE_BUSY;
    return 0;
}

static void *send_data_cb(void *arg)
{
    struct assuan_cmd_s *data = arg;
    gint old;
    gpg_error_t *rc;

    pth_cancel_state(PTH_CANCEL_ENABLE|PTH_CANCEL_ASYNCHRONOUS, &old);
    rc = g_malloc(sizeof(gpg_error_t));
    *rc = assuan_send_data(data->ctx, data->line, data->line_len);
    pth_cancel_state(old, NULL);
    pth_exit(rc);
    return NULL;
}

/* For every assuan command that needs to be sent to the client, a timeout is
 * needed to determine if the client lost the connection. The timeout is the
 * same as the "keepalive" configuration parameter or a default if unset.
 */
gpg_error_t do_assuan_command(assuan_context_t ctx,
	void *(*cb)(void *data), void *data)
{
    pth_attr_t attr = pth_attr_new();
    pth_t tid;
    gint to = get_key_file_integer("global", "keepalive");
    pth_event_t ev, tev;
    pth_status_t st;
    gpg_error_t rc = 0;
    void *p;

    pth_attr_init(attr);
    pth_attr_set(attr, PTH_ATTR_JOINABLE, TRUE);
    tid = pth_spawn(attr, cb, data);
    rc = gpg_error_from_syserror();
    pth_attr_destroy(attr);

    if (!tid) {
	log_write("%s(%i): pth_spawn(): %s", __FILE__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    pth_cleanup_push(cleanup_cancel_cb, tid);
    to = to <= 0 ? DEFAULT_KEEPALIVE_TO : to;
    ev = pth_event(PTH_EVENT_TID|PTH_UNTIL_TID_DEAD, tid);
    tev = to ? pth_event(PTH_EVENT_TIME, pth_timeout(to, 0)) : NULL;
    ev = pth_event_concat(ev, tev, NULL);
    pth_cleanup_push(cleanup_ev_cb, ev);
    pth_yield(tid);
    pth_wait(ev);

    if (tev) {
	st = pth_event_status(tev);

	if (st == PTH_STATUS_OCCURRED) {
	    pth_cleanup_pop(1);
	    pth_cleanup_pop(1);
	    return GPG_ERR_TIMEOUT;
	}
    }

    st = pth_event_status(ev);

    if (st == PTH_STATUS_FAILED) {
	pth_cancel(tid);
	pth_join(tid, &p);
	g_free(p);
	rc = GPG_ERR_ASS_WRITE_ERROR;
    }
    else if (st == PTH_STATUS_OCCURRED) {
	pth_join(tid, &p);
	rc = *(gpg_error_t *)p;
	g_free(p);
    }

    pth_cleanup_pop(1);
    pth_cleanup_pop(0);
    return rc;
}

static gpg_error_t xfer_data(assuan_context_t ctx, const gchar *line,
	gint total)
{
    gint to_send;
    gint sent = 0;
    gpg_error_t rc;
    struct assuan_cmd_s data;
    gint progress = get_key_file_integer("global", "xfer_progress");
    gint flush = 0;

    progress = progress>0 ? (progress/ASSUAN_LINELENGTH)*ASSUAN_LINELENGTH : 0;
    to_send = total < ASSUAN_LINELENGTH ? total : ASSUAN_LINELENGTH;
    data.ctx = ctx;
    rc = send_status(ctx, STATUS_XFER, "%li %li", sent, total);

    if (rc)
	return rc;

again:
    do {
	if (sent + to_send > total)
	    to_send = total - sent;

	data.line = flush ? NULL : (gchar *)line+sent;
	data.line_len = flush ? 0 : to_send;
	rc = do_assuan_command(ctx, send_data_cb, &data);

	if (!rc) {
	    sent += flush ? 0 : to_send;

	    if ((progress && !(sent % progress) && sent != total) ||
		    (sent == total && flush))
		rc = send_status(ctx, STATUS_XFER, "%li %li", sent, total);

	    if (!flush && !rc && sent == total) {
		flush = 1;
		goto again;
	    }
	}
    } while (!rc && sent < total);

    return rc;
}

static gint get_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    gpg_error_t rc;
    xmlNodePtr n;

    log_write2("ARGS=\"%s\"", line);
    req = split_input_line(line, "\t", -1);

    if (!req || !*req) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    n = find_root_element(client->doc, &req, &rc, NULL, 0, FALSE);

    if (!n) {
	g_strfreev(req);
	return send_error(ctx, rc);
    }

    if (req[1])
	n = find_elements(client->doc, n->children, req+1, &rc, NULL, NULL, NULL, FALSE, 0, NULL, FALSE);

    g_strfreev(req);

    if (rc)
	return send_error(ctx, rc);

    if (!n || !n->children)
	return send_error(ctx, GPG_ERR_NO_VALUE);

    n = find_text_node(n->children);

    if (!n || !n->content || !*n->content)
	return send_error(ctx, GPG_ERR_NO_VALUE);

    rc = xfer_data(ctx, (gchar *)n->content, xmlStrlen(n->content));
    return send_error(ctx, rc);
}

static xmlNodePtr realpath_elements_cb(xmlNodePtr node, gchar **target,
	gpg_error_t *rc, gchar **req_orig, void *data)
{
    gchar *path = *(gchar **)data;
    gchar *tmp = NULL, *result;

    if (path) {
	g_free(path);
	*(gchar **)data = NULL;
    }

    path = g_strjoinv("\t", target);

    if (!path) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	*rc = GPG_ERR_ENOMEM;
	return NULL;
    }

    if (req_orig) {
	tmp = g_strjoinv("\t", req_orig);

	if (!tmp) {
	    g_free(path);
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    *rc = GPG_ERR_ENOMEM;
	    return NULL;
	}
    }

    if (tmp && *tmp)
	result = g_strdup_printf("%s\t%s", path, tmp);
    else
	result = g_strdup(path);

    if (!result) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	*rc = GPG_ERR_ENOMEM;
	g_free(path);
	g_free(tmp);
	return NULL;
    }

    g_free(path);
    g_free(tmp);
    *(gchar **)data = result;
    return node;
}

static void list_command_cleanup1(void *arg);
static gint realpath_command(assuan_context_t ctx, gchar *line)
{
    gpg_error_t rc;
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    gchar *t;
    gint i;
    xmlNodePtr n;
    GString *string;
    gchar *rp = NULL;

    log_write2("ARGS=\"%s\"", line);

    if (strchr(line, '\t') != NULL) {
	if ((req = split_input_line(line, "\t", 0)) == NULL)
	    return send_error(ctx, GPG_ERR_SYNTAX);
    }
    else {
	if ((req = split_input_line(line, " ", 0)) == NULL)
	    return send_error(ctx, GPG_ERR_SYNTAX);
    }

    n = find_root_element(client->doc, &req, &rc, NULL, 0, FALSE);

    if (!n) {
	g_strfreev(req);
	return send_error(ctx, rc);
    }

    rp = g_strjoinv("\t", req);

    if (!rp) {
	g_strfreev(req);
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return send_error(ctx, GPG_ERR_ENOMEM);
    }

    if (req[1]) {
	n = find_elements(client->doc, n->children, req+1, &rc,
		NULL, realpath_elements_cb, NULL, FALSE, 0, &rp, FALSE);

	if (!n) {
	    g_free(rp);
	    g_strfreev(req);
	    return send_error(ctx, rc);
	}
    }

    string = g_string_new(rp);
    g_free(rp);
    g_strfreev(req);

    if (!string) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return send_error(ctx, GPG_ERR_ENOMEM);
    }

again:
    for (i = 0, t = string->str + i; *t; t++, i++) {
	if ((!i && *t != '!') || (*t == '\t' && *(t+1) && *(t+1) != '!')) {
	    string = g_string_insert_c(string, !i ? i++ : ++i, '!');
	    goto again;
	}
    }

    pth_cleanup_push(list_command_cleanup1, string);
    rc = xfer_data(ctx, string->str, string->len);
    pth_cleanup_pop(1);
    return send_error(ctx, rc);
}

static void list_command_cleanup1(void *arg)
{
    g_string_free((GString *)arg, TRUE);
}

static void list_command_cleanup2(void *arg)
{
    struct element_list_s *elements = arg;

    if (elements) {
	if (elements->list) {
	    gint total = g_slist_length(elements->list);
	    gint i;

	    for (i = 0; i < total; i++) {
		gchar *tmp = g_slist_nth_data(elements->list, i);
		g_free(tmp);
	    }

	    g_slist_free(elements->list);
	}

	if (elements->prefix)
	    g_free(elements->prefix);

	if (elements->req)
	    g_strfreev(elements->req);

	g_free(elements);
    }
}

static gpg_error_t parse_list_opt_norecurse(gpointer data, gpointer value)
{
    struct element_list_s *elements = data;

    elements->recurse = FALSE;
    return 0;
}

static gpg_error_t parse_list_opt_verbose(gpointer data, gpointer value)
{
    struct element_list_s *elements = data;

    elements->verbose = TRUE;
    return 0;
}

static gint list_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    struct element_list_s *elements = NULL;
    gchar *tmp;
    struct argv_s *args[] = {
	&(struct argv_s) { "no-recurse", OPTION_TYPE_NOARG, parse_list_opt_norecurse },
	&(struct argv_s) { "verbose", OPTION_TYPE_NOARG, parse_list_opt_verbose },
	NULL
    };

    if (disable_list_and_dump == TRUE)
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);

    elements = g_malloc0(sizeof(struct element_list_s));

    if (!elements) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    elements->recurse = TRUE; // default
    pth_cleanup_push(list_command_cleanup2, elements);
    rc = parse_options(&line, args, elements);

    if (rc)
	goto fail;

    if (!*line) {
	GString *str;

	rc = list_root_elements(client->doc, &str, elements->verbose);

	if (rc) {
	    pth_cleanup_pop(1);
	    return send_error(ctx, rc);
	}

	pth_cleanup_push(list_command_cleanup1, str);
	rc = xfer_data(ctx, str->str, str->len);
	pth_cleanup_pop(1);
	pth_cleanup_pop(1);
	return send_error(ctx, rc);
    }

    elements->req = split_input_line(line, " ", 0);

    if (!elements->req)
	strv_printf(&elements->req, "%s", line);

    rc = create_path_list(client->doc, elements, *elements->req);

    if (rc)
	goto fail;

    if (elements) {
	gint total = g_slist_length(elements->list);
	gint i;
	GString *str;

	if (!total) {
	    rc = GPG_ERR_NO_VALUE;
	    goto fail;
	}

	str = g_string_new(NULL);

	if (!str) {
	    rc = GPG_ERR_ENOMEM;
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    goto fail;
	}

	for (i = 0; i < total; i++) {
	    tmp = g_slist_nth_data(elements->list, i);
	    g_string_append_printf(str, "%s%s", tmp, i+1 == total ? "" : "\n");
	}

	pth_cleanup_push(list_command_cleanup1, str);
	rc = xfer_data(ctx, str->str, str->len); 
	pth_cleanup_pop(1);
    }
    else
	rc = GPG_ERR_NO_VALUE;

fail:
    pth_cleanup_pop(1);
    return send_error(ctx, rc);
}

/*
 * req[0] - element path
 */
static gpg_error_t attribute_list(assuan_context_t ctx, gchar **req)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **attrlist = NULL;
    gint i = 0;
    gchar **path = NULL;
    xmlAttrPtr a;
    xmlNodePtr n, an;
    gchar *line;
    gpg_error_t rc;

    if (!req || !req[0])
	return GPG_ERR_SYNTAX;

    if ((path = split_input_line(req[0], "\t", 0)) == NULL) {
	/*
	 * The first argument may be only a root element.
	 */
	if ((path = split_input_line(req[0], " ", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }

    n = find_root_element(client->doc, &path, &rc, NULL, 0, FALSE);

    if (!n) {
	g_strfreev(path);
	return rc;
    }

    if (path[1]) {
	n = find_elements(client->doc, n->children, path+1, &rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);
	
	if (!n) {
	    g_strfreev(path);
	    return rc;
	}
    }

    g_strfreev(path);

    for (a = n->properties; a; a = a->next) {
	gchar **pa;

	if ((pa = g_realloc(attrlist, (i + 2) * sizeof(gchar *))) == NULL) {
	    if (attrlist)
		g_strfreev(attrlist);

	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    return GPG_ERR_ENOMEM;
	}

	attrlist = pa;
	an = a->children;
	attrlist[i] = g_strdup_printf("%s %s", (gchar *)a->name,
		an && an->content ? (gchar *)an->content : "");

	if (!attrlist[i]) {
	    g_strfreev(attrlist);
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    return GPG_ERR_ENOMEM;
	}

	attrlist[++i] = NULL;
    }

    if (!attrlist)
	return GPG_ERR_NO_VALUE;

    line = g_strjoinv("\n", attrlist);

    if (!line) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	g_strfreev(attrlist);
	return GPG_ERR_ENOMEM;
    }

    pth_cleanup_push(g_free, line);
    pth_cleanup_push(req_cleanup, attrlist);
    rc = xfer_data(ctx, line, strlen(line));
    pth_cleanup_pop(1);
    pth_cleanup_pop(1);
    return rc;
}

/*
 * req[0] - attribute
 * req[1] - element path
 */
static gpg_error_t attribute_delete(struct client_s *client, gchar **req)
{
    xmlNodePtr n;
    gchar **path = NULL;
    gpg_error_t rc;

    if (!req || !req[0] || !req[1])
	return GPG_ERR_SYNTAX;

    if ((path = split_input_line(req[1], "\t", 0)) == NULL) {
	/*
	 * The first argument may be only a root element.
	 */
	if ((path = split_input_line(req[1], " ", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }

    /*
     * Don't remove the "_name" attribute for the root element. To remove an
     * root element use DELETE <name>.
     */
    if (!path[1] && xmlStrEqual((xmlChar *)req[0], (xmlChar *)"_name")) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    n = find_root_element(client->doc, &path, &rc, NULL, 0, FALSE);

    if (!n)
	goto fail;

    if (path[1]) {
	n = find_elements(client->doc, n->children, path+1, &rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);
	
	if (!n)
	    goto fail;
    }

    rc = delete_attribute(n, (xmlChar *)req[0]);

fail:
    g_strfreev(path);
    return rc;
}

static xmlNodePtr create_element_path(struct client_s *client, gchar ***path,
	gpg_error_t *rc)
{
    gchar **src = *path;
    gchar **src_orig = g_strdupv(src);
    xmlNodePtr n = NULL;

    *rc = 0;

    if (!src_orig) {
	*rc = GPG_ERR_ENOMEM;
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

again:
    n = find_root_element(client->doc, &src, rc, NULL, 0, FALSE);

    if (!n) {
	if (*rc == GPG_ERR_ELEMENT_NOT_FOUND) {
	    *rc = new_root_element(client->doc, src[0]);

	    if (*rc)
		goto fail;

	    goto again;
	}
	else
	    goto fail;
    }

    if (src[1]) {
	if (!n->children)
	    n = create_target_elements_cb(n, src+1, rc, NULL);
	else
	    n = find_elements(client->doc, n->children, src+1, rc,
		    NULL, NULL, create_target_elements_cb, FALSE, 0, NULL, FALSE);

	if (!n)
	    goto fail;

	/*
	 * Reset the position of the element tree now that the elements
	 * have been created.
	 */
	g_strfreev(src);
	src = src_orig;
	src_orig = NULL;
	n = find_root_element(client->doc, &src, rc, NULL, 0, FALSE);

	if (!n)
	    goto fail;

	n = find_elements(client->doc, n->children, src+1, rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);

	if (!n)
	    goto fail;
    }

fail:
    if (src_orig)
	g_strfreev(src_orig);

    *path = src;
    return n;
}

/*
 * Creates a "target" attribute. When other commands encounter an element with
 * this attribute, the element path is modified to the target value. If the
 * source element path doesn't exist when using 'ATTR SET target', it is
 * created, but the destination element path must exist.
 *
 * req[0] - source element path
 * req[1] - destination element path
 */
static gpg_error_t target_attribute(struct client_s *client, gchar **req)
{
    gchar **src, **dst, *line = NULL, **odst = NULL;
    gpg_error_t rc;
    xmlNodePtr n;

    if (!req || !req[0] || !req[1])
	return GPG_ERR_SYNTAX;

    if ((src = split_input_line(req[0], "\t", 0)) == NULL) {
	/*
	 * The first argument may be only a root element.
	 */
	if ((src = split_input_line(req[0], " ", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }

    if ((dst = split_input_line(req[1], "\t", 0)) == NULL) {
	/*
	 * The first argument may be only a root element.
	 */
	if ((dst = split_input_line(req[1], " ", 0)) == NULL) {
	    rc = GPG_ERR_SYNTAX;
	    goto fail;
	}
    }

    odst = g_strdupv(dst);

    if (!odst) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    n = find_root_element(client->doc, &dst, &rc, NULL, 0, FALSE);

    /*
     * Make sure the destination element path exists.
     */
    if (!n)
	goto fail;

    if (dst[1]) {
	n = find_elements(client->doc, n->children, dst+1, &rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);

	if (!n)
	    goto fail;
    }

    n = create_element_path(client, &src, &rc);

    if (rc)
	goto fail;

    line = g_strjoinv("\t", odst);

    if (!line) {
	rc = GPG_ERR_ENOMEM;
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	goto fail;
    }

    rc = add_attribute(n, "target", line);

fail:
    g_free(line);
    g_strfreev(src);
    g_strfreev(dst);
    g_strfreev(odst);
    return rc;
}

/*
 * req[0] - name
 * req[1] - new name
 */
static gpg_error_t name_attribute(struct client_s *client, gchar **req)
{
    gpg_error_t rc;
    gchar **tmp;
    xmlNodePtr n;

    tmp = g_strdupv(req);

    if (!tmp) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    n = find_root_element(client->doc, &tmp, &rc, NULL, 0, FALSE);
    g_strfreev(tmp);

    if (!n)
	return rc;

    if (g_utf8_collate(req[0], req[1]) == 0)
	return 0;

    /*
     * Will not overwrite an existing root.
     */
    tmp = g_strdupv(req+1);

    if (!tmp) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    n = find_root_element(client->doc, &tmp, &rc, NULL, 0, FALSE);
    g_strfreev(tmp);

    if (rc && rc != GPG_ERR_ELEMENT_NOT_FOUND)
	return rc;

    if (n)
	return GPG_ERR_AMBIGUOUS_NAME;

    if (!valid_xml_element((xmlChar *)req[1]))
	return GPG_ERR_SYNTAX;

    tmp = g_strdupv(req);

    if (!tmp) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    n = find_root_element(client->doc, &tmp, &rc, NULL, 0, FALSE);
    g_strfreev(tmp);

    if (!n)
	return GPG_ERR_ELEMENT_NOT_FOUND;

    return add_attribute(n, "_name", req[1]);
}

/*
 * req[0] - attribute
 * req[1] - element path
 */
static gpg_error_t attribute_get(assuan_context_t ctx, gchar **req)
{
    struct client_s *client = assuan_get_pointer(ctx);
    xmlNodePtr n;
    xmlChar *a;
    gchar **path= NULL;
    gpg_error_t rc;

    if (!req || !req[0] || !req[1])
	return GPG_ERR_SYNTAX;

    if (strchr(req[1], '\t')) {
	if ((path = split_input_line(req[1], "\t", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }
    else {
	if ((path = split_input_line(req[1], " ", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }

    n = find_root_element(client->doc, &path, &rc, NULL, 0, FALSE);

    if (!n)
	goto fail;

    if (path[1]) {
	n = find_elements(client->doc, n->children, path+1, &rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);

	if (!n)
	    goto fail;
    }

    g_strfreev(path);
    
    if ((a = xmlGetProp(n, (xmlChar *)req[0])) == NULL)
	return GPG_ERR_NOT_FOUND;

    pth_cleanup_push(xmlFree, a);

    if (*a)
	rc = xfer_data(ctx, (gchar *)a, xmlStrlen(a));
    else
	rc = GPG_ERR_NO_VALUE;

    pth_cleanup_pop(1);
    return rc;

fail:
    g_strfreev(path);
    return rc;
}

/*
 * req[0] - attribute
 * req[1] - element path
 * req[2] - value
 */
static gpg_error_t attribute_set(struct client_s *client, gchar **req)
{
    gchar **path = NULL;
    gpg_error_t rc;
    xmlNodePtr n;

    if (!req || !req[0] || !req[1])
	return GPG_ERR_SYNTAX;

    /*
     * Reserved attribute names.
     */
    if (!strcmp(req[0], "_name")) {
	/*
	 * Only reserved for the root element. Not the rest of the
	 * document.
	 */
	if (strchr(req[1], '\t') == NULL)
	    return name_attribute(client, req + 1);
    }
    else if (!strcmp(req[0], "target"))
	return target_attribute(client, req + 1);
    
    if ((path = split_input_line(req[1], "\t", 0)) == NULL) {
	/*
	 * The first argument may be only a root element.
	 */
	if ((path = split_input_line(req[1], " ", 0)) == NULL)
	    return GPG_ERR_SYNTAX;
    }

    n = find_root_element(client->doc, &path, &rc, NULL, 0, FALSE);

    if (!n)
	goto fail;

    if (path[1]) {
	n = find_elements(client->doc, n->children, path+1, &rc,
		NULL, NULL, NULL, FALSE, 0, NULL, FALSE);

	if (!n)
	    goto fail;
    }

    rc = add_attribute(n, req[0], req[2]);

fail:
    g_strfreev(path);
    return rc;
}

/*
 * req[0] - command
 * req[1] - attribute name or element path if command is LIST
 * req[2] - element path
 * req[2] - element path or value
 */
static gint attr_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gchar **req;
    gpg_error_t rc = 0;

    log_write2("ARGS=\"%s\"", line);
    req = split_input_line(line, " ", 4);

    if (!req || !req[0] || !req[1]) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    pth_cleanup_push(req_cleanup, req);

    if (g_ascii_strcasecmp(req[0], "SET") == 0)
	rc = attribute_set(client, req+1);
    else if (g_ascii_strcasecmp(req[0], "GET") == 0)
	rc = attribute_get(ctx, req+1);
    else if (g_ascii_strcasecmp(req[0], "DELETE") == 0)
	rc = attribute_delete(client, req+1);
    else if (g_ascii_strcasecmp(req[0], "LIST") == 0)
	rc = attribute_list(ctx, req+1);
    else
	rc = GPG_ERR_SYNTAX;

    pth_cleanup_pop(1);
    return send_error(ctx, rc);
}

static gint iscached_command(assuan_context_t ctx, gchar *line)
{
    gchar **req = split_input_line(line, " ", 0);
    guchar md5file[16];
    gchar *path, *tmp;

    if (!req || !*req) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    log_write2("ARGS=\"%s\"", line);

    if (!valid_filename(req[0])) {
	g_strfreev(req);
	return GPG_ERR_INV_VALUE;
    }

    gcry_md_hash_buffer(GCRY_MD_MD5, md5file, req[0], strlen(req[0]));
    CACHE_LOCK(ctx);

    if (cache_iscached(md5file)) {
	g_strfreev(req);
	CACHE_UNLOCK;
	return send_error(ctx, 0);
    }

    CACHE_UNLOCK;
    tmp = get_key_file_string("global", "data_directory");

    if (!tmp) {
	g_strfreev(req);
	return GPG_ERR_ENOMEM;
    }

    path = expand_homedir(tmp);

    if (!path) {
	g_strfreev(req);
	g_free(tmp);
	return GPG_ERR_ENOMEM;
    }

    g_free(tmp);
    tmp = path;
    path = g_strdup_printf("%s/%s", tmp, req[0]);
    g_free(tmp);

    if (!path) {
	g_strfreev(req);
	return GPG_ERR_ENOMEM;
    }

    if (access(path, R_OK) == -1) {
	gpg_error_t rc = gpg_error_from_syserror();

	g_free(path);
	g_strfreev(req);
	return send_error(ctx, rc);
    }

    g_free(path);
    return send_error(ctx, GPG_ERR_NOT_FOUND);
}

static gint clearcache_command(assuan_context_t ctx, gchar *line)
{
    gchar **req = split_input_line(line, " ", 0);
    guchar md5file[16];

    log_write2("ARGS=\"%s\"", line);
    CACHE_LOCK(ctx);

    if (!req || !*req) {
	g_strfreev(req);
	cache_clear(NULL, 2);
	CACHE_UNLOCK;
	return send_error(ctx, 0);
    }

    gcry_md_hash_buffer(GCRY_MD_MD5, md5file, req[0], strlen(req[0]));
    g_strfreev(req);

    if (cache_clear(md5file, 1) == FALSE) {
	CACHE_UNLOCK;
	return send_error(ctx, GPG_ERR_NOT_FOUND);
    }

    CACHE_UNLOCK;
    return send_error(ctx, 0);
}

static gint cachetimeout_command(assuan_context_t ctx, gchar *line)
{
    guchar md5file[16];
    glong timeout;
    gchar **req = split_input_line(line, " ", 0);
    gchar *p;

    if (!req || !*req || !req[1]) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    errno = 0;
    timeout = strtol(req[1], &p, 10);

    if (errno != 0 || *p != 0 || timeout < -1) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    gcry_md_hash_buffer(GCRY_MD_MD5, md5file, req[0], strlen(req[0]));
    CACHE_LOCK(client->ctx);

    if (cache_set_timeout(md5file, timeout) == FALSE) {
	CACHE_UNLOCK;
	return send_error(ctx, GPG_ERR_NOT_FOUND);
    }

    CACHE_UNLOCK;
    return send_error(ctx, 0);
}

static gint dump_command(assuan_context_t ctx, gchar *line)
{
    xmlChar *xml;
    gint len;
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;

    if (disable_list_and_dump == TRUE)
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);

    xmlDocDumpFormatMemory(client->doc, &xml, &len, 1);

    if (!xml) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return send_error(ctx, GPG_ERR_ENOMEM);
    }

    pth_cleanup_push(xmlFree, xml);
    rc = xfer_data(ctx, (gchar *)xml, len);
    pth_cleanup_pop(1);
    return send_error(ctx, rc);
}

static gint getconfig_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc = 0;
    gchar filename[255]={0}, param[747]={0};
    gchar *p, *tmp, *fp = client->filename, *paramp = line;

    log_write2("ARGS=\"%s\"", line);

    if (strchr(line, ' ')) {
	sscanf(line, " %254[^ ] %746c", filename, param);
	paramp = param;
	fp = filename;
    }

    if (fp && !valid_filename(fp))
	return send_error(ctx, GPG_ERR_INV_VALUE);

    paramp = g_ascii_strdown(paramp, -1);

    if (!paramp) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return send_error(ctx, GPG_ERR_ENOMEM);
    }

    if (fp && !g_ascii_strcasecmp(paramp, "iterations")) {
	if (!(client->opts & OPT_ITERATIONS) || fp != client->filename) {
	    file_header_internal_t *fh = read_file_header(fp, FALSE, &rc);

	    if (!fh && rc != GPG_ERR_ENOENT)
		return send_error(ctx, rc);

	    if (!rc) {
		g_free(paramp);
		p = g_strdup_printf("%lu", (unsigned long)fh->ver.fh2.iter);
		close_file_header(fh);

		if (!p) {
		    log_write("%s(%i): %s", __FILE__, __LINE__,
			    pwmd_strerror(GPG_ERR_ENOMEM));
		    return send_error(ctx, GPG_ERR_ENOMEM);
		}

		goto done;
	    }
	}
    }
    else if (!g_ascii_strcasecmp(paramp, "enable_pinentry")) {
#ifdef WITH_PINENTRY
	gboolean n;

	if (fp == client->filename && (client->opts & OPT_PINENTRY))
	    n = client->pinentry->enable;
	else
	    n = get_key_file_boolean(fp, "enable_pinentry");

	p = g_strdup_printf("%s", n ? "true" : "false");

	if (!p) {
	    log_write("%s(%i): %s", __FILE__, __LINE__,
		    pwmd_strerror(GPG_ERR_ENOMEM));
	    return send_error(ctx, GPG_ERR_ENOMEM);
	}

	goto done;
#else
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);
#endif
    }
    else if (!g_ascii_strcasecmp(paramp, "pinentry_timeout")) {
#ifdef WITH_PINENTRY
	p = g_strdup_printf("%i", get_key_file_integer(fp, "pinentry_timeout"));

	if (!p) {
	    log_write("%s(%i): %s", __FILE__, __LINE__,
		    pwmd_strerror(GPG_ERR_ENOMEM));
	    return send_error(ctx, GPG_ERR_ENOMEM);
	}

	goto done;
#else
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);
#endif
    }

    p = get_key_file_string(fp ? fp : "global", paramp);
    g_free(paramp);

    if (!p)
	return send_error(ctx, GPG_ERR_NO_VALUE);

    tmp = expand_homedir(p);
    g_free(p);

    if (!tmp) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return send_error(ctx, GPG_ERR_ENOMEM);
    }

    p = tmp;
done:
    pth_cleanup_push(g_free, p);
    rc = xfer_data(ctx, p, strlen(p));
    pth_cleanup_pop(1);
    return send_error(ctx, rc);
}

struct xpath_s {
    xmlXPathContextPtr xp;
    xmlXPathObjectPtr result;
    xmlBufferPtr buf;
    gchar **req;
};

static void xpath_command_cleanup(void *arg)
{
    struct xpath_s *xpath = arg;

    req_cleanup(xpath->req);

    if (xpath->buf)
	xmlBufferFree(xpath->buf);

    if (xpath->result)
	xmlXPathFreeObject(xpath->result);

    if (xpath->xp)
	xmlXPathFreeContext(xpath->xp);
}

static gint xpath_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    struct xpath_s xpath;

    log_write2("ARGS=\"%s\"", line);

    if (disable_list_and_dump == TRUE)
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);

    if (!line || !*line)
	return send_error(ctx, GPG_ERR_SYNTAX);

    memset(&xpath, 0, sizeof(struct xpath_s));

    if ((xpath.req = split_input_line(line, "\t", 2)) == NULL) {
	if (strv_printf(&xpath.req, "%s", line) == FALSE)
	    return send_error(ctx, GPG_ERR_ENOMEM);
    }

    xpath.xp = xmlXPathNewContext(client->doc);

    if (!xpath.xp) {
	xpath_command_cleanup(&xpath);
	return send_error(ctx, EPWMD_LIBXML_ERROR);
    }

    xpath.result = xmlXPathEvalExpression((xmlChar *)xpath.req[0], xpath.xp);

    if (!xpath.result) {
	xpath_command_cleanup(&xpath);
	return send_error(ctx, EPWMD_LIBXML_ERROR);
    }

    if (xmlXPathNodeSetIsEmpty(xpath.result->nodesetval)) {
	rc = GPG_ERR_ELEMENT_NOT_FOUND;
	goto fail;
    }

    rc = recurse_xpath_nodeset(client->doc, xpath.result->nodesetval,
	    (xmlChar *)xpath.req[1], &xpath.buf, 0, NULL);

    if (rc)
	goto fail;
    else if (!xpath.req[1] && !xmlBufferLength(xpath.buf)) {
	rc = GPG_ERR_NO_VALUE;
	goto fail;
    }
    else if (xpath.req[1])
	goto fail;

    pth_cleanup_push(xpath_command_cleanup, &xpath);
    rc = xfer_data(ctx, (gchar *)xmlBufferContent(xpath.buf),
	    xmlBufferLength(xpath.buf));
    pth_cleanup_pop(0);

fail:
    xpath_command_cleanup(&xpath);
    return send_error(ctx, rc);
}

/* XPATHATTR SET|DELETE <name> <expression>[<TAB>[value]] */
static gint xpathattr_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    struct xpath_s xpath;
    gchar **req = NULL;
    gboolean cmd = FALSE; //SET

    log_write2("ARGS=\"%s\"", line);

    if (disable_list_and_dump == TRUE)
	return send_error(ctx, GPG_ERR_NOT_IMPLEMENTED);

    if (!line || !*line)
	return send_error(ctx, GPG_ERR_SYNTAX);

    memset(&xpath, 0, sizeof(struct xpath_s));

    if ((req = split_input_line(line, " ", 3)) == NULL)
	return send_error(ctx, GPG_ERR_ENOMEM);

    if (!req[0]) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    if (!g_ascii_strcasecmp(req[0], "SET"))
	cmd = FALSE;
    else if (!g_ascii_strcasecmp(req[0], "DELETE"))
	cmd = TRUE;
    else {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    if (!req[1] || !req[2]) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    if ((xpath.req = split_input_line(req[2], "\t", 3)) == NULL) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    if (!xpath.req[0] || (!xpath.req[1] && !cmd) || (xpath.req[1] && cmd)) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    xpath.xp = xmlXPathNewContext(client->doc);

    if (!xpath.xp) {
	rc = EPWMD_LIBXML_ERROR;
	goto fail;
    }

    xpath.result = xmlXPathEvalExpression((xmlChar *)xpath.req[0], xpath.xp);

    if (!xpath.result) {
	rc = EPWMD_LIBXML_ERROR;
	goto fail;
    }

    if (xmlXPathNodeSetIsEmpty(xpath.result->nodesetval)) {
	rc = GPG_ERR_ELEMENT_NOT_FOUND;
	goto fail;
    }

    rc = recurse_xpath_nodeset(client->doc, xpath.result->nodesetval,
	    (xmlChar *)xpath.req[1], &xpath.buf, cmd, (xmlChar *)req[1]);

fail:
    g_strfreev(req);
    xpath_command_cleanup(&xpath);
    return send_error(ctx, rc);
}

static gint import_command_finalize(gpointer data, gint assuan_rc, guchar *line,
	gsize len)
{
    struct client_s *client = assuan_get_pointer((assuan_context_t)data);
    gpg_error_t rc = file_modified(client);
    gchar **req, **path = NULL, **path_orig = NULL, *content;
    xmlDocPtr doc = NULL;
    xmlNodePtr n, root, copy;

    if (assuan_rc || rc) {
	if (line)
	    xfree(line);
	return assuan_rc ? assuan_rc : rc;
    }

    req = split_input_line((gchar *)line, "\t", 2);
    xfree(line);

    if (!req || !*req)
	return GPG_ERR_SYNTAX;

    content = req[0];
    path = split_input_line(req[1], "\t", 0);

    if (!content || !*content) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    doc = xmlReadDoc((xmlChar *)content, NULL, "UTF-8", XML_PARSE_NOBLANKS);

    if (!doc) {
	rc = EPWMD_LIBXML_ERROR;
	goto fail;
    }

    root = xmlDocGetRootElement(doc);
    rc = validate_import(root);

    if (rc)
	goto fail;

    if (path) {
	path_orig = g_strdupv(path);

	if (!path_orig) {
	    log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	    rc = GPG_ERR_ENOMEM;
	    goto fail;
	}

	xmlChar *a = xmlGetProp(root, (xmlChar *)"_name");

	if (!a) {
	    g_strfreev(path_orig);
	    rc = GPG_ERR_ENOMEM;
	    goto fail;
	}

	if (strv_printf(&path, "%s", (gchar *)a) == FALSE) {
	    xmlFree(a);
	    g_strfreev(path_orig);
	    rc = GPG_ERR_ENOMEM;
	    goto fail;
	}

	xmlFree(a);
	n = find_root_element(client->doc, &path, &rc, NULL, 0, FALSE);

	if (rc && rc != GPG_ERR_ELEMENT_NOT_FOUND) {
	    g_strfreev(path_orig);
	    goto fail;
	}

	if (!rc) {
	    n = find_elements(client->doc, n->children, path+1, &rc, NULL, NULL, NULL, FALSE, 0, NULL, TRUE);

	    if (rc && rc != GPG_ERR_ELEMENT_NOT_FOUND) {
		g_strfreev(path_orig);
		goto fail;
	    }
	    else if (!rc) {
		xmlNodePtr parent = n->parent;

		xmlUnlinkNode(n);
		xmlFreeNode(n);
		n = parent;
	    }
	}

	g_strfreev(path);
	path = path_orig;

	if (rc == GPG_ERR_ELEMENT_NOT_FOUND) {
	    n = create_element_path(client, &path, &rc);

	    if (rc)
		goto fail;
	}

	copy = xmlCopyNodeList(root);
	n = xmlAddChildList(n, copy);

	if (!n)
	    rc = EPWMD_LIBXML_ERROR;
    }
    else {
	/* Check if the content root element can create a DTD root element. */
	if (!xmlStrEqual((xmlChar *)"element", root->name)) {
	    rc = GPG_ERR_SYNTAX;
	    goto fail;
	}

	xmlChar *a;

	if ((a = xmlGetProp(root, (xmlChar *)"_name")) == NULL) {
	    rc = GPG_ERR_SYNTAX;
	    goto fail;
	}

	gchar *tmp = g_strdup((gchar *)a);
	xmlFree(a);
	gboolean literal = is_literal_element(&tmp);

	if (!valid_xml_element((xmlChar *)tmp) || literal) {
	    g_free(tmp);
	    rc = GPG_ERR_INV_VALUE;
	    goto fail;
	}

	if (strv_printf(&path, "%s", tmp) == FALSE) {
	    g_free(tmp);
	    rc = GPG_ERR_ENOMEM;
	    goto fail;
	}

	g_free(tmp);
	n = find_root_element(client->doc, &path, &rc, NULL, 0, TRUE);

	if (rc && rc != GPG_ERR_ELEMENT_NOT_FOUND) {
	    rc = EPWMD_LIBXML_ERROR;
	    goto fail;
	}

	/* Overwriting the existing tree. */
	if (!rc) {
	    xmlUnlinkNode(n);
	    xmlFreeNodeList(n);
	}

	rc = 0;
	xmlSetProp(root, (xmlChar *)"_name", (xmlChar *)path[0]);
	n = xmlCopyNode(root, 1);
	n = xmlAddChildList(xmlDocGetRootElement(client->doc), n);
    }

    if (n && !rc)
	rc = update_element_mtime(n->parent);

fail:
    if (doc)
	xmlFreeDoc(doc);

    if (path)
	g_strfreev(path);

    g_strfreev(req);
    client->inquire_status = INQUIRE_DONE;
    return rc;
}

static gint import_command(assuan_context_t ctx, gchar *line)
{
    gpg_error_t rc;
    struct client_s *client = assuan_get_pointer(ctx);

    rc = assuan_inquire_ext(ctx, "IMPORT", 0, import_command_finalize, ctx);

    if (rc)
	return send_error(ctx, rc);

    /* Don't return with assuan_process_done() here. This is an INQUIRE. */
    client->inquire_status = INQUIRE_BUSY;
    return 0;
}

static gpg_error_t do_lock_command(struct client_s *client)
{
    gpg_error_t rc = lock_file_mutex(client);

    if (!rc)
	client->is_lock_cmd = TRUE;

    return client->opts & OPT_INQUIRE ? rc : send_error(client->ctx, rc);
}

static gint lock_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);

    return do_lock_command(client);
}

static gint unlock_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);

    unlock_file_mutex(client);
    return send_error(ctx, 0);
}

static gint getpid_command(assuan_context_t ctx, gchar *line)
{
    gpg_error_t rc;
    gchar buf[32];
    pid_t pid = getpid();

    print_fmt(buf, sizeof(buf), "%i", pid);
    rc = xfer_data(ctx, buf, strlen(buf));
    return send_error(ctx, rc);
}

static gint version_command(assuan_context_t ctx, gchar *line)
{
    gpg_error_t rc;
    gchar buf[32];

    print_fmt(buf, sizeof(buf), "0x%X", VERSION_HEX);
    rc = xfer_data(ctx, buf, strlen(buf));
    return send_error(ctx, rc);
}

#ifdef WITH_PINENTRY
static void set_option_value(gchar **opt, const gchar *value)
{
    if (opt)
	g_free(*opt);

    *opt = NULL;

    if (value)
	*opt = g_strdup(value);
}
#endif

static gint set_unset_common(assuan_context_t ctx, const gchar *name,
	const gchar *value)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;

    if (g_ascii_strcasecmp(name, (gchar *)"log_level") == 0) {
	glong l = 0;

	if (value) {
	    l = strtol(value, NULL, 10);

	    if (l < 0 || l > 2)
		return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);
	}

	MUTEX_LOCK(&rcfile_mutex);
	g_key_file_set_integer(keyfileh, "global", "log_level", (gint)l);
	MUTEX_UNLOCK(&rcfile_mutex);
	goto done;
    }
    else if (g_ascii_strcasecmp(name, (gchar *)"rc_on_locked") == 0) {
	glong l = 0;

	if (value) {
	    l = strtol(value, NULL, 10);

	    if (l < 0 || l > 1)
		return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);
	}

	client->rc_on_locked = l ? TRUE : FALSE;
	goto done;
    }
    else if (g_ascii_strcasecmp(name, (gchar *)"lock_on_open") == 0) {
	rc = parse_open_opt_lock(client, (gpointer)value);

	if (rc)
	    return rc;

	client->opts |= OPT_LOCK;
    }
    else if (g_ascii_strcasecmp(name, (gchar *)"cipher") == 0) {
	if (!value) {
	    client->opts &= ~(OPT_CIPHER);
	    goto done;
	}

	rc = parse_save_opt_cipher(client, (gpointer)value);

	if (rc)
	    return rc;

	client->opts |= OPT_CIPHER;
	goto done;
    }
    else if (g_ascii_strcasecmp(name, (gchar *)"iterations") == 0) {
	rc = parse_save_opt_iterations(client, (gpointer)value);

	if (rc)
	    return rc;

	goto done;
    }
    else if (g_ascii_strcasecmp(name, (gchar *)"NAME") == 0) {
	pth_attr_t attr = pth_attr_of(pth_self());
	gchar buf[41];

	if (!value) {
	    pth_attr_destroy(attr);
	    goto done;
	}

	print_fmt(buf, sizeof(buf), "%s", value);
	pth_attr_set(attr, PTH_ATTR_NAME, buf);
	pth_attr_destroy(attr);
#ifdef WITH_PINENTRY
	if (client->pinentry->name)
	    g_free(client->pinentry->name);

	client->pinentry->name = g_strdup(buf);

	if (!client->pinentry->name)
	    return GPG_ERR_ENOMEM;
#endif

	goto done;
    }
#ifdef WITH_PINENTRY
    else if (g_ascii_strcasecmp(name, (gchar *)"lc_messages") == 0)
	set_option_value(&client->pinentry->lcmessages, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"lc_ctype") == 0)
	set_option_value(&client->pinentry->lcctype, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"ttyname") == 0)
	set_option_value(&client->pinentry->ttyname, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"ttytype") == 0)
	set_option_value(&client->pinentry->ttytype, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"display") == 0)
	set_option_value(&client->pinentry->display, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"pinentry_path") == 0)
	set_option_value(&client->pinentry->path, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"title") == 0)
	set_option_value(&client->pinentry->title, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"prompt") == 0)
	set_option_value(&client->pinentry->prompt, value);
    else if (g_ascii_strcasecmp(name, (gchar *)"desc") == 0)
	set_option_value(&client->pinentry->desc, value);
    else if (g_ascii_strcasecmp(name, "pinentry_timeout") == 0) {
	gchar *p = NULL;
	gint n;
	
	if (!value)
	    goto done;

	n = strtol(value, &p, 10);

	if (*p || n < 0)
	    return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_INV_VALUE);

	MUTEX_LOCK(&rcfile_mutex);
	g_key_file_set_integer(keyfileh, client->filename ? client->filename :
		"global", "pinentry_timeout", n);
	MUTEX_UNLOCK(&rcfile_mutex);
	goto done;
    }
    else if (g_ascii_strcasecmp(name, "enable_pinentry") == 0) {
	rc = parse_opt_pinentry(client, (gpointer)value);

	if (rc)
	    return rc;

	goto done;
    }
#else
    else if (g_ascii_strcasecmp(name, (gchar *)"lc_messages") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"lc_ctype") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"ttyname") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"ttytype") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"display") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"pinentry_path") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"title") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"prompt") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, (gchar *)"desc") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, "pinentry_timeout") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
    else if (g_ascii_strcasecmp(name, "enable_pinentry") == 0)
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_NOT_IMPLEMENTED);
#endif
    else
	return gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_UNKNOWN_OPTION);

done:
    return 0;
}

static gint unset_command(assuan_context_t ctx, gchar *line)
{
    log_write2("ARGS=\"%s\"", line);
    return send_error(ctx, set_unset_common(ctx, line, NULL));
}

static gint set_command(assuan_context_t ctx, gchar *line)
{
    gchar name[64] = {0}, value[256] = {0};

    log_write2("ARGS=\"%s\"", line);

    if (sscanf(line, " %63[_a-zA-Z] = %255c", name, value) != 2)
	return send_error(ctx, gpg_err_make(PWMD_ERR_SOURCE, GPG_ERR_SYNTAX));

    return send_error(ctx, set_unset_common(ctx, name, value));
}

static gint rename_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    gchar **req, **src, *dst;
    xmlNodePtr n, ndst;

    log_write2("ARGS=\"%s\"", line);
    req = split_input_line(line, " ", -1);

    if (!req || !req[0] || !req[1]) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    dst = req[1];
    is_literal_element(&dst);

    if (!valid_xml_element((xmlChar *)dst)) {
	g_strfreev(req);
	return GPG_ERR_INV_VALUE;
    }

    if (strchr(req[0], '\t'))
	src = split_input_line(req[0], "\t", -1);
    else
	src = split_input_line(req[0], " ", -1);

    if (!src || !*src) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    n = find_root_element(client->doc, &src, &rc, NULL, 0, FALSE);

    if (src[1] && n)
	n = find_elements(client->doc, n->children, src+1, &rc, NULL, NULL,
		NULL, FALSE, 0, NULL, FALSE);

    if (!n)
	goto fail;


    xmlChar *a = xmlGetProp(n, (xmlChar *)"_name");

    if (!a) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    /* To prevent unwanted effects:
     *
     * <root name="a"><b/></root>
     *
     * RENAME a<TAB>b b
     */
    if (xmlStrEqual(a, (xmlChar *)dst)) {
	xmlFree(a);
	rc = GPG_ERR_AMBIGUOUS_NAME;
	goto fail;
    }

    xmlFree(a);
    gchar **tmp = NULL;

    if (src[1]) {
	gchar **p;

	for (p = src; *p; p++) {
	    if (!*(p+1))
		break;

	    strv_printf(&tmp, "%s", *p);
	}
    }

    strv_printf(&tmp, "!%s", dst);
    ndst = find_root_element(client->doc, &tmp, &rc, NULL, 0, FALSE);

    if (!ndst && rc && rc != GPG_ERR_ELEMENT_NOT_FOUND) {
	g_strfreev(tmp);
	goto fail;
    }

    if (tmp[1] && ndst)
	ndst = find_elements(client->doc, ndst->children, tmp+1, &rc, NULL,
		NULL, NULL, FALSE, 0, NULL, FALSE);

    g_strfreev(tmp);

    if (!ndst && rc && rc != GPG_ERR_ELEMENT_NOT_FOUND)
	goto fail;

    rc = 0;

    /* Target may exist:
     *
     * <root name="a"/>
     * <root name="b" target="a"/>
     *
     * RENAME b a
     *
     * Would need to do:
     * RENAME !b a
     */
    if (ndst == n) {
	rc = GPG_ERR_AMBIGUOUS_NAME;
	goto fail;
    }

    if (ndst) {
	unlink_node(ndst);
	xmlFreeNodeList(ndst);
    }

    rc = add_attribute(n, "_name", dst);

fail:
    g_strfreev(req);
    g_strfreev(src);
    return send_error(ctx, rc);
}

static gint copy_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    gchar **req, **src = NULL, **dst = NULL;
    xmlNodePtr nsrc, ndst, new;

    log_write2("ARGS=\"%s\"", line);
    req = split_input_line(line, " ", -1);

    if (!req || !req[0] || !req[1]) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    if (strchr(req[0], '\t'))
	src = split_input_line(req[0], "\t", -1);
    else
	src = split_input_line(req[0], " ", -1);

    if (!src || !*src) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    if (strchr(req[1], '\t'))
	dst = split_input_line(req[1], "\t", -1);
    else
	dst = split_input_line(req[1], " ", -1);

    if (!dst || !*dst) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    nsrc = find_root_element(client->doc, &src, &rc, NULL, 0, FALSE);

    if (nsrc && src[1])
	nsrc = find_elements(client->doc, nsrc->children, src+1, &rc, NULL,
		NULL, NULL, FALSE, 0, NULL, FALSE);

    if (!nsrc)
	goto fail;

    ndst = find_root_element(client->doc, &dst, &rc, NULL, 0, FALSE);

    if (ndst && dst[1])
	ndst = find_elements(client->doc, ndst->children, dst+1, &rc, NULL,
		NULL, NULL, FALSE, 0, NULL, FALSE);

    if (!ndst && rc != GPG_ERR_ELEMENT_NOT_FOUND)
	goto fail;

    new = xmlCopyNodeList(nsrc);

    if (!new) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    if (!ndst)
	ndst = create_element_path(client, &dst, &rc);

    if (!ndst) {
	xmlUnlinkNode(new);
	xmlFreeNodeList(new);
	goto fail;
    }

    /* Merge any attributes from the src node to the initial dst node. */
    for (xmlAttrPtr attr = new->properties; attr; attr = attr->next) {
	if (xmlStrEqual(attr->name, (xmlChar *)"_name"))
	    continue;

	xmlAttrPtr a = xmlHasProp(ndst, attr->name);

	if (a)
	    xmlRemoveProp(a);

	xmlChar *tmp = xmlNodeGetContent(attr->children);
	xmlNewProp(ndst, attr->name, tmp);
	xmlFree(tmp);
	rc = add_attribute(ndst, NULL, NULL);
    }

    xmlNodePtr n = ndst->children;
    xmlUnlinkNode(n);
    xmlFreeNodeList(n);
    ndst->children = NULL;

    if (!new->children) {
	xmlUnlinkNode(new);
	xmlFreeNodeList(new);
	goto fail;
    }

    n = xmlCopyNodeList(new->children);

    if (!n) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    xmlUnlinkNode(new);
    xmlFreeNodeList(new);
    n = xmlAddChildList(ndst, n);

    if (!n) {
	rc = GPG_ERR_ENOMEM;
	goto fail;
    }

    rc = update_element_mtime(xmlDocGetRootElement(client->doc) == ndst->parent ? ndst : ndst->parent);

fail:
    if (req)
	g_strfreev(req);

    if (src)
	g_strfreev(src);

    if (dst)
	g_strfreev(dst);

    return send_error(ctx, rc);
}

static gint move_command(assuan_context_t ctx, gchar *line)
{
    struct client_s *client = assuan_get_pointer(ctx);
    gpg_error_t rc;
    gchar **req, **src = NULL, **dst = NULL;
    xmlNodePtr nsrc, ndst = NULL;

    log_write2("ARGS=\"%s\"", line);
    req = split_input_line(line, " ", -1);

    if (!req || !req[0] || !req[1]) {
	g_strfreev(req);
	return send_error(ctx, GPG_ERR_SYNTAX);
    }

    if (strchr(req[0], '\t'))
	src = split_input_line(req[0], "\t", -1);
    else
	src = split_input_line(req[0], " ", -1);

    if (!src || !*src) {
	rc = GPG_ERR_SYNTAX;
	goto fail;
    }

    if (strchr(req[1], '\t'))
	dst = split_input_line(req[1], "\t", -1);
    else
	dst = split_input_line(req[1], " ", -1);

    nsrc = find_root_element(client->doc, &src, &rc, NULL, 0, FALSE);

    if (nsrc && src[1])
	nsrc = find_elements(client->doc, nsrc->children, src+1, &rc, NULL,
		NULL, NULL, FALSE, 0, NULL, FALSE);

    if (!nsrc)
	goto fail;

    if (dst) {
	ndst = find_root_element(client->doc, &dst, &rc, NULL, 0, FALSE);

	if (ndst && dst[1])
	    ndst = find_elements(client->doc, ndst->children, dst+1, &rc, NULL,
		    NULL, NULL, FALSE, 0, NULL, FALSE);
    }
    else
	ndst = xmlDocGetRootElement(client->doc);

    for (xmlNodePtr n = ndst; n; n = n->parent) {
	if (n == nsrc) {
	    rc = GPG_ERR_CONFLICT;
	    goto fail;
	}
    }

    if (rc && rc != GPG_ERR_ELEMENT_NOT_FOUND)
	goto fail;

    rc = 0;

    if (ndst) {
	xmlChar *a = node_has_attribute(nsrc, (xmlChar *)"_name");
	xmlNodePtr dup = find_element(ndst->children, (gchar *)a, NULL);

	xmlFree(a);

	if (dup) {
	    if (dup == nsrc)
		goto fail;
	    
	    if (ndst == xmlDocGetRootElement(client->doc)) {
		xmlNodePtr n = nsrc;
		gboolean match = FALSE;

		while (n->parent && n->parent != ndst)
		    n = n->parent;

		xmlChar *a = node_has_attribute(n, (xmlChar *)"_name");
		xmlChar *b = node_has_attribute(nsrc, (xmlChar *)"_name");

		if (xmlStrEqual(a, b)) {
		    match = TRUE;
		    xmlUnlinkNode(nsrc);
		    xmlUnlinkNode(n);
		    xmlFreeNodeList(n);
		}

		xmlFree(a);
		xmlFree(b);

		if (!match) {
		    xmlUnlinkNode(dup);
		    xmlFreeNodeList(dup);
		}
	    }
	    else
		xmlUnlinkNode(dup);
	}
    }

    if (!ndst && dst)
	ndst = create_element_path(client, &dst, &rc);

    if (!ndst)
	goto fail;

    update_element_mtime(nsrc->parent);
    xmlUnlinkNode(nsrc);
    ndst = xmlAddChildList(ndst, nsrc);

    if (!ndst)
	rc = GPG_ERR_ENOMEM;

    update_element_mtime(ndst->parent);

fail:
    if (req)
	g_strfreev(req);

    if (src)
	g_strfreev(src);

    if (dst)
	g_strfreev(dst);

    return send_error(ctx, rc);
}

static int ls_command(assuan_context_t ctx, gchar *line)
{
    log_write2("ARGS=\"%s\"", line);
    gpg_error_t rc;
    gchar *tmp = g_key_file_get_string(keyfileh, "global", "data_directory", NULL);
    gchar *dir = expand_homedir(tmp);
    DIR *d = opendir(dir);

    rc = gpg_error_from_syserror();
    g_free(tmp);

    if (!d) {
	g_free(dir);
	return send_error(ctx, rc);
    }

    size_t len = offsetof(struct dirent, d_name)+pathconf(dir, _PC_NAME_MAX)+1;
    struct dirent *p = g_malloc(len), *cur = NULL;
    gchar *list = NULL;

    g_free(dir);
    rc = 0;

    while (!readdir_r(d, p, &cur) && cur) {
	if (cur->d_name[0] == '.' && cur->d_name[1] == '\0')
	    continue;
	else if (cur->d_name[0] == '.' && cur->d_name[1] == '.' && cur->d_name[2] == '\0')
	    continue;

	tmp = g_strdup_printf("%s%s\n", list ? list : "", cur->d_name);

	if (!tmp) {
	    if (list)
		g_free(list);

	    rc = GPG_ERR_ENOMEM;
	    break;
	}

	g_free(list);
	list = tmp;
    }

    closedir(d);
    g_free(p);

    if (rc)
	return send_error(ctx, rc);

    if (!list)
	return send_error(ctx, GPG_ERR_NO_VALUE);

    list[strlen(list)-1] = 0;
    rc = xfer_data(ctx, list, strlen(list));
    g_free(list);
    return send_error(ctx, rc);
}

static void bye_notify(assuan_context_t ctx)
{
    struct client_s *cl = assuan_get_pointer(ctx);

    /* This will let assuan_process_next() return. */
    fcntl(cl->thd->fd, F_SETFL, O_NONBLOCK);
}

static void reset_notify(assuan_context_t ctx)
{
    struct client_s *cl = assuan_get_pointer(ctx);

    if (cl)
	cleanup_client(cl);
}

/* 
 * This is called before every Assuan command.
 */
gint command_startup(assuan_context_t ctx, const gchar *name)
{
    struct client_s *cl = assuan_get_pointer(ctx);
    gpg_error_t rc;

    log_write1("%s", name);

    if (!g_ascii_strcasecmp(name, "ISCACHED") ||
	    !g_ascii_strcasecmp(name, "CLEARCACHE") ||
	    !g_ascii_strcasecmp(name, "CACHETIMEOUT") ||
	    !g_ascii_strcasecmp(name, "GETCONFIG") ||
	    !g_ascii_strcasecmp(name, "GETPID") ||
	    !g_ascii_strcasecmp(name, "VERSION") ||
	    !g_ascii_strcasecmp(name, "SET") ||
	    !g_ascii_strcasecmp(name, "BYE") ||
	    !g_ascii_strcasecmp(name, "NOP") ||
	    !g_ascii_strcasecmp(name, "CANCEL") ||
	    !g_ascii_strcasecmp(name, "RESET") ||
	    !g_ascii_strcasecmp(name, "END") ||
	    !g_ascii_strcasecmp(name, "HELP") ||
	    !g_ascii_strcasecmp(name, "OPTION") ||
	    !g_ascii_strcasecmp(name, "INPUT") ||
	    !g_ascii_strcasecmp(name, "OUTPUT") ||
	    !g_ascii_strcasecmp(name, "LS") ||
	    !g_ascii_strcasecmp(name, "UNSET"))
	return 0;

#ifdef WITH_PINENTRY
    if (!(cl->opts & OPT_PINENTRY))
	reset_pin_defaults(cl->pinentry);
#endif

    cl->last_rc = rc = file_modified(cl);

    if (rc) {
	if ((rc == EPWMD_NO_FILE || rc == EPWMD_FILE_MODIFIED) &&
		!g_ascii_strcasecmp(name, "OPEN"))
	    rc = 0;
    }

    return rc;
}

/*
 * This is called after every Assuan command.
 */
void command_finalize(assuan_context_t ctx, gint rc)
{
    struct client_s *client = assuan_get_pointer(ctx);

    if (!client->is_lock_cmd)
	unlock_file_mutex(client);

    log_write1(N_("command completed (rc=%u)"), client->last_rc);
    client->opts &= ~(OPT_INQUIRE);
    client->opts &= ~(OPT_BASE64);
}

gpg_error_t register_commands(assuan_context_t ctx)
{
    static struct {
	const gchar *name;
	gint (*handler)(assuan_context_t, gchar *line);
    } table[] = {
	{ "OPEN", open_command },
	{ "SAVE", save_command },
	{ "LIST", list_command },
	{ "REALPATH", realpath_command },
	{ "STORE", store_command },
	{ "DELETE", delete_command },
	{ "GET", get_command },
	{ "ATTR", attr_command },
	{ "ISCACHED", iscached_command },
	{ "CLEARCACHE", clearcache_command },
	{ "CACHETIMEOUT", cachetimeout_command },
	{ "GETCONFIG", getconfig_command },
	{ "DUMP", dump_command },
	{ "XPATH", xpath_command },
	{ "XPATHATTR", xpathattr_command },
	{ "IMPORT", import_command },
	{ "LOCK", lock_command },
	{ "UNLOCK", unlock_command },
	{ "GETPID", getpid_command },
	{ "VERSION", version_command },
	{ "SET", set_command },
	{ "UNSET", unset_command },
	{ "RENAME", rename_command },
	{ "COPY", copy_command },
	{ "LS", ls_command },
	{ "MOVE", move_command },
	{ "INPUT", NULL }, 
	{ "OUTPUT", NULL }, 
	{ NULL, NULL }
    };
    gint i, rc;

    for (i=0; table[i].name; i++) {
	rc = assuan_register_command (ctx, table[i].name, table[i].handler);

	if (rc)
	    return rc;
    } 

    rc = assuan_register_bye_notify(ctx, bye_notify);

    if (rc)
	return rc;

    rc = assuan_register_reset_notify(ctx, reset_notify);

    if (rc)
	return rc;

    rc = assuan_register_pre_cmd_notify(ctx, command_startup);

    if (rc)
	return rc;

    return assuan_register_post_cmd_notify(ctx, command_finalize);
}

gpg_error_t try_xml_decrypt(assuan_context_t ctx,
	struct crypto_s *crypto, gpointer *dst, goffset *dst_len)
{
    goffset insize, len;
    struct client_s *client = ctx ? assuan_get_pointer(ctx) : NULL;
    guint64 iter = 0ULL, n_iter = 0ULL, iter_progress = 0ULL;
    gulong outsize = 0;
    gpg_error_t rc;
    gsize fh_size = crypto->fh->v1 ? sizeof(crypto->fh->ver.fh1) : sizeof(crypto->fh->ver.fh2);
    guint64 fh_iter = crypto->fh->v1 ? crypto->fh->ver.fh1.iter : crypto->fh->ver.fh2.iter;
    gsize hashlen = gcry_md_get_algo_dlen(GCRY_MD_SHA256);

    lseek(crypto->fh->fd, fh_size, SEEK_SET);
    insize = crypto->fh->st.st_size - fh_size;
    crypto->iv = gcry_malloc(crypto->blocksize);

    if (!crypto->iv) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    if (crypto->fh->v1)
	memcpy(crypto->iv, crypto->fh->ver.fh1.iv, crypto->blocksize);
    else
	memcpy(crypto->iv, crypto->fh->ver.fh2.iv, crypto->blocksize);

    crypto->inbuf = gcry_malloc(insize);

    if (!crypto->inbuf) {
	log_write("%s(%i): %s", __FILE__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    crypto->insize = insize;
    len = pth_read(crypto->fh->fd, crypto->inbuf, crypto->insize);

    if (len != crypto->insize)
	return GPG_ERR_INV_LENGTH;

    /* No encryption iterations. This is a plain (gzipped) file. */
    if ((crypto->fh->v1 && (long)fh_iter < 0L) ||
	    (!crypto->fh->v1 && fh_iter <= 0L)) {
	/*
	 * cache_file_count() needs both .used == TRUE and a valid key in
	 * order for it to count as a used cache entry. Fixes CACHE status
	 * messages.
	 */
	memset(crypto->key, '!', hashlen);
	goto decompress;
    }

    if ((rc = gcry_cipher_setiv(crypto->gh, crypto->iv, crypto->blocksize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    if ((rc = gcry_cipher_setkey(crypto->gh, crypto->key, crypto->keysize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    iter_progress = (guint64)get_key_file_double(client && client->filename ?
	    client->filename : "global", "iteration_progress");

    if (iter_progress > 0ULL && fh_iter >= iter_progress) {
	rc = send_status(ctx, STATUS_DECRYPT, "0 %llu", fh_iter);
	
	if (rc)
	    return rc;
    }

    rc = iterate_crypto_once(client, crypto, STATUS_DECRYPT);

    if (rc)
	return rc;

    crypto->tkey = gcry_malloc(hashlen);

    if (!crypto->tkey) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(GPG_ERR_ENOMEM));
	return GPG_ERR_ENOMEM;
    }

    memcpy(crypto->tkey, crypto->key, hashlen);
    guchar *tkey = crypto->tkey;
    tkey[0] ^= 1;

    if ((rc = gcry_cipher_setkey(crypto->gh, crypto->tkey, crypto->keysize))) {
	log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	return rc;
    }

    while (iter < (crypto->fh->v1 ? fh_iter : fh_iter-1)) {
	if (iter_progress > 0ULL && iter >= iter_progress) {
	    if (!(iter % iter_progress)) {
		rc = send_status(ctx, STATUS_DECRYPT, "%llu %llu",
			++n_iter * iter_progress, fh_iter);
	
		if (rc)
		    return rc;
	    }
	}

	if ((rc = gcry_cipher_setiv(crypto->gh, crypto->iv, crypto->blocksize))) {
	    log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	    return rc;
	}

	rc = iterate_crypto_once(client, crypto, STATUS_DECRYPT);

	if (rc) {
	    log_write("%s(%i): %s", __FUNCTION__, __LINE__, pwmd_strerror(rc));
	    return rc;
	}

	iter++;
    }

    if (iter_progress && fh_iter >= iter_progress) {
	rc = send_status(ctx, STATUS_DECRYPT, "%llu %llu", fh_iter, fh_iter);

	if (rc)
	    return rc;
    }

decompress:
    if (!crypto->fh->v1 && crypto->fh->ver.fh2.version >= 0x218) {
	len = 0;

	if (crypto->fh->ver.fh2.iter > 0ULL) {
	    if (memcmp(crypto->inbuf, crypto_magic, sizeof(crypto_magic)))
		return GPG_ERR_INV_PASSPHRASE;

	    len = sizeof(crypto_magic);
	}

	rc = do_decompress(ctx, (guchar *)crypto->inbuf+len, crypto->insize-len,
		    (gpointer *)&crypto->outbuf, &outsize);

	if (rc)
	    return rc;
    }
    else {
	rc = do_decompress(ctx, crypto->inbuf, crypto->insize,
		    (gpointer *)&crypto->outbuf, &outsize);

	if (rc == GPG_ERR_ENOMEM)
	    return rc;
	else if (rc)
	    return GPG_ERR_INV_PASSPHRASE; // Not a valid gzip header. Must be a bad key.

	if (g_strncasecmp(crypto->outbuf, "<?xml ", 6) != 0) {
	    gcry_free(crypto->outbuf);
	    crypto->outbuf = NULL;
	    return GPG_ERR_INV_PASSPHRASE;
	}
    }

    if (ctx) {
	client->xml = crypto->outbuf;
	client->len = outsize;
	crypto->outbuf = NULL;
    }
    else if (dst) {
	*dst = crypto->outbuf;
	*dst_len = outsize;
	crypto->outbuf = NULL;
    }

    /* The calling function should free the crypto struct. */
    return 0;
}
