/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>
#include <ctype.h>
#include <gpg-error.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "misc.h"

gboolean strv_printf(gchar ***array, const gchar *fmt, ...)
{
    gchar **a;
    va_list ap;
    gchar *buf;
    gint len = *array ? g_strv_length(*array) : 0;
    gint ret;

    if (!fmt)
	return FALSE;

    if ((a = g_realloc(*array, (len + 2) * sizeof(gchar *))) == NULL) {
	log_write("%s(%i): %s", __FILE__, __LINE__, strerror(ENOMEM));
	return FALSE;
    }

    va_start(ap, fmt);
    ret = g_vasprintf(&buf, fmt, ap);
    va_end(ap);

    if (ret == -1) {
	log_write("%s(%i): %s", __FILE__, __LINE__, strerror(ENOMEM));
	return FALSE;
    }

    a[len++] = buf;
    a[len] = NULL;
    *array = a;
    return TRUE;
}

gchar **strvcatv(gchar **dst, gchar **src)
{
    gchar **p;
    gint i;
    gchar **d;

    if (!src)
	return NULL;

    d = g_strdupv(dst);

    if (!d)
	return NULL;

    i = g_strv_length(d);

    for (p = src; *p; p++) {
	gchar **pa;

	pa = g_realloc(d, (i + 2) * sizeof(gchar *));

	if (!pa) {
	    g_strfreev(d);
	    return NULL;
	}

	d = pa;
	d[i] = g_strdup(*p);

	if (!d[i]) {
	    g_strfreev(d);
	    return NULL;
	}

	d[++i] = NULL;
    }

    return d;
}

gboolean valid_filename(const gchar *filename)
{
    const gchar *p;

    if (!filename || !*filename)
	return FALSE;

    if (!strcmp(filename, "-"))
	return FALSE;

    for (p = filename; *p; p++) {
	if (*p == '/' || isspace(*p))
	    return FALSE;
    }

    return TRUE;
}

gchar *print_fmt(gchar *buf, gsize len, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    g_vsnprintf(buf, len, fmt, ap);
    va_end(ap);
    return buf;
}

gboolean contains_whitespace(const gchar *str)
{
    const gchar *p = str;
    gunichar c;
    glong len;

    len = g_utf8_strlen(p++, -1) -1;

    while (len--) {
	c = g_utf8_get_char(p++);

	if (g_unichar_isspace(c))
	    return TRUE;
    }

    return FALSE;
}

gchar *expand_homedir(gchar *str)
{
    gchar *p = str;

    if (*p++ == '~')
	return g_strdup_printf("%s%s", g_get_home_dir(), p);

    return g_strdup(str);
}

gchar *strip_char(gchar *str, gchar c)
{
    gchar *p, *p2;

    for (p = p2 = str; *p; p++) {
	if (*p == c)
	    continue;

	*p2++ = *p;
    }

    *p2 = 0;
    return str;
}

gchar *tohex(const guchar *str, gsize len)
{
    gchar *p;
    gint i;

    if (!str || len <= 0)
	return NULL;

    p = g_malloc0(len*2+1);

    if (!p)
	return NULL;

    for (i = 0; i < len; i++) {
	gchar buf[3];

	g_sprintf(buf, "%02x", str[i]);
	strcat(p, buf);
    }

    return p;
}

gchar **split_input_line(gchar *str, gchar *delim, gint n)
{
    if (!str || !*str)
	return NULL;

    return g_strsplit(str, delim, n);
}

gpg_error_t parse_options(gchar **line, struct argv_s *args[], gpointer data)
{
    gchar *p = *line;
    gpg_error_t rc = 0;

    for (; p && *p; p++) {
	while (g_ascii_isspace(*p))
	    p++;

	if (!*p)
	    break;

	if (*p == '-' && *(p+1) == '-') {
	    p += 2;
	    gchar opt[255] = {0}, value[255] = {0};
	    gchar *tp;
	    guint len;

	    for (tp = opt, len = 0; *p; p++, len++) {
		if (len+1 == 255)
		    return GPG_ERR_LINE_TOO_LONG;

		if (*p == ' ' || *p == '=') {
		    gboolean inquote = FALSE;

		    *tp = 0;

		    if (*p == '=') {
			p++;

			for (tp = value, len = 0; *p; p++, len++) {
			    if (len+1 == 255)
				return GPG_ERR_LINE_TOO_LONG;

			    if (*p == '\"') {
				inquote = !inquote;
				continue;
			    }
			    else if (*p == ' ' && !inquote)
				break;

			    *tp++ = *p;
			}

			*tp = 0;
		    }

		    break;
		}

		*tp++ = *p;
	    }

	    *tp = 0;
	    gboolean match = FALSE;

	    for (int i = 0; args[i]; i++) {
		if (!g_strcmp0(args[i]->opt, opt)) {
		    if (args[i]->type == OPT_NOARG && *value)
			return GPG_ERR_SYNTAX;
		    else if (args[i]->type == OPT_ARG && !*value)
			return GPG_ERR_SYNTAX;

		    rc = args[i]->func(data, value);

		    if (rc)
			return rc;

		    log_write2("OPTION='%s' VALUE='%s'", opt, value);
		    match = TRUE;
		    break;
		}
	    }

	    if (!match)
		return GPG_ERR_UNKNOWN_OPTION;

	    if (!*p)
		break;

	    continue;
	}

	break;
    }

    *line = p;
    return rc;
}
