;/*
scanning program for teh Technotrend DVB-s PCI card.

Copyright (C) 2000  Jan Mourer
	jan@panteltje.demon.nl

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <pwd.h>

#include "pscan.h"

#include "dvb.h"
#include "indentifier.h"

#include <pthread.h>
#include <sys/poll.h>


#define ASTRA_START			11798
#define ASTRA_END			12750
#define ASTRA_STEP			5
#define ASTRA_SYMBOL_RATE	22000

#define HOTBIRD_START		10700
#define HOTBIRD_END			12570
/*
looked up hotbird 1
minimum spacings are (half shift opposite polarity):
(spacing 1(41.5), 2(38.42), 3(38), 4(38.36)
*/
#define HOTBIRD_STEP		15 
#define HOTBIRD_SYMBOL_RATE	27500


#define NO_SCAN			0
#define ASTRA_SCAN		1
#define HOTBIRD_SCAN	2
#define MAN_SCAN		3


int debug_flag;
int symbolrate;
char polarization;
int frequency;
int sat_scan_mode;
char *home_dir;
int start_frequency;
int end_frequency;
int frequency_step;

#define MAXDVBAPI 10
#define VIDEODEV "/dev/video"


char *strsave(char *s) /* save char array s somewhere */
{
char *p;

/* argument check */
if(! s) return 0;

p = malloc(strlen(s) +  1);
if(p) strcpy(p, s);
return(p);
}/* end function strsave */


int SetChannel(int videoDev,\
	int FrequencyMHz, char Polarization, int Diseqc, int Srate,\
	int Vpid, int Apid, int Tpid, int Ca, int Pnr)
{
unsigned int freq;

if(debug_flag)
	{
	fprintf(stdout,\
	"SetChannel(): arg\n\
	FrequencyMHz=%d Polarization=%c Diseqc=%d Srate=%d\n\
	Vpid=%d Apid=%d Tpid=%d Ca=%d Pnr=%d\n",\
	FrequencyMHz, Polarization, Diseqc, Srate,\
	Vpid, Apid, Tpid, Ca, Pnr);
	}

if (videoDev >= 0)
	{
	struct frontend front;
	ioctl(videoDev, VIDIOCGFRONTEND, &front);

	freq = FrequencyMHz;
	front.ttk = (freq < 11700UL) ? 0 : 1;
	if (freq < 11700UL) freq -=  9750UL;
	else freq -= 10600UL;
	front.channel_flags = Ca ? DVB_CHANNEL_CA : DVB_CHANNEL_FTA;
	front.pnr       = Pnr;
	front.freq      = freq * 1000000UL;
	front.diseqc    = Diseqc;
	front.srate     = Srate * 1000;
	front.volt      = (Polarization == 'v' || Polarization == 'V') ? 0 : 1;
	front.video_pid = Vpid;
	front.audio_pid = Apid;
	front.tt_pid	= Tpid;
	front.fec       = 8;
	front.AFC       = 1;
	front.channel_flags = DVB_CHANNEL_FTA; /* (0) else DVB_CHANNEL_CA (1) */

	ioctl(videoDev, VIDIOCSFRONTEND, &front);

	if (front.sync & 0x1F == 0x1F)
		{
//		fprintf(stdout, "SetChannel(): set\n");

        return 1;
		}

//	esyslog(LOG_ERR,\
//	"ERROR: channel not sync'ed (front.sync=%X)!", front.sync);

//	fprintf(stdout,\
//	"channel not sync'ed (front.sync=%X)!\n",\
//	front.sync);
	}

return 0;
}/* end function SetChannel */


main(int argc, char **argv)
{
int a, i, f, r;
struct video_capability cap;
char temp[1024];
char *station;
int diseqc;
int vpid;
int apid;
int tpid;
int ca;
int pnr;
int selected_channel;
char channel_name[100];
char temp2[100];
struct passwd *userinfo;
int man_flag;
char tin1[80], tin2[80], tin3[80], tin4[80], tin5[80], tin6[80], tin7[80];
int sort_mode;
FILE *pptr;

/*select con io 1 byte at the time*/
setbuf(stdout,NULL);
setbuf(stdin,NULL);

iopl(3);

#ifdef PCTV1000
tv1000_ini();
#endif

sort_mode = SORT_NO;
sat_scan_mode = NO_SCAN;
channel_name[0] = 0;
while(1)
	{
	a = getopt(argc, argv, "abc:de:f:hp:r:s:");
	if(a == -1) break;
	switch(a)
		{
		case 'a':
			sat_scan_mode = ASTRA_SCAN;
			break;
		case 'b':
			sat_scan_mode = HOTBIRD_SCAN;
			break;
//        case 'c':
//       	sscanf(optarg, "%500s", channel_name);
//        	break;
        case 'd': /* not to conflict with -d display option for x */
			debug_flag = 1;
			break;
		case 'e':
			end_frequency = atoi(optarg);
			sat_scan_mode = MAN_SCAN;
			break;
		case 'f':
			start_frequency = atoi(optarg);
			sat_scan_mode = MAN_SCAN;
			break;
		case 'h':
			fprintf(stdout,\
"Usage:\n\
pscan [-a] [-b] [-e scan end] [-f scan start] [-d] [-h] [-r alpha / freq]
[-p polarization] [-s symbolrate]\n\
\t-a  scan astra and exit (~/programs.dat and ~/channels.conf).\n\
\t-b  scan hotbird and exit (~/programs.dat and ~/channels.conf).\n\
\t-e  scan end frequency MHz.\n\
\t-f  scan start frequency MHz.\n\
\t-d  send debug output to console.\n\
\t-h  help (this help).\n\
\t-p  polarization (h or v).\n\
\t-r  [alpha or freq] sort list alphabetically or on frequency and exit.\n\
\t-s  symbolrate.\n
");
			exit(0);	
			break;
		case 'p':
			polarization = optarg[0];
			break;
		case 'r':
			if(strncasecmp(optarg, "alpha", 2) == 0)
				{
				sort_mode = SORT_ALPHA;
				}
			else if(strncasecmp(optarg, "freq", 2) == 0)
				{
				sort_mode = SORT_FREQUENCY;
				}
			break;
		case 's':
			symbolrate = atoi(optarg);
			break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stdout, "Unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stdout, "Unknown option character `\\x%x'.\n",\
				optopt);
				}
			exit(1);
			break;			
		default:
			break;
		}/* end switch a */
	}/* end while getopt() */

fprintf(stdout, "\nPanteltje pscan-%s\n\n", PSCAN_VERSION);

userinfo = getpwuid(getuid() );
if(! userinfo)
	{
	fprintf(stdout, "could not get userinfo\n");
	exit(1);
	}

/* get home directory */ 
home_dir = strsave(userinfo -> pw_dir);
if(! home_dir)
	{
	fprintf(stdout, "could not allocate space for home_dir\n");
	exit(1);
	}

/* this if we do sorting of an existing file */
if(sort_mode != SORT_NO)
	{
	/* load the existing file */
	sprintf(temp, "%s/channels.conf", home_dir);
	if(! load_programs_vdr_format(temp) ) return 0;

	if(! sort_program_list(sort_mode) ) exit(1);
	
	/* create a VDR program list */
	if(! save_programs_vdr_format() ) return 0;

	fprintf(stdout, "Ready\n");

	exit(0);
	}

/* connect to video device */
for(i = 0; i < 10; i++)
	{
	sprintf(temp, "%s%d", VIDEODEV, i);
	if(access(temp, F_OK | R_OK | W_OK) != 0)
		{
		fprintf(stdout, "main(): could not access device %s\n", temp);
		perror("access");
		exit(1);
		}

	f = open(temp, O_RDWR);
	if(f < 0) 
		{
		fprintf(stdout,\
		"main(): could not open device %s for read_write\n", temp);
exit(1);		
		}

	r = ioctl(f, VIDIOCGCAP, &cap);

	if(debug_flag)
		{
		fprintf(stdout, "r=%d dev=%s\n", r, temp);
		}

	if(r == 0 && (cap.type & VID_TYPE_DVB) )
//	if(r == 0 && (cap.type & VID_TYPE_TELETEXT) )
		{
		fprintf(stdout, "connected to %s\n", temp);

		/* these are command line option for from within a script */
		if(sat_scan_mode == ASTRA_SCAN)
			{
			scan_channels(f, ASTRA_SCAN);

			if(! sort_program_list(SORT_ALPHA) ) exit(1);

			fprintf(stdout, "Ready\n");
	
			exit(0);
			}
		else if(sat_scan_mode == HOTBIRD_SCAN)
			{
			scan_channels(f, HOTBIRD_SCAN);

			fprintf(stdout, "Ready\n");

			exit(0);
			}	
		else if(sat_scan_mode == MAN_SCAN)
			{
			scan_channels(f, MAN_SCAN);

			fprintf(stdout, "Ready\n");

			exit(0);
			}	

#define _ZDF "ZDF:11954:h:1:27500:110:120:0:0:0"
#define _CNN "CNN:12168:v:1:27500:165:100:0:0:0"
#define _3SAT "3sat:11954:h:1:27500:210:220:0:0:0"
#define _VISIONS_CA "Visions:12092:h:1:27500:4416:4417:0:1:0"
#define _CMILAGNO "C.MILAGNO:12092:h:1:27500:4368:4369:0:0:0"
#define _TELESIERA "TELESIERA:12092:h:1:27500:4160:4161:0:0:0"
#define _SATISFACTION "SATISFACTION:12092:h:1:27500:4192:4193:0:0:0"
#define _RCT "RCT:12092:h:1:27500:4520:4520:0:0:0"
#define _TVGALICA "TVGALICA:12092:h:1:27500:4224:4225:0:0:0"
#define _FIESTA "FIESTA:12092:h:1:27500:4432:4433:0:0:0"
#define _VTV "VTV:12092:h:1:27500:0:0:0:0:0"
#define _TVE "TVE:12092:h:1:27500:4208:4209:0:0:0"
#define _TELE24 "Tele 24:12379:v:1:27500:3023:3033:3500:0:0"
#define _MOU2 "MOU.2:12475:h:1:27500:42:43:0:0:0"
#define _TTEST "TELETEXT TEST RAI 1:11765:v:1:27500:160:80:3401:0:0"
#define _VISIONS "Visions:12092:h:1:27500:4416:4417:0:0:0"

		a = -1;
		selected_channel = -1;
		while(1)
			{
			fprintf(stdout,\
			"\nMenu  (Use pscan -h for help on command line options)\n\n");
			fprintf(stdout, "-2  %s\n", "scan channels hotbird");
			fprintf(stdout, "-1  %s\n", "scan channels astra");
			fprintf(stdout, "0  %s\n", "exit");
#ifdef PCTV1000
			fprintf(stdout, "1  %s\n", "computer");
			fprintf(stdout, "2  %s\n", "video");
#endif
			fprintf(stdout, "3  %s\n", _3SAT);
			fprintf(stdout, "4  %s\n", _VISIONS_CA);
			fprintf(stdout, "5  %s\n", _CMILAGNO);
			fprintf(stdout, "6  %s\n", _TELESIERA);
			fprintf(stdout, "7  %s\n", _SATISFACTION);
			fprintf(stdout, "8  %s\n", _RCT);
			fprintf(stdout, "9  %s\n", _TVGALICA);
			fprintf(stdout, "10 %s\n", _FIESTA);
			fprintf(stdout, "11 %s\n", _VTV);
			fprintf(stdout, "12 %s\n", _TVE);
			fprintf(stdout, "13 %s\n", _TELE24);
			fprintf(stdout, "14 %s\n", _MOU2);
			fprintf(stdout, "15 %s\n", _TTEST);
			fprintf(stdout, "16  %s\n", _VISIONS);
			fprintf(stdout, "98 %s\n", "manual parse from output");
			fprintf(stdout, "99 %s\n", "manual 27500 v 11765 160 80 3401");

			if(a > 2) selected_channel = a;
			if(selected_channel > 0)
				{
				fprintf(stdout, "selected=%d %s\n", selected_channel, temp);
				}

			fprintf(stdout, "Enter choice\n");
			fscanf(stdin, "%10s", temp2);
			a = atoi(temp2);

			man_flag = 0;
			switch(a)
				{
				case -2: /* scan hotbird */
					scan_channels(f, HOTBIRD_SCAN);
					break;
				case -1: /* scan astra */
					scan_channels(f, ASTRA_SCAN);
					break;
				case 0: /* exit */
					fprintf(stdout, "User abort\n");
					exit(0);
					break;
#ifdef PCTV1000
				case 1: /* select computer if using the Philips PCTV1000
							analog terrestial card */
					tv1000_set_screen(0);
//					sprintf(temp, "%s", "computer selected");
					continue;
					break;
				case 2: /* select AUX and full screen mode on on the Philps
						PCTV1000 */
					tv1000_set_screen(1);
					continue;
					break;
#endif
				case 3:
					sprintf(temp, "%s", _3SAT);
					break;
				case 4:
					sprintf(temp, "%s", _VISIONS_CA);
					break;
				case 5:
					sprintf(temp, "%s", _CMILAGNO);
					break;
				case 6:
					sprintf(temp, "%s", _TELESIERA);
					break;
				case 7:
					sprintf(temp, "%s", _SATISFACTION);
					break;
				case 8:
					sprintf(temp, "%s", _RCT);
					break;
				case 9:
					sprintf(temp, "%s", _TVGALICA);
					break;
				case 10:
					sprintf(temp, "%s", _FIESTA);
					break;
				case 11:
					sprintf(temp, "%s", _VTV);
					break;
				case 12:
					sprintf(temp, "%s", _TVE);
					break;
				case 13:
					sprintf(temp, "%s", _TELE24);
					break;
				case 14:
					sprintf(temp, "%s", _MOU2);
					break;
				case 15:
					sprintf(temp, "%s", _TTEST);
					break;
				case 16:
					sprintf(temp, "%s", _VISIONS);
					break;
				case 98: /* manual in, but be a cut and paste entry from
							channels.conf */
					man_flag = 1;
					/*
					Format accepted:
					TVE Internacional:12100:h:0:27500:4208:4500:0:8707
					*/

					fprintf(stdout,\
					"Enter output format (use cut & paste)\n");

					fscanf(stdin,\
					"%a[^:]:%d:%c:%d:%d:%d:%d:%d:%d:%d",\
					&station, &frequency, &polarization, &diseqc,\
					&symbolrate, &vpid, &apid, &tpid, &ca, &pnr);

					break;
				case 99: /* manual in, asks for parameters */ 
					man_flag = 1;
					fprintf(stdout,\
					"symbolrate polarization frequency vpid apid pnr?\n");
					
					fscanf(stdin, "%d %c %d %d %d %d",\
					&symbolrate, &polarization, &frequency,\
					&vpid, &apid, &pnr);

					diseqc = 0;
					ca = 0;

					station = strsave("man station");
					if(! station) exit(1);				

					fprintf(stdout, "\n");

					break;
				default:
					fprintf(stdout, "Invalid choice\n");
					break;
				}/* end switch */

			if(! man_flag)
				{
				sscanf(temp, "%a[^:]:%d:%c:%d:%d:%d:%d:%d:%d:%d",\
				&station, &frequency, &polarization, &diseqc, &symbolrate,\
				&vpid, &apid, &tpid, &ca, &pnr);
				}

			fprintf(stdout, "station=%s\n", station);
			fprintf(stdout, "freq=%d\n", frequency);
			fprintf(stdout, "pol=%c\n", polarization);
			fprintf(stdout, "diseqc=%d\n", diseqc);
			fprintf(stdout, "srate=%d\n", symbolrate);
			fprintf(stdout, "vpid=%d\n", vpid);
			fprintf(stdout, "apid=%d\n", apid);
			fprintf(stdout, "tpid=%d\n", tpid);
			fprintf(stdout, "ca=%d\n", ca);
			fprintf(stdout, "pnr=%d\n", pnr);

			SetChannel(\
				f,					//videoDev
				frequency,			//FrequencyMHz
				polarization,		//char Polarization
				diseqc,				//int Diseqc
				symbolrate,			//int Srate
				vpid,				//int Vpid
				apid,				//int Apid
				tpid,				//int Tpid
				ca,					//int Ca 
				pnr					//int Pnr
				);

			print_signal_strength(f);
			}/* end while user input */

		/* close video device */
		close(f);
		}/* end if f OK */
	else
		{
//		fprintf(stdout, "BAD %s\n", temp);
		}


	}/* end for i */

exit(0);
}/* end function main */


int print_signal_strength(int videoDev)
{
int a, c, i;
int top;
struct frontend front;
double dstep, deebee;
char temp[128];
char temp1[20];
int signal_quality;
double quality;

if(debug_flag)
	{
	fprintf(stdout,\
	"print_signal_strength(): arg videoDev=%d\n", videoDev);
	}

/* this how many stars maximum in display */
#define MAX_SCALE	30

/*
the agc value is now in front.agc (16 bits integer),
the stronger the signal, the more the agc (automatic gain control)
drops.

65280 = 0dB no signal	  0 % scale
19000 = 10 dB
	0 = ?? dB			100 % scale
*/

/* if 19000 = 10dB and 65280 = 0dB then 1 dB = */
dstep = (65280.0 - 19000.0) / 10.0;
	
/* prevent devide by zero later on */
if(front.agc == 65535) front.agc = 1;

while(1)
	{
	if(c == 27) return 1; // ESCAPE exits

	ioctl(videoDev, VIDIOCGFRONTEND, &front);  

	top = 65535 - front.agc;
	top = top * (MAX_SCALE / 65536.0);

	// scale is linear due to agc log
	deebee =  (65535 - front.agc) / dstep ;  
	
	signal_quality = front.nest; 
	quality = (65535 - signal_quality) / 655.35;	

	/* display */
	temp[0] = 0;
	sprintf(temp1, "%4.1f dB q=%2.0f", deebee, quality);
	for(i = 0; i < MAX_SCALE; i++)
		{
		if(i < top) strcat(temp, "*");
		else strcat(temp, " ");
		}		
	 strcat(temp, temp1);

	fprintf(stdout, "%s", temp);

break;
	fprintf(stdout, "\r");
	}/* end while display signal strength loop */

return 1;
}/* end function print signal_strength */


int test_lock(int videoDev)
{
struct frontend front;
int signal, viterbi, carrier, frame_lock;

if(debug_flag)
	{
	fprintf(stdout, "test_lock(): arg videoDev=%d\n", videoDev);
	}

if (videoDev >= 0)
	{
	ioctl(videoDev, VIDIOCGFRONTEND, &front);
	// for a channel found we need signal, viterbi, carrier, frame_lock
	
	signal = front.sync & DVB_SYNC_SIGNAL;
	viterbi = front.sync & DVB_SYNC_VITERBI;
	carrier = front.sync & DVB_SYNC_CARRIER;
	frame_lock = front.sync & DVB_SYNC_FSYNC;
	
//	fprintf(stdout, "Signal=%d Viterbi=%d Carrier=%d framelock=%d\n",\
//	signal, viterbi, carrier, frame_lock); 
	
	if(signal) fprintf(stdout, "S");
	else fprintf(stdout, "s");

	if(viterbi) fprintf(stdout, "V");
	else fprintf(stdout, "v");

	if(carrier) fprintf(stdout, "C");
	else fprintf(stdout, "c");

	if(frame_lock) fprintf(stdout, "F");
	else fprintf(stdout, "f");
	
	if(signal && viterbi && carrier && frame_lock) return 1;
	}	

return 0;
}


#define MAXSECSIZE 32768

__u16 SetFilter(int videoDev, __u16 pid, __u16 section, __u16 mode)
{ 
struct bitfilter filt =\
	{
	pid,
	{ section, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,  
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000},
	mode,0,
	FILTER_MEM,
	{},
	};

if(debug_flag)
	{
	fprintf(stdout,\
	"SetFilter(): arg videoDev=%d pid=%d section=%d mode=%d\n",\
	videoDev, pid, section, mode);
	}

if (ioctl(videoDev, VIDIOCSBITFILTER, &filt) < 0) return 0xffff;

return 0;
}/* end function SetFilter */


int CloseFilter(int videoDev, __u16 handle)
{
if(debug_flag)
	{
	fprintf(stdout,\
	"CloseFilter(): arg videoDev=%d handle=%d\n",\
	videoDev, handle);
	}

if (ioctl(videoDev, VIDIOCSSHUTDOWNFILTER, &handle) < 0) return -1;

return 0;
}/* end function CloseFilter */
        

int GetSection(\
	int videoDev,\
	unsigned char *buf, ushort table, unsigned char filter,\
	unsigned char section,\
	unsigned char *msecnum)
{
int seclen = 0;
_u16 handle, table_id;
unsigned char filter_id, section_id = 0xff, maxsec = 0;
struct pollfd pfd;
int a1, a2;
int i;
		
if(debug_flag)
	{
	fprintf(stdout,\
	"GetSection(): arg videoDev=%d buf=%lu table=%d\n\
	filter=%d section=%d *msecnum=%d\n",\
	videoDev, buf, table, filter, section, *msecnum);
	}

if ((handle = SetFilter(\
	videoDev, table, (filter << 8) | 0x00ff, 0) ) == 0xffff)
	{
	return -1;
	}
do
	{
	seclen = 0;
	pfd.fd = videoDev;
	pfd.events = POLLIN;
	if (poll(&pfd, 1, 20000) == 0)
//	if (poll(&pfd, 1, 200) == 0)
		{
//		cerr << "Timeout\n";

		if(debug_flag)
			{
			fprintf(stdout, "GetSection(): timeout poll\n");
			}

		break;
		}
	a1 = read(videoDev, buf, 8);
	if(debug_flag)
		{
		fprintf(stdout, "a1\n");
		for(i = 0; i < a1; i++)
			{
			fprintf(stdout, "%d ", buf[i]);
			}
		fprintf(stdout, "\n");
		}

	seclen = (buf[6] << 8) | buf[7];
	seclen &= 0xfff;

	table_id = (buf[4] << 8) | buf[5];

	a2 = read(videoDev, buf, seclen);
	if(debug_flag)
		{
		fprintf(stdout, "a2\n");
		for(i = 0; i < a2; i++)
			{
			fprintf(stdout, "%d ", buf[i]);
			}
		fprintf(stdout, "\n");
		}

	filter_id = buf[0];
	section_id = buf[6];
	maxsec = buf[7];

	}
while (table_id != table || filter_id != filter || section_id != section);

*msecnum = maxsec;

if(debug_flag)
	{
	fprintf(stdout,\
	"GetSection(): a1=%d a2=%d\n\
	seclen=%d\n\
	table_id=%d\n\
	filter_id=%d\n\
	section_id=%d\n\
	maxsec=%d\n\
	*msecnum=%d\n",\
	a1, a2, seclen, table_id, filter_id, section_id, maxsec, *msecnum);
	}

CloseFilter(videoDev, handle);

return seclen;
}/* end function GetSection */


int GetPAT(int videoDev, dvb_pat_t *pat)
{
unsigned char buf[MAXSECSIZE];
unsigned char msec = 0, sec = 0;
int ret = -1;

if(debug_flag)
	{
	fprintf(stdout, "GetPAT(): arg videoDev=%d pat=%lu\n",\
	videoDev, pat);
	}

while(sec <= msec)
	{
	if(debug_flag)
		{
		fprintf(stdout, "GetPAT(): in while\n");
		}

	if (GetSection(videoDev, buf, PID_PAT, TID_PAT, sec, &msec) > 0)
		ret &= parse_pat(buf, pat);
	sec++;
	}

return ret;
}/* end function GetPAT */


int GetNIT(int videoDev, dvb_nit_t *nit)
{
unsigned char buf[MAXSECSIZE];
unsigned char sec=0, msec=0;
int ret = -1;

if(debug_flag)
	{
	fprintf(stdout,\
	"GetNIT(): arg videoDev=%d nit=%lu\n",\
	videoDev, nit);
	}

while (sec <= msec)
	{
	if( GetSection(videoDev, buf, PID_NIT, TID_NIT_ACT, sec, &msec) > 0)
		ret &= parse_nit(buf, nit);

	//hdump(buf, 0x120);

	sec++;
	}

return ret;
}/* end function GetNIT */


int GetPMT(int videoDev, dvb_pmt_t *pmt, unsigned short pid)
{
unsigned char buf[MAXSECSIZE];
unsigned char sec = 0, msec = 0;
int ret = -1;
int i;

if(debug_flag)
	{
	fprintf(stdout,\
	"GetPMT(): arg videoDev=%d pmt=%d pid=%d\n",\
	videoDev, pmt, pid);
	}

while(sec <= msec)
	{
	if(debug_flag)
		{
		fprintf(stdout,\
		"GetPMT(): pid=%d sec=%d msec=%d\n",\
		pid, sec, msec);
		}

	if( GetSection(videoDev, buf, pid, TID_PMT, sec, &msec) > 0)
        ret &= parse_pmt(buf, pmt);

	if(debug_flag)
		{
		fprintf(stdout, "PMT sec=%d msec=%d\n", sec, msec);
		for(i = 0; i < 100; i++)
			{
			fprintf(stdout, "%d ", buf[i]);
			}
		fprintf(stdout, "\n");
		}

//	if( GetSection(videoDev, buf, pid, TID_CAT, sec, &msec) > 0)
//        ret &= parse_pmt(buf, pmt);

//	if(debug_flag)
//		{
//		fprintf(stdout, "CAT sec=%d msec=%d\n", sec, msec);
//		for(i = 0; i < 100; i++)
//			{
//			fprintf(stdout, "%d ", buf[i]);
//			}
//		fprintf(stdout, "\n");
//		}

//	if( GetSection(videoDev, buf, pid, TID_PAT, sec, &msec) > 0)
//       ret &= parse_pmt(buf, pmt);

	sec++;
	}

return ret;
}/* end function GetPMT */


int GetSDT(int videoDev, dvb_sdt_t *sdt)
{
unsigned char buf[MAXSECSIZE];
unsigned char msec=0, sec=0;
int ret = -1;
		
if(debug_flag)
	{
	fprintf(stdout,\
	"GetSDT(): arg videoDev=%d sdt=%lu\n",\
	videoDev, sdt);
	}

while(sec <= msec)
	{
	if(GetSection(videoDev, buf, PID_SDT, TID_SDT_ACT, sec, &msec) > 0)
		ret &= parse_sdt(buf, sdt);
	sec++;
	}

return ret;
}/* end function GetSDT */


int scan_transponder(int videoDev)
{
dvb_sdt_t sdt;
dvb_pat_t pat;
dvb_pat_prog_t *prog;
int pmtnum = 0;
char temp[1024];
int transponder_frequency;

if(debug_flag)
	{
	fprintf(stdout,\
	"scan_transponderr(): arg videoDev=%d\n",\
	videoDev);
	}

memset(&pat, 0, sizeof(pat) );

if (GetPAT(videoDev, &pat) < 0) return 0;
if(debug_flag)
	{
	fprintf(stdout, "scan_transponder(): have PAT\n");
	}

fprintf(stdout, "PAT\n");

prog = pat.prog;
while(prog)
	{
	if (!prog->prog_nr)
		{
        dvb_nit_t *nit = (dvb_nit_t *)calloc (1, sizeof (dvb_nit_t) );
		if( GetNIT(videoDev, nit) >= 0)
			{
			if(debug_flag) 
				{
				fprintf(stdout, "scan_transponder(): have NIT\n");
				}	

	        if (prog->prog.nit) free(prog->prog.nit);
			prog->prog.nit = nit;
			}
		else free(nit);

		fprintf(stdout, "NIT\n");
//		osd.Text(64, 120, 2, 1, "NIT");
//		cerr << "NIT\n";

		// FIXME: fix NIT parsing 
		//if (front.type != FRONT_DVBC)

//		tp.freq = (__u32) (1000000 * nit->freq + 0.5);

		if(debug_flag)
			{
			fprintf(stdout,\
			"freq=%d\n",\
			(__u32) (1000000 * nit->freq + 0.5) );
			}

		}
	else
		{
		dvb_pmt_t *pmt = (dvb_pmt_t *)calloc (1, sizeof(dvb_pmt_t) );
		pmt = (dvb_pmt_t *)calloc (1, sizeof(dvb_pmt_t) );

		if(debug_flag)
			{
			fprintf(stdout, "scan_transponder(): prog_nr=%d pid=%d\n",\
			prog->prog_nr, prog->pid);
			}
/*__________________________*/
		if (GetPMT(videoDev, pmt, prog->pid) >= 0)
			{
			if(debug_flag)
				{
				fprintf(stdout, "scan_transponder(): have PMT\n");
				}

			pmtnum++;
			if (prog->prog.pmt) free (prog->prog.pmt);
			prog->prog.pmt = pmt;
			}
		else free (pmt);
		}
	prog = prog->next;
	}

fprintf(stdout, "PMT\n");
//osd.Text(128, 120, 2, 1, "PMT");

//cerr << "freq:" << tp.freq << endl;

/* Ignore TPs with no PMT or SDT */
if (!pmtnum)
	{
	free_pat(&pat);
	return -1;
	}
memset(&sdt, 0, sizeof(dvb_sdt_t) );
if (GetSDT(videoDev,  &sdt) < 0)
	{
	free_pat(&pat);
	return -1;
	}
if(debug_flag)
	{
	fprintf(stdout, "scan_transponder(): have SDT\n");
	}

fprintf(stdout, "SDT\n");
//osd.Text(196, 120, 2, 1, "SDT");

transponder_frequency = 0;
/* print to screen and fill database */
/* print_al() gets the true transponder frequency */
print_all (&pat, &sdt, &transponder_frequency);

/* if print_all() got valid data */
if(transponder_frequency != 0)
	{
	/*
	We came from a frequncy <= then the transponder (scan),
	So now it is safe to start at the transponder, and have a good guess for 
	the next transponder with the same polarity being 38 MHz higher.
	*/
	/* set the global */
//	frequency = transponder_frequency + 38; 
	}

return 1;
}/* end function scan_transponder */


int scan_channels(int videoDev, int scan_mode)
{
int i, vbiDev;
int diseqc;
int vpid;
int apid;
int tpid;
int ca;
int pnr;
struct frontend front;
char temp[1024];
FILE *fptr;

if(debug_flag)
	{
	fprintf(stdout,\
	"scan_channels(): arg videoDev=%d scan_mode=%d\n",\
	videoDev, scan_mode);
	}

#define VBIDEV "/dev/vbi"


for(i = 0; i < 10; i++)
	{
	sprintf(temp, "%s%d", VBIDEV, i);
	if(access(temp, F_OK | R_OK | W_OK) != 0)
		{
		fprintf(stdout,\
		"scan_channels(): could not access device %s\n", temp);
		perror("access");
//		exit(1);
		}

	vbiDev = open(temp, O_RDWR);
	if(vbiDev < 0) 
		{
		fprintf(stdout,\
		"scan_channels(): could not open device %s for read_write\n", temp);
		}
	else
		{
		break;
		}
	}/* end for all vbi devices */

fprintf(stdout, "using %s\n", temp);

if(scan_mode == ASTRA_SCAN)
	{
	fprintf(stdout, "astra channel scan\n");

	polarization = 'h';
	symbolrate = ASTRA_SYMBOL_RATE;
	start_frequency = ASTRA_START;
	end_frequency = ASTRA_END;
	frequency_step = ASTRA_STEP;
	}
else if(scan_mode == HOTBIRD_SCAN)
	{
	fprintf(stdout, "hotbird channel scan\n");

	polarization = 'h';
	symbolrate = HOTBIRD_SYMBOL_RATE;
	start_frequency = HOTBIRD_START;
	end_frequency = HOTBIRD_END;
	frequency_step = HOTBIRD_STEP;
	}
else if(scan_mode == MAN_SCAN)
	{
	fprintf(stdout, "manual channel scan\n");


	frequency_step = HOTBIRD_STEP;
	}
else
	{
	fprintf(stdout, "scan_channels(): unknow scan_mode %d\n", scan_mode);

	return 0;
	}

apid = 0;
vpid = 0;
tpid = 0;
ca = 0;
pnr = 0;
diseqc = 1;
while(1) /* all polarizations */
	{
	frequency = start_frequency;
	while(1)/* all frequencies */
		{
		fprintf(stdout, "\n");

		fprintf(stdout, "%d %c %d ", symbolrate, polarization, frequency);

		SetChannel(\
			videoDev,			//videoDev
			frequency,			//FrequencyMHz
			polarization,		//char Polarization
			diseqc,				//int Diseqc
			symbolrate,			//int Srate
			vpid,				//int Vpid
			apid,				//int Apid
			tpid,				//int Tpid
			ca,					//int Ca 
			pnr					//int Pnr
			);

		print_signal_strength(videoDev);

		fprintf(stdout, " ");

		if(test_lock(videoDev) )
			{
			fprintf(stdout, " LOCK");

			ioctl(videoDev, VIDIOCGFRONTEND, &front);
//	if ((front.sync & 0x07) == 0x07) frequency = front.curfreq / 1000000; 

			fprintf(stdout, "\n");

			/* this also adds entries to the data base */
			scan_transponder(vbiDev);
			}

		frequency += frequency_step;
		if(frequency > end_frequency) break;
		}/* end while all frequencies */

	if(scan_mode == MAN_SCAN) break;

	if(polarization == 'v') break;
	if(polarization == 'h') polarization = 'v';
	}/* end while all polarizations */

fprintf(stdout, "\nReady scanning and filling database\n");

/* sort the program list alphabetically */
if(! sort_program_list(SORT_ALPHA) ) return 0;
	
/* create a VDR program list */
if(! save_programs_vdr_format() ) return 0;

/* create gVideo program list */
if(! save_programs_gvideo_format() ) return 0;

return 1;
}/* end function scan_channels */


