/*
pscan scanner for the Technotrend DVB-s card
pscan is registered Copyright (C) 2000 <Jan Mourer>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define TEMP_SIZE 1024
#define READSIZE 65535

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <pwd.h>

#include "pscan.h"

#include "dvb.h"
#include "indentifier.h"

#include <pthread.h>
#include <sys/poll.h>


extern int debug_flag;
extern char *home_dir;
extern char *strsave();

struct program
	{
	char *name;/* combine service name and video audio pid to get unique */
	char *service_name;
	int transponder_id;
	int orig_nw_id;
	float orbit;
	char west_east;
	char *country_avail;
	char *service_type;
	int symbolrate;
	char *fec;
	char *provider;
	int version;
	int frequency;
	char polarization;
	int disecq;
	int cpid;
	int vpid;
	int apid;
	int tpid;
	int ca;
	int pnr;	
	int lnb;
	struct program *nxtentr;
	struct program *prventr;
	};
struct program *programtab[2]; /* first element points to first entry,
			second element to last entry */


struct program *lookup_program(char *name)
{
struct program *pa;

/*pa points to next entry*/
for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)
	{
	if(strcmp(pa -> name, name) == 0) return(pa);
	}
return(0);/*not found*/
}/* end function lookup_program */


struct program *install_program_at_end_of_list(char *name)
{
struct program *plast, *pnew;

if(debug_flag)
	{
	fprintf(stdout,\
	"install_program_at_end_off_list(): arg name=%s\n", name);
	}

pnew = lookup_program(name);
if(pnew)
	{
	/* free previous definition */
/*	free(pnew -> subject);*/
	return(pnew);/* already there */
	}

/* create new structure */
pnew = (struct program *) calloc(1, sizeof(*pnew) );
if(! pnew)
	{
	fprintf(stdout,\
	"install_program_at_end_of_list(): coul not allocate space struct\n");

	return(0);
	}

pnew -> name = strsave(name);
if(! pnew -> name) return(0);

/* get previous structure */
plast = programtab[1]; /* end list */

/* set new structure pointers */
pnew -> nxtentr = 0; /* new points top zero (is end) */
pnew -> prventr = plast; /* point to previous entry, or 0 if first entry */

/* set previuos structure pointers */
if( !programtab[0] ) programtab[0] = pnew; /* first element in list */
else plast -> nxtentr = pnew;

/* set array end pointer */
programtab[1] = pnew;

return(pnew);/* pointer to new structure */
}/* end function install_program */


int delete_program(char *name)/* delete entry from double linked list */
{
struct program *pa, *pprev, *pdel, *pnext;

if(debug_flag)
	{
	fprintf(stdout, "delete_program(): arg name=%s\n", name);
	}

/* argument check */
if(! name) return(0);

pa = programtab[0];
while(1)
	{
	/* if end list, return not found */
	if(! pa) return(0);

	/* test for match in name */
	if(strcmp(name, pa -> name) != 0) /* no match */
		{
		/* point to next element in list */
		pa = pa -> nxtentr;

		/* loop for next element in list */
		continue;
		}

	/* we now know which struture to delete */
	pdel = pa;

	/* get previous and next structure */
	pnext = pa -> nxtentr;
	pprev = pa -> prventr;

	/* set pointers for previous structure */
	/* if first one, modify programtab[0] */
	if(pprev == 0) programtab[0] = pnext;
	else pprev -> nxtentr = pnext;

	/* set pointers for next structure */
	/* if last one, modify programtab[1] */
	if(pnext == 0) programtab[1] = pprev;
	else pnext -> prventr = pprev;
	
	/* delete structure */	
	free(pdel -> name);
	free(pdel -> service_name);
	free(pdel -> country_avail);
	free(pdel -> service_type);
	free(pdel -> fec);
	free(pdel -> provider);
	free(pdel); /* free structure */

	/* return OK deleted */
	return(1);
	}/* end for all structures */
}/* end function delete_program */


int delete_all_programs()/* delete all entries from table */
{
struct program *pa;

if(debug_flag)
	{
	fprintf(stdout, "delete_all_programs() arg none\n");
	}

while(1)
	{	
	pa = programtab[0];
	if(! pa) break;
	programtab[0] = pa -> nxtentr;
	free(pa -> name);
	free(pa -> service_name);
	free(pa -> country_avail);
	free(pa -> service_type);
	free(pa -> fec);
	free(pa -> provider);
	free(pa);/* free structure */
	}/* end while all structures */

programtab[1] = 0;
return(1);
}/* end function delete_all_programs */


int save_programs_vdr_format()
{
char temp[TEMP_SIZE];
struct program *pa;
int a;
FILE *fptr;
FILE *pptr;

if(debug_flag)
	{
	fprintf(stdout, "save_programs_vdr_format(): arg none\n");
	}

/* save the new file as a tmp */
sprintf(temp, "%s/channels.conf.tmp", home_dir);
fptr = fopen(temp, "w");
if(! fptr)
	{
	fprintf(stdout,\
	"save_programs_vdr_format(): could not open file %s for write\n",\
	temp);
	return 0;
	}

for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)	
	{
	/* VDR format */
	fprintf(fptr, "%s:%d:%c:%d:%d:%d:%d:%d:%d\n",\
	pa -> service_name, pa -> frequency, pa -> polarization,\
	pa -> disecq, pa -> symbolrate, pa -> vpid, pa -> apid,\
	pa -> ca, pa -> pnr);
	}	

/* remove any old .bak file */
sprintf(temp, "%s/channels.conf.bak", home_dir);
if(unlink(temp) != 0)
	{
//	fprintf(stdout, "scan_channels(): could not unlink %s\n",\
//	temp);

//	return 0;
	}

/* rename the old file to .bak */
sprintf(temp, "mv %s/channels.conf %s/channels.conf.bak",\
home_dir, home_dir);
pptr = popen(temp, "r");
pclose(pptr);

/* rename the temp file to the corrrect name */
sprintf(temp, "mv %s/channels.conf.tmp %s/channels.conf",\
home_dir, home_dir);
pptr = popen(temp, "r");
pclose(pptr);

return(1);
}/* end function save_programs_vdr_format */


int load_programs_vdr_format(char *pathfilename)
{
int a;
FILE *fptr;
char temp[READSIZE];
struct program *pa;
extern char *strsave();
char name[4096];
int frequency;
char polarization;
int disecq;
int symbolrate;
int vpid;
int apid;
int tpid;
int ca;
int pnr;
char *station_name;
struct stat *statptr;
int file_size;

if(debug_flag)
	{
	fprintf(stdout,\
	"load_programs_vdr_format() arg pathfilename=%s\n",\
	pathfilename);
	}
	
if(! pathfilename) return 0;

delete_all_programs();

errno = 0;
fptr = fopen("/root/channels.conf", "r");
//perror("");
if(! fptr)
	{
	if(debug_flag)
		{
		fprintf(stdout,\
		"load_programs_vdr_format(): could not open file %s for read\n",\
		pathfilename);
		}
	return(0);
	} 

/*
need the length of this file, to prevent generating zero output length
files from defective )zero length) channels.conf
*/
/* allocate space for structure stat */
statptr = (struct stat*) malloc(sizeof(struct stat) );
if(! statptr)
	{
	fprintf(stdout,\
	"load_programs_vdr_format(): could not alllocate space for stat\n");

	return(0);
	}

/* fill the structure with data for this file */
fstat(fileno(fptr), statptr);

/* get filesize */
file_size = statptr -> st_size;

/* free the structure */
free(statptr);

/* test the filesize */
if(file_size == 0)
	{
	fprintf(stdout,\
	"load_programs_vdr_format(): %s has zero length aborting\n",\
	pathfilename);

	return 0;
	}

/* read the file */
while(1)
	{
	a = readline(fptr, temp);/* closes file on EOF */
	if(a == EOF) return(1);/* empty file or last line */

	a = sscanf(temp,\
	"%a[^:]:%d:%c:%d:%d:%d:%d:%d:%d:%d",\
	&station_name, &frequency, &polarization, &disecq,\
	&symbolrate, &vpid, &apid, &tpid, &ca, &pnr);

//	if(a == 10)
		{
		/* create a structure entry */
		pa = install_program_at_end_of_list(temp);
		if(! pa)
			{
			if(debug_flag)
				{
				fprintf(stdout,\
				"load_programs_vdr_format(): cannot install url %s\n", temp);
				}
			return(0);
			}
		pa -> frequency = frequency;

		pa -> service_name = strsave(station_name);
		if(! pa -> service_name) return 0;

		pa -> polarization = polarization;
		pa -> disecq = disecq;
		pa -> symbolrate = symbolrate;
		pa -> vpid = vpid;
		pa -> apid = apid;
		pa -> ca = ca;
		pa -> pnr = pnr;

		}/* end if a == 10 */
	}/* end while all lines in urls.dat */

return 1;
}/* end function load_programs_vdr_format */


int readline(FILE *file, char *contents)
{
int a, c, i;

for(i = 0; i < READSIZE - 1; i++)
	{
	while(1)
		{
		c = getc(file);
		a = ferror(file);
		if(a)
			{
			perror("readline():");
			continue;
			}
		break;
		}
	if(feof(file) )
		{
		fclose(file);
		contents[i] = 0;/* EOF marker */
		return(EOF);
		}
	if(c == '\n')
		{
		contents[i] = 0;/* string termination */
		return(1);/* end of line */
		} 
	contents[i] = c;
	}
/*
mmm since we are here, the line must be quite long, possibly something
is wrong.
Since I do not always check for a return 0 in the use uf this function,
just to be safe, gona force a string termination.
This prevents stack overflow, and variables getting overwritten.
*/
contents[i] = 0;/* force string termination */
/*fclose(file);*/
if(debug_flag)
	{
	fprintf(stdout,\
	"readline(): line to long, returning 0 contents=%s\n", contents);
	}
return(0);
}/* end function readline */


int add_service(
char *name,\
char *service_name,\
int transponder_id,\
int orig_nw_id,\
float orbit,\
char west_east,\
char *country_avail,\
char *service_type,\
int symbolrate,\
char *fec,\
char *provider,\
int version,\
int frequency,\
char polarization,\
int disecq,\
int cpid,\
int vpid,\
int apid,\
int tpid,\
int ca,\
int pnr)
{
struct program *pa;

if(debug_flag)
	{
	fprintf(stdout, "add_service(): arg\n\
	name=%s\n\
	service_name=%s\n\
	transponder_id=%d\n\
	orig_nw_id=%d\n\
	orbit=%.2f\n\
	west_east=%c\n\
	*country_avail=%s\n\
	*service_type=%s\n\
	symbolrate=%d\n\
	*fec=%s\n\
	*provider=%s\n\
	version=%d\n\
	frequency=%d\n\
	polarization=%c\n\
	disecq=%d\n\
	cpid=%d\n\
	vpid=%d\n\
	apid=%d\n\
	tpid=%d\n\
	ca=%d\n\
	pnr=%d\n",\
	name, service_name, transponder_id, orig_nw_id, orbit, west_east,\
	country_avail, service_type, symbolrate, fec, provider,\
	version, frequency, polarization, disecq,\
	cpid, vpid, apid, tpid, ca, pnr);
	}

pa = install_program_at_end_of_list(name);
	if(! pa) return 0;
pa -> service_name = strsave(service_name);
	if(! pa -> service_name) return 0;
pa -> transponder_id = transponder_id;
pa -> orig_nw_id = orig_nw_id;
pa -> orbit = orbit;
pa -> west_east = west_east;
pa -> country_avail = strsave(country_avail);
	if(! pa -> country_avail) return 0;
pa -> service_type = strsave(service_type);
	if(! pa -> service_type) return 0;
pa -> fec = strsave(fec);
	if(! pa -> fec) return 0;
pa -> provider = strsave(provider);
	if(! pa -> provider) return 0;
pa -> version = version;
pa -> frequency = frequency;
pa -> polarization = polarization;
pa -> disecq = disecq;
pa -> symbolrate = symbolrate;
pa -> cpid = cpid;
pa -> vpid = vpid;
pa -> apid = apid;
pa -> tpid = tpid;
pa -> ca = ca;
pa -> pnr = pnr;

return 1;
}/* end function add_program */


int swap_program_position(struct program *ptop, struct program *pbottom)
{
struct program *punder;
struct program *pabove;

if(debug_flag)
	{
	fprintf(stdout,\
	"swap_position(): swapping top=%lu bottom=%lu\n", ptop, pbottom);
	}

/* get one below the bottom */
punder = pbottom -> prventr;/* could be zero if first entry */
if(debug_flag)
	{
	fprintf(stdout,\
	"swap_position(): punder=%lu\n", punder);
	}

/* get the one above the top */
pabove = ptop -> nxtentr;/* could be zero if last entry */
if(debug_flag)
	{
	fprintf(stdout,\
	"swap_position(): pabove=%lu\n", pabove);
	}

/* the next pointer in punder (or programtab[0]) must now point to ptop */
if(! punder)
	{
	programtab[0] = ptop; 
	}
else
	{
	punder -> nxtentr = ptop;
	}

/* the prev pointer in in ptop must now point to punder */
ptop -> prventr = punder;/* could be zero if first entry */

/* the next pointer in ptop must now point to pbottom */
ptop -> nxtentr = pbottom;

/* the next pointer in pbottom must now point to pabove */
pbottom -> nxtentr = pabove;

/* mark last one in programtabtab */
if(! pabove)
	{
	programtab[1] = pbottom;
	}
else
	{
	/* the prev pointer in pabove must now point to pbottom */
	pabove -> prventr = pbottom;
	}

/* the prev pointer in pbottom must now point to ptop */
pbottom -> prventr = ptop;

/* return swapped */
return 1;
}/* end function swap_program_position */


int sort_program_list(int mode)
/*
sorts the double linked program with criterium set by  mode.
doing some sort of bubble sort.
*/
{
struct program *pa;
struct program *pb;
int swap_flag;
char temp[4096];
FILE *pptr;
FILE *fptr;

if(debug_flag)
	{
	fprintf(stdout, "sort_program_list(): arg mode=%d\n", mode);
	}

if(mode == SORT_ALPHA)
	{
	fprintf(stdout,\
	"Sorting database alphanumeric, this may take a while...\n");
	}
else if(mode == SORT_FREQUENCY)
	{
	fprintf(stdout,\
	"Sorting database by frequency, this may take a wile\n");
	}

/* sort the double linked list */
while(1)/* go through program again and again */
	{
	swap_flag = 0;
	for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)
		{
		pb = pa -> prventr;
		if(pb)
			{
			if(mode == SORT_ALPHA)
				{
				if(debug_flag)
					{
					fprintf(stdout, "pa->name=%s pb->name=%s\n",\
					pa -> name, pb -> name);
					}

				if(strcasecmp(pa -> name, pb -> name) < 0)
					{
					swap_flag = swap_program_position(pa, pb);
					/* indicate position was swapped */
					}/* end if condition */
				}
			else if(mode == SORT_FREQUENCY)
				{
				if(pa -> frequency < pb -> frequency)
					{
					swap_flag = swap_program_position(pa, pb);
					}		
				}			
			}/* end if pb */
		}/* end for all entries */

	/* if no more swapping took place, ready, program is sorted */
	if(! swap_flag) break;
	}/* end while go through program again and again */

return 1;
}/* end function sort_program_list */


#ifdef USE_GTK
int load_program_list(\
char *buffer, gchar *list_item_data_key, GtkWidget *gtklist)
{
int a;
struct program *pa;
GtkWidget *list_item;
FILE *fptr;
//extern struct program *programtab[2];
//extern gtk_misc_set_alignment();

for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)
	{
	GtkWidget       *label;
	gchar           *string;
	
	a = readline(fptr, buffer); /* closes file on EOF */
	if(a == EOF) break;

	label = gtk_label_new(buffer);

	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);

	list_item = gtk_list_item_new();/* centers display */

	gtk_container_add(GTK_CONTAINER(list_item), label);

	gtk_widget_show(label);

	gtk_container_add(GTK_CONTAINER(gtklist), list_item);

	gtk_widget_show(list_item);

	gtk_label_get(GTK_LABEL(label), &string);

	gtk_object_set_data(GTK_OBJECT(list_item),
		list_item_data_key,
		string);

	}/* end for all entries */


return 1;
}/* end function load_program_list */
#endif


int save_programs_gvideo_format()
{
char temp[TEMP_SIZE];
struct program *pa;
int a;
FILE *fptr;
int polarization, signalrate, vpid, apid, lnb, type, dvbnum;
int fec;

if(debug_flag)
	{
	fprintf(stdout,\
	"save_programs_gvideo_format(): arg none\n");
	}

sprintf(temp, "%s/gvideorc", home_dir);
fprintf(stdout, "Writing output to %s\n", temp);

fptr = fopen(temp, "w");
if(! fptr)
	{
	fprintf(stdout,\
	"save_programs_vdr_format(): could not open file %s for write\n",\
	temp);
	return 0;
	}

fprintf(fptr,\
"#file generated by pscan\n\
# gvideo format:\n\
#\n\
#	* = start of a channel declaration\n\
#	Channel    : <Name>\n\
#	Frequency  : <frequency> (unit depends on tuner)\n\
#	CHBC       : <Color> <Hue> <Brightness> <Contrast>\n\
#					(numbers depend on your hardware)\n\
#	NI         : <TV norm> <Input> (if norm=-1 then SAT is expected\n\
#	SAT        : <Polarization> (0 = vertical, 1 horizontal)\n\
#					<Signal Rate>\n\
#					<FEC> (0=1/2, 1=2/3, 2=3/4, 3=4/5, 4=5/6, 5=6/7\n\
#						6=7/8, 7=8/9 8=Auto\n\
#					<VPID> <APID> <LNB Nr.>\n\
#					<type> (Analog,DVBS,DVBT,DVBC)=(0,1,2,3)\n\
#					<dvbnum> id of channel in DVB structure\n\
");

for(pa = programtab[0]; pa != 0; pa = pa -> nxtentr)	
	{
	if(pa -> polarization == 'v') polarization = 0;
	else polarization = 1;
	type = 1; /* DVB-s */
	dvbnum = 0;
	lnb = 0;

	if(strcmp(pa -> fec, "1/2") == 0) fec = 0;
	else if(strcmp(pa -> fec, "2/3") == 0) fec = 1;
	else if(strcmp(pa -> fec, "3/4") == 0) fec = 2;
	else if(strcmp(pa -> fec, "4/5") == 0) fec = 3;
	else if(strcmp(pa -> fec, "5/6") == 0) fec = 4;
	else if(strcmp(pa -> fec, "6/7") == 0) fec = 5;
	else if(strcmp(pa -> fec, "7/8") == 0) fec = 7;
	else if(strcmp(pa -> fec, "8/9") == 0) fec = 8;

	fprintf(fptr, "*\n");
	fprintf(fptr, "Channel: %s\n", pa -> service_name);
	fprintf(fptr, "Frequency: %d\n", pa -> frequency);
	fprintf(fptr, "CBHC: 255 0 0 255\n");
	fprintf(fptr, "NI: -1 0\n");
	fprintf(fptr, "SAT: %d %d %d %d %d %d %d %d\n",\
			polarization, pa -> symbolrate, fec,\
			pa -> vpid, pa -> apid, lnb,\
			type, dvbnum);
	fprintf(fptr, "\n");

	}/* end for all structure entries */	

return 1;
}/* end function save_programs_gvideo_format */


