/*
	Copyright (c) 2004-2006, The Dojo Foundation
	All Rights Reserved.

	Licensed under the Academic Free License version 2.1 or above OR the
	modified BSD license. For more information on Dojo licensing, see:

		http://dojotoolkit.org/community/licensing.shtml
*/

dojo.provide("dojo.gfx.svg");

dojo.require("dojo.lang.declare");
dojo.require("dojo.svg");

dojo.require("dojo.gfx.color");
dojo.require("dojo.gfx.common");
dojo.require("dojo.gfx.shape");
dojo.require("dojo.gfx.path");

dojo.require("dojo.experimental");
dojo.experimental("dojo.gfx.svg");

dojo.gfx.svg.getRef = function(name){
	// summary: returns a DOM Node specified by the name argument or null
	// name: String: an SVG external reference 
	if(!name || name == "none") return null;
	if(name.match(/^url\(#.+\)$/)){
		return dojo.byId(name.slice(5, -1));	// Node
	}
	// alternative representation of a reference
	if(name.match(/^#dj_unique_.+$/)){
		// we assume here that a reference was generated by dojo.gfx
		return dojo.byId(name.slice(1));	// Node
	}
	return null;	// Node
};

dojo.gfx.svg.dasharray = {
	solid:				"none",
	shortdash:			[4, 1],
	shortdot:			[1, 1],
	shortdashdot:		[4, 1, 1, 1],
	shortdashdotdot:	[4, 1, 1, 1, 1, 1],
	dot:				[1, 3],
	dash:				[4, 3],
	longdash:			[8, 3],
	dashdot:			[4, 3, 1, 3],
	longdashdot:		[8, 3, 1, 3],
	longdashdotdot:		[8, 3, 1, 3, 1, 3]
};

dojo.lang.extend(dojo.gfx.Shape, {
	// summary: SVG-specific implementation of dojo.gfx.Shape methods
	
	setFill: function(fill){
		// summary: sets a fill object (SVG)
		// fill: Object: a fill object
		//	(see dojo.gfx.defaultLinearGradient, 
		//	dojo.gfx.defaultRadialGradient, 
		//	dojo.gfx.defaultPattern, 
		//	or dojo.gfx.color.Color)
		var h = dojo.render.html;
		
		if(!fill){
			// don't fill
			this.fillStyle = null;
			this.rawNode.setAttribute("fill", "none");
			this.rawNode.setAttribute("fill-opacity", 0);
			return this;
		}
		if(typeof(fill) == "object" && "type" in fill){
			// gradient
			switch(fill.type){
				case "linear":
					var f = dojo.gfx.makeParameters(dojo.gfx.defaultLinearGradient, fill);
					var gradient = this._setFillObject(f, "linearGradient");
					dojo.lang.forEach(["x1", "y1", "x2", "y2"], function(x){
						gradient.setAttribute(x, (h.safari ? parseInt(f[x]) : f[x].toFixed(8)));
					});
					break;
				case "radial":
					var f = dojo.gfx.makeParameters(dojo.gfx.defaultRadialGradient, fill);
					var gradient = this._setFillObject(f, "radialGradient");
					dojo.lang.forEach(["cx", "cy", "r"], function(x){
						gradient.setAttribute(x, f[x].toFixed(8));
					});
					break;
				case "pattern":
					var f = dojo.gfx.makeParameters(dojo.gfx.defaultPattern, fill);
					var pattern = this._setFillObject(f, "pattern");
					dojo.lang.forEach(["x", "y", "width", "height"], function(x){
						pattern.setAttribute(x, f[x].toFixed(8));
					});
					break;
			}
			return this;
		}
		// color object
		var f = dojo.gfx.normalizeColor(fill);
		this.fillStyle = f;
		this.rawNode.setAttribute("fill", f.toCss());
		this.rawNode.setAttribute("fill-opacity", f.a);
		return this;	// self
	},

	setStroke: function(stroke){
		// summary: sets a stroke object (SVG)
		// stroke: Object: a stroke object
		//	(see dojo.gfx.defaultStroke) 
	
		if(!stroke){
			// don't stroke
			this.strokeStyle = null;
			this.rawNode.setAttribute("stroke", "none");
			this.rawNode.setAttribute("stroke-opacity", 0);
			return this;
		}
		// normalize the stroke
		if(typeof stroke == "string"){
			stroke = {color: stroke};
		}
		var s = this.strokeStyle = dojo.gfx.makeParameters(dojo.gfx.defaultStroke, stroke);
		s.color = dojo.gfx.normalizeColor(s.color);
		// generate attributes
		var rn = this.rawNode;
		if(s){
			rn.setAttribute("stroke", s.color.toCss());
			rn.setAttribute("stroke-opacity", s.color.a);
			rn.setAttribute("stroke-width",   s.width);
			rn.setAttribute("stroke-linecap", s.cap);
			if(typeof(s.join) == "number"){
				rn.setAttribute("stroke-linejoin",   "miter");
				rn.setAttribute("stroke-miterlimit", s.join);
			}else{
				rn.setAttribute("stroke-linejoin",   s.join);
			}
			var da = s.style.toLowerCase();
			if(da in dojo.gfx.svg.dasharray){ da = dojo.gfx.svg.dasharray[da]; }
			if(da instanceof Array){
				for(var i = 0; i < da.length; ++i){
					da[i] *= s.width;
				}
				if(s.cap != "butt"){
					for(var i = 0; i < da.length; i += 2){
						da[i] -= s.width;
						if(da[i] < 1){ da[i] = 1; }
					}
					for(var i = 1; i < da.length; i += 2){
						da[i] += s.width;
					}
				}
				da = da.join(",");
			}
			rn.setAttribute("stroke-dasharray", da);
			rn.setAttribute("dojoGfxStrokeStyle", s.style);
		}
		return this;	// self
	},
	
	_getParentSurface: function(){
		var surface = this.parent;
		for(; surface && !(surface instanceof dojo.gfx.Surface); surface = surface.parent);
		return surface;
	},

	_setFillObject: function(f, nodeType){
		this.fillStyle = f;
		var surface = this._getParentSurface();
		var defs = surface.defNode;
		var fill = this.rawNode.getAttribute("fill");
		var ref  = dojo.gfx.svg.getRef(fill);
		if(ref){
			fill = ref;
			if(fill.tagName.toLowerCase() != nodeType.toLowerCase()){
				var id = fill.id;
				fill.parentNode.removeChild(fill);
				fill = document.createElementNS(dojo.svg.xmlns.svg, nodeType);
				fill.setAttribute("id", id);
				defs.appendChild(fill);
			}else{
				while(fill.childNodes.length){
					fill.removeChild(fill.lastChild);
				}
			}
		}else{
			fill = document.createElementNS(dojo.svg.xmlns.svg, nodeType);
			fill.setAttribute("id", dojo.dom.getUniqueId());
			defs.appendChild(fill);
		}
		if(nodeType == "pattern"){
			fill.setAttribute("patternUnits", "userSpaceOnUse");
			var img = document.createElementNS(dojo.svg.xmlns.svg, "image");
			img.setAttribute("x", 0);
			img.setAttribute("y", 0);
			img.setAttribute("width",  f.width .toFixed(8));
			img.setAttribute("height", f.height.toFixed(8));
			img.setAttributeNS(dojo.svg.xmlns.xlink, "href", f.src);
			fill.appendChild(img);
		}else{
			fill.setAttribute("gradientUnits", "userSpaceOnUse");
			for(var i = 0; i < f.colors.length; ++i){
				f.colors[i].color = dojo.gfx.normalizeColor(f.colors[i].color);
				var t = document.createElementNS(dojo.svg.xmlns.svg, "stop");
				t.setAttribute("offset",     (dojo.render.html.safari ? parseInt(f.colors[i].offset) : f.colors[i].offset.toFixed(8)));
				t.setAttribute("stop-color", f.colors[i].color.toCss());
				fill.appendChild(t);
			}
		}
		this.rawNode.setAttribute("fill", "url(#" + fill.getAttribute("id") +")");
		this.rawNode.removeAttribute("fill-opacity");
		return fill;
	},
	
	_applyTransform: function() {
		var matrix = this.matrix;
		if(matrix){
			var tm = this.matrix;
			if(dojo.render.html.safari){
				this.rawNode.setAttribute("transform", "matrix(" +
					parseInt(tm.xx) + "," + parseInt(tm.yx) + "," +
					parseInt(tm.xy) + "," + parseInt(tm.yy) + "," +
					parseInt(tm.dx) + "," + parseInt(tm.dy) + ")");
			}else{
				this.rawNode.setAttribute("transform", "matrix(" +
					tm.xx.toFixed(8) + "," + tm.yx.toFixed(8) + "," +
					tm.xy.toFixed(8) + "," + tm.yy.toFixed(8) + "," +
					tm.dx.toFixed(8) + "," + tm.dy.toFixed(8) + ")");
			}
		}else{
			this.rawNode.removeAttribute("transform");
		}
		return this;
	},

	setRawNode: function(rawNode){
		// summary:
		//	assigns and clears the underlying node that will represent this
		//	shape. Once set, transforms, gradients, etc, can be applied.
		//	(no fill & stroke by default)
		with(rawNode){
			setAttribute("fill", "none");
			setAttribute("fill-opacity", 0);
			setAttribute("stroke", "none");
			setAttribute("stroke-opacity", 0);
			setAttribute("stroke-width", 1);
			setAttribute("stroke-linecap", "butt");
			setAttribute("stroke-linejoin", "miter");
			setAttribute("stroke-miterlimit", 4);
		}
		this.rawNode = rawNode;
	},

	moveToFront: function(){
		// summary: moves a shape to front of its parent's list of shapes (SVG)
		this.rawNode.parentNode.appendChild(this.rawNode);
		return this;	// self
	},
	moveToBack: function(){
		// summary: moves a shape to back of its parent's list of shapes (SVG)
		this.rawNode.parentNode.insertBefore(this.rawNode, this.rawNode.parentNode.firstChild);
		return this;	// self
	},
	
	setShape: function(newShape){
		// summary: sets a shape object (SVG)
		// newShape: Object: a shape object
		//	(see dojo.gfx.defaultPath,
		//	dojo.gfx.defaultPolyline,
		//	dojo.gfx.defaultRect,
		//	dojo.gfx.defaultEllipse,
		//	dojo.gfx.defaultCircle,
		//	dojo.gfx.defaultLine,
		//	or dojo.gfx.defaultImage)
		this.shape = dojo.gfx.makeParameters(this.shape, newShape);
		for(var i in this.shape){
			if(i != "type"){ this.rawNode.setAttribute(i, this.shape[i]); }
		}
		return this;	// self
	},
	
	attachFill: function(rawNode){
		// summary: deduces a fill style from a Node.
		// rawNode: Node: an SVG node
		var fillStyle = null;
		if(rawNode){
			var fill = rawNode.getAttribute("fill");
			if(fill == "none"){ return null; }
			var ref  = dojo.gfx.svg.getRef(fill);
			if(ref){
				var gradient = ref;
				switch(gradient.tagName.toLowerCase()){
					case "lineargradient":
						fillStyle = this._getGradient(dojo.gfx.defaultLinearGradient, gradient);
						dojo.lang.forEach(["x1", "y1", "x2", "y2"], function(x){
							fillStyle[x] = gradient.getAttribute(x);
						});
						break;
					case "radialgradient":
						fillStyle = this._getGradient(dojo.gfx.defaultRadialGradient, gradient);
						dojo.lang.forEach(["cx", "cy", "r"], function(x){
							fillStyle[x] = gradient.getAttribute(x);
						});
						fillStyle.cx = gradient.getAttribute("cx");
						fillStyle.cy = gradient.getAttribute("cy");
						fillStyle.r  = gradient.getAttribute("r");
						break;
					case "pattern":
						fillStyle = dojo.lang.shallowCopy(dojo.gfx.defaultPattern, true);
						dojo.lang.forEach(["x", "y", "width", "height"], function(x){
							fillStyle[x] = gradient.getAttribute(x);
						});
						fillStyle.src = gradient.firstChild.getAttributeNS(dojo.svg.xmlns.xlink, "href");
						break;
				}
			}else{
				fillStyle = new dojo.gfx.color.Color(fill);
				var opacity = rawNode.getAttribute("fill-opacity");
				if(opacity != null) fillStyle.a = opacity;
			}
		}
		return fillStyle;	// Object
	},
	
	_getGradient: function(defaultGradient, gradient){
		var fillStyle = dojo.lang.shallowCopy(defaultGradient, true);
		fillStyle.colors = [];
		for(var i = 0; i < gradient.childNodes.length; ++i){
			fillStyle.colors.push({
				offset: gradient.childNodes[i].getAttribute("offset"),
				color:  new dojo.gfx.color.Color(gradient.childNodes[i].getAttribute("stop-color"))
			});
		}
		return fillStyle;
	},

	attachStroke: function(rawNode){
		// summary: deduces a stroke style from a Node.
		// rawNode: Node: an SVG node
		if(!rawNode){ return null; }
		var stroke = rawNode.getAttribute("stroke");
		if(stroke == null || stroke == "none") return null;
		var strokeStyle = dojo.lang.shallowCopy(dojo.gfx.defaultStroke, true);
		var color = new dojo.gfx.color.Color(stroke);
		if(color){
			strokeStyle.color = color;
			strokeStyle.color.a = rawNode.getAttribute("stroke-opacity");
			strokeStyle.width = rawNode.getAttribute("stroke-width");
			strokeStyle.cap = rawNode.getAttribute("stroke-linecap");
			strokeStyle.join = rawNode.getAttribute("stroke-linejoin");
			if(strokeStyle.join == "miter"){
				strokeStyle.join = rawNode.getAttribute("stroke-miterlimit");
			}
			strokeStyle.style = rawNode.getAttribute("dojoGfxStrokeStyle");
		}
		return strokeStyle;	// Object
	},

	attachTransform: function(rawNode){
		// summary: deduces a transformation matrix from a Node.
		// rawNode: Node: an SVG node
		var matrix = null;
		if(rawNode){
			matrix = rawNode.getAttribute("transform");
			if(matrix.match(/^matrix\(.+\)$/)){
				var t = matrix.slice(7, -1).split(",");
				matrix = dojo.gfx.matrix.normalize({
					xx: parseFloat(t[0]), xy: parseFloat(t[2]), 
					yx: parseFloat(t[1]), yy: parseFloat(t[3]), 
					dx: parseFloat(t[4]), dy: parseFloat(t[5])
				});
			}
		}
		return matrix;	// dojo.gfx.matrix.Matrix
	},
	
	attachShape: function(rawNode){
		// summary: builds a shape from a Node.
		// rawNode: Node: an SVG node
		var shape = null;
		if(rawNode){
			shape = dojo.lang.shallowCopy(this.shape, true);
			for(var i in shape) {
				shape[i] = rawNode.getAttribute(i);
			}
		}
		return shape;	// dojo.gfx.Shape
	},

	attach: function(rawNode){
		// summary: reconstructs all shape parameters from a Node.
		// rawNode: Node: an SVG node
		if(rawNode) {
			this.rawNode = rawNode;
			this.fillStyle = this.attachFill(rawNode);
			this.strokeStyle = this.attachStroke(rawNode);
			this.matrix = this.attachTransform(rawNode);
			this.shape = this.attachShape(rawNode);
		}
	},
	
	_getRealMatrix: function(){
		var m = this.matrix;
		var p = this.parent;
		while(p){
			if(p.matrix){
				m = dojo.gfx.matrix.multiply(p.matrix, m);
			}
			p = p.parent;
		}
		return m;
	}
});

dojo.declare("dojo.gfx.Group", dojo.gfx.Shape, {
	// summary: a group shape (SVG), which can be used 
	//	to logically group shapes (e.g, to propagate matricies)

	setRawNode: function(rawNode){
		// summary: sets a raw SVG node to be used by this shape
		// rawNode: Node: an SVG node
		this.rawNode = rawNode;
	}
});
dojo.gfx.Group.nodeType = "g";

dojo.declare("dojo.gfx.Rect", dojo.gfx.shape.Rect, {
	// summary: a rectangle shape (SVG)

	attachShape: function(rawNode){
		// summary: builds a rectangle shape from a Node.
		// rawNode: Node: an SVG node
		var shape = null;
		if(rawNode){
			shape = dojo.gfx.Rect.superclass.attachShape.apply(this, arguments);
			shape.r = Math.min(rawNode.getAttribute("rx"), rawNode.getAttribute("ry"));
		}
		return shape;	// dojo.gfx.shape.Rect
	},
	setShape: function(newShape){
		// summary: sets a rectangle shape object (SVG)
		// newShape: Object: a rectangle shape object
		this.shape = dojo.gfx.makeParameters(this.shape, newShape);
		this.bbox = null;
		for(var i in this.shape){
			if(i != "type" && i != "r"){ this.rawNode.setAttribute(i, this.shape[i]); }
		}
		if(this.shape.r){
			this.rawNode.setAttribute("ry", this.shape.r);
			this.rawNode.setAttribute("rx", this.shape.r);
		}
		return this;	// self
	}
});
dojo.gfx.Rect.nodeType = "rect";

dojo.gfx.Ellipse = dojo.gfx.shape.Ellipse;
dojo.gfx.Ellipse.nodeType = "ellipse";

dojo.gfx.Circle = dojo.gfx.shape.Circle;
dojo.gfx.Circle.nodeType = "circle";

dojo.gfx.Line = dojo.gfx.shape.Line;
dojo.gfx.Line.nodeType = "line";

dojo.declare("dojo.gfx.Polyline", dojo.gfx.shape.Polyline, {
	// summary: a polyline/polygon shape (SVG)
	
	setShape: function(points, closed){
		// summary: sets a polyline/polygon shape object (SVG)
		// points: Object: a polyline/polygon shape object
		if(points && points instanceof Array){
			// branch
			// points: Array: an array of points
			this.shape = dojo.gfx.makeParameters(this.shape, { points: points });
			if(closed && this.shape.points.length){ 
				this.shape.points.push(this.shape.points[0]);
			}
		}else{
			this.shape = dojo.gfx.makeParameters(this.shape, points);
		}
		this.box = null;
		var attr = [];
		var p = this.shape.points;
		if(dojo.render.html.safari){
			for(var i = 0; i < p.length; ++i){
				if(typeof p[i] == "number"){
					attr.push(parseInt(p[i]));
				}else{
					attr.push(parseInt(p[i].x));
					attr.push(parseInt(p[i].y));
				}
			}
		}else{
			for(var i = 0; i < p.length; ++i){
				if(typeof p[i] == "number"){
					attr.push(p[i].toFixed(8));
				}else{
					attr.push(p[i].x.toFixed(8));
					attr.push(p[i].y.toFixed(8));
				}
			}
		}
		this.rawNode.setAttribute("points", attr.join(" "));
		return this;	// self
	}
});
dojo.gfx.Polyline.nodeType = "polyline";

dojo.declare("dojo.gfx.Image", dojo.gfx.shape.Image, {
	// summary: an image (SVG)

	setShape: function(newShape){
		// summary: sets an image shape object (SVG)
		// newShape: Object: an image shape object
		this.shape = dojo.gfx.makeParameters(this.shape, newShape);
		this.bbox = null;
		var rawNode = this.rawNode;
		for(var i in this.shape){
			if(i != "type" && i != "src"){ rawNode.setAttribute(i, this.shape[i]); }
		}
		rawNode.setAttributeNS(dojo.svg.xmlns.xlink, "href", this.shape.src);
		return this;	// self
	},
	setStroke: function(){
		// summary: ignore setting a stroke style
		return this;	// self
	},
	setFill: function(){
		// summary: ignore setting a fill style
		return this;	// self
	},
	attachStroke: function(rawNode){
		// summary: ignore attaching a stroke style
		return null;
	},
	attachFill: function(rawNode){
		// summary: ignore attaching a fill style
		return null;
	}
});
dojo.gfx.Image.nodeType = "image";

dojo.declare("dojo.gfx.Text", dojo.gfx.shape.Text, {
	// summary: an anchored text (SVG)

	attachShape: function(rawNode){
		// summary: builds a text shape from a Node.
		// rawNode: Node: an SVG node
		var shape = null;
		if(rawNode){
			shape = dojo.lang.shallowCopy(dojo.gfx.defaultText, true);
			shape.x = rawNode.getAttribute("x");
			shape.y = rawNode.getAttribute("y");
			shape.align = rawNode.getAttribute("text-anchor");
			shape.decoration = rawNode.getAttribute("text-decoration");
			shape.rotated = parseFloat(rawNode.getAttribute("rotate")) != 0;
			shape.kerning = rawNode.getAttribute("kerning") == "auto";
			shape.text = rawNode.firstChild.nodeValue;
		}
		return shape;	// dojo.gfx.shape.Text
	},
	setShape: function(newShape){
		// summary: sets a text shape object (SVG)
		// newShape: Object: a text shape object
		this.shape = dojo.gfx.makeParameters(this.shape, newShape);
		this.bbox = null;
		var r = this.rawNode;
		var s = this.shape;
		r.setAttribute("x", s.x);
		r.setAttribute("y", s.y);
		r.setAttribute("text-anchor", s.align);
		r.setAttribute("text-decoration", s.decoration);
		r.setAttribute("rotate", s.rotated ? 90 : 0);
		r.setAttribute("kerning", s.kerning ? "auto" : 0);
		r.textContent = s.text;
		return this;	// self
	},
	attach: function(rawNode){
		// summary: reconstructs all shape parameters from a Node.
		// rawNode: Node: an SVG node
		dojo.gfx.Shape.prototype.attach.call(this, rawNode);
		if(rawNode) {
			this.fontStyle = this.attachFont(rawNode);
		}
	}
});
dojo.gfx.Text.nodeType = "text";

dojo.declare("dojo.gfx.Path", dojo.gfx.path.Path, {
	// summary: a path shape (SVG)

	_updateWithSegment: function(segment){
		// summary: updates the bounding box of path with new segment
		// segment: Object: a segment
		dojo.gfx.Path.superclass._updateWithSegment.apply(this, arguments);
		if(typeof(this.shape.path) == "string"){
			this.rawNode.setAttribute("d", this.shape.path);
		}
	},
	setShape: function(newShape){
		// summary: forms a path using a shape (SVG)
		// newShape: Object: an SVG path string or a path object (see dojo.gfx.defaultPath)
		dojo.gfx.Path.superclass.setShape.apply(this, arguments);
		this.rawNode.setAttribute("d", this.shape.path);
		return this;	// self
	}
});
dojo.gfx.Path.nodeType = "path";

dojo.declare("dojo.gfx.TextPath", dojo.gfx.path.TextPath, {
	// summary: a textpath shape (SVG)

	_updateWithSegment: function(segment){
		// summary: updates the bounding box of path with new segment
		// segment: Object: a segment
		dojo.gfx.Path.superclass._updateWithSegment.apply(this, arguments);
		this._setTextPath();
	},
	setShape: function(newShape){
		// summary: forms a path using a shape (SVG)
		// newShape: Object: an SVG path string or a path object (see dojo.gfx.defaultPath)
		dojo.gfx.Path.superclass.setShape.apply(this, arguments);
		this._setTextPath();
		return this;	// self
	},
	_setTextPath: function(){
		if(typeof this.shape.path != "string"){ return; }
		var r = this.rawNode;
		if(!r.firstChild){
			var tp = document.createElementNS(dojo.svg.xmlns.svg, "textPath");
			var tx = document.createTextNode("");
			tp.appendChild(tx);
			r.appendChild(tp);
		}
		var ref  = r.firstChild.getAttributeNS(dojo.svg.xmlns.xlink, "href");
		var path = ref && dojo.gfx.svg.getRef(ref);
		if(!path){
			var surface = this._getParentSurface();
			if(surface){
				var defs = surface.defNode;
				path = document.createElementNS(dojo.svg.xmlns.svg, "path");
				var id = dojo.dom.getUniqueId();
				path.setAttribute("id", id);
				defs.appendChild(path);
				r.firstChild.setAttributeNS(dojo.svg.xmlns.xlink, "href", "#" + id);
			}
		}
		if(path){
			path.setAttribute("d", this.shape.path);
		}
	},
	_setText: function(){
		var r = this.rawNode;
		if(!r.firstChild){
			var tp = document.createElementNS(dojo.svg.xmlns.svg, "textPath");
			var tx = document.createTextNode("");
			tp.appendChild(tx);
			r.appendChild(tp);
		}
		r = r.firstChild;
		var t = this.text;
		r.setAttribute("alignment-baseline", "middle");
		switch(t.align){
			case "middle":
				r.setAttribute("text-anchor", "middle");
				r.setAttribute("startOffset", "50%");
				break;
			case "end":
				r.setAttribute("text-anchor", "end");
				r.setAttribute("startOffset", "100%");
				break;
			default:
				r.setAttribute("text-anchor", "start");
				r.setAttribute("startOffset", "0%");
				break;
		}
		//r.parentNode.setAttribute("alignment-baseline", "central");
		//r.setAttribute("dominant-baseline", "central");
		r.setAttribute("baseline-shift", "0.5ex");
		r.setAttribute("text-decoration", t.decoration);
		r.setAttribute("rotate", t.rotated ? 90 : 0);
		r.setAttribute("kerning", t.kerning ? "auto" : 0);
		r.firstChild.data = t.text;
	},
	attachText: function(rawNode){
		// summary: builds a textpath shape from a Node.
		// rawNode: Node: an SVG node
		var shape = null;
		if(rawNode){
			shape = dojo.lang.shallowCopy(dojo.gfx.defaultTextPath, true);
			shape.align = rawNode.getAttribute("text-anchor");
			shape.decoration = rawNode.getAttribute("text-decoration");
			shape.rotated = parseFloat(rawNode.getAttribute("rotate")) != 0;
			shape.kerning = rawNode.getAttribute("kerning") == "auto";
			shape.text = rawNode.firstChild.nodeValue;
		}
		return shape;	// dojo.gfx.shape.TextPath
	},
	attach: function(rawNode){
		// summary: reconstructs all shape parameters from a Node.
		// rawNode: Node: an SVG node
		dojo.gfx.Shape.prototype.attach.call(this, rawNode);
		if(rawNode) {
			this.fontStyle = this.attachFont(rawNode);
			this.text = this.attachText(rawNode);
		}
	}
});
dojo.gfx.TextPath.nodeType = "text";


dojo.gfx.svg._font = {
	_setFont: function(){
		// summary: sets a font object (SVG)
		var f = this.fontStyle;
		// next line doesn't work in Firefox 2 or Opera 9
		//this.rawNode.setAttribute("font", dojo.gfx.makeFontString(this.fontStyle));
		this.rawNode.setAttribute("font-style", f.style);
		this.rawNode.setAttribute("font-variant", f.variant);
		this.rawNode.setAttribute("font-weight", f.weight);
		this.rawNode.setAttribute("font-size", f.size);
		this.rawNode.setAttribute("font-family", f.family);
	},
	attachFont: function(rawNode){
		// summary: deduces a font style from a Node.
		// rawNode: Node: an SVG node
		if(!rawNode){ return null; }
		var fontStyle = dojo.lang.shallowCopy(dojo.gfx.defaultFont, true);
		fontStyle.style = rawNode.getAttribute("font-style");
		fontStyle.variant = rawNode.getAttribute("font-variant");
		fontStyle.weight = rawNode.getAttribute("font-weight");
		fontStyle.size = rawNode.getAttribute("font-size");
		fontStyle.family = rawNode.getAttribute("font-family");
		return fontStyle;	// Object
	}
};

dojo.lang.extend(dojo.gfx.Text, dojo.gfx.svg._font);
dojo.lang.extend(dojo.gfx.TextPath, dojo.gfx.svg._font);

delete dojo.gfx.svg._font;

dojo.gfx.svg._creators = {
	// summary: SVG shape creators
	createPath: function(path){
		// summary: creates an SVG path shape
		// path: Object: a path object (see dojo.gfx.defaultPath)
		return this.createObject(dojo.gfx.Path, path);	// dojo.gfx.Path
	},
	createRect: function(rect){
		// summary: creates an SVG rectangle shape
		// rect: Object: a path object (see dojo.gfx.defaultRect)
		return this.createObject(dojo.gfx.Rect, rect);	// dojo.gfx.Rect
	},
	createCircle: function(circle){
		// summary: creates an SVG circle shape
		// circle: Object: a circle object (see dojo.gfx.defaultCircle)
		return this.createObject(dojo.gfx.Circle, circle);	// dojo.gfx.Circle
	},
	createEllipse: function(ellipse){
		// summary: creates an SVG ellipse shape
		// ellipse: Object: an ellipse object (see dojo.gfx.defaultEllipse)
		return this.createObject(dojo.gfx.Ellipse, ellipse);	// dojo.gfx.Ellipse
	},
	createLine: function(line){
		// summary: creates an SVG line shape
		// line: Object: a line object (see dojo.gfx.defaultLine)
		return this.createObject(dojo.gfx.Line, line);	// dojo.gfx.Line
	},
	createPolyline: function(points){
		// summary: creates an SVG polyline/polygon shape
		// points: Object: a points object (see dojo.gfx.defaultPolyline)
		//	or an Array of points
		return this.createObject(dojo.gfx.Polyline, points);	// dojo.gfx.Polyline
	},
	createImage: function(image){
		// summary: creates an SVG image shape
		// image: Object: an image object (see dojo.gfx.defaultImage)
		return this.createObject(dojo.gfx.Image, image);	// dojo.gfx.Image
	},
	createText: function(text){
		// summary: creates an SVG text shape
		// text: Object: a text object (see dojo.gfx.defaultText)
		return this.createObject(dojo.gfx.Text, text);	// dojo.gfx.Text
	},
	createTextPath: function(text){
		// summary: creates an SVG text shape
		// text: Object: a textpath object (see dojo.gfx.defaultTextPath)
		return this.createObject(dojo.gfx.TextPath, {}).setText(text);	// dojo.gfx.TextPath
	},
	createGroup: function(){
		// summary: creates an SVG group shape
		return this.createObject(dojo.gfx.Group);	// dojo.gfx.Group
	},
	createObject: function(shapeType, rawShape){
		// summary: creates an instance of the passed shapeType class
		// shapeType: Function: a class constructor to create an instance of
		// rawShape: Object: properties to be passed in to the classes "setShape" method
		if(!this.rawNode){ return null; }
		var shape = new shapeType();
		var node = document.createElementNS(dojo.svg.xmlns.svg, shapeType.nodeType); 
		shape.setRawNode(node);
		this.rawNode.appendChild(node);
		shape.setShape(rawShape);
		this.add(shape);
		return shape;	// dojo.gfx.Shape
	},
	// group control
	add: function(shape){
		// summary: adds a shape to a group/surface
		// shape: dojo.gfx.Shape: an SVG shape object
		var oldParent = shape.getParent();
		if(oldParent){
			oldParent.remove(shape, true);
		}
		shape._setParent(this, null);
		this.rawNode.appendChild(shape.rawNode);
		return this;	// self
	},
	remove: function(shape, silently){
		// summary: remove a shape from a group/surface
		// shape: dojo.gfx.Shape: an SVG shape object
		// silently: Boolean?: if true, regenerate a picture
		if(this.rawNode == shape.rawNode.parentNode){
			this.rawNode.removeChild(shape.rawNode);
		}
		shape._setParent(null, null);
		return this;	// self
	},
	clear: function(){
		// summary: removes all shapes from a group/surface
		var r = this.rawNode;
		while(r.lastChild && r.firstChild != r.lastChild){
			r.removeChild(r.lastChild);
		}
		return this;	// self
	}
};

dojo.gfx.attachNode = function(node){
	// summary: creates a shape from a Node
	// node: Node: an SVG node
	if(!node) return null;
	var s = null;
	switch(node.tagName.toLowerCase()){
		case dojo.gfx.Rect.nodeType:
			s = new dojo.gfx.Rect();
			break;
		case dojo.gfx.Ellipse.nodeType:
			s = new dojo.gfx.Ellipse();
			break;
		case dojo.gfx.Polyline.nodeType:
			s = new dojo.gfx.Polyline();
			break;
		case dojo.gfx.Path.nodeType:
			s = new dojo.gfx.Path();
			break;
		case dojo.gfx.Circle.nodeType:
			s = new dojo.gfx.Circle();
			break;
		case dojo.gfx.Line.nodeType:
			s = new dojo.gfx.Line();
			break;
		case dojo.gfx.Image.nodeType:
			s = new dojo.gfx.Image();
			break;
		case dojo.gfx.Text.nodeType:
			var t = node.getElementsByTagName("textPath");
			if(t && t.length){
				s = new dojo.gfx.TextPath();
			}else{
				s = new dojo.gfx.Text();
			}
			break;
		default:
			dojo.debug("FATAL ERROR! tagName = " + node.tagName);
	}
	s.attach(node);
	return s;	// dojo.gfx.Shape
};

dojo.lang.extend(dojo.gfx.Surface, {
	// summary: a surface object to be used for drawings (SVG)

	setDimensions: function(width, height){
		// summary: sets the width and height of the rawNode
		// width: String: width of surface, e.g., "100px"
		// height: String: height of surface, e.g., "100px"
		if(!this.rawNode){ return this; }
		this.rawNode.setAttribute("width",  width);
		this.rawNode.setAttribute("height", height);
		return this;	// self
	},
	getDimensions: function(){
		// summary: returns an object with properties "width" and "height"
		return this.rawNode ? {width: this.rawNode.getAttribute("width"), height: this.rawNode.getAttribute("height")} : null; // Object
	}
});

dojo.gfx.createSurface = function(parentNode, width, height){
	// summary: creates a surface (SVG)
	// parentNode: Node: a parent node
	// width: String: width of surface, e.g., "100px"
	// height: String: height of surface, e.g., "100px"

	var s = new dojo.gfx.Surface();
	s.rawNode = document.createElementNS(dojo.svg.xmlns.svg, "svg");
	s.rawNode.setAttribute("width",  width);
	s.rawNode.setAttribute("height", height);

	var defs = new dojo.gfx.svg.Defines();
	var node = document.createElementNS(dojo.svg.xmlns.svg, dojo.gfx.svg.Defines.nodeType); 
	defs.setRawNode(node);
	s.rawNode.appendChild(node);
	s.defNode = node;
	
	dojo.byId(parentNode).appendChild(s.rawNode);
	return s;	// dojo.gfx.Surface
};

dojo.gfx.attachSurface = function(node){
	// summary: creates a surface from a Node
	// node: Node: an SVG node
	var s = new dojo.gfx.Surface();
	s.rawNode = node;
	var def_elems = node.getElementsByTagName("defs");
	if(def_elems.length == 0){
		return null;	// dojo.gfx.Surface
	}
	s.defNode = def_elems[0];
	return s;	// dojo.gfx.Surface
};

dojo.lang.extend(dojo.gfx.Group, dojo.gfx.svg._creators);
dojo.lang.extend(dojo.gfx.Surface, dojo.gfx.svg._creators);

delete dojo.gfx.svg._creators;

// Gradient and pattern

dojo.gfx.svg.Defines = function(){
	this.rawNode = null;
};
dojo.lang.extend(dojo.gfx.svg.Defines, {
	setRawNode: function(rawNode){
		this.rawNode = rawNode;
	}
});
dojo.gfx.svg.Defines.nodeType = "defs";
