/* Usage: cat file_list | modify_rpmdb --name=pkgname --version=version --mode=replace */

#include <unistd.h>
#include <getopt.h>
#include <malloc.h>
#include <string.h>
#include <sys/param.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>

#include <rpm/header.h>
#include <rpm/dbindex.h>
#include <rpm/rpmlib.h>
#include "rpmdb.h"
#include "md5.h"

int patch=0;
int replace=0;
char **excludes;
char **excludes_versions;
unsigned char *excludes_flags;
char **requires;
char **requires_versions;
unsigned char *requires_flags;
char **provides;
char **files;
int release = 0;

#define COOKIE_TAG 1000

const struct option long_opts[] = {
#    undef rpm_field
#    define rpm_field(name,tag,type) {name, required_argument,NULL,COOKIE_TAG},
#    include "rpmfields.h"

  {"excludes", required_argument, NULL, 'e'},
  {"help", no_argument, NULL, 'h'},
  {"mode", required_argument, NULL, 'm'},
  {"provides", required_argument, NULL, 'p'},
  {"requires", required_argument, NULL, 'r'},
  {0,0,0,0},
};

const int rpm_tags[] = {
#   undef rpm_field
#   define rpm_field(name,tag,type) (tag),
#   include "rpmfields.h"
};

const int rpm_tag_types[] = {
#   undef rpm_field
#   define rpm_field(name,tag,type) (type),
#   include "rpmfields.h"
};

#define NUM_RPM_TAGS	(sizeof(rpm_tags)/sizeof(int))

void *rpm_tag_values[NUM_RPM_TAGS] = {0,} ;
int rpm_tag_counts[NUM_RPM_TAGS] = {0,} ;
char rpm_tags_selected[NUM_RPM_TAGS] = {0,};

rpmdb db = NULL;

static int
find_rpm_tag_selected(int tag) {
    int i;
    for(i = 0; i < NUM_RPM_TAGS; i++) {
        if (rpm_tags[i] == tag) return rpm_tags_selected[i];
    }
    fprintf (stderr, "rpm_tag_selected: unable to find tag %d.\n", tag);
    return 0;
}
    
static void*
find_rpm_tag_value(int tag) {
    int i;
    for(i = 0; i < NUM_RPM_TAGS; i++) {
        if (rpm_tags[i] == tag) return rpm_tag_values[i];
    }
    fprintf (stderr, "rpm_tag_selected: unable to find tag %d.\n", tag);
    return 0;
}

static void *
mem_alloc_check(void *ptr) {
    if (ptr == NULL) {
        fprintf (stderr, "Memory allocation failure\n");
	exit(1);
    }
    return ptr;
}

static void *
xmalloc(int size) {
    return mem_alloc_check(malloc(size));
}

static void *
xrealloc(void *ptr, int size) {
    if (ptr == NULL)
        return xmalloc(size);
    else
        return mem_alloc_check(realloc(ptr,size));
}

static void *
xstrdup(const char *string) {
    return mem_alloc_check(strdup(string));
}

static char **
alloc_argvector(int num_strings) {
    const int size = sizeof(char*) * num_strings;
    char **result = xmalloc(size);
    memset(result, '\0', size);
    return result;
}

static void
append_to_vector(char **vector, char *string) {
    while (*vector) vector++;
    *vector = string;
}

static int
vector_len(char** vector) {
    char **const orig = vector;
    while(*vector) vector++;
    return vector - orig;
}

static void
parse_require_conflict(char *input, char **name, char **version,
		       unsigned char *flags) {
    /*STUB*/
    *name = xstrdup(input);
    *version = "";
    *flags = 0;
}
    
static void
usage(FILE *output_file, int exit_status) {
    int i;
    fprintf (output_file,
	     "Usage: cat file_list | modify_rpmdb --name=name --version=version [options]\n"
	     "Options:\n"
	     "\t--excludes=pkgspec (repeatable)\n"
	     "\t--group=groupname\n"
             "\t--provides=pkgname (repeatable)\n"
	     "\t--requires=pkgspec (repeatable)\n");
    for(i = 0; long_opts[i].val == COOKIE_TAG; i++) {
        fprintf(output_file, "\t--%s=%s\n", long_opts[i].name,
	       (rpm_tag_types[i] == RPM_STRING_TYPE ? "string" :
		(rpm_tag_types[i] == RPM_BIN_TYPE ? "file (containing binary data)" :
		 (rpm_tag_types[i] == RPM_STRING_ARRAY_TYPE ?
		  "string (repeatable)" : "number"))));
    }
    
    exit(exit_status);
}

static void*
read_file(const char *filename,
	  int *byte_count /* output value */) {
    int fd;
    struct stat statbuf;
    char *result, *ptr;
    int remaining;

    if (stat(filename, &statbuf) < 0) {
        perror(filename);
	exit(1);
    }
    if (!S_ISREG(statbuf.st_mode)) {
        fprintf (stderr, "%s: not a regular file.\n", filename);
	exit(1);
    }
    ptr = result = xmalloc((remaining = statbuf.st_size));
    while (remaining > 0) {
        int read_result = read(fd, ptr, remaining);
	if (read_result < 0 && errno == EAGAIN) continue;
	if (read_result < 0) {
	    fprintf (stderr, "reading %s: %s.\n", filename, strerror(errno));
	    exit(1);
	}
	if (read_result == 0) {
	    fprintf (stderr, "Warning: only read %d bytes of file %s, which appeared to be %ld bytes long.\n",
		     ptr - result, filename, statbuf.st_size);
	    exit(1);
	}
	ptr += read_result;
	remaining -= read_result;
    }
    return result;
}

static void *
append_value(void *old_vector, int *count_ptr,
	     void *new_value, int bytes_per_elt) {
    void *result = xrealloc(old_vector,
			    ((*count_ptr) + 1) * bytes_per_elt);
    memcpy(result + (*count_ptr)*bytes_per_elt, new_value, bytes_per_elt);
    (*count_ptr)++;
    return result;
}

static void *
convert_to_rpm_type(char *input_string, int rpm_type, void *previous_value,
		    int *count /* input and output variable.*/ ) {
    char **vector;
    switch (rpm_type) {
        case RPM_STRING_TYPE:
	    *count = 1;
	    return xstrdup(input_string);
        case RPM_INT8_TYPE:
	    {
	        unsigned char val = atoi(input_string);
		return append_value(previous_value, count, &val, sizeof(val));
	    }
        case RPM_INT16_TYPE:
	    {
	        unsigned short val = atoi(input_string);
		return append_value(previous_value, count, &val, sizeof(val));
	    }
        case RPM_INT32_TYPE:
	    {
	        unsigned int val = atoi(input_string);
		return append_value(previous_value, count, &val, sizeof(val));
	    }
        case RPM_STRING_ARRAY_TYPE:
	    return append_value(previous_value, count, xstrdup(input_string),
				sizeof(char*));
	    return vector;
        case RPM_BIN_TYPE:
	    return read_file(input_string, count);
        default:
	    fprintf (stderr, "convert_to_rpm_type: Unrecognized type %d.\n", rpm_type);
	    exit(1);
	    /*NOTREACHED*/
    }
}

static void
parse_args (int argc, char **argv) {
   int opt;
   int index;
   int i;
   while ((opt = getopt_long(argc,argv,"e:n:p:r:v", long_opts, &index)) != -1){
        switch(opt) {
	  case 'e': 
	    i = vector_len(excludes);
	    parse_require_conflict(optarg, &excludes[i],
				   &excludes_versions[i], &excludes_flags[i]);
	    break;
	  case 'h':
	    usage(stdout,0);
	    /*NOTREACHED*/
	    break;
	  case 'm':
	    if (strcmp(optarg, "replace") == 0)
	      replace = 1;
	    else if (strcmp(optarg, "patch") == 0)
	      patch = 1;
	    else {
	      fprintf (stderr, "Unrecognized mode %s.\n", optarg);
	      exit(1);
	    }
	  case 'p': append_to_vector(provides, optarg); break;
	  case 'r':
	    i = vector_len(requires);
	    parse_require_conflict(optarg, &requires[i],
				   &requires_versions[i], &requires_flags[i]);
	    break;
	  case ':': fprintf (stderr, "argument required\n"); break;
	  case '?': exit(1);	/* unrecognized option */
	  case COOKIE_TAG:
	      rpm_tag_values[index] =
		  convert_to_rpm_type(optarg, rpm_tag_types[index],
				      rpm_tag_values[index],
				      &rpm_tag_counts[index]);
	      rpm_tags_selected[index] = 1;
	      break;
	  default: fprintf (stderr, "Unrecognized option '%c'\n", opt); break;
	}
    }
    if (optind != argc) usage(stderr, 1);
}

static void
check_required_args(void) {
    if (!find_rpm_tag_selected(RPMTAG_NAME)) {
        fprintf(stderr,"Missing required argument \"--name=name\"\n");
	exit(1);
    }
    
    if (!find_rpm_tag_selected(RPMTAG_VERSION)) {
        fprintf(stderr,"Missing required argument \"--version=version\"\n");
	exit(1);
    }

    if (!patch && !replace) {
        fprintf (stderr, "Must specify \"--mode=patch\" or \"--mode=replace.\"\n");
	exit(1);
    }
    if (patch && replace) {
        fprintf (stderr, "Cannot specify both \"--mode=patch\" and \"--mode=replace.\"\n");
	exit(1);
    }
}

static char **
read_lines(FILE *input) {
    char **result;
    int max_size;
    int size;
    char buf[MAXPATHLEN+2];

    size = max_size = 1;
    result = xmalloc(sizeof(char*));

    while (fgets(buf,sizeof(buf),input) != NULL) {
        char *lastchar;
        size++;
	if (size > max_size) {
	    max_size *= 2;
	    result = xrealloc(result, max_size * sizeof(char*));
	}
	lastchar = buf + strlen(buf) - 1;
	if (*lastchar == '\n') *lastchar = '\0';
	result[size-2] = xstrdup(buf);
    }
    return result;
}

static int
in_directory(char *filename, char *prefix) {
    if (strcmp(filename,prefix) == 0) {
        return 0;
    } else {
        const int len = strlen(prefix);
	return (strncmp(filename,prefix,len) == 0 && filename[len] == '/');
    }
}

static unsigned int
filename_to_flags(char *filename) {
    if (in_directory(filename, "/etc") ||
	in_directory(filename, "/var/db")) {
        return RPMFILE_CONFIG;
    }
    if (in_directory(filename, "/var/man") ||
	in_directory(filename, "/usr/man") ||
	in_directory(filename, "/usr/share/info") ||
	in_directory(filename, "/usr/share/man") ||
	in_directory(filename, "/usr/info")) {
        return RPMFILE_DOC;
    }
    return 0;
}

static void
register_files(Header header, char **files) {
   int count = vector_len(files);
   unsigned int uids[count], gids[count], sizes[count],
       mtimes[count],devices[count],inodes[count], flags[count];
   unsigned short modes[count],rdevs[count];
   unsigned char states[count];
   char *md5s[count];
   char *linktos[count];
   char *usernames[count];
   char *groupnames[count];
   /* RPMTAG_FILELANGS? */
   int i;

   for (i = 0; i < count; i++ ) {
       struct stat statbuf;
       struct passwd *pwd;
       struct group *grp;

       if (lstat(files[i], &statbuf) < 0) {
	   perror(files[i]);
	   exit(1);
       }
       uids[i] = statbuf.st_uid;
       gids[i] = statbuf.st_gid;
       sizes[i] = statbuf.st_size;
       modes[i] = statbuf.st_mode;
       rdevs[i] = statbuf.st_rdev;
       mtimes[i] = statbuf.st_mtime;
       devices[i] = statbuf.st_dev;
       inodes[i] = statbuf.st_ino;
       states[i] = RPMFILE_STATE_NORMAL;
       flags[i] = filename_to_flags(files[i]);
       if (S_ISLNK(statbuf.st_mode)) {
	   char link_contents[MAXPATHLEN+1];
	   readlink(files[i], link_contents, sizeof(link_contents));
	   linktos[i] = xstrdup(link_contents);
       } else {
	   linktos[i] = "";
       }
       if (S_ISREG(statbuf.st_mode)) {
	   char md5buf[100];
	   mdfile(files[i], md5buf);
	   md5s[i] = xstrdup(md5buf);
       } else {
	   md5s[i] = "";
       }
       pwd = getpwuid(uids[i]);
       usernames[i] = pwd ? xstrdup(pwd->pw_name) : NULL;
       grp = getgrgid(gids[i]);
       groupnames[i] = grp ? xstrdup(grp->gr_name) : NULL;
   }

   headerAddOrAppendEntry(header,RPMTAG_FILENAMES,RPM_STRING_ARRAY_TYPE,files,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEUIDS,RPM_INT32_TYPE,uids,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEGIDS,RPM_INT32_TYPE,gids,count);
   headerAddOrAppendEntry(header,RPMTAG_FILESIZES,RPM_INT32_TYPE,sizes,count);
   headerAddOrAppendEntry(header,RPMTAG_FILESTATES,RPM_INT8_TYPE,states,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEMODES,RPM_INT16_TYPE,modes,count);
   headerAddOrAppendEntry(header,RPMTAG_FILERDEVS,RPM_INT16_TYPE,rdevs,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEMTIMES,RPM_INT32_TYPE,mtimes,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEMD5S,RPM_STRING_ARRAY_TYPE,md5s,count);
   headerAddOrAppendEntry(header,RPMTAG_FILELINKTOS,RPM_STRING_ARRAY_TYPE,linktos,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEFLAGS,RPM_INT32_TYPE,flags,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEUSERNAME,RPM_STRING_ARRAY_TYPE,usernames,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEGROUPNAME,RPM_STRING_ARRAY_TYPE,groupnames,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEDEVICES,RPM_INT32_TYPE,devices,count);
   headerAddOrAppendEntry(header,RPMTAG_FILEINODES,RPM_INT32_TYPE,inodes,count);
   /*headerAddOrAppendEntry(header,RPMTAG_FILELANGS,RPM_INT32_TYPE,langs,count);*/
}


static void
update_rpmdb(void) {
    int i;
    Header header = NULL;
    dbiIndexSet matches;

    rpmSetVar(RPMVAR_DBPATH, "/var/lib/rpm");
    if (rpmdbOpen("", &db, O_RDWR, 0644) != 0) {
	rpmError(RPMERR_DBOPEN, "failed to open RPM database /var/lib/rpm");
        perror("rpmdbOpen");
	exit(1);
    }

    if (rpmdbFindPackage(db, find_rpm_tag_value(RPMTAG_NAME),
			 &matches) != 0) {
        perror("rpmdbFindPackage");
	exit(1);
    }
    if (patch) {
        if (matches.count > 0) {
	    header = rpmdbGetRecord(db, matches.recs[0].recOffset);
	}
	/* STUB XXX: Print error if matches.count == 0? */
    }
    if (!header) header = headerNew();
    
    register_files(header, files);
    for (i = 0; i < NUM_RPM_TAGS; i++) {
        if (rpm_tags_selected[i]) {
            headerAddOrAppendEntry(header, rpm_tags[i], rpm_tag_types[i],
			   rpm_tag_values[i], rpm_tag_counts[i]);
	}
    }
    if (requires[0]) {
        headerAddOrAppendEntry(header, RPMTAG_REQUIRENAME,
			       RPM_STRING_ARRAY_TYPE,
			       requires, vector_len(requires));
        headerAddOrAppendEntry(header, RPMTAG_REQUIREVERSION,
			       RPM_STRING_ARRAY_TYPE,
			       requires_versions, vector_len(requires));
        headerAddOrAppendEntry(header, RPMTAG_REQUIREFLAGS,
			       RPM_INT8_TYPE,
			       requires_flags, vector_len(requires));
    }
    if (provides[0]) {
        headerAddOrAppendEntry(header, RPMTAG_PROVIDES, RPM_STRING_ARRAY_TYPE,
			       provides, vector_len(provides));
    }
    if (excludes[0]) {
        headerAddOrAppendEntry(header, RPMTAG_CONFLICTNAME,
			       RPM_STRING_ARRAY_TYPE,
			       excludes, vector_len(excludes));
        headerAddOrAppendEntry(header, RPMTAG_CONFLICTVERSION,
			       RPM_STRING_ARRAY_TYPE,
			       excludes_versions, vector_len(excludes));
        headerAddOrAppendEntry(header, RPMTAG_CONFLICTFLAGS,
			       RPM_INT8_TYPE,
			       excludes_flags, vector_len(excludes));
    }
    if (patch && matches.count > 0) {
       if (rpmdbUpdateRecord(db, matches.recs[0].recOffset, header) != 0) {
	    rpmError(RPMERR_DBPUTINDEX, "failed to update rpm database");
	    perror("rpmdbUpdateRecord");
       }
    } else {
        if (replace && matches.count > 0) {
	    int i;
	    for (i = 0; i < matches.count; i++) {
	        rpmdbRemove(db, matches.recs[i].recOffset, 1);
	    }
	}
        if (rpmdbAdd(db,header) != 0) {
	    rpmError(RPMERR_DBPUTINDEX, "failed to update rpm database");
	    perror("rpmdbAdd");
	}
    }
    rpmdbClose(db);
}

int
main(int argc, char **argv) {
    excludes = alloc_argvector(argc);	
    excludes_versions = alloc_argvector(argc);
    excludes_flags = xmalloc(argc * sizeof(*excludes_flags));
    memset(excludes_flags, '\0', argc*sizeof(*excludes_flags));
    provides = alloc_argvector(argc);	
    requires = alloc_argvector(argc);
    requires_versions = alloc_argvector(argc);
    requires_flags = xmalloc(argc * sizeof(unsigned char));
    memset(requires_flags, '\0', argc*sizeof(*requires_flags));

    parse_args(argc, argv);
    check_required_args();

    files = read_lines(stdin);

    update_rpmdb();
    return 0;
}
