//////////////////////////////////////////////////////////////         
// 	$Id: rpmInstall.cpp,v 1.2 1998/12/13 15:44:45 toivo Exp $	
#include "../config.h"
#ifdef HAVE_RPM
#define HAVE_ALLOCA_H

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_ALLOCA_H
# include <alloca.h>
#endif

extern "C"
{
#include <rpm/rpmlib.h>
#include "rpmInstall.h"
#include "rpmMessages.h"
}
#include "rpmutils.h"
#include "kpackage.h"

static int hashesPrinted = 0;

static void printHash(const unsigned long amount, const unsigned long total);
static void printPercent(const unsigned long amount, const unsigned long total);
static void printDepProblems(FILE * f, struct rpmDependencyConflict * conflicts,
			     int numConflicts);

static void printHash(const unsigned long amount, const unsigned long total) {
    int hashesNeeded;

    if (hashesPrinted != 50) {
	hashesNeeded = (int)(50 * (total ? (((float) amount) / total) : 1));
	while (hashesNeeded > hashesPrinted) {
	    printf("#");
	    hashesPrinted++;
	}
	fflush(stdout);
	hashesPrinted = hashesNeeded;

	if (hashesPrinted == 50)
	    printf("\n");
    }
}

static void printPercent(const unsigned long amount, const unsigned long total) 
{
  int percentage =  (int)(total
			  ? ((float) ((((float) amount) / total) * 100))
			  : 100.0);
  kpkg->kp->setPercent(percentage);
}

static int installPackages(char * rootdir, char ** packages, 
			    int numPackages, int installFlags, 
			    int interfaceFlags, rpmdb db) {
    int i, fd;
    int numFailed = 0;
    char ** filename;
    const char * printFormat = NULL;
    char * chptr;
    int rc;
    rpmNotifyFunction fn;
    char * netsharedPath = NULL;

    if (interfaceFlags & INSTALL_PERCENT)
	fn = printPercent;
    else if (interfaceFlags & INSTALL_HASH)
	fn = printHash;
    else
	fn = NULL;

    netsharedPath = rpmGetVar(RPMVAR_NETSHAREDPATH);

    for (i = 0, filename = packages; i < numPackages; i++, filename++) {
	if (!*filename) continue;

	hashesPrinted = 0;

	fd = open(*filename, O_RDONLY);
	if (fd < 0) {
	 KpMsgE(i18n("cannot open file %s"),*filename,FALSE);
	    numFailed++;
	    *filename = NULL;
	    continue;
	} 

	if (interfaceFlags & INSTALL_PERCENT) 
	    printFormat = "%%f %s:%s:%s\n";
	else if (rpmIsVerbose() && (interfaceFlags & INSTALL_HASH)) {
	    chptr = strrchr((const char *)*filename, '/');
	    if (!chptr)
		chptr = *filename;
	    else
		chptr++;

	    printFormat = "%-28s";
	} else if (rpmIsVerbose())
	    rpmMessage(RPMMESS_DEBUG, i18n("Installing %s\n"), *filename);

	if (db) {
	    rc = rpmInstallPackage(rootdir, db, fd, 0, installFlags, fn, 
				   NULL
#ifndef RPMSENSE_TRIGGERPOSTUN 
				   ,netsharedPath
#endif				   
				   ); 
	} else {
	    if (installFlags &= RPMINSTALL_TEST) {
		rpmMessage(RPMMESS_DEBUG, "stopping source install as we're "
			"just testing");
		rc = 0;
	    } else {
		rc = rpmInstallSourcePackage(rootdir, fd, NULL, fn,
					     (char *)printFormat,0);
	    }
	} 

	if (rc == 1) {
	  KpMsgE(i18n("%s does not appear to be a RPM package"),*filename,FALSE);
	}
	    
	if (rc) {
	  KpMsgE(i18n("%s cannot be installed"),*filename,TRUE);
	    numFailed++;
	}

	close(fd);
    }

    return numFailed;
}

int doInstall(char * rootdir, char ** argv, int installFlags, 
	      int interfaceFlags) {
    rpmdb db;
    int fd, i;
    int mode, rc;
    char ** packages, ** tmpPackages;
    char ** filename;
    int numPackages;
    int numTmpPackages = 0, numBinaryPackages = 0, numSourcePackages = 0;
    int numFailed = 0;
    Header * binaryHeaders;
    int isSource;
    rpmDependencies rpmdep;
    struct rpmDependencyConflict * conflicts;
    int numConflicts;
    int stopInstall = 0;

    if (installFlags & RPMINSTALL_TEST) 
	mode = O_RDONLY;
    else
	mode = O_RDWR | O_CREAT;

    rpmMessage(RPMMESS_DEBUG, i18n("counting packages to install"));
    for (filename = argv, numPackages = 0; *filename; filename++, numPackages++)
	;

    rpmMessage(RPMMESS_DEBUG, i18n("found %d packages"), numPackages);
    packages = (char**)alloca((numPackages + 1) * sizeof(char *));
    packages[numPackages] = NULL;
    tmpPackages = (char**)alloca((numPackages + 1) * sizeof(char *));
    binaryHeaders = (Header*)alloca((numPackages + 1) * sizeof(Header));
	
    for (filename = argv, i = 0; *filename; filename++) 
      {
	packages[i++] = *filename;
      }
    
    rpmMessage(RPMMESS_DEBUG,
	       i18n("finding source and binary packages"));
    for (filename = packages; *filename; filename++) 
      {
	fd = open(*filename, O_RDONLY);
	if (fd < 0) 
	  {
	  KpMsgE(i18n("cannot open file %s"),*filename,FALSE);
	    numFailed++;
	    *filename = NULL;
	    continue;
	  }
	
	rc = rpmReadPackageHeader(fd, &binaryHeaders[numBinaryPackages], &isSource,
				  NULL, NULL);
	
	close(fd);
	
	if (rc == 1) 
	  {
	  KpMsgE(i18n("%s does not appear to be a RPM package"),*filename,FALSE);	  }
	
	if (rc) 
	  {
	  KpMsgE(i18n("%s cannot be installed"),*filename,TRUE);
	    numFailed++;
	    *filename = NULL;
	  } 
	else if (isSource) 
	  {
	    /* the header will be NULL if this is a v1 source package */
	    if (binaryHeaders[numBinaryPackages])
	      headerFree(binaryHeaders[numBinaryPackages]);
	    
	    numSourcePackages++;
	  } 
	else 
	  {
	    numBinaryPackages++;
	  }
      }
    
    rpmMessage(RPMMESS_DEBUG,
	       i18n("found %d source and %d binary packages"), 
	       numSourcePackages, numBinaryPackages);
    
    if (numBinaryPackages) 
      {
	rpmMessage(RPMMESS_DEBUG,
		   i18n("opening database mode: 0%o"), mode);
	if (rpmdbOpen(rootdir, &db, mode, 0644)) {
	  KpMsgE(i18n("cannot open /var/lib/rpm/packages.rpm\nKpackage needs to be running as root"),"",TRUE);
	  return 1;
	}
	if (!(interfaceFlags & INSTALL_NODEPS)) 
	  {
	    rpmdep = rpmdepDependencies(db);
	    for (i = 0; i < numBinaryPackages; i++)
	      if (installFlags & RPMINSTALL_UPGRADE)
		rpmdepUpgradePackage(rpmdep, binaryHeaders[i],0);
	      else
		rpmdepAddPackage(rpmdep, binaryHeaders[i],0);
	    
	    if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) 
	      {
		numFailed = numPackages;
		stopInstall = 1;
	      }
	    
	    rpmdepDone(rpmdep);
	    
	    if (!stopInstall && conflicts) 
	      {
		rpmMessage(RPMMESS_DEBUG,
			   i18n("failed dependencies:"));
		printDepProblems(stderr, conflicts, numConflicts);
		rpmdepFreeConflicts(conflicts, numConflicts);
		numFailed = numPackages;
		stopInstall = 1;
	      }
	  }
      }
    else
      db = NULL;
    
    if (!stopInstall) 
      {
	rpmMessage(RPMMESS_DEBUG,
		   i18n("installing binary packages"));
	numFailed += installPackages(rootdir, packages, numPackages, 
				     installFlags, interfaceFlags, db);
      }
    
    for (i = 0; i < numTmpPackages; i++)
      unlink(tmpPackages[i]);
    
    for (i = 0; i < numBinaryPackages; i++) 
      headerFree(binaryHeaders[i]);
    
    if (db) rpmdbClose(db);
    
    return numFailed;
}

int doUninstall(char * rootdir, char ** argv, int uninstallFlags,
		 int interfaceFlags) {
    rpmdb db;
    dbiIndexSet matches;
    int i, j;
    int mode;
    int rc;
    int count;
    int numPackages;
    int * packageOffsets;
    char ** arg;
    int numFailed = 0;
    rpmDependencies rpmdep;
    struct rpmDependencyConflict * conflicts;
    int numConflicts;
    int stopUninstall = 0;

    rpmMessage(RPMMESS_DEBUG,
	       i18n("counting packages to uninstall"));
    for (arg = argv, numPackages = 0; *arg; arg++, numPackages++)
	;
    rpmMessage(RPMMESS_DEBUG,
	       i18n("found %d packages to uninstall"), numPackages);

    packageOffsets = (int*)alloca(sizeof(int *) * numPackages);

    if (uninstallFlags & RPMUNINSTALL_TEST) 
	mode = O_RDONLY;
    else
	mode = O_RDWR | O_EXCL;
	
    if (rpmdbOpen(rootdir, &db, mode, 0644)) {
      KpMsgE(i18n("cannot open /var/lib/rpm/packages.rpm\nKpackage needs to be running as root"),"",TRUE);
      return 1;
    }

    j = 0;
    for (arg = argv, numPackages = 0; *arg; arg++, numPackages++) {
	rc = findPackageByLabel(db, *arg, &matches);
	if (rc == 1) {
	  KpMsgE(i18n("package %s is not installed"),*arg,FALSE);
	    numFailed++;
	} else if (rc == 2) {
	  KpMsgE(i18n("error searching for package %s"),*arg,FALSE);
	    numFailed++;
	} else {
	    count = 0;
	    for (i = 0; i < matches.count; i++)
		if (matches.recs[i].recOffset) count++;

	    if (count > 1) {
	      KpMsgE(i18n("\"%s\" specifies multiple packages"),*arg,FALSE);
		numFailed++;
	    }
	    else { 
		for (i = 0; i < matches.count; i++) {
		    if (matches.recs[i].recOffset) {
			packageOffsets[j++] = matches.recs[i].recOffset;
		    }
		}
	    }

	    dbiFreeIndexRecord(matches);
	}
    }
    numPackages = j;

    if (!(interfaceFlags & UNINSTALL_NODEPS)) {
	rpmdep = rpmdepDependencies(db);
	for (i = 0; i < numPackages; i++)
	    rpmdepRemovePackage(rpmdep, packageOffsets[i]);

	if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) {
	    numFailed = numPackages;
	    stopUninstall = 1;
	}

	rpmdepDone(rpmdep);

	if (!stopUninstall && conflicts) {
	  rpmMessage(RPMMESS_WARNING,i18n("removing these packages would break "
			    "dependencies:\n"));
	    printDepProblems(stderr, conflicts, numConflicts);
	    rpmdepFreeConflicts(conflicts, numConflicts);
	    numFailed += numPackages;
	    stopUninstall = 1;
	}
    }

    if (!stopUninstall) {
	for (i = 0; i < numPackages; i++) {
	    rpmMessage(RPMMESS_DEBUG,
		       i18n("uninstalling record number %d"),
			packageOffsets[i]);
	    rpmRemovePackage(rootdir, db, packageOffsets[i], uninstallFlags);
	}
    }

    rpmdbClose(db);

    return numFailed;
}

int doSourceInstall(char * rootdir, char * arg, char ** specFile) {
    int fd;
    int rc;

    fd = open(arg, O_RDONLY);
    if (fd < 0) {
      KpMsgE(	    i18n("cannot open %s"),arg,TRUE);
	return 1;
    }

    if (rpmIsVerbose())
	rpmMessage(RPMMESS_DEBUG, i18n("Installing %s\n"), arg);

    rc = rpmInstallSourcePackage(rootdir, fd, specFile, NULL, NULL, 0);
    if (rc == 1) {
      KpMsgE(	    i18n("%s cannot be installed"),arg,TRUE);
    }

    close(fd);

    return rc;
}

void printDepFlags(FILE * f, QString s, char * version, int flags) {

  if (flags) {
    fprintf(f, " ");
    s += " ";
  }

  if (flags & RPMSENSE_LESS) {
    fprintf(f, "<");
    s += "<";
  }
  if (flags & RPMSENSE_GREATER) {
    fprintf(f, ">");
    s += ">";
  }
  if (flags & RPMSENSE_EQUAL) {
    fprintf(f, "=");
    s += "=";
  }
  if (flags & RPMSENSE_SERIAL) {
    fprintf(f, "S");
    s += "S";
  }
  if (flags) {
    fprintf(f, " %s", version);
    s += " ";
    s += version;
  }
}

static void printDepProblems(FILE * f, struct rpmDependencyConflict * conflicts,
			     int numConflicts) {
    int i;
    QString s, st;

    st += i18n("Dependency Problem:\n");
    for (i = 0; i < numConflicts; i++) {
	fprintf(f, "\t%s", conflicts[i].needsName);
	st += s.sprintf( "\t%s", conflicts[i].needsName);
	if (conflicts[i].needsFlags) {
	    printDepFlags(stderr, st, conflicts[i].needsVersion, 
			  conflicts[i].needsFlags);
	}

	if (conflicts[i].sense == rpmDependencyConflict::RPMDEP_SENSE_REQUIRES) {
	    fprintf(f, i18n(" is needed by %s-%s-%s\n"),
		    conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
	    st += s.sprintf( i18n(" is needed by %s-%s-%s\n"),
			     conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
	} else {
	    fprintf(f, i18n(" conflicts with %s-%s-%s\n"),
		    conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
	    st += s.sprintf( i18n(" conflicts with %s-%s-%s\n"),
			     conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
	}
    }
    KMsgBox::message(kpkg, i18n("Package not installed:"),
		     st.data(), KMsgBox::STOP);
}
#endif
